import math
import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
from copy import deepcopy
from torch.nn import CrossEntropyLoss
from transformers import BertPreTrainedModel, BertModel
    

class UtteranceEncoding(BertPreTrainedModel):
    def __init__(self, config):
        super(UtteranceEncoding, self).__init__(config)

        self.config = config
        self.bert = BertModel(config)
        
        self.init_weights()
        
    def forward(self, input_ids, attention_mask, token_type_ids, output_attentions=False, output_hidden_states=False):
        return self.bert(input_ids=input_ids, 
                         attention_mask=attention_mask, 
                         token_type_ids=token_type_ids, 
                         output_attentions=output_attentions, 
                         output_hidden_states=output_hidden_states)

    
class MultiHeadAttention(nn.Module):
    def __init__(self, heads, d_model, dropout=0.1):
        super().__init__()

        self.d_model = d_model
        self.d_k = d_model // heads
        self.h = heads

        self.q_linear = nn.Linear(d_model, d_model)
        self.v_linear = nn.Linear(d_model, d_model)
        self.k_linear = nn.Linear(d_model, d_model)
        self.dropout = nn.Dropout(dropout)
        self.out = nn.Linear(d_model, d_model)

        self.scores = None

    def attention(self, q, k, v, d_k, mask=None, dropout=None):
        scores = torch.matmul(q, k.transpose(-2, -1)) / math.sqrt(d_k)
        if mask is not None:
            mask = mask.unsqueeze(1)
            scores = scores.masked_fill(mask == 0, -1e9)
        scores = F.softmax(scores, dim=-1)

        if dropout is not None:
            scores = dropout(scores)

        self.scores = scores
        output = torch.matmul(scores, v)
        return output

    def forward(self, q, k, v, mask=None):
        bs = q.size(0)

        # perform linear operation and split into h heads
        k = self.k_linear(k).view(bs, -1, self.h, self.d_k)
        q = self.q_linear(q).view(bs, -1, self.h, self.d_k)
        v = self.v_linear(v).view(bs, -1, self.h, self.d_k)

        # transpose to get dimensions bs * h * sl * d_model
        k = k.transpose(1, 2)
        q = q.transpose(1, 2)
        v = v.transpose(1, 2)

        scores = self.attention(q, k, v, self.d_k, mask, self.dropout)

        # concatenate heads and put through final linear layer
        concat = scores.transpose(1, 2).contiguous().view(bs, -1, self.d_model)
        output = self.out(concat)
        return output

    def get_scores(self):
        return self.scores  
    
    
class MultiHeadAttentionTanh(nn.Module):
    def __init__(self, heads, d_model, dropout=0.1):
        super().__init__()

        self.d_model = d_model
        self.d_k = d_model // heads
        self.h = heads

        self.q_linear = nn.Linear(d_model, d_model)
        self.v_linear = nn.Linear(d_model, d_model)
        self.k_linear = nn.Linear(d_model, d_model)
        self.dropout = nn.Dropout(dropout)
        self.out = nn.Linear(d_model, d_model)

        self.scores = None

    def attention(self, q, k, v, d_k, mask=None, dropout=None):
        scores = torch.matmul(q, k.transpose(-2, -1)) / math.sqrt(d_k)
        scores = torch.tanh(scores)
#         scores = torch.sigmoid(scores)
        if mask is not None:
            mask = mask.unsqueeze(1)
            scores = scores.masked_fill(mask == 0, 0.)
#         scores = F.softmax(scores, dim=-1)

        if dropout is not None:
            scores = dropout(scores)

        self.scores = scores
        output = torch.matmul(scores, v)
        return output

    def forward(self, q, k, v, mask=None):
        bs = q.size(0)

        # perform linear operation and split into h heads
        k = self.k_linear(k).view(bs, -1, self.h, self.d_k)
        q = self.q_linear(q).view(bs, -1, self.h, self.d_k)
        v = self.v_linear(v).view(bs, -1, self.h, self.d_k)

        # transpose to get dimensions bs * h * sl * d_model
        k = k.transpose(1, 2)
        q = q.transpose(1, 2)
        v = v.transpose(1, 2)

        scores = self.attention(q, k, v, self.d_k, mask, self.dropout)

        # concatenate heads and put through final linear layer
        concat = scores.transpose(1, 2).contiguous().view(bs, -1, self.d_model)
        output = self.out(concat)
        return output

    def get_scores(self):
        return self.scores  
    
    
def clones(module, N):
    "Produce N identical layers."
    return nn.ModuleList([deepcopy(module) for _ in range(N)])
            

class SelfAttention(nn.Module):
    "A stack of N layers"
    def __init__(self, layer, N):
        super(SelfAttention, self).__init__()
        self.layers = clones(layer, N)
        self.norm = nn.LayerNorm(layer.size)
        
    def forward(self, x, mask=None):
        "Pass the input (and mask) through each layer in turn."
        for layer in self.layers:
            x = layer(x, mask)
        return self.norm(x)    
    
    
class SublayerConnection(nn.Module):
    """
    A residual connection followed by a layer norm.
    """
    def __init__(self, size, dropout):
        super(SublayerConnection, self).__init__()
        self.norm = nn.LayerNorm(size)
        self.dropout = nn.Dropout(dropout)

    def forward(self, x, sublayer):
        "Apply residual connection to any sublayer with the same size."
        x = self.norm(x)
        return x + self.dropout(sublayer(x))
    
    
class SelfAttentionLayer(nn.Module):
    def __init__(self, size, self_attn, feed_forward, dropout):
        super(SelfAttentionLayer, self).__init__()
        self.self_attn = self_attn
        self.feed_forward = feed_forward
        self.sublayer = clones(SublayerConnection(size, dropout), 2)
        self.size = size

    def forward(self, x, mask):
        x = self.sublayer[0](x, lambda x: self.self_attn(x, x, x, mask))
        return self.sublayer[1](x, self.feed_forward)
    
    
class PositionwiseFeedForward(nn.Module):
    "Implements FFN equation."
    def __init__(self, d_model, d_ff, dropout=0.1):
        super(PositionwiseFeedForward, self).__init__()
        self.w_1 = nn.Linear(d_model, d_ff)
        self.w_2 = nn.Linear(d_ff, d_model)
        self.dropout = nn.Dropout(dropout)
        self.gelu = nn.ReLU() # use gelu or relu

    def forward(self, x):
        return self.w_2(self.dropout(self.gelu(self.w_1(x))))    
    

class CrossAttention(nn.Module):
    def __init__(self, attn_head, model_output_dim, dropout=0., attn_type="softmax"):
        super(CrossAttention, self).__init__()
        self.attn_head = attn_head
        self.model_output_dim = model_output_dim
        self.dropout = dropout
        if attn_type == "tanh":
            self.attn_fun = MultiHeadAttentionTanh(self.attn_head, self.model_output_dim, dropout=0.)
        else:
            self.attn_fun = MultiHeadAttention(self.attn_head, self.model_output_dim, dropout=0.)
        
    def forward(self, query, value, attention_mask=None):
        num_query = query.size(0)
        batch_size = value.size(0)
        seq_length = value.size(1)
        
        expanded_query = query.unsqueeze(0).expand(batch_size, *query.shape)
        if attention_mask is not None:
            expanded_attention_mask = attention_mask.view(-1, seq_length, 1).expand(value.size()).float()
            new_value = torch.mul(value, expanded_attention_mask)
            attn_mask = attention_mask.unsqueeze(1).expand(batch_size, num_query, seq_length)
        else:
            new_value = value
            attn_mask = None
        
        attended_embedding = self.attn_fun(expanded_query, new_value, new_value, mask=attn_mask)
        
        return attended_embedding
        
        
class Decoder(nn.Module):
    def __init__(self, args, model_output_dim, num_labels, slot_value_pos, device):
        super(Decoder, self).__init__()
        self.model_output_dim = model_output_dim
        self.num_slots = len(num_labels)
        self.num_total_labels = sum(num_labels)
        self.num_labels = num_labels
        self.slot_value_pos = slot_value_pos
        self.attn_head = args.attn_head
        self.device = device
        self.args = args
        self.dropout_prob = self.args.dropout_prob
        self.dropout = nn.Dropout(self.dropout_prob)
        self.attn_type = self.args.attn_type
        
        ### slot utterance attention
        self.slot_utter_attn = UtteranceAttention(self.attn_head, self.model_output_dim, dropout=0., attn_type=self.attn_type)
        
        ### MLP
        self.SlotMLP = nn.Sequential(nn.Linear(self.model_output_dim * 2, self.model_output_dim),
                                     nn.ReLU(),
                                     nn.Dropout(p=self.dropout_prob),
                                     nn.Linear(self.model_output_dim, self.model_output_dim))

        ### basic modues, attention dropout is 0.1 by default
        attn = MultiHeadAttention(self.attn_head, self.model_output_dim)
        ffn = PositionwiseFeedForward(self.model_output_dim, self.model_output_dim, self.dropout_prob)
        
        ### attention layer, multiple self attention layers
        self.slot_self_attn = SlotSelfAttention(SlotAttentionLayer(self.model_output_dim, deepcopy(attn), 
                                                                   deepcopy(ffn), self.dropout_prob),
                                                                   self.args.num_self_attention_layer)
        
        ### prediction
        self.pred = nn.Sequential(nn.Dropout(p=self.dropout_prob), 
                                  nn.Linear(self.model_output_dim, self.model_output_dim),
                                  nn.LayerNorm(self.model_output_dim))
        
        ### measure
        self.distance_metric = args.distance_metric
        if self.distance_metric == "cosine":
            self.metric = torch.nn.CosineSimilarity(dim=-1, eps=1e-08)
        elif self.distance_metric == "euclidean":
            self.metric = torch.nn.PairwiseDistance(p=2.0, eps=1e-06, keepdim=False)
            
        self.logsoftmax = nn.LogSoftmax(dim=-1)
        self.softmax = nn.Softmax(dim=-1)
        self.nll = CrossEntropyLoss(ignore_index=-1)
       
    def slot_value_matching(self, value_lookup, hidden, target_slots, labels):
        loss = 0.
        loss_slot = []
        pred_slot = []
        
        batch_size = hidden.size(0)
        value_emb = value_lookup.weight[0:self.num_total_labels, :]
        
        for s, slot_id in enumerate(target_slots): # note: target_slots are successive
            hidden_label = value_emb[self.slot_value_pos[slot_id][0]:self.slot_value_pos[slot_id][1], :]
            num_slot_labels = hidden_label.size(0) # number of value choices for each slot
            
            _hidden_label = hidden_label.unsqueeze(0).repeat(batch_size, 1, 1).reshape(batch_size * num_slot_labels, -1)
            _hidden = hidden[:,s,:].unsqueeze(1).repeat(1, num_slot_labels, 1).reshape(batch_size * num_slot_labels, -1)
            _dist = self.metric(_hidden_label, _hidden).view(batch_size, num_slot_labels)
            
            if self.distance_metric == "euclidean":
                _dist = -_dist
               
            _, pred = torch.max(_dist, -1)
            pred_slot.append(pred.view(batch_size, 1))
            
            _loss = self.nll(_dist, labels[:, s])
            
            loss += _loss
            loss_slot.append(_loss.item())
            
        pred_slot = torch.cat(pred_slot, 1) # [batch_size, num_slots]
        
        return loss, loss_slot, pred_slot
     
    def forward(self, hidden, labels, value_lookup, eval_type="train"):
        
        batch_size = hidden.size(0)
        target_slots = list(range(0, self.num_slots))
        # slot value matching
        loss, loss_slot, pred_slot = self.slot_value_matching(value_lookup, hidden, target_slots, labels)
        
        return loss, loss_slot, pred_slot
        
class ListNet():
    def listnet(scores_pred, perm_true, eps=1e-10):
        max_pos = perm_true.max(dim=1,keepdim=True)[0] + 1
        scores_true = (max_pos-perm_true).float()/max_pos
                
        top1prob_pred = torch.softmax(scores_pred, 1) + eps
        top1prob_true = torch.softmax(scores_true, 1)
        loss_type = 'KL'
        if loss_type == 'CE':
            loss = - top1prob_true * torch.log(top1prob_pred)
        elif loss_type == 'KL':
            loss = nn.KLDivLoss(reduction='none')(torch.log(top1prob_pred), top1prob_true)
        elif loss_type == 'JS': # JS Divergence between two probabilities
            top1prob_mean = 0.5*(top1prob_pred+top1prob_true)
            kld = nn.KLDivLoss(reduction='none')
            loss = 0.5*kld(torch.log(top1prob_pred),top1prob_mean)+0.5*kld(torch.log(top1prob_true+eps), top1prob_mean)  
        loss = loss.sum(1).sum()
        return loss 

class BeliefTracker(nn.Module):
    def __init__(self, args, slot_lookup, value_lookup, num_labels, slot_value_pos, device):
        super(BeliefTracker, self).__init__()
        
        self.num_slots = len(num_labels)
        self.num_labels = num_labels
        self.slot_value_pos = slot_value_pos
        self.args = args
        self.device = device
        self.slot_lookup = slot_lookup 
        self.value_lookup = value_lookup
        self.ffn = PositionwiseFeedForward(768, 768, 0.1)
        self.attn = MultiHeadAttention(4, 768)
        target_slots = list(range(0, self.num_slots))
        self.slot_embedding = slot_lookup.weight[target_slots, :]  # select target slots' embeddings  
        self.softmax = nn.Softmax(dim=-1)
        self.nll = CrossEntropyLoss(ignore_index=-1)

        # context bert encoder 
        self.encoder = UtteranceEncoding.from_pretrained(self.args.pretrained_model)

        # context_2_slot_attn
        self.c2s_attn = CrossAttention(4, 768, dropout=0.)

        # slot_self_attn
        self.s_sa = self.slot_self_attn = SelfAttention(SelfAttentionLayer(self.model_output_dim, deepcopy(self.attn), 
                                                                   deepcopy(self.ffn), 0.1), 4)
        # order ranking
        self.listNet = ListNet()

        # slot_2_turn_single
        self.s2t_attn_s = CrossAttention(4, 768, dropout=0.)

        # turn_self_attn
        self.t_sa = SelfAttention(SelfAttentionLayer(self.model_output_dim, deepcopy(self.attn), 
                                                                   deepcopy(self.ffn), 0.1), 2)
        # slot_2_turn_overall
        self.s2t_attn_o = CrossAttention(4, 768, dropout=0.)
        self.slotMLP = nn.Linear(768, 1)

        # match value
        self.model_output_dim = self.encoder.config.hidden_size
        self.decoder = Decoder(args, self.model_output_dim, self.num_labels, self.slot_value_pos, device)

    def forward(self, input_ids, slot_order, split_list, align_labels, attention_mask, token_type_ids, labels, eval_type="train"):
        
        batch_size = input_ids.size(0)
        num_slots = self.num_slots
        
        # context_encode
        sequence_output = self.encoder(input_ids, attention_mask, token_type_ids)[0]    
        
        # c2s        
        slot_base_cont_emb = self.c2s_attn(self.slot_embedding, sequence_output, attention_mask)
        slot_base_cont_emb = torch.cat((self.slot_embedding.unsqueeze(0).repeat(batch_size, 1, 1), slot_base_cont_emb), 2)
        hidden_slot = self.s_sa(slot_base_cont_emb) 

        # rank loss
        r_loss = self.listNet.listnet(hidden_slot, slot_order)

        # slot_2_turn_single
        turn_embed_list = [torch.split(sequence_output[i,:,:], split_list[i]) for i in range(sequence_output.size(0))]
        s2c_list = []
        for i in range(len(turn_embed_list)):
            s2c_list_inner = []
            for j in range(len(turn_embed_list[i])):
                turn_base_slot_emb = self.s2t_attn_s(self.slot_embedding, turn_embed_list[i][j])
                s2c_list_inner.append(turn_base_slot_emb)
            s2c_list.append(s2c_list_inner)
        
        # slot_self_attn
        align_loss_tot = 0.
        value_match_loss = 0
        for i in range(len(s2c_list)):
            c_tensor = torch.stack(s2c_list[i], dim=0) # N * 768
            hidden_turn = self.s_sa(c_tensor) 
            slot_embed = torch.squeeze(self.slot_embedding, 1)
            
            turn_attn_o = self.s2t_attn_o(hidden_turn, slot_embed)
            turn_align_logits = self.slotMLP(turn_attn_o)
            pro_turn_align = self.softmax(turn_align_logits)
            align_label = align_labels[i]
            align_loss = self.nll(turn_align_logits, align_label)
            align_loss_tot += align_loss
            soft_turn_hidden = torch.mm(torch.transpose(pro_turn_align, 1,0), turn_attn_o)
            

    
            
       









        # calculate accuracy
        accuracy = pred_slot == labels
        acc_slot = torch.true_divide(torch.sum(accuracy, 0).float(), batch_size).cpu().detach().numpy() # slot accuracy
        acc = torch.sum(torch.floor_divide(torch.sum(accuracy, 1), num_slots)).float().item() / batch_size # joint accuracy
        
        return loss, loss_slot, acc, acc_slot, pred_slot