// Copyright 2013 Thomas Müller
// This file is part of MarMoT, which is licensed under GPLv3.

package marmot.core;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Options extends java.util.Properties {

	public static final long serialVersionUID = 1L;
	public static final String BEAM_SIZE = "beam-size";
	public static final String ORDER = "order";
	public static final String PRUNE = "prune";
	public static final String NUM_ITERATIONS = "num-iterations";
	public static final String PENALTY = "penalty";
	public static final String PROB_THRESHOLD = "prob-threshold";
	public static final String SHUFFLE = "shuffle";
	public static final String CANDIDATES_PER_STATE = "candidates-per-state";
	public static final String EFFECTIVE_ORDER = "effective-order";
	public static final String INITIAL_VECTOR_SIZE = "initial-vector-size";
	public static final String VERBOSE = "verbose";
	public static final String QUADRATIC_PENALTY = "quadratic-penalty";
	public static final String ORACLE = "oracle";
	public static final String MAX_TRANSITION_FEATURE_LEVEL = "max-transition-feature-level";
	public static final String VERY_VERBOSE = "very-verbose";
	public static final String TRAINER = "trainer";
	public static final String AVERAGING = "averaging";

	private static final Map<String, String> DEFALUT_VALUES_ = new HashMap<String, String>();

	static {
		DEFALUT_VALUES_.put(BEAM_SIZE, "5");
		DEFALUT_VALUES_.put(ORDER, "2");
		DEFALUT_VALUES_.put(PRUNE, "true");
		DEFALUT_VALUES_.put(NUM_ITERATIONS, "10");
		DEFALUT_VALUES_.put(PENALTY, "0.1");
		DEFALUT_VALUES_.put(PROB_THRESHOLD, "0.01");
		DEFALUT_VALUES_.put(SHUFFLE, "true");
		DEFALUT_VALUES_.put(CANDIDATES_PER_STATE, "[4, 2, 1.5]");
		DEFALUT_VALUES_.put(EFFECTIVE_ORDER, "1");
		DEFALUT_VALUES_.put(INITIAL_VECTOR_SIZE, "10000000");
		DEFALUT_VALUES_.put(VERBOSE, "false");
		DEFALUT_VALUES_.put(QUADRATIC_PENALTY, "0.0");
		DEFALUT_VALUES_.put(ORACLE, "false");
		DEFALUT_VALUES_.put(MAX_TRANSITION_FEATURE_LEVEL, "-1");
		DEFALUT_VALUES_.put(VERY_VERBOSE, "false");
		DEFALUT_VALUES_.put(TRAINER, CrfTrainer.class.getCanonicalName());
		DEFALUT_VALUES_.put(AVERAGING, "true");
	}

	public Options() {
		super();
		putAll(DEFALUT_VALUES_);
	}

	public Options(Options options) {
		this();
		putAll(options);
	}

	public String toSimpleString() {
		String string = "";

		Set<Object> key_set = keySet();
		List<String> key_list = new ArrayList<String>(key_set.size());
		for (Object key : keySet()) {
			key_list.add((String) (key));
		}
		Collections.sort(key_list);

		for (String key : key_list) {
			String value = getProperty(key);
			string += String.format("%s = %s\n", key, value);
		}
		return string;
	}

	public void writePropertiesToFile(String filename) {
		try {
			BufferedWriter writer = new BufferedWriter(new FileWriter(filename));
			writer.write(toSimpleString());
			writer.close();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	public void setPropertiesFromFile(String filename) {
		try {
			BufferedReader reader = new BufferedReader(new FileReader(filename));
			setPropertiesFromReader(reader);
			reader.close();
		} catch (FileNotFoundException e) {
			throw new RuntimeException(e);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	private String normalizeOption(String option) {
		return option.trim().replace("_", "-").toLowerCase();
	}

	public void setPropertiesFromReader(BufferedReader reader) {

		Pattern p = Pattern.compile("([^:=]*)[:=](.*)");

		try {
			while (reader.ready()) {
				String line = reader.readLine();

				line = line.trim();

				if (line.length() == 0) {
					continue;
				}

				Matcher m = p.matcher(line);

				if (!m.matches()) {
					throw new RuntimeException(String.format(
							"Invalid line: %s\n", line));
				}

				String key = normalizeOption(m.group(1));

				if (!this.containsKey(key)) {
					throw new RuntimeException(String.format(
							"Unknown property: %s\n", key));
				}

				String value = m.group(2).trim();

				if (value.endsWith(";")) {
					value = value.substring(0, value.length() - 1);
				}
				if (value.endsWith("\"") && value.startsWith("\"")) {
					value = value.substring(1, value.length() - 1);
				}
				value = new String(value);

				this.setProperty(key, value);
			}
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	private static final Pattern OPTION_PATTERN = Pattern.compile("-*(.*)");

	public void setPropertiesFromStrings(String[] args) {
		int index = 0;
		while (index < args.length) {
			String option = args[index++];
			Matcher m = OPTION_PATTERN.matcher(option);

			if (!m.matches()) {
				throw new RuntimeException("Unexpected argument: " + option
						+ ". Missing '-'?");
			}

			option = normalizeOption(m.group(1));
			if (option.equalsIgnoreCase("props")) {
				checkBoundaries(index, args);
				setPropertiesFromFile(args[index++]);
			} else if (this.containsKey(option)) {
				checkBoundaries(index, args);
				this.setProperty(option, args[index++]);
			} else {
				throw new RuntimeException(String.format(
						"Unknown property: %s\n", option));
			}
		}

		if (getVerbose()) {
			for (Map.Entry<Object, Object> prop : this.entrySet()) {
				System.err.println(prop.getKey() + ": " + prop.getValue());
			}
		}
	}

	private void checkBoundaries(int index, String[] args) {
		if (index >= args.length) {
			throw new RuntimeException("Missing argument");
		}
	}
	
	public void dieIfPropertyIsEmpty(String property) {
		if (getProperty(property).isEmpty()) {
			System.err.format("Error: Property '%s' needs to be set!\n", property);
			System.exit(1);
		}
	}

	public boolean getPrune() {
		return Boolean.parseBoolean(getProperty(PRUNE));
	}

	public int getBeamSize() {
		return Integer.parseInt(getProperty(BEAM_SIZE));
	}

	public int getOrder() {
		return Integer.parseInt(getProperty(ORDER));
	}

	public int getNumIterations() {
		return Integer.parseInt(getProperty(NUM_ITERATIONS));
	}

	public double getPenalty() {
		return Double.parseDouble(getProperty(PENALTY));
	}

	public double getProbThreshold() {
		return Double.parseDouble(getProperty(PROB_THRESHOLD));
	}

	public boolean getShuffle() {
		return Boolean.parseBoolean(getProperty(SHUFFLE));
	}

	public double[] getCandidatesPerState() {
		String array_string = getProperty(CANDIDATES_PER_STATE).trim();
		boolean error = false;
		double[] array = null;

		if (array_string.length() > 2 && array_string.charAt(0) == '['
				&& array_string.charAt(array_string.length() - 1) == ']') {
			array_string = array_string.substring(1, array_string.length() - 1);
			String[] element_strings = array_string.split(",");
			array = new double[element_strings.length];
			for (int index = 0; index < element_strings.length; index++) {
				double element = Double.valueOf(element_strings[index]);
				array[index] = element;
				if (element < 1.0) {
					error = true;
				}
			}
		} else {
			error = true;
		}

		if (error) {
			throw new InvalidParameterException("Not an array: "
					+ getProperty(CANDIDATES_PER_STATE));
		}

		return array;
	}

	public int getEffectiveOrder() {
		return Integer.parseInt(getProperty(EFFECTIVE_ORDER));
	}

	public int getInitialVectorSize() {
		return (int) Double.parseDouble(getProperty(INITIAL_VECTOR_SIZE));
	}

	public boolean getVerbose() {
		return Boolean.parseBoolean(getProperty(VERBOSE));
	}

	public double getQuadraticPenalty() {
		return Double.parseDouble(getProperty(QUADRATIC_PENALTY));
	}

	public boolean getOracle() {
		return Boolean.parseBoolean(getProperty(ORACLE));
	}

	public int getMaxTransitionFeatureLevel() {
		return Integer.parseInt(getProperty(MAX_TRANSITION_FEATURE_LEVEL));
	}

	public boolean getVeryVerbose() {
		return Boolean.parseBoolean(getProperty(VERY_VERBOSE));
	}

	public String getTrainer() {
		return getProperty(TRAINER);
	}

	public boolean getAveraging() {
		return Boolean.parseBoolean(getProperty(AVERAGING));
	}

}
