import torch
import random
import torch.nn as nn
from transformers import BartTokenizer, BartForConditionalGeneration
import torch.nn.functional as F
from typing import Optional


def move_device(tensor, device):
    if tensor is None:
        return None
    else:
        tensor = tensor.to(device)
        return tensor


def _expand_mask(mask: torch.Tensor, dtype: torch.dtype, tgt_len: Optional[int] = None):
    """
    Expands attention_mask from `[bsz, seq_len]` to `[bsz, 1, tgt_seq_len, src_seq_len]`.
    """
    bsz, src_len = mask.size()
    tgt_len = tgt_len if tgt_len is not None else src_len
    expanded_mask = mask[:, None, None, :].expand(bsz, 1, tgt_len, src_len).to(dtype)
    inverted_mask = 1.0 - expanded_mask
    return inverted_mask.masked_fill(inverted_mask.bool(), torch.finfo(dtype).min)


def shift_tokens_right(input_ids: torch.Tensor, pad_token_id: int, decoder_start_token_id: int):
    """
    Shift input ids one token to the right.
    """
    shifted_input_ids = input_ids.new_zeros(input_ids.shape)
    shifted_input_ids[:, 1:] = input_ids[:, :-1].clone()
    shifted_input_ids[:, 0] = decoder_start_token_id

    assert pad_token_id is not None, "self.model.config.pad_token_id has to be defined."
    # replace possible -100 values in labels by `pad_token_id`
    shifted_input_ids.masked_fill_(shifted_input_ids == -100, pad_token_id)

    return shifted_input_ids


class ShardedBART(nn.Module):
    def __init__(self, args):
        super(ShardedBART, self).__init__()
        self.args = args
        self.tokenizer = BartTokenizer.from_pretrained(args.checkpoint)
        self.model = BartForConditionalGeneration.from_pretrained(args.checkpoint)

        self.device_enc1 = self.device_enc2 = self.device_dec1 = self.device_dec2 = None
        self.loss_fct = nn.CrossEntropyLoss(ignore_index=self.tokenizer.pad_token_id)

    def shard(self):
        """ Shard the model to at most 4 gpus"""
        assert self.args.gpu > 0 and torch.cuda.is_available()
        gpu = min(self.args.gpu, 4)
        if gpu == 4:
            assert torch.cuda.device_count() >= 4
            self.device_enc1 = 'cuda:0'
            self.device_enc2 = 'cuda:1'
            self.device_dec1 = 'cuda:2'
            self.device_dec2 = 'cuda:3'
            self.cuda()
        elif gpu == 3:
            assert torch.cuda.device_count() >= 3
            self.device_enc1 = 'cuda:0'
            self.device_enc2 = 'cuda:1'
            self.device_dec1 = 'cuda:1'
            self.device_dec2 = 'cuda:2'
        elif gpu == 2:
            assert torch.cuda.device_count() >= 2
            self.device_enc1 = self.device_enc2 = 'cuda:0'
            self.device_dec1 = self.device_dec2 = 'cuda:1'
        else:
            self.device_enc1 = self.device_enc2 = self.device_dec1 = self.device_dec2 = 'cuda:0'

        # Model sharding
        self.encoder.to(self.device_enc1)
        self.decoder.to(self.device_dec1)

        # We further shard the model if needed.
        encoder_layer_num = len(self.encoder.layers)
        for i in range(encoder_layer_num):
            if i >= (encoder_layer_num // 2):
                self.encoder.layers[i].to(self.device_enc2)

        decoder_layer_num = len(self.decoder.layers)
        for i in range(decoder_layer_num):
            if i >= (decoder_layer_num // 2):
                self.decoder.layers[i].to(self.device_dec2)

        # For calculating lm logits
        self.model.final_logits_bias = move_device(
            self.model.final_logits_bias, self.device_dec2
        )

        self.model.model.shared = move_device(self.model.model.shared, self.device_dec2)
        torch.cuda.empty_cache()
        print(f'Sharded to {gpu} GPUs.')

    def encode(self, sentence, max_len):
        """ Encode text (up to max_length)
            Example output:
            {
                'input_ids': tensor([[   0,  713,   16, 1531,    2]]),
                'attention_mask': tensor([[1, 1, 1, 1, 1]])
            }
        """
        encoded = self.tokenizer([sentence], max_length=max_len,
                                 truncation=True, return_tensors='pt')
        return encoded

    def forward(
            self,
            input_ids=None,
            attention_mask=None,
            decoder_attention_mask=None,
            labels=None,
    ):

        decoder_input_ids = shift_tokens_right(
            labels, self.config.pad_token_id, self.config.decoder_start_token_id
        )

        # Go through encoder
        hidden_states = forward_encoder(
            self=self.encoder,
            input_ids=input_ids,
            attention_mask=attention_mask
        )

        # Go through decoder
        hidden_states = forward_decoder(
            self=self.decoder,
            input_ids=decoder_input_ids,
            attention_mask=decoder_attention_mask,
            encoder_hidden_states=hidden_states,
            encoder_attention_mask=attention_mask
        )

        lm_logits = self.model.lm_head(hidden_states) + self.model.final_logits_bias
        masked_lm_loss = self.loss_fct(lm_logits.view(-1, self.config.vocab_size),
                                       labels.view(-1).to(lm_logits.device))
        return masked_lm_loss

    @property
    def config(self):
        return self.model.config

    @property
    def encoder(self):
        return self.model.model.encoder

    @property
    def decoder(self):
        return self.model.model.decoder

    @property
    def generate(self):
        return self.model.generate


def forward_encoder(
        self,
        input_ids=None,
        attention_mask=None
):
    """ Here self is self.encoder"""
    # retrieve input_ids and inputs_embeds

    input_shape = input_ids.size()
    input_ids = input_ids.view(-1, input_shape[-1]).to(self.embed_tokens.weight.device)

    inputs_embeds = self.embed_tokens(input_ids) * self.embed_scale

    embed_pos = self.embed_positions(input_shape)

    hidden_states = inputs_embeds.to(self.layernorm_embedding.weight.device) \
                    + embed_pos.to(self.layernorm_embedding.weight.device)
    hidden_states = self.layernorm_embedding(hidden_states)
    hidden_states = F.dropout(hidden_states, p=self.dropout, training=self.training)

    # expand attention_mask
    if attention_mask is not None:
        # [bsz, seq_len] -> [bsz, 1, tgt_seq_len, src_seq_len]
        attention_mask = _expand_mask(attention_mask, inputs_embeds.dtype)

    for idx, encoder_layer in enumerate(self.layers):
        # add LayerDrop (see https://arxiv.org/abs/1909.11556 for description)
        dropout_probability = random.uniform(0, 1)
        if self.training and (dropout_probability < self.layerdrop):  # skip the layer
            pass
        else:
            layer_outputs = encoder_layer(
                hidden_states.to(encoder_layer.fc1.weight.device),
                attention_mask.to(encoder_layer.fc1.weight.device),
                layer_head_mask=None,
                output_attentions=False,
            )
            hidden_states = layer_outputs[0]
    return hidden_states


def forward_decoder(
        self,
        input_ids=None,
        attention_mask=None,
        encoder_hidden_states=None,
        encoder_attention_mask=None,
):
    """ Here self is self.decoder """

    # retrieve input_ids and inputs_embeds

    input_shape = input_ids.size()
    input_ids = input_ids.view(-1, input_shape[-1])
    inputs_embeds = self.embed_tokens(input_ids.to(self.embed_tokens.weight.device)) * self.embed_scale

    # past_key_values_length
    past_key_values_length = 0

    attention_mask = self._prepare_decoder_attention_mask(
        attention_mask.to(inputs_embeds.device), input_shape, inputs_embeds, past_key_values_length
    )

    # expand encoder attention mask
    if encoder_hidden_states is not None and encoder_attention_mask is not None:
        # [bsz, seq_len] -> [bsz, 1, tgt_seq_len, src_seq_len]
        encoder_attention_mask = _expand_mask(encoder_attention_mask, inputs_embeds.dtype, tgt_len=input_shape[-1])

    # embed positions
    positions = self.embed_positions(input_shape, past_key_values_length)

    hidden_states = inputs_embeds.to(self.layernorm_embedding.weight.device) \
                    + positions.to(self.layernorm_embedding.weight.device)
    hidden_states = self.layernorm_embedding(hidden_states)

    hidden_states = F.dropout(hidden_states, p=self.dropout, training=self.training)

    for idx, decoder_layer in enumerate(self.layers):
        # add LayerDrop (see https://arxiv.org/abs/1909.11556 for description)
        dropout_probability = random.uniform(0, 1)
        if self.training and (dropout_probability < self.layerdrop):
            continue
        curr_device = decoder_layer.fc1.weight.device
        layer_outputs = decoder_layer(
            hidden_states.to(curr_device),
            attention_mask=attention_mask.to(curr_device),
            encoder_hidden_states=encoder_hidden_states.to(curr_device),
            encoder_attention_mask=encoder_attention_mask.to(curr_device),
            layer_head_mask=None,
            cross_attn_layer_head_mask=None,
            past_key_value=None,
            output_attentions=False,
            use_cache=False,
        )
        hidden_states = layer_outputs[0]

    return hidden_states
