import { Meteor } from 'meteor/meteor';
import { Mongo } from 'meteor/mongo';
import { check, Match } from 'meteor/check';

export const Texts = new Mongo.Collection('texts');
export const Assignments = new Mongo.Collection('assignments');
export const Progress = new Mongo.Collection('progress');
export const Annotations = new Mongo.Collection('annotations');
export const Tutorials = new Mongo.Collection('tutorials');
export const Feedback = new Mongo.Collection('feedback');
export const ActionLog = new Mongo.Collection('actionlog');

if (Meteor.isServer) {
  // Create indices
  Annotations.rawCollection().createIndex({ _id: 1})
  Annotations.rawCollection().createIndex({ user: 1})
  Assignments.rawCollection().createIndex({ _id: 1})
  Assignments.rawCollection().createIndex({ group: 1})
  Progress.rawCollection().createIndex({ _id: 1})
  Texts.rawCollection().createIndex({ _id: 1 })
  Tutorials.rawCollection().createIndex({ _id: 1})

  Meteor.publish('texts-all', function textsPublication() {
    return Texts.find();
  });
  Meteor.publish('texts-single', function textsPublication(text_id) {
    return Texts.find({ _id: text_id });
  });
  Meteor.publish('texts-tutorials', function textsTutorialPublication() {
    return Texts.find({ _id: /^tutorial/ });
  });

  Meteor.publish('annotations-all', function annotationsPublication() {
    return Annotations.find();
  });
  Meteor.publish('annotations-user', function annotationsPublication(user_id) {
    return Annotations.find({ user: user_id });
  });
  Meteor.publish('annotations-single', function annotationsPublication(ann_id) {
    return Annotations.find({ _id: ann_id });
  });

  Meteor.publish('assignments-all', function assignmentsPublication() {
    return Assignments.find();
  });
  Meteor.publish('assignments-group', function assignmentsGroupPublication(group) {
    return Assignments.find({ group: group });
  });
  Meteor.publish('assignments-single', function assignmentsPublication(assign_id) {
    return Assignments.find({ _id: assign_id });
  });

  Meteor.publish('progress-all', function progressPublication() {
    return Progress.find();
  });

  Meteor.publish('tutorials-all', function tutorialsPublication() {
    return Tutorials.find();
  });
  Meteor.publish('tutorials-single', function tutorialsPublication(tut_id) {
    return Tutorials.find({ _id: tut_id });
  });
}

function checkMention(mention, nullAllowed=false) {
  if (nullAllowed && mention == null) {
  } else {
    check(mention, {start: Number, end: Number, label: String});
    const IsNotNegative = Match.Where((x) => {
      return x >= 0;
    });
    check(mention.start, IsNotNegative);
    check(mention.end, IsNotNegative);
  }
}

function checkCluster(cluster) {
  check(cluster._id, String);
  cluster.mentions.forEach(mention => checkMention(mention));
}

function checkLabels(labels) {
  labels.forEach(label => check(label, String));
}

function checkChecks(checks) {
  checks.forEach(c => {
    check(c.decision, String);
    checkMention(c.mention);
  });
}

Meteor.methods({
  'logToConsole'(msg) {
    console.log(msg);
  },
  'logError'(user_id, text) {
    check(user_id, String);
    check(text, String);

    Feedback.insert({
      user: user_id,
      text: text,
      confidence: 0,
      time: new Date(),
    });
    ActionLog.insert({
      user: user_id,
      action: "logError",
      text: text,
      time: new Date(),
    });
  },
  'feedback.submit'(user_id, text, confidence, assignment_id) {
    check(user_id, String);
    check(text, String);

    Feedback.insert({
      user: user_id,
      text: text,
      confidence: confidence,
      assignment: assignment_id,
      time: new Date(),
    });
    ActionLog.insert({
      user: user_id,
      action: "feedback.submit",
      assignment: assignment_id,
      time: new Date(),
    });
  },
  'tutorial.begin'(user_id, text_id, init, labels, checks, focus, tutorial_id) {
    check(user_id, String);
    check(text_id, String);
    check(tutorial_id, String);
    checkMention(focus, true);
    checkLabels(labels);
    checkChecks(checks);
    const id = text_id +"-"+ tutorial_id +"-"+ user_id;
    const tid = tutorial_id +"-"+ user_id;

    Annotations.upsert({ _id: id }, {
      _id: id,
      user: user_id,
      text_id: text_id,
      clusters: init,
      checks: checks,
      labels: labels,
      focus_mention: focus,
      createdAt: new Date(),
    });
    ActionLog.insert({
      user: user_id,
      action: "tutorial.begin/reset",
      tutorial: text_id,
      time: new Date(),
    });
    Progress.upsert({ _id: tid }, {
      _id: tid,
      tutorial: tutorial_id,
      user: user_id,
      assigned: new Date(),
      completed: false,
    });
  },
  'tutorial.complete'(user_id, tutorial_id) {
    check(user_id, String);
    check(tutorial_id, String);

    const id = tutorial_id +"-"+ user_id;
    Progress.update(
      {_id: id},
      {$set: {completed: new Date()}}
    );
    ActionLog.insert({
      user: user_id,
      action: "tutorial.complete",
      tutorial: tutorial_id,
      time: new Date(),
    });
  },
  'assignments.preview'(assignment_id, group_id) {
    check(assignment_id, String);
    check(group_id, String);

    ActionLog.insert({
      user: "NO_USER_PROVIDED",
      action: "assignments.preview",
      assignment: assignment_id,
      group: group_id,
      time: new Date(),
    });
  },
  'assignments.assign'(assignment_id, text_id, user_id, init_clusters, labels, checks, focus) {
    check(assignment_id, String);
    check(text_id, String);
    check(user_id, String);
    checkLabels(labels);
    checkMention(focus, true);
    checkChecks(checks);

    const id = assignment_id + "-"+ user_id;

    Annotations.insert({
      _id: id,
      text_id: text_id,
      user: user_id,
      clusters: init_clusters,
      checks: checks,
      labels: labels,
      focus_mention: focus,
      assignment: assignment_id,
      createdAt: new Date(),
    });
    Progress.insert({
      _id: id,
      assignment: assignment_id,
      user: user_id,
      assigned: new Date(),
      completed: false,
    });
    ActionLog.insert({
      user: user_id,
      action: "assignments.assign",
      assignment: assignment_id,
      time: new Date(),
    });
  },
  'assignments.complete'(assignment_id, user_id) {
    check(assignment_id, String);
    check(user_id, String);

    const id = assignment_id + "-"+ user_id;
    Progress.update(
      {_id: id},
      {$set: {completed: new Date()}}
    );
    ActionLog.insert({
      user: user_id,
      action: "assignments.complete",
      assignment: assignment_id,
      time: new Date(),
    });
  },
  'clusters.insert'(text_id, ann_id, cluster, user_id) {
    check(text_id, String);
    check(ann_id, String);
    checkCluster(cluster);

    Annotations.update(
      {_id: ann_id},
      {$push: {clusters: cluster}}
    );
    ActionLog.insert({
      user: user_id,
      action: 'clusters.insert',
      text: text_id,
      annotation: ann_id,
      cluster: cluster.mentions,
      time: new Date(),
    });
  },
  'clusters.remove'(text_id, ann_id, cluster_id, mentions, user_id) {
    check(text_id, String);
    check(ann_id, String);
    check(cluster_id, String);

    Annotations.update(
      {_id: ann_id},
      {$pull: {clusters: {_id: cluster_id}}}
    );
    ActionLog.insert({
      user: user_id,
      action: 'clusters.remove',
      text: text_id,
      annotation: ann_id,
      cluster: mentions,
      time: new Date(),
    });
  },
  'clusters.extend'(text_id, ann_id, cluster_id, start, end, user_id) {
    check(text_id, String);
    check(ann_id, String);
    check(cluster_id, String);
    check(start, Number);
    check(end, Number);

    const mention = {start: start, end: end};
    Annotations.update(
      {_id: ann_id, "clusters._id": cluster_id},
      {$push: {"clusters.$.mentions": mention}}
    );
    ActionLog.insert({
      user: user_id,
      action: 'clusters.extend',
      text: text_id,
      annotation: ann_id,
      time: new Date(),
    });
  },
  'clusters.contract'(text_id, ann_id, cluster_id, start, end, label, user_id) {
    check(text_id, String);
    check(ann_id, String);
    check(cluster_id, String);
    check(start, Number);
    check(end, Number);

    const mention = {start: start, end: end, label: label};
    Annotations.update(
      {_id: ann_id, "clusters._id": cluster_id},
      {$pull: {"clusters.$.mentions": mention}}
    );
    ActionLog.insert({
      user: user_id,
      action: 'clusters.contract',
      text: text_id,
      annotation: ann_id,
      time: new Date(),
    });
  },
  'clusters.merge'(text_id, ann_id, cluster_id0, cluster_id1, new_cluster, user_id) {
    check(text_id, String);
    check(ann_id, String);
    check(cluster_id0, String);
    check(cluster_id1, String);
    checkCluster(new_cluster);

    Annotations.update(
      {_id: ann_id},
      {$pull: {clusters: {_id: cluster_id0}}}
    );
    Annotations.update(
      {_id: ann_id},
      {$pull: {clusters: {_id: cluster_id1}}}
    );
    Annotations.update(
      {_id: ann_id},
      {$push: {clusters: new_cluster}}
    );
    ActionLog.insert({
      user: user_id,
      action: 'clusters.merge',
      text: text_id,
      annotation: ann_id,
      time: new Date(),
    });
  },
  'clusters.split'(text_id, ann_id, cluster_id, cluster0, cluster1, user_id) {
    check(text_id, String);
    check(ann_id, String);
    check(cluster_id, String);
    checkCluster(cluster0);
    checkCluster(cluster1);

    Annotations.update(
      {_id: ann_id},
      {$pull: {clusters: {_id: cluster_id}}}
    );
    Annotations.update(
      {_id: ann_id},
      {$push: {clusters: cluster0}}
    );
    Annotations.update(
      {_id: ann_id},
      {$push: {clusters: cluster1}}
    );
    ActionLog.insert({
      user: user_id,
      action: 'clusters.split',
      text: text_id,
      annotation: ann_id,
      time: new Date(),
    });
  },
  'mentions.check'(text_id, user_id, ann_id, mention, decision) {
    check(text_id, String);
    check(ann_id, String);
    check(user_id, String);
    check(decision, String);
    checkMention(mention);

    Annotations.update(
      {_id: ann_id,},
      {$push: {checks: {mention: mention, decision: decision}}}
    );
    ActionLog.insert({
      user: user_id,
      action: 'mentions.check',
      text: text_id,
      annotation: ann_id,
      mention: mention,
      decision: decision,
      time: new Date(),
    });
  },
  'mentions.uncheck'(text_id, user_id, ann_id, mention, decision) {
    check(text_id, String);
    check(ann_id, String);
    check(user_id, String);
    check(decision, String);
    checkMention(mention);

    Annotations.update(
      {_id: ann_id,},
      {$pull: {checks: {mention: mention, decision: decision}}}
    );
    ActionLog.insert({
      user: user_id,
      action: 'mentions.uncheck',
      text: text_id,
      annotation: ann_id,
      mention: mention,
      decision: decision,
      time: new Date(),
    });
  },
  'mentions.check.update'(text_id, user_id, ann_id, mention, decision) {
    check(text_id, String);
    check(ann_id, String);
    check(user_id, String);
    check(decision, String);
    checkMention(mention);

    Annotations.update(
      {_id: ann_id, checks: {$elemMatch: {
        "mention.start": mention.start,
        "mention.end": mention.end,
        "mention.label": mention.label}
      }},
      {$set: {"checks.$.decision": decision}}
    );
    ActionLog.insert({
      user: user_id,
      action: 'mentions.check.update',
      text: text_id,
      annotation: ann_id,
      mention: mention,
      decision: decision,
      time: new Date(),
    });
  },
});

