#include <memory>
#include <iostream>
#include <iomanip>
#include <cstring>
#include <fstream>
#include <sstream>
#include <unordered_map>

#include "include/basic.h"
#include "io/token_manager.h"

#include "baseline/baseline_run.h"
#include "data_driven/data_driven_run.h"

#include "io/command_line.h"
#include "io/utility.h"

int main(int argc, char* argv[]) {
    std::ios_base::sync_with_stdio(false);
    std::cin.tie(nullptr);
    std::cout << std::fixed << std::setprecision(4);

    srand(time(nullptr));

    initializeCommandLineVariables(argc, argv);
    auto& vm = commandLineVariables();

    int iterations = 10;
    std::string method(vm["method"].as< std::string >());
    std::string rules_file(vm["rules"].as< std::string >());
    std::string input_file(vm["input"].as< std::string >());
    std::string feature_file(vm["feature"].as< std::string >());

    if (vm.count("iterations"))
        iterations = vm["iterations"].as< int >();

    LOG_INFO(<< "Loading Literals ...");
    TokenManager::loadLiterals(vm["literals"].as< std::string >());
    LOG_INFO(<< "Loaded " << TokenManager::literalCount() << " Literals");

    LOG_INFO(<< "Loading States ...");
    TokenManager::loadStates(vm["states"].as< std::string >());
    LOG_INFO(<< "Loaded " << TokenManager::stateCount() << " States");

    std::unique_ptr< RunBase > runner;

    auto& model = vm["model"].as< std::string >();
    if (model == "baseline")
        runner = std::unique_ptr< RunBase >(new baseline::Run());
    else if (stringStartsWith(model, "data_driven"))
        runner = std::unique_ptr< RunBase >(new data_driven::Run());
    else {
        LOG_ERROR(<< "Unknown model !!!");
        std::exit(1);
    }

#ifdef DECODER_CHECK
    LOG_INFO(<< "Decoder check mode is enabled");
#endif

    if (method == "train") {
        for (int i = 0; i < iterations; ++i) {
            LOG_INFO(<< "Iteration " << i + 1);
            runner->train(input_file,
                          rules_file,
                          feature_file,
                          feature_file);
        }
    } else if (method == "goldTest") {
        runner->goldTest(input_file, rules_file, feature_file);
    } else if (method == "transduce") {
        if (!vm.count("output")) {
            LOG_ERROR(<< "No output option found !!");
            std::exit(1);
        }
        runner->transduce(input_file, vm["output"].as< std::string >(),
                          rules_file, feature_file, false /* with_detail*/);
    } else if (method == "transduceWithDetail") {
        if (!vm.count("output")) {
            LOG_ERROR(<< "No output option found !!");
            std::exit(1);
        }
        runner->transduce(input_file, vm["output"].as< std::string >(),
                          rules_file, feature_file, true /* with_detail*/);
    }
}
