import numpy as np
import pandas as pd


def emg_plot_signal(emg_signals, info=None, ax=None, label=None):
    if not isinstance(emg_signals, pd.DataFrame):
        raise ValueError(
            "NeuroKit error: The `emg_signals` argument must"
            " be the DataFrame returned by `emg_process()`."
        )

    sampling_rate = info["sampling_rate"]
    x_axis = np.linspace(
        0, emg_signals.shape[0] / info["sampling_rate"], emg_signals.shape[0]
    )

    if sampling_rate is not None:
        ax.set_xlabel("Time (seconds)")
    elif sampling_rate is None:
        ax.set_xlabel("Samples")

    # Plot cleaned and raw EMG.
    ax.set_title("Raw and Cleaned Signal")
    # ax.plot(x_axis, emg_signals["EMG_Raw"], color="#B0BEC5", label="Raw", zorder=1)
    ax.plot(
        x_axis,
        emg_signals["EMG_Clean"],
        # color="#FFC107",
        label=f"Cleaned {label}",
        zorder=1,
        linewidth=1.5,
    )
    # ax.legend(loc="upper right")


def emg_plot_act(emg_signals, info=None, ax=None, label=None):
    # Sanity-check input.
    if not isinstance(emg_signals, pd.DataFrame):
        raise ValueError(
            "NeuroKit error: The `emg_signals` argument must"
            " be the DataFrame returned by `emg_process()`."
        )

    # Determine what to display on the x-axis, mark activity.
    x_axis = np.linspace(
        0, emg_signals.shape[0] / info["sampling_rate"], emg_signals.shape[0]
    )

    sampling_rate = info["sampling_rate"]
    if sampling_rate is not None:
        ax.set_xlabel("Time (seconds)")
    elif sampling_rate is None:
        ax.set_xlabel("Samples")

    # Plot Amplitude.
    ax.set_title("Muscle Activation")
    ax.plot(
        x_axis,
        emg_signals["EMG_Amplitude"],
        # color="#FF9800",
        # label="Amplitude",
        label=label,
        linewidth=1.5,
    )

    # Shade activity regions.
    # activity_signal = _emg_plot_activity(emg_signals, onsets, offsets)
    # ax.fill_between(
    #     x_axis,
    #     emg_signals["EMG_Amplitude"],
    #     activity_signal,
    #     where=emg_signals["EMG_Amplitude"] > activity_signal,
    #     # color="#f7c568",
    #     alpha=0.5,
    #     # label=None,
    #     label=label,
    # )

    # # Mark onsets and offsets.
    # ax.scatter(
    #     x_axis[onsets],
    #     emg_signals["EMG_Amplitude"][onsets],
    #     # color="#f03e65",
    #     # label=None,
    #     label=label,
    #     zorder=3,
    # )
    # ax.scatter(
    #     x_axis[offsets],
    #     emg_signals["EMG_Amplitude"][offsets],
    #     # color="#f03e65",
    #     # label=None,
    #     label=label,
    #     zorder=3,
    # )

    # if sampling_rate is not None:
    #     onsets = onsets / sampling_rate
    #     offsets = offsets / sampling_rate

    # for i, j in zip(list(onsets), list(offsets)):
    #     ax.axvline(i, color="#4a4a4a", linestyle="--", label=None, zorder=2)
    #     ax.axvline(j, color="#4a4a4a", linestyle="--", label=None, zorder=2)
    # ax.legend(loc="upper right")


def _emg_plot_activity(emg_signals, onsets, offsets):
    activity_signal = pd.Series(np.full(len(emg_signals), np.nan))
    activity_signal[onsets] = emg_signals["EMG_Amplitude"][onsets].values
    activity_signal[offsets] = emg_signals["EMG_Amplitude"][offsets].values
    activity_signal = activity_signal.bfill()

    if np.any(activity_signal.isna()):
        index = np.min(np.where(activity_signal.isna())) - 1
        value_to_fill = activity_signal[index]  # pylint: disable=unsubscriptable-object
        activity_signal = activity_signal.fillna(value_to_fill)

    return activity_signal
