package handlers;

import utils.Utils;

public class NeedlemanWunschHandler 
{
	// alignment matrix
	int[][] d;
	// strings for which the alignment should be determined
	String a, b;
	// lengths of the two strings 
	int n, m;

	/**
	 * initialize handler for Needleman-Wunsch alignment
	 * @param _a first input string
	 * @param _b second input string
	 */
	public NeedlemanWunschHandler(String _a, String _b) 
	{
		a = _a;
		b = _b;

		n = a.length();
		m = b.length();
	}
	
	/**
	 * compute alignment matrix for Needleman-Wunsch algorithm
	 */
	public void computeMatrix()
	{
		d = new int[n + 1][m + 1];

		d[0][0] = 0;

		for (int i = 1; i <= n; i++) 
			d[i][0] = d[i - 1][0] + w(a.charAt(i - 1), '\0');

		for (int j = 1; j <= m; j++)
			d[0][j] = d[0][j - 1] + w('\0', b.charAt(j - 1));

		for (int i = 1; i <= n; i++) 
		{
			for (int j = 1; j <= m; j++) 
			{
				d[i][j] = min(d[i - 1][j - 1] + w(a.charAt(i - 1), b.charAt(j - 1)),
						min(d[i - 1][j] + w(a.charAt(i - 1), '\0'), d[i][j - 1]
								+ w('\0', b.charAt(j - 1))));
			}
		}
	}

	/**
	 * determine Needleman-Wunsch alignment of strings a and b
	 * @return orthographic alignment
	 * for example, for words 'exhaustiv' and 'esaustivo' the output is 
	 * 'exhaustiv-_es-austivo'
	 */
	public String computeAlignment() 
	{
		computeMatrix();
		
		String alignedA = "";
		String alignedB = "";

		int i = n, j = m;

		while (i > 0 || j > 0) 
		{
			if (i > 0
					&& d[i][j] == d[i - 1][j] + w(a.charAt(i - 1), '\0')) 
			{
				alignedA = a.charAt(--i) + alignedA;
				alignedB = "-" + alignedB;
			} 
			else if (j > 0
					&& d[i][j] == d[i][j - 1] + w('\0', b.charAt(j - 1))) 
			{
				alignedA = "-" + alignedA;
				alignedB = b.charAt(--j) + alignedB;
			}
			else if (i > 0
					&& j > 0
					&& d[i][j] == d[i - 1][j - 1]
							+ w(a.charAt(i - 1), b.charAt(j - 1))) 
			{
				alignedA = a.charAt(--i) + alignedA;
				alignedB = b.charAt(--j) + alignedB;
			} 
		}

		return alignedA + "_" + alignedB;
	}

	/**
	 * weight of character transition from x to y
	 * @param x source character
	 * @param y target character
	 * @return weight of x->y transition
	 */
	private int w(char x, char y) 
	{
		if (x == y) 
			return 0;
		
		if (Utils.removeDiacritics(x + "").equals(Utils.removeDiacritics(y + "")))
			return 0;
		
		return 1;
	}

	/**
	 * get mininum of u and v
	 * @param u integer
	 * @param v integer
	 * @return minimum of u and v
	 */
	private int min(int u, int v) 
	{
		if (u <= v)
			return u;

		return v;
	}

	/**
	 * print alignment matrix
	 */
	public void printMatrix() 
	{
		for (int i = 0; i <= n; i++) 
		{
			for (int j = 0; j <= m; j++) 
				System.out.print(d[i][j] + " ");

			System.out.println();
		}
	}
}