import argparse
import difflib
import os

import numpy as np
import pandas as pd
import torch

from scipy.stats import ttest_ind
from scipy.stats import bootstrap
from tqdm import tqdm
from transformers import AutoModelForMaskedLM, AutoTokenizer

### I/O Functions
########################


def prompt_parameters():
    parser = argparse.ArgumentParser()
    parser.add_argument("--input", type=str, required=True)
    parser.add_argument("--out_dir", type=str, required=True)
    parser.add_argument(
        "--model",
        type=str,
        required=True,
        choices=[
            "bert-multi",
            "xlm-roberta",
            "xlm-roberta-L",
            "bert",
            "roberta",
            "albert",
        ],
    )
    parser.add_argument("--perturb", default=False, action="store_true")
    args = parser.parse_args()
    return args


def load_model(args):
    if args.model == "bert-multi":
        pretrained_weights = "bert-base-multilingual-cased"
    elif args.model == "xlm-roberta":
        pretrained_weights = "xlm-roberta-base"
    elif args.model == "xlm-roberta-L":
        pretrained_weights = "xlm-roberta-large"
    elif args.model == "bert":
        pretrained_weights = "bert-base-uncased"
    elif args.model == "roberta":
        pretrained_weights = "roberta-large"
    elif args.model == "albert":
        pretrained_weights = "albert-xxlarge-v2"
    model = AutoModelForMaskedLM.from_pretrained(
        pretrained_weights, output_hidden_states=True, output_attentions=True
    )
    tokenizer = AutoTokenizer.from_pretrained(pretrained_weights)

    model = model.eval()
    if torch.cuda.is_available():
        model.to("cuda")
    return tokenizer, model


def read_sentences(input_file):
    usecols = [
        "ID",
        "A_x",
        "B_x",
        "stereo_antistereo",
    ]
    new_col_names = [
        "ID",
        "sent_more",
        "sent_less",
        "stereo_antistereo",
    ]
    df_pairs = pd.read_csv(input_file, delimiter=",", encoding="utf-8", usecols=usecols)
    df_pairs.columns = new_col_names
    df_pairs = df_pairs[df_pairs["sent_more"].notna()]
    return df_pairs


def get_out_file_name(args):
    input_file_name = os.path.basename(args.input)
    input_file_name_pieces = input_file_name.split("_")
    input_file_name_pieces.insert(-1, args.model)
    output_file_name = "_".join(input_file_name_pieces)
    output_file_path = os.path.join(args.out_dir, output_file_name)
    return output_file_path


### Masking and Model Runs
########################

"""With slight modifications from: 
Nikita Nangia, Clara Vania, Rasika Bhalerao, and Samuel R. Bowman. 
2020. CrowS-Pairs: A Challenge Dataset for Measuring Social Biases in Masked Language Models. 
arXiv:2010.00133 [cs], September. arXiv: 2010.00133."""


def get_span(seq1, seq2):

    seq1 = [str(x) for x in seq1.tolist()]
    seq2 = [str(x) for x in seq2.tolist()]

    matcher = difflib.SequenceMatcher(None, seq1, seq2)
    template1, template2 = [], []
    for op in matcher.get_opcodes():
        if op[0] == "equal":
            template1 += [x for x in range(op[1], op[2], 1)]
            template2 += [x for x in range(op[3], op[4], 1)]

    # Remove ids added by tokenizer
    template1 = template1[1:-1]
    template2 = template2[1:-1]
    return template1, template2


"""Used a setup similar to:
Masahiro Kaneko and Danushka Bollegala. 
2021. Unmasking the Mask -- Evaluating Social Biases in Masked Language Models. 
arXiv:2104.07496 [cs], April. arXiv: 2104.07496.
"""


def get_model_predictions(
    model, token_ids, mask_template, mask_id, softmax, log_softmax
):
    masked_token_ids = token_ids.repeat(len(mask_template), 1)
    masked_token_ids[range(masked_token_ids.size(0)), mask_template] = mask_id
    model_predictions = model(masked_token_ids)
    model_predictions = model_predictions.logits
    mask_probabilities = softmax(
        model_predictions[range(model_predictions.size(0)), mask_template, :]
    )
    mask_log_probabilities = log_softmax(
        model_predictions[range(model_predictions.size(0)), mask_template, :]
    )
    return mask_probabilities, mask_log_probabilities


### Calculate CPS Scores
########################


def calc_cps_score(
    sent_more_mask_log_probs,
    sent_less_mask_log_probs,
    sent_more_ids,
    sent_less_ids,
    sent_more_template,
    sent_less_template,
):
    common_token_ids = sent_more_ids.view(-1)[sent_more_template]

    sent_more_gold_mask_predictions = sent_more_mask_log_probs[
        range(sent_more_mask_log_probs.size(0)), common_token_ids
    ]
    sent_less_gold_mask_predictions = sent_less_mask_log_probs[
        range(sent_less_mask_log_probs.size(0)), common_token_ids
    ]
    sent_more_score = sum(sent_more_gold_mask_predictions).item()
    sent_less_score = sum(sent_less_gold_mask_predictions).item()

    if sent_more_score > sent_less_score:
        return 1, sent_more_score, sent_less_score
    else:
        return 0, sent_more_score, sent_less_score


### Calculate JSD token predictions
########################


def calc_analytic_jsd(prob_at_gold):
    analytic_jsd_at_gold = torch.sqrt(
        (
            prob_at_gold * torch.log2(prob_at_gold)
            - (prob_at_gold + 1) * torch.log2(prob_at_gold + 1)
            + 2
        )
        / 2.0
    )
    return analytic_jsd_at_gold


def calc_analytic_jsd_dist_from_gold(
    sent_more_predictions,
    sent_less_predictions,
    sent_more_ids,
    sent_less_ids,
    sent_more_template,
    sent_less_template,
):
    common_token_ids = sent_more_ids.view(-1)[sent_more_template]
    sent_more_gold_token_pred = sent_more_predictions[
        range(sent_more_predictions.size(0)), common_token_ids
    ]
    sent_less_gold_token_pred = sent_less_predictions[
        range(sent_less_predictions.size(0)), common_token_ids
    ]
    sent_more_analytic_gold_jsd = calc_analytic_jsd(sent_more_gold_token_pred)
    sent_less_analytic_gold_jsd = calc_analytic_jsd(sent_less_gold_token_pred)
    return sent_more_analytic_gold_jsd, sent_less_analytic_gold_jsd


### S_JSD scores
########################


def calc_s_jsd_token_score(sent_more_gold_jsd, sent_less_gold_jsd):
    s_jsd_token_scores = sent_more_gold_jsd - sent_less_gold_jsd
    return s_jsd_token_scores


def calc_avg_s_jsd(sent_more_gold_jsd, sent_less_gold_jsd):
    s_jsd_token_scores = calc_s_jsd_token_score(sent_more_gold_jsd, sent_less_gold_jsd)
    sentence_s_jsd_score = calc_avg_sentence_score(s_jsd_token_scores)
    return sentence_s_jsd_score


def calc_jsd_binary_score(sent_more_gold_jsd, sent_less_gold_jsd):
    sent_more_gold_jsd_sum = sum(sent_more_gold_jsd)
    sent_less_gold_jsd_sum = sum(sent_less_gold_jsd)
    if sent_more_gold_jsd_sum < sent_less_gold_jsd_sum:
        return 1
    else:
        return 0


### Sentence-level averaging
########################


def calc_avg_sentence_score(jsd_distances):
    jsd_score = torch.mean(jsd_distances).item()
    return jsd_score


### Statistical Tests
########################


def calc_bootstraped_std_err(list_of_values_to_average):
    standard_error = bootstrap(
        (list_of_values_to_average,), np.mean, confidence_level=0.95
    ).standard_error
    return standard_error


def get_number_of_correct_token_predictions(mask_log_probs, token_ids, token_template):
    stacked_token_ids = token_ids[0][token_template].view(-1, 1)
    sorted_indices = torch.sort(mask_log_probs, dim=1, descending=True)[1]
    correct_hits_tensor = torch.where(sorted_indices == stacked_token_ids, 1, 0)[:, 0]
    correct_hits_count = correct_hits_tensor.sum()
    return correct_hits_count


### Printing
########################


def print_score_and_sdv(sentence_scores_list):
    avg_score = np.average(sentence_scores_list)
    boot_strapped_sdv = calc_bootstraped_std_err(sentence_scores_list)
    print("Avg.: {:.8f} Boot-strapped sdv.: {}".format(avg_score, boot_strapped_sdv))


### Main
########################


if __name__ == "__main__":

    args = prompt_parameters()
    df_pairs = read_sentences(args.input)
    tokenizer, model = load_model(args)
    mask_id = tokenizer.mask_token_id
    softmax = torch.nn.Softmax(dim=1)
    log_softmax = torch.nn.LogSoftmax(dim=1)

    sentence_data = []

    for index, row in tqdm(df_pairs.iterrows(), total=df_pairs.shape[0]):

        if args.perturb:
            row["sent_more"] = row["sent_more"][:-1]
            row["sent_less"] = row["sent_less"][:-1]

        sent_more_ids = tokenizer.encode(row["sent_more"], return_tensors="pt")
        sent_less_ids = tokenizer.encode(row["sent_less"], return_tensors="pt")

        with torch.no_grad():
            sent_more_template, sent_less_template = get_span(
                sent_more_ids[0], sent_less_ids[0]
            )
            sent_more_mask_probs, sent_more_mask_log_probs = get_model_predictions(
                model, sent_more_ids, sent_more_template, mask_id, softmax, log_softmax
            )
            sent_less_mask_probs, sent_less_mask_log_probs = get_model_predictions(
                model, sent_less_ids, sent_less_template, mask_id, softmax, log_softmax
            )

            # Evaluation
            (
                sent_more_gold_jsd,
                sent_less_gold_jsd,
            ) = calc_analytic_jsd_dist_from_gold(
                sent_more_mask_probs,
                sent_less_mask_probs,
                sent_more_ids,
                sent_less_ids,
                sent_more_template,
                sent_less_template,
            )

            sentence_s_jsd_score = calc_avg_s_jsd(
                sent_more_gold_jsd, sent_less_gold_jsd
            )

            jsd_binary_score = calc_jsd_binary_score(
                sent_more_gold_jsd, sent_less_gold_jsd
            )

            # CPS reproduction
            cps_score, sent_more_cps_score, sent_less_cps_score = calc_cps_score(
                sent_more_mask_log_probs,
                sent_less_mask_log_probs,
                sent_more_ids,
                sent_less_ids,
                sent_more_template,
                sent_less_template,
            )
            # Recording
            matching_tokens = tokenizer.convert_ids_to_tokens(
                sent_more_ids[0][sent_more_template]
            )

            sent_more_token_pred_hits = get_number_of_correct_token_predictions(
                sent_more_mask_log_probs, sent_more_ids, sent_more_template
            )

            sent_less_token_pred_hits = get_number_of_correct_token_predictions(
                sent_less_mask_log_probs, sent_less_ids, sent_less_template
            )

            sentence_data.append(
                {
                    "ID": row["ID"],
                    "sent_more": row["sent_more"],
                    "sent_less": row["sent_less"],
                    "stereo_antistereo": row["stereo_antistereo"],
                    "matching_tokens": matching_tokens,
                    "number_of_matching_tokens": len(matching_tokens),
                    "sent_more_token_pred_hits": sent_more_token_pred_hits,
                    "sent_less_token_pred_hits": sent_less_token_pred_hits,
                    "cps_score": cps_score,
                    "cps_score_delta": (sent_more_cps_score - sent_less_cps_score),
                    "sentence_s_jsd_score": sentence_s_jsd_score,
                    "jsd_binary_score": jsd_binary_score,
                }
            )

    # Output
    df_scores = pd.DataFrame(sentence_data)
    out_file_name = get_out_file_name(args)
    df_scores.to_csv(
        out_file_name, sep=",", encoding="utf-8", index=False,
    )

    print("=" * 50)
    print("Input: {}".format(args.input))
    print("Model: {}".format(args.model))
    print("Perturbation: {}".format(args.perturb))
    print("*" * 25)

    sum_maching_tokens = sum([d["number_of_matching_tokens"] for d in sentence_data])
    print("Number of matching tokens: {}".format(sum_maching_tokens))

    sent_more_token_pred_hits = sum(
        [d["sent_more_token_pred_hits"] for d in sentence_data]
    )
    print(
        "Correct token predictions for sent_more: {}".format(sent_more_token_pred_hits)
    )

    sent_less_token_pred_hits = sum(
        [d["sent_less_token_pred_hits"] for d in sentence_data]
    )
    print(
        "Correct token predictions for sent_less: {}".format(sent_less_token_pred_hits)
    )

    print("CPS score [0,100]:")
    cps_score_list = [100 * d["cps_score"] for d in sentence_data]
    print_score_and_sdv(cps_score_list)

    print("Avg. S_JSD score [-1,1]:")
    s_jsd_score_list = [d["sentence_s_jsd_score"] for d in sentence_data]
    print_score_and_sdv(s_jsd_score_list)

    print("Avg. JSD binary score [0,100]")
    jsd_binary_score_list = [100 * d["jsd_binary_score"] for d in sentence_data]
    print_score_and_sdv(jsd_binary_score_list)

