# -*- coding: utf-8 -*-

import argparse
import os
import pickle
import shutil
import sys
import tempfile
from importlib import import_module
from io import BytesIO
from zipfile import ZipFile


def magic_load_internal(data_bytes, working_dir, packer=None, use_shell=False):
    header = pickle.load(open(os.path.join(working_dir, '__HEAD__'), 'rb'))

    if packer is None:
        packer = import_module(header['packer'])
    packer.install_given_packages(header['packages'])

    this_module = import_module('__this_module__')
    if not use_shell:
        entry_point = getattr(this_module, header['entry_point'])
        entry_point(data_bytes, entry_class=header['entry_class'])
    else:
        try:
            import IPython
            IPython.embed()
        except ModuleNotFoundError:
            import code
            code.interact(local=locals())
    return 0


def magic_load_main(mode, packed_file, code_bytes, data_bytes, directory=None):
    use_shell = (mode == 'shell')

    packed_dir = tempfile.mktemp(os.path.basename(packed_file))
    if os.path.exists(packed_dir):
        print(f'{packed_dir} already exists, please remove it first')
        return 1

    try:
        ZipFile(BytesIO(code_bytes)).extractall(packed_dir)
        sys.path.insert(0, os.path.abspath(packed_dir))

        if mode == 'main' or mode == 'shell':
            return magic_load_internal(data_bytes, working_dir=packed_dir, use_shell=use_shell)
        elif mode == 'extract':
            if directory is None:
                print(f'Usage: {sys.argv[0]} extract -C <directory>')
                return 1
            if os.path.exists(directory):
                print(f'"{directory}" already exists, please remove it first')
                return 1
            shutil.move(packed_dir, directory)
            with open(os.path.join(directory, '__MODEL__'), 'wb') as fp:
                fp.write(data_bytes)
    finally:
        if os.path.exists(packed_dir):
            shutil.rmtree(packed_dir)

    return 0


def split_packed_file(script_path, code_length, data_length, fp=None):
    try:
        shell_length = os.path.getsize(script_path) - code_length - data_length
        if fp is None:
            fp = open(script_path, 'rb')
        fp.seek(shell_length, 0)
        code_bytes = fp.read(code_length)
        data_bytes = fp.read()
        fp.close()
        return script_path, code_bytes, data_bytes
    except Exception as err:
        print('Fail to read packed file:', type(err).__name__, err, file=sys.stderr)


def main(argv=None):
    if argv is None:
        argv = sys.argv

    is_self_extracting = os.getenv('RUN_IN_PACKED_MODE') is not None

    parser = argparse.ArgumentParser('magic_load.py', add_help=False)
    parser.add_argument('mode', default='main', type=str,
                        choices=['main', 'extract', 'shell', 'help'],
                        help='main: run entry point of this packed file;\n'
                             'extract: extract internal codes of this packed_file;\n'
                             'shell: start REPL in this environment.\n'
                             'help: show this help message and exit')
    parser.add_argument('--directory', '-C', default=None, help='destination of extraction')

    if is_self_extracting:  # run packed model
        cmd_args = argv[4:]
        packed_file = argv[1]
        code_length = int(argv[2])
        data_length = int(argv[3])
        argv[0] = packed_file
    else:
        cmd_args = argv[1:]
        parser.add_argument('--packed-file', help='Packed file you want to use', required=True)
        packed_file = None

    options, cmd_args = parser.parse_known_args(cmd_args)
    if options.mode == 'help':
        parser.print_help()
        sys.exit(0)

    if packed_file is None:
        packed_file = options.packed_file

        with open(packed_file, 'rb') as fp:
            fp.readline()
            code_length = int(fp.readline().decode().split('=', 1)[1])
            data_length = int(fp.readline().decode().split('=', 1)[1])

    sys.argv = [argv[0]] + cmd_args  # directly modify sys.argv

    packed_parts = split_packed_file(packed_file, code_length, data_length)
    sys.exit(magic_load_main(options.mode, *packed_parts, directory=options.directory))


if __name__ == '__main__':
    main()
