import re

import pandas as pd

from .final_feedback import generate_feedback
from .icebreaker import is_ice_breaker
from .openai_utils import comments_summarizer, join_hist, message_reeditor
from .price_extractor import detected_offer_chatgpt
from .rationales import (edit_rationale, edit_rationale_first_offer,
                         is_rationale, is_rationale_first_offer)
from .strategic_closing import edit_strategic_closing, is_strategic_closing
from .strong_offer import (edit_strong_counteroffer, edit_strong_offer_first,
                           edit_strong_offer_second, is_strong_counteroffer,
                           is_strong_offer_first, is_strong_offer_second)


class Negotiation:

    def __init__(
        self,
        history,
        buyer_targetPrice,
        buyer_reservationPrice,
        seller_defaultFirstOffer,
    ) -> None:

        self.history = history
        self.buyer_targetPrice = buyer_targetPrice
        self.buyer_reservationPrice = buyer_reservationPrice
        self.first_offer_happend = False
        self.seller_made_an_offer = False
        self.current_buyer_offer = None
        self.current_seller_offer = seller_defaultFirstOffer
        self.buyer_nb_offer = 0
        self.deal = False


def join_comments(list_of_comments):
    concatenated_comments = ""
    for num, comment in enumerate(list_of_comments):
        concatenated_comments += f"comment {num + 1}: " + comment + "\n"
    return concatenated_comments


def generate_comments_summary(list_of_comments, message):

    prompt = f"""  We are in the context of a negotiation.
    Different teachers gave comments to the buyer:
    Your task is to fusion all the comments and advices into one that would englobe all of them.
    The text has to be well written and logic. You have to be friendly with the user, and use the conditional temp.
    If there is only one comment, then just copy paste it.
    Stay precise, do not add any extra information.
    
      #EXAMPLE1 :
      - MESSAGE:
        "Seems a little steep, steep for me. You know, I can do something in the, you know, $12,000 range would really be, you know, near the top of the end of my budget. Do you have any flexibility there? You know, anything we can do to, you know, work on that price?"

      -COMMENTS:
            comment 1: "Negotiation research finds a benefit to speaking your opening offer first. It can “anchor” the other person’s judgment of the price range, setting the stage for a more favorable outcome."
            comment 2: "Considering your target price of $10000, a strong first offer would ideally be below $9000. This approach helps to keep your target price near the midpoint of the range under discussion."
  
      - ANSWER: Negotiation research finds a benefit to speaking your opening offer first. It can “anchor” the other person’s judgment of the price range, setting the stage for a more favorable outcome.
            Moreover given the context, when you are setting the price to $12,000, you're essentially revealing your upper limit. However, to keep the negotiation flexible and within your budget, it's wise to aim for an initial offer below $9,000. This gives you room to maneuver and respond to counteroffers while ensuring that the final price stays within your financial comfort zone
            
        #EXAMPLE2:
        
        -MESSAGE:
            Hi, I'm looking for probably a Honda Accord with reasonable mileage around maybe $15000. Do you have anything like that?"

        -COMMENTS:
            comment 1: "Begin your negotiation conversation with some brief social conversation before delving into the economic issues. Show esteem for the other person (your counterpart) by praising what they are selling or asking about their day. “Breaking the ice” in some way through initial personal conversation creates rapport, which tends to increase openness and cooperativeness.
            comment 2: "Negotiation research finds that opening offers are most effective when accompanied by a rationale in terms of some objective reference point, such as an expert’s valuation of the object under negotiation or market value indicated by past sales prices."
        -ANSWER: Begin your negotiation conversation with some brief social conversation before delving into the economic issues. Show esteem for the other person (your counterpart) by praising what they are selling or asking about their day. “Breaking the ice” in some way through initial personal conversation creates rapport, which tends to increase openness and cooperativeness.
            Also, negotiation research finds that opening offers are most effective when accompanied by a rationale in terms of some objective reference point, such as an expert’s valuation of the object under negotiation or market value indicated by past sales prices.
        
        -MESSAGE:
            {message}
        - COMMENTS:
            {list_of_comments}
        - ANSWER: """
    edit = comments_summarizer(
        messages=[
            {"role": "system", "content": prompt},
        ],
    )
    return edit


def generate_message_reedition(list_of_comments, message):

    prompt = f"""  We are in the context of a negotiation.
    Different teachers gave comments to the buyer:
    Your task is to propose an alternative message the buyer could have sent that would match all the comments given by teachers.

    For example if a comment is saying that the buyer should open the conversation with an ice breaker, then propose an icebreaker.
    If a comment is saying that they should add rationales to their offers, then rewrite the offer and add a few rationales to it.
    You have to put yourself in the buyer's position. Assume that you are talking to the seller.
    
      #EXAMPLE1 :
      - MESSAGE:
        "Seems a little steep, steep for me. You know, I can do something in the, you know, $12,000 range would really be, you know, near the top of the end of my budget. Do you have any flexibility there? You know, anything we can do to, you know, work on that price?"

      -COMMENTS:
        "comment 1: "Negotiation research finds a benefit to speaking your opening offer first. It can “anchor” the other person’s judgment of the price range, setting the stage for a more favorable outcome."
        comment 2: "Considering your target price of $10000, a strong first offer would ideally be below $9000. This approach helps to keep your target price near the midpoint of the range under discussion."
  
      - ANSWER: "The price seems a little steep for me. I can work with something in the $9,000 range, which is near the top end of my budget. I want to ensure that we can reach a mutually beneficial agreement. Is there any flexibility on the price from your end?"

        #EXAMPLE2:
        -MESSAGE:
            "Hi, I'm looking for probably a Honda Accord with reasonable mileage around maybe $15000. Do you have anything like that?"

        -COMMENTS:
            "comment 1: "Begin your negotiation conversation with some brief social conversation before delving into the economic issues. Show esteem for the other person (your counterpart) by praising what they are selling or asking about their day. “Breaking the ice” in some way through initial personal conversation creates rapport, which tends to increase openness and cooperativeness.
            comment 2: "Negotiation research finds that opening offers are most effective when accompanied by a rationale in terms of some objective reference point, such as an expert’s valuation of the object under negotiation or market value indicated by past sales prices."
        -ANSWER: "Hey! It has been a long time are you doing ?"
         
        #YOUR TURN TO DO IT NOW
        -MESSAGE:
            {message}
        - COMMENTS:
            {list_of_comments}
        - ANSWER:"""
    edit = message_reeditor(
        messages=[
            {"role": "system", "content": prompt},
        ],
    )
    return edit


def turnwise_error_detection(history, reservation_price, target_price):

    nego = Negotiation(
        history=history,
        buyer_reservationPrice=reservation_price,
        buyer_targetPrice=target_price,
        seller_defaultFirstOffer=20000,
    )
    processed_turns = []
    flagged_errors = []
    first_offer_error = False
    hardball_error = False
    strategic_closing = True

    for idx in range(len(nego.history)):

        prev_conv = nego.history[:idx]
        message = nego.history[idx]
        turn = {"id": message["id"], "role": message["role"], "feedback": None}
        edit = []
        comment = None

        labels = {
            "breakingTheIce": None,
            "first_offer": None,
            "strongFirstOffer": None,
            "strongCounterOffer": None,
            "rationale": None,
            "strategicClosing": None,
            "isHardball": None,
        }

        if message["role"] == "Seller":
            # PRICE DETECTION
            price_detected = detected_offer_chatgpt(
                history=prev_conv,
                message=message,
                delusional_offer=nego.buyer_targetPrice / 100,
            )

            if price_detected != -1:
                if price_detected != nego.current_buyer_offer:
                    nego.current_seller_offer = price_detected

                # FIRST OFFER ERROR FOR THE NEXT TURN OF THE BUYER

                if not nego.first_offer_happend:
                    nego.first_offer_happend = True
                    first_offer_error = True
                    if price_detected > 2 * nego.buyer_targetPrice:
                        hardball_error = True
                        labels["isHardball"] = True
                    else:
                        labels["isHardball"] = False

            turn["current_offer"] = nego.current_seller_offer

        if message["role"] == "Buyer":
            # PRICE DETECTION

            price_detected = detected_offer_chatgpt(
                history=prev_conv,
                message=message,
                nb_offers=nego.buyer_nb_offer,
                delusional_offer=nego.buyer_targetPrice / 100,
            )

            # IF FIRST OFFER MADE BY THE SELLER
            if first_offer_error:
                labels["first_offer"] = False
                # flagged_errors.append('Did not make the first offer')
                first_offer_error = False
                edit.append(
                    """Negotiation research finds a benefit to speaking your opening offer first. It can “anchor” the other person’s judgment of the price range, setting the stage for a more favorable outcome."""
                )

            # HARDBALLS
            if hardball_error and price_detected > 0:
                edit.append(
                    "The Seller clearly offered an unreasonable price here. You should have refused this offer and not treated it instead of counteroffering it."
                )
                hardball_error = False
            # ICEBREAKER
            if idx <= 1:
                icebreaker = "True" == is_ice_breaker(message["content"])
                labels["breakingTheIce"] = icebreaker
                if not icebreaker:
                    edit.append(
                        "Begin your negotiation conversation with some brief social conversation before delving into the economic issues. Show esteem for the other person (your counterpart) by praising what they are selling or asking about their day. “Breaking the ice” in some way through initial personal conversation creates rapport, which tends to increase openness and cooperativeness."
                    )

            # STRATEGIC CLOSING
            if idx >= len(nego.history) - 4:

                if idx >= len(nego.history) - 2:
                    if not strategic_closing or not (
                        "True" == is_strategic_closing(message["content"])
                    ):
                        passage = join_hist(nego.history[-2:])
                        edit.append(edit_strategic_closing(passage))

                strategic_closing = "True" == is_strategic_closing(message["content"])
                labels["strategicClosing"] = strategic_closing

            if price_detected != -1:
                # RATIONALE:
                if idx < 2:
                    frame = nego.history[idx : idx + 2]

                if len(nego.history) - (idx + 1) <= 2:
                    frame = nego.history[idx - 2 : idx + 1]
                else:
                    frame = nego.history[idx - 2 : idx + 2]

                passage = join_hist(frame)

                if nego.buyer_nb_offer == 0:
                    classif_rationale = is_rationale_first_offer(passage)
                    if not re.search(r"True", classif_rationale):
                        labels["rationale"] = False
                        edit.append(edit_rationale_first_offer(passage))
                    else:
                        labels["rationale"] = False

                else:
                    classif_rationale = is_rationale(passage)
                    if not re.search(r"True", classif_rationale):
                        labels["rationale"] = False
                        edit.append(edit_rationale(passage))
                    else:
                        labels["rationale"] = True

                # FIRST OFFER EVER
                if not nego.first_offer_happend:
                    nego.first_offer_happend = True
                    # labels['first_offer'] = is_first_offer(negotiation=nego)
                    labels["first_offer"] = True

                    # STRONG OFFER FIRST
                    if is_strong_offer_first(price=price_detected, negotiation=nego):
                        labels["strongFirstOffer"] = True
                    else:
                        labels["strongFirstOffer"] = False
                        # Generate an example of better offer
                        edit.append(
                            edit_strong_offer_first(
                                history=prev_conv,
                                message=message,
                                offer=price_detected,
                                buyer_targetPrice=nego.buyer_targetPrice,
                            )
                        )
                        # flagged_errors.append('Made a weak first offer')

                # FIRST OFFER (AFTER SELLER)
                else:
                    if nego.buyer_nb_offer == 0:
                        if is_strong_offer_second(
                            price=price_detected, negotiation=nego
                        ):
                            labels["strongFirstOffer"] = True
                        else:
                            labels["strongFirstOffer"] = False
                            edit.append(
                                edit_strong_offer_second(
                                    history=prev_conv,
                                    message=message,
                                    offer=price_detected,
                                    buyer_targetPrice=nego.buyer_targetPrice,
                                    seller_currentOffer=nego.current_seller_offer,
                                )
                            )
                            # flagged_errors.append('Made a weak first offer')

                # STRONG COUNTER OFFER
                if 0 < nego.buyer_nb_offer < 4:
                    if is_strong_counteroffer(
                        price=price_detected, negotiation=nego, debugging=False
                    ):
                        labels["strongCounterOffer"] = True
                    else:
                        labels["strongCounterOffer"] = False
                        # Generate an example of better counteroffer
                        edit.append(
                            edit_strong_counteroffer(
                                history=prev_conv,
                                message=message,
                                offer=price_detected,
                                previous_offer=nego.current_buyer_offer,
                                buyer_reservationPrice=nego.buyer_reservationPrice,
                                seller_currentOffer=nego.current_seller_offer,
                            )
                        )
                        # flagged_errors.append('Did a weak counter offer')

                nego.buyer_nb_offer += 1
                nego.current_buyer_offer = price_detected

            turn["current_offer"] = nego.current_buyer_offer

        turn["content"] = message["content"]
        turn["labels"] = labels
        turn["comments"] = edit
        if len(edit) > 0:
            joined_comments = join_comments(edit)
            comment = generate_comments_summary(joined_comments, message["content"])
            alternative_msg = generate_message_reedition(
                joined_comments, message["content"]
            )

            turn["feedback"] = "Message feedback: " + comment + "\n\nA suggested alternative: " + alternative_msg
        processed_turns.append(turn)

    return processed_turns, flagged_errors
