#!/usr/bin/env python3

import re
import string


from comp_med_dsum_eval.preprocess.sec_tag.concept_graph import *
from comp_med_dsum_eval.preprocess.sec_tag.normalize import NORM_MAP
from comp_med_dsum_eval.preprocess.constants import HEADER_REGEX

DATA_DIR = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'data')

# file containing mapping of concepts to synonyms, tab-delimited, generated by db-extract.py
CS_MAP_FILE = os.path.join(DATA_DIR, 'concepts_and_synonyms.txt')

# name of the file containing the concept graph, generated by db-extract.py
GRAPH_FILENAME = os.path.join(DATA_DIR, 'graph.txt')

# recognize XML character entities
regex_xml_character_entity = re.compile(r'&(?:#([0-9]+)|#x([0-9a-fA-F]+)|([0-9a-zA-Z]+));')

# one or more letters, hyphen, one or more letters, space
regex_hyphenated_word = re.compile(r'[a-zA-Z]+-[a-zA-Z]+')

# ICD-9 or IDC-10 code
regex_icd_hyphen_num = re.compile(r'(entity|ICD)-(9|10)')
regex_icd_code = re.compile(r'entity-(9|10)\s+codes?(:|--?)', re.IGNORECASE)

# one or more letters, hyphen, floating pt. or integer number followed by an optional '*'
regex_hyphenated_lab_result = re.compile(r'[a-zA-Z]+-[\d.]+\*?')

# whitespace, words separated by space or forward slash, optional bracketed anonymized date,
# terminated by colon, dash, or double-dash
regex_multi_word = re.compile(r'(?<=\s)[a-zA-Z][a-zA-Z /]+(\[[\-*\d]+])?(:|--?)')

# line start, words separated by space or forward slash, optional bracketed anonymized date,
# terminated by colon, dash, or double-dash
regex_multi_word_line_start = re.compile(r'(?<=\A)[a-zA-Z][a-zA-Z /]+(\[[\-*\d]+])?(:|--?)')

# newline, words, whitespace, anonymized date in parens, newline
regex_header = re.compile(r'\n[a-zA-Z][\-a-zA-Z /]+\s\(\[[\-*\d]+]\)(\n|:|--?)')
regex_header_line_start = re.compile(r'\A[a-zA-Z][\-a-zA-Z /]+\s\(\[[\-*\d]+]\)(\n|:|--?)')

# ends with a number followed by a period (such as a numbered list)
regex_ends_with_number = re.compile(r'\d+.\Z')

# one or more whitespace chars
regex_multi_whitespace = re.compile(r'\s+')

# only whitespace chars
regex_only_whitespace = re.compile(r'\A\s+\Z')

# one or more spaces and newlines
regex_multi_space = re.compile(r' +')
regex_multi_newline = re.compile(r'\n+')

synonym_map = {}  # synonym => [list of associated concepts]
cid_to_concept_map = {}  # cid => concept
concept_to_cid_map = {}  # concept => cid

graph = ConceptGraph()

# ignore these headers if in a medications section
IGNORE_FOR_MEDS = ['DISPOSITION_PLAN',  # interpret 'disp' as 'dispensing', not 'dispensation'
                   'HEART_RATE',  # interpret 'hr' as 'hour', not 'heart rate'
                   'HEADACHE_REVIEW',  # another hr
                   'UNIT',  # dispensing units
                   'PAIN',  # warnings about pain, etc.
                   'LOWER_EXTREMITY_EXAM',  # 'pain in extremities' is not extremity exam or review
                   'EXTREMITY_REVIEW'
                   ]

IGNORE_FOR_VITALS = ['BLOOD_PRESSURE', 'HEART_RATE', 'RESPIRATORY_RATE']

IGNORE_FOR_HISTORY = ['LABORATORY_DATA',  # medicine review is not a lab data section
                      'HISTORY_PRESENT_ILLNESS']  # each mention of the term 'history' is not a new section

IGNORE_FOR_INSTRUCTIONS = ['PAIN', 'MEDICATIONS']

IGNORE_FOR_ASSESSMENT = ['VALVULAR_DATA']

# don't interpret a phrase with 'discharged' as a discharge date
IGNORE_FOR_DISPOSITION_PLAN = ['DATE_OF_DISCHARGE']

# ignore a 'time' concept if any of these words precedes the word 'time'
IGNORE_BEFORE_TIME = ['and', 'at', 'bed', 'by', 'bypass', 'cross-clamp', 'difficult',
                      'hard', 'in', 'next', 'of', 'one', 'over', 'prothrombin', 'response',
                      'run', 'same', 'that', 'this', 'the', 'thromboplastin', 'with', 'which']

# synonyms; need to examine context before assigning to a concept, if any
IGNORE_SYN_NEED_CONTEXT = ['location', 'length']

# ingore these for NORM_COMP forms
STOP_WORDS = ['and', 'at', 'by', 'for', 'from', 'in', 'of', 'on', 'or', 'the', 'to', 'with']

# synonyms containing hyphens
HYPHENATED_SYNONYMS = ['decision-makin', 'half-bro', 'half-sis', 'pre-admission', 't-max',
                       't-min', 'x-ray', 'X-Ray', 'cine-esophogram', 'Bronchial-Alveolar',
                       'skin-to-muscle', 'follow-up']

# explicit HTML character entities that are non-printing (need more)
HTML_CHARACTER_ENTITIES = ['&#8206;']


###############################################################################
class SectionHeader():
    # the section header spans [start, end) in the sentence
    def __init__(self, sentence_index, start, end, synonym, concept, treecode_list, raw=''):
        self.sentence_index = sentence_index
        self.start_index = start;
        self.end_index = end;
        self.synonym = synonym;
        self.concept = concept;
        self.treecode_list = treecode_list;
        self.raw = raw

    def __str__(self):
        result = "({0}, {1}, {2}, {3}, {4})".format(self.start_index,
                                                    self.end_index,
                                                    self.synonym,
                                                    self.concept,
                                                    self.treecode_list)
        return result

    def to_output_string(self):
        # concatenate the treecode_list into a sequence of dotted ints
        treecode_str = ".".join([str(i) for i in self.treecode_list])

        result = "{0} [{1}]".format(self.concept, treecode_str)
        return result


UNKNOWN = SectionHeader(0, 0, 0, 'unknown', 'UNKNOWN', '-1', 'unknown')


###############################################################################
def remove_newlines(text):
    # replace newline with space
    no_newlines = regex_multi_newline.sub(' ', text)

    # replace multiple consecutive spaces with single space
    cleaned_text = regex_multi_space.sub(' ', no_newlines)
    return cleaned_text



###############################################################################
def in_named_section(name, stack):
    """Return True if in the named section, False otherwise."""

    n = len(stack)
    if 0 == n:
        return False

    for i in reversed(range(0, n)):
        concept_name = stack[i].concept.lower()
        if name in concept_name:
            return True

    return False


###############################################################################
def nearest_whitespace(s, pos_start):
    """
    Search string backwards and forwards for the nearest whitespace chars
    to 'pos_start'. Return the results as a tuple. Returns -1 if no whitespace
    found.
    """

    # positions of whitespace to either side
    pos_left = -1
    pos_right = -1

    p = pos_start
    while p >= 0:
        c = s[p]
        if ' ' == c or '\n' == c or '\t' == c:
            pos_left = p
            break
        else:
            p = p - 1

    p = pos_start + 1
    while p < len(s):
        c = s[p]
        if ' ' == c or '\n' == c or '\t' == c:
            pos_right = p
            break
        else:
            p = p + 1

    return (pos_left, pos_right)


###############################################################################
def longest_in_lists(sequence):
    # sequence is a list of lists

    index_of_longest = 0
    longest = len(sequence[0])
    all_same_length = True

    for i in range(1, len(sequence)):
        l = len(sequence[i])
        if l > longest:
            longest = l
            index_of_longest = i

    # could have multiple of the longest length
    indices_of_longest = []
    for i in range(0, len(sequence)):
        l = len(sequence[i])
        if l == longest:
            indices_of_longest.append(i)
        elif l != longest:
            all_same_length = False

    return (longest, indices_of_longest, all_same_length)


###############################################################################
def shortest_in_lists(sequence):
    # sequence is a list of lists

    index_of_shortest = 0
    shortest = len(sequence[0])
    all_same_length = True

    for i in range(1, len(sequence)):
        l = len(sequence[i])
        if l < shortest:
            shortest = l
            index_of_shortest = i

    # could have multiple of the shortest length
    indices_of_shortest = []
    for i in range(0, len(sequence)):
        l = len(sequence[i])
        if l == shortest:
            indices_of_shortest.append(i)
        elif l != shortest:
            all_same_length = False

    return (shortest, indices_of_shortest, all_same_length)


###############################################################################
def word_offsets(text, words):
    """
    Return lists of start and end offsets for each word in text.
    The text is known to contain each word.
    """

    start_offsets = []
    end_offsets = []

    prev_pos = 0
    for i in range(0, len(words)):
        pos = text.find(words[i], prev_pos)
        start_offsets.append(pos)

        end_pos = pos + len(words[i])
        end_offsets.append(end_pos)
        prev_pos = end_pos + 1

    return (start_offsets, end_offsets)


###############################################################################
def remove_stop_words(words, start_offsets, end_offsets):
    """
    Remove stop words and the corresponding offsets from the supplied lists.
    """
    assert len(words) == len(start_offsets)
    assert len(words) == len(end_offsets)

    new_words = []
    new_start_offsets = []
    new_end_offsets = []

    for i in range(0, len(words)):
        w = words[i]
        o = start_offsets[i]
        e = end_offsets[i]
        if w in STOP_WORDS:
            continue
        else:
            new_words.append(w)
            new_start_offsets.append(o)
            new_end_offsets.append(e)

    return (new_words, new_start_offsets, new_end_offsets)


###############################################################################
def try_exact_match(sentence_index, start, end, lc_text):
    """Return candidate concepts for exactly matching lowercase text."""
    candidates = []

    if lc_text in synonym_map:
        for c in synonym_map[lc_text]:
            cid = concept_to_cid_map[c]
            treecode_list = graph.treecode_list(cid)
            sh = SectionHeader(sentence_index, start, end, lc_text, c, treecode_list)
            candidates.append(sh)

    return candidates


###############################################################################
def try_match_anchored_right(sentence_index, start, end, start_offsets, end_offsets, words):
    """
    Remove initial words one at a time and try for exact match. This
    finds word sequences anchored at the right, near the terminating
    symbol.
    """
    candidates = []

    new_end = start + end_offsets[-1]
    for i in range(1, len(words)):
        test_text = ' '.join(words[i:])
        if test_text in synonym_map:
            # found exact match, so save all associated concepts
            new_start = start + start_offsets[i]
            for c in synonym_map[test_text]:
                cid = concept_to_cid_map[c]
                treecode_list = graph.treecode_list(cid)
                sh = SectionHeader(sentence_index, new_start, new_end, test_text, c, treecode_list)
                candidates.append(sh)
            break

    return candidates


###############################################################################
def try_match_anchored_left(sentence_index, start, end, start_offsets, end_offsets, words):
    """If no match, try removing final words one at a time. This is less
    preferable than the previous loop, which anchors at the right to
    the terminating symbol.
    """
    candidates = []

    new_start = start + start_offsets[0]
    for i in reversed(range(1, len(words))):
        test_text = ' '.join(words[0:i])
        if test_text in synonym_map:
            # found exact match, so save all associated concepts
            new_end = start + end_offsets[i - 1]
            for c in synonym_map[test_text]:
                cid = concept_to_cid_map[c]
                treecode_list = graph.treecode_list(cid)
                sh = SectionHeader(sentence_index, new_start, new_end, test_text, c, treecode_list)
                candidates.append(sh)
            break

    return candidates


###############################################################################
def candidate_headers(sentence_index, start, end, text):
    assert end - start == len(text)

    # candidate section headers for this matching text
    candidates = []

    # if text contains '/', replace by space
    if '/' in text:
        text = text.replace('/', ' ')

    # convert text to lower case
    lc_text = text.lower()

    # if exact match, return all associated concepts for later disambiguation
    candidates = try_exact_match(sentence_index, start, end, lc_text)
    if 0 == len(candidates):
        # no exact match found, so split into individual words
        words = lc_text.split()
        word_count = len(words)
        # compute original word offsets in lc_text
        start_offsets, end_offsets = word_offsets(lc_text, words)
        assert 0 == start_offsets[0]

        # try match anchored right, remove words one-by-one from the left
        if 0 == len(candidates):
            candidates = try_match_anchored_right(sentence_index, start, end,
                                                  start_offsets, end_offsets, words)

            # try match anchored left, remove words one-by-one from the right
            if 0 == len(candidates):
                candidates = try_match_anchored_left(sentence_index, start, end,
                                                     start_offsets, end_offsets, words)

                if 0 == len(candidates):

                    # remove stop words and the corresponding offsets
                    words, start_offsets, end_offsets = remove_stop_words(words, start_offsets, end_offsets)
                    word_count = len(words)

                    if 0 == word_count:
                        # nothing left, unfortunately
                        return []
                    # replace words with their 'normalized' forms, if any
                    for i in range(0, word_count):
                        w = words[i]
                        if w in NORM_MAP:
                            words[i] = NORM_MAP[w]

                    test_text = ' '.join(words)

                    if test_text == lc_text:
                        pass
                    else:
                        # Try again for an exact match, this time to the normalized form.
                        new_start = start + start_offsets[0]
                        new_end = start + end_offsets[-1]
                        candidates = try_exact_match(sentence_index, new_start, new_end, test_text)

                        if 0 == len(candidates):
                            candidates = try_match_anchored_right(sentence_index, start, end,
                                                                  start_offsets, end_offsets, words)

                            if 0 == len(candidates):
                                candidates = try_match_anchored_left(sentence_index, start, end,
                                                                     start_offsets, end_offsets, words)

    return candidates


###############################################################################
def resolve_ambiguities(candidate_headers, stack):
    n = len(candidate_headers)
    # get cids of all candidates
    cids = [concept_to_cid_map[h.concept] for h in candidate_headers]

    # index of stack element being considered
    stack_index = len(stack) - 1

    # best candidates found at each level of the stack
    best_candidates_map = {}

    # if stack is empty, need treecode_lists for resolution below
    if 0 == len(stack):
        treecode_lists = [graph.treecode_list(cid) for cid in cids]

    # walk the stack and find nearest common ancestor codes for all stack elts
    while (stack_index >= 0):

        stack_elt = stack[stack_index]

        # cid of stack elt
        stackelt_cid = concept_to_cid_map[stack_elt.concept]

        # get the treecode lists for each candidate
        treecode_lists = [graph.treecode_list(cid) for cid in cids]

        # get the nearest common ancestors for the stack element and the candidates
        ancestor_codes = [graph.nearest_common_ancestor(stackelt_cid, cid) for cid in cids]
        (longest, indices_of_longest, all_same_length) = longest_in_lists(ancestor_codes)

        best_candidates_map[stack_index] = (longest, deepcopy(indices_of_longest), all_same_length)

        # go up to the next level in the stack and try again, may find a nearer common ancestor
        stack_index -= 1

    # find the code length of the nearest common ancestor among all stack levels
    longest_code_len = -1
    for key in best_candidates_map.keys():
        (longest, indices_of_longest, all_same_length) = best_candidates_map[key]
        if longest > longest_code_len:
            longest_code_len = longest

    # Extract all ancestors with code lengths equal to longest_code_len.
    # Need to use a set, since could find the same ancestor at multiple levels
    # of the stack.
    best_candidate_set = set()
    for key in best_candidates_map.keys():
        (longest, indices_of_longest, all_same_length) = best_candidates_map[key]
        if longest == longest_code_len:
            for i in indices_of_longest:
                best_candidate_set.add(i)

    best_candidates = list(best_candidate_set)
    num_best = len(best_candidates)
    if 1 == num_best:
        return candidate_headers[best_candidates[0]]

    elif num_best > 0:

        # Walking up the stack did not lead to a single nearest common ancestor. For the
        # remaining best candidates, take the highest level concept (shortest treecode, or
        # most general of the candidate concepts).

        treecode_lists_2 = [treecode_lists[i] for i in best_candidates]
        (shortest, indices_of_shortest, all_same_length) = shortest_in_lists(treecode_lists_2)

        if not all_same_length and 1 == len(indices_of_shortest):
            # found the winner
            return candidate_headers[best_candidates[indices_of_shortest[0]]]
        else:
            # still no clear winner, take the first of the best candidates
            return candidate_headers[best_candidates[0]]

    else:
        # take the highest-level (most general) of the candidate concept, if any
        (shortest, indices_of_shortest, all_same_length) = shortest_in_lists(treecode_lists)
        if not all_same_length and 1 == len(indices_of_shortest):
            return candidate_headers[indices_of_shortest[0]]

    # default is to not return anything
    return None


###############################################################################
def merge(list1, list2):
    """
    Merge two lists of regex match objects. Take the match that spans the
    longest sequence of chars in case of overlap. The matches in each list
    are ordered sequentially.
    """

    len1 = len(list1)
    len2 = len(list2)

    assert len1 > 0
    assert len2 > 0

    result = []

    count = len1 + len2
    start1 = 0
    start2 = 0
    end1 = 0
    end2 = 0
    index1 = 0
    index2 = 0

    for i in range(0, count):

        start1 = list1[index1].start()
        end1 = list1[index1].end()

        start2 = list2[index2].start()
        end2 = list2[index2].end()

        if end1 < start2:
            # list1 item occurs before list2 item starts
            result.append(list1[index1])
            index1 += 1
        elif start1 > end2:
            # list2 item occurs before list1 item starts
            result.append(list2[index2])
            index2 += 1
        else:
            # list1 and list2 items overlap; take longest match
            len_item1 = end1 - start1
            len_item2 = end2 - start2
            if len_item1 > len_item2:
                result.append(list1[index1])
            else:
                result.append(list2[index2])
            index1 += 1
            index2 += 1

        if len1 == index1:
            # finished list 1, push remainder of list2
            while index2 < len2:
                result.append(list2[index2])
                index2 += 1
            break

        elif len2 == index2:
            # finished list 2, push remainder of list 1
            while index1 < len1:
                result.append(list1[index1])
                index1 += 1
            break

    return result


###############################################################################
def process_report(report, spacy=None, verbose=False):
    doc = spacy(report)
    sentences = [str(sent) for sent in doc.sents]

    # context stack
    stack = []

    section_headers = []
    section_texts = []

    # need to examine context before definitive assignment
    headers_pending_context = []
    for sentence_index in range(0, len(sentences)):
        s = sentences[sentence_index]
        section_indices = [(m.start(), m.end()) for m in re.finditer(HEADER_REGEX, s, flags=re.M)]
        # print("\nHERE IS THE NEXT SENTENCE:\n->{0}<-".format(s))

        header_count = 0
        match_objects = []
        prev_end = 0

        # Any matches at the start of the sentence?

        match_obj1 = regex_header_line_start.match(s)
        match_obj2 = regex_multi_word_line_start.match(s)

        # take the longest match at the start of the line, if any
        if match_obj1 and match_obj2:
            len1 = match_obj1.end() - match_obj1.start()
            len2 = match_obj2.end() - match_obj2.start()
            if len1 > len2:
                match_objects.append(match_obj1)
            else:
                match_objects.append(match_obj2)
        else:
            if match_obj1:
                match_objects.append(match_obj1)
            if match_obj2:
                match_objects.append(match_obj2)

        if len(match_objects) > 0:
            prev_end = match_objects[-1].end()

        # now check for matches after the initial match, if any

        matches1 = []
        iterator = regex_header.finditer(s)
        for match_obj in iterator:
            matches1.append(match_obj)

        matches2 = []
        iterator = regex_multi_word.finditer(s)
        for match_obj in iterator:
            matches2.append(match_obj)

        merged_obects = []
        if 0 == len(matches1):
            merged_objects = matches2
        elif 0 == len(matches2):
            merged_objects = matches1
        else:
            merged_objects = merge(matches1, matches2)

        # extend match_objects with any that have been merged
        for m in merged_objects:
            if m.start() >= prev_end:
                match_objects.append(m)
                prev_end = m.end()

        prev_end = -1

        for match_obj in match_objects:
            start = match_obj.start()
            end = match_obj.end()
            if end <= prev_end:
                continue

            matching_text = s[start:end]

            # if the match ends in a single hyphen, check to see if hyphenated word
            # or hyphenated lab result
            if end >= 2 and '-' == s[end - 1] and s[end - 2] != '-':
                # find closest delimiting whitespace positions to the hyphenated word
                ws_left, ws_right = nearest_whitespace(s, end - 1)
                if -1 == ws_left:
                    ws_left = 0
                if -1 == ws_right:
                    ws_right = len(s)
                # print("\tHYPHEN TERMINATION, EXTRACTED: ->{0}<-".format(s[ws_left:ws_right+1]))

                hyph_match = regex_hyphenated_word.search(s[ws_left:ws_right])
                if hyph_match:
                    # found a hyphenated word
                    hyphenated_word = hyph_match.group().lower()

                    # Check for a termination char at the end of the hyphenated
                    # word. If present, this may be a section header.

                    end_char_index = ws_left + hyph_match.end()
                    if end_char_index < len(s):
                        end_char = s[ws_left + hyph_match.end()]
                        if ':' == end_char or '-' == end_char:
                            # extend the matching text to include the hyphenated word
                            end = ws_left + hyph_match.end() + 1
                            matching_text = s[start:end]
                            prev_end = end

                        elif not hyphenated_word in HYPHENATED_SYNONYMS:
                            continue

                # Is this an ICD 9 or 10 code?
                icd_hyphen_num_match = regex_icd_hyphen_num.search(s[start:ws_right + 1])
                if icd_hyphen_num_match:
                    # extend the match to include the code digit
                    end = start + icd_hyphen_num_match.end()
                    matching_text = s[start:end]

                    # Does the word 'code' or 'codes' appear?
                    icd_code_match = regex_icd_code.search(s[start:])  # len(codes)+1
                    if icd_code_match:
                        # extend the match to include the full expression
                        end = start + icd_code_match.end()
                        matching_text = s[start:end]

                else:
                    # Is this a hyphenated lab result?
                    lab_result_match = regex_hyphenated_lab_result.search(s[ws_left:ws_right + 1])
                    if lab_result_match:
                        continue

            prev_end = end

            # strip termination character(s); 'delta' is the number of termination chars

            # match must end in a known termination character; do not include it in the
            # search for candidate headers
            if not s[end - 1] in {':', '-', '\n'}:
                # print('Should be in set \':-newline\' yet it is {}'.format(s[end - 1]))
                continue
            end_delta = -1

            # if terminated by '--', strip both dashes
            if end >= 2 and '-' == s[end - 1] and '-' == s[end - 2]:
                end_delta = -2

            # if terminated by ' -' (space followed by dash), strip the space
            if end >= 2 and '-' == s[end - 1] and ' ' == s[end - 2]:
                end_delta = -2

            # strip leading newline, if any; matches: \n[all caps words] ([anonymized date])\n
            start_delta = 0
            if '\n' == matching_text[0]:
                start_delta = 1

            # remove any parenthesized bracketed anonymized data
            if ')' == matching_text[end_delta - 1] and ']' == matching_text[end_delta - 2]:
                pos = matching_text.rfind('(')
                if -1 != pos:
                    # move backwards over whitespace between all-caps text and open paren
                    pos -= 1
                    end_delta = start + pos - end

            # remove bracketed anonymized data from the matching text
            if ']' == matching_text[end_delta - 1]:
                pos = matching_text.rfind('[')
                if -1 != pos:
                    # move backwards over whitespace
                    pos -= 1
                    end_delta = start + pos - end

            # get all candidate headers for this matched text
            candidates = candidate_headers(sentence_index, start + start_delta, end + end_delta,
                                           matching_text[start_delta:end_delta])
            num_candidates = len(candidates)
            if 0 == num_candidates:
                continue
            elif 1 == len(candidates):
                header = candidates[0]
            else:
                header = resolve_ambiguities(candidates, stack)

            # if None, couldn't resolve ambiguities
            if header is None:
                continue

            # If a space both precedes and follows the matching synonym, it is probably a word
            # in a narrative section and not a section header.
            # print("header.start_index: {0}".format(header.start_index))
            # print("  header.end_index: {0}".format(header.end_index))
            # print("        sentence s: {0}".format(s))
            if header.start_index > 0:
                if (' ' == s[header.start_index - 1]) and (' ' == s[header.end_index]):
                    continue

            # ignore matches to the letter 'a' (synonym for ASSESSMENT)
            match_len = len(s[header.start_index:header.end_index])
            if 1 == match_len and ('a' == s[header.start_index] or 'A' == s[header.start_index]):
                continue

            # ignore certain terms in various situations
            if header.concept in IGNORE_FOR_MEDS and in_named_section('medication', stack):
                continue
            if header.concept in IGNORE_FOR_VITALS and in_named_section('vital', stack):
                continue
            if header.concept in IGNORE_FOR_VITALS and in_named_section('history', stack):
                continue
            if header.concept in IGNORE_FOR_HISTORY and in_named_section('history', stack):
                continue
            if header.concept in IGNORE_FOR_ASSESSMENT and in_named_section('assessment', stack):
                continue
            if header.concept in IGNORE_FOR_DISPOSITION_PLAN and in_named_section('disposition', stack):
                continue
            if header.concept in IGNORE_FOR_INSTRUCTIONS and in_named_section('instructions', stack):
                continue

            # if 'time' is the synonym, ignore if preceded by a word ending in '-tion',
            # such as 'deceleration', 'propagation', etc., or other common narrative words
            if 'TIME' == header.concept and header.start_index >= 6:
                # find the word preceding 'time', if any
                pos = s[0:header.start_index - 1].rfind(' ')
                if -1 != pos:
                    preceding_word = s[pos + 1:header.start_index - 1]
                    if preceding_word.endswith('tion') or preceding_word in IGNORE_BEFORE_TIME:
                        continue

            # if 'unit' is the synonym, ignore if immediately followed by a forward slash,
            # which likely indicates a unit of measurement, such as unit/mL
            if 'UNIT' == header.concept and '/' == s[header.end_index]:
                continue

            # ignore if need more context to interpret; save for later
            if header.synonym.lower() in IGNORE_SYN_NEED_CONTEXT:
                headers_pending_context.append(header)
                continue

            # level of this header in the graph
            level = len(header.treecode_list) - 1

            # update the context stack
            if 0 == len(stack):
                stack.append(header)
            else:
                stack_top = stack[len(stack) - 1]
                level_top = len(stack_top.treecode_list) - 1
                assert level >= 0 and level_top >= 0

                if level > level_top:
                    # new tag at a lower level in the graph (level > level_prev), push onto stack
                    stack.append(header)
                elif level == level_top:
                    # new tag at same level, replace top of stack
                    stack[len(stack) - 1] = header
                else:
                    # new tag at higher level, so pop everything at same or lower level
                    while level <= level_top:
                        stack.pop()
                        if 0 == len(stack):
                            break
                        else:
                            stack_top = stack[len(stack) - 1]
                            level_top = len(stack_top.treecode_list) - 1

                    stack.append(header)

            def overlap(a, b):
                a_range = set(range(a[0], a[1]))
                b_range = set(range(b[0], b[1]))
                return len(a_range.intersection(b_range)) > 0
            header_range = (header.start_index, header.end_index)
            matching_regex_headers = [x for x in section_indices if overlap(x, header_range)]
            if len(matching_regex_headers) == 0:
                extra_str = s[header.end_index:]
                extra_match = re.search(r'\n', extra_str)
                if extra_match is not None:
                    extra_str = extra_str[:extra_match.end()]
                extra_str = extra_str[:min(30, len(extra_str))]

                starts_line = header.start_index == 0
                if not starts_line:
                    for newline_idx in range(header.start_index - 1, -1, -1):
                        if s[newline_idx].isalpha():
                            break
                        if s[newline_idx] == '\n':
                            starts_line = True
                            break
                first_capital = s[header.start_index] == s[header.start_index].upper()
                raw_temp = s[header.start_index:header.end_index]
                full_extra_str = raw_temp + extra_str

                # starting a sentence and followed by newline
                island_regex = raw_temp + '\s{0,2}([-:]+)?\s{0,2}\n'
                colon_regex = raw_temp + r'\s{0,2}:'
                dash_regex = raw_temp + r'\s{0,2}-+'

                dash_match = re.match(dash_regex, full_extra_str)
                colon_match = re.match(colon_regex, full_extra_str)
                island_match = re.match(island_regex, full_extra_str)

                if starts_line and any([dash_match, colon_match, island_match]):
                    is_valid = True
                    if dash_match is not None:
                        header.end_index += dash_match.start()
                    elif colon_match is not None:
                        header.end_index += colon_match.start()
                elif not first_capital:
                    is_valid = False
                elif colon_match is not None:
                    is_valid = True
                    header.end_index += colon_match.start()
                elif dash_match is not None:
                    is_valid = True
                    header.end_index += dash_match.start()
                elif island_match is not None:
                    is_valid = True
                else:
                    # there's a chance it starts a line and we just needed to extend the header (it was truncated)
                    if starts_line and first_capital:
                        if ': ' in extra_str or ':\n' in extra_str:
                            header.end_index += extra_str.index(':')
                            is_valid = True
                        elif '- ' in extra_str or '-\n' in extra_str:
                            if 'x' in extra_str and 'ray' in extra_str:
                                is_valid = False
                            else:
                                header.end_index += extra_str.index('-')
                                is_valid = True
                        else:
                            is_valid = False
                    else:
                        is_valid = False
            elif len(matching_regex_headers) == 1:
                is_valid = True
                matching_regex_header = matching_regex_headers[0]
                rhs, rhe = matching_regex_header
                regex_end_in_punc = s[rhe - 1] in {':', '-'}
                if regex_end_in_punc:
                    rhe -= 1
                if rhs > 0 and s[rhs - 1] == '_':
                    pass
                else:
                    header.start_index = min(header.start_index, rhs)
                    if regex_end_in_punc:
                        header.end_index = max(header.end_index, rhe)
            else:
                raise Exception('Should not have multiple overlapping sections')
            header.raw = s[header.start_index:header.end_index].strip(string.punctuation + ' ')
            if is_valid:
                if verbose and header.raw.lower() == header.raw:
                    print(f'Warning: all lowercase section --> {header.raw}')
                section_headers.append(header)
                header_count += 1
            else:
                if verbose:
                    print(f'Ignoring Clarity section --> Raw={header.raw} Concept={header.concept}')

        if 0 == header_count:
            # no matches for this sentence, so concatenate to previous
            clean_text = s  # remove_newlines(s)
            if 0 == len(section_texts):
                section_texts.append(clean_text)
            else:
                if not section_texts[-1] == ' ':
                    section_texts[-1] += ' '
                section_texts[-1] += clean_text
        else:
            sh_len = len(section_headers)
            assert sh_len >= header_count
            # if start of first header match > 0, extract preceding text and append to previous
            start = section_headers[sh_len - header_count].start_index
            if start > 0:
                extra = s[0:start]  # remove_newlines(s[0:start])
                if 0 == len(section_texts):
                    section_texts.append(extra)
                else:
                    if not section_texts[-1] == ' ':
                        section_texts[-1] += ' '
                    section_texts[-1] += extra

            for i in range(0, header_count):
                end = section_headers[sh_len - header_count + i].end_index

                if i < header_count - 1:
                    next_start = section_headers[sh_len - header_count + i + 1].start_index
                    text = s[end + 1:next_start]
                else:
                    text = s[end + 1:]

                # could have a section with no text, in which case print a single space
                if 0 == len(text):
                    text = ' '

                clean_text = text  # remove_newlines(text)
                section_texts.append(clean_text)

    # print out tagged report
    num_headers = len(section_headers)
    num_sections = len(section_texts)
    # could have one additional text section if no header at start of report
    assert (num_sections == num_headers) or (num_sections == num_headers + 1)
    if num_sections == 0:
        if verbose:
            print('0 sections found.')
        return None, None

    # section_start = 0
    if num_sections > num_headers:
        # section_start = 1
        section_headers.insert(0, UNKNOWN)
        # print("<UNKNOWN []>\n\t{0}".format(section_texts[0]))

    # texts = section_texts[section_start:]
    # for i in range(0, num_headers):
    #    print("<{0}>\n\t{1}".format(section_headers[i].to_output_string(), section_texts[i]))

    section_texts = [s.strip(':-\n\t ') for s in section_texts]
    return section_headers, section_texts


inited = False
init_in_progress = False


def section_tagger_init():
    global init_in_progress, inited
    if inited or init_in_progress:
        print("section tagger init already done or in progress")
        return

    init_in_progress = True

    print("section_tagger_init...")

    # load the mapping of concept strings to synonyms
    infile = open(CS_MAP_FILE, 'rt')
    # skip the first line, which contains format information
    infile.readline()

    # build the map of synonyms to concepts; concepts will become section headers;
    # save synonyms in LOWER CASE, since all comparisons with synonyms will be done
    # in lower case (also shortens list, since some synonyms only differ by capitalization)
    for line in infile:
        (cid, concept_name, synonym_id, synonym_name, synonym_type) = line.split('\t')

        c_name = concept_name.upper()  # save concepts in UPPER CASE
        s_name = synonym_name.lower()  # save synonyms in LOWER CASE

        # cids are integers
        cid = int(cid)

        cid_to_concept_map[cid] = c_name
        concept_to_cid_map[c_name] = cid

        if not s_name in synonym_map:
            # have not seen this synonym before
            synonym_map[s_name] = [c_name]  # cid??
        else:
            if not c_name in synonym_map[s_name]:
                # have seen this synonym before, but associated concept is new
                synonym_map[s_name].append(c_name)  # cid??

    # map 'discharge_disposition' to 'discharge_condition', since
    # 'discharge_disposition' is not in the graph because of a missing treecode
    DC = ['DISCHARGE_CONDITION']
    synonym_map['discharge disposition'] = DC
    synonym_map['discharge_disposition'] = DC
    synonym_map['disposition at discharge'] = DC

    # map 'labs on admission' to 'laboratory_data', same reason
    LD = ['LABORATORY_DATA']
    synonym_map['laboratory_data_admission'] = LD
    synonym_map['laboratory data on admission'] = LD
    synonym_map['laboratory data admission'] = LD
    synonym_map['admission lab'] = LD
    synonym_map['labs on admission'] = LD

    infile.close()

    # build the concept graph, including extra entries added by db-extract.py
    db_extra = {}
    db_extra[63] = 2
    db_extra[544] = 24

    graph.load_from_file(GRAPH_FILENAME, db_extra)

    inited = True
    init_in_progress = False
    return True


if __name__ == '__main__':
    section_tagger_init()
    x, y = process_report("CV  : The patient's vital signs were routinely monitored, and was put on vasopressin, "
                          "norepinephrine and epinephrine during her stay to maintain appropriate hemodynamics. "
                          "A final sentence yay! Pulmonary:  "
                          "Vital signs were routinely monitored. She was intubated and sedated throughout her "
                          "admission, and her ventilation settings were adjusted based on ABG values")
    for a, b in zip(x, y):
        print(a.to_output_string(), a.synonym, b)
