import os
from enum import Enum
from dataclasses import dataclass, field
from filelock import FileLock
import numpy as np
import pandas as pd
from transformers import AutoTokenizer, AutoModelForSequenceClassification, HfArgumentParser, Trainer, \
    TrainingArguments, DataProcessor, set_seed, EvalPrediction
from sklearn.model_selection import train_test_split
from sklearn.metrics import accuracy_score, precision_recall_fscore_support
from typing import List, Optional, Union, Callable, Dict
from torch.utils.data.dataset import Dataset
import torch
from utils import InputExample, InputFeatures
import time
import logging
from collections import Counter
from OptimTrainer import OptimTrainer, PREFIX_CHECKPOINT_DIR
logger = logging.getLogger(__name__)
import joblib
from pathlib import Path

def split_stratified_into_train_val_test(df_input, stratify_colname='y',
                                         frac_train=0.7, frac_val=0.10, frac_test=0.20,
                                         random_state=None):
    '''
    Splits a Pandas dataframe into three subsets (train, val, and test)
    following fractional ratios provided by the user, where each subset is
    stratified by the values in a specific column (that is, each subset has
    the same relative frequency of the values in the column). It performs this
    splitting by running train_test_split() twice.

    Parameters
    ----------
    df_input : Pandas dataframe
        Input dataframe to be split.
    stratify_colname : str
        The name of the column that will be used for stratification. Usually
        this column would be for the label.
    frac_train : float
    frac_val   : float
    frac_test  : float
        The ratios with which the dataframe will be split into train, val, and
        test data. The values should be expressed as float fractions and should
        sum to 1.0.
    random_state : int, None, or RandomStateInstance
        Value to be passed to train_test_split().

    Returns
    -------
    df_train, df_val, df_test :
        Dataframes containing the three splits.
    '''

    if frac_train + frac_val + frac_test != 1.0:
        raise ValueError('fractions %f, %f, %f do not add up to 1.0' % \
                         (frac_train, frac_val, frac_test))

    if stratify_colname not in df_input.columns:
        raise ValueError('%s is not a column in the dataframe' % (stratify_colname))

    X = df_input # Contains all columns.
    y = df_input[[stratify_colname]] # Dataframe of just the column on which to stratify.

    # Split original dataframe into train and temp dataframes.
    df_train, df_temp, y_train, y_temp = train_test_split(X,
                                                          y,
                                                          stratify=y,
                                                          test_size=(1.0 - frac_train),
                                                          random_state=random_state)

    # Split the temp dataframe into val and test dataframes.
    relative_frac_test = frac_test / (frac_val + frac_test)
    df_val, df_test, y_val, y_test = train_test_split(df_temp,
                                                      y_temp,
                                                      stratify=y_temp,
                                                      test_size=relative_frac_test,
                                                      random_state=random_state)

    assert len(df_input) == len(df_train) + len(df_val) + len(df_test)

    return df_train, df_val, df_test

@dataclass
class DataTrainingArguments:
    """
    Arguments pertaining to what data we are going to input our model for training and eval.
    Using `HfArgumentParser` we can turn this class
    into argparse arguments to be able to specify them on
    the command line.
    """
    data_file: str = field(
        metadata={"help": "The input data dir. Should contain the .tsv files (or other data files) for the task."}
    )
    task_name: str = field(default="Twitter")
    max_seq_length: int = field(
        default=256,
        metadata={
            "help": "The maximum total input sequence length after tokenization. Sequences longer "
            "than this will be truncated, sequences shorter will be padded."
        },
    )
    overwrite_cache: bool = field(
        default=False, metadata={"help": "Overwrite the cached training and evaluation sets"}
    )

    def __post_init__(self):
        self.task_name = self.task_name.lower()

@dataclass
class ModelArguments:
    """
    Arguments pertaining to which model/config/tokenizer we are going to fine-tune from.
    """

    model_name_or_path: str = field(
        default=None, metadata={"help": "Path to pretrained model or model identifier from huggingface.co/models"}
    )
    config_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained config name or path if not the same as model_name"}
    )
    tokenizer_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained tokenizer name or path if not the same as model_name"}
    )
    cache_dir: Optional[str] = field(
        default=None, metadata={"help": "Where do you want to store the pretrained models downloaded from s3"}
    )

def convert_examples_to_features(
    examples: List[InputExample],
    tokenizer: AutoTokenizer,
    max_length: Optional[int] = None,
    label_list: List = None,
    output_mode="classification",
):
    if max_length is None:
        max_length = tokenizer.max_len
    label_map = {label: i for i, label in enumerate(label_list)}

    def label_from_example(example: InputExample) -> Union[int, float, None]:
        if example.label is None:
            return None
        if output_mode == "classification":
            return label_map[example.label]
        elif output_mode == "regression":
            return float(example.label)
        raise KeyError(output_mode)

    labels = [label_from_example(example) for example in examples]

    batch_encoding = tokenizer(
       # [(example.text_a, example.text_b) for example in examples],
        [example.text_a for example in examples],
        max_length=max_length,
        padding="max_length",
        truncation=True,
    )

    features = []
    for i in range(len(examples)):
        inputs = {k: batch_encoding[k][i] for k in batch_encoding}

        feature = InputFeatures(**inputs, label=labels[i])
        features.append(feature)

    for i, example in enumerate(examples[:5]):
        logger.info("*** Example ***")
        logger.info("guid: %s" % (example.guid))
        logger.info("features: %s" % features[i])

    return features

class Processor(DataProcessor):
    """Processor for the CoLA data set (GLUE version)."""

    def __init__(self, labels, train, dev, test):
        self.labels = labels
        self.train = train
        self.dev = dev
        self.test = test

    def get_train_examples(self, data_dir=None):
        """See base class."""
        return self._create_examples(self._read_tsv(self.train), "train")

    def get_train_data(self):
        return self._read_tsv(self.train)
    def get_dev_data(self):
        return self._read_tsv(self.dev)
    def get_test_data(self):
        return self._read_tsv(self.test)

    def get_dev_examples(self, data_dir=None):
        """See base class."""
        return self._create_examples(self._read_tsv(self.dev), "dev")

    def get_test_examples(self, data_dir=None):
        """See base class."""
        return self._create_examples(self._read_tsv(self.test), "test")

    def get_test_labels(self):
        return self._read_tsv(self.test)

    def get_labels(self):
        """See base class."""
        return self.labels

    def _read_tsv(cls, input, quotechar=None):
        lines = []
        for i in input.itertuples():
            lines.append({
                'text': i.full_text,
                'sent_id': i.id_str,
                'label': i.label
            })
        return lines

    def _create_examples(self, lines, set_type):
        """Creates examples for the training, dev and test sets."""
        test_mode = set_type == "test"
        examples = []
        for (i, line) in enumerate(lines):
            guid = "%s-%s" % (set_type, i)
            text_a = line["text"]
            label = None if test_mode else line["label"]
            examples.append(InputExample(guid=guid, text_a=text_a, text_b=None, label=label))
        return examples

class Split(Enum):
    train = "train"
    dev = "dev"
    test = "test"

class TwitterDataset(Dataset):
    """
    This will be superseded by a framework-agnostic approach
    soon.
    """

    args: DataTrainingArguments
    output_mode: str
    features: List[InputFeatures]

    def __init__(
        self,
        args: DataTrainingArguments,
        tokenizer: AutoTokenizer,
        processor: DataProcessor,
        limit_length: Optional[int] = None,
        mode: Union[str, Split] = Split.train,
        cache_dir: Optional[str] = None
    ):
        self.args = args
        self.processor = processor
        self.output_mode = "classification"
        if isinstance(mode, str):
            try:
                mode = Split[mode]
            except KeyError:
                raise KeyError("mode is not a valid split name")
        # Load data features from cache or dataset file
        cached_features_file = os.path.join(
            cache_dir if cache_dir is not None else args.data_dir,
            "cached_{}_{}_{}_{}".format(
                mode.value,
                tokenizer.__class__.__name__,
                str(args.max_seq_length),
                args.task_name,
            ),
        )
        label_list = self.processor.get_labels()
        self.label_list = label_list

        # Make sure only the first process in distributed training processes the dataset,
        # and the others will use the cache.
        lock_path = cached_features_file + ".lock"
        with FileLock(lock_path):

            if os.path.exists(cached_features_file) and not args.overwrite_cache:
                start = time.time()
                self.features = torch.load(cached_features_file)
                logger.info(
                    f"Loading features from cached file {cached_features_file} [took %.3f s]", time.time() - start
                )
            else:
                logger.info(f"Creating features from dataset")

                if mode == Split.dev:
                    examples = self.processor.get_dev_examples(None)
                elif mode == Split.test:
                    examples = self.processor.get_test_examples(None)
                else:
                    examples = self.processor.get_train_examples(None)
                if limit_length is not None:
                    examples = examples[:limit_length]
                self.features = convert_examples_to_features(
                    examples,
                    tokenizer,
                    max_length=args.max_seq_length,
                    label_list=label_list,
                    output_mode=self.output_mode,
                )
                start = time.time()
                torch.save(self.features, cached_features_file)
                # ^ This seems to take a lot of time so I want to investigate why and how we can improve.
                logger.info(
                    "Saving features into cached file %s [took %.3f s]", cached_features_file, time.time() - start
                )
    def __len__(self):
        return len(self.features)

    def __getitem__(self, i) -> InputFeatures:
        return self.features[i]

BERT_model = "bert-base-german-cased"

model_args = ModelArguments(
    model_name_or_path=BERT_model,
    tokenizer_name=BERT_model,
    cache_dir="./cache",
    config_name=None,
)

data_args = DataTrainingArguments(
    data_file="",
    overwrite_cache=True,
    max_seq_length=256
)

training_args = TrainingArguments(
    logging_steps=100,
    per_device_train_batch_size=8,
    per_device_eval_batch_size=8,
    save_steps=1000,
    #evaluate_during_training=True,
    output_dir="./model_output/",
    overwrite_output_dir=True,
    do_train=True,
    do_eval=True,
    do_predict=False,
    learning_rate=0.00008,
    num_train_epochs=10,
)

label_column = "label"
inputpath = "../data/expertdata_resolved.tsv"
df = pd.read_csv(inputpath, sep="\t")
num_labels = len(df[label_column].unique())

seeds = [2020, 2021, 2022, 2023, 2024, 2025, 2026, 2027, 2028, 2029]
current_seed = seeds[9]

df_train, df_test = train_test_split(df, test_size=0.20, train_size=0.80, shuffle=True, stratify=df[[label_column]],
                                    random_state=current_seed)
processor = Processor(train=df_train, dev=df_test, test=None, labels=list(df[label_column].unique()))

# Setup logging
logging.basicConfig(
    format="%(asctime)s - %(levelname)s - %(name)s -   %(message)s",
    datefmt="%m/%d/%Y %H:%M:%S",
    level=logging.INFO,
    #level=logging.WARN,
)
logger.warning(
    "Process rank: %s, device: %s, n_gpu: %s, distributed training: %s, 16-bits training: %s",
    training_args.local_rank,
    training_args.device,
    training_args.n_gpu,
    bool(training_args.local_rank != -1),
    training_args.fp16,
)
logger.info("Training/evaluation parameters %s", training_args)

# Set seed
set_seed(training_args.seed)
output_mode = "classification"

# model
tokenizer = AutoTokenizer.from_pretrained("bert-base-german-cased")
model = AutoModelForSequenceClassification.from_pretrained("bert-base-german-cased", num_labels=num_labels)

# datasets
train_dataset = (
    TwitterDataset(data_args, tokenizer=tokenizer, processor=processor, cache_dir=model_args.cache_dir)
    if training_args.do_train
    else None
)
eval_dataset = (
    TwitterDataset(data_args, tokenizer=tokenizer, processor=processor, mode="dev", cache_dir=model_args.cache_dir)
    if training_args.do_eval
    else None
)
test_dataset = (
    TwitterDataset(data_args, tokenizer=tokenizer, processor=processor, mode="test", cache_dir=model_args.cache_dir)
    if training_args.do_predict
    else None
)

def compute_metrics(pred):
    labels = pred.label_ids
    preds = pred.predictions.argmax(-1)
    precision, recall, f1, _ = precision_recall_fscore_support(labels, preds, average='macro')
    acc = accuracy_score(labels, preds)
    return {
        'accuracy': acc,
        'f1': f1,
        'precision': precision,
        'recall': recall
    }

trainer = OptimTrainer(
    model=model,
    args=training_args,
    train_dataset=train_dataset,
    eval_dataset=eval_dataset,
    compute_metrics=compute_metrics
)

# Training
if training_args.do_train:
    # pass metric which will be used to determine best model during training
    trainer.train(eval_metric='f1')

# load best model
checkpoint_folder = f"{PREFIX_CHECKPOINT_DIR}-best"
output_dir = os.path.join(trainer.args.output_dir, checkpoint_folder)
model = AutoModelForSequenceClassification.from_pretrained(output_dir, num_labels=num_labels)
trainer = OptimTrainer(
    model=model,
    args=training_args,
    train_dataset=train_dataset,
    eval_dataset=eval_dataset,
    compute_metrics=compute_metrics
)

# Evaluation
if training_args.do_eval:
    logger.info("*** Evaluate ***")
    eval_result = trainer.evaluate()
    print(eval_result)

# save best performing model
model_path = Path("/home/tilman/Repositories/inception-external-recommender/models/BertClassifier/twitter_opinion_bert_finetuned.joblib")
model_path.parent.mkdir(parents=True, exist_ok=True)
tmp_model_path = model_path.with_suffix(".joblib.tmp")
joblib.dump(model, tmp_model_path)
os.replace(tmp_model_path, model_path)
exit()

# Prediction
# if training_args.do_predict:
#     predictions = trainer.predict(test_dataset=test_dataset).predictions
#     predictions = np.argmax(predictions, axis=1)
#     label_list = processor.get_labels()
#     label_map = {label: i for i, label in enumerate(label_list)}
#     label_map_inversed = {v: k for k, v in label_map.items()}
#     labels = [label_map[i['label']] for i in processor.get_test_labels()]
#     precision, recall, f1, _ = precision_recall_fscore_support(labels, predictions, average='macro')
#     accuracy = accuracy_score(labels, predictions)
#     print('TEST:', 'test_f1', f1, 'precision', precision, 'recall', recall)
#     precision, recall, f1, _ = precision_recall_fscore_support(labels, [Counter(labels).most_common(1)[0][0]] * len(labels), average='macro')
#     print('TEST MAJORITY:', 'test_f1', f1, 'precision', precision, 'recall', recall)
#     test_data = processor.get_test_data()
#     df_pred = pd.DataFrame({'sentence': [i['text'] for i in test_data], 'label': [i['label'] for i in test_data], 'predicted': [label_map_inversed[i] for i in predictions]})
#     df_pred.to_csv("result.tsv", sep="\t", index=False)
