""" Class which implements several sampling strategies """
import random
import math
from scipy.stats import entropy
from scipy.special import softmax
import numpy as np
from operator import itemgetter

class InstanceSampler:
    def __init__(self):
        """ Init sampler """
        # Default strategy is random sampling
        self.strategy = 'random'
        self.pred_prob = []
        self.user_goal = -1
        self.sample_set = []
        self.pool_size = -1
        self.h_max = entropy([0.25,0.25,0.25,0.25]) # Can be precomputed since we know the number of classes
        self.h_normalize_max = entropy([1/20 for i in range(20)]) # Same here
        self.uncertainty_comp = 'softmax'
        self.lmbda = 0.5 # Set lambda to 0.5
        self.lmbda_strategy = 'None'

    #########################
    #   Active Learning
    #########################

    def set_lambda(self, strategy : str) -> None:
        """ Set strategy for the lambda """
        self.lmbda_strategy = strategy

    def update_lambda(self, iteration : int) -> None:
        """ Update lambda according to the given iteration """
        if self.lmbda_strategy == 'root':
            self.lmbda = 1/math.sqrt(iteration+1)

    def set_uncertainty_comp(self, comp : str) -> None:
        self.uncertainty_comp = comp

    def finish_iteration(self) -> None:
        """ After finishing a single iteration, do the required clean-ups. """
        self.user_goal = -1
        self.pred_prob = []
        self.sample_set = []

    def set_pool_size(self, pool_size: int) -> None:
        self.pool_size = pool_size

    def set_sampling_strategy(self, strategy: str) -> None:
        """ Set the sampling strategy. Possible inputs are:
            random
            uncertainty
            user
            joint
            pipeline        
        """        
        self.strategy = strategy

    def set_user_goal(self, user_query: int) -> None:
        """ Sets the user queried class. Instead of using the classes 1-5,
        we scale it to the range of 0-4 for easier comparison. """
        self.user_goal = user_query

    def sample_instance(self) -> int:
        """ Sample instance according to the specified strategy. Returns
         the index of the instance from the according sampling strategy. """
        if len(self.pred_prob) == 0:
            # If the logits have not been set throw error:
            raise Exception("No logits provided.")
        if self.strategy == 'uncertainty':
            self.uncertainty_sampling()
        elif self.strategy == 'user':
            self.user_sampling()
        elif self.strategy == 'tradeoff':
            self.tradeoff_sampling()
        elif self.strategy == 'combined':
            self.combined_sampling()
        else:
            self.random_sampling()

    def get_sampled_instances(self) -> list:
        return self.sample_set[0][1]

    def set_pred_probs(self, probs: list) -> None:
        """ Set the logits to compute the sampling. We compute the prediction
        probabilities using the softmax to do entropy sampling. """
        self.pred_prob = []
        # Compute entropy of entropy for 20 gaps!
        if self.uncertainty_comp == 'softmax':
            for i,batch in enumerate(probs):
                self.pred_prob.append((self._get_softmax_entropy(batch),i,self._get_ctest_label([pred  for pred in batch])))
        else:
            for i,batch in enumerate(probs):
                self.pred_prob.append((self._get_token_entropy(batch),i,self._get_ctest_label([pred  for pred in batch])))

    def random_sampling(self) -> None:
        """ Random sampling strategy. """
        self.sample_set = random.sample(self.pred_prob,1)
        
    def uncertainty_sampling(self) -> None:
        """ Uncertainty sampling based on the prediction probabilities.
        Uses entropy sampling. """
        self.sample_set = sorted(self.pred_prob, reverse=True, key=itemgetter(0))

    def user_sampling(self) -> None:
        """ We sample the instance with the class queried by the user and the 
        highest prediction of the model probability (the most certain sample). """        
        if self.user_goal < 0:
            raise Exception("No user goal specified.")
        tmp_sample_set = []
        min_class_distance = 0
        min_set = []
        while len(tmp_sample_set) < 1:
            if min_class_distance > 5:
                print("Something is wrong with the data. Please fix this.")
                break
            for prob in self.pred_prob:
                if abs(self.user_goal - prob[2]) == min_class_distance:
                    min_set.append(prob)
            tmp_sample_set.extend(sorted(min_set, key=itemgetter(0)))
            if len(tmp_sample_set) < 1:
                min_class_distance += 1
        # Set sample set sorted by 
        self.sample_set = tmp_sample_set[:]

    def combined_sampling(self) -> None:
        """ Pipeline sampling. First get all instances predicted to have the class
        queried by the user. Then do uncertainty sampling. """
        if self.user_goal < 0:
            raise Exception("No user goal specified.")
        tmp_sample_set = []
        min_class_distance = 0
        max_set = []
        while len(tmp_sample_set) < 1:
            if min_class_distance > 5:
                print("Something is wrong with the data. Please fix this.")
                break
            for prob in self.pred_prob:
                if abs(self.user_goal - prob[2]) == min_class_distance:
                    max_set.append(prob)
            tmp_sample_set.extend(sorted(max_set, reverse=True, key=itemgetter(0)))
            if len(tmp_sample_set) < 1:
                min_class_distance += 1
        self.sample_set = tmp_sample_set

    def tradeoff_sampling(self) -> None:
        """ Joint sampling, simply do """
        tmp_sample_set = []
        if self.user_goal < 0:
            raise Exception("No user goal specified.")
        # V1: Use 1/entropy to convert the max-term into a min-term
        # V2: Use -entropy instead
        for prob in self.pred_prob:
            tmp_sample_set.append(((1-self.lmbda)*(1-(abs(self.user_goal - prob[2])/4)) + self.lmbda*prob[0],prob[1]))
        # Set sample set sorted by 
        self.sample_set = sorted(tmp_sample_set[:], reverse=True, key=itemgetter(0))

    def _get_ctest_label(self, probs : list ) -> int: 
        """ Compute scores depending on the gap class """
        classes = (np.argmax(x) for x in probs)
        score = sum([1 for pred_cl in classes if random.random() <= (pred_cl+1)/4])
        score = score*5
        if score <= 54:
            return 0
        elif score <= 64:
            return 1
        elif score <= 74:
            return 2
        elif score <= 84:
            return 3
        else:
            return 4

    def _get_token_entropy(self, probs : list) -> float:
        """ Compute the total token entropy for a C-test """
        return sum([entropy(pred)/self.h_max for pred in probs])/20.0

    def _get_softmax_entropy(self, probs : list) -> float:
        """ Computes the entropy over the softmax squashed entropies of a C-test"""
        scaling_factor = np.mean([entropy(pred)/self.h_max for pred in probs])/self.h_normalize_max
        return scaling_factor*entropy(softmax([entropy(pred) for pred in probs]))





