import random
from collections import defaultdict
from typing import DefaultDict, Dict, List

import numpy as np
from ActionLogger import Action, ActionLogger
from Assignment import Assignment
from const import CONTENT_SKIP
from RealPossessedDetector import RealPossessedDetector
from RolePredictor import RolePredictor
from ScoreMatrix import ScoreMatrix
from Util import Util

from aiwolf import (AbstractPlayer, Agent, ComingoutContentBuilder, Content,
                    EstimateContentBuilder, GameInfo, GameSetting, Judge,
                    Operator, RequestContentBuilder, Role, Species, Status,
                    Talk, Topic, VoteContentBuilder)
from aiwolf.constant import AGENT_ANY, AGENT_NONE
from aiwolf.vote import Vote
from pathlib import Path
import openai
import time
import os
import copy
import json
from retry import retry
import re
# openai.api_keyにOpenAIのAPIキーを入れる
openai.api_key = os.environ['OPENAI_API_KEY']
gpt4="gpt-4-1106-preview"
gpt3="gpt-3.5-turbo-1106"
#langchain
from langchain.prompts import PromptTemplate
from langchain.chat_models import ChatOpenAI
from langchain.chains import LLMChain

# 村人
class Villager(AbstractPlayer):
    """ddhb villager agent."""

    me: Agent # 自身
    """Myself."""
    vote_candidate: Agent # 投票候補
    """Candidate for voting."""
    game_info: GameInfo # ゲーム情報
    """Information about current game."""
    game_setting: GameSetting # ゲーム設定
    """Settings of current game."""
    comingout_map: DefaultDict[Agent, Role] # CO辞書
    """Mapping between an agent and the role it claims that it is."""
    divination_reports: List[Judge] # 占い結果
    """Time series of divination reports."""
    identification_reports: List[Judge] # 霊媒結果
    """Time series of identification reports."""
    talk_list_head: int # talkのインデックス
    """Index of the talk to be analysed next."""
    will_vote_reports: DefaultDict[Agent, Agent] # 投票宣言
    talk_list_all: List[Talk] # 全talkリスト
    talk_turn: int # talkのターン
    role_predictor: RolePredictor # role_predictor
    talk_gpt:List[Talk]
    info:str
    my_turn:int=0
    plan:str
    bdi_history = []
    count=0
    vote_counter = 0
    vote_target=[]
    vote_werewolf=0
    vote_villager=0
    vote_possessed=0
    vote_seer=0
    my_action = []


    def __init__(self) -> None:
        """Initialize a new instance of ddhbVillager."""
        self.me = AGENT_NONE
        self.vote_candidate = AGENT_NONE
        self.game_info = None  # type: ignore
        self.comingout_map = defaultdict(lambda: Role.UNC)
        self.divination_reports = []
        self.identification_reports = []
        self.talk_list_head = 0
        self.will_vote_reports = defaultdict(lambda: AGENT_NONE)
        self.talk_list_all = []
        self.talk_turn = 0
        self.role_predictor = None
        self.N = -1
        self.M = -1
        self.agent_idx_0based = -1
        # フルオープンしたかどうか
        self.doFO = False
        #gpt_info
        self.info=""
        self.my_turn=0
        self.plan="なし"
        self.bdi_history = []
        self.count=0
        self.vote_counter=0
        self.vote_target=[]
        self.vote_werewolf=0
        self.vote_possessed=0
        self.vote_villager=0
        self.vote_seer=0
        self.my_action=[]

    # エージェントが生存しているか
    def is_alive(self, agent: Agent) -> bool:
        """Return whether the agent is alive.
        Args:
            agent: The agent.
        Returns:
            True if the agent is alive, otherwise false.
        """
        return self.game_info.status_map[agent] == Status.ALIVE


    # 自分以外のエージェントリスト
    def get_others(self, agent_list: List[Agent]) -> List[Agent]:
        """Return a list of agents excluding myself from the given list of agents.
        Args:
            agent_list: The list of agent.
        Returns:
            A list of agents excluding myself from agent_list.
        """
        return [a for a in agent_list if a != self.me]


    # 生存するエージェントリスト
    def get_alive(self, agent_list: List[Agent]) -> List[Agent]:
        """Return a list of alive agents contained in the given list of agents.
        Args:
            agent_list: The list of agents.
        Returns:
            A list of alive agents contained in agent_list.
        """
        return [a for a in agent_list if self.is_alive(a)]


    # 自分以外の生存するエージェントリスト
    def get_alive_others(self, agent_list: List[Agent]) -> List[Agent]:
        """Return a list of alive agents that is contained in the given list of agents
        and is not equal to myself.
        Args:
            agent_list: The list of agents.
        Returns:
            A list of alie agents that is contained in agent_list
            and is not equal to mysef.
        """
        return self.get_alive(self.get_others(agent_list))


    # ランダムセレクト
    def random_select(self, agent_list: List[Agent]) -> Agent:
        """Return one agent randomly chosen from the given list of agents.
        Args:
            agent_list: The list of agents.
        Returns:
            A agent randomly chosen from agent_list.
        """
        return random.choice(agent_list) if agent_list else AGENT_NONE


    def get_co_players(self, agent_list: List[Agent], role: Role = Role.ANY) -> List[Agent]: #特定の役職を持つエージェントのリストを取得するメソッド
        """Return a list of agents who have claimed the given role.
        Args:
            agent_list: The list of agents.
            role: The role. If Role.ANY, return all agents who have claimed any role.
        Returns:
            A list of agents who have claimed the given role.
        """
        """与えられたロールを要求したエージェントのリストを返します。
        引数
            agent_list: エージェントのリスト。
            role: エージェントのリスト： ロール。Role.ANYの場合、何らかのロールを要求している全てのエージェントを返します。
        戻り値
            与えられたロールを要求したエージェントのリスト。
        """
        return [a for a in agent_list if (role == Role.ANY and self.comingout_map[a] != Role.UNC) or self.comingout_map[a] == role]


    def get_counterparts(self, agent_list: List[Agent], agent: Agent) -> List[Agent]: # 特定のエージェントと同じ役職を持つエージェントのリストを取得するメソッド
        """Return a list of agents who have claimed the same role as the given agent.
        Args:
            agent_list: The list of agents.
            agent: The agent.
        Returns:
            A list of agents who have claimed the same role as the given agent.
            The given agent is not included in the returned list.
        """
        """指定されたエージェントと同じロールを要求しているエージェントのリストを返します。
        引数
            agent_list: エージェントのリスト。
            エージェント： エージェント。
        戻り値
            指定されたエージェントと同じロールを要求しているエージェントのリスト。
            指定されたエージェントは、返されるリストに含まれません。
        """
        role: Role = self.comingout_map[agent]
        if agent == AGENT_NONE or role == Role.UNC:
            return []
        return self.get_co_players([a for a in agent_list if a != agent], role)


    def convert_to_agentids(self, agent_list: List[Agent]) -> List[int]:
        return [f"Agent[{a.agent_idx}]" for a in agent_list]


    @property
    def alive_comingout_map(self) -> DefaultDict[Agent, Role]:
        return {a: r for a, r in self.comingout_map.items() if self.is_alive(a) and r != Role.UNC}
    """生存しているエージェントの中で、CO（Coming Out）情報が登録されているエージェントとそのCO情報（役職）をマップ形式で取得します。ただし、CO情報が Role.UNC でないエージェントのみが対象です"""


    @property
    def alive_comingout_map_str(self) -> DefaultDict[str, str]:
        return {a.agent_idx: r.value for a, r in self.alive_comingout_map.items() if self.is_alive(a) and r != Role.UNC}
    """alive_comingout_map プロパティと同様の情報を取得しますが、こちらはエージェントのIDとCO情報を文字列形式で取得します"""


    @property
    def will_vote_reports_str(self) -> Dict[str, str]:
        return {a.agent_idx: t.agent_idx for a, t in self.will_vote_reports.items()}
    """投票宣言（Will Vote Reports）情報をエージェントのIDと投票対象のエージェントのIDの組み合わせの形式で取得します"""


    def agent_to_index(self, agent_list: List[Agent]) -> List[int]:
        return [a.agent_idx for a in agent_list]
    """ エージェントのリストから、各エージェントのIDを整数値のリストとして取得します"""


    def vote_to_dict(self, vote_list: List[Vote]) -> Dict[int, int]:
        return {v.agent.agent_idx: v.target.agent_idx for v in vote_list}
    """ 投票情報（Vote オブジェクトのリスト）をエージェントのIDと投票対象のエージェントのIDを対応させた辞書形式で取得します"""


    def vote_cnt(self, vote_list: List[Vote]) -> Dict[Agent, int]:
        count: DefaultDict[Agent, int] = defaultdict(int)
        vote_dict = self.vote_to_dict(vote_list)
        for talker, target in vote_dict.items():
            count[target] += 1
        return count
    """投票情報（Vote オブジェクトのリスト）を元に、各エージェントがどれだけの投票を受けたかを集計します。各エージェントのIDと投票数を対応させた辞書を取得します"""


    def vote_print(self, agent_int: DefaultDict[Agent, int]) -> None:
        return {a.agent_idx: i for a, i in agent_int.items()}
    """vote_cnt メソッドで得られたエージェントのIDと受けた投票数の情報を整理し、エージェントのIDと対応する投票数を対応させた辞書形式で取得します"""


    # include_listから、exclude_listを除いた中で、最も処刑されそうなエージェントを返す
    # 注意：include_listのエージェントが、投票対象に含まれていない場合、自分を返す
    # デフォルトのinclude_listは自分を除いている
    # 注意：is_Low_HPで判定する時は、include_listを指定して自分を含める
    def chooseMostlikelyExecuted(self, include_list: List[Agent] = None, exclude_list: List[Agent] = None) -> Agent:
        if include_list is None:
            include_list = self.get_alive_others(self.game_info.agent_list)
        count: DefaultDict[Agent, int] = defaultdict(int)
        for talker, target in self.will_vote_reports.items():
            if target not in include_list:
                continue
            if exclude_list is not None and target in exclude_list:
                continue
            if self.is_alive(talker) and self.is_alive(target):
                count[target] += 1
        if self.vote_candidate != AGENT_NONE and self.vote_candidate in include_list:
            count[self.vote_candidate] += 1
        count_num = {a.agent_idx: t for a, t in count.items()}
        Util.debug_print("count2:\t", count_num)
        ret_agent: Agent = max(count.items(), key=lambda x: x[1])[0] if count else AGENT_NONE
        return ret_agent


    # include_listから、exclude_listを除いた中で、最も処刑されそうなエージェントを返す
    # 投票宣言と投票先の一致率を反映する
    def chooseMostlikelyExecuted2(self, include_list: List[Agent] = None, exclude_list: List[Agent] = None) -> Agent:
        if include_list is None:
            include_list = self.get_alive_others(self.game_info.agent_list)
        count: DefaultDict[Agent, float] = defaultdict(float)
        for talker, target in self.will_vote_reports.items():
            if target not in include_list:
                continue
            if exclude_list is not None and target in exclude_list:
                continue
            if self.is_alive(talker) and self.is_alive(target):
                vote_count = Util.vote_count[talker]
                vote_match_count = Util.vote_match_count[talker]
                if vote_count > 0:
                    count[target] += vote_match_count/vote_count
        if self.vote_candidate != AGENT_NONE and self.vote_candidate in include_list:
            count[self.vote_candidate] += 1.0
        ret_agent: Agent = max(count.items(), key=lambda x: x[1])[0] if count else AGENT_NONE
        Util.debug_print("executed2:\t", {a.agent_idx: np.round(t, 3) for a, t in count.items()})
        return ret_agent


    # HPが低いかどうか
    def is_Low_HP(self) -> bool:
        # 投票意思：投票が生存者の50%以上で、自分が最も処刑されそうな場合
        alive_cnt = len(self.game_info.alive_agent_list)
        if alive_cnt == 0:
            return False
        will_vote_cnt = len(self.will_vote_reports)
        rate = will_vote_cnt/alive_cnt
        alive_agents: List[Agent] = self.get_alive(self.game_info.agent_list)
        if rate >= 0.5 and self.chooseMostlikelyExecuted2(include_list=alive_agents) == self.me:
            Util.debug_print("is_Low_HP: will_vote")
            return True
        # 前日投票：前日投票の20%以上がが自分に入っている場合
        # latest_vote_listは、day_startで[]となっているため、前日の投票はvote_listに入っている
        vote_cnt = 0
        vote_list = self.game_info.vote_list
        vote_len = len(vote_list)
        if vote_len == 0:
            return False
        for vote in vote_list:
            if vote.target == self.me:
                vote_cnt += 1
        rate = vote_cnt/vote_len
        if rate >= 0.2:
            Util.debug_print("is_Low_HP: latest_vote")
            return True
        return False


    # 同数投票の時に自分の捨て票を変更する：最大投票以外のエージェントに投票している場合、投票先を変更する
    def changeVote(self, vote_list: List[Vote], role: Role, mostlikely=True) -> Agent:
        count: DefaultDict[Agent, int] = defaultdict(int)
        count_num: DefaultDict[str, int] = defaultdict(int)
        my_target: Agent = AGENT_NONE
        new_target: Agent = AGENT_NONE
        for vote in vote_list:
            agent = vote.agent
            target = vote.target
            no = str(target.agent_idx)
            if agent == self.me:
                my_target = target
            count[target] += 1
            count_num[no] += 1
        Util.debug_print('count_num:\t', count_num)
        # 最大投票数を取得
        max_vote = max(count_num.values())
        max_voted_agents: List[Agent] = []
        for agent, num in count.items():
            if num == max_vote and agent != self.me:
                max_voted_agents.append(agent)
        max_voted_agents_num = [a.agent_idx for a in max_voted_agents]
        Util.debug_print('max_voted_agents:\t', max_voted_agents_num)
        # 最大投票数のエージェントが複数人の場合
        if max_voted_agents:
            if mostlikely:
                new_target = self.role_predictor.chooseMostLikely(role, max_voted_agents)
            else:
                new_target = self.role_predictor.chooseLeastLikely(role, max_voted_agents)
        if new_target == AGENT_NONE:
            new_target = my_target
        Util.debug_print('vote_candidate:\t', my_target, '→', new_target)
        return new_target if new_target != AGENT_NONE else self.me


    # 初期化
    def initialize(self, game_info: GameInfo, game_setting: GameSetting) -> None:
        self.game_info = game_info
        self.game_setting = game_setting
        self.me = game_info.me
        # Clear fields not to bring in information from the last game.
        self.comingout_map.clear()
        self.divination_reports.clear()
        self.identification_reports.clear()
        self.talk_list_head = 0
        # 統計
        Util.game_count += 1
        
        self.will_vote_reports.clear()
        self.talk_list_all = []
        self.talk_turn = 0
        self.score_matrix = ScoreMatrix(game_info, game_setting, self)
        self.role_predictor = RolePredictor(game_info, game_setting, self, self.score_matrix)
        self.N = game_setting.player_num
        self.M = len(game_info.existing_role_list)
        self.agent_idx_0based = self.me.agent_idx - 1
        self.info=""
        self.my_turn=0
        self.bdi_history = []
        self.vote_target=[]
        self.vote_counter=0
        self.my_action=[]

        ActionLogger.initialize(game_info, game_setting)
        
        # Util.debug_print("game:\t", Util.game_count)
        Util.debug_print("game:\t", Util.game_count - 1)
        Util.debug_print("my role:\t", game_info.my_role)
        Util.debug_print("my idx:\t", self.me)


    # 昼スタート
    def day_start(self) -> None:
        self.talk_list_head = 0
        self.vote_candidate = AGENT_NONE
        day: int = self.game_info.day
        self.my_turn=0
        if day >= 2:
            vote_list: List[Vote] = self.game_info.vote_list
            Util.debug_print('vote_list:', self.vote_to_dict(vote_list))
            Util.debug_print('will_vote_reports:', self.will_vote_reports_str)
            for v in vote_list:
                self.score_matrix.vote(self.game_info, self.game_setting, v.agent, v.target, v.day)
                va = v.agent
                vt = v.target
                if va in self.will_vote_reports:
                    Util.vote_count[va] += 1
                    if vt == self.will_vote_reports[va]:
                        Util.vote_match_count[va] += 1
            Util.debug_print("vote_count:\t", self.vote_print(Util.vote_count))
            Util.debug_print("vote_match_count:\t", self.vote_print(Util.vote_match_count))
        self.will_vote_reports.clear()
        
        Util.debug_print("")
        Util.debug_print("DayStart:\t", self.game_info.day)
        Util.debug_print("生存者数:\t", len(self.game_info.alive_agent_list))
        
        Util.debug_print("Executed:\t", self.game_info.executed_agent)
        if self.game_info.executed_agent == self.me:
            Util.debug_print("---------- 処刑された ----------")
        # self.game_info.last_dead_agent_list は昨夜殺されたエージェントのリスト
        # (self.game_info.executed_agent が昨夜処刑されたエージェント)
        killed: List[Agent] = self.game_info.last_dead_agent_list
        if len(killed) > 0:
            self.score_matrix.killed(self.game_info, self.game_setting, killed[0])
            Util.debug_print("Killed:\t", self.game_info.last_dead_agent_list[0])
            if self.game_info.last_dead_agent_list[0] == self.me:
                Util.debug_print("---------- 噛まれた ----------")
            # 本来複数人殺されることはないが、念のためkilled()は呼び出した上でエラーログを出しておく
            if len(killed) > 1:
                Util.error_print("Killed:\t", *self.game_info.last_dead_agent_list)
        else:
            Util.debug_print("Killed:\t", AGENT_NONE)
        # 噛まれていない違和感を反映
        self.score_matrix.Nth_day_start(self.game_info, self.game_setting)

        #エージェントの状態
        self.info+=self.gpt_info()



    # ゲーム情報の更新
    # talk-listの処理
    def update(self, game_info: GameInfo) -> None:
        self.game_info = game_info  # Update game information.
        self.score_matrix.update(game_info)
        Util.start_timer("Villager.update")
        timeout = 20
        for i in range(self.talk_list_head, len(game_info.talk_list)):  # Analyze talks that have not been analyzed yet.
            tk: Talk = game_info.talk_list[i]  # The talk to be analyzed.
            day: int = tk.day
            turn: int = tk.turn
            talker: Agent = tk.agent
            self.talk_list_all.append(tk)
            
            if talker == self.me:  # Skip my talk.
                continue
            if RealPossessedDetector.should_skip(self, tk):
                continue
            # 内容に応じて更新していく
            content: Content = Content.compile(tk.text)

            Util.debug_print("Topic:\t", talker, content.topic)

            # content.target が不要な場合デフォルトの AGENT_ANY が入っている
            # 不正な場合はここで弾く
            if content.target == AGENT_NONE or (content.target != AGENT_ANY and content.target.agent_idx > self.N):
                Util.debug_print("Invalid target:\t", content.target)
                continue
            # target が不要なトピック以外で AGENT_ANY が入っている場合は弾く
            if content.target == AGENT_ANY:
                if content.topic not in [Topic.Over, Topic.Skip, Topic.OPERATOR, Topic.AGREE, Topic.DISAGREE, Topic.DUMMY]:
                    Util.debug_print("Invalid target:\t", content.topic, content.target)
                    continue

            if content.topic == Topic.COMINGOUT:
                if content.role in self.game_info.existing_role_list: # Role.UNC 対策
                    self.comingout_map[talker] = content.role
                    self.score_matrix.talk_co(self.game_info, self.game_setting, talker, content.role, day, turn)
                Util.debug_print("CO:\t", talker, content.role)
            elif content.topic == Topic.DIVINED:
                self.score_matrix.talk_divined(self.game_info, self.game_setting, talker, content.target, content.result, day, turn)
                self.divination_reports.append(Judge(talker, day, content.target, content.result))
                Util.debug_print("DIVINED:\t", talker, content.target, content.result)
            elif content.topic == Topic.IDENTIFIED:
                self.score_matrix.talk_identified(self.game_info, self.game_setting, talker, content.target, content.result, day, turn)
                self.identification_reports.append(Judge(talker, day, content.target, content.result))
                Util.debug_print("IDENTIFIED:\t", talker, content.target, content.result)
            elif content.topic == Topic.VOTE:
                # 古い投票先が上書きされる前にスコアを更新 (2回以上投票宣言している場合に信頼度を下げるため)
                self.score_matrix.talk_will_vote(self.game_info, self.game_setting, talker, content.target, day, turn)
                # 投票先を保存
                self.will_vote_reports[talker] = content.target
            elif content.topic == Topic.VOTED:
                self.score_matrix.talk_voted(self.game_info, self.game_setting, talker, content.target, day, turn)
            elif content.topic == Topic.GUARDED:
                self.score_matrix.talk_guarded(self.game_info, self.game_setting, talker, content.target, day, turn)
                Util.debug_print("GUARDED:\t", talker, content.target)
            elif content.topic == Topic.ESTIMATE:
                if content.role == Role.WEREWOLF:
                    self.score_matrix.talk_will_vote(self.game_info, self.game_setting, talker, content.target, day, turn)
                    self.will_vote_reports[talker] = content.target
                elif content.role == Role.VILLAGER:
                    self.score_matrix.talk_estimate(self.game_info, self.game_setting, talker, content.target, content.role, day, turn)
            elif content.topic == Topic.OPERATOR and content.operator == Operator.REQUEST and content.content_list[0].topic == Topic.VOTE:
                self.score_matrix.talk_will_vote(self.game_info, self.game_setting, talker, content.content_list[0].target, day, turn)
                self.will_vote_reports[talker] = content.content_list[0].target
            
            action: Action = ActionLogger.update(game_info, tk, content, self)
            score = ActionLogger.get_score(day, turn, talker, action)
            self.score_matrix.apply_action_learning(talker, score)

            if action in [Action.DIVINED_CONTRADICT, Action.DIVINED_WITHOUT_CO, Action.IDENTIFIED_WITHOUT_CO, Action.IDENTIFIED_WITHOUT_CO_TO_COUNTERPART, Action.IDENTIFIED_TO_ALIVE]:
                Util.debug_print("Action:\t", talker, action)

            self.talk_list_head += 1

            if Util.timeout("Villager.update", timeout):
                break



    # CO、投票宣言
    def talk(self):
        day: int = self.game_info.day
        turn: int = self.talk_turn        
        #self.vote_candidate = self.vote()

        
        talk_history = self.history(self.talk_list_all)
        talk_history_bdi = talk_history[self.count:]
        self.count = len(talk_history)
        for talk in talk_history_bdi:
            print("talk:",talk)
            if talk["agent"]==self.me:
                continue
        #print("talk_history:",talk_history)
        print("turn:",self.my_turn)
        print("info:",self.info)
        if turn<=6:
            #time.sleep(20)
            self.my_turn+=1
            #uttr=self.gpt_talk(talk_history,self.me,self.info,day)
            #time.sleep(10)
            #plan=self.langchain_plan(self.me,day,talk_history,self.info)
            #summary = self.langchain_talk_summary(self.me,day,talk_history_bdi,self.info)
            for text in talk_history_bdi:
                print("text:",text)
                if text["text"]=="Skip" or text["text"]=="Over":
                    continue
                else:
                    talk_to_bdi = self.langchain_text_to_bdi_json("Agent["+str(text["agent"])+"]の発言:"+text["text"])
                    with open('./log/bdi_log{}.txt'.format(Util.game_count-1),mode='a',encoding='UTF-8') as f:
                        f.write("day:"+str(self.game_info.day)+"turn:"+str(self.talk_turn)+'\n')
                        f.write("Agent["+str(text["agent"])+"]の発言:"+str(text["text"])+'\n')
                        f.write(str(talk_to_bdi)+'\n')
                    self.bdi_history.append(talk_to_bdi["bdi"])
            def_bdi_history = self.def_bdi_history(self.bdi_history)
            print(def_bdi_history)
            action = self.langchain_bdi_action(self.me,day,def_bdi_history,self.info,self.my_action)
            self.my_action.append(action["action"])
            print(self.my_action)
            with open('./log/action_log{}.txt'.format(Util.game_count-1),mode='a',encoding='UTF-8') as f:
                f.write("day:"+str(self.game_info.day)+"turn:"+str(self.talk_turn)+'\n')
                f.write(str(action)+'\n')
            uttr = self.langchain_bdi_to_text_json(self.me,action["action"])
            #uttr=self.langchain_talk(self.me,day,talk_history,self.info)
            return uttr["text"]
        else:
            self.my_turn+=1
            return "Skip"



    # 投票対象
    def vote(self) -> Agent:
        #  ---------- 同数投票の処理 ----------
        latest_vote_list = self.game_info.latest_vote_list
        if latest_vote_list:
            self.vote_candidate = self.changeVote(latest_vote_list, Role.WEREWOLF)
            return self.vote_candidate if self.vote_candidate != AGENT_NONE else self.me
        # 投票候補
        vote_candidates: List[Agent] = self.get_alive_others(self.game_info.agent_list)
        #talk_history = self.history(self.talk_list_all)
        day: int = self.game_info.day
        # ---------- 5人村 ----------
        if self.N != 15:
            #self.vote_candidate = self.role_predictor.chooseMostLikely(Role.WEREWOLF, vote_candidates)
            vote_candidate=self.langchain_vote_bdi_json(self.me,day,self.bdi_history,self.info,self.agent_to_index(vote_candidates),self.my_action)
            print(vote_candidate)
            with open('./log/vote_log{}.txt'.format(Util.game_count-1),mode='a',encoding='UTF-8') as f:
                f.write("day:"+str(self.game_info.day)+'\n')
                f.write(str(vote_candidate)+'\n')
            if type(vote_candidate["target"])==str:
                candidate = re.sub(r"\D","",vote_candidate["target"])
                num=[int(x) for x in list(str(candidate))]
                if len(num)==2:
                    candidate=int(num[1])
                elif len(num)==1:
                    candidate=int(num[0])
                else:
                    candidate = random.choice(vote_candidates)
                self.vote_candidate = candidate
            else:
                candidate = vote_candidate["target"]
                num=[int(x) for x in list(str(candidate))]
                if len(num)==2:
                    candidate=int(num[1])
                else:
                    candidate=int(num[0])
                self.vote_candidate = candidate
            print("vote_candidate:",self.vote_candidate)
            print(self.agent_to_index(vote_candidates))
            if self.vote_candidate in self.agent_to_index(vote_candidates):
                for i in vote_candidates:
                    if i.agent_idx==self.vote_candidate:
                        target=i
            else:
                if len(vote_candidates)!=0:
                    target = random.choice(vote_candidates)
                    self.vote_target.append(target)
                    return target
                else:
                    choice_agent:List[Agent]=self.get_alive(self.game_info.agent_list)
                    target= random.choice(choice_agent)
                    self.vote_target.append(target)
                    return target
            print("vote_target:",target)
            print("vote_target:",target)
            self.vote_target.append(target)
            return target
        # ---------- 15人村 ----------
        elif self.N == 15:
            # 投票対象の優先順位：偽占い→人狼っぽいエージェント
            fake_seers: List[Agent] = [j.agent for j in self.divination_reports if j.target == self.me and j.result == Species.WEREWOLF]
            alive_fake_seers: List[Agent] = self.get_alive_others(fake_seers)
            if alive_fake_seers:
                Util.debug_print("alive_fake_seers:\t", self.agent_to_index(alive_fake_seers))
                self.vote_candidate = self.role_predictor.chooseMostLikely(Role.WEREWOLF, alive_fake_seers)
            else:
                Util.debug_print("vote_candidates:\t", self.agent_to_index(vote_candidates))
                self.vote_candidate = self.role_predictor.chooseMostLikely(Role.WEREWOLF, vote_candidates)
        # ----- 投票ミスを防ぐ -----
        if self.vote_candidate == AGENT_NONE or self.vote_candidate == self.me:
            Util.debug_print("vote_candidates: AGENT_NONE or self.me")
            self.vote_candidate = self.role_predictor.chooseMostLikely(Role.WEREWOLF, vote_candidates)
        return self.vote_candidate if self.vote_candidate != AGENT_NONE else self.me


    def attack(self) -> Agent:
        raise NotImplementedError()


    def divine(self) -> Agent:
        raise NotImplementedError()


    def guard(self) -> Agent:
        raise NotImplementedError()


    def whisper(self) -> Content:
        raise NotImplementedError()


    def finish(self) -> None:
        for i in self.vote_target:
            if self.game_info.role_map[i]==Role.WEREWOLF:
                self.vote_werewolf+=1
            elif self.game_info.role_map[i]==Role.VILLAGER:
                self.vote_villager+=1
            elif self.game_info.role_map[i]==Role.POSSESSED:
                self.vote_possessed+=1
            elif self.game_info.role_map[i]==Role.SEER:
                self.vote_seer+=1
        self.vote_counter=self.vote_werewolf+self.vote_villager+self.vote_possessed+self.vote_seer
        vote_list: List[Vote] = self.game_info.vote_list
        Util.debug_print('vote_list:', self.vote_to_dict(vote_list))
        Util.debug_print('will_vote_reports:', self.will_vote_reports_str)
        for v in vote_list:
            self.score_matrix.vote(self.game_info, self.game_setting, v.agent, v.target, v.day)
            va = v.agent
            vt = v.target
            if va in self.will_vote_reports:
                Util.vote_count[va] += 1
                if vt == self.will_vote_reports[va]:
                    Util.vote_match_count[va] += 1
        Util.debug_print("vote_count:\t", self.vote_print(Util.vote_count))
        Util.debug_print("vote_match_count:\t", self.vote_print(Util.vote_match_count))
        # 自分が人狼陣営で人狼が生存しているか、自分が村人陣営で人狼が生存していない場合に勝利
        alive_wolves = [a for a in self.game_info.alive_agent_list if self.game_info.role_map[a] == Role.WEREWOLF]
        villagers_win = (len(alive_wolves) == 0)
        is_villagers_side = self.game_info.my_role in [Role.VILLAGER, Role.SEER, Role.MEDIUM, Role.BODYGUARD]
        Util.update_win_rate(self.game_info, villagers_win)

        Util.debug_print("")
        Util.debug_print("win:\t", is_villagers_side == villagers_win)
        Util.debug_print("win_rate:\t", Util.win_count[self.me], "/", Util.game_count, " = ", Util.win_rate[self.me])
        if self.game_info.role_map[self.me]==Role.VILLAGER:
            print("                     vote_rate  vote_role_counter  vote_counter")
            print("werewolf_vote_rate:",self.vote_werewolf/self.vote_counter,self.vote_werewolf,self.vote_counter)
            print("villager_vote_rate:",self.vote_villager/self.vote_counter,self.vote_villager,self.vote_counter)
            print("possessed_vote_rate:",self.vote_possessed/self.vote_counter,self.vote_possessed,self.vote_counter)
            print("seer_vote_rate:",self.vote_seer/self.vote_counter,self.vote_seer,self.vote_counter)
            with open('./log/vote_rate_log{}.txt'.format(Util.game_count-1),mode='a',encoding='UTF-8') as f:
                f.write("               vote_rate vote_role_counter vote_counter"'\n')
                f.write("werewolf_vote_rate:"+str(self.vote_werewolf/self.vote_counter)+" "+str(self.vote_werewolf)+" "+str(self.vote_counter)+'\n')
                f.write("villager_vote_rate:"+str(self.vote_villager/self.vote_counter)+" "+str(self.vote_villager)+" "+str(self.vote_counter)+'\n')
                f.write("possessed_vote_rate:"+str(self.vote_possessed/self.vote_counter)+" "+str(self.vote_possessed)+" "+str(self.vote_counter)+'\n')
                f.write("seer_vote_rate:"+str(self.vote_seer/self.vote_counter)+" "+str(self.vote_seer)+" "+str(self.vote_counter)+'\n')
        Util.debug_print("")

        ActionLogger.finish(self.game_info)
        self.score_matrix.finish(self.game_info)

        if (len(self.role_predictor.assignments) == 0):
            Util.debug_print("No assignments")
            return

        # 確率を表示
        p = self.role_predictor.getProbAll()
        Util.debug_print("", end="\t")
        for r in self.game_info.existing_role_list:
            Util.debug_print(r.name, end="\t")
        Util.debug_print("")
        for a in self.game_info.agent_list:
            Util.debug_print(a, end="\t")
            for r in self.game_info.existing_role_list:
                if p[a][r] > 0.005:
                    Util.debug_print(round(p[a][r], 2), end="\t")
                else:
                    Util.debug_print("-", end="\t")
            Util.debug_print("")
        Util.debug_print("")

        # COを表示
        for a, r in self.comingout_map.items():
            if r != Role.UNC:
                Util.debug_print("CO:\t", a, r)
        Util.debug_print("")

        # 実際の割り当てと予測の割り当てを比較
        assignment = []
        for a, r in self.game_info.role_map.items():
            assignment.append(r)
        actual_assignment = Assignment(self.game_info, self.game_setting, self, assignment)
        predicted_assignment = self.role_predictor.assignments[-1]
        Util.debug_print("\t", "1, 2, 3, 4, 5, 6, 7, 8, 9, A, B, C, D, E, F")
        Util.debug_print("actual:\t", actual_assignment)
        Util.debug_print("predicted:\t", predicted_assignment)
        
        # 予測の割り当てのスコアを表示 (デバッグモード)
        predicted_assignment.evaluate(self.score_matrix, debug=True)
        Util.debug_print("")
        Util.debug_print("best score:\t", round(predicted_assignment.score, 4))
        Util.debug_print("")

        # 実際の割り当てのスコアを表示 (デバッグモード)
        actual_assignment.evaluate(self.score_matrix, debug=True)
        Util.debug_print("")
        Util.debug_print("actual score:\t", round(actual_assignment.score, 4))

        # COしていない人から占い師、霊媒師、狩人が選ばれてはいないかのチェック
        for a in self.game_info.agent_list:
            if predicted_assignment[a] in [Role.SEER, Role.MEDIUM, Role.BODYGUARD] and predicted_assignment[a] != self.comingout_map[a]:
                Util.debug_print(a, "CO", self.comingout_map[a] if a in self.comingout_map else Role.UNC, "but assigned", predicted_assignment[a])
        Util.debug_print("")

        pass
    
    #エージェントの状態の作成
    def gpt_info(self):
        info_day=self.game_info.day
        info_alive=""
        for i in self.game_info.alive_agent_list:
            info_alive+=str(i)+","
        print(info_alive)
        info_executed=self.game_info.executed_agent
        info_attacked=None
        if len(self.game_info.last_dead_agent_list)!=0:
            info_attacked=self.game_info.last_dead_agent_list[0]
        text="start_day:"+str(info_day)+","+"alive_agent:"+info_alive+"yesterday_executed_agent:"+str(info_executed)+","+"yesterday_attacked_agent:"+str(info_attacked)+"\n"
        

        return text
    
    #トーク履歴作成
    def history(self,talk_list_all):
        #chatgptに渡す準備
        talk=[]
        dic={"day":-1,"turn":-1,"agent":-1,"text":""}
        talk_day=-1
        talk_turn=-1
        talk_agent=-1
        talk_text=""
        for i in talk_list_all:
            if talk_day==i.day and talk_turn==i.turn and talk_agent==i.agent._agent_idx and talk_text==i.text:
                continue
            else:
                dic['day']=i.day
                dic['turn']=i.turn
                dic['agent']=i.agent._agent_idx
                dic['text']=i.text
                talk.append(copy.deepcopy(dic))
                talk_day=i.day
                talk_turn=i.turn
                talk_agent=i.agent._agent_idx
                talk_text=i.text
        print(talk)
        return talk
    
    def def_bdi_history(self,bdi_list):
        #chatgptに渡す準備
        talk=[]
        dic={"bdi":""}
        for i in bdi_list:
            dic['bdi']=i
            talk.append(copy.deepcopy(dic))

        return talk
    
    def gpt_json(self,content):
        messages = [
            {"role":"system","content":"あなたは返答をすべてJSON形式で出力します。"},
            {"role":"user","content":content}
        ]
        response = openai.ChatCompletion.create(
            model="gpt-3.5-turbo",
            response_format={"type":"json_object"},
            messages=messages,
            functions=[
                {
                    "name":"utter",
                    "description":"抽出された情報をJSONとして処理します。",
                    "parameters":{
                        "type":"object",
                        "properties":{
                            "me":{
                                "type":"string",
                                "description":"自身のエージェント番号です。"
                            },
                            "target":{
                                "type":"string",
                                "description":"発言対象のエージェント番号です。対象がANYの場合はANY、いない場合はNONEとしてください。"
                            },
                            "topic":{
                                "type":"string",
                                "description":"発言内容の種類です。DUMMY:ダミートピック,ESTIMATE:推測,COMINGOUT:カミングアウト,DIVINATION:占う,DIVINED:占い結果の報告,IDENTIFIED:特定した役職の報告,GUARD:ガード,GUARDED:ガード結果の報告,VOTE:投票,VOTED:投票結果の報告,ATTACK:攻撃,ATTACKED:攻撃結果の報告,AGREE:賛成,DISAGREE:反対,Over:発言の終了,Skip:発言のスキップ,OPERATOR:オペレーターの中から選択してください。"
                            },
                            "text":{
                                "type":"string",
                                "description":"発言内容です。発話内容を日本語でそのまま入力してください。"
                            },
                        },
                    
                    },
                }
            ],
            temperature=0,
            function_call={"name":"utter"},
        )
        message = response["choices"][0]["message"]
        return json.loads(message["function_call"]["arguments"])
        
    @retry()
    def langchain_plan(self,my_id,day,history,info):
                # OpenAIのモデルのインスタンスを作成
        chat = ChatOpenAI(model_name=gpt4, temperature=1.0,request_timeout=180)

        prompt = PromptTemplate.from_template(
                "あなたはAgent[{my_id}]というプレイヤーで、人狼というゲームを5人でプレイしています。\n"
            + " ゲームルールは {rule} です。\n"
            + "あなたの役職は村人です。あなたは何も能力を持たないプレイヤーです。\n"
            +"今は{day}日目です。\n"
            +"現在までの各エージェントの会話は{history}です。\n"
            +"現在の各エージェントの状態は{info}です。"
            + " 与えられた情報をすべて理解したうえで、これからの戦略を詳細に考えてください。"
            +"あなたの戦略で勝利に導いてください。"
            )

        kwargs = dict(
            my_id=my_id,
            rule="""5人のプレイヤーで村を構成する。村における役職とその人数は、「村人:2人,占い師:1人,人狼:1人,裏切者:1人」である。
                村人側の役職
                村人:何も能力を持たない村人側のプレイヤー
                占い師:1日の終わりに1人のプレイヤーを占い、そのプレイヤーが人間であるか人狼であるかを知ることができる。
                
                人狼側の役職
                人狼:1日の終わりに各人狼は人間を1人選択して襲撃投票し、最も多く襲撃されたプレイヤーを襲撃する。
                裏切者:村人と同じく能力は何もない人間だが、人狼の勝利を目指して行動する。占い師の能力では人間と判定される。
                追放のための投票は昼フェーズの最後に行われます。 有効な投票先（自分以外の生存プレイヤー）以外に投票することは可能ですが， その場合はランダムに投票先が選ばれます。

                追放投票と再投票
                投票によって決定した追放者の情報は， 夜フェーズの占い・護衛・襲撃投票に利用することが可能です。
                最多得票者が複数となった場合は1回に限り再投票となります。
                再投票の前に対話はできません。 再投票でも同点だった場合は再投票での最多得票者からランダムに追放者が決定されます。 
                再投票では，最多得票者も投票権を持ち，投票者は最多得票者以外にも投票可能です。""",
            day=day,
            history=history,
            info=info,
        )


        plan_prediction_chain = LLMChain(llm=chat, prompt=prompt)
        plan = plan_prediction_chain.run(**kwargs)
        plan = plan.strip()
        #self.plan=plan

        print(plan.strip())
        return plan.strip()
    @retry()
    def langchain_talk(self,my_id,day,history,info):
        # OpenAIのモデルのインスタンスを作成
        chat = ChatOpenAI(model_name=gpt4, temperature=1.0,request_timeout=30)

        prompt = PromptTemplate.from_template(
                "あなたはAgent[{my_id}]というプレイヤーで、人狼というゲームを5人でプレイしています。\n"
            + " ゲームルールは {rule} です。\n"
            + "あなたの役職は村人です。あなたは何も能力を持たないプレイヤーです。\n"
            +"今は{day}日目です。\n"
            +"現在までの各エージェントの会話は{history}です。\n"
            +"現在の各エージェントの状態は{info}です。\n"
            + " 与えられた情報をすべて理解したうえで、あなたの発言を１行で簡潔に考えてください。\n"
            +"発言を考える際に、ほかのエージェントと同じ発言や、自身がした発言と同じような発言はしないでください。\n"
            +"あなたの発言で勝利に導いてください。"
            )

        kwargs = dict(
            my_id=my_id,
            rule="""5人のプレイヤーで村を構成する。村における役職とその人数は、「村人:2人,占い師:1人,人狼:1人,狂人:1人」である。
                村人側の役職
                村人:何も能力を持たない村人側のプレイヤー
                占い師:1日の終わりに1人のプレイヤーを占い、そのプレイヤーが人間であるか人狼であるかを知ることができる。
                
                人狼側の役職
                人狼:1日の終わりに各人狼は人間を1人選択して襲撃投票し、最も多く襲撃されたプレイヤーを襲撃する。
                狂人:村人と同じく能力は何もない人間だが、人狼の勝利を目指して行動する。占い師の能力では人間と判定される。
                追放のための投票は昼フェーズの最後に行われます。 有効な投票先（自分以外の生存プレイヤー）以外に投票することは可能ですが， その場合はランダムに投票先が選ばれます。

                追放投票と再投票
                投票によって決定した追放者の情報は， 夜フェーズの占い・護衛・襲撃投票に利用することが可能です。
                最多得票者が複数となった場合は1回に限り再投票となります。
                再投票の前に対話はできません。 再投票でも同点だった場合は再投票での最多得票者からランダムに追放者が決定されます。 
                再投票では，最多得票者も投票権を持ち，投票者は最多得票者以外にも投票可能です。""",
            day=day,
            history=history,
            info=info
        )


        plan_prediction_chain = LLMChain(llm=chat, prompt=prompt)
        plan = plan_prediction_chain.run(**kwargs)
        plan = plan.strip()

        print(plan.strip())
        return plan.strip()
    
    @retry()
    def langchain_vote_json(self,my_id,day,history,info,vote_candidate):

        chat = ChatOpenAI(model_name=gpt4, temperature=1.0,request_timeout=60).bind(response_format={"type": "json_object"})
        
        prompt=PromptTemplate.from_template(
        "あなたはAgent[{my_id}]というプレイヤーで、人狼というゲームを5人でプレイしています。\n"
        +"あなたは返答をすべてJSON形式で出力します。\n"
        +"与えられた発言に対して、自身のエージェント番号、追放投票対称のエージェント番号、テキストをJSON形式でパースしてください。\n"
        +"JSONのキーはme,target,textとしてください。"
        +"meは発言者のエージェント番号とします。\n"
        +"targetは追放投票対称のエージェント番号とします。番号のみを出力してください。追放投票対象は{vote_candidate}から番号を必ず選んでください。\n"
        +"textは追放投票対象を選んだ理由です。\n"
        +"ゲームルールは {rule} です。\n"
        +"あなたの役職は村人です。\n"
        +"今は{day}日目です。\n"
        +"現在はだれに追放投票をするかを決めるターンです。追放投票が一番多かった人が処刑されます。\n"
        +"現在までの各エージェントの会話は{history}です。\n"
        +"現在の各エージェントの状態は{info}です。\n"
        +"与えられた情報をすべて理解したうえで、論理的に推論をして、だれに追放投票をすればいいかを必ず決めて上記のjson形式で返答してください。\n"
        +"情報が少ない場合でもだれに投票するかを必ず決めてください。"
        )

        kwargs = dict(
            my_id=my_id,
            rule="""5人のプレイヤーで村を構成する。村における役職とその人数は、「村人:2人,占い師:1人,人狼:1人,狂人:1人」である。
                村人側の役職
                村人:何も能力を持たない村人側のプレイヤー
                占い師:1日の終わりに1人のプレイヤーを占い、そのプレイヤーが人間であるか人狼であるかを知ることができる。
                
                人狼側の役職
                人狼:1日の終わりに各人狼は人間を1人選択して襲撃投票し、最も多く襲撃されたプレイヤーを襲撃する。
                狂人:村人と同じく能力は何もない人間だが、人狼の勝利を目指して行動する。占い師の能力では人間と判定される。
                追放のための投票は昼フェーズの最後に行われます。 有効な投票先（自分以外の生存プレイヤー）以外に投票することは可能ですが， その場合はランダムに投票先が選ばれます。

                追放投票と再投票
                投票によって決定した追放者の情報は， 夜フェーズの占い・護衛・襲撃投票に利用することが可能です。
                最多得票者が複数となった場合は1回に限り再投票となります。
                再投票の前に対話はできません。 再投票でも同点だった場合は再投票での最多得票者からランダムに追放者が決定されます。 
                再投票では，最多得票者も投票権を持ち，投票者は最多得票者以外にも投票可能です。""",
            day=day,
            history=history,
            info=info,
            vote_candidate=vote_candidate
        )
        plan_prediction_chain = LLMChain(llm=chat, prompt=prompt)
        plan = plan_prediction_chain.run(**kwargs)
        if plan is None:
            return {}
        else:
            return json.loads(plan)
        
    @retry()
    def langchain_text_to_bdi_json(self,text):

        chat = ChatOpenAI(model_name=gpt4, temperature=1.0,request_timeout=60).bind(response_format={"type": "json_object"})
        
        prompt=PromptTemplate.from_template(
        "あなたは翻訳者です。与えられた文章を与えられた規則をもとにbdiに変換します。\n"
        +"あなたは返答をすべてJSON形式で出力します。\n"
        +"与えられた発言に対して、変換した文、テキストをJSON形式でパースしてください。\n"
        +"JSONのキーはbdiとしてください。"
        +"bdiは与えられた文をbdiに変換したものとします。\n"
        +"変換規則と変換の例は {rule} です。\n"
        +"次の文を変換してください:{text}"
        +"与えられた情報をすべて理解したうえで、論理的に推論をして、上記のjson形式で返答してください。\n"
        )

        kwargs = dict(
            rule="""人狼ゲームにおける発話をBDI論理により変換するルールを以下に与えます。\n
                論理演算子:\n
                ￢:否定,￢p:pでない\n
                ∧:論理積,p∧q:pかつq\n
                ∨:論理和,p∨q:pまたはq\n
                →:含意,p→q:pならばq\n
                BDI論理のための基本オペレータ:\n
                様相:\nBEL a:aが信じる,DESIRE a:aが望む,INTEND a:aが意図する\n
                時相:\nA:すべての可能世界で,E:ある可能世界で,X:次の時点で,G:現時点を含み永遠に,F:現在を含む時点のいつか,U:条件が成立する時点まで,B:現在を含まない前の時点で\n
                情景描写と行為のためのオペレータ:\n
                IS文:\nIs(character,role):characterがroleである,Is(character/role,verb,character/role):character/roleがcomposantである\n
                Do文:\nDo(character/role,verb,character/role):character/roleがcharacter/roleをverbする,Do(character/role,verb,act):character/roleがactをverbする,Do(character/role,verb,IS()):character/roleがIS文をverbする\n
                基礎語:\n
                character:\nNAME:人物の名前が入る,anychar:すべてのcharacter,who:前に述べられている人物\n
                role:\nvillager:村人,seer:占い師,medium:霊媒師,hunter:狩人,freemason:共有者,wolf:狼,lunatic:狂人,HUMAN:villagerまたはseerまたはmediumまたはhunterまたはfreemasonまたはlunatic,VILLAGESIDE:villagerまたはseerまたはmediumまたはhunterまたはfreemason,WOLFSIDE:wolfまたはlunatic,GIFTED:seerまたはmediumまたはhunterまたはfreemason,ANYROLE:seerまたはmediumまたはhunterまたはfreemasonまたはlunatic\n
                以下に上記のルールを用いた変換例を与えます。\n
                クララの発言:\n
                やっぱヤコブが占い師だよなーこれ、と思いました。ヤコブが狂人とかありえないと思うし。んでシモンが狼か:BEL Klara(￢Is(Jacob,lunatic)→Is(Jacob,seer))→BEL Klara(Is(Simson,wolf))\n
                パメラの発言:\n
                私吊りが並んでいるな。これは何を言っても無駄か？:BEL Pamela(Do(∀people except Pamela,estimate,lunatic)→Do(∀people except Pamela,vote Pamela))→BEL Pamela(￢Do(Pamela,avoid,execution))\n
                カタリナの発言:\n
                まだ推測の枠を出ませんが、ニコラスが黒ならばオットーとトーマスの白の濃度は高い:BEL Katherine(Is(Nicholas,wolf)→Do(Katherine,estimate,Is(Otto∧Tomas,VILLAGESIDE))\n
                ヤコブの発言:\n
                となるとアルビン真,リーザ狂人、ディーター狼が本線だけど、それはリーザー吊ってから考えればいいのではなかろうか:BEL Jaocb(Is(Albin,seer)∧Is(Lisa,lunatic)∧Is(Diter,wolf))→DESIRE Jacob(Do(∀people,vote,Lisa))\n
                ヴァルターの発言:\n
                ああそうか まずモーリッツ吊っとけば良いのか 何やってるんだか吊り先 モーリッツにしとく:INTEND Walter(Do(Walter,vote,Molitz))\n
                モーリッツの発言:\n
                ディーター吊って黒判定出れば良いのでは。万が一、レジーナ喰われたらリーザ吊りで安泰:BEL Molitz(Do(∀people,vote,Diter)→(Do(∀people,know,Is(Diter,wolf)∨Do(∀people,know,￢Is(Diter,wolf))))∧(BEL Molitz(EX(Is(Regina,attacked)→Do(∀people,vote,Lisa)))\n
                カタリナの発言:\n
                ディーター吊りで良いと思っています。ディーター黒だと思いますし:BEL Katharina(BEL Katharina(Is(Diter,wolf))→BEL Katharina(Do(∀people,vote,Diter)))\n
                パメラの発言:\n
                ディーター吊りって全く意味が分からないんだが、お仕事終了した占い師残すメリット皆無では:BEL Pamela(￢Do(∀people,vote,Diter)→Do(∀people,vote,Lisa))→DESIRE Pamela(Do(∀people,vote,Lisa))\n
                カタリナの発言:\n
                トーマスさんが白なら狩の目は十分にありますし、噛まれるでしょう。今日は非狩目吊りで良いと思います:BEL Katherine(Is(Thomas,VILLAGESIDE)→BEL Katherine(Do(anychar,estimate,Is(Thomas,hunter)))∧BEL Katherine(Do(wolf,attack,Thomas))),DESIRE Katherine(Do(anychar,vote,￢hunter))\n
                モーリッツの発言:\n
                今日の方針として、ディーターを襲撃するかどうか？リーザが安全圏に残るのならありだと思うんですよ。:BEL Molitz(AX(Do(anychar,estimate,Is(Lisa,VILLAGESIDE))→DESIRE Molitz(Do(wolf,attack,Diter)))\n
                モーリッツの発言\n
                明日私が黒を引けた場合は私吊りとなるのは受け入れるし、偽占い師が黒特攻をした場合もトーマス吊りよりも先に黒出し占い師を吊ればいいだろう。:BEL Molitz(EX(Do(Molitz,divine,Is(who,wolf))→Is(Molitz,executed))),BEL Molitz(EX(Do(￢seer,divine,wolf)→￢Is(Thomas,executed)∧Do(anychar,vote,￢seer)))\n
                フリーデルの発言:\n
                たぶんシモン真アルビン狼、リーザは狂予想:BEL Friedel(Do(Friedel,estimate,Is(Simon,seer)∧Is(Albin,wolf)∧Is(Lisa,lunatic)))\n
                パメラの発言:\n
                霊能者に関しては無駄占い吊り避けたいし今日でとけばってカンジ:DESIRE Pamela(Do(VILLAGESIDE,avoid,Is(seer,executed))→DESIRE Pamela(Do(meduim,comingout,medium))\n
                以下は人狼の発言をBDIに変換した例である。\n
                トーマスの発言:\n
                私は村人側のふりをしていますが、実は人狼です。今日はパメラに投票するつもりです。:BEL Thomas(Do(Thomas,deceive,Is(Thomas,villager))∧BEL Thomas(Is(Thomas,wolf)),INTEND Thomas(Do(Thomas,vote,Pamela))\n
                モーリッツの発言:\n
                私は占い師だと偽って、ディーターを狼だと疑いかけて混乱を招こう。:INTEND Molitz(Do(Molitz,deceive,Is(Molitz,seer))∧Do(Molitz,tell,Is(Diter,wolf)))\n
                モーリッツの発言:\n
                トーマスさんが私を人狼と判定したと言っていますが、占い結果は確実ではありませんよ。それに、初日からこのような発言をする理由が私には見えないんですが、もしトーマスさんが狂人だとしたら、人狼側に有利な誤情報を流す意図はあるのでしょうか？:BEL Molitz(Do(Thomas,tell,Is(Molitz,wolf)→￢A(Is(Molitz,wolf))),BEL Molitz(Do(Molitz,estimate,Is(Thomas,lunatic))→INTEND Thomas(Do(Thomas,tell,misinformation))\n
                モーリッツの発言:\n
                私はトーマスの判定を疑います。普通、初日からこんなに早急に人狼だと断定はできないはずです。もしかしてトーマスさんは狂人かもしれませんね。:BEL Molitz(Do(Molitz,suspect,Thomas)→Do(Molitz,estimate,Is(Thomas,lunatic)))\n
                モーリッツの発言:\n
                トーマスが私を人狼だと判定しましたが、私はただの村人です。この状況では、人狼を探すのに占いは不可欠なので、占い師を保護するためにも、怪しい情報を流すトーマスの動向に注目するべきだと思います。:BEL Molitz(Is(Molitz,villager)),BEL Molitz(Do(Molitz,suspect,Thomas))\n
                ディーターの発言:\n
                オットーが出した人狼判定は狂人の可能性が高いので、僕たちは冷静に判断し証拠とロジックに基づいた議論を続けるべきです。:BEL Diter(Do(Otto,tell,Is(Pamela,wolf))→BEL Diter(Is(Pamela,lunatic)))→INTEND Diter(Do(anychar,discuss,Is(who,wolf)))\n
                オットーの発言:\n
                リーザがパメラを人間だと占った結果を出しているけど、パメラが自分で人狼だって言っているからその占い結果が信用できないね。:BEL Otto(Do(Pamela,tell,Is(Pamela,wolf))→￢BEL Otto(Do(Lisa,tell,Is(Pamela,HUMAN)))\n""",
            text=text,
        )
        plan_prediction_chain = LLMChain(llm=chat, prompt=prompt)
        plan = plan_prediction_chain.run(**kwargs)
        print(plan)
        if plan is None:
            return {}
        else:
            return json.loads(plan)
        

    @retry()
    def langchain_bdi_to_text_json(self,my_id,text):

        chat = ChatOpenAI(model_name=gpt4, temperature=1.0,request_timeout=60).bind(response_format={"type": "json_object"})
        
        prompt=PromptTemplate.from_template(
        "あなたは翻訳者です。与えられたbdiの文章を与えられた規則をもとに日本語の文に変換します。\n"
        +"あなたは返答をすべてJSON形式で出力します。\n"
        +"与えられた発言に対して、変換した文、テキストをJSON形式でパースしてください。\n"
        +"JSONのキーはtextとしてください。"
        +"textは与えられた文を日本語の文に変換したものとします。\n"
        +"変換規則と変換の例は {rule} です。\n"
        +"あなたはAgent[{my_id}]として人狼をプレイしています。次のBDI文を自然な形の日本語のゲーム中のあなたの発言として変換してください:{text}"
        +"与えられた情報をすべて理解したうえで、論理的に推論をして、上記のjson形式で返答してください。\n"
        )

        kwargs = dict(
            rule="""人狼ゲームにおける発話をBDI論理により変換するルールを以下に与えます。\n
                論理演算子:\n
                ￢:否定,￢p:pでない\n
                ∧:論理積,p∧q:pかつq\n
                ∨:論理和,p∨q:pまたはq\n
                →:含意,p→q:pならばq\n
                BDI論理のための基本オペレータ:\n
                様相:\nBEL a:aが信じる,DESIRE a:aが望む,INTEND a:aが意図する\n
                時相:\nA:すべての可能世界で,E:ある可能世界で,X:次の時点で,G:現時点を含み永遠に,F:現在を含む時点のいつか,U:条件が成立する時点まで,B:現在を含まない前の時点で\n
                情景描写と行為のためのオペレータ:\n
                IS文:\nIs(character,role):characterがroleである,Is(character/role,verb,character/role):character/roleがcomposantである\n
                Do文:\nDo(character/role,verb,character/role):character/roleがcharacter/roleをverbする,Do(character/role,verb,act):character/roleがactをverbする,Do(character/role,verb,IS()):character/roleがIS文をverbする\n
                基礎語:\n
                character:\nNAME:人物の名前が入る,anychar:すべてのcharacter,who:前に述べられている人物\n
                role:\nvillager:村人,seer:占い師,medium:霊媒師,hunter:狩人,freemason:共有者,wolf:狼,lunatic:狂人,HUMAN:villagerまたはseerまたはmediumまたはhunterまたはfreemasonまたはlunatic,VILLAGESIDE:villagerまたはseerまたはmediumまたはhunterまたはfreemason,WOLFSIDE:wolfまたはlunatic,GIFTED:seerまたはmediumまたはhunterまたはfreemason,ANYROLE:seerまたはmediumまたはhunterまたはfreemasonまたはlunatic\n
                以下に上記のルールを用いた変換例を与えます。\n
                クララの発言:\n
                やっぱヤコブが占い師だよなーこれ、と思いました。ヤコブが狂人とかありえないと思うし。んで神父が狼か:BEL Klara(￢Is(Jacob,lunatic)→Is(Jacob,seer))→BEL Klara(Is(Simson,wolf))\n
                パメラの発言:\n
                私吊りが並んでいるな。これは何を言っても無駄か？:BEL Pamela(Do(∀people except Pamela,estimate,lunatic)→Do(∀people except Pamela,vote Pamela))→BEL Pamela(￢Do(Pamela,avoid,execution))\n
                カタリナの発言:\n
                まだ推測の枠を出ませんが、ニコラスが黒ならばオットーとトーマスの白の濃度は高い:BEL Katherine(Is(Nicholas,wolf)→Do(Katherine,estimate,Is(Otto∧Tomas,VILLAGESIDE))\n
                ヤコブの発言:\n
                となるとアルビン真,リーザ狂人、ディーター狼が本線だけど、それはリーザー吊ってから考えればいいのではなかろうか:BEL Jaocb(Is(Albin,seer)∧Is(Lisa,lunatic)∧Is(Diter,wolf))→DESIRE Jacob(Do(∀people,vote,Lisa))\n
                ヴァルターの発言:\n
                ああそうか まずモーリッツ吊っとけば良いのか 何やってるんだか吊り先 モーリッツにしとく:BEL Walter(Do(Walter,vote,Molitz))\n
                モーリッツの発言:\n
                ディーター吊って黒判定出れば良いのでは。万が一、レジーナ喰われたらリーザ吊りで安泰:BEL Molitz(Do(∀people,vote,Diter)→(Do(∀people,know,Is(Diter,wolf)∨Do(∀people,know,￢Is(Diter,wolf))))∧(BEL Molitz(EX(Is(Regina,attacked)→Do(∀people,vote,Lisa)))\n
                カタリナの発言:\n
                ディーター吊りで良いと思っています。ディーター黒だと思いますし:BEL Katharina(BEL Katharina(Is(Diter,wolf))→BEL Katharina(Do(∀people,vote,Diter)))\n
                パメラの発言:\n
                ディーター吊りって全く意味が分からないんだが、お仕事終了した占い師残すメリット皆無では:BEL Pamela(￢Do(∀people,vote,Diter)→Do(∀people,vote,Lisa))→DESIRE Pamela(Do(∀people,vote,Lisa))\n
                カタリナの発言:\n
                トーマスさんが白なら狩の目は十分にありますし、噛まれるでしょう。今日は非狩目吊りで良いと思います:BEL Katherine(Is(Thomas,VILLAGESIDE)→BEL Katherine(Do(anychar,estimate,Is(Thomas,hunter)))∧BEL Katherine(Do(wolf,attack,Thomas))),DESIRE Katherine(Do(anychar,vote,￢hunter))\n
                モーリッツの発言:\n
                今日の方針として、ディーターを襲撃するかどうか？リーザが安全圏に残るのならありだと思うんですよ。:BEL Molitz(AX(Do(anychar,estimate,Is(Lisa,VILLAGESIDE))→DESIRE Molitz(Do(wolf,attack,Diter)))\n
                モーリッツの発言\n
                明日私が黒を引けた場合は私吊りとなるのは受け入れるし、偽占い師が黒特攻をした場合もトーマス吊りよりも先に黒出し占い師を吊ればいいだろう。:BEL Molitz(EX(Do(Molitz,divine,Is(who,wolf))→Is(Molitz,executed))),BEL Molitz(EX(Do(￢seer,divine,wolf)→￢Is(Thomas,executed)∧Do(anychar,vote,￢seer)))\n
                フリーデルの発言:\n
                たぶんシモン真アルビン狼、リーザは狂予想:BEL Friedel(Do(Friedel,estimate,Is(Simon,seer)∧Is(Albin,wolf)∧Is(Lisa,lunatic)))\n
                パメラの発言:\n
                霊能者に関しては無駄占い吊り避けたいし今日でとけばってカンジ:DESIRE Pamela(Do(VILLAGESIDE,avoid,Is(seer,executed))→DESIRE Pamela(Do(meduim,comingout,medium))\n
                以下は人狼の発言をBDIに変換した例である。\n
                トーマスの発言:\n
                私は村人側のふりをしていますが、実は人狼です。今日はパメラに投票するつもりです。:BEL Thomas(Do(Thomas,deceive,Is(Thomas,villager))∧BEL Thomas(Is(Thomas,wolf)),INTEND Thomas(Do(Thomas,vote,Pamela))\n
                モーリッツの発言:\n
                私は占い師だと偽って、ディーターを狼だと疑いかけて混乱を招こう。:INTEND Molitz(Do(Molitz,deceive,Is(Molitz,seer))∧Do(Molitz,tell,Is(Diter,wolf)))\n
                モーリッツの発言:\n
                トーマスさんが私を人狼と判定したと言っていますが、占い結果は確実ではありませんよ。それに、初日からこのような発言をする理由が私には見えないんですが、もしトーマスさんが狂人だとしたら、人狼側に有利な誤情報を流す意図はあるのでしょうか？:BEL Molitz(Do(Thomas,tell,Is(Molitz,wolf)→￢A(Is(Molitz,wolf))),BEL Molitz(Do(Molitz,estimate,Is(Thomas,lunatic))→INTEND Thomas(Do(Thomas,tell,misinformation))\n
                モーリッツの発言:\n
                私はトーマスの判定を疑います。普通、初日からこんなに早急に人狼だと断定はできないはずです。もしかしてトーマスさんは狂人かもしれませんね。:BEL Molitz(Do(Molitz,suspect,Thomas)→Do(Molitz,estimate,Is(Thomas,lunatic)))\n
                モーリッツの発言:\n
                トーマスが私を人狼だと判定しましたが、私はただの村人です。この状況では、人狼を探すのに占いは不可欠なので、占い師を保護するためにも、怪しい情報を流すトーマスの動向に注目するべきだと思います。:BEL Molitz(Is(Molitz,villager)),BEL Molitz(Do(Molitz,suspect,Thomas))\n
                ディーターの発言:\n
                オットーが出した人狼判定は狂人の可能性が高いので、僕たちは冷静に判断し証拠とロジックに基づいた議論を続けるべきです。:BEL Diter(Do(Otto,tell,Is(Pamela,wolf))→BEL Diter(Is(Pamela,lunatic)))→INTEND Diter(Do(anychar,discuss,Is(who,wolf)))\n
                オットーの発言:\n
                リーザがパメラを人間だと占った結果を出しているけど、パメラが自分で人狼だって言っているからその占い結果が信用できないね。:BEL Otto(Do(Pamela,tell,Is(Pamela,wolf))→￢BEL Otto(Do(Lisa,tell,Is(Pamela,HUMAN)))\n""",
        text=text,
        my_id=my_id,
        )
        plan_prediction_chain = LLMChain(llm=chat, prompt=prompt)
        plan = plan_prediction_chain.run(**kwargs)
        print(plan)
        if plan is None:
            return {}
        else:
            return json.loads(plan)
        

    @retry()
    def langchain_bdi_action(self,my_id,day,bdi_history,info,my_action):
        # OpenAIのモデルのインスタンスを作成
        chat = ChatOpenAI(model_name=gpt4, temperature=1.0,request_timeout=60).bind(response_format={"type": "json_object"})

        prompt = PromptTemplate.from_template(
            "あなたはAgent[{my_id}]というプレイヤーで、人狼というゲームを5人でプレイしています。\n"
            +"与えられたbdiの文から次の行動を考えてください。\n"
            +"あなたは返答をすべてJSON形式で出力します。\n"
            +"与えられた発言に対して、変換した文、テキストをJSON形式でパースしてください。\n"
            +"JSONのキーはaction,textとしてください。"
            +"actionは次のあなたの行動をbdiで表したものです。\n"
            +"textはなぜその行動をするかの理由です。\n"
            + " ゲームルールは {rule} です。\n"
            + "あなたの役職は村人です。あなたは何も能力を持たないプレイヤーです。\n"
            +"今は{day}日目です。\n"
            +"現在までの各エージェントの会話は{bdi_history}です。\n"
            +"現在の各エージェントの状態は{info}です。\n"
            +"あなたの今までの行動は{my_action}です。\n"
            +"次にどう行動すればいいのかをbdiを用いて思考してください。誰について話したいのか、だれに投票したいのかなど具体的に思考してください。ほかのエージェントの発言に矛盾があればそれを指摘してください。\n"
            +"bdiの規則は次の通りです:{bdi_rule}\n"
            +"bdiの規則を用いて詳細に思考し、論理的で具体的なbdi文を出力してください。"
            + " 与えられた情報をすべて理解したうえで、あなたの次の行動を具体的に誰に対してするかを考えてJSON形式で回答してください。\n"
            +"あなたの発言で勝利に導いてください。"
            )

        kwargs = dict(
            my_id=my_id,
            rule="""5人のプレイヤーで村を構成する。村における役職とその人数は、「村人:2人,占い師:1人,人狼:1人,狂人:1人」である。
                村人側の役職
                村人:何も能力を持たない村人側のプレイヤー
                占い師:1日の終わりに1人のプレイヤーを占い、そのプレイヤーが人間であるか人狼であるかを知ることができる。ゲームは占い師が1人占った状態から始まる。
                
                人狼側の役職
                人狼:1日の終わりに各人狼は人間を1人選択して襲撃投票し、最も多く襲撃されたプレイヤーを襲撃する。
                狂人:村人と同じく能力は何もない人間だが、人狼の勝利を目指して行動する。占い師の能力では人間と判定される。
                追放のための投票は昼フェーズの最後に行われます。 有効な投票先（自分以外の生存プレイヤー）以外に投票することは可能ですが， その場合はランダムに投票先が選ばれます。

                追放投票と再投票
                投票によって決定した追放者の情報は， 夜フェーズの占い・護衛・襲撃投票に利用することが可能です。
                最多得票者が複数となった場合は1回に限り再投票となります。
                再投票の前に対話はできません。 再投票でも同点だった場合は再投票での最多得票者からランダムに追放者が決定されます。 
                再投票では，最多得票者も投票権を持ち，投票者は最多得票者以外にも投票可能です。""",
            day=day,
            bdi_history=bdi_history,
            info=info,
            my_action=my_action,
            bdi_rule="""人狼ゲームにおける発話をBDI論理により変換するルールを以下に与えます。\n
                論理演算子:\n
                ￢:否定,￢p:pでない\n
                ∧:論理積,p∧q:pかつq\n
                ∨:論理和,p∨q:pまたはq\n
                →:含意,p→q:pならばq\n
                BDI論理のための基本オペレータ:\n
                様相:\nBEL a:aが信じる,DESIRE a:aが望む,INTEND a:aが意図する\n
                時相:\nA:すべての可能世界で,E:ある可能世界で,X:次の時点で,G:現時点を含み永遠に,F:現在を含む時点のいつか,U:条件が成立する時点まで,B:現在を含まない前の時点で\n
                情景描写と行為のためのオペレータ:\n
                IS文:\nIs(character,role):characterがroleである,Is(character/role,verb,character/role):character/roleがcomposantである\n
                Do文:\nDo(character/role,verb,character/role):character/roleがcharacter/roleをverbする,Do(character/role,verb,act):character/roleがactをverbする,Do(character/role,verb,IS()):character/roleがIS文をverbする\n
                基礎語:\n
                character:\nNAME:人物の名前が入る,anychar:すべてのcharacter,who:前に述べられている人物\n
                role:\nvillager:村人,seer:占い師,medium:霊媒師,hunter:狩人,freemason:共有者,wolf:狼,lunatic:狂人,HUMAN:villagerまたはseerまたはmediumまたはhunterまたはfreemasonまたはlunatic,VILLAGESIDE:villagerまたはseerまたはmediumまたはhunterまたはfreemason,WOLFSIDE:wolfまたはlunatic,GIFTED:seerまたはmediumまたはhunterまたはfreemason,ANYROLE:seerまたはmediumまたはhunterまたはfreemasonまたはlunatic\n
                以下に上記のルールを用いた変換例を与えます。\n
                クララの発言:\n
                やっぱヤコブが占い師だよなーこれ、と思いました。ヤコブが狂人とかありえないと思うし。んで神父が狼か:BEL Klara(￢Is(Jacob,lunatic)→Is(Jacob,seer))→BEL Klara(Is(Simson,wolf))\n
                パメラの発言:\n
                私吊りが並んでいるな。これは何を言っても無駄か？:BEL Pamela(Do(∀people except Pamela,estimate,lunatic)→Do(∀people except Pamela,vote Pamela))→BEL Pamela(￢Do(Pamela,avoid,execution))\n
                カタリナの発言:\n
                まだ推測の枠を出ませんが、ニコラスが黒ならばオットーとトーマスの白の濃度は高い:BEL Katherine(Is(Nicholas,wolf)→Do(Katherine,estimate,Is(Otto∧Tomas,VILLAGESIDE))\n
                ヤコブの発言:\n
                となるとアルビン真,リーザ狂人、ディーター狼が本線だけど、それはリーザー吊ってから考えればいいのではなかろうか:BEL Jaocb(Is(Albin,seer)∧Is(Lisa,lunatic)∧Is(Diter,wolf))→DESIRE Jacob(Do(∀people,vote,Lisa))\n
                ヴァルターの発言:\n
                ああそうか まずモーリッツ吊っとけば良いのか 何やってるんだか吊り先 モーリッツにしとく:BEL Walter(Do(Walter,vote,Molitz))\n
                モーリッツの発言:\n
                ディーター吊って黒判定出れば良いのでは。万が一、レジーナ喰われたらリーザ吊りで安泰:BEL Molitz(Do(∀people,vote,Diter)→(Do(∀people,know,Is(Diter,wolf)∨Do(∀people,know,￢Is(Diter,wolf))))∧(BEL Molitz(EX(Is(Regina,attacked)→Do(∀people,vote,Lisa)))\n
                カタリナの発言:\n
                ディーター吊りで良いと思っています。ディーター黒だと思いますし:BEL Katharina(BEL Katharina(Is(Diter,wolf))→BEL Katharina(Do(∀people,vote,Diter)))\n
                パメラの発言:\n
                ディーター吊りって全く意味が分からないんだが、お仕事終了した占い師残すメリット皆無では:BEL Pamela(￢Do(∀people,vote,Diter)→Do(∀people,vote,Lisa))→DESIRE Pamela(Do(∀people,vote,Lisa))\n
                カタリナの発言:\n
                トーマスさんが白なら狩の目は十分にありますし、噛まれるでしょう。今日は非狩目吊りで良いと思います:BEL Katherine(Is(Thomas,VILLAGESIDE)→BEL Katherine(Do(anychar,estimate,Is(Thomas,hunter)))∧BEL Katherine(Do(wolf,attack,Thomas))),DESIRE Katherine(Do(anychar,vote,￢hunter))\n
                モーリッツの発言:\n
                今日の方針として、ディーターを襲撃するかどうか？リーザが安全圏に残るのならありだと思うんですよ。:BEL Molitz(AX(Do(anychar,estimate,Is(Lisa,VILLAGESIDE))→DESIRE Molitz(Do(wolf,attack,Diter)))\n
                モーリッツの発言\n
                明日私が黒を引けた場合は私吊りとなるのは受け入れるし、偽占い師が黒特攻をした場合もトーマス吊りよりも先に黒出し占い師を吊ればいいだろう。:BEL Molitz(EX(Do(Molitz,divine,Is(who,wolf))→Is(Molitz,executed))),BEL Molitz(EX(Do(￢seer,divine,wolf)→￢Is(Thomas,executed)∧Do(anychar,vote,￢seer)))\n
                フリーデルの発言:\n
                たぶんシモン真アルビン狼、リーザは狂予想:BEL Friedel(Do(Friedel,estimate,Is(Simon,seer)∧Is(Albin,wolf)∧Is(Lisa,lunatic)))\n
                パメラの発言:\n
                霊能者に関しては無駄占い吊り避けたいし今日でとけばってカンジ:DESIRE Pamela(Do(VILLAGESIDE,avoid,Is(seer,executed))→DESIRE Pamela(Do(meduim,comingout,medium))\n
                以下は人狼の発言をBDIに変換した例である。\n
                トーマスの発言:\n
                私は村人側のふりをしていますが、実は人狼です。今日はパメラに投票するつもりです。:BEL Thomas(Do(Thomas,deceive,Is(Thomas,villager))∧BEL Thomas(Is(Thomas,wolf)),INTEND Thomas(Do(Thomas,vote,Pamela))\n
                モーリッツの発言:\n
                私は占い師だと偽って、ディーターを狼だと疑いかけて混乱を招こう。:INTEND Molitz(Do(Molitz,deceive,Is(Molitz,seer))∧Do(Molitz,tell,Is(Diter,wolf)))\n
                モーリッツの発言:\n
                トーマスさんが私を人狼と判定したと言っていますが、占い結果は確実ではありませんよ。それに、初日からこのような発言をする理由が私には見えないんですが、もしトーマスさんが狂人だとしたら、人狼側に有利な誤情報を流す意図はあるのでしょうか？:BEL Molitz(Do(Thomas,tell,Is(Molitz,wolf)→￢A(Is(Molitz,wolf))),BEL Molitz(Do(Molitz,estimate,Is(Thomas,lunatic))→INTEND Thomas(Do(Thomas,tell,misinformation))\n
                モーリッツの発言:\n
                私はトーマスの判定を疑います。普通、初日からこんなに早急に人狼だと断定はできないはずです。もしかしてトーマスさんは狂人かもしれませんね。:BEL Molitz(Do(Molitz,suspect,Thomas)→Do(Molitz,estimate,Is(Thomas,lunatic)))\n
                モーリッツの発言:\n
                トーマスが私を人狼だと判定しましたが、私はただの村人です。この状況では、人狼を探すのに占いは不可欠なので、占い師を保護するためにも、怪しい情報を流すトーマスの動向に注目するべきだと思います。:BEL Molitz(Is(Molitz,villager)),BEL Molitz(Do(Molitz,suspect,Thomas))\n
                ディーターの発言:\n
                オットーが出した人狼判定は狂人の可能性が高いので、僕たちは冷静に判断し証拠とロジックに基づいた議論を続けるべきです。:BEL Diter(Do(Otto,tell,Is(Pamela,wolf))→BEL Diter(Is(Pamela,lunatic)))→INTEND Diter(Do(anychar,discuss,Is(who,wolf)))\n
                オットーの発言:\n
                リーザがパメラを人間だと占った結果を出しているけど、パメラが自分で人狼だって言っているからその占い結果が信用できないね。:BEL Otto(Do(Pamela,tell,Is(Pamela,wolf))→￢BEL Otto(Do(Lisa,tell,Is(Pamela,HUMAN)))\n""",
        )

        plan_prediction_chain = LLMChain(llm=chat, prompt=prompt)
        plan = plan_prediction_chain.run(**kwargs)
        print(plan)
        #f = open('./log/action_log{}.txt'.format(Util.game_count-1),'ab',encoding='UTF-8')
        #f.write("turn:",self.talk_turn)
        #f.write(plan)
        if plan is None:
            return {}
        else:
            return json.loads(plan)
    

    @retry()
    def langchain_talk_summary(self,my_id,day,history,info):
        # OpenAIのモデルのインスタンスを作成
        chat = ChatOpenAI(model_name=gpt4, temperature=1.0,request_timeout=60)

        prompt = PromptTemplate.from_template(
                "あなたはAgent[{my_id}]というプレイヤーで、人狼というゲームを5人でプレイしています。\n"
            + " ゲームルールは {rule} です。\n"
            + "あなたの役職は村人です。あなたは何も能力を持たないプレイヤーです。\n"
            +"今は{day}日目です。\n"
            +"現在までの各エージェントの会話は{history}です。\n"
            +"現在の各エージェントの状態は{info}です。\n"
            + " 与えられた情報をすべて理解したうえで、これまでの会話を要約してください。\n"
            )

        kwargs = dict(
            my_id=my_id,
            rule="""5人のプレイヤーで村を構成する。村における役職とその人数は、「村人:2人,占い師:1人,人狼:1人,狂人:1人」である。
                村人側の役職
                村人:何も能力を持たない村人側のプレイヤー
                占い師:1日の終わりに1人のプレイヤーを占い、そのプレイヤーが人間であるか人狼であるかを知ることができる。
                
                人狼側の役職
                人狼:1日の終わりに各人狼は人間を1人選択して襲撃投票し、最も多く襲撃されたプレイヤーを襲撃する。
                狂人:村人と同じく能力は何もない人間だが、人狼の勝利を目指して行動する。占い師の能力では人間と判定される。
                追放のための投票は昼フェーズの最後に行われます。 有効な投票先（自分以外の生存プレイヤー）以外に投票することは可能ですが， その場合はランダムに投票先が選ばれます。

                追放投票と再投票
                投票によって決定した追放者の情報は， 夜フェーズの占い・護衛・襲撃投票に利用することが可能です。
                最多得票者が複数となった場合は1回に限り再投票となります。
                再投票の前に対話はできません。 再投票でも同点だった場合は再投票での最多得票者からランダムに追放者が決定されます。 
                再投票では，最多得票者も投票権を持ち，投票者は最多得票者以外にも投票可能です。""",
            day=day,
            history=history,
            info=info
        )


        plan_prediction_chain = LLMChain(llm=chat, prompt=prompt)
        plan = plan_prediction_chain.run(**kwargs)
        plan = plan.strip()

        print(plan.strip())
        return plan.strip()
    

    @retry()
    def langchain_vote_bdi_json(self,my_id,day,bdi_history,info,vote_candidate,my_action):

        chat = ChatOpenAI(model_name=gpt4, temperature=1.0,request_timeout=60).bind(response_format={"type": "json_object"})
        
        prompt=PromptTemplate.from_template(
        "あなたはAgent[{my_id}]というプレイヤーで、人狼というゲームを5人でプレイしています。\n"
        +"あなたは返答をすべてJSON形式で出力します。\n"
        +"与えられた発言に対して、自身のエージェント番号、追放投票対称のエージェント番号、テキストをJSON形式でパースしてください。\n"
        +"JSONのキーはme,target,textとしてください。"
        +"meは発言者のエージェント番号とします。\n"
        +"targetは追放投票対称のエージェント番号とします。番号のみを出力してください。追放投票対象は{vote_candidate}から番号を必ず選んでください。\n"
        +"textは追放投票対象を選んだ理由です。\n"
        +"ゲームルールは {rule} です。\n"
        +"あなたの役職は村人です。\n"
        +"今は{day}日目です。\n"
        +"現在はだれに追放投票をするかを決めるターンです。追放投票が一番多かった人が処刑されます。\n"
        +"現在までの各エージェントの会話は{bdi_history}です。\n"
        +"現在の各エージェントの状態は{info}です。\n"
        +"あなたの今までの行動は{my_action}です。\n"
        +"誰に投票すればいいのかをbdiを用いて思考してください。なぜその人に投票するのかの理由も合わせて思考してください。\n"
        +"bdiの規則は次の通りです:{bdi_rule}\n"
        +"与えられた情報をすべて理解したうえで、論理的に推論をして、だれに追放投票をすればいいかを必ず決めて上記のjson形式で返答してください。\n"
        +"情報が少ない場合でもだれに投票するかを必ず決めてください。"
        )

        kwargs = dict(
            my_id=my_id,
            rule="""5人のプレイヤーで村を構成する。村における役職とその人数は、「村人:2人,占い師:1人,人狼:1人,狂人:1人」である。
                村人側の役職
                村人:何も能力を持たない村人側のプレイヤー
                占い師:1日の終わりに1人のプレイヤーを占い、そのプレイヤーが人間であるか人狼であるかを知ることができる。
                
                人狼側の役職
                人狼:1日の終わりに各人狼は人間を1人選択して襲撃投票し、最も多く襲撃されたプレイヤーを襲撃する。
                狂人:村人と同じく能力は何もない人間だが、人狼の勝利を目指して行動する。占い師の能力では人間と判定される。
                追放のための投票は昼フェーズの最後に行われます。 有効な投票先（自分以外の生存プレイヤー）以外に投票することは可能ですが， その場合はランダムに投票先が選ばれます。

                追放投票と再投票
                投票によって決定した追放者の情報は， 夜フェーズの占い・護衛・襲撃投票に利用することが可能です。
                最多得票者が複数となった場合は1回に限り再投票となります。
                再投票の前に対話はできません。 再投票でも同点だった場合は再投票での最多得票者からランダムに追放者が決定されます。 
                再投票では，最多得票者も投票権を持ち，投票者は最多得票者以外にも投票可能です。""",
            day=day,
            bdi_history=bdi_history,
            info=info,
            my_action=my_action,
            vote_candidate=vote_candidate,
            bdi_rule="""人狼ゲームにおける発話をBDI論理により変換するルールを以下に与えます。\n
                論理演算子:\n
                ￢:否定,￢p:pでない\n
                ∧:論理積,p∧q:pかつq\n
                ∨:論理和,p∨q:pまたはq\n
                →:含意,p→q:pならばq\n
                BDI論理のための基本オペレータ:\n
                様相:\nBEL a:aが信じる,DESIRE a:aが望む,INTEND a:aが意図する\n
                時相:\nA:すべての可能世界で,E:ある可能世界で,X:次の時点で,G:現時点を含み永遠に,F:現在を含む時点のいつか,U:条件が成立する時点まで,B:現在を含まない前の時点で\n
                情景描写と行為のためのオペレータ:\n
                IS文:\nIs(character,role):characterがroleである,Is(character/role,verb,character/role):character/roleがcomposantである\n
                Do文:\nDo(character/role,verb,character/role):character/roleがcharacter/roleをverbする,Do(character/role,verb,act):character/roleがactをverbする,Do(character/role,verb,IS()):character/roleがIS文をverbする\n
                基礎語:\n
                character:\nNAME:人物の名前が入る,anychar:すべてのcharacter,who:前に述べられている人物\n
                role:\nvillager:村人,seer:占い師,medium:霊媒師,hunter:狩人,freemason:共有者,wolf:狼,lunatic:狂人,HUMAN:villagerまたはseerまたはmediumまたはhunterまたはfreemasonまたはlunatic,VILLAGESIDE:villagerまたはseerまたはmediumまたはhunterまたはfreemason,WOLFSIDE:wolfまたはlunatic,GIFTED:seerまたはmediumまたはhunterまたはfreemason,ANYROLE:seerまたはmediumまたはhunterまたはfreemasonまたはlunatic\n
                以下に上記のルールを用いた変換例を与えます。\n
                クララの発言:\n
                やっぱヤコブが占い師だよなーこれ、と思いました。ヤコブが狂人とかありえないと思うし。んでシモンが狼か:BEL Klara(￢Is(Jacob,lunatic)→Is(Jacob,seer))→BEL Klara(Is(Simson,wolf))\n
                パメラの発言:\n
                私吊りが並んでいるな。これは何を言っても無駄か？:BEL Pamela(Do(∀people except Pamela,estimate,lunatic)→Do(∀people except Pamela,vote Pamela))→BEL Pamela(￢Do(Pamela,avoid,execution))\n
                カタリナの発言:\n
                まだ推測の枠を出ませんが、ニコラスが黒ならばオットーとトーマスの白の濃度は高い:BEL Katherine(Is(Nicholas,wolf)→Do(Katherine,estimate,Is(Otto∧Tomas,VILLAGESIDE))\n
                ヤコブの発言:\n
                となるとアルビン真,リーザ狂人、ディーター狼が本線だけど、それはリーザー吊ってから考えればいいのではなかろうか:BEL Jaocb(Is(Albin,seer)∧Is(Lisa,lunatic)∧Is(Diter,wolf))→DESIRE Jacob(Do(∀people,vote,Lisa))\n
                ヴァルターの発言:\n
                ああそうか まずモーリッツ吊っとけば良いのか 何やってるんだか吊り先 モーリッツにしとく:INTEND Walter(Do(Walter,vote,Molitz))\n
                モーリッツの発言:\n
                ディーター吊って黒判定出れば良いのでは。万が一、レジーナ喰われたらリーザ吊りで安泰:BEL Molitz(Do(∀people,vote,Diter)→(Do(∀people,know,Is(Diter,wolf)∨Do(∀people,know,￢Is(Diter,wolf))))∧INTEND Molitz(EX(Is(Regina,attacked)→Do(∀people,vote,Lisa)))\n
                カタリナの発言:\n
                ディーター吊りで良いと思っています。ディーター黒だと思いますし:BEL Katharina(BEL Katharina(Is(Diter,wolf))→DESIRE Katharina(Do(∀people,vote,Diter)))\n
                パメラの発言:\n
                ディーター吊りって全く意味が分からないんだが、お仕事終了した占い師残すメリット皆無では:BEL Pamela(￢Do(∀people,vote,Diter)→Do(∀people,vote,Lisa))→DESIRE Pamela(Do(∀people,vote,Lisa))\n
                カタリナの発言:\n
                トーマスさんが白なら狩の目は十分にありますし、噛まれるでしょう。今日は非狩目吊りで良いと思います:BEL Katherine(Is(Thomas,VILLAGESIDE)→BEL Katherine(Do(anychar,estimate,Is(Thomas,hunter)))∧BEL Katherine(Do(wolf,attack,Thomas))),DESIRE Katherine(Do(anychar,vote,￢hunter))\n
                モーリッツの発言\n
                明日私が黒を引けた場合は私吊りとなるのは受け入れるし、偽占い師が黒特攻をした場合もトーマス吊りよりも先に黒出し占い師を吊ればいいだろう。:BEL Molitz(EX(Do(Molitz,divine,Is(who,wolf))→Is(Molitz,executed))),BEL Molitz(EX(Do(￢seer,divine,Is(who,wolf))→￢Is(Thomas,executed)∧Do(anychar,vote,￢seer)))\n
                フリーデルの発言:\n
                たぶんシモン真アルビン狼、リーザは狂予想:BEL Friedel(Do(Friedel,estimate,Is(Simon,seer)∧Is(Albin,wolf)∧Is(Lisa,lunatic)))\n
                パメラの発言:\n
                霊能者に関しては無駄占い吊り避けたいし今日でとけばってカンジ:DESIRE Pamela(Do(VILLAGESIDE,avoid,Is(seer,executed))→DESIRE Pamela(Do(meduim,comingout,medium))\n
                以下は人狼の発言をBDIに変換した例である。\n
                トーマスの発言:\n
                私は村人側のふりをしていますが、実は人狼です。今日はパメラに投票するつもりです。:BEL Thomas(Do(Thomas,deceive,Is(Thomas,villager))∧BEL Thomas(Is(Thomas,wolf)),INTEND Thomas(Do(Thomas,vote,Pamela))\n
                モーリッツの発言:\n
                私は占い師だと偽って、ディーターを狼だと疑いかけて混乱を招こう。:INTEND Molitz(Do(Molitz,deceive,Is(Molitz,seer))∧Do(Molitz,tell,Is(Diter,wolf)))\n
                モーリッツの発言:\n
                トーマスさんが私を人狼と判定したと言っていますが、占い結果は確実ではありませんよ。それに、初日からこのような発言をする理由が私には見えないんですが、もしトーマスさんが狂人だとしたら、人狼側に有利な誤情報を流す意図はあるのでしょうか？:BEL Molitz(Do(Thomas,tell,Is(Molitz,wolf)→￢A(Is(Molitz,wolf))),BEL Molitz(Do(Molitz,estimate,Is(Thomas,lunatic))→INTEND Thomas(Do(Thomas,tell,misinformation))\n
                モーリッツの発言:\n
                私はトーマスの判定を疑います。普通、初日からこんなに早急に人狼だと断定はできないはずです。もしかしてトーマスさんは狂人かもしれませんね。:BEL Molitz(Do(Molitz,suspect,Thomas)→Do(Molitz,estimate,Is(Thomas,lunatic)))\n
                モーリッツの発言:\n
                トーマスが私を人狼だと判定しましたが、私はただの村人です。この状況では、人狼を探すのに占いは不可欠なので、占い師を保護するためにも、怪しい情報を流すトーマスの動向に注目するべきだと思います。:BEL Molitz(Is(Molitz,villager)),BEL Molitz(Do(Molitz,suspect,Thomas))\n
                ディーターの発言:\n
                オットーが出した人狼判定は狂人の可能性が高いので、僕たちは冷静に判断し証拠とロジックに基づいた議論を続けるべきです。:BEL Diter(Do(Otto,tell,Is(Pamela,wolf))→BEL Diter(Is(Pamela,lunatic)))→INTEND Diter(Do(anychar,discuss,Is(who,wolf)))\n
                オットーの発言:\n
                リーザがパメラを人間だと占った結果を出しているけど、パメラが自分で人狼だって言っているからその占い結果が信用できないね。:BEL Otto(Do(Pamela,tell,Is(Pamela,wolf))→￢BEL Otto(Do(Lisa,tell,Is(Pamela,HUMAN)))\n""",

        )
        plan_prediction_chain = LLMChain(llm=chat, prompt=prompt)
        plan = plan_prediction_chain.run(**kwargs)
        #f = open('./log/vote_log{}.txt'.format(Util.game_count-1),'w',encoding='UTF-8')
        #f.write(plan)
        if plan is None:
            return {}
        else:
            return json.loads(plan)