import copy
import logging
import os
import torch
import random
import sys
from dataclasses import dataclass, field
from typing import Optional
import json
import numpy as np
import transformers
from transformers import RobertaTokenizer, RobertaForSequenceClassification
from transformers.trainer_callback import EarlyStoppingCallback
from transformers import (
    AutoConfig,
    AutoModelForSequenceClassification,
    AutoTokenizer,
    EvalPrediction,
    HfArgumentParser,
    PretrainedConfig,
    Trainer,
    TrainingArguments,
    default_data_collator,
    set_seed,
)
from transformers.trainer_utils import is_main_process
from data.config import activate_kind

logger = logging.getLogger(__name__)

@dataclass
class DataTrainingArguments:
    """
    Arguments pertaining to what data we are going to input our model for training and eval.

    Using `HfArgumentParser` we can turn this class
    into argparse arguments to be able to specify them on
    the command line.
    """

    task_name: Optional[str] = field(
        default=None,
        metadata={"help": "The name of the task"},
    )
    max_seq_length: int = field(
        default=128,
        metadata={
            "help": "The maximum total input sequence length after tokenization. Sequences longer "
            "than this will be truncated, sequences shorter will be padded."
        },
    )
    overwrite_cache: bool = field(
        default=False, metadata={"help": "Overwrite the cached preprocessed datasets or not."}
    )
    pad_to_max_length: bool = field(
        default=False,
        metadata={
            "help": "Whether to pad all samples to `max_seq_length`. "
            "If False, will pad the samples dynamically when batching to the maximum length in the batch."
        },
    )
    train_file: Optional[str] = field(
        default=None, metadata={"help": "A csv or a json file containing the training data."}
    )
    validation_file: Optional[str] = field(
        default=None, metadata={"help": "A csv or a json file containing the validation data."}
    )

    def __post_init__(self):
        if self.train_file is None or self.validation_file is None:
            raise ValueError("Need either a GLUE task or a training/validation file.")
        else:
            extension = self.train_file.split(".")[-1]
            assert extension in ["csv", "json"], "`train_file` should be a csv or a json file."
            extension = self.validation_file.split(".")[-1]
            assert extension in ["csv", "json"], "`validation_file` should be a csv or a json file."


@dataclass
class ModelArguments:
    """
    Arguments pertaining to which model/config/tokenizer we are going to fine-tune from.
    """

    model_name_or_path: str = field(
        metadata={"help": "Path to pretrained model or model identifier from huggingface.co/models"}
    )
    config_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained config name or path if not the same as model_name"}
    )
    tokenizer_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained tokenizer name or path if not the same as model_name"}
    )
    cache_dir: Optional[str] = field(
        default=None,
        metadata={"help": "Where do you want to store the pretrained models downloaded from huggingface.co"},
    )
    use_fast_tokenizer: bool = field(
        default=True,
        metadata={"help": "Whether to use one of the fast tokenizer (backed by the tokenizers library) or not."},
    )


# class ROCDataset(torch.utils.data.Dataset):
#     def __init__(self, encodings_labels_dict):
#         self.encodings = encodings_labels_dict["encodings"]
#         self.labels = encodings_labels_dict["labels"]

#     def __getitem__(self, idx):
#         item = {key: torch.tensor(val[idx]) for key, val in self.encodings.items()}
#         item['labels'] = torch.tensor(self.labels[idx])
#         return item

#     def __len__(self):
#         return len(self.labels)

class StoriumDataset(torch.utils.data.Dataset):
    def __init__(self, encodings_labels_dict, do_predict = False):
        self.encodings = encodings_labels_dict["encodings"]
        self.do_predict = do_predict
        self.labels = encodings_labels_dict["labels"]
        # print('do_predict = ', self.do_predict)

    def __getitem__(self, idx):
        item = {key: torch.tensor(val[idx]) for key, val in self.encodings.items()}
        if not self.do_predict:
            item['labels'] = torch.tensor(self.labels[idx])
        return item

    def __len__(self):
        return len(self.labels)
    
def main():
    # See all possible arguments in src/transformers/training_args.py
    # or by passing the --help flag to this script.
    # We now keep distinct sets of args, for a cleaner separation of concerns.

    parser = HfArgumentParser((ModelArguments, DataTrainingArguments, TrainingArguments))
    if len(sys.argv) == 2 and sys.argv[1].endswith(".json"):
        # If we pass only one argument to the script and it's the path to a json file,
        # let's parse it to get our arguments.
        model_args, data_args, training_args = parser.parse_json_file(json_file=os.path.abspath(sys.argv[1]))
    else:
        model_args, data_args, training_args = parser.parse_args_into_dataclasses()
    print('training args = ', training_args)
    if (
        os.path.exists(training_args.output_dir)
        and os.listdir(training_args.output_dir)
        and training_args.do_train
        and not training_args.overwrite_output_dir
    ):
        raise ValueError(
            f"Output directory ({training_args.output_dir}) already exists and is not empty. "
            "Use --overwrite_output_dir to overcome."
        )

    # Setup logging
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s -   %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        level=logging.INFO if is_main_process(training_args.local_rank) else logging.WARN,
    )

    # Log on each process the small summary:
    logger.warning(
        f"Process rank: {training_args.local_rank}, device: {training_args.device}, n_gpu: {training_args.n_gpu}"
        + f"distributed training: {bool(training_args.local_rank != -1)}, 16-bits training: {training_args.fp16}"
    )
    # Set the verbosity to info of the Transformers logger (on main process only):
    if is_main_process(training_args.local_rank):
        transformers.utils.logging.set_verbosity_info()
        transformers.utils.logging.enable_default_handler()
        transformers.utils.logging.enable_explicit_format()
    logger.info(f"Training/evaluation parameters {training_args}")

    # Set seed before initializing model.
    set_seed(training_args.seed)

    # Get the datasets: you can either provide your own CSV/JSON training and evaluation files (see below)
    # or specify a GLUE benchmark task (the dataset will be downloaded automatically from the datasets Hub).
    #
    # For CSV/JSON files, this script will use as labels the column called 'label' and as pair of sentences the
    # sentences in columns called 'sentence1' and 'sentence2' if such column exists or the first two columns not named
    # label if at least two columns are provided.
    #
    # If the CSVs/JSONs contain only one non-label column, the script does single sentence classification on this
    # single column. You can easily tweak this behavior (see below)
    #
    # In distributed training, the load_dataset function guarantee that only one local process can concurrently
    # download the dataset.
    if data_args.train_file.endswith(".csv"):
        # Loading a dataset from local csv files
        pass
        # datasets = load_dataset(
        #     "csv", data_files={"train": data_args.train_file, "validation": data_args.validation_file}
        # )
    else:
        # Loading a dataset from local json files
        # datasets = load_dataset(
        #     "json", data_files={"train": data_args.train_file, "validation": data_args.validation_file, "field": "data"}
        # )
        datasets = {}
        print('data_args = ', data_args)
        with open(data_args.train_file) as fin:
            datasets["train"] = json.load(fin)['data']
            print('train size = ', len(datasets['train']))
        with open(data_args.validation_file) as fin:
            datasets["validation"] = json.load(fin)['data']
            print('val size = ', len(datasets['validation']))

    # See more about loading any type of standard or custom dataset at
    # https://huggingface.co/docs/datasets/loading_datasets.html.

    # Labels
    # Trying to have good defaults here, don't hesitate to tweak to your needs.
    is_regression = False #datasets["train"].features["label"].dtype in ["float32", "float64"]
    # A useful fast method:
    # https://huggingface.co/docs/datasets/package_reference/main_classes.html#datasets.Dataset.unique
    label_list = list(range(len(activate_kind) + 1))
    print('label_list = ', label_list)
    #label_list = [0, 1] #datasets["train"].unique("label")
    label_list.sort()  # Let's sort it for determinism
    num_labels = len(label_list)

    # Load pretrained model and tokenizer
    #
    # In distributed training, the .from_pretrained methods guarantee that only one local process can concurrently
    # download model & vocab.
    config = AutoConfig.from_pretrained(
        model_args.config_name if model_args.config_name else model_args.model_name_or_path,
        num_labels=num_labels,
        # finetuning_task=data_args.task_name,
        cache_dir=model_args.cache_dir,
    )
    tokenizer = AutoTokenizer.from_pretrained(
        model_args.tokenizer_name if model_args.tokenizer_name else model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        use_fast=model_args.use_fast_tokenizer,
    )
    model = AutoModelForSequenceClassification.from_pretrained(
        model_args.model_name_or_path,
        from_tf=bool(".ckpt" in model_args.model_name_or_path),
        config=config,
        cache_dir=model_args.cache_dir,
    )

    # Preprocessing the datasets
    # Again, we try to have some nice defaults but don't hesitate to tweak to your use case.
    non_label_column_names = ["text"]
    sentence1_key, sentence2_key = "text", None

    # Padding strategy
    if data_args.pad_to_max_length:
        padding = "max_length"
        max_length = data_args.max_seq_length
    else:
        # We will pad later, dynamically at batch creation, to the max sequence length in each batch
        padding = False
        max_length = data_args.max_seq_length

    # Some models have set the order of the labels to use, so let's make sure we do use it.
    label_to_id = {v: i for i, v in enumerate(label_list)}

    def preprocess_function(examples, split):
        result = {}
        cnt = 0
        if split == 'train': # 训练时丢掉超过长度的数据
            new_examples = []
            for l in examples:
                tmp_len = len(tokenizer([l["text"]])["input_ids"][0])
                if tmp_len > data_args.max_seq_length:
                    cnt += 1 
                    continue
                else:
                    new_examples.append(l)
            examples = copy.deepcopy(new_examples)
        print('split = ', split)
        print('len = ', len(examples))
        print('cnt = ', cnt)
        print('='*100)
        # 默认在首尾加上bos和sep, 中间的sep设成default的</s>即可
        result["encodings"] = tokenizer([l["text"] for l in examples], truncation=True, padding=padding, max_length=max_length)
        result["labels"] = [label_to_id[l["label"]] if l['label'] is not None else -1 for l in examples]
        #result["labels"] = [1 for l in examples]
        # print(data_args.max_seq_length)
        # for r in result["encodings"]["input_ids"]:
        #     if len(r) > data_args.max_seq_length:
        #         print(len(r))
        # exit()
        return result
        
    train_dataset = StoriumDataset(preprocess_function(datasets["train"], 'train')) # encodings(id), labels(id)
    eval_dataset = StoriumDataset(preprocess_function(datasets["validation"], 'val'), do_predict = training_args.do_predict)

    # Log a few random samples from the training set:
    for index in random.sample(range(len(train_dataset)), 3):
        logger.info(f"Sample {index} of the training set: {train_dataset[index]}.")

    # TODO: When datasets metrics include regular accuracy, make an else here and remove special branch from
    # compute_metrics

    # You can define your custom compute_metrics function. It takes an `EvalPrediction` object (a namedtuple with a
    # predictions and label_ids field) and has to return a dictionary string to float.
    def compute_metrics(p: EvalPrediction):
        preds = p.predictions[0] if isinstance(p.predictions, tuple) else p.predictions
        print('is regression = ', is_regression)
        preds = np.squeeze(preds) if is_regression else np.argmax(preds, axis=1)

        result = {}
        temp = []
        for pred, label in zip(preds, p.label_ids):
            if (pred==0 and label==0) or (pred!=0 and label!=0):
                temp.append(1)
            else:
                temp.append(0)
        result['score_accuracy'] = np.mean(temp)
        result['accuracy'] = (preds == p.label_ids).astype(np.float32).mean().item()
        for i in range(6):
            mask = p.label_ids == i
            result[f"type({i})"] = (preds[mask] == p.label_ids[mask]).astype(np.float32).mean().item()
        return result

        # positive_mask = p.label_ids == 1
        # negative_mask = p.label_ids == 0
        
        # return {
        #     "accuracy": (preds == p.label_ids).astype(np.float32).mean().item(),
        #     "positive_acc": (preds[positive_mask] == p.label_ids[positive_mask]).astype(np.float32).mean().item(),
        #     "negative_acc": (preds[negative_mask] == p.label_ids[negative_mask]).astype(np.float32).mean().item(),
        #     }



    # Initialize our Trainer
    trainer = Trainer(
        model=model,
        args=training_args,
        train_dataset=train_dataset,
        eval_dataset=eval_dataset if training_args.do_eval else None,
        compute_metrics=compute_metrics,
        tokenizer=tokenizer,
        callbacks=[EarlyStoppingCallback(early_stopping_patience=5)], # threshold default to 0.0
        # Data collator will default to DataCollatorWithPadding, so we change it if we already did the padding.
        data_collator=default_data_collator if data_args.pad_to_max_length else None,
    )

    # Training
    if training_args.do_train:
        trainer.train(
            model_path=model_args.model_name_or_path if os.path.isdir(model_args.model_name_or_path) else None
        )
        trainer.save_model()  # Saves the tokenizer too for easy upload

    # Evaluation
    eval_results = {}
    print('training_args.do_eval = ', training_args.do_eval)
    print('training_args.do_predict = ', training_args.do_predict)
    if training_args.do_eval:
        logger.info("*** Evaluate ***")

        # Loop to handle MNLI double evaluation (matched, mis-matched)
        tasks = [data_args.task_name]
        eval_datasets = [eval_dataset]

        for eval_dataset, task in zip(eval_datasets, tasks):
            eval_result = trainer.evaluate(eval_dataset=eval_dataset)

            output_eval_file = os.path.join(training_args.output_dir, f"eval_results_{task}.txt")
            if trainer.is_world_process_zero():
                with open(output_eval_file, "w") as writer:
                    logger.info(f"***** Eval results {task} *****")
                    for key, value in eval_result.items():
                        logger.info(f"  {key} = {value}")
                        writer.write(f"{key} = {value}\n")

            eval_results.update(eval_result)

    if training_args.do_predict:
        logger.info("*** Test ***")

        # Loop to handle MNLI double evaluation (matched, mis-matched)
        tasks = [data_args.task_name]
        test_datasets = [eval_dataset]
        #test_datasets = [test_dataset]
        if data_args.task_name == "mnli":
            tasks.append("mnli-mm")
            test_datasets.append(datasets["test_mismatched"])

        for test_dataset, task in zip(test_datasets, tasks):
            # Removing the `label` columns because it contains -1 and Trainer won't like that.
            # test_dataset.remove_columns_("label")
            predictions = trainer.predict(test_dataset=test_dataset).predictions
            print('is regression = ', is_regression)
            logit_score = torch.softmax(torch.tensor(predictions), dim=-1)[:, 1].numpy()
            logit_score = np.mean(logit_score)
            predictions = np.squeeze(predictions) if is_regression else np.argmax(predictions, axis=1)
            
            print('file = ', data_args.validation_file)
            print(f'score1 = {np.sum(predictions==1) / len(predictions)}, total = {len(predictions)}')
            print(f'score2 = {logit_score}')
            output_test_file = os.path.join(training_args.output_dir, f"test_results_{task}.txt")
            if trainer.is_world_process_zero():
                with open(output_test_file, "w") as writer:
                    logger.info(f"***** Test results {task} *****")
                    writer.write("index\tprediction\n")
                    for index, item in enumerate(predictions):
                        if is_regression:
                            writer.write(f"{index}\t{item:3.3f}\n")
                        else:
                            item = label_list[item]
                            writer.write(f"{index}\t{item}\n")
    return eval_results


if __name__ == "__main__":
    main()