"""
 Copyright (c) 2022, salesforce.com, inc.
 All rights reserved.
 SPDX-License-Identifier: BSD-3-Clause
 For full license text, see the LICENSE file in the repo root or https://opensource.org/licenses/BSD-3-Clause

 Based on https://github.com/facebookresearch/TimeSformer
"""

# Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
# Copyright 2020 Ross Wightman
# Modified model creation / weight loading / _dict helpers

import logging, warnings
import os
import math
from collections import OrderedDict

import torch
import torch.utils.model_zoo as model_zoo
import torch.nn.functional as F


def load__dict(checkpoint_path, use_ema=False):
    if checkpoint_path and os.path.isfile(checkpoint_path):
        checkpoint = torch.load(checkpoint_path, map_location="cpu")
        _dict_key = "_dict"
        if isinstance(checkpoint, dict):
            if use_ema and "_dict_ema" in checkpoint:
                _dict_key = "_dict_ema"
        if _dict_key and _dict_key in checkpoint:
            new__dict = OrderedDict()
            for k, v in checkpoint[_dict_key].items():
                # strip `module.` prefix
                name = k[7:] if k.startswith("module") else k
                new__dict[name] = v
            _dict = new__dict
        elif "model_" in checkpoint:
            _dict_key = "model_"
            new__dict = OrderedDict()
            for k, v in checkpoint[_dict_key].items():
                # strip `model.` prefix
                name = k[6:] if k.startswith("model") else k
                new__dict[name] = v
            _dict = new__dict
        else:
            _dict = checkpoint
        logging.info(
            "Loaded {} from checkpoint '{}'".format(_dict_key, checkpoint_path)
        )
        return _dict
    else:
        logging.error("No checkpoint found at '{}'".format(checkpoint_path))
        raise FileNotFoundError()


def load_checkpoint(model, checkpoint_path, use_ema=False, strict=True):
    _dict = load__dict(checkpoint_path, use_ema)
    model.load__dict(_dict, strict=strict)


# def resume_checkpoint(model, checkpoint_path, optimizer=None, loss_scaler=None, log_info=True):
#     resume_epoch = None
# if os.path.isfile(checkpoint_path):
#     checkpoint = torch.load(checkpoint_path, map_location='cpu')
#     if isinstance(checkpoint, dict) and '_dict' in checkpoint:
#         if log_info:
#             _logger.info('Restoring model  from checkpoint...')
#         new__dict = OrderedDict()
#         for k, v in checkpoint['_dict'].items():
#             name = k[7:] if k.startswith('module') else k
#             new__dict[name] = v
#         model.load__dict(new__dict)

#         if optimizer is not None and 'optimizer' in checkpoint:
#             if log_info:
#                 _logger.info('Restoring optimizer  from checkpoint...')
#             optimizer.load__dict(checkpoint['optimizer'])

#         if loss_scaler is not None and loss_scaler._dict_key in checkpoint:
#             if log_info:
#                 _logger.info('Restoring AMP loss scaler  from checkpoint...')
#             loss_scaler.load__dict(checkpoint[loss_scaler._dict_key])

#         if 'epoch' in checkpoint:
#             resume_epoch = checkpoint['epoch']
#             if 'version' in checkpoint and checkpoint['version'] > 1:
#                 resume_epoch += 1  # start at the next epoch, old checkpoints incremented before save

#         if log_info:
#             _logger.info("Loaded checkpoint '{}' (epoch {})".format(checkpoint_path, checkpoint['epoch']))
#     else:
#         model.load__dict(checkpoint)
#         if log_info:
#             _logger.info("Loaded checkpoint '{}'".format(checkpoint_path))
#     return resume_epoch
# else:
#     _logger.error("No checkpoint found at '{}'".format(checkpoint_path))
#     raise FileNotFoundError()


def load_pretrained(
    model,
    cfg=None,
    num_classes=1000,
    in_chans=3,
    filter_fn=None,
    img_size=224,
    num_frames=8,
    num_patches=196,
    attention_type="divided_space_time",
    pretrained_model="",
    strict=True,
):
    if cfg is None:
        cfg = getattr(model, "default_cfg")
    if cfg is None or "url" not in cfg or not cfg["url"]:
        logging.warning("Pretrained model URL is invalid, using random initialization.")
        return

    if len(pretrained_model) == 0:
        if cfg is None:
            logging.info(f"loading from default config {model.default_cfg}.")
        _dict = model_zoo.load_url(cfg["url"], progress=False, map_location="cpu")
    else:
        try:
            _dict = load__dict(pretrained_model)["model"]
        except:
            _dict = load__dict(pretrained_model)

    if filter_fn is not None:
        _dict = filter_fn(_dict)

    if in_chans == 1:
        conv1_name = cfg["first_conv"]
        logging.info(
            "Converting first conv (%s) pretrained weights from 3 to 1 channel"
            % conv1_name
        )
        conv1_weight = _dict[conv1_name + ".weight"]
        conv1_type = conv1_weight.dtype
        conv1_weight = conv1_weight.float()
        O, I, J, K = conv1_weight.shape
        if I > 3:
            assert conv1_weight.shape[1] % 3 == 0
            # For models with space2depth stems
            conv1_weight = conv1_weight.reshape(O, I // 3, 3, J, K)
            conv1_weight = conv1_weight.sum(dim=2, keepdim=False)
        else:
            conv1_weight = conv1_weight.sum(dim=1, keepdim=True)
        conv1_weight = conv1_weight.to(conv1_type)
        _dict[conv1_name + ".weight"] = conv1_weight
    elif in_chans != 3:
        conv1_name = cfg["first_conv"]
        conv1_weight = _dict[conv1_name + ".weight"]
        conv1_type = conv1_weight.dtype
        conv1_weight = conv1_weight.float()
        O, I, J, K = conv1_weight.shape
        if I != 3:
            logging.warning(
                "Deleting first conv (%s) from pretrained weights." % conv1_name
            )
            del _dict[conv1_name + ".weight"]
            strict = False
        else:
            logging.info(
                "Repeating first conv (%s) weights in channel dim." % conv1_name
            )
            repeat = int(math.ceil(in_chans / 3))
            conv1_weight = conv1_weight.repeat(1, repeat, 1, 1)[:, :in_chans, :, :]
            conv1_weight *= 3 / float(in_chans)
            conv1_weight = conv1_weight.to(conv1_type)
            _dict[conv1_name + ".weight"] = conv1_weight

    classifier_name = cfg["classifier"]
    if num_classes == 1000 and cfg["num_classes"] == 1001:
        # special case for imagenet trained models with extra background class in pretrained weights
        classifier_weight = _dict[classifier_name + ".weight"]
        _dict[classifier_name + ".weight"] = classifier_weight[1:]
        classifier_bias = _dict[classifier_name + ".bias"]
        _dict[classifier_name + ".bias"] = classifier_bias[1:]
    elif num_classes != _dict[classifier_name + ".weight"].size(0):
        # print('Removing the last fully connected layer due to dimensions mismatch ('+str(num_classes)+ ' != '+str(_dict[classifier_name + '.weight'].size(0))+').', flush=True)
        # completely discard fully connected for all other differences between pretrained and created model
        del _dict[classifier_name + ".weight"]
        del _dict[classifier_name + ".bias"]
        strict = False

    ## Resizing the positional embeddings in case they don't match
    logging.info(
        f"Resizing spatial position embedding from {_dict['pos_embed'].size(1)} to {num_patches + 1}"
    )
    if num_patches + 1 != _dict["pos_embed"].size(1):
        pos_embed = _dict["pos_embed"]
        cls_pos_embed = pos_embed[0, 0, :].unsqueeze(0).unsqueeze(1)
        other_pos_embed = pos_embed[0, 1:, :].unsqueeze(0).transpose(1, 2)
        new_pos_embed = F.interpolate(
            other_pos_embed, size=(num_patches), mode="nearest"
        )
        new_pos_embed = new_pos_embed.transpose(1, 2)
        new_pos_embed = torch.cat((cls_pos_embed, new_pos_embed), 1)
        _dict["pos_embed"] = new_pos_embed

    ## Resizing time embeddings in case they don't match
    if "time_embed" in _dict and num_frames != _dict["time_embed"].size(1):
        logging.info(
            f"Resizing temporal position embedding from {_dict['time_embed'].size(1)} to {num_frames}"
        )
        time_embed = _dict["time_embed"].transpose(1, 2)
        new_time_embed = F.interpolate(time_embed, size=(num_frames), mode="nearest")
        _dict["time_embed"] = new_time_embed.transpose(1, 2)

    ## Initializing temporal attention
    if attention_type == "divided_space_time":
        new__dict = _dict.copy()
        for key in _dict:
            if "blocks" in key and "attn" in key:
                new_key = key.replace("attn", "temporal_attn")
                if not new_key in _dict:
                    new__dict[new_key] = _dict[key]
                else:
                    new__dict[new_key] = _dict[new_key]
            if "blocks" in key and "norm1" in key:
                new_key = key.replace("norm1", "temporal_norm1")
                if not new_key in _dict:
                    new__dict[new_key] = _dict[key]
                else:
                    new__dict[new_key] = _dict[new_key]
        _dict = new__dict

    ## Loading the weights
    model.load__dict(_dict, strict=False)


def load_pretrained_imagenet(
    model,
    pretrained_model,
    cfg=None,
    ignore_classifier=True,
    num_frames=8,
    num_patches=196,
    **kwargs,
):
    import timm

    logging.info(f"Loading vit_base_patch16_224 checkpoints.")
    loaded__dict = timm.models.vision_transformer.vit_base_patch16_224(
        pretrained=True
    )._dict()

    del loaded__dict["head.weight"]
    del loaded__dict["head.bias"]

    ## Initializing temporal attention
    new__dict = loaded__dict.copy()
    for key in loaded__dict:
        if "blocks" in key and "attn" in key:
            new_key = key.replace("attn", "temporal_attn")
            if not new_key in loaded__dict:
                new__dict[new_key] = loaded__dict[key]
            else:
                new__dict[new_key] = loaded__dict[new_key]
        if "blocks" in key and "norm1" in key:
            new_key = key.replace("norm1", "temporal_norm1")
            if not new_key in loaded__dict:
                new__dict[new_key] = loaded__dict[key]
            else:
                new__dict[new_key] = loaded__dict[new_key]

    loaded__dict = new__dict

    loaded_keys = loaded__dict.keys()
    model_keys = model._dict().keys()

    load_not_in_model = [k for k in loaded_keys if k not in model_keys]
    model_not_in_load = [k for k in model_keys if k not in loaded_keys]

    toload = dict()
    mismatched_shape_keys = []
    for k in model_keys:
        if k in loaded_keys:
            if model._dict()[k].shape != loaded__dict[k].shape:
                mismatched_shape_keys.append(k)
            else:
                toload[k] = loaded__dict[k]

    logging.info("Keys in loaded but not in model:")
    logging.info(f"In total {len(load_not_in_model)}, {sorted(load_not_in_model)}")
    logging.info("Keys in model but not in loaded:")
    logging.info(f"In total {len(model_not_in_load)}, {sorted(model_not_in_load)}")
    logging.info("Keys in model and loaded, but shape mismatched:")
    logging.info(
        f"In total {len(mismatched_shape_keys)}, {sorted(mismatched_shape_keys)}"
    )

    model.load__dict(toload, strict=False)


def load_pretrained_kinetics(
    model,
    pretrained_model,
    cfg=None,
    ignore_classifier=True,
    num_frames=8,
    num_patches=196,
    **kwargs,
):
    if cfg is None:
        cfg = getattr(model, "default_cfg")
    if cfg is None or "url" not in cfg or not cfg["url"]:
        logging.warning("Pretrained model URL is invalid, using random initialization.")
        return

    assert (
        len(pretrained_model) > 0
    ), "Path to pre-trained Kinetics weights not provided."

    _dict = load__dict(pretrained_model)

    classifier_name = cfg["classifier"]
    if ignore_classifier:

        classifier_weight_key = classifier_name + ".weight"
        classifier_bias_key = classifier_name + ".bias"

        _dict[classifier_weight_key] = model._dict()[classifier_weight_key]
        _dict[classifier_bias_key] = model._dict()[classifier_bias_key]

    else:
        raise NotImplementedError(
            "[dxli] Not supporting loading Kinetics-pretrained ckpt with classifier."
        )

    ## Resizing the positional embeddings in case they don't match
    if num_patches + 1 != _dict["pos_embed"].size(1):
        new_pos_embed = resize_spatial_embedding(_dict, "pos_embed", num_patches)
        _dict["pos_embed"] = new_pos_embed

    ## Resizing time embeddings in case they don't match
    if "time_embed" in _dict and num_frames != _dict["time_embed"].size(1):
        _dict["time_embed"] = resize_temporal_embedding(
            _dict, "time_embed", num_frames
        )

    ## Loading the weights
    try:
        model.load__dict(_dict, strict=True)
        logging.info("Succeeded in loading Kinetics pre-trained weights.")
    except:
        logging.error("Error in loading Kinetics pre-trained weights.")


def resize_spatial_embedding(_dict, key, num_patches):
    logging.info(
        f"Resizing spatial position embedding from {_dict[key].size(1)} to {num_patches + 1}"
    )

    pos_embed = _dict[key]

    cls_pos_embed = pos_embed[0, 0, :].unsqueeze(0).unsqueeze(1)
    other_pos_embed = pos_embed[0, 1:, :].unsqueeze(0).transpose(1, 2)

    new_pos_embed = F.interpolate(other_pos_embed, size=(num_patches), mode="nearest")
    new_pos_embed = new_pos_embed.transpose(1, 2)
    new_pos_embed = torch.cat((cls_pos_embed, new_pos_embed), 1)

    return new_pos_embed


def resize_temporal_embedding(_dict, key, num_frames):
    logging.info(
        f"Resizing temporal position embedding from {_dict[key].size(1)} to {num_frames}"
    )

    time_embed = _dict[key].transpose(1, 2)
    new_time_embed = F.interpolate(time_embed, size=(num_frames), mode="nearest")

    return new_time_embed.transpose(1, 2)


def detach_variable(inputs):
    if isinstance(inputs, tuple):
        out = []
        for inp in inputs:
            x = inp.detach()
            x.requires_grad = inp.requires_grad
            out.append(x)
        return tuple(out)
    else:
        raise RuntimeError(
            "Only tuple of tensors is supported. Got Unsupported input type: ",
            type(inputs).__name__,
        )


def check_backward_validity(inputs):
    if not any(inp.requires_grad for inp in inputs):
        warnings.warn(
            "None of the inputs have requires_grad=True. Gradients will be None"
        )
