import numpy as np
import argparse
import os
import jsonlines
import gzip

import torch
import torch.nn as nn
import torch.optim as optim
from torch.nn import DataParallel
from torch.utils.data import DataLoader
from torch.autograd import Variable

from utils.config import load_config

from models.Oracle import Oracle

from gw_utils import *

def get_qas(pgame):
    """
    Gets the list of questions and their answers from a played game. These
    played games have to be generated with the Gameplay/inference.py script.

    Parameters
    ----------
    pgame: dict
        data of a played game, made with the inference script

    Returns
    -------
    out: list
        a list of tuples (question, answer) for each question in pgame
    """
    qas = []
    q = ''
    for wd in pgame.lower().strip().split():
        if wd == '<start>':
            continue
        if wd == '<no>' or wd == '<yes>' or wd == '<n/a>':
            qas.append((q.strip(),wd[1:-1]))
            q = ''
        else:
            q += ' '+wd
    return qas


def get_raw_qas(game):
    """
    Gets the list of questions and their answers from a played game. These
    played games are taken from the human-human GuessWhat!? corpus.

    Parameters
    ----------
    pgame: dict
        data of a played game from the human-human corpus

    Returns
    -------
    out: list
        a list of tuples (question, answer) for each question in game
    """
    qas = []
    for qa in game['qas']:
        qfixed = qa['question'][:-1]+' ?'
        # Get rid of the '?' of the last word
        qas.append((qfixed, qa['answer'].lower()))
    return qas

def load_visual_features(split='val'):
    """
    Load visual features from file and a given split.

    Parameters
    ----------
    split: str, optional
        Name of the partition to load. Options are 'train', 'val' or 'test' 

    Returns
    -------
    out: list
        a list of tuples (question, answer) for each question in pgame
    """
    # TODO make this function take paths as arguments
    partitions = {'train': ('train2id', 'train_img_features'),
                  'val'  : ('val2id', 'val_img_features'),
                  'test' : ('test2id', 'test_img_features')}

    map_split, feat_split = partitions[split]
    print('Using {}'.format(split))
    with open('data/ResNet_avg_image_features2id.json', 'rb') as fl:
        vf_map = json.load(fl)
        vf_map = vf_map[map_split]
    visual_features = np.asarray(h5py.File('data/ResNet_avg_image_features.h5',
        'r')[feat_split])
    return vf_map, visual_features


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument("-breaking", action='store_true', help='Run with just one sample for test purpose')
    parser.add_argument("-split", type=str,default='val', help='Partition of the dataset to  use. Options are "train", "val" or "test". Default: val.')
    parser.add_argument("-out_fname", type=str, default='out.json', help='Name of the generated file.')
    parser.add_argument("-bin_path", type=str, required=True, help='Path to the Oracle bin model.')
    parser.add_argument("-img_feats", action='store_true', help='Use image features')
    parser.add_argument("-crop_feats", action='store_true', help='Use crop features')
    parser.add_argument("-from_gameplay", action='store_true', help='If the question_data file was generated by the Gameplay/inference.py script')
    parser.add_argument("-question_data", type=str, help='Path to the file of questions. Not needed if -from_gameplay is not used.')
    
    args = parser.parse_args()

    # Raw GuessWhat?! dataset
    print('Crops: ', args.crop_feats)
    print('Image: ', args.img_feats)
    
    # TODO make this path not so hardcoded
    datapath = 'data/guesswhat.{}.jsonl.gz'.format('valid' if args.split == 'val' else args.split)

    rawdata = get_raw_dataset(datapath)

    # TODO make this load with parameters
    # Load the visual mapping and visual features
    vf_map, visual_feat = load_visual_features(args.split)
    with open('./data/objects_features_index_test.json', 'r') as fl:
        cf_map = json.load(fl)
    viscrop_feat = np.asarray(h5py.File('./data/objects_features_test.h5', 'r')['objects_features'])

    # Load the vocabulary manager and Oracle model
    # TODO make this load with parameters
    vocab, oracle = load_vocab_oracle(args.bin_path)

    # Dicts to transfrom from token to word
    ans2tok = {'no': 0, 'yes': 1, 'n/a': 2}
    tok2ans = {0:'no', 1: 'yes', 2: 'n/a'}

    # TODO make the path a parameter
    # If we get our dialogues from the Gameplay/inference.py script
    # load the dialogues
    if args.from_gameplay:
        with open(args.question_data, 'r') as fl:
            played_games = json.loads(fl.read())

    # Data to write to file
    data = {}

    # Check effectiveness for each game played in the dataset
    for j, game in enumerate(rawdata): 

        # For test purposes
        if args.breaking and j > 5:
            break

        # Get image related information
        im_width = game['image']['width']
        im_height = game['image']['height']

        # Get a dict of all objects in the image indexed by their id 
        objects = {obj['id']: obj for obj in game['objects']}

        # Get the id of the target object
        target_obj_id = game['object_id']
        target_obj = objects[target_obj_id]

        # Create a sample to make inference with the oracle
        # It has to have the fields:
        # answers: non-important
        # crop_features: non-important
        # img_features: visual features of the image
        # lenght: lenght of the question
        # question: a list of numbers that represent the words in ids
        # obj_cat: the category of the target object for the question
        # spatial: the bounding box of the target object for the question
        sample = {}
        sample['answer']         = torch.tensor([2])
        sample['crop_features']  = torch.tensor([0])
        sample['img_features']   = torch.tensor([0])
        if args.crop_feats:
            crop_feat_idx = cf_map[str(game['id'])]
            crop_feat = viscrop_feat[crop_feat_idx]
            sample['crop_features']  = torch.tensor([crop_feat])
        if args.img_feats:
            img_feat_idx = vf_map[game['image']['file_name']]
            img_feat = visual_feat[img_feat_idx]
            sample['img_features']   = torch.tensor([img_feat])
        sample['obj_cat'] = torch.tensor([target_obj['category_id']])
        sample['spatial'] = torch.tensor([get_spatial_feat(target_obj['bbox'],
                                                           im_width,
                                                           im_height)])

        # If we're using questions from the Gameplay/inference.py script,
        # parse question with get_qas. Otherwise parse them with get_raw_qas
        # For models
        #try:
        #    qas = get_qas(played_games[str(game['id'])]['gen_dialogue'])
        #    status = played_games[str(game['id'])]['state']
        #except KeyError:
        #    print("{} skipped".format(game['id']))
        #    continue

        # For humans
        qas = get_raw_qas(game)
        status = game['status']

        #print("{}: target object: {}".format(game['id'], target_obj_id))

        # Take the last question like this because it's easier
        proc_qas = []
        for q, ans in qas:

            # Transform the question string in a list of word ids
            qseq = vocab.words2seq(q)

            # For each question in the gameplay:
            # set the questions and its lenght
            sample['length'] = torch.tensor([len(q.strip().split(' '))])
            sample['question'] = torch.tensor([qseq])

            objs_ans = [] # Dict of objects
            for o in objects:
                obj = objects[o]
                # For each object in the image:
                # set that object's caterogy and spatial information
                sample['obj_cat'] = torch.tensor([obj['category_id']])
                sample['spatial'] = torch.tensor([get_spatial_feat(obj['bbox'],
                                                              im_width,
                                                              im_height)])

                # Get prediction from the oracle
                pred = oracle.forward(sample).argmax(dim=1).numpy()

                if o == target_obj_id:
                    objs_ans.append({'object_id': o, 'ans':ans,
                                     'is_target': o == target_obj_id})
                else:
                    objs_ans.append({'object_id': o, 'ans':tok2ans[int(pred[0])],
                                     'is_target': o == target_obj_id})

            proc_q = {'question': q, 'ans': ans,
                      'objs': objs_ans}
            proc_qas.append(proc_q)


        # Store data for this game
        data[str(game['id'])] = {'qas': proc_qas,
                                 'state': status}

        #data[str(game['id'])] = {'question': q, 'ans': ans,
        #                         'state':status,
        #                         'objs':objs_ans}


    with open(args.out_fname, 'w') as fl:
        json.dump(data, fl)
