import React, { Component } from 'react';
import { withTracker } from 'meteor/react-meteor-data';
import { Link } from "react-router-dom";

import { Annotations, Texts, Tutorials } from '../api/documents';
import TextDoc from './TextDoc.jsx';
import { gup } from '../util/gup.js';

function formatInstructions(inst) {
  var context = [];
  var prev = null;
  var buffer = [];
  var itemBuffer = [];
  var inItem = false;
  inst.split(" ").forEach(token => {
    if (inItem) {
      if (token == "ITEM" || token == "\n") {
        itemBuffer.push((<li key={itemBuffer.length.toString()}>{buffer.join(" ")}</li>));
        buffer = [];
        if (token == "\n") {
          context.push((<ul key={context.length.toString()}>{itemBuffer}</ul>));
          inItem = false;
          itemBuffer = [];
        } else {
          return;
        }
      } else {
        buffer.push(token);
        return;
      }
    }

    if (token == "\n") {
      if (prev == "\n") {
        context.push((<span key={context.length.toString()} className="instructions">&nbsp;</span>));
      }
      context.push((<div key={context.length.toString()} className="clear"></div>));
    } else if (token == "ITEM") {
      inItem = true;
      if (prev != "\n") {
        context.push((<div key={context.length.toString()} className="clear"></div>));
      }
    } else if (prev == "\n" || prev == "ITEM" || context.length == 0) {
      context.push(token);
    } else {
      context[context.length -1] += " "+ token;
    }
    prev = token;
  });

  if (buffer.length > 0) {
    itemBuffer.push((<li key={itemBuffer.length.toString()}>{buffer.join(" ")}</li>));
  }
  if (itemBuffer.length > 0) {
    context.push((<ul key={context.length.toString()}>{itemBuffer}</ul>));
  }
  return context;
}

export class Tutorial extends Component {
  constructor(props) {
    super(props);

    this.handleClick = this.handleClick.bind(this);
    this.toggleHints = this.toggleHints.bind(this);
    this.showHintButton = this.showHintButton.bind(this);
    this.resetCurrent = this.resetCurrent.bind(this);
    this.prepareTutorial = this.prepareTutorial.bind(this);
    this.tutorialComplete = this.tutorialComplete.bind(this);
    this.noPuncSpan = this.noPuncSpan.bind(this);

    this.state = {
      currentStage: 0,
      finished: false,
      hints: "no-button",
      rerender: false,
    };
  }

  toggleHints() {
    this.setState((state, props) => {
      if (state.hints == "hide") return { hints: "show" };
      else if (state.hints == "show") return { hints: "hide" };
    });
  }

  showHintButton() {
    if (this.state.hints == "no-button") {
      this.setState({
        hints: "hide"
      });
    }
  }

  resetCurrent() {
    const user = gup("workerId");

    const task = this.props.tutorials.filter(option => {
      return option._id == this.props.match.params.id;
    })[0].tasks[this.state.currentStage];

    const init_anno = this.props.annotations.find(annotation => {
      return annotation._id == task.init_anno_id;
    });
    if (init_anno != undefined) {
      Meteor.call('tutorial.begin', user, task.text_id, init_anno.clusters, init_anno.labels, init_anno.checks, init_anno.focus_mention, this.props.match.params.id);
    }
  }

  tutorialComplete() {
    const user = gup("workerId");
    Meteor.call('tutorial.complete', user, this.props.match.params.id);
  }

  noPuncSpan(mention, isStart, text_id) {
    const text = this.props.texts.find(text => {
      return text._id == text_id;
    });
    var pos = isStart ? mention.start : mention.end;
    while (/[\u2000-\u206F\u2E00-\u2E7F\\'!"#$%&()*+,\-.\/:;<=>?@\[\]^_`{|}~]/.test(text.tokens[pos].text)) {
      if (isStart) {
        if (pos == mention.end) {
          pos = mention.start;
          break;
        }
        pos += 1;
      } else {
        if (pos == mention.start) {
          pos = mention.end;
          break;
        }
        pos -= 1;
      }
    }
    return pos;
  }

  handleClick = (e) => {
    const user = gup("workerId");
    var ui = "";

    // Check that the annotations for visible stages match
    const outcomes = this.props.tutorials.find(option => {
      // Get the relevant tutorial
      if (option._id == this.props.match.params.id) {
        ui = option.ui;
        return true;
      } else {
        return false;
      }
    }).tasks.filter((task, index) => {
      // Get the relevant stages
      return index <= this.state.currentStage;
    }).map(task => {
      // Check them
      const gold = this.props.annotations.find(annotation => {
        return annotation.user == 'server' && annotation.text_id == task.text_id;
      }).clusters;
      const rawAnno = this.props.annotations.find(annotation => {
        return annotation.user == user && annotation.text_id == task.text_id;
      });
      const include = [];
      const exclude = [];
      if (rawAnno.checks != undefined) {
        rawAnno.checks.forEach(check => {
          if (check.decision == "remove") {
            exclude.push(check.mention);
          } else {
            include.push(check.mention);
          }
        });
      }
      const anno = rawAnno.clusters.map(cluster => {
        return {
          _id: cluster._id,
          color: cluster.color,
          mentions: cluster.mentions.filter(mention => {
            const included = include.some(m => { return m.start == mention.start && m.end == mention.end  && m.label == mention.label });
            const excluded = exclude.some(m => { return m.start == mention.start && m.end == mention.end  && m.label == mention.label });
            if (ui.startsWith("check-mentions")) {
              if (ui.includes("exclude")) {
                return ! excluded;
              } else {
                return included && (! excluded);
              }
            } else {
              return true;
            }
          }),
        };
      }).filter(cluster => {
        return cluster.mentions.length > 0;
      });
      const overlap = gold.filter(gcluster => {
        return anno.some(acluster => {
          const matched = acluster.mentions.filter(amention => {
            // shift the ends of the mention to ignore punctuation
            const astart = this.noPuncSpan(amention, true, task.text_id);
            const aend = this.noPuncSpan(amention, false, task.text_id);
            return gcluster.mentions.some(gmention => {
              const gstart = this.noPuncSpan(gmention, true, task.text_id);
              const gend = this.noPuncSpan(gmention, false, task.text_id);
              return astart == gstart && aend == gend && amention.label == gmention.label;
            });
          }).length;
          return matched == acluster.mentions.length && matched == gcluster.mentions.length;
        });
      });
      return {
        done: overlap.length == gold.length && overlap.length == anno.length,
        matched: overlap.length,
        gold: gold.length,
        annotation: anno.length,
      };
    });

    if (outcomes.every(outcome => { return outcome.done; })) {
      const tasks = this.props.tutorials.find(option => {
        // Get the relevant tutorial
        if (option._id == this.props.match.params.id) {
          ui = option.ui;
          return true;
        } else {
          return false;
        }
      }).tasks;
      this.setState((state, props) => {
        return {
          currentStage: Math.min(state.currentStage + 1, tasks.length - 1),
          finished: state.currentStage + 1 == tasks.length,
        };
      });
    } else {
      this.setState({
        finished: false,
      });
      alert("Your answer does not match what we are looking for. Please try again. Use the hint button if you are stuck.");
      this.showHintButton();
    }
  }

  prepareTutorial() {
    if (this.props.annotations.length > 0) {
      const user = gup("workerId");

      this.props.tutorials.filter(option => {
        return option._id == this.props.match.params.id;
      }).forEach(tutorial => {
        tutorial.tasks.forEach(task => {
          const anno = this.props.annotations.find(annotation => {
            return annotation.user == user && annotation.text_id == task.text_id;
          });
          if (anno === undefined) {
            const init_anno = this.props.annotations.find(annotation => {
              return annotation._id == task.init_anno_id;
            });
            if (init_anno != undefined) {
              Meteor.call('tutorial.begin', user, task.text_id, init_anno.clusters, init_anno.labels, init_anno.checks, init_anno.focus_mention, tutorial._id);
            }
          }
        });
      });
    }
  }

  componentDidUpdate() {
    this.prepareTutorial();
  }

  componentDidMount() {
    this.prepareTutorial();
  }

  render() {
    const user = gup("workerId");

    // Get the relevant tutorial (text, gold annotations) based on the URL parameters
    const tutorial = this.props.tutorials.find(option => {
      return option._id == this.props.match.params.id;
    });
    // If we are not able to get something, return a placeholder
    if (tutorial === undefined) {
      return (
        <p>Loading</p>
      );
    }

    // Get the annotations for this user for this tutorial
    const stages = tutorial.tasks.map(task => {
      return {
        task: task,
        text:
          this.props.texts.find(text => {
            return text._id == task.text_id;
          }),
        anno:
          this.props.annotations.find(annotation => {
            return annotation.user == user && annotation.text_id == task.text_id;
          }),
      };
    });
    // If we are not able to get something, create the annotations and return a stub
    if (stages.some(stage => { return stage.anno === undefined || stage.text === undefined; })) {
      return (
        <p>Loading...</p>
      );
    }

    // Render up until the current stage
    const toShow = stages.filter((stage, index) => {
      return index <= this.state.currentStage;
    });
    const content = toShow.map((stage, index) => {
      const gold = this.props.annotations.find(annotation => {
        return annotation.user == 'server' && annotation.text_id == stage.task.text_id;
      });
      const hint = this.state.hints == 'show' ? gold.clusters : undefined;
      const fade = this.state.finished || index != toShow.length - 1 ? (<div className="fade"></div>) : "";
      const tick = (this.state.finished || index != toShow.length - 1) ? (
        <div className="center-buttons">
          <div className="fade"></div>
          {"\u2713 - Correct"}
        </div>
      ) : "";

      const context = formatInstructions(stage.task.context);
      return (
        <div key={stage.anno._id}>
          <div className="instructions">
            {fade}
            {"Tutorial Step "+ index.toString() +": "} {context}
          </div>
          <div className="tutorial-textdoc">
            {fade}
            <TextDoc
              text={stage.text}
              clusters={stage.anno.clusters}
              checks={stage.anno.checks}
              ref_clusters={hint}
              ann_id={stage.anno._id}
              key={stage.anno._id}
              ui={tutorial.ui}
              validate={() => {}}
              labelOptions={stage.anno.labels}
              focusMention={stage.task.focus_mention}
            />
          </div>
          <div className="clear">
          </div>
          {tick}
          <br />
          <br />
        </div>
      );
    });

    // If they have completed all stages, show a button to go to the task
    // (which links to assign).
    const button = (
      <button className="noselect button" onClick={this.handleClick} >
        {toShow[toShow.length - 1].task.progress}
      </button>
    );
    const hintButton = this.state.hints == "no-button" ? "" : (
      <button className="noselect button" onClick={this.toggleHints} >
        Show/Hide Hints
      </button>
    );
    const hintInfo = this.state.hints == "show" ? (
      <p>
        Green: Missing selection
        <br />
        Orange: Incorrect selection
      </p>
    ) : "";
    const reset = toShow[toShow.length - 1].task.progress != "Check" ? "" : (
      <button className="noselect button" onClick={this.resetCurrent} >
        Reset this step
      </button>
    );
    const target = (
      <button className="noselect button" onClick={this.tutorialComplete}>
        <Link to={"/assign/" + gup("group") + this.props.location.search}>Begin Task</Link>
      </button>
    );
    const finish = this.state.finished ? target : "";
   
    const buttonFade = this.state.finished ? (<div className="fade"></div>) : "";
    return (
      <div>
        {content}
        <div className="center-buttons">
          {finish}
        </div>
        <div className="center-buttons">
          {buttonFade}
          {reset}
          {hintButton}
        </div>
        <div className="center-buttons">
          {buttonFade}
          {hintInfo}
        </div>
        <div className="center-buttons">
          {buttonFade}
          {button}
        </div>
      </div>
    );
  }
}

export default TutorialContainer = withTracker(() => {
  const user_id = gup("workerId");

  Meteor.subscribe('tutorials-all');
  Meteor.subscribe('texts-tutorials');
  Meteor.subscribe('annotations-user', 'server');
  Meteor.subscribe('annotations-user', user_id);

  return {
    tutorials: Tutorials.find().fetch(),
    texts: Texts.find().fetch(),
    annotations: Annotations.find().fetch(),
  };
})(Tutorial);
