# -*- coding: utf-8 -*-

import pyshrg
from framework.common.logger import LOGGER
from framework.common.utils import ProgressReporter
from nn_generator.evaluate.utils import BLEU
from pyshrg_utils.parser import PySHRGParserOptions, PySHRGPool, pyshrg_parse_args


class MainOptions(PySHRGParserOptions):
    output_prefix: str = '-'


def _parsing_worker(context: pyshrg.Context, graph_index):
    code = context.parse(graph_index)
    if code == pyshrg.ParserError.kNone:
        context.generate()
    return graph_index, code, str(context.sentence)


def main(argv=None):
    manager, options = pyshrg_parse_args(argv, MainOptions, abbrevs={'output_prefix': '-o'})

    output_prefix = options.output_prefix

    error_count = 0

    all_arguments = []
    for i in range(manager.graph_size):
        all_arguments.append((i,))

    progress = ProgressReporter(stop=len(all_arguments),
                                message_fn=lambda _: f'error: {error_count}',
                                print_time=True,
                                newline=True)
    print_to_screen = output_prefix == '-'
    pool = PySHRGPool()
    try:
        bleu = BLEU()
        for graph_index, code, sentence in \
                progress(pool.imap_unordered(_parsing_worker, all_arguments)):
            graph = manager.get_graph(graph_index)

            if code != pyshrg.ParserError.kNone:
                error_count += 1
                # logger.error('%s %s', graph.sentence_id, code)
            else:
                bleu.add(sentence.lower(), graph.lemma_sequence.lower(), graph.sentence_id)
                if print_to_screen:
                    bleu.print_sample()

        bleu_score = bleu.get()
        LOGGER.info('BLEU score: %.4f', bleu_score)

        if not print_to_screen:
            bleu.save(output_prefix)
    finally:
        pool.terminate()


if __name__ == '__main__':
    main()
