using System;
using System.Linq;
using System.Collections.Generic;

namespace SpreadsheetLLM.Heuristic
{
    internal static class Utils
    {
        #region remove from Box List
        public static void RemoveTheseCandidates(IEnumerable<Boundary> CandidatesToRemoved, List<Boundary> boxes)
        {
            if (CandidatesToRemoved.Any())
            {
                boxes.RemoveAll(CandidatesToRemoved.Contains);
            }
        }

        public static void AppendTheseCandidates(IEnumerable<Boundary> CandidatesToAppend, List<Boundary> boxes)
        {
            if (CandidatesToAppend.Any())
            {
                if (!(CandidatesToAppend is HashSet<Boundary> candidatesHashSet))
                    candidatesHashSet = new HashSet<Boundary>(CandidatesToAppend);
                candidatesHashSet.ExceptWith(boxes);
                boxes.AddRange(candidatesHashSet);
            }
        }

        public static void RemoveAndAppendCandidates(IEnumerable<Boundary> CandidatesToRemoved, IEnumerable<Boundary> CandidatesToAppend, List<Boundary> boxes)
        {
            RemoveTheseCandidates(CandidatesToRemoved, boxes);
            AppendTheseCandidates(CandidatesToAppend, boxes);
        }
        #endregion

        #region Utils
        public static int CountTure(List<bool> diffs)
        {
            return diffs.Count(diff => diff);
        }

        public static bool VerifyValuesOpposite(int x, int y)
        {
            return x == 0 != (y == 0);
        }

        public static bool VerifyValuesDiff(int x, int y, int cellDiffThresh)
        {
            return Math.Abs(x - y) > cellDiffThresh;
        }

        public static bool IsFillBox(Boundary box1, List<Boundary> inBoxes, int step = 0)
        {
            if (inBoxes.Count == 0)
                return false;

            var rowLinesCombine = new List<int>();
            var colLinesCombine = new List<int>();
            foreach (var box2 in inBoxes)
            {
                if (!(box2.top <= box1.top + 2 && box2.bottom >= box1.bottom - 2))
                {
                    for (int i = box2.top - 2; i <= box2.bottom; i++)
                    {
                        rowLinesCombine.Add(i);
                    }
                }
                if (!(box2.left <= box1.left + 2 && box2.right >= box1.right - 2))
                {
                    for (int i = box2.left - 2; i <= box2.right; i++)
                    {
                        colLinesCombine.Add(i);
                    }
                }
            }

            if (Min(rowLinesCombine) > box1.top || Max(rowLinesCombine) < box1.bottom || Min(colLinesCombine) > box1.left || Max(colLinesCombine) < box1.right)
                return false;

            if (!IsFillLines(box1.top, box1.bottom, rowLinesCombine))
                return false;

            if (!IsFillLines(box1.left, box1.right, colLinesCombine))
                return false;

            return true;
        }

        public static bool IsFillBoxRowColLines(Boundary box1, List<int> rowLinesCombine, List<int> colLinesCombine, int step = 0)
        {
            return IsFillLines(box1.top, box1.bottom, rowLinesCombine)
                || IsFillLines(box1.left, box1.right, colLinesCombine);
        }
        public static bool IsFillLines(int start, int end, List<int> lines, int step = 0)
        {
            if (lines.Count == 0)
                return false;

            var linesInside = lines.Where(line => !(line < start || line > end)).ToList();
            if (Min(linesInside) <= start && Max(linesInside) >= end)
            {
                if (linesInside.Distinct().Count() > (end - start + 1) * 0.8)
                {
                    return true;
                }
                if (end - start > 5)
                {
                    int maxGap = 0;
                    linesInside.Sort();
                    int lineUp = linesInside[0];
                    foreach (int line in linesInside)
                    {
                        maxGap = Math.Max(maxGap, line - lineUp);
                        lineUp = line;
                    }
                    maxGap = Math.Max(maxGap, end - lineUp);
                    if (maxGap <= Math.Max(5, (end - start) * 0.2))
                        return true;
                }
            }

            return false;
        }

        public static List<Boundary> DistinctBoxes(List<Boundary> boxes)
        {
            return boxes.Distinct().ToList();
        }

        public static int DistinctStrs(string[,] contents, int up, int down, int left, int right)
        {
            var strings = new HashSet<string>();
            for (int i = up - 1; i <= down - 1; i++)
            {
                for (int j = left - 1; j <= right - 1; j++)
                {
                    if (contents[i, j] != "")
                        strings.Add(contents[i, j]);
                }
            }
            return strings.Count;
        }

        public static double AreaSize(Boundary box)
        {
            if (box.bottom < box.top || box.right < box.left)
                return 0;

            return (box.bottom - box.top + 1) * (box.right - box.left + 1);
        }

        public static double Height(Boundary box)
        {
            if (box.bottom < box.top)
                return 0;

            return box.bottom - box.top + 1;
        }
        public static double Width(Boundary box)
        {
            if (box.right < box.left)
                return 0;

            return box.right - box.left + 1;
        }

        public static Boundary LeftCol(Boundary box, int start = 0, int step = 1)
        {
            return new Boundary(box.top, box.bottom, box.left + start, box.left + start + step - 1);
        }

        public static Boundary UpRow(Boundary box, int start = 0, int step = 1)
        {
            return new Boundary(box.top + start, box.top + start + step - 1, box.left, box.right);
        }
        public static Boundary RightCol(Boundary box, int start = 0, int step = 1)
        {
            return new Boundary(box.top, box.bottom, box.right - start - step + 1, box.right - start);
        }
        public static Boundary DownRow(Boundary box, int start = 0, int step = 1)
        {
            return new Boundary(box.bottom - start - step + 1, box.bottom - start, box.left, box.right);
        }
        public static double AreaSize(List<Boundary> boxes)
        {
            double sizeSum = 0;
            var boxesCalculated = new List<Boundary>();
            foreach (var box in boxes)
            {
                sizeSum = sizeSum + AreaSize(box);
                bool markIn = false;
                foreach (var boxOut in boxes)
                {
                    if (ContainsBox(boxOut, box) && !boxOut.Equals(box))
                    {
                        markIn = true;
                        break;
                    }
                }
                if (markIn) continue;
                foreach (var boxIn in boxesCalculated)
                {
                    if (isOverlap(boxIn, box))
                    {
                        sizeSum = sizeSum - AreaSize(OverlapBox(boxIn, box));

                    }
                }
                foreach (var boxIn1 in boxesCalculated)
                {
                    foreach (var boxIn2 in boxesCalculated)
                    {
                        if (isOverlap(boxIn1, box) && isOverlap(boxIn2, box) && isOverlap(boxIn2, boxIn1) && !boxIn2.Equals(boxIn1) && !boxIn2.Equals(box) && !box.Equals(boxIn1))
                        {
                            sizeSum = sizeSum + AreaSize(OverlapBox(box, boxIn1, boxIn2));

                        }
                    }
                }
                boxesCalculated.Add(box);
            }
            return sizeSum;
        }

        public static List<Boundary> GetUnifiedRanges(List<Boundary> mergeBoxes)
        {
            // unify the overlapping and neighboring boxes

            List<Boundary> newBoxes = new List<Boundary>(mergeBoxes);
            List<Boundary> prevBoxes;

            int cntUnify = -1;

            // cycling in the loop until no unify happens
            while (cntUnify != 0)
            {
                cntUnify = 0;
                //dic record which boxes are unified already
                Dictionary<int, bool> boxDic = new Dictionary<int, bool>();

                prevBoxes = newBoxes;
                newBoxes = new List<Boundary>();

                for (int i = 0; i < prevBoxes.Count; i++)
                {
                    if (boxDic.ContainsKey(i)) continue;
                    bool markUnified = false;
                    // search for boxes overlap or neibouring with  the selected one
                    for (int j = i + 1; j < prevBoxes.Count; j++)
                    {
                        if (boxDic.ContainsKey(j)) continue;
                        if (isOverlapOrNeighbor(prevBoxes[i], prevBoxes[j]))
                        {
                            // unify the two boxes
                            Boundary newBox = UnifyBox(prevBoxes[i], prevBoxes[j]);
                            newBoxes.Add(newBox);

                            boxDic[j] = true;
                            markUnified = true;
                            cntUnify++;

                            break;
                        }
                    }
                    if (!markUnified) newBoxes.Add(prevBoxes[i]);
                }
            }
            return newBoxes;
        }

        public static double Min(List<int> arr)
        {
            if (arr.Count == 0)
                return int.MaxValue;

            return arr.Min();
        }

        public static double Max(List<int> arr)
        {
            if (arr.Count == 0)
                return int.MinValue;

            return arr.Max();
        }

        public static Boundary OverlapBox(Boundary box1, Boundary box2)
        {
            return new Boundary(
                Math.Max(box1.top, box2.top),
                Math.Min(box1.bottom, box2.bottom),
                Math.Max(box1.left, box2.left),
                Math.Min(box1.right, box2.right));
        }

        public static Boundary UnifyBox(Boundary box1, Boundary box2)
        {
            return new Boundary(
                Math.Min(box1.top, box2.top),
                Math.Max(box1.bottom, box2.bottom),
                Math.Min(box1.left, box2.left),
                Math.Max(box1.right, box2.right));
        }
        public static Boundary OverlapBox(Boundary box1, Boundary box2, Boundary box3)
        {
            return OverlapBox(box1, OverlapBox(box2, box3));
        }
        public static int GetNumberFormatCode(string format)
        {
            switch (format)
            {
                case "": return 0;
                case "General": return 0;
                case "0": return 1;
                case "0.00": return 1;
                case "#,##0": return 1;
                case "#,##0.00": return 1;
                case "0%": return 1;
                case "0.00%": return 1;
                case "0.00E+00": return 1;
                case "# ?/?": return 2;
                case "# ??/??": return 2;
                case "d/m/yyyy": return 3;
                case "d-mmm-yy": return 3;
                case "d-mmm": return 3;
                case "mmm-yy": return 3;
                case "h:mm tt": return 3;
                case "h:mm:ss tt": return 3;
                case "H:mm": return 3;
                case "H:mm:ss": return 3;
                case "m/d/yyyy H:mm": return 3;
                case "#,##0 ;(#,##0)": return 1;
                case "#,##0 ;[Red](#,##0)": return 1;
                case "#,##0.00;(#,##0.00)": return 1;
                case "#,##0.00;[Red](#,##0.00)": return 1;
                case "mm:ss": return 3;
                case "[h]:mm:ss": return 3;
                case "mmss.0": return 3;
                case "##0.0E+0": return 1;
                case "@": return 2;
                default: return 0;
            }
        }
        #endregion

        #region statistics features for rows/cols
        public static double average(List<double> Valist)
        {
            double sum = 0;
            foreach (double d in Valist)
            {
                sum = sum + d;
            }
            double revl = Math.Round(sum / Valist.Count, 3);
            return revl;
        }

        public static double stdev(List<double> ValList)
        {
            double avg = average(ValList);
            double sumstdev = 0;
            foreach (double d in ValList)
            {
                sumstdev = sumstdev + (d - avg) * (d - avg);
            }

            double stdeval = System.Math.Sqrt(sumstdev / ValList.Count);
            return Math.Round(stdeval, 3);
        }

        public static double r1(List<double> ValList1, List<double> ValList2)
        {
            double sumR1 = 0;
            for (int i = 0; i < ValList1.Count; i++)
            {
                sumR1 = sumR1 + Math.Abs(ValList1[i] - ValList2[i]);
            }
            return sumR1 / ValList1.Count;
        }

        public static double correl(List<double> array1, List<double> array2)
        {
            double avg1 = average(array1);
            double stdev1 = stdev(array1);

            double avg2 = average(array2);
            double stdev2 = stdev(array2);

            double sum = 0;
            for (int i = 0; i < array1.Count && i < array2.Count; i++)
            {
                sum = sum + (array1[i] - avg1) / stdev1 * ((array2[i] - avg2) / stdev2);
            }
            return Math.Round(sum / array1.Count, 3);
        }
        #endregion

        #region rank boxes
        public static List<Boundary> RankBoxesByLocation(List<Boundary> ranges)
        {
            ranges.Sort((x, y) => ComputeBoxByLocation(x).CompareTo(ComputeBoxByLocation(y)));
            return ranges;
        }

        public static List<Boundary> RankBoxesBySize(List<Boundary> ranges)
        {
            ranges.Sort((x, y) => AreaSize(y).CompareTo(AreaSize(x)));
            return ranges;
        }

        public static double ComputeBoxByLocation(Boundary box)
        {
            return box.left + 0.00001 * box.top;
        }
        #endregion

        #region box position related
        public static bool ContainsBox(Boundary box1, Boundary box2, int step = 0)
        {
            return box1.top <= box2.top + step && box1.bottom >= box2.bottom - step && box1.left <= box2.left + step && box1.right >= box2.right - step;
        }

        public static bool ContainsBox(List<Boundary> boxes1, Boundary box2, int step = 0)
        {
            return boxes1.Any(box1 => ContainsBox(box1, box2, step) && !box1.Equals(box2));
        }

        public static bool ContainsBox(Boundary box1, List<Boundary> boxes2, int step = 0)
        {
            return boxes2.Any(box2 => ContainsBox(box1, box2, step) && !box1.Equals(box2));
        }

        public static bool isOverlapOrNeighbor(Boundary box1, Boundary box2)
        {
            return !((box1.top > box2.bottom + 1) || (box1.bottom + 1 < box2.top) || (box1.left > box2.right + 1) || (box1.right + 1 < box2.left));
        }

        public static bool isOverlap(Boundary box1, List<Boundary> boxes2, bool exceptForward = false, bool exceptBackward = false, bool exceptSuppression = false)
        {
            return boxes2 != null && boxes2.Any(box2 => isOverlap(box1, box2, exceptForward, exceptBackward, exceptSuppression) && !box1.Equals(box2));
        }

        public static bool isOverlap(Boundary box1, Boundary box2)
        {
            return !((box1.top > box2.bottom) || (box1.bottom < box2.top) || (box1.left > box2.right) || (box1.right < box2.left));
        }

        public static bool isOverlap(Boundary box1, Boundary box2, bool exceptForward = false, bool exceptBackward = false, bool exceptSuppression = false)
        {
            return isOverlap(box1, box2) && !(exceptForward && ContainsBox(box1, box2)) && !(exceptBackward && ContainsBox(box2, box1)) && !(exceptSuppression && isSuppressionBox(box1, box2));
        }

        public static bool isSuppressionBox(Boundary box1, Boundary box2, int step = 2, int directionNum = -1)
        {
            for (int i = 0; i < 4; i++)
            {
                if (directionNum >= 0 && i != directionNum && directionNum != 4)
                {
                    if (Math.Abs(box1[i] - box2[i]) > 0)
                        return false;
                }
                else
                {
                    if (Math.Abs(box1[i] - box2[i]) > step)
                        return false;
                }
            }
            return true;
        }
        #endregion

        #region sheet map related
        /// <summary>
        /// Calculate sum value to each cell of the submatrix from upper-left to the cell.
        /// </summary>
        public static int[,] CalcSumMatrix(this int[,] valueMap)
        {
            // calculate the sum map from raw map
            int height = valueMap.GetLength(0);
            int width = valueMap.GetLength(1);
            if (height == 0 || width == 0)
                return valueMap;

            int[,] result = new int[height, width];
            result[0, 0] = valueMap[0, 0];
            for (int i = 1; i < height; ++i)
                result[i, 0] = result[i - 1, 0] + valueMap[i, 0];
            for (int j = 1; j < width; ++j)
                result[0, j] = result[0, j - 1] + valueMap[0, j];
            for (int i = 1; i < height; ++i)
                for (int j = 1; j < width; ++j)
                    result[i, j] = result[i - 1, j] + result[i, j - 1] - result[i - 1, j - 1] + valueMap[i, j];

            return result;
        }

        /// <summary>
        /// Summarize rectangular values in submatrix from summarized value map.
        /// </summary>
        public static int SubmatrixSum(this int[,] valueSumMap, Boundary box)
        {
            box.top = Math.Min(Math.Max(box.top, 1), valueSumMap.GetLength(0));
            box.bottom = Math.Min(Math.Max(box.bottom, 1), valueSumMap.GetLength(0));
            box.left = Math.Min(Math.Max(box.left, 1), valueSumMap.GetLength(1));
            box.right = Math.Min(Math.Max(box.right, 1), valueSumMap.GetLength(1));
            if (box.top > box.bottom || box.left > box.right) return 0;

            int result = valueSumMap[box.bottom - 1, box.right - 1];
            if (box.left > 1) result -= valueSumMap[box.bottom - 1, box.left - 2];
            if (box.top > 1) result -= valueSumMap[box.top - 2, box.right - 1];
            if (box.left > 1 && box.top > 1) result += valueSumMap[box.top - 2, box.left - 2];

            return result;
        }
        #endregion
    }

}
