# coding=utf-8
#
# Copyright 2020 Heinrich Heine University Duesseldorf
#
# Part of this code is based on the source code of BERT-DST
# (arXiv:1907.03040)
# Part of this code is based on the source code of Transformers
# (arXiv:1910.03771)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import torch
import copy
from torch import nn
from torch.nn import CrossEntropyLoss, BCEWithLogitsLoss

# from transformers.file_utils import (add_start_docstrings, add_start_docstrings_to_callable)
from transformers.modeling_bert import (BertModel, BertPreTrainedModel, BertOnlyMLMHead)


# @add_start_docstrings(
#     """BERT Model with a classification heads for the DST task. """,
#     BERT_START_DOCSTRING,
# )
class BertForDST(BertPreTrainedModel):
    def __init__(self, config):
        super(BertForDST, self).__init__(config)
        self.slot_list = config.dst_slot_list
        self.class_types = config.dst_class_types  # "none", "dontcare", "copy_value", "true", "false", "refer", "inform"
        self.class_labels = config.dst_class_labels  # len(dst_class_types)
        self.token_loss_for_nonpointable = config.dst_token_loss_for_nonpointable
        self.refer_loss_for_nonpointable = config.dst_refer_loss_for_nonpointable
        self.class_aux_feats_inform = config.dst_class_aux_feats_inform
        self.class_aux_feats_ds = config.dst_class_aux_feats_ds
        self.class_loss_ratio = config.dst_class_loss_ratio
        
        # Only use refer loss if refer class is present in dataset.
        if 'refer' in self.class_types:
            self.refer_index = self.class_types.index('refer')
        else:
            self.refer_index = -1
        
        self.copy_value_index = self.class_types.index('copy_value')
        
        self.bert = BertModel(config)
        self.dropout = nn.Dropout(config.dst_dropout_rate)
        self.dropout_heads = nn.Dropout(config.dst_heads_dropout_rate)
        
        #         if self.class_aux_feats_inform:
        #                 self.add_module("inform_projection", nn.Linear(len(self.slot_list), len(self.slot_list)))
        #         if self.class_aux_feats_ds:
        #             self.add_module("ds_projection", nn.Linear(len(self.slot_list), len(self.slot_list)))
        
        #         aux_dims = len(self.slot_list) * (
        #                 self.class_aux_feats_inform + self.class_aux_feats_ds)  # second term is 0, 1 or 2
        
        #         for slot in self.slot_list:
        #             self.add_module("class_" + slot,
        #                             nn.Linear(config.hidden_size + aux_dims, self.class_labels))  # 预测 slot gates
        #             self.add_module("token_" + slot, nn.Linear(config.hidden_size, 2))  # 预测 span
        #             self.add_module("refer_" + slot,
        #                             nn.Linear(config.hidden_size + aux_dims, len(self.slot_list) + 1))  # 预测 DS copy
        
        if config.phase == 'finetune':
            if self.class_aux_feats_inform:
                self.add_module("inform_projection", nn.Linear(len(self.slot_list), len(self.slot_list)))
            if self.class_aux_feats_ds:
                self.add_module("ds_projection", nn.Linear(len(self.slot_list), len(self.slot_list)))
            
            aux_dims = len(self.slot_list) * (
                    self.class_aux_feats_inform + self.class_aux_feats_ds)  # second term is 0, 1 or 2
            
            for slot in self.slot_list:
                self.add_module("class_" + slot,
                                nn.Linear(config.hidden_size + aux_dims, self.class_labels))  # 预测 slot gates
                self.add_module("token_" + slot, nn.Linear(config.hidden_size, 2))  # 预测 span
                self.add_module("refer_" + slot,
                                nn.Linear(config.hidden_size + aux_dims, len(self.slot_list) + 1))  # 预测 DS copy
        
        if config.phase == 'preview':
            # this is for preview ########
            self.cls = BertOnlyMLMHead(config)
            self.add_module("preview_span_weight1", nn.Linear(2 * config.hidden_size, config.hidden_size))
            self.add_module("preview_span_weight2", nn.Linear(config.hidden_size, 1))
            self.add_module("preview_class_weight1",
                            nn.Linear(2 * config.hidden_size, config.hidden_size))  # 两个 CLS 合在一起
            self.add_module("preview_class_weight2", nn.Linear(config.hidden_size, self.class_labels))
            #############
        
        self.init_weights()
    
    def resize(self, new_num_tokens=None):
        # 处理 mlm
        old_num_tokens, old_embedding_dim = self.cls.predictions.decoder.weight.size()
        
        # Build new embeddings
        new_decoder = nn.Linear(old_embedding_dim, new_num_tokens, bias=False)
        new_decoder.to(self.cls.predictions.decoder.weight.device)
        # initialize all new embeddings (in particular added tokens)
        self._init_weights(new_decoder)
        # Copy word embeddings from the previous weights
        num_tokens_to_copy = min(old_num_tokens, new_num_tokens)
        new_decoder.weight.data[:num_tokens_to_copy, :] = \
            self.cls.predictions.decoder.weight.data[:num_tokens_to_copy, :]
        self.cls.predictions.decoder = new_decoder
        
        # Build new bias
        new_bias = nn.Parameter(torch.zeros(new_num_tokens), requires_grad=True)
        new_bias.to(self.cls.predictions.bias.device)
        # Copy word embeddings from the previous weights
        new_bias.data[:num_tokens_to_copy] = \
            self.cls.predictions.bias.data[:num_tokens_to_copy]
        self.cls.predictions.bias = new_bias
        self.cls.predictions.decoder.bias = self.cls.predictions.bias
    
    def forward(self,
                input_ids,
                input_mask=None,
                segment_ids=None,
                position_ids=None,
                head_mask=None,
                start_pos=None,
                end_pos=None,
                inform_slot_id=None,  # for auxiliary inform label
                refer_id=None,  # DS memory refer label
                class_label_id=None,  # for slot gate label
                diag_state=None):  # for auxiliary ds label
        outputs = self.bert(
            input_ids,
            attention_mask=input_mask,
            token_type_ids=segment_ids,
            position_ids=position_ids,
            head_mask=head_mask
        )
        
        sequence_output = outputs[0]
        pooled_output = outputs[1]
        
        sequence_output = self.dropout(sequence_output)
        pooled_output = self.dropout(pooled_output)
        
        # TODO: establish proper format in labels already?
        if inform_slot_id is not None:
            inform_labels = torch.stack(list(inform_slot_id.values()), 1).float()  # auxiliary inform label
        if diag_state is not None:
            diag_state_labels = torch.clamp(torch.stack(list(diag_state.values()), 1).float(), 0.0,
                                            1.0)  # auxiliary ds label
        
        total_loss = 0
        per_slot_per_example_loss = {}
        per_slot_class_logits = {}
        per_slot_start_logits = {}
        per_slot_end_logits = {}
        per_slot_refer_logits = {}
        for slot in self.slot_list:
            if self.class_aux_feats_inform and self.class_aux_feats_ds:
                pooled_output_aux = torch.cat(
                    (pooled_output, self.inform_projection(inform_labels), self.ds_projection(diag_state_labels)), 1)
            elif self.class_aux_feats_inform:
                pooled_output_aux = torch.cat((pooled_output, self.inform_projection(inform_labels)), 1)
            elif self.class_aux_feats_ds:
                pooled_output_aux = torch.cat((pooled_output, self.ds_projection(diag_state_labels)), 1)
            else:
                pooled_output_aux = pooled_output
            class_logits = self.dropout_heads(getattr(self, 'class_' + slot)(pooled_output_aux))
            
            token_logits = self.dropout_heads(getattr(self, 'token_' + slot)(sequence_output))
            start_logits, end_logits = token_logits.split(1, dim=-1)
            start_logits = start_logits.squeeze(-1)
            end_logits = end_logits.squeeze(-1)
            
            refer_logits = self.dropout_heads(getattr(self, 'refer_' + slot)(pooled_output_aux))
            
            per_slot_class_logits[slot] = class_logits
            per_slot_start_logits[slot] = start_logits
            per_slot_end_logits[slot] = end_logits
            per_slot_refer_logits[slot] = refer_logits
            
            # If there are no labels, don't compute loss
            if class_label_id is not None and start_pos is not None and end_pos is not None and refer_id is not None:
                # If we are on multi-GPU, split add a dimension
                if len(start_pos[slot].size()) > 1:
                    start_pos[slot] = start_pos[slot].squeeze(-1)
                if len(end_pos[slot].size()) > 1:
                    end_pos[slot] = end_pos[slot].squeeze(-1)
                # sometimes the start/end positions are outside our model inputs, we ignore these terms
                ignored_index = start_logits.size(1)  # This is a single index
                start_pos[slot].clamp_(0, ignored_index)
                end_pos[slot].clamp_(0, ignored_index)
                
                # class_loss_fct = CrossEntropyLoss(reduction='none', weight=torch.tensor([0.8, 2., 1.0, 1.5, 1.5, 2., 1.]).cuda())
                class_loss_fct = CrossEntropyLoss(reduction='none')
                token_loss_fct = CrossEntropyLoss(reduction='none', ignore_index=ignored_index)
                refer_loss_fct = CrossEntropyLoss(reduction='none')
                
                start_loss = token_loss_fct(start_logits, start_pos[slot])
                end_loss = token_loss_fct(end_logits, end_pos[slot])
                token_loss = (start_loss + end_loss) / 2.0
                
                token_is_pointable = (start_pos[slot] > 0).float()
                # token_is_pointable = torch.eq(class_label_id[slot], self.copy_value_index).float()
                if not self.token_loss_for_nonpointable:
                    token_loss *= token_is_pointable
                
                refer_loss = refer_loss_fct(refer_logits, refer_id[slot])
                token_is_referrable = torch.eq(class_label_id[slot], self.refer_index).float()
                if not self.refer_loss_for_nonpointable:
                    refer_loss *= token_is_referrable
                
                class_loss = class_loss_fct(class_logits, class_label_id[slot])
                
                if self.refer_index > -1:
                    per_example_loss = (self.class_loss_ratio) * class_loss + (
                            (1 - self.class_loss_ratio) / 2) * token_loss + (
                                               (1 - self.class_loss_ratio) / 2) * refer_loss
                else:
                    per_example_loss = self.class_loss_ratio * class_loss + (1 - self.class_loss_ratio) * token_loss
                
                total_loss += per_example_loss.sum()
                per_slot_per_example_loss[slot] = per_example_loss
        
        # add hidden states and attention if they are here
        outputs = (total_loss,) + (
            per_slot_per_example_loss, per_slot_class_logits, per_slot_start_logits, per_slot_end_logits,
            per_slot_refer_logits,) + outputs[2:]
        
        return outputs
    
    def get_mlm_loss(self,
                     mlm_input_ids,
                     mlm_input_mask,
                     mlm_segment_ids,
                     masked_lm_labels,
                     ):
        mlm_outputs = self.bert(
            input_ids=mlm_input_ids,
            attention_mask=mlm_input_mask,
            token_type_ids=mlm_segment_ids)
        mlm_sequence_output = mlm_outputs[0]
        # print('mlm_sequence_output shape', mlm_sequence_output.shape)
        prediction_scores = self.cls(mlm_sequence_output)
        # print('prediction_scores shape:', prediction_scores.shape)
        
        loss_fct = CrossEntropyLoss()  # -100 index = padding token
        masked_lm_loss = loss_fct(prediction_scores.view(-1, self.config.vocab_size), masked_lm_labels.view(-1))
        return masked_lm_loss
    
    def preview(self,
                mlm_input_ids,
                mlm_input_mask,
                mlm_segment_ids,
                masked_lm_labels,
                schema_input_ids,
                schema_input_mask,
                schema_segment_ids,
                input_ids,
                input_mask,
                segment_ids,
                span_label_id=None,
                class_label_id=None,
                ):
        
        # print('mlm_input_ids shape:', mlm_input_ids.shape)
        # print('mlm_input_mask shape:', mlm_input_mask.shape)
        # print('mlm_segment_ids shape:', mlm_segment_ids.shape)
        # print('masked_lm_labels shape:', masked_lm_labels.shape)
        masked_lm_loss = self.get_mlm_loss(mlm_input_ids, mlm_input_mask, mlm_segment_ids, masked_lm_labels)
        
        # get schema
        schema_outputs = self.bert(
            input_ids=schema_input_ids,
            attention_mask=schema_input_mask,
            token_type_ids=schema_segment_ids
        )
        slot_encodings = schema_outputs[1]  # slotnum * hid
        
        outputs = self.bert(
            input_ids=input_ids,
            attention_mask=input_mask,
            token_type_ids=segment_ids)
        
        sequence_output = outputs[0]  # b * seq * hid
        pooled_output = outputs[1]  # b * hid
        
        total_span_loss = 0
        total_class_loss = 0
        
        class_loss_fct = CrossEntropyLoss(reduction='none')
        span_loss_fct = BCEWithLogitsLoss(reduction='none', )
        bsz, seqlen = input_ids.shape[0], input_ids.shape[1]
        
        for idx, slot in enumerate(self.slot_list):
            slot_emb = slot_encodings[idx]
            slot_emb_tile = slot_emb.repeat(bsz, 1)
            hid_vec = self.preview_class_weight1(torch.cat([pooled_output, slot_emb_tile], 1))
            class_logits = self.preview_class_weight2(self.dropout(hid_vec))
            class_loss = class_loss_fct(class_logits, class_label_id[slot])
            total_class_loss += class_loss.mean()
            
            if slot not in ['hotel-parking', 'hotel-internet', 'hotel-type']:
                slot_emb_tile2 = slot_emb.repeat(bsz, seqlen, 1)
                token_vec = self.preview_span_weight1(torch.cat([sequence_output, slot_emb_tile2], 2))  # b * seq * 2hid
                span_logits = torch.squeeze(self.preview_span_weight2(self.dropout(token_vec)), -1)  # b * seq
                span_loss = span_loss_fct(span_logits, span_label_id[slot])  # b*seq
                #                 A = torch.ones_like(span_label_id[slot])
                #                 B = torch.ones_like(span_label_id[slot])
                #                 C = torch.where(span_label_id[slot].to(torch.bool), A, B)
                #                 span_loss_masked = span_loss * input_mask * C
                span_loss_masked = span_loss * input_mask
                total_span_loss += span_loss_masked.mean()
        
        total_loss = 0.1 * total_span_loss + 0.1 * total_class_loss + 0.8 * masked_lm_loss
        return total_loss, total_class_loss, total_span_loss, masked_lm_loss
    
    def one_hot(self, label, class_num):
        label_ = torch.unsqueeze(label, 1)
        batch_size = label_.size(0)
        one_hot = torch.zeros(batch_size, class_num).scatter_(1, label_, 1)
        return one_hot
    
    def presudo_infer(self,
                      input_ids,
                      input_mask=None,
                      segment_ids=None,
                      position_ids=None,
                      head_mask=None,
                      start_pos=None,
                      end_pos=None,
                      inform_slot_id=None,  # for auxiliary inform label
                      refer_id=None,  # DS memory refer label
                      class_label_id=None,  # for slot gate label
                      diag_state=None):  # for auxiliary ds label
        
        total_loss = 0
        per_slot_per_example_loss = {}
        per_slot_class_logits = {}
        per_slot_start_logits = {}
        per_slot_end_logits = {}
        per_slot_refer_logits = {}
        
        for slot in self.slot_list:
            per_slot_class_logits[slot] = self.one_hot(class_label_id[slot], self.class_labels)
            per_slot_start_logits[slot] = self.one_hot(start_pos[slot], input_ids.size(1))
            per_slot_end_logits[slot] = self.one_hot(end_pos[slot], input_ids.size(1))
            per_slot_refer_logits[slot] = self.one_hot(refer_id[slot], len(self.slot_list) + 1)
            
            per_slot_per_example_loss[slot] = 0
        
        # add hidden states and attention if they are here
        outputs = (total_loss,) + (
            per_slot_per_example_loss, per_slot_class_logits, per_slot_start_logits, per_slot_end_logits,
            per_slot_refer_logits,)
        
        return outputs
