import argparse
import logging
import copy
import readers as dp 
import numpy as np

from keras.layers import Dense, Input
from keras.models import Model
from keras import losses
from keras.utils import to_categorical
from keras.callbacks import Callback

from sklearn.metrics import accuracy_score

#######################
#   helper functions
#######################

def get_data(data):
    X_dat = []
    y_dat = []
    toks = []
    for batch in data:
        y_small = []
        x_small = []
        tok_small = []
        for t,y,x in batch:  
            x_small.append(x)
            y_small.append(y)
            tok_small.append(t)
        X_dat.append(np.stack(x_small[:]))
        y_dat.append(np.stack([to_categorical(y,4) for y in y_small]))
        toks.append(tok_small[:])
    return X_dat, y_dat, toks

def flatten_data(data):
    result = []
    for i in data:
        result.extend(i)    
    return np.stack(result)

logging.basicConfig(
    format='%(asctime)s %(levelname)-8s %(message)s',
    level=logging.INFO,
    datefmt='%Y-%m-%d %H:%M:%S')

from active_learning import InstanceSampler
from user_simulation import UserQuery

def main():
    parser = argparse.ArgumentParser(description='Code for the paper empowering active learning to jointly optimize system and user demands.')
    parser.add_argument('--train', default='data/train.txt', help='Path to main train data')
    parser.add_argument('--test', default='data/test.txt', help='Path to main test data')
    parser.add_argument('--seed', type=int, default=42, help='random seed')
    parser.add_argument('--epochs', type=int, default=50, help='number of epochs')
    parser.add_argument('--init-weights', default='models/init.pt', help='Path to model initialization')
    parser.add_argument('--best-model', default='models/best_model.pt', help='Path to store best model')
    parser.add_argument('--sampling-strategy', default='joint', help='sampling strategies: random, uncertainty, user, combined, tradeoff')
    parser.add_argument('--user-static-class',type=int,default=2, help='User proficiency for their level.')
    parser.add_argument('--user-step-size', type=int, default=8, help='t for increasing or decreasing proficiency')
    parser.add_argument('--user-strategy', default='interruped', help='Learner behavior. increasing (motivated), decreasing, interruped, or static.')
    parser.add_argument('--al-iterations', type=int, default=40, help='Number of sampling steps')
    parser.add_argument('--history', default='results/history.txt')
    parser.add_argument('--results', default='results/scores.txt', help='Path to result file')
    parser.add_argument('--uncertainty',default='softmax')
    parser.add_argument('--lambda-schedule', default='None', help='schedule for decreasing lambda. None or root')
    args = parser.parse_args()

    np.random.seed(args.seed)

    ########################
    #      Init stuff
    ########################
    train_dev = dp.load_data(args.train)
    test = dp.load_data(args.test)

    # Class for adding acc measure during training
    class AccuracyScore(Callback):
        def on_train_begin(self, logs={}):
            self.best_acc = 0.0
            self.no_improvment_counter = 0
        def on_epoch_end(self, batch, logs={}):
            # Get predictions
            predict = self.model.predict(self.validation_data[0])
            class_preds = [np.argmax(x) for x in predict]
            true = [np.argmax(x) for x in self.validation_data[1]]

            self.acc=accuracy_score(true, class_preds)
            if self.acc > self.best_acc:
                self.best_acc=self.acc
                self.model.save_weights(args.best_model)
                self.no_improvment_counter = 0
            else:
                self.no_improvment_counter += 1
            if self.no_improvment_counter > 20: # Early stopping at 20
                self.model.stop_training = True 
            return 

    acc_met= AccuracyScore()

    inputs = Input(shape=(61,))
    layer1 = Dense(61)(inputs)
    output_layer = Dense(4,activation='softmax')(layer1)
    model = Model(inputs=inputs,outputs=output_layer)
    model.compile(optimizer='adam', loss=losses.categorical_crossentropy)
    print(model.summary())

    logging.info("Start training setup...")

    #######################
    #   Active learning
    #######################
    # Init sampler and set strategy
    instance_sampler = InstanceSampler() 
    instance_sampler.set_lambda(args.lambda_schedule) 
    instance_sampler.set_uncertainty_comp(args.uncertainty)
    instance_sampler.set_sampling_strategy(args.sampling_strategy)
    # Init user simulation model
    user_simulation = UserQuery(step_size=args.user_step_size, 
                            strategy=args.user_strategy,
                            static_class=args.user_static_class)

    try:
        model.load_weights(args.init_weights)
    except OSError:
        model.save_weights(args.best_model)
        model.load_weights(args.best_model)

    al_train_data = []
    al_pool_data = copy.deepcopy(train_dev) # Note, we only have a single task!
    user_queries = []
    sample_history = []
    sample_index_history = []
    al_history = open(args.history,'w')
    resultlog = open(args.results,'w')
    resultlog.write("Iteration\tDelta U\tM-Acc\n")
    al_history.write("Iteration\tUser Goal\tPredicted Instance\tSampled Instances\tClassified Instances\n")
    # Start AL
    for iteration in range(args.al_iterations):
        # Load previous model:
        model.load_weights(args.best_model)
        logging.info("Iteration {} of {}.".format(iteration, int(args.al_iterations)))
        # Number of total iterations is al_iterations divided by batch size
        user_query = user_simulation.sample_class()
        user_queries.append(user_query)
        instance_sampler.update_lambda(iteration) # Update lambda if we have a strategy selected
        instance_sampler.set_user_goal(user_query)
        x_al, y_al, tok_al = get_data(al_pool_data)
        logits = [model.predict(x) for i,x in enumerate(x_al) if i not in sample_index_history]
        instance_sampler.set_pred_probs(logits)
        instance_sampler.sample_instance()
        # Update the training and pool data
        sample_index = instance_sampler.get_sampled_instances()
        al_train_data.append(al_pool_data[sample_index])
        sample_index_history.append(sample_index)
        
        instance_sampler.finish_iteration() # Do clean up after a sampling iteration
        logging.info("Already sampled instances: {}.".format(len(sample_index_history)))
        logging.info("Remaining pool data {}.".format(len(al_pool_data)))
        X_train, y_train, y_tok = get_data(al_train_data)
        # TODO: hacky way 
        if len(X_train) > 1:
            X_train = flatten_data(X_train)
            y_train = flatten_data(y_train)
        model.fit(X_train,y_train, batch_size=20, epochs=args.epochs, validation_split=0.2,shuffle=True,verbose=0, callbacks=[acc_met])
        X_test, y_test, y_tok = get_data(test)
        test_pred = [model.predict(x) for x in X_test]
        y_true, y_pred = [],[]
        for i in test_pred:
            y_pred.extend([np.argmax(x) for x in i])
        for i in y_test:
            y_true.extend([np.argmax(x) for x in i])
        # Evaluate user objective
        _,_,gaps = get_data([al_pool_data[sample_index]])
        estimated_difficulty = user_simulation.get_ctest_class(gaps[0])
        resultlog.write("{}\t{}\t{}\n".format(iteration,abs(user_query-estimated_difficulty),accuracy_score(y_true,y_pred)))
        al_history.write("{}\t{}\t{}\t{}\t{}\n".format(iteration, user_query, estimated_difficulty, ','.join([str(x) for x in y_pred]), ','.join([str(x) for x in y_true])))
    al_history.close()
    resultlog.close()



if __name__ == '__main__':
    main()
