#include <fstream>
#include <algorithm>

#include "token_manager.h"
#include "include/basic.h"
#include "io/utility.h"

Token TokenManager::lemmas_;
Token TokenManager::node_labels_;
Token TokenManager::edge_labels_;
Token TokenManager::senses_;

// load Tokenmanually
std::vector< ttoken > TokenManager::states_;
std::vector< int > TokenManager::states_appropriate_var_count_;
std::vector< ttoken > TokenManager::literals_;
std::unordered_map< std::string, std::vector< int > > TokenManager::states_map_;
std::unordered_map< std::string, int > TokenManager::states_to_indices_;

bool TokenManager::loadTokens(const std::string& file_name,
                              std::vector< ttoken >& tokens) {
    std::ifstream is(file_name);

    if (!is) {
        LOG_ERROR(<< "Can't found file " << file_name);
        return false;
    }

    std::string token;
    while (is >> token)
        tokens.emplace_back(std::move(token));

    return true;
}

bool TokenManager::loadStates(const std::string& state_file) {
    if (!loadTokens(state_file, states_))
        return false;
    int index = 0;
    for (auto& state_token : states_) {
        int var_count;
        if (index < EMPTY_STATE_COUNT)
            var_count = 0;
        else if (std::isupper(state_token[0])) {
            var_count = std::count(state_token.begin(),
                                   state_token.end(), '_') +
                        1;
        } else
            var_count = 1;
        states_appropriate_var_count_.push_back(var_count);

        auto pos = state_token.find_last_of(':');
        if (pos != std::string::npos)
            states_map_[state_token.substr(pos + 1)].push_back(index);

        states_to_indices_[state_token] = index;
        ++index;
    }
    return true;
}

int TokenManager::indexOfState(const ttoken& state) {
    auto iter = states_to_indices_.find(state);
    if (iter == states_to_indices_.end()) {
        int index = states_.size();
        states_to_indices_[state] = index;
        states_.push_back(state);
        return states_.size();
    }
    return iter->second;
}

bool TokenManager::stateMatch(int state_index, const ttoken& node_label) {
    if (state_index < EMPTY_STATE_COUNT || node_label[0] == '_')
        return true;
    int size = node_label.size();
    if (size > 2 &&
        stringEndsWith(node_label, "_q") &&
        stringStartsWith(states_[state_index], "DET"))
        return true;
    const auto& state = states_[state_index];
    return stringStartsWith(state, node_label) &&
           state[node_label.length()] == ':';
}
