#include <fstream>
#include <algorithm>

#include <boost/filesystem.hpp>

#include "include/basic.h"
#include "io/token_manager.h"
#include "drawer.h"

namespace fs = boost::filesystem;

void drawStream(const std::string& filename_base,
                const std::vector< int > stream_target,
                const std::unordered_map< EdsGraph::Arc, RuleSet::State >& edge_states,
                const EdsGraph& graph) {
    fs::path tmp("dots");
    std::string filename(filename_base + ".dot");
    std::replace(filename.begin(), filename.end(), '/', '-');
    LOG_DEBUG(<< "Write dot file: " << filename);
    try {
        if (!fs::exists(tmp))
            fs::create_directory(tmp);
        filename = (tmp / filename).string();
    } catch (fs::filesystem_error& err) {
        LOG_ERROR(<< "filesystem_error: " << err.what());
        return;
    }
    std::ofstream os(filename);
    if (!os) {
        LOG_ERROR(<< "Can not create file " << filename);
        return;
    }

    os << "digraph {\n";
    int node_count = graph.nodes.size();
    for (int node_index = 0; node_index < node_count; ++node_index) {
        auto& node = graph.nodes[node_index];
        os << node_index << "[label=\"" << node.toString() << "\"];\n";
    }
    for (int node_index = 0; node_index < node_count; ++node_index) {
        auto& node = graph.nodes[node_index];
        for (auto& e : node.out_edges) {
            auto arc = MAKE_ARC(node_index, e.target_index);
            auto iter = edge_states.find(arc);
            os << node_index << " -> " << e.target_index << "[label=\"";
            if (iter != edge_states.end())
                printState(os, iter->second);
            else
                os << "??";
            os << "\"];\n";
        }
        if (stream_target[node_index] != -1)
            os << node_index << " -> " << stream_target[node_index]
               << "[color=red];\n";
    }

    os << "}";
    os.close();
}
