#include <unordered_set>

#include "include/basic.h"
#include "io/token_manager.h"
#include "solver.h"

#ifndef NDEBUG
static std::unordered_set< int > visited_nodes;
static std::vector< std::pair< int, int > > history_stack;
#endif

void EquationSolverBase::initialize() {
    pron_index = TokenManager::indexOfNodeLabel("pron");
    poss_index = TokenManager::indexOfNodeLabel("poss");
    sense_1_index = TokenManager::indexOfSense("1");
    parentheical_index = TokenManager::indexOfNodeLabel("parentheical");
}

void EquationSolverBase::solve(std::string& result) {
    assert(transducer_ptr_->hasResultComputed());
    assert(transducer_ptr_->currentTopIndex() != -1);

#ifndef NDEBUG
    visited_nodes.clear();
    history_stack.clear();
#endif

    solveVariable(transducer_ptr_->currentTopIndex(), 0, result);
}

void EquationSolverBase::computeSpecialLemma(const EdsGraph::Node& node,
                                             int node_index,
                                             std::string& result) {
    result.push_back('<');
    // int target_index = transducer_ptr_->currentStreamTarget(node_index);
    // auto& graph = transducer_ptr_->currentGraph();
    // // poss <- pron 的情况
    // if (node.label_index == pron_index &&
    //     target_index != -1 &&
    //     graph.nodes[target_index].label_index == poss_index)
    //     result.append("pron-poss");
    // else
    //     result.append(TokenManager::nodeLabelAt(node.label_index));
    result.append(TokenManager::nodeLabelAt(node.label_index));
    result.push_back('>');
}

void EquationSolverBase::solveVariable(int node_index, int var_index,
                                       std::string& result) {

    auto& graph = transducer_ptr_->currentGraph();

#ifndef NDEBUG
    if (visited_nodes.count(node_index)) {
        LOG_ERROR(<< "Stream has a cycle " << graph.filename);
        return;
    }
    visited_nodes.insert(node_index);
    history_stack.push_back({node_index, var_index});
#endif

    // 计算 node_index 结点 输出的第 var_index 个变量
    auto rule_index = transducer_ptr_->currentRuleChoice(node_index);
    auto& rule = transducer_ptr_->ruleAt(rule_index);

// bool is_parentheical = graph.nodes[node_index].label_index == parentheical_index &&
//                        rule.tail.equations.size() == 1;
// if (is_parentheical)
//     result.append("( ");

#ifndef NDEBUG
    bool flag = rule.tail.equations.size() > ( std::size_t )var_index;
    if (!flag) {
        for (auto& history : history_stack) {
            int i = history.first, v = history.second;
            int rule_index = transducer_ptr_->currentRuleChoice(i);
            std::cerr << transducer_ptr_->currentGraph().nodes[i]
                      << '@' << v << ' ';
            std::cerr << transducer_ptr_->ruleAt(rule_index) << std::endl;
        }
    }
    assert(flag);
#endif

    auto& node = graph.nodes[node_index];
    auto& equation = rule.tail.equations[var_index];

    for (std::size_t j = 1; j < equation.size(); ++j) {
        int var = equation[j];
        if (IS_LEMMA_VAR(var)) {
            bool append_hyphen = false;
            if (node.carg != NULL_CARG) {
                result.append(node.carg);
                if (result.back() == '-') {
                    append_hyphen = true;
                    result.pop_back();
                }
            } else if (node.is_special)
                computeSpecialLemma(node, node_index, result);
            else
                result.append(TokenManager::lemmaAt(node.lemma_index));
            for (int i = 0; i < 5; ++i) {
                result.push_back('_');
                result.append(node.properties[i]);
            }
            if (append_hyphen)
                result.append(" \\@-\\@");
        } else if (IS_SENSE_VAR(var)) {
            if (node.sense_index != sense_1_index) {
                // result.push_back('[');
                result.append(TokenManager::senseAt(node.sense_index));
                // result.push_back(']');
            }
        } else if (IS_THAT_VAR(var)) {
            result.append(THAT_VAR);
        } else if (IS_LITERAL_VAR(var)) {
            // result.push_back('"');
            result.append(TokenManager::literalAt(var));
            // result.push_back('"');
        } else {
            int next_index =
                transducer_ptr_->targetOrSourceIndex(node_index,
                                                     VAR_MAJOR(var));
            int target_index =
                transducer_ptr_->currentStreamTarget(next_index);
            if (target_index == node_index)
                solveVariable(next_index, VAR_MINOR(var), result);
            else
                LOG_DEBUG(<< "Solver: target index ("
                          << target_index << ") != current index ("
                          << node_index << ')');
        }
        if (j != equation.size() - 1)
            result.push_back(' ');
    }

// if (is_parentheical)
//     result.append(" )");

#ifndef NDEBUG
    visited_nodes.erase(node_index);
    history_stack.pop_back();
#endif
}
