function X = lyapunov_symmetric_eig(V, lambda, C, tol)
% Solves AX + XA = C when A = A', as a pseudo-inverse, given eig(A).
%
% function X = lyapunov_symmetric_eig(V, lambda, C)
% function X = lyapunov_symmetric_eig(V, lambda, C, tol)
%
% Same as lyapunov_symmetric(A, C, [tol]), where A is symmetric, its
% eigenvalue decomposition [V, lambda] = eig(A, 'vector') is provided as
% input directly, and C is a single matrix of the same size as A.
%
% See also: lyapunov_symmetric sylvester lyap sylvester_nocheck

% This file is part of Manopt: www.manopt.org.
% Original author: Nicolas Boumal, Aug. 31, 2018.
% Contributors: 
% Change log: 

    % AX + XA = C  is equivalent to DY + YD = M with
    % Y = V'XV, M = V'CV and D = diag(lambda).
    M = V'*C*V;
    
    % W(i, j) = lambda(i) + lambda(j)
    W = bsxfun(@plus, lambda, lambda');
    
    % Normally, the solution Y is simply this:
    Y = M ./ W;
    
    % But this may involve divisions by (almost) 0 in certain places.
    % Thus, we go for a pseudo-inverse.
    absW = abs(W);
    if ~exist('tol', 'var') || isempty(tol)
        tol = numel(C)*eps(max(absW(:))); % similar to pinv tolerance
    end
    Y(absW <= tol) = 0;
    
    % Undo the change of variable
    X = V*Y*V';

end
