function M = obliquecomplexfactory(n, m, transposed)
% Returns a manifold struct defining complex matrices w/ unit-norm columns.
%
% function M = obliquecomplexfactory(n, m)
% function M = obliquecomplexfactory(n, m, transposed)
%
% Oblique manifold: deals with complex matrices of size n x m such that
% each column has unit 2-norm, i.e., is a point on the unit sphere in C^n.
% The geometry is a product geometry of m unit spheres in C^n. For the
% metric, C^n is treated as R^(2n), so that the real part and imaginary
% parts are treated separately as 2n real coordinates. As such, the complex
% oblique manifold is a Riemannian submanifold of (R^2)^(n x m), with the
% usual metric <u, v> = real(u'*v).
% 
% If transposed is set to true (it is false by default), then the matrices
% are transposed: a point Y on the manifold is a matrix of size m x n and
% each row has unit 2-norm. It is the same geometry, just a different
% representation.
%
% In transposed form, a point Y is such that Y*Y' is a Hermitian, positive
% semidefinite matrix of size m and of rank at most n, such that all the
% diagonal entries are equal to 1.
%
% Note: obliquecomplexfactory(1, n, true) is equivalent to (but potentially
% slower than) complexcirclefactory(n).
%
% See also: spherecomplexfactory complexcirclefactory obliquefactory

% This file is part of Manopt: www.manopt.org.
% Original author: Nicolas Boumal, Sep. 3, 2014.
% Contributors: 
% Change log: 
%
%   Oct. 21, 2016 (NB)
%       Formatted for inclusion in Manopt release.
%
%   July 20, 2017 (NB)
%       Distance function is now accurate for close-by points. See notes
%       inside the spherefactory file for details. Also improvies distances
%       computation as part of the log function.

    
    if ~exist('transposed', 'var') || isempty(transposed)
        transposed = false;
    end
    
    if transposed
        trnsp = @(X) X.';
    else
        trnsp = @(X) X;
    end

    M.name = @() sprintf('Complex oblique manifold COB(%d, %d)', n, m);
    
    M.dim = @() (2*n-1)*m;
    
    M.inner = @(x, d1, d2) real(d1(:)'*d2(:));
    
    M.norm = @(x, d) norm(d(:));
    
    M.dist = @(x, y) norm(real(2*asin(.5*sqrt(sum(trnsp(abs(x - y).^2), 1)))));
    
    M.typicaldist = @() pi*sqrt(m);
    
    M.proj = @(X, U) trnsp(projection(trnsp(X), trnsp(U)));
    
    M.tangent = M.proj;
    
    % For Riemannian submanifolds, converting a Euclidean gradient into a
    % Riemannian gradient amounts to an orthogonal projection.
	M.egrad2rgrad = M.proj;
    
    M.ehess2rhess = @ehess2rhess;
    function rhess = ehess2rhess(X, egrad, ehess, U)
        X = trnsp(X);
        egrad = trnsp(egrad);
        ehess = trnsp(ehess);
        U = trnsp(U);
        
        PXehess = projection(X, ehess);
        inners = sum(real(conj(X).*egrad), 1);
        rhess = PXehess - bsxfun(@times, U, inners);
        
        rhess = trnsp(rhess);
    end
    
    M.exp = @exponential;
    % Exponential on the complex oblique manifold
    function y = exponential(x, d, t)
        x = trnsp(x);
        d = trnsp(d);
        
        if nargin == 2
            % t = 1;
            td = d;
        else
            td = t*d;
        end

        nrm_td = sqrt(sum(real(td).^2 + imag(td).^2, 1));

        y = bsxfun(@times, x, cos(nrm_td)) + ...
            bsxfun(@times, td, sin(nrm_td) ./ nrm_td);
        
        % For those columns where the step is 0, replace y by x
        exclude = (nrm_td == 0);
        y(:, exclude) = x(:, exclude);

        y = trnsp(y);
    end

    M.log = @logarithm;
    function v = logarithm(x1, x2)
        x1 = trnsp(x1);
        x2 = trnsp(x2);
        
        v = projection(x1, x2 - x1);
        dists = real(2*asin(.5*sqrt(sum(trnsp(abs(x - y).^2), 1))));
        norms = sqrt(sum(real(v).^2 + imag(v).^2, 1));
		factors = dists./norms;
        % For very close points, dists is almost equal to norms, but
        % because they are both almost zero, the division above can return
        % NaN's. To avoid that, we force those ratios to 1.
		factors(dists <= 1e-10) = 1;
		v = bsxfun(@times, v, factors);
        
        v = trnsp(v);
    end

    M.retr = @retraction;
    % Retraction on the oblique manifold
    function y = retraction(x, d, t)
        x = trnsp(x);
        d = trnsp(d);
        
        if nargin < 3
            td = d;
        else
            td = t*d;
        end

        y = normalize_columns(x + td);
        
        y = trnsp(y);
    end

    M.hash = @(x) ['z' hashmd5([real(x(:)) ; imag(x(:))])];
    
    M.rand = @() trnsp(random(n, m));
    
    M.randvec = @(x) trnsp(randomvec(n, m, trnsp(x)));
    
    M.lincomb = @matrixlincomb;
    
    M.zerovec = @(x) trnsp(zeros(n, m));
    
    M.transp = @(x1, x2, d) M.proj(x2, d);
    
    M.pairmean = @pairmean;
    function y = pairmean(x1, x2)
        y = trnsp(x1+x2);
        y = normalize_columns(y);
        y = trnsp(y);
    end

    % vec returns a vector representation of an input tangent vector which
    % is represented as a matrix. mat returns the original matrix
    % representation of the input vector representation of a tangent
    % vector. vec and mat are thus inverse of each other. They are
    % furthermore isometries between a subspace of R^2nm and the tangent
    % space at x.
    vect = @(X) X(:);
    M.vec = @(x, u_mat) [vect(real(trnsp(u_mat))) ; ...
                         vect(imag(trnsp(u_mat)))];
    M.mat = @(x, u_vec)    trnsp(reshape(u_vec(1:(n*m)),     [n, m])) + ...
                        1i*trnsp(reshape(u_vec((n*m+1):end), [n, m]));
    M.vecmatareisometries = @() true;

end

% Given a matrix X, returns the same matrix but with each column scaled so
% that they have unit 2-norm.
function X = normalize_columns(X)
	norms = sqrt(sum(real(X).^2 + imag(X).^2, 1));
	X = bsxfun(@times, X, 1./norms);
end

% Orthogonal projection of the ambient vector H onto the tangent space at X
function PXH = projection(X, H)

    % Compute the inner product between each vector H(:, i) with its root
    % point X(:, i), that is, real(X(:, i)' * H(:, i)).
    % Returns a row vector.
    inners = real(sum(conj(X).*H, 1));
    
    % Subtract from H the components of the H(:, i)'s that are parallel to
    % the root points X(:, i).
    PXH = H - bsxfun(@times, X, inners);

end

% Uniform random sampling on the sphere.
function x = random(n, m)

    x = normalize_columns(randn(n, m) + 1i*randn(n, m));

end

% Random normalized tangent vector at x.
function d = randomvec(n, m, x)

    d = randn(n, m) + 1i*randn(n, m);
    d = projection(x, d);
    d = d / norm(d(:));

end
