import argparse
import asyncio
import os
import sys

import flask
from feedback_module.modules import FeedbackModule
from flask import Flask, jsonify, request
from flask_cors import CORS
from openai import OpenAI
from prenegotiation_feedback.modules import PreNegotiationFeedbackModule

sys.path.insert(0, "../")


client = OpenAI(
    api_key=os.environ["OPENAI_API_KEY"],
    organization=os.environ["OPENAI_ORGANIZATION_KEY"]
)

pre_negotiation_cache = {}
turnwise_cache = {}
overall_cache = {}


class API:
    def __init__(self):
        self.prenegotiation = PreNegotiationFeedbackModule()
        self.negotiation = FeedbackModule()

    def get_prenegotiation_feedback(self, history, deal_data):
        return self.prenegotiation.get_prenegotiation_feedback(history=history, deal_data=deal_data)

    def get_turnwise_feedback(self, history, params):
        return self.negotiation.get_turnwise_feedback(history=history, params=params)

    def get_overall_feedback(self, history):
        return self.negotiation.get_overall_feedback(history=history)


if __name__ == "__main__":

    api = API()

    parser = argparse.ArgumentParser(description="Simple API for chat bot")
    parser.add_argument(
        "--hostname", default="localhost", help="web server hostname."
    )
    parser.add_argument(
        "--port", type=int, default=35496, help="web server port."
    )
    parser.add_argument(
        "--serving_hostname", default="0.0.0.0", help="API web server hostname."
    )
    parser.add_argument(
        "--serving_port", type=int, default=8080, help="API web server port."
    )

    args = parser.parse_args()

    hostname = args.hostname
    port = args.port
    serving_hostname = args.serving_hostname
    serving_port = args.serving_port

    app = Flask(__name__)
    CORS(app)
    blueprint = flask.Blueprint(
        "feedback_module", __name__, template_folder="templates"
    )

    TURN_LOOKUP = {}

    PUNK = ["?", ".", "!"]

    class Feedback:
        @staticmethod
        async def send_prenegotiation_feedback(history, deal_data):
            return api.get_prenegotiation_feedback(history, deal_data)

        @staticmethod
        async def send_turnwise_feedback(history, params):
            return api.get_turnwise_feedback(history, params)

        @staticmethod
        async def send_overall_feedback(history):
            return api.get_overall_feedback(history)

    @blueprint.route("/api/v1/prenegotiation_feedback", methods=["POST"])
    def send_prenegotiation_feedback():
        data = request.get_json()
        
        chat_id = data[1]['data'][0]['id']
        if pre_negotiation_cache.get(chat_id, None) is not None:
            return pre_negotiation_cache.get(chat_id, None)

        loop = asyncio.new_event_loop()
        asyncio.set_event_loop(loop)

        answers = data[0].get("data", [])
        deal_data = data[-2].get("data")

        result = loop.run_until_complete(Feedback.send_prenegotiation_feedback(answers, deal_data))
        chat_id = data[1]['data'][0]['id']
        pre_negotiation_cache[chat_id] = result
        return result

    @blueprint.route("/api/v1/turnwise_feedback", methods=["POST"])
    def send_turnwise_feedback():
        data = request.get_json()

        chat_id = data[1]['data'][0]['id']
        if turnwise_cache.get(chat_id, None) is not None:
            return turnwise_cache.get(chat_id, None)

        loop = asyncio.new_event_loop()
        asyncio.set_event_loop(loop)

        answers = data[0].get("data", [])

        history, parameters = data[1].get("data", [])[:], {
            "targetPrice": int(answers[1]["answer"]),
            "reservationPrice": int(answers[0]["answer"]),
        }

        result = loop.run_until_complete(
            Feedback.send_turnwise_feedback(history, parameters)
        )
        chat_id = data[1]['data'][0]['id']
        turnwise_cache[chat_id] = result
        return result

    @blueprint.route("/api/v1/overall_feedback", methods=["POST"])
    def send_overall_feedback():
        data = request.get_json()

        chat_id = data[1]['data'][0]['id']
        if overall_cache.get(chat_id, None) is not None:
            return overall_cache.get(chat_id, None)

        loop = asyncio.new_event_loop()
        asyncio.set_event_loop(loop)

        history = data[1].get("data", [])[:]

        result = loop.run_until_complete(Feedback.send_overall_feedback(history))
        chat_id = data[1]['data'][0]['id']
        overall_cache[chat_id] = result
        return result
    
    @blueprint.route("/api/v1/summary_feedback", methods=["POST"])
    def send_summary_feedback():
        data = request.get_json()
        chat_id = data[1]['data'][0]['id']
        result = {
            'pre_negotiation_feedback': pre_negotiation_cache.get(chat_id, None),
            'turnwise_feedback': turnwise_cache.get(chat_id, None),
            'overall_feedback': overall_cache.get(chat_id, None)
        }
        return result

    @blueprint.route("/health", methods=["GET"])
    def get_health():
        return "OK"

    async def main():
        app.register_blueprint(blueprint)
        app.run(host=serving_hostname, port=serving_port, debug=False)

    main_loop = asyncio.get_event_loop()
    main_loop.run_until_complete(main())
