# coding=utf-8
# Copyright 2022 The HuggingFace Datasets Authors and the current dataset script contributor.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
The LTI LangID corpus is a dataset for language identification.
The most recent version, v5, contains training data for 1266 languages, and some (possibly very tiny) amount of text for a total of 1706 languages.
"""
import os
import subprocess
from glob import glob
from pathlib import Path
from typing import Dict, List, Tuple

import datasets

from seacrowd.utils import schemas
from seacrowd.utils.configs import SEACrowdConfig
from seacrowd.utils.constants import Licenses, Tasks

_CITATION = """\
@inproceedings{brown2014non,
  title={Non-linear mapping for improved identification of 1300+ languages},
  author={Brown, Ralf D},
  booktitle={Proceedings of the 2014 Conference on Empirical Methods in Natural Language Processing (EMNLP)},
  pages={627--632},
  year={2014}
}
"""

_DATASETNAME = "lti_langid_corpus"

_DESCRIPTION = """\
The LTI LangID corpus is a dataset for language identification.
The most recent version, v5, contains training data for 1266 languages, and some (possibly very tiny) amount of text for a total of 1706 languages.
This dataloader can only be executed in a BASH environment at the moment. (See https://github.com/SEACrowd/seacrowd-datahub/pull/405)
"""

_HOMEPAGE = "https://www.cs.cmu.edu/~ralf/langid.html"

_LANGUAGES = [
    "ifa",
    "ace",
    "btm",
    "mqj",
    "mbs",
    "nbq",
    "gor",
    "zyp",
    "tbl",
    "kjp",
    "kmk",
    "kqe",
    "ptu",
    "blw",
    "ceb",
    "prf",
    "yva",
    "zlm",
    "bps",
    "tdt",
    "mya",
    "dgc",
    "lus",
    "wrs",
    "abx",
    "rgu",
    "aaz",
    "agn",
    "ccp",
    "cmn",
    "jav",
    "obo",
    "due",
    "msk",
    "xsb",
    "syb",
    "ind",
    "lbk",
    "min",
    "smk",
    "att",
    "nod",
    "tdj",
    "atb",
    "atd",
    "pag",
    "hvn",
    "ksc",
    "lao",
    "kkl",
    "lti",
    "dao",
    "cgc",
    "tbk",
    "gdg",
    "amk",
    "mbd",
    "clu",
    "msb",
    "sbl",
    "cek",
    "khm",
    "yue",
    "sgb",
    "beu",
    "eip",
    "ifu",
    "mnw",
    "suc",
    "bgs",
    "pam",
    "ebk",
    "eng",
    "nfa",
    "cbk",
    "ify",
    "csy",
    "heg",
    "shn",
    "mta",
    "mbt",
    "lex",
    "tha",
    "mmn",
    "sun",
    "vie",
    "llg",
    "xnn",
    "txq",
    "bcl",
    "kje",
    "kne",
    "san",
    "hlt",
    "kyu",
    "bkd",
    "duo",
    "tet",
    "ury",
    "yka",
    "bjn",
    "tiy",
    "ivv",
    "agt",
    "ban",
    "blz",
    "mbi",
    "ilo",
    "mkn",
    "isd",
    "cth",
    "bpr",
    "por",
    "mbb",
    "tgl",
    "msm",
    "ivb",
    "tam",
    "plw",
    "alp",
    "row",
]

_ALL_LANGS = [
    "aai",
    "aak",
    "aau",
    "aaz",
    "abk",
    "abt",
    "abx",
    "aby",
    "aca",
    "acc",
    "ace",
    "acf",
    "acr",
    "acu",
    "adj",
    "ady",
    "adz",
    "aer",
    "aey",
    "afr",
    "agd",
    "agg",
    "agm",
    "agn",
    "agr",
    "agt",
    "agu",
    "ahr",
    "aia",
    "aii",
    "ake",
    "akh",
    "alb",
    "alp",
    "als",
    "aly",
    "ame",
    "amf",
    "amh",
    "ami",
    "amk",
    "amm",
    "amn",
    "amo",
    "amp",
    "amr",
    "amu",
    "amx",
    "ang",
    "anh",
    "anp",
    "anv",
    "aoi",
    "aoj",
    "aom",
    "aon",
    "apb",
    "ape",
    "apn",
    "apr",
    "apu",
    "apw",
    "apy",
    "apz",
    "ara",
    "arb",
    "arc",
    "are",
    "arg",
    "arl",
    "arn",
    "arp",
    "arq",
    "ary",
    "arz",
    "ase",
    "asm",
    "aso",
    "ast",
    "ata",
    "atb",
    "atd",
    "atg",
    "atj",
    "att",
    "aty",
    "auc",
    "aui",
    "auy",
    "ava",
    "avk",
    "avt",
    "awa",
    "awb",
    "awk",
    "awx",
    "aym",
    "ayp",
    "azb",
    "azd",
    "aze",
    "azg",
    "azz",
    "bak",
    "ban",
    "bao",
    "bar",
    "bba",
    "bbb",
    "bbr",
    "bch",
    "bcl",
    "bco",
    "bdd",
    "bdv",
    "bea",
    "bef",
    "bel",
    "ben",
    "beo",
    "ber",
    "beu",
    "bfb",
    "bfz",
    "bgc",
    "bgn",
    "bgs",
    "bgt",
    "bgw",
    "bhd",
    "bhg",
    "bhl",
    "bho",
    "bht",
    "big",
    "bjk",
    "bjn",
    "bjp",
    "bjr",
    "bjv",
    "bjz",
    "bkd",
    "bki",
    "bkq",
    "bla",
    "blw",
    "blz",
    "bmh",
    "bmr",
    "bmu",
    "bnp",
    "boa",
    "bod",
    "boj",
    "bon",
    "bos",
    "box",
    "bpr",
    "bps",
    "bpy",
    "bqc",
    "bqi",
    "bqj",
    "bqp",
    "bre",
    "bsj",
    "bsn",
    "bsp",
    "bss",
    "btm",
    "buk",
    "bul",
    "bus",
    "bvd",
    "bvr",
    "bwd",
    "bwo",
    "bxh",
    "bxr",
    "byr",
    "byx",
    "bzd",
    "bzh",
    "bzj",
    "caa",
    "cab",
    "cac",
    "caf",
    "cak",
    "cao",
    "cap",
    "car",
    "cat",
    "cav",
    "cax",
    "cbc",
    "cbi",
    "cbk",
    "cbm",
    "cbr",
    "cbs",
    "cbt",
    "cbu",
    "cbv",
    "cco",
    "ccp",
    "cdh",
    "cdj",
    "cdo",
    "ceb",
    "ceg",
    "cek",
    "ces",
    "cgc",
    "cha",
    "chd",
    "che",
    "chf",
    "chk",
    "chq",
    "chr",
    "chu",
    "chv",
    "chz",
    "cjo",
    "cjv",
    "ckb",
    "cke",
    "cki",
    "ckw",
    "cle",
    "clu",
    "cly",
    "cme",
    "cmn",
    "cni",
    "cnl",
    "cnr",
    "cnt",
    "coe",
    "cof",
    "cok",
    "con",
    "cop",
    "cor",
    "cos",
    "cot",
    "cox",
    "cpa",
    "cpb",
    "cpc",
    "cpu",
    "cpy",
    "crh",
    "crm",
    "crn",
    "cro",
    "crx",
    "csb",
    "cso",
    "csy",
    "cta",
    "cth",
    "cti",
    "cto",
    "ctp",
    "ctu",
    "cub",
    "cuc",
    "cui",
    "cuk",
    "cul",
    "cut",
    "cux",
    "cwe",
    "cya",
    "cym",
    "daa",
    "dad",
    "dag",
    "dah",
    "dan",
    "dao",
    "ded",
    "des",
    "deu",
    "dgc",
    "dgo",
    "dgz",
    "dhi",
    "dhn",
    "dif",
    "dik",
    "din",
    "diq",
    "div",
    "djk",
    "djr",
    "dnj",
    "dob",
    "dop",
    "dov",
    "dsb",
    "dso",
    "dty",
    "dub",
    "due",
    "duo",
    "dwr",
    "dww",
    "dyi",
    "dzo",
    "ebk",
    "egl",
    "eip",
    "eka",
    "ekk",
    "eko",
    "ell",
    "emi",
    "emp",
    "eng",
    "enm",
    "enq",
    "epo",
    "eri",
    "ese",
    "esi",
    "esk",
    "ess",
    "est",
    "etr",
    "eus",
    "ewe",
    "ext",
    "eza",
    "faa",
    "fai",
    "fan",
    "fao",
    "far",
    "fas",
    "ffm",
    "fin",
    "fit",
    "for",
    "fra",
    "frp",
    "frr",
    "fry",
    "fue",
    "fuf",
    "fuh",
    "fur",
    "fuv",
    "gag",
    "gah",
    "gai",
    "gam",
    "gan",
    "gaq",
    "gas",
    "gaw",
    "gaz",
    "gbk",
    "gbl",
    "gbm",
    "gcr",
    "gdg",
    "gdn",
    "gdr",
    "geb",
    "gfk",
    "ghs",
    "gla",
    "gle",
    "glg",
    "glk",
    "glv",
    "gmv",
    "gng",
    "gnn",
    "gnw",
    "gof",
    "gok",
    "gom",
    "gor",
    "got",
    "gqr",
    "grc",
    "grn",
    "gsw",
    "gub",
    "guc",
    "guh",
    "gui",
    "guj",
    "gul",
    "gum",
    "gun",
    "guo",
    "gup",
    "gux",
    "gvc",
    "gvf",
    "gvn",
    "gvs",
    "gwi",
    "gym",
    "gyr",
    "hak",
    "hat",
    "hau",
    "haw",
    "hbo",
    "hch",
    "heb",
    "heg",
    "hif",
    "hin",
    "hix",
    "hla",
    "hlb",
    "hlt",
    "hmo",
    "hns",
    "hop",
    "hot",
    "hoy",
    "hrv",
    "hsb",
    "hto",
    "hub",
    "hui",
    "hun",
    "hus",
    "huu",
    "huv",
    "hva",
    "hvn",
    "hwc",
    "hye",
    "hyw",
    "ian",
    "ibo",
    "icr",
    "ido",
    "ifa",
    "ifu",
    "ify",
    "ign",
    "iii",
    "ikk",
    "ikw",
    "ile",
    "ilo",
    "imo",
    "ina",
    "inb",
    "ind",
    "inh",
    "ino",
    "iou",
    "ipi",
    "iqw",
    "iri",
    "isd",
    "isl",
    "isn",
    "ita",
    "ivb",
    "ivv",
    "iws",
    "ixl",
    "izz",
    "jaa",
    "jac",
    "jae",
    "jam",
    "jao",
    "jav",
    "jbo",
    "jic",
    "jid",
    "jiv",
    "jni",
    "jpn",
    "juy",
    "jvn",
    "kaa",
    "kab",
    "kaj",
    "kal",
    "kan",
    "kaq",
    "kat",
    "kaz",
    "kbc",
    "kbd",
    "kbh",
    "kbm",
    "kbp",
    "kbq",
    "kca",
    "kdc",
    "kde",
    "kdl",
    "kek",
    "ken",
    "kew",
    "kfs",
    "kfw",
    "kfx",
    "kfy",
    "kgf",
    "kgk",
    "kgp",
    "khm",
    "khn",
    "khs",
    "khw",
    "khz",
    "kij",
    "kik",
    "kin",
    "kip",
    "kiu",
    "kiz",
    "kjb",
    "kje",
    "kjo",
    "kjp",
    "kjs",
    "kkc",
    "kkl",
    "klt",
    "klv",
    "kmg",
    "kmh",
    "kmk",
    "kmo",
    "kms",
    "kmu",
    "kne",
    "knf",
    "knj",
    "knv",
    "kog",
    "koi",
    "kom",
    "kor",
    "kos",
    "kpf",
    "kpg",
    "kpj",
    "kpr",
    "kpw",
    "kpx",
    "kqa",
    "kqc",
    "kqe",
    "kqf",
    "kql",
    "kqw",
    "krc",
    "ksc",
    "ksd",
    "ksh",
    "ksj",
    "ksr",
    "ktj",
    "ktm",
    "kto",
    "kud",
    "kue",
    "kum",
    "kup",
    "kur",
    "kvg",
    "kvn",
    "kwd",
    "kwf",
    "kwi",
    "kwj",
    "kxp",
    "kxv",
    "kyc",
    "kyf",
    "kyg",
    "kyq",
    "kyu",
    "kyv",
    "kyz",
    "kze",
    "lac",
    "lad",
    "lao",
    "lat",
    "lbb",
    "lbf",
    "lbk",
    "lbm",
    "lcm",
    "leu",
    "lex",
    "lez",
    "lfn",
    "lgl",
    "lid",
    "lif",
    "lij",
    "lim",
    "lin",
    "lit",
    "liv",
    "lki",
    "lld",
    "llg",
    "lmo",
    "lrc",
    "ltg",
    "lti",
    "ltz",
    "lug",
    "luo",
    "lus",
    "luz",
    "lvs",
    "lww",
    "lzh",
    "lzz",
    "maa",
    "mah",
    "mai",
    "maj",
    "mal",
    "mam",
    "maq",
    "mar",
    "mau",
    "mav",
    "maz",
    "mbb",
    "mbc",
    "mbd",
    "mbh",
    "mbi",
    "mbj",
    "mbl",
    "mbs",
    "mbt",
    "mca",
    "mcb",
    "mcd",
    "mcf",
    "mco",
    "mcp",
    "mcq",
    "mcr",
    "mdf",
    "mdy",
    "med",
    "mee",
    "meh",
    "mek",
    "meq",
    "met",
    "meu",
    "mfy",
    "mgc",
    "mgh",
    "mhr",
    "mib",
    "mie",
    "mig",
    "mih",
    "mil",
    "mim",
    "min",
    "mio",
    "mir",
    "mit",
    "mix",
    "miy",
    "miz",
    "mjc",
    "mjl",
    "mkd",
    "mkj",
    "mkl",
    "mkn",
    "mks",
    "mle",
    "mlg",
    "mlh",
    "mlp",
    "mlt",
    "mmn",
    "mmo",
    "mmx",
    "mna",
    "mnc",
    "mni",
    "mnw",
    "moh",
    "mon",
    "mop",
    "mox",
    "mpd",
    "mpj",
    "mpm",
    "mpp",
    "mps",
    "mpt",
    "mpx",
    "mqb",
    "mqj",
    "mri",
    "mrj",
    "msb",
    "msc",
    "msk",
    "msm",
    "msy",
    "mta",
    "mti",
    "mto",
    "mux",
    "muy",
    "mva",
    "mvn",
    "mwc",
    "mwe",
    "mwf",
    "mwl",
    "mwp",
    "mxb",
    "mxp",
    "mxq",
    "mxt",
    "mxv",
    "mya",
    "myk",
    "myu",
    "myv",
    "myw",
    "myy",
    "mza",
    "mzi",
    "mzn",
    "nab",
    "naf",
    "nag",
    "nah",
    "nak",
    "nan",
    "nap",
    "nas",
    "nav",
    "nbq",
    "nca",
    "nce",
    "nch",
    "ncj",
    "ncl",
    "ncu",
    "nde",
    "ndg",
    "ndj",
    "nds",
    "new",
    "nfa",
    "nfr",
    "ngp",
    "ngu",
    "nhe",
    "nhg",
    "nhi",
    "nho",
    "nhr",
    "nhu",
    "nhw",
    "nhx",
    "nhy",
    "nif",
    "nii",
    "nin",
    "niy",
    "nko",
    "nld",
    "nlg",
    "nlx",
    "nmw",
    "nna",
    "nno",
    "nnq",
    "noa",
    "nob",
    "nod",
    "noi",
    "nop",
    "not",
    "nou",
    "nov",
    "npi",
    "npl",
    "nqo",
    "nsn",
    "nso",
    "nss",
    "nsu",
    "ntj",
    "ntp",
    "ntu",
    "nuy",
    "nvm",
    "nwi",
    "nya",
    "nys",
    "nyu",
    "obo",
    "oci",
    "okv",
    "olo",
    "omw",
    "ong",
    "ons",
    "ood",
    "opm",
    "ory",
    "oss",
    "ota",
    "ote",
    "otm",
    "otn",
    "otq",
    "ots",
    "oym",
    "pab",
    "pad",
    "pag",
    "pah",
    "pam",
    "pan",
    "pao",
    "pap",
    "pbb",
    "pbc",
    "pcd",
    "pdc",
    "peg",
    "pes",
    "pfl",
    "pib",
    "pio",
    "pir",
    "piu",
    "pjt",
    "pli",
    "pls",
    "plt",
    "plu",
    "plw",
    "pma",
    "pms",
    "pnb",
    "pnt",
    "pob",
    "poe",
    "poh",
    "poi",
    "pol",
    "pon",
    "por",
    "poy",
    "ppo",
    "pps",
    "prf",
    "pri",
    "prq",
    "pst",
    "ptp",
    "ptu",
    "pwg",
    "qub",
    "quc",
    "quf",
    "quh",
    "qul",
    "qup",
    "quy",
    "qvc",
    "qve",
    "qvh",
    "qvm",
    "qvn",
    "qvs",
    "qvw",
    "qvz",
    "qwh",
    "qxh",
    "qxl",
    "qxn",
    "qxo",
    "rai",
    "ram",
    "rap",
    "reg",
    "rgu",
    "rkb",
    "rmc",
    "rmn",
    "rmy",
    "roh",
    "ron",
    "roo",
    "rop",
    "row",
    "rro",
    "rtw",
    "rue",
    "ruf",
    "rug",
    "rup",
    "rus",
    "rwo",
    "sab",
    "sah",
    "san",
    "sat",
    "sbe",
    "sbk",
    "sbl",
    "sbs",
    "sby",
    "sch",
    "sck",
    "scn",
    "sco",
    "sdh",
    "seh",
    "sey",
    "sgb",
    "sgj",
    "sgs",
    "sgz",
    "shi",
    "shj",
    "shn",
    "shp",
    "sim",
    "sin",
    "sja",
    "skr",
    "sli",
    "slk",
    "sll",
    "slv",
    "sme",
    "smk",
    "smo",
    "sna",
    "snc",
    "snd",
    "snn",
    "snp",
    "sny",
    "som",
    "soq",
    "soy",
    "spa",
    "spl",
    "spm",
    "spp",
    "sps",
    "spy",
    "srd",
    "sri",
    "srm",
    "srn",
    "srp",
    "srq",
    "srx",
    "ssd",
    "ssg",
    "ssx",
    "stp",
    "stq",
    "sua",
    "suc",
    "sue",
    "sun",
    "sus",
    "suz",
    "swe",
    "swh",
    "swp",
    "sxb",
    "syb",
    "szl",
    "szy",
    "tac",
    "taj",
    "tam",
    "tar",
    "tat",
    "tau",
    "tav",
    "taw",
    "tay",
    "tbc",
    "tbf",
    "tbg",
    "tbk",
    "tbl",
    "tbo",
    "tbz",
    "tca",
    "tcf",
    "tcs",
    "tcy",
    "tcz",
    "tdj",
    "tdt",
    "tdx",
    "ted",
    "tee",
    "tel",
    "ter",
    "tet",
    "tew",
    "tfr",
    "tgk",
    "tgl",
    "tgo",
    "tgp",
    "tha",
    "thr",
    "tif",
    "tik",
    "tim",
    "tir",
    "tiw",
    "tiy",
    "tke",
    "tkr",
    "tku",
    "tlf",
    "tlh",
    "tly",
    "tmd",
    "tna",
    "tnc",
    "tnk",
    "tnn",
    "tnp",
    "toc",
    "tod",
    "tof",
    "toj",
    "tok",
    "ton",
    "too",
    "top",
    "tos",
    "tpi",
    "tpt",
    "tpx",
    "tpz",
    "trc",
    "trq",
    "trs",
    "trv",
    "tsn",
    "tso",
    "tsw",
    "ttc",
    "tte",
    "tuc",
    "tue",
    "tuf",
    "tuk",
    "tuo",
    "tur",
    "tvk",
    "tvt",
    "twi",
    "txq",
    "txu",
    "tyv",
    "tzc",
    "tze",
    "tzj",
    "tzm",
    "tzo",
    "tzs",
    "tzz",
    "ubr",
    "ubu",
    "udm",
    "udu",
    "uig",
    "ukr",
    "uli",
    "ulk",
    "unx",
    "upv",
    "ura",
    "urb",
    "urd",
    "uri",
    "urt",
    "urw",
    "ury",
    "usa",
    "usp",
    "uvh",
    "uvl",
    "uzb",
    "vah",
    "vai",
    "var",
    "vec",
    "vep",
    "vid",
    "vie",
    "viv",
    "vls",
    "vmy",
    "vol",
    "vro",
    "waj",
    "wal",
    "wap",
    "wat",
    "waw",
    "wbi",
    "wbp",
    "wed",
    "wer",
    "wim",
    "wiu",
    "wiv",
    "wln",
    "wmt",
    "wmw",
    "wnc",
    "wnu",
    "wol",
    "wos",
    "wrk",
    "wrs",
    "wsk",
    "wuu",
    "wuv",
    "xal",
    "xav",
    "xbi",
    "xed",
    "xer",
    "xho",
    "xir",
    "xla",
    "xmf",
    "xnj",
    "xnn",
    "xno",
    "xnr",
    "xon",
    "xri",
    "xsb",
    "xsi",
    "xta",
    "xtd",
    "xtm",
    "xtn",
    "xuo",
    "yaa",
    "yad",
    "yal",
    "yam",
    "yao",
    "yap",
    "yaq",
    "yby",
    "ycn",
    "yid",
    "yka",
    "yle",
    "yml",
    "yom",
    "yon",
    "yor",
    "yrb",
    "yre",
    "yss",
    "yua",
    "yue",
    "yuj",
    "yut",
    "yuw",
    "yva",
    "zaa",
    "zab",
    "zac",
    "zad",
    "zae",
    "zai",
    "zaj",
    "zam",
    "zao",
    "zaq",
    "zar",
    "zas",
    "zat",
    "zav",
    "zaw",
    "zca",
    "zea",
    "zga",
    "zgh",
    "zia",
    "ziw",
    "zlm",
    "zos",
    "zpc",
    "zpg",
    "zpi",
    "zpl",
    "zpm",
    "zpo",
    "zpq",
    "zpt",
    "zpu",
    "zpv",
    "zpz",
    "zsr",
    "ztg",
    "ztq",
    "zty",
    "zul",
    "zyp",
]

_LICENSE = Licenses.UNKNOWN.value

_LOCAL = False

_URL = "http://sourceforge.net/projects/la-strings/files/Language-Data/LTI-LangID-rel5.txz/download"

_SUPPORTED_TASKS = [Tasks.LANGUAGE_IDENTIFICATION]

_SOURCE_VERSION = "5.0.0"
_SEACROWD_VERSION = "1.0.0"


# TODO: Name the dataset class to match the script name using CamelCase instead of snake_case
class LTILangIDDataset(datasets.GeneratorBasedBuilder):
    """LTI LangID corpus is a dataset for language identification for 1266 languages."""

    BUILDER_CONFIGS = [
        SEACrowdConfig(
            name=f"{_DATASETNAME}_source",
            version=datasets.Version(_SOURCE_VERSION),
            description=f"{_DATASETNAME} source schema",
            schema="source",
            subset_id=f"{_DATASETNAME}",
        ),
        SEACrowdConfig(
            name=f"{_DATASETNAME}_seacrowd_text",
            version=datasets.Version(_SEACROWD_VERSION),
            description=f"{_DATASETNAME} SEACrowd schema",
            schema="seacrowd_text",
            subset_id=f"{_DATASETNAME}",
        ),
    ]

    DEFAULT_CONFIG_NAME = f"{_DATASETNAME}_source"

    def _info(self) -> datasets.DatasetInfo:
        if self.config.schema == "source":
            features = datasets.Features({"text": datasets.Value("string"), "language": datasets.Value("string")})

        elif self.config.schema == "seacrowd_text":
            features = schemas.text_features(label_names=_ALL_LANGS)

        return datasets.DatasetInfo(
            description=_DESCRIPTION,
            features=features,
            homepage=_HOMEPAGE,
            license=_LICENSE,
            citation=_CITATION,
        )

    def _split_generators(self, dl_manager: datasets.DownloadManager) -> List[datasets.SplitGenerator]:
        """Returns SplitGenerators."""
        data_dir = dl_manager.download_and_extract(_URL)
        data_dir = os.path.join(data_dir, "MIL-TALE", "5")

        datasplits_dir = os.path.join(data_dir, "datasplits")
        if not Path(datasplits_dir).exists() or not Path(os.path.join(datasplits_dir, "DONE")).exists():
            # Run provided install.sh to generate train/dev/test splits in "datasplits" folder
            subprocess.call([os.path.join(data_dir, "code", "install.sh"), datasplits_dir])
            with open(os.path.join(datasplits_dir, "DONE"), "w"):
                pass

        train_filepaths = []
        dev_filepaths = []
        test_filepaths = []

        dataset_dir = os.path.join(data_dir, "datasplits")
        print(dataset_dir)

        for lang_id in _ALL_LANGS:
            train_filepaths.append(
                (
                    lang_id,
                    glob(os.path.join(dataset_dir, "train", f"{lang_id}*")),
                )
            )
            dev_filepaths.append(
                (
                    lang_id,
                    glob(os.path.join(dataset_dir, "devtest", f"{lang_id}*")),
                )
            )
            test_filepaths.append(
                (
                    lang_id,
                    glob(os.path.join(dataset_dir, "test", f"{lang_id}*")),
                )
            )

        return [
            datasets.SplitGenerator(
                name=datasets.Split.TRAIN,
                # Whatever you put in gen_kwargs will be passed to _generate_examples
                gen_kwargs={
                    "all_filepaths": train_filepaths,
                },
            ),
            datasets.SplitGenerator(
                name=datasets.Split.VALIDATION,
                # Whatever you put in gen_kwargs will be passed to _generate_examples
                gen_kwargs={
                    "all_filepaths": dev_filepaths,
                },
            ),
            datasets.SplitGenerator(
                name=datasets.Split.TEST,
                # Whatever you put in gen_kwargs will be passed to _generate_examples
                gen_kwargs={
                    "all_filepaths": test_filepaths,
                },
            ),
        ]

    def _generate_examples(self, all_filepaths: List[Path]) -> Tuple[int, Dict]:

        if self.config.schema == "source":
            key = 0
            for lang_id, filepaths in all_filepaths:
                for filepath in filepaths:
                    try:
                        for line in open(filepath):
                            text = line.strip().replace("\u200d", " ").replace("\u200b", " ")
                            yield key, {"text": text, "language": lang_id}
                            key += 1
                    except UnicodeDecodeError:
                        continue

        elif self.config.schema == "seacrowd_text":
            key = 0
            for lang_id, filepaths in all_filepaths:
                for filepath in filepaths:
                    try:
                        for line in open(filepath):
                            text = line.strip().replace("\u200d", " ").replace("\u200b", " ")
                            yield key, {"id": f"{filepath.split('/')[-1]}_{key}", "text": text, "label": lang_id}
                            key += 1
                    except UnicodeDecodeError:
                        continue
