// Part of Curveship-js version 0.4
//  Copyright 2020-2021 Nick Montfort
//
// licensed under the GNU General Public License v3.0. See the file LICENSE
// for complete terms.
//
// Noun class, along with English-language irregular plurals. These are for
// all but -s and -es plurals.

var names = {};

class Noun {
  constructor(base) {
    this.base = base;
  }
  singular() {
    return this.base;
  }
  plural() {
    if (this.hasOwnProperty("plural")) {
      return this.plural;
    }
    let plural = "";
    if (this.base in irregularNouns) {
      plural = irregularNouns[this.base][0];
    } else {
      let ending = /((ch|sh|s|z|x)|[bcdfghjklmnpqrstvwxz]y|[^o]o)$/;
      if (ending.test(this.base)) {
        plural = this.base + "es";
      } else {
        plural = this.base + "s";
      }
    }
    this.plural = plural;
    return plural;
  }
}

// ### PRONOUNS ###

class PronounSet {
  constructor(thirdPersonSingular) {
    this.pronoun = [];
    this.pronoun.push([[], [], []]); // There is no 0th person or number
    this.pronoun.push([
      [],
      ["I", "me", "mine", "my", "myself"],
      ["we", "us", "ours", "our", "ourselves"],
    ]);
    this.pronoun.push([
      [],
      ["you", "you", "yours", "your", "yourself"],
      ["you", "you", "yours", "your", "yourselves"],
    ]);
    this.pronoun.push([
      [],
      thirdPersonSingular,
      ["they", "them", "theirs", "their", "themselves"],
    ]);
  }
  getSubject(person, number = 1) {
    return this.pronoun[person][number][0];
  }
  getObject(person, number = 1) {
    return this.pronoun[person][number][1];
  }
  getPossessiveAdj(person, number = 1) {
    return this.pronoun[person][number][2];
  }
  getPossessivePronoun(person, number = 1) {
    return this.pronoun[person][number][3];
  }
  getReflexive(person, number = 1) {
    return this.pronoun[person][number][4];
  }
}

var pronoun = {};
pronoun.feminine = new PronounSet(["she", "her", "hers", "her", "herself"]);
pronoun.masculine = new PronounSet(["he", "him", "his", "his", "himself"]);
pronoun.neuter = new PronounSet(["it", "it", "its", "its", "itself"]);
pronoun.unknownBinary = new PronounSet([
  "he or she",
  "him or her",
  "his or hers",
  "his or her",
  "himself or herself",
]);
pronoun.nonBinary = new PronounSet([
  "they",
  "them",
  "theirs",
  "their",
  "themself",
]); // If you prefer, you can make the last entry "themselves"

const articles = new Set(["a", "an", "one", "some", "that", "this", "the"]);

function bareName(np) {
  return articles.has(np.split(" ")[0]) ? np = np.substring(np.indexOf(" ") + 1) : np;
}

class Names {
  constructor(
    initial,
    subsequent = null,
    pronouns = null,
    multiple_plurality = true
  ) {
    this.initial = initial;
    this.subsequent = subsequent ? subsequent : "the " + bareName(initial);
    this.pronouns = pronouns;
    if (typeof this.possessive === "undefined") {
      this.possessive = null;
    }
    this.nameByCategory = false;
  }
  setGenericPronouns(tag) {
    let pronounSet = pronoun.neuter;
    if (tag in actor && actor[tag].gender === "male") {
      pronounSet = pronoun.masculine;
    } else if (tag in actor && actor[tag].gender === "female") {
      pronounSet = pronoun.feminine;
    }
    this.pronouns = pronounSet;
  }
}

class ProperNames extends Names {
  constructor(
    given,
    family = null,
    pronouns = null,
    common = null,
    title = null,
    possessive = null
  ) {
    let initial, subsequent;
    if (given === null) {
      initial = (title !== null ? title + " " : "") + family;
      subsequent = initial;
    } else if (family !== null && family !== "") {
      initial = (title !== null ? title + " " : "") + given + " " + family;
      subsequent = title !== null ? title + " " + family : given;
    } else {
      initial = given;
      subsequent = given;
    }
    super(initial, subsequent, pronouns);
    this.title = title;
    this.given = given;
    this.family = family;
    this.common = common;
    this.possessive = possessive;
  }
}

class BrandNames extends Names {
  constructor(
    generic, // if left out by default, it is wrapped in parens by the author
             // "(a soda)" means it is referred to *only* by brand name,
             // e.g. “a Sprite” whereas "a cola" means the generic term is
             // included, e.g. “a Royal Crown cola”
    make, // if an article is ever used it must be included: "a Honda"
    model, // if an article is ever used it must be included: "an Accord"
    source = null, // where it was purchased from, with connecting language, e.g.
                   // "purchased from K-Mart" "won at a state fair"
    attributes = null,
    pronouns = pronoun.neuter,
    plural = false
  ) {
    let initial;
    if (make) {
      initial = model ? make + " " + bareName(model) : make;
    } else {
      initial = model ? model : "";
    }
    initial += (generic[0] === "(") ? "" : " " + bareName(generic);
    let subsequent = initial;
    initial += source ? " " + source : "";
    initial = attributes ? attributes + " " + bareName(initial) : initial;
    super(initial, subsequent, pronouns, plural);
    this.make = make;
    this.model = model;
    this.source = source;
    this.attributes = attributes;
  }
}

class CategoryNames extends Names {
  constructor(name, proper = null) {
    // FIXME shuold be the plural of the name???
    let initial = "some " + (proper !== null ? proper + " " : "") + name + "s";
    let subsequent =
      "the " + (proper !== null ? proper + " " : "") + name + "s";
    super(initial, subsequent, pronoun.neuter);
    this.name = (proper !== null ? proper + " " : "") + name;
    this.proper = proper;
  }
}

class GenericNames extends Names {
  constructor(tag) {
    let name;
    if (tag in actor && actor[tag].gender === "male") {
      if (actor[tag].age == "adult") {
        name = "man";
      } else {
        name = "boy";
      }
    } else if (tag in actor && actor[tag].gender === "female") {
      if (actor[tag].age == "adult") {
        name = "woman";
      } else {
        name = "girl";
      }
    } else if (tag in thing) {
      name = "thing";
    } else if (tag in place) {
      name = "location";
    }
    super("a " + name, "the " + name);
    this.nameByCategory = true;
  }
}

// NOTE that this irregular noun list contains offensive works.
// Curveship-js does not use offensive nouns (or any other words)
// for you. But because this list is built from a dictionary that
// contains offensive words, they appear in the list, and the
// system will pluralize those words according to to this list.
//
// Rather than tagging the offensive words (and calling attention
// to them) or removing them (which would not prevent their use
// in variable narratives), they have been left in this list without
// further comment.

const irregularNouns = {
  abaculus: ["abaculi"],
  abacus: ["abacuses", "abaci"],
  abattoir: ["abattoirs"],
  abbacy: ["abbacies"],
  abbey: ["abbeys"],
  abdominal: ["abdominals", "abdominales"],
  ability: ["abilities"],
  abnormality: ["abnormalities"],
  abnormity: ["abnormities"],
  abscess: ["abscesses"],
  absciss: ["abscisses"],
  abscissa: ["abscissas", "abscissae"],
  absurdity: ["absurdities"],
  acacia: ["acacias", "acaciae"],
  academy: ["academies"],
  acaleph: ["acalephs", "acalephans"],
  acanthus: ["acanthuses", "acanthi"],
  acarus: ["acari"],
  accessary: ["accessaries"],
  accessory: ["accessories"],
  accipiter: ["accipiters", "accipitres"],
  acclivity: ["acclivities"],
  ace: ["aces"],
  acicula: ["aciculae"],
  acinus: ["acini"],
  acrimony: ["acrimonies"],
  acroterium: ["acrotplwia"],
  actinia: ["actiniae", "actinias"],
  activity: ["activities"],
  actuality: ["actualities"],
  actuary: ["actuaries"],
  aculeus: ["aculei"],
  addendum: ["addenda"],
  adieu: ["adieus"],
  admiralty: ["admiralties"],
  adultery: ["adulteries"],
  adversary: ["adversaries"],
  adversity: ["adversities"],
  adytum: ["adyta"],
  aecidium: ["aecidia"],
  affinity: ["affinities"],
  affix: ["affixes"],
  agami: ["agamis"],
  agape: ["agapae"],
  agency: ["agencies"],
  agendum: ["agenda"],
  agio: ["agios"],
  agnus: ["agnuses", "agni"],
  agon: ["agones"],
  agony: ["agonies"],
  ai: ["ais"],
  "aid-de-camp": ["aids-de-camp"],
  ala: ["alae"],
  alabastrum: ["alabastra"],
  albino: ["albinos"],
  albugo: ["albugines"],
  alcarraza: ["alcarrazas"],
  alderman: ["aldplwmen"],
  alewife: ["alewives"],
  alga: ["algae"],
  alias: ["aliases"],
  alkali: ["alkalis", "alkalies"],
  allegory: ["allegories"],
  alley: ["alleys"],
  alluvium: ["alluviums", "alluvia"],
  ally: ["allies"],
  almonry: ["almonries"],
  aloe: ["aloes"],
  alto: ["altos"],
  "alto-rilievo": ["alto-rilievos"],
  alumna: ["alumnae"],
  alumnus: ["alumni"],
  alveary: ["alvearies"],
  alveolus: ["alveoli"],
  alveus: ["alvei"],
  amanuensis: ["amanuenses"],
  ambiguity: ["ambiguities"],
  ambo: ["ambos"],
  ambry: ["ambries"],
  ambulacrum: ["ambulacra"],
  ambulatory: ["ambulatories"],
  amenity: ["amenities"],
  amentum: ["amenta"],
  amity: ["amities"],
  amoeba: ["amoebae", "amoebas"],
  amorpha: ["amorphas"],
  amphibium: ["amphibia", "amphibiums"],
  amphibology: ["amphibologies"],
  amphiboly: ["amphibolies"],
  amphictyony: ["amphictyonies"],
  amphora: ["amophorae"],
  ampulla: ["ampullae"],
  analogy: ["analogies"],
  analysis: ["analyses"],
  anaptychus: ["anaptichi"],
  anastomosis: ["anastomoses"],
  anathema: ["anathemas"],
  anatifa: ["anatifae"],
  anatomy: ["anatomies"],
  ancon: ["ancones"],
  anfractuosity: ["anfractuosities"],
  animalculum: ["animalcula"],
  animosity: ["animosities"],
  animus: ["animi"],
  anniversary: ["anniversaries"],
  annuity: ["annuities"],
  annulus: ["annuli"],
  anomaly: ["anomalies"],
  ansa: ["ansae"],
  anta: ["antae"],
  antefix: ["antefixes", "antefixa"],
  antenna: ["antennae"],
  anthelion: ["anthelia"],
  antheridium: ["anthplwidia"],
  antichthon: ["antichthones"],
  anticlinorium: ["anticlinoria"],
  antilogy: ["antilogies"],
  antinomy: ["antinomies"],
  antipathy: ["antipathies"],
  antiphony: ["antiphonies"],
  antiquary: ["antiquaries"],
  antiquity: ["antiquities"],
  antithesis: ["antitheses"],
  antlia: ["antilae"],
  antrum: ["antra"],
  anxiety: ["anxieties"],
  apathy: ["apathies"],
  apery: ["aperies"],
  apex: ["apexes", "apices"],
  aphelion: ["aphelia"],
  aphis: ["aphides"],
  apocrypha: ["apocryphas"],
  apod: ["apods", "apodes"],
  apology: ["apologies"],
  aponeurosis: ["aponeuroses"],
  apophysis: ["apophyses"],
  aporia: ["aporias"],
  apostasy: ["apostasies"],
  apothecary: ["apothecaries"],
  apothecium: ["apothecia"],
  apotheosis: ["apotheoses"],
  apparatus: ["apparatus", "apparatuses"],
  appendix: ["appendixes", "appendices"],
  appetency: ["appetencies"],
  apronful: ["apronfuls"],
  apsis: ["apsides"],
  aquarium: ["aquariums", "aquaria"],
  aquila: ["aquilae"],
  arango: ["arangoes"],
  arboretum: ["arboreta"],
  arcanum: ["arcana"],
  archipelago: ["archipelagoes", "archipelagos"],
  archive: ["archives"],
  area: ["areas"],
  arena: ["arenas", "arenae"],
  areola: ["areolae"],
  argosy: ["argosies"],
  aridity: ["aridities"],
  aristocracy: ["aristocracies"],
  armadillo: ["armadillos"],
  armful: ["armfulus"],
  armilla: ["armillas", "armillae"],
  armory: ["armories"],
  arroyo: ["arroyos"],
  artery: ["arteries"],
  articulus: ["articuli"],
  aruspex: ["aruspices"],
  as: ["asses"],
  ascarid: ["ascarides"],
  ascidium: ["ascidia"],
  ascococcus: ["ascococci"],
  ashantee: ["ashantees"],
  ashtoreth: ["ashtaroth"],
  asperity: ["asperities"],
  aspersorium: ["aspplwsoria"],
  assembly: ["assemblies"],
  assemblyman: ["assemblymen"],
  assiduity: ["assiduities"],
  asylum: ["asylums", "asyla"],
  atheneum: ["atheneums", "athenaea"],
  atlas: ["atlases"],
  atrium: ["atria"],
  atrocity: ["atrocities"],
  attorney: ["attorneys"],
  augury: ["auguries"],
  aura: ["aurae"],
  auricula: ["auriculae", "auriculas"],
  aurora: ["auroras", "aurorae"],
  auspice: ["auspices"],
  austerity: ["austerities"],
  authority: ["authorities"],
  autobiography: ["autobiographies"],
  autochthon: ["authochthons", "autochthones"],
  autocracy: ["autocracies"],
  "auto-da-fé": ["autos-da-fé"],
  "auto-de-fe": ["autos-de-fe"],
  automaton: ["automata", "automatons"],
  auxiliary: ["auxiliaries"],
  availability: ["availabilities"],
  aviary: ["aviaries"],
  axilla: ["axillae"],
  axis: ["axes"],
  axman: ["axmen"],
  azalea: ["azaleas"],
  baa: ["baas"],
  baal: ["baalim"],
  baby: ["babies"],
  bacchant: ["bacchants", "bacchantes"],
  bacchante: ["bacchantes"],
  bacchius: ["bacchii"],
  bacillus: ["bacilli"],
  backwoodsman: ["backwoodsmen"],
  bacterium: ["bacteria"],
  bagman: ["bagmen"],
  balcony: ["balconies"],
  ballista: ["ballistae"],
  banality: ["banalities"],
  bandeau: ["bandeaux"],
  bandit: ["bandits", "banditti"],
  bandy: ["bandies"],
  bankruptcy: ["bankruptcies"],
  baptistery: ["baptisteries", "baptistries"],
  barbarity: ["barbarities"],
  barony: ["baronies"],
  basilica: ["basilicas", "basilicae"],
  basis: ["bases"],
  basketful: ["basketfuls"],
  bass: ["bass", "basses"],
  bastinado: ["bastinadoes"],
  bateau: ["bateaux"],
  bath: ["baths"],
  batman: ["batmen"],
  batsman: ["batsmen"],
  battery: ["batteries"],
  batz: ["batzen"],
  bayou: ["bayous"],
  beach: ["beaches"],
  beadsman: ["beadsmen"],
  beadswoman: ["beadswomen"],
  beau: ["beaux", "beaus"],
  beauty: ["beautie"],
  beccafico: ["beccaficos"],
  bedstaff: ["bedstaves"],
  beech: ["beeches"],
  "bel-esprit": ["beaux-esprits"],
  belly: ["bellies"],
  bench: ["benches"],
  beneficiary: ["beneficiaries"],
  berry: ["berries"],
  betony: ["betonies"],
  bevy: ["bevies"],
  bias: ["biases"],
  bibliography: ["bibliographies"],
  bijou: ["bijoux"],
  bilberry: ["bilberries"],
  bilbo: ["bilboes"],
  "billet-doux": ["billets-doux"],
  billman: ["billmen"],
  biography: ["biographies"],
  birch: ["birches"],
  bistoury: ["bistouries"],
  blastema: ["blastemata"],
  bleachery: ["bleacheries"],
  blemish: ["blemishes"],
  blenny: ["blennies"],
  bliss: ["blisses"],
  boa: ["boas"],
  boatful: ["boatfuls"],
  boatman: ["boatmen"],
  boatwoman: ["boatwomen"],
  body: ["bodies"],
  bogy: ["bogies"],
  bolus: ["boluses"],
  bombolo: ["bomboloes"],
  bondman: ["bondmen"],
  bondsman: ["bondsmen"],
  bondwoman: ["bondwomen"],
  bonito: ["bonitoes"],
  bonmot: ["bonsmots"],
  "bonne bouche": ["bonnes bouches"],
  bonus: ["bonuses"],
  "bon vivant": ["bons vivants"],
  booby: ["boobies"],
  bookman: ["bookmen"],
  bookshelf: ["bookshelves"],
  booly: ["boolies"],
  bosh: ["boshes"],
  boss: ["bosses"],
  botany: ["botanies"],
  botch: ["botches"],
  bothy: ["bothies"],
  boundary: ["boundaries"],
  bounty: ["bounties"],
  "bow-compass": ["bow-compasses"],
  bowery: ["boweries"],
  bowman: ["bowmen"],
  box: ["boxes"],
  boyau: ["boyaux", "boyaus"],
  brachium: ["bracchia"],
  brahman: ["brahmans", "brahmins"],
  brakeman: ["brakemen"],
  branch: ["branches"],
  branchia: ["branchiae"],
  brandy: ["brandies"],
  brass: ["brasses"],
  bravado: ["bravadoes"],
  bravo: ["bravoes"],
  brettice: ["brettices"],
  brevetcy: ["brevetcies"],
  breviary: ["breviaries"],
  brevity: ["brevities"],
  bribeery: ["briberies"],
  bridesman: ["bridesmen"],
  bronchus: ["bronchi"],
  brother: ["brothers", "brethren"],
  "brother-in-law": ["brothers-in-law"],
  brutality: ["brutalities"],
  bubo: ["buboes"],
  bucranium: ["bucrania"],
  buffalo: ["buffaloes"],
  buffoonery: ["buffooneries"],
  buggy: ["buggies"],
  bugloss: ["buglosses"],
  bulla: ["bullae"],
  bullary: ["bullaries"],
  bully: ["bullies"],
  bumbelo: ["bumbeloes"],
  buoyancy: ["buoyancies"],
  bureau: ["bureaus", "bureaux"],
  burglary: ["burglaries"],
  burman: ["burmans"],
  bursa: ["bursae"],
  bursary: ["bursaries"],
  bursch: ["burschen"],
  busby: ["busbies"],
  bushman: ["bushmen"],
  business: ["businesses"],
  busto: ["bustoes"],
  busybody: ["busybodies"],
  butterfly: ["butterflies"],
  butterman: ["buttermen"],
  buttery: ["butteries"],
  bypath: ["bypaths"],
  byssus: ["byssuses", "byssi"],
  cabman: ["cabmen"],
  cacophony: ["cacophonies"],
  cactus: ["cactuses", "cacti"],
  caddy: ["caddies"],
  caecum: ["caecums", "caeca"],
  caesura: ["caesuras", "caesurae"],
  cajolery: ["cajoleries"],
  calamity: ["calamities"],
  calamus: ["calami"],
  calcaneum: ["neums", "nea"],
  calcar: ["calcaria"],
  calculus: ["calculi"],
  calf: ["calves"],
  calico: ["calicoes"],
  callosity: ["callosities"],
  calumny: ["calumnies"],
  calx: ["calxes", "calces"],
  calyx: ["calyxes", "calyces"],
  cameo: ["cameos"],
  camera: ["cameras", "camerae"],
  canaliculus: ["canaliculi"],
  canary: ["canaries"],
  candelabrum: ["candelabra", "candelabrums"],
  canis: ["canes"],
  cannon: ["cannons", "cannon"],
  canoe: ["canoes"],
  canoeman: ["canoemen"],
  canopy: ["canopies"],
  cantharis: ["cantharides"],
  canthus: ["canthi"],
  canticle: ["canticles"],
  canto: ["cantos"],
  capability: ["capabilities"],
  capacity: ["capacities"],
  capitulary: ["capitularies"],
  capoch: ["capoches"],
  caput: ["capita"],
  caravansary: ["caravansaries"],
  carbonado: ["carbonadoes"],
  carbonaro: ["carbonari"],
  carcass: ["carcasses"],
  cardo: ["cardies"],
  cargo: ["cargoes"],
  carib: ["caries"],
  carolus: ["caroluses", "caroli"],
  carp: ["carp", "carps"],
  carpale: ["carpalia"],
  carpus: ["carpi"],
  carry: ["carries"],
  "carte de visite": ["cartes de visite"],
  cartouch: ["cartouches"],
  cartulary: ["cartularies"],
  caryatid: ["caryatids"],
  caryopsis: ["caryopses"],
  casino: ["casinos", "casini"],
  cassowary: ["cassowaries"],
  castellany: ["castellanies"],
  casualty: ["casualties"],
  catalysis: ["catalyse"],
  category: ["categories"],
  catena: ["catene"],
  cathetus: ["catheti"],
  catso: ["catsos"],
  caudex: ["caudices", "caudexes"],
  cauliculus: ["cauliculi"],
  caulis: ["caules"],
  cautery: ["cauteries"],
  cavity: ["cavities"],
  cavy: ["cavies"],
  celebrity: ["celebriries"],
  cello: ["cellos", "celli"],
  cemetery: ["cemeteries"],
  centenary: ["centenaries"],
  centesimo: ["centesimi"],
  cento: ["centos"],
  centrality: ["centralities"],
  centrum: ["centrums", "centra"],
  centumvir: ["centumviri"],
  century: ["centuries"],
  cercaria: ["cercarle"],
  cercus: ["cerci"],
  cerebellum: ["cerebellums", "cerebella"],
  cerebrum: ["cerebrums", "cerebra"],
  ceremony: ["ceremonies"],
  certainty: ["certainties"],
  cervix: ["cervixes", "cervices"],
  chairman: ["chairmen"],
  chalaza: ["chalazas", "chalazae"],
  chalcedony: ["chalcedonies"],
  chansonnette: ["chansonnettes"],
  chantry: ["chantries"],
  chapeau: ["chapeux"],
  chapellany: ["chapellanies"],
  chapman: ["chapmen"],
  "char-a-bancs": ["chars-a-banc"],
  "chargé d’affaires": ["chargés d’affaires"],
  charity: ["charities"],
  charwoman: ["charwomen"],
  chateau: ["chateux"],
  checkerberry: ["checkerberries"],
  "chef-d’oeuvre": ["chefs-d’oeuvre"],
  cheiropterygium: ["cheiropterygia"],
  chela: ["chelae"],
  chelicera: ["chelicerae"],
  cherub: ["cherubs", "cherubim"],
  chessman: ["chessmen"],
  cheval: ["chevaux"],
  "cheval-de-frise": ["chevaux-de-frise"],
  chich: ["chiches"],
  child: ["children"],
  chimera: ["chimeras"],
  chimney: ["chimneys"],
  chinaman: ["chinamen"],
  chintz: ["chintzes"],
  chlamys: ["chlamyses", "chlamydes"],
  chondroma: ["chondromata"],
  choragus: ["choragi"],
  chorepiscopus: ["chorepiscopi"],
  choriamb: ["choriambs"],
  choriambus: ["choriambi", "choriambuses"],
  chorus: ["choruses"],
  chose: ["choses"],
  chromo: ["chromos"],
  chronology: ["chronologies"],
  chrysalid: ["chrysalids"],
  chrysalis: ["chrysalides"],
  churchman: ["churchmen"],
  ciborium: ["ciboria"],
  cicada: ["cicadas", "cicadae"],
  cicatrix: ["cicatrices"],
  cicerone: ["ciceroni", "cicerones"],
  cicisbeo: ["cicisbei"],
  cimex: ["cimices"],
  cippus: ["cippi"],
  circus: ["circuses"],
  cirrus: ["cirri"],
  city: ["cities"],
  civillty: ["civilities"],
  clansman: ["clansmen"],
  classis: ["classes"],
  classman: ["classmen"],
  claustrum: ["claustra"],
  clavis: ["claves", "clavises"],
  clavy: ["clavies"],
  clemency: ["clemencies"],
  clergyman: ["clergymen"],
  clivity: ["clivities"],
  cloaca: ["cloacae"],
  cloth: ["cloths", "clothes"],
  clypeus: ["clypei"],
  cnida: ["cnidae"],
  coachman: ["coachmen"],
  coagulum: ["coagula"],
  "co-ally": ["co-allies"],
  coccobacterium: ["coccobacteria"],
  coccus: ["cocci"],
  coccyx: ["coccyges"],
  cockney: ["cockneys"],
  codex: ["codices"],
  cognoscente: ["cognoscenti"],
  colliery: ["collieries"],
  colloquy: ["colloquies"],
  collum: ["colla"],
  collyrium: ["collyriums", "collyria"],
  colony: ["colonies"],
  colorman: ["colormen"],
  colossus: ["colossi", "colossuses"],
  columbarium: ["columbaria"],
  columbary: ["columbaries"],
  colure: ["colures"],
  coly: ["colies"],
  comedo: ["comedones"],
  comedy: ["comedies"],
  comicality: ["comicalities"],
  comity: ["comities"],
  commandery: ["commanderies"],
  commentary: ["commentaries"],
  commissary: ["commissaries"],
  commodity: ["commodities"],
  commonalty: ["commonalties"],
  community: ["communities"],
  company: ["companies"],
  compendium: ["compendiums", "compendia"],
  complexity: ["complexities"],
  complice: ["complices"],
  complicity: ["complicities"],
  concavity: ["concavities"],
  concerto: ["concertos"],
  concetto: ["concetti"],
  concubinary: ["concubinaries"],
  conditory: ["conditories"],
  condottiere: ["condottieri"],
  condyloma: ["condylomata", "condylomes"],
  confederacy: ["confederacies"],
  conferva: ["confervae"],
  congiary: ["congiaries"],
  congress: ["congresses"],
  congressman: ["congressmen"],
  congruity: ["congruities"],
  conidium: ["conida"],
  consistory: ["consistories"],
  conspectuity: ["conspectuities"],
  conspiracy: ["conspiracies"],
  constituency: ["constituencies"],
  constitutionality: ["ties"],
  cussuetudinary: ["consuetudinaries"],
  contango: ["contangoes"],
  contemporary: ["contemporaries"],
  content: ["contents"],
  contingency: ["contingencies"],
  continuity: ["continuities"],
  contradictory: ["contradictories"],
  contrariety: ["contrarieties"],
  contrary: ["contraries"],
  contributory: ["contributories"],
  controversy: ["controversies"],
  contumacy: ["contumacies"],
  conventionality: ["conventionalities"],
  "conversazi-one": ["conversazioni"],
  convexity: ["convexities"],
  convicinity: ["convicinities"],
  conviviality: ["convivialities"],
  convolvulus: ["convolvuli", "convoluluses"],
  cooky: ["cookies"],
  cooly: ["coolies"],
  coparcenary: ["coparcenaries"],
  copartnery: ["copartneries"],
  copy: ["copies"],
  coquetry: ["coquetries"],
  corbie: ["corbies"],
  cordiality: ["cordialities"],
  corf: ["corves"],
  cornea: ["corneas"],
  "cornet-à-piston": ["cornets-à-piston"],
  corniculum: ["cornicula"],
  "corno di bassetto": ["corni di basseto"],
  "corno inglese": ["corni inglesi"],
  cornu: ["cornua"],
  "cornu ammonis": ["cornua ammonis"],
  cornucopia: ["cornucopias"],
  corollary: ["corollaries"],
  corona: ["coronae", "coronas"],
  corporality: ["corporalities"],
  corporeality: ["corporealities"],
  corpus: ["corpora"],
  correspondency: ["correspondencies"],
  corrigendum: ["corrigenda"],
  cortex: ["cortices"],
  corundum: ["corundums"],
  corybant: ["corybants", "corybantes"],
  corypheus: ["corypheuses", "coryphei"],
  cosignitary: ["cosignitaries"],
  cosmogony: ["cosmogonies"],
  cosmography: ["cosmographies"],
  cosurety: ["cosureties"],
  cotemporary: ["cotemporaries"],
  councilman: ["councilmen"],
  counterman: ["countermen"],
  countess: ["countesses"],
  country: ["countries"],
  countryman: ["countrymen"],
  countrywoman: ["countrywomen"],
  county: ["counties"],
  "couple-close": ["couple-closes"],
  courtesy: ["courtesies"],
  "court-martial": ["courts-martial"],
  cow: ["cows", "kine"],
  cowberry: ["cowberries"],
  "cowrie cowry": ["cowries"],
  cracksman: ["cracksmen"],
  craftsman: ["craftsmen"],
  cragsman: ["cragsmen"],
  cranberry: ["cranberries"],
  cranium: ["craniums", "crania"],
  cranny: ["crannies"],
  crawfish: ["crawfishes", "crawfish"],
  creamery: ["creameries"],
  credendum: ["credenda"],
  crematorium: ["crematoriums", "crematories"],
  cress: ["cresses"],
  crisis: ["crises"],
  crissum: ["crissa"],
  criterion: ["criteria", "criterions"],
  crony: ["cronies"],
  crotch: ["crotches"],
  "crow’s-foot": ["crow’s-feet"],
  crucifix: ["crucifixes"],
  crudity: ["crudities"],
  cruelty: ["cruelties"],
  crus: ["crura"],
  crutch: ["crutches"],
  crux: ["cruxes", "cruces"],
  cry: ["cries"],
  cryptogamia: ["cryptogamiae"],
  cuirass: ["cuirasses"],
  "cul-de-sac": ["culs-de-sac"],
  cullis: ["cullises"],
  cully: ["cullies"],
  culpability: ["culpabilities"],
  cultus: ["cultuses"],
  cumulus: ["cumuli"],
  cupful: ["cupfuls"],
  cupola: ["cupolas"],
  curacy: ["curacies"],
  curculio: ["curculios"],
  curia: ["curle"],
  curio: ["curios"],
  curiosity: ["curiosities"],
  curioso: ["curiosos"],
  currency: ["currencies"],
  curriculum: ["curriculums", "curricula"],
  curtesy: ["curtesies"],
  custos: ["custodes"],
  cutlass: ["cutlasses"],
  cypress: ["cypresses"],
  cypris: ["cyprides"],
  cytococcus: ["cytococci"],
  czarowitz: ["czarowitzes"],
  dado: ["dadoes"],
  dago: ["dagos"],
  dahlia: ["dahlias"],
  daily: ["dailies"],
  daimio: ["daimios"],
  dainty: ["dainties"],
  dairy: ["dairies"],
  dairyman: ["dairymen"],
  dairywoman: ["dairywomen"],
  daisy: ["daisies"],
  dalesman: ["dalesmen"],
  dandy: ["dandies"],
  "dare-deviltry": ["dare-deviltries"],
  datum: ["data"],
  daughter: ["daughters", "daughtren"],
  "daughter-in-law": ["daughters-in-law"],
  deanery: ["deaneries"],
  debauchery: ["debaucheries"],
  decahedron: ["decahedrons", "decahedra"],
  decemvir: ["decemvirs", "decemviri"],
  decency: ["decencies"],
  decennary: ["decennaries"],
  decennium: ["decenniums", "decennia"],
  declivity: ["declivities"],
  "decoy-man": ["decoy-men"],
  decury: ["decuries"],
  deficiency: ["deficiencies"],
  deformity: ["deformities"],
  deity: ["deities"],
  delay: ["delays"],
  delicacy: ["delicacies"],
  delinquency: ["delinquencies"],
  delivery: ["deliveries"],
  delta: ["deltas"],
  democracy: ["democracies"],
  demy: ["demies"],
  denarius: ["denarii"],
  dentiroster: ["dentirostres"],
  dependency: ["dependencies"],
  depositary: ["depositaries"],
  depository: ["depositories"],
  deputy: ["deputies"],
  desideratum: ["desiderata"],
  desperado: ["desperadoes"],
  desponsory: ["desponsories"],
  destiny: ["destinies"],
  devilry: ["devilries"],
  deviltry: ["deviltries"],
  dey: ["deys"],
  diaeresis: ["diaereses", "diereses"],
  diagnosis: ["diagnoses"],
  dialysis: ["dialyses"],
  diary: ["diaries"],
  dice: ["die"],
  dictionary: ["dictionaries"],
  dictum: ["dicta", "dictums"],
  dido: ["didos"],
  die: ["dice", "dies"],
  diesis: ["dieses"],
  "dies juridicus": ["dies juridici"],
  dietary: ["dietaries"],
  differentia: ["differentiae"],
  difficulty: ["difficulties"],
  dignitary: ["dignitaries"],
  dignity: ["dignities"],
  dilettante: ["dilettanti"],
  dilogy: ["dilogies"],
  diluvium: ["diluviums", "diluvia"],
  diocese: ["dioceses"],
  diplococcus: ["diplococci"],
  diploma: ["diplomas"],
  dipody: ["dipodies"],
  directory: ["directories"],
  directrix: ["directrixes"],
  disability: ["disabilities"],
  discobolus: ["discoboli"],
  discovery: ["discoveries"],
  discrepance: ["discrepances", "discrepancies"],
  discus: ["discuses", "disci"],
  dishful: ["dishfuls"],
  disparity: ["disparities"],
  dispathy: ["dispathies"],
  dispensary: ["dispensaries"],
  dispensatory: ["dispensatories"],
  distaff: ["distaffs", "distaves"],
  distillery: ["distilleries"],
  ditch: ["ditches"],
  ditto: ["dittos"],
  ditty: ["ditties"],
  diversity: ["diversities"],
  diverticulum: ["diverticula"],
  divertimento: ["divertimenti"],
  divinity: ["divinities"],
  djinnee: ["jjinn", "djinns"],
  dodo: ["dodoes"],
  dogma: ["dogmas", "dogmata"],
  dogtooth: ["dogteeth"],
  doing: ["doings"],
  dolly: ["dollies"],
  domesman: ["domesmen"],
  domino: ["dominos", "dominoes"],
  dominus: ["domini"],
  donkey: ["donkeys"],
  dooly: ["doolies"],
  dormitory: ["dormitories"],
  dormouse: ["dormice"],
  dory: ["dories"],
  dowdy: ["dowdies"],
  dowry: ["dowries"],
  doxology: ["doxologies"],
  doxy: ["doxies"],
  dozen: ["dozen", "dozens"],
  drachma: ["drachmas", "drachmae"],
  dracunculus: ["dracunculi"],
  dragman: ["dragmen"],
  dragoman: ["dragomans"],
  drapery: ["draperies"],
  draughtsman: ["draughtsmen"],
  drayman: ["draymen"],
  drollery: ["drolleries"],
  dromedary: ["dromedaries"],
  drongo: ["drongos"],
  dropsy: ["dropsies"],
  drosky: ["droskies"],
  dryas: ["dryades"],
  dubiety: ["dubieties"],
  dubiosity: ["dubiosities"],
  duchy: ["duchies"],
  duenna: ["duennas"],
  dummy: ["dummies"],
  duodecimo: ["duodecimos"],
  duplicity: ["duplicities"],
  dustman: ["dustmen"],
  dutchman: ["dutchmen"],
  duty: ["duties"],
  duumvir: ["duumvirs", "duumviri"],
  dvergr: ["dvergar"],
  dwarf: ["dwarfs"],
  dynasty: ["dynasties"],
  dyscrasy: ["discrasies"],
  earning: ["earnings"],
  ebony: ["ebonies"],
  ebriety: ["ebrieties"],
  eccentricity: ["eccentricities"],
  ecchymosis: ["ecchymoses"],
  ecclesia: ["ecclesiae"],
  ecdysis: ["ecdyses"],
  echinus: ["echini"],
  echo: ["echoes"],
  economy: ["economies"],
  ecstasy: ["ecstasies"],
  ecthoreum: ["ecthorea"],
  ecthyma: ["ecthymata"],
  ectobronchium: ["ectobronchia"],
  ectozoön: ["ectozoa"],
  edda: ["eddas"],
  eddy: ["eddies"],
  effeminacy: ["effeminacies"],
  effigy: ["effigies"],
  effluvium: ["effluvia"],
  effrontery: ["effronteries"],
  "el dorado": ["el doradoes"],
  electricity: ["electricities"],
  electrophorus: ["electrophori"],
  electuary: ["electuaries"],
  eleemosynary: ["eleemosynaries"],
  elegy: ["elegies"],
  elench: ["elenchs"],
  elf: ["elves"],
  ellipsis: ["ellipses"],
  elves: ["elf"],
  elysium: ["elysiums", "elysia"],
  elytron: ["elytra"],
  embargo: ["embargoes"],
  embassy: ["embassies"],
  emblazonry: ["emblazonries"],
  embolus: ["emboli"],
  embroidery: ["embroideries"],
  embryo: ["embryos"],
  emergence: ["emergences"],
  emergency: ["emergencies"],
  emeritus: ["emeriti"],
  eminency: ["eminences"],
  emissary: ["emissaries"],
  emphasis: ["emphases"],
  emporium: ["emporiums", "emporia"],
  empty: ["empties"],
  emunctory: ["emunctories"],
  emyd: ["emyds"],
  encomium: ["encomiums"],
  encrinus: ["encrini"],
  endorhiza: ["endorhizae"],
  endothelium: ["endothelia"],
  endysis: ["endyses"],
  enema: ["enemata"],
  enemy: ["enemies"],
  energy: ["energies"],
  engineman: ["enginemen"],
  englishman: ["englishmen"],
  englishwoman: ["englishwomen"],
  enigma: ["enigmas"],
  enmity: ["enmities"],
  enormity: ["enormities"],
  ensigncy: ["ensigncies"],
  entirety: ["entireness"],
  entity: ["entities"],
  entobronchium: ["entobronchia"],
  entomology: ["entomologies"],
  entoplastron: ["entoplastra"],
  entosternum: ["entosterna"],
  entozoön: ["entozoa"],
  entreaty: ["entreaties"],
  entry: ["entries"],
  envy: ["envies"],
  eozoön: ["eozoöns", "eozoa"],
  epenthesis: ["epentheses"],
  ephemeris: ["ephemerides"],
  ephemeron: ["ephemera"],
  ephor: ["ephors", "ephori"],
  epichirema: ["epichiremata"],
  epimeron: ["epimera"],
  epiphysis: ["epiphyses"],
  epiplastron: ["epiplastra"],
  epiploön: ["epiploa"],
  epipodiale: ["epipodialia"],
  epipodium: ["epipodia"],
  epipubis: ["epipubes"],
  episternum: ["episterna"],
  epithalamium: ["epithalamiums", "epithalamia"],
  epithalamy: ["epithalamies"],
  epithelium: ["epitheliums", "epithelia"],
  epitome: ["epitomes"],
  epizoön: ["epizoa"],
  equality: ["equalities"],
  equerry: ["equerries"],
  equilibrium: ["equilibriums", "equilibria"],
  equisetum: ["equiseta"],
  equity: ["equities"],
  era: ["eras"],
  erf: ["erven"],
  erinys: ["erinyes"],
  erratum: ["errata"],
  eruca: ["erucae"],
  eskimo: ["eskimos"],
  espy: ["espies"],
  esquimau: ["esquimaux"],
  essay: ["essays"],
  essene: ["essenes"],
  estuary: ["estuaries"],
  estufa: ["estufas"],
  eternity: ["eternities"],
  etymology: ["etymologies"],
  etymon: ["etymons", "etyma"],
  eulogium: ["eulogiums"],
  eulogy: ["eulogies"],
  euphony: ["euphonies"],
  eventuality: ["eventualities"],
  exanthema: ["exanthemata"],
  excellency: ["excellencies"],
  exciseman: ["excisemen"],
  exedra: ["exedrae"],
  exegesis: ["exegeses"],
  exequy: ["exequies"],
  exigency: ["exigencies"],
  "ex officio": ["ex officiis"],
  exordium: ["exordiums", "exordia"],
  exorhiza: ["exorhize"],
  exotery: ["exoteries"],
  expressman: ["expressmen"],
  extra: ["extras"],
  extraordinary: ["extraordinaries"],
  extravagancy: ["extravagancies"],
  extremity: ["extremities"],
  "ex-voto": ["ex-votos"],
  ey: ["eyren"],
  eyetooth: ["eyeteeth"],
  eyrie: ["eyries"],
  fabella: ["fabellae"],
  fabliau: ["fabliaux"],
  facility: ["facilities"],
  facsimile: ["facsimiles"],
  factory: ["factories"],
  factotum: ["factotums"],
  factum: ["facta"],
  faculty: ["faculties"],
  fairy: ["fairies"],
  fallacy: ["fallacies"],
  falsetto: ["falsettos"],
  falsity: ["falsities"],
  familiarity: ["familiarities"],
  familistery: ["familisteries"],
  family: ["families"],
  fancy: ["fancies"],
  fandango: ["fandangoes"],
  fantasy: ["fantasies"],
  fascia: ["fasciae"],
  fasciculus: ["fasciculi"],
  fasciola: ["fasciolae"],
  fatality: ["fatalities"],
  "father-in-law": ["fathers-in-law"],
  faunus: ["fauni"],
  faux: ["fauces"],
  feasibility: ["feasibilities"],
  fecula: ["feculae"],
  felicity: ["felicities"],
  fellah: ["fellahin", "fellahs"],
  felly: ["fellies"],
  "felo-de-se": ["felos-de-se"],
  felony: ["felonies"],
  femur: ["femora"],
  fenestra: ["fenestrae"],
  feria: ["feriae"],
  ferry: ["ferries"],
  ferryman: ["ferrymen"],
  festivity: ["festivities"],
  fetter: ["fetters"],
  fetus: ["fetuses"],
  feudatory: ["feudatories"],
  fiasco: ["fiascoes"],
  fibrilla: ["fibrillae"],
  fibula: ["fibulae"],
  fibulare: ["fibularia"],
  fico: ["ficoes"],
  fifty: ["fifties"],
  filly: ["fillies"],
  fimbria: ["fimbriae"],
  finality: ["finalities"],
  finch: ["fishes"],
  fingrigo: ["fingrigos"],
  firefly: ["fireflies"],
  fireman: ["firemen"],
  firman: ["firmans"],
  fish: ["fishes", "fish"],
  fisherman: ["fishermen"],
  fishery: ["fisheries"],
  fishwoman: ["fishwomen"],
  fistula: ["fistulae"],
  fitch: ["fitches"],
  flagellum: ["flagellums", "flagella"],
  flagman: ["flagmen"],
  flagrancy: ["flagrancies"],
  flagstaff: ["flagstaves", "flagstaffs"],
  flambeau: ["flambeaux", "flambeaus"],
  flamen: ["flammens", "flamines"],
  flamingo: ["flamingoes"],
  flanch: ["flanches"],
  flash: ["flashes"],
  flattery: ["flatteries"],
  flatus: ["flatuses", "flatus"],
  "fleur-de-lis": ["fleurs-de-lis"],
  flitch: ["flitches"],
  flo: ["flon"],
  flocculus: ["flocculi"],
  floccus: ["flocci"],
  flourish: ["flourishes"],
  flunky: ["flunkies"],
  flurry: ["flurries"],
  fly: ["flies"],
  flyman: ["flymen"],
  focus: ["focuses", "foci"],
  foeman: ["foemen"],
  fogy: ["fogies"],
  folio: ["folios"],
  folium: ["foliums", "folia"],
  folly: ["follies"],
  fomes: ["fomites"],
  foolery: ["fooleries"],
  foot: ["feet"],
  footman: ["footmen"],
  footpath: ["footpaths"],
  foppery: ["fopperies"],
  foramen: ["foramina", "foramines"],
  foreman: ["foremen"],
  "fore tooth": ["fore teeth"],
  forewomen: ["forewomen"],
  forgeman: ["forgemen"],
  forgery: ["forgeries"],
  formality: ["formalities"],
  formula: ["formulas", "formulae"],
  formulary: ["formularies"],
  fornix: ["fornices"],
  fortress: ["fortresses"],
  forty: ["forties"],
  forum: ["forums", "fora"],
  fossa: ["fossae"],
  foundery: ["founderies"],
  foundry: ["foundries"],
  fovea: ["foveae"],
  foveola: ["foveolae"],
  fovilla: ["fovillae"],
  fowl: ["fowls"],
  fox: ["foxes"],
  fraenulum: ["fraenula"],
  fraenum: ["fraenums", "fraena"],
  frailty: ["frailties"],
  fraternity: ["fraternities"],
  freedman: ["freedmen"],
  freeman: ["freemen"],
  frenchman: ["frenchmen"],
  frenum: ["frenums", "frena"],
  frenzy: ["frenzies"],
  frequency: ["frequencies"],
  fresco: ["frescoes", "frescos"],
  fresh: ["freshes"],
  freshman: ["freshmen"],
  fretum: ["freta"],
  frigidarium: ["frigidaria"],
  frivolity: ["frivolities"],
  friz: ["frizzes"],
  floatiersman: ["frontiersmen"],
  fructuary: ["fructuaries"],
  frugality: ["frugalities"],
  fruitery: ["fruiteries"],
  frustum: ["frusta", "frustums"],
  fuchsia: ["fuchsias", "fuchslae"],
  fucus: ["fuci"],
  fugleman: ["fuglemen"],
  fulcrum: ["fulcra", "fulcrums"],
  fullery: ["fulleries"],
  fumade: ["fumades", "fumadoes"],
  functionary: ["functionaries"],
  fungus: ["fungi", "funguses"],
  funiculus: ["funiculi"],
  funny: ["funnies"],
  fury: ["furies"],
  futurity: ["futurities"],
  gadfly: ["gadflies"],
  galago: ["galagos"],
  galaxy: ["galaxies"],
  gallantry: ["gallantries"],
  gallery: ["galleries"],
  galley: ["galleys"],
  gallfly: ["gallflies"],
  gallimaufry: ["gallimaufries"],
  gallows: ["gallowses", "gallows"],
  ganglion: ["ganglia", "ganglions"],
  gas: ["gases"],
  gastrula: ["gastrulae"],
  gaucho: ["gauchos"],
  gaudy: ["gaudies"],
  gayety: ["gayeties"],
  gecko: ["geckoes"],
  gemma: ["gemmae"],
  gendarme: ["gendarmes", "gens d’armes"],
  genealogy: ["genealogies"],
  generality: ["generalities"],
  generatrix: ["generatrices", "generatrixes"],
  genius: ["geniuses", "genii"],
  gens: ["gentes"],
  gentleman: ["gentlemen"],
  gentlewoman: ["gentlewomen"],
  gentoo: ["gentoos"],
  genu: ["genua"],
  genus: ["genera"],
  geography: ["geographies"],
  geology: ["geologies"],
  geometry: ["geometries"],
  german: ["germans"],
  germen: ["germens", "germina"],
  gigerium: ["gigeria"],
  ginglymus: ["ginglymi"],
  ginkgo: ["ginkgoes"],
  ginnee: ["ginn"],
  glabella: ["glabellas", "glabellae"],
  glabellum: ["glabella"],
  gladiolus: ["gladioli", "gladioluses"],
  gladius: ["gladii"],
  glans: ["glandes"],
  glassful: ["glassfuls"],
  gleeman: ["gleemen"],
  globigerina: ["globigerin&ae"],
  glochidium: ["glochidia"],
  glomerulus: ["glomeruli"],
  glossa: ["glossae"],
  glossary: ["gossaries"],
  gluttony: ["gluttonies"],
  gnathidium: ["gnathidia"],
  gnathotheca: ["gnathothecae"],
  goaf: ["goafs", "goaves"],
  goby: ["gobies"],
  gonad: ["gonads"],
  gonangium: ["gonangia", "gonangiums"],
  gonidium: ["gonidia"],
  gonoblastidium: ["gonoblastidia"],
  gonotheca: ["gonothecae"],
  goody: ["goodies"],
  goose: ["geese"],
  gooseberry: ["gooseberries"],
  goosery: ["gooseries"],
  gorfly: ["gorflies"],
  gorgoneion: ["gorgoneia"],
  gownsman: ["gownsmen"],
  gradino: ["gradinos"],
  grampus: ["grampuses"],
  granary: ["granaries"],
  gratuity: ["gtratuities"],
  gravamen: ["gravamina ", "gravamens"],
  gravity: ["gravities"],
  gravy: ["gravies"],
  gree: ["grees", "greece", "grice", "grise", "grize"],
  grizzly: ["grizzlies"],
  grocery: ["groceries"],
  groggery: ["groggeries"],
  groomsman: ["groomsmen"],
  grotto: ["grottoes"],
  guacho: ["guachos"],
  guanaco: ["guanacos"],
  guano: ["guanos"],
  guarantee: ["guarantees"],
  guaranty: ["guaranies"],
  guardsman: ["guardsmen"],
  gula: ["gulae", "gulas"],
  gully: ["gulles"],
  gumma: ["gummata"],
  gutta: ["guttae"],
  gymnasium: ["gymnasiums", "gymnasia"],
  gypsy: ["gypsies"],
  gyrus: ["gyri"],
  hackman: ["hackmen"],
  hackney: ["hackneys"],
  hackneyman: ["hackneymen"],
  haematozoön: ["haematozoa"],
  haggada: ["haggadoth"],
  halacha: ["halachoth"],
  half: ["halves"],
  halo: ["halos"],
  hamadryad: ["hamadryads", "hamadryades"],
  hammerman: ["hammermen"],
  hamulus: ["hamuli"],
  handcraftsman: ["handcraftsmen"],
  handful: ["hand flus"],
  "handi-craftsman": ["handicraftsmen"],
  "hang-by": ["hang-bies"],
  "hanger-on": ["hangers-on"],
  hangman: ["hangmen"],
  hardwareman: ["hardwaremen"],
  harmony: ["harmonies"],
  harpy: ["harpies"],
  harvestman: ["harvestmen"],
  haustellum: ["haustella"],
  haustorium: ["haustoria"],
  headman: ["headmen"],
  headsman: ["headsmen"],
  hearty: ["hearties"],
  heathen: ["heathens", "heathen"],
  hectocotylus: ["hectocotyli"],
  heddle: ["heddles"],
  helix: ["helices", "helixes"],
  helmsman: ["helmsmen"],
  hemapophysis: ["hemapophyses"],
  hemelytron: ["hemelytra"],
  hemina: ["heminae"],
  henchman: ["henchmen"],
  henhouse: ["henhouses"],
  henry: ["henrys"],
  hepatica: ["hepaticae"],
  herbarium: ["herbariums", "herbaria"],
  "herb-woman": ["herb-women"],
  herdman: ["herdmen"],
  herdswoman: ["women"],
  heresy: ["heresies"],
  herma: ["hermae"],
  hernia: ["hernias", "herniae"],
  hero: ["heroes"],
  hetman: ["hetmans"],
  hexahedron: ["hexahedrons", "hexahedra"],
  hiatus: ["hiatus", "hiatuses"],
  hierarchy: ["hierarchies"],
  hierotheca: ["hierothecae"],
  "high-churchman": ["highchurchmen"],
  highwayman: ["highwaymen"],
  hindoo: ["hindoos", "hindus"],
  hinny: ["hinnies"],
  hippopotamus: ["hippopotamuses", "hippopotami"],
  history: ["histories"],
  hobby: ["hobbies"],
  hodman: ["hodmen"],
  hogsty: ["hogsties"],
  homily: ["homilies"],
  homunculus: ["homunculi"],
  hoof: ["hoofs", "hooves"],
  horsefly: ["horseflies"],
  horsefoot: ["horsefeet"],
  horseman: ["horsemen"],
  horsewoman: ["horsewomen"],
  hosanna: ["hosannas"],
  hose: ["hose", "hosen"],
  hospitality: ["hospitalities"],
  hostility: ["hostilities"],
  houri: ["houris"],
  house: ["houses"],
  hoyman: ["hoymen"],
  humanity: ["humanities"],
  humerus: ["humeri"],
  humility: ["humilities"],
  huntsman: ["huntsmen"],
  hurricano: ["hurricanoes"],
  husbandman: ["husbandmen"],
  hydra: ["hydras", "hydrae"],
  hydrocaulus: ["hydrocauli"],
  hydromedusa: ["hydromedusae"],
  hydrophyllium: ["hydrophyllia", "hydrophylliums"],
  hydrorhiza: ["hydrorhizae", "hydrorhizas"],
  hydrotheca: ["hydrothecae", "hydrothecas"],
  hydrozoön: ["hydrozoa", "hydrozoöns"],
  hyena: ["hyenas"],
  hymenium: ["hymenia", "hymeniums"],
  hypanthium: ["hypanthia", "hypanthiums"],
  hypapophysis: ["hypapophyles"],
  hyperapophysis: ["hyperapophyses"],
  hypoarion: ["hypoaria"],
  hypochondrium: ["hypochondria", "hypochondriums"],
  hypocleidium: ["hypocleida", "hypocleidiums"],
  hpocrisy: ["hypocrisies"],
  hypodactylum: ["hypodactyla"],
  hypogeum: ["hypogea"],
  hypoplastron: ["hypoplastra"],
  hypoptilum: ["hypoptila", "hypoptilums"],
  hyporadius: ["hyporadii"],
  hyporhachis: ["hyporhachides"],
  hypostasis: ["hypostases"],
  hyposternum: ["hyposterna", "hyposternums"],
  hypotarsus: ["hypotarsi"],
  hypothesis: ["hypotheses"],
  i: ["we", "our", "ours", "us"],
  iambus: ["iambi", "iambuses"],
  ianthina: ["ianthinae", "ianthinas"],
  ibex: ["ibexes", "ibices"],
  iceman: ["icemen"],
  ichthyosaurus: ["ichthyosauri"],
  idea: ["ideas"],
  ideality: ["idealities"],
  identity: ["identities"],
  idiocracy: ["idiocrasies"],
  idiopathy: ["idiopathies"],
  idiosyncrasy: ["idiosyncrasies"],
  idolatry: ["idolatries"],
  "ignis fatuus": ["ignes fatui"],
  ignominy: ["ignominies"],
  illegality: ["illegalities"],
  illiteracy: ["illiteracies"],
  imago: ["imagoes"],
  imbecility: ["imbecilities"],
  imbrocado: ["imbrocadoes"],
  imbroglio: ["imbroglios"],
  immateriality: ["immaterialities"],
  immensity: ["immensities"],
  immorality: ["immoralities"],
  immortality: ["immortalities"],
  immortelle: ["immortelles"],
  immunity: ["immunities"],
  imperiality: ["imperialities"],
  impiety: ["impieties"],
  importunity: ["importunities"],
  impossibility: ["impossibilities"],
  impracticability: ["impracticabilities"],
  impresario: ["impresarios"],
  impress: ["impresses"],
  improbability: ["improbabilities"],
  impropriatrix: ["impropriatrixes", "impropriatrices"],
  impropriety: ["improprieties"],
  improvvisatore: ["improvvisatori"],
  improvvisatrice: ["improvvisatrici"],
  inaccuracy: ["inaccuracies"],
  inadvertence: ["inadvertences"],
  inamorato: ["inamoratos"],
  inanity: ["inanities"],
  incapacity: ["incapacities"],
  incendiary: ["incendiaries"],
  incensory: ["incensories"],
  incivility: ["incivilities"],
  inclemency: ["inclemencies"],
  incognito: ["incognitos"],
  incommodity: ["incommodities"],
  incompatibility: ["incompatibilities"],
  incongruity: ["incongruities"],
  inconsistency: ["inconsistencies"],
  incubus: ["incubuses", "incubi"],
  incumbency: ["incumbencies"],
  incunabulum: ["incunabula"],
  indecency: ["indecencies"],
  indelicacy: ["indelicacies"],
  indemnity: ["indemnities"],
  index: ["indexes", "indices"],
  indiaman: ["indiamen"],
  indignity: ["indignities"],
  indigo: ["indigoes"],
  individuality: ["individualities"],
  inductorium: ["inductoriums", "inductoria"],
  indusium: ["indusia"],
  industry: ["industries"],
  inelegance: ["inelegances", "inelegancies"],
  inequality: ["inequalities"],
  infamy: ["infamies"],
  infelicity: ["infelicities"],
  infidelity: ["infidelities"],
  infinity: ["infinities"],
  infirmary: ["infirmaries"],
  infirmity: ["infirmities"],
  informality: ["informalities"],
  infula: ["infule"],
  infundibulum: ["infundibula", "infundibulums"],
  infusory: ["infusories"],
  inhumanity: ["inhumanities"],
  iniquity: ["iniquities"],
  injuria: ["injurie"],
  injury: ["injuries"],
  innuendo: ["innuedoes"],
  inquiry: ["inquiries"],
  insectivore: ["insectivores"],
  insecurity: ["insecurities"],
  insessor: ["insessores"],
  insolvency: ["insolvencies"],
  instability: ["instabilities"],
  instrumentality: ["instrumentalities"],
  intaglio: ["intaglius", "intagli"],
  intangibility: ["intangibilities"],
  intendancy: ["intendancies"],
  interambulacrum: ["interambulacra", "interambulacrums"],
  interaxis: ["interaxes"],
  intercentrum: ["intercentra"],
  interleaf: ["interleaves"],
  intermaxilla: ["intermaxillae"],
  intermediary: ["intermediaries"],
  intermedium: ["intermediums", "intermedia"],
  internuncio: ["internuncios"],
  interoperculum: ["interopercula"],
  interregnum: ["interregnums"],
  interrex: ["interrexes", "interreges"],
  interrogatory: ["interrogatories"],
  interstice: ["interstices"],
  intervallum: ["intervallums", "intervalla"],
  intestine: ["intestines"],
  intimacy: ["intimacies"],
  intricacy: ["intricacies"],
  inventory: ["inventories"],
  invisibility: ["invisibilities"],
  invitatory: ["invitatories"],
  involucellum: ["involucella"],
  involucrum: ["involucra", "involucrums"],
  io: ["ios"],
  iris: ["irises", "irides"],
  irishman: ["irishmen"],
  irregularity: ["irregularities"],
  isthmus: ["isthmuses"],
  italic: ["italics"],
  itinerary: ["itineraries"],
  ivory: ["ivories"],
  ivy: ["ivies"],
  jackman: ["jackmen"],
  jacobus: ["jacobuses"],
  janizary: ["janizaries"],
  jealousy: ["jealousies"],
  jelly: ["jellies"],
  jenny: ["jennies"],
  jersey: ["jerseys"],
  jess: ["jesses"],
  "jet d’eau": ["jets d’eau"],
  jetty: ["jetties"],
  jimmy: ["jimmies"],
  jingo: ["jingoes"],
  jinnee: ["jinn"],
  jo: ["joes"],
  jockey: ["jockeys"],
  johnny: ["johnnies"],
  journey: ["journeys"],
  journeyman: ["journeymen"],
  juba: ["jubae"],
  jugulum: ["jugula"],
  jugum: ["juga", "jugums"],
  julus: ["juli"],
  july: ["julies"],
  jungermannia: ["jungermanniae"],
  june: ["junos"],
  junta: ["juntas"],
  junto: ["juntos"],
  juramentum: ["juramenta"],
  jury: ["juries"],
  juryman: ["jurymen"],
  juvenility: ["juvenilities"],
  karreo: ["karroos"],
  kavass: ["kavasses"],
  kecksy: ["kecksies"],
  keelman: ["men"],
  kelpie: ["kelpies"],
  kerchief: ["kerchiefs"],
  kersey: ["kerseys"],
  kibitka: ["kibitkas"],
  kickshaws: ["kickshawses"],
  kidney: ["kidneys"],
  kilo: ["kilos"],
  kinsman: ["kinsmen"],
  kinswoman: ["kinswomen"],
  kirkman: ["kirkmen"],
  kivikivi: ["kivikivies", "kiwikiwies"],
  knavery: ["knaveries"],
  knife: ["knives"],
  "knight bachelor": ["knights bachelors"],
  "knight banneret": ["knights bannerets"],
  "knight-errant": ["knight-errants", "knights-errant"],
  "knight-errantry": ["knight-errantries"],
  "knight templar": ["knights templars"],
  "kohl-rabi": ["kohl-rabies"],
  krooman: ["kroomen"],
  labarum: ["labara"],
  labellum: ["labella", "labellums"],
  labipalpus: ["labipalpi"],
  labium: ["labia", "labiums"],
  laboratory: ["laboratories"],
  labrum: ["labra", "labrums"],
  labrus: ["labri"],
  laceman: ["lacemen"],
  lacertus: ["lacerti"],
  lachrymatory: ["ries"],
  lacinia: ["laciniae"],
  lacinula: ["lacinulae", "lacinulas"],
  lackey: ["lackeys"],
  lacuna: ["lacunae", "lacunas"],
  lacunar: ["lacunars", "lacunaria"],
  ladino: ["ladinos"],
  ladleful: ["ladlefuls"],
  lady: ["ladies"],
  lagena: ["lagenae", "lagenas"],
  lamella: ["lamellae", "lamellas"],
  lamina: ["lamella"],
  lamprey: ["lampreys"],
  landlady: ["landladies"],
  landman: ["landmen"],
  landsman: ["landsmen"],
  lapful: ["lapfuls"],
  lapidary: ["lapidaries"],
  lapis: ["lapides"],
  laquear: ["laquearia"],
  lar: ["lares", "lars"],
  larceny: ["larcenies"],
  larva: ["larvae", "larvas"],
  larve: ["larves"],
  lasso: ["lassos"],
  later: ["lateres"],
  lath: ["laths"],
  latibulum: ["latibula"],
  laundry: ["laundries"],
  laundryman: ["laundrymen"],
  lavatory: ["lavatories"],
  layman: ["laymen"],
  leadman: ["leadmen"],
  leadsman: ["leadsmen"],
  leaf: ["leaves"],
  lectica: ["lecticae"],
  lectionary: ["lectionaries"],
  lee: ["lees"],
  leetman: ["leetmen"],
  legacy: ["legacies"],
  legionary: ["legionaries"],
  legumen: ["legumina", "legumens"],
  lemma: ["lemmata", "lemmas"],
  lemniscus: ["lemnisci"],
  lens: ["lenses"],
  lenticula: ["lenticulas", "lenticulae"],
  lethargy: ["lethargies"],
  levy: ["levies"],
  lex: ["leges"],
  liability: ["liabilities"],
  liberality: ["liberalities"],
  liberty: ["liberties"],
  libra: ["librae"],
  library: ["libraries"],
  libretto: ["librettos", "libretti"],
  lied: ["lieder"],
  liegeman: ["liegemen"],
  lienculus: ["lienculi"],
  life: ["lives"],
  lighterman: ["lightermen"],
  "light-horseman": ["lighthorsemen"],
  lighthouse: ["lighthouses"],
  lightman: ["lightmen"],
  ligula: ["ligulae", "ligulas"],
  lily: ["lilies"],
  limitary: ["limitaries"],
  limulus: ["limuli"],
  lineman: ["linemen"],
  lingua: ["linguae"],
  lingula: ["lingulae"],
  "lion’s tooth": ["lions’ teeth"],
  lira: ["lire"],
  liriodendron: ["liriodendra"],
  litany: ["litanies"],
  literatus: ["literati"],
  liturgy: ["liturgies"],
  lituus: ["litui"],
  livery: ["liveries"],
  liveryman: ["liverymen"],
  llano: ["llanos"],
  loaf: ["loaves"],
  lobby: ["lobbies"],
  locality: ["localitiees"],
  loculus: ["loculi"],
  locus: ["loci", "loca"],
  logman: ["logmen"],
  loma: ["lomata"],
  longiroster: ["longirostres", "longirosters"],
  longshoreman: ["longshoremen"],
  looby: ["loobies"],
  lophosteon: ["lophostea", "lophosteons"],
  lorica: ["loricae"],
  lorrie: ["lorries"],
  lory: ["lories"],
  lottery: ["lotteries"],
  louse: ["lice"],
  "low-churchman": ["lowchurchmen"],
  lumberman: ["lumbermen"],
  luminary: ["luminaries"],
  lunacy: ["lunacies"],
  lunula: ["lunulae"],
  lustrum: ["lustrums", "lustra"],
  luxury: ["luxuries"],
  lyceum: ["lyceums", "lycea"],
  lytta: ["lyttae"],
  macula: ["maculae"],
  madam: ["madams", "mesdames"],
  madame: ["mesdames"],
  mademoiselle: ["mesdemoiselles"],
  madman: ["madmen"],
  magistracy: ["magistracies"],
  magistrality: ["magistralities"],
  magnifico: ["magnificoes"],
  majesty: ["majesties"],
  majority: ["majorities"],
  malady: ["maladies"],
  malleolus: ["malleoli"],
  malleus: ["mallei"],
  maltman: ["maltmen"],
  malum: ["mala"],
  mamma: ["mammae"],
  mammal: ["mammals"],
  mammilla: ["mammilae"],
  mammy: ["mammies"],
  man: ["men"],
  mango: ["mangoes"],
  manifest: ["manifests"],
  manifesto: ["manifestoes"],
  manofwar: ["men-of-war"],
  manteau: ["manteaux", "manteaus"],
  manubrium: ["manubria", "manubriums"],
  manufactory: ["manufactories"],
  manus: ["manus"],
  maori: ["maoris"],
  marksman: ["marksmen"],
  maronite: ["maronites"],
  marsupium: ["marsupia"],
  martyrology: ["martyrologies"],
  mastery: ["masteries"],
  masticatory: ["masticatories"],
  mastiff: ["mastiffs", "mastives"],
  matrix: ["matrices"],
  mausoleum: ["mausoleums", "mausolea"],
  maxilla: ["maxillae"],
  maximum: ["maxima"],
  meatus: ["meatuses"],
  media: ["mediae"],
  medicornu: ["medicornua"],
  medium: ["media", "e", "mediums"],
  medius: ["medii"],
  medley: ["medleys"],
  melisma: ["melismata"],
  melody: ["melodies"],
  memento: ["mementos"],
  memorandum: ["memorandums", "memoranda"],
  memory: ["memories"],
  menaion: ["menaia"],
  mendacity: ["mendacities"],
  meniscus: ["menisci", "meniscuses"],
  menologium: ["menologia", "menologies"],
  menstruum: ["menstruums", "menstrua"],
  mercenary: ["mercenaries"],
  merchantman: ["merchantmen"],
  mercy: ["mercies"],
  merino: ["merinos"],
  merman: ["mermen"],
  mesobronchium: ["mesobronchia"],
  mesopodiale: ["mesopodialia"],
  mestino: ["mestinos"],
  mestizo: ["mestizos"],
  metabasis: ["metabases"],
  metalepsis: ["metalepses"],
  metalman: ["metalmen"],
  metamorphosis: ["metamorphoses"],
  metapodiale: ["metapodialia"],
  metapodium: ["metapodia"],
  metapophysis: ["metapophyses"],
  metastasis: ["metastases"],
  metatarsus: ["metatarsi"],
  metathesis: ["metatheses"],
  metazoan: ["metazoans"],
  miasma: ["miasmata"],
  micella: ["micellae"],
  micrococcus: ["micrococci"],
  middleman: ["middlemen"],
  middy: ["middies"],
  midrash: ["midrashim", "midrashoth"],
  midshipman: ["midshipmen"],
  midwife: ["midwives"],
  mighty: ["mighties"],
  militiaman: ["militiamen"],
  milkman: ["milkmen"],
  milliary: ["milliaries"],
  mina: ["minae", "minas"],
  mineralogy: ["mineralogies"],
  minimum: ["minima"],
  minimus: ["minimi"],
  minority: ["minorities"],
  mintman: ["mintmen"],
  minuteman: ["minutemen"],
  minutia: ["minutiae"],
  mirabilary: ["mirabilaries"],
  miscellany: ["miscellanies"],
  misery: ["miseries"],
  miss: ["misses"],
  missa: ["missae"],
  missionary: ["missionaries"],
  mockery: ["mockeries"],
  modiolus: ["modioli"],
  modius: ["modii"],
  modulus: ["moduli"],
  modus: ["modi"],
  moiety: ["moieties"],
  momentum: ["momenta", "momentums"],
  monarchy: ["monarchies"],
  monastery: ["monasteries"],
  moneron: ["monera", "monerons"],
  money: ["moneys"],
  monkery: ["monkeries"],
  monkey: ["monkeys"],
  monody: ["monodies"],
  monopodium: ["monopodia", "monopodiums"],
  monopoly: ["monopolies"],
  monopteron: ["monoptera"],
  monseigneur: ["messeigneurs"],
  monsieur: ["messieurs"],
  monsignore: ["monsignors"],
  monstrosity: ["monstrosities"],
  monthly: ["monthlies"],
  mootman: ["mootmen"],
  morality: ["moralities"],
  mortuary: ["mortuaries"],
  morula: ["morulae"],
  moslem: ["moslems", "moslem"],
  mosquito: ["mosquitoes"],
  moth: ["moths"],
  motto: ["mottoes"],
  mouse: ["mice"],
  mouth: ["mouths"],
  mouthful: ["mouthfuls"],
  movable: ["movables"],
  mufti: ["muftis"],
  mulatto: ["mulattoes"],
  mulberry: ["mulberries"],
  mummery: ["mummeries"],
  mummy: ["mummies"],
  municipality: ["municipalities"],
  murex: ["murices"],
  mus: ["mures"],
  musa: ["musae"],
  musca: ["muscae"],
  mussulman: ["mussulmans"],
  mustache: ["mustaches"],
  mustacho: ["mustachios"],
  mutandum: ["mutanda"],
  mutiny: ["mutinies"],
  myeloplax: ["myeloplaxes", "myeloplaces"],
  myocomma: ["myocommata", "myocommas"],
  myself: ["ourselves"],
  mystery: ["mysteries"],
  mythology: ["mythologies"],
  myxoma: ["myxomata"],
  navus: ["naevi"],
  nailery: ["naileries"],
  napery: ["naperies"],
  nappy: ["nappies"],
  narcissus: ["narcissuses"],
  narrow: ["narrows"],
  nassa: ["nassas", "nassae"],
  natica: ["naticas", "naticae"],
  nationality: ["nationalities"],
  nativity: ["nativies"],
  nauplius: ["nauplii"],
  nautilus: ["nautiluses", "nautili"],
  navvy: ["navies"],
  navy: ["navies"],
  nay: ["nays"],
  nebula: ["nebulae"],
  necessary: ["necessaries"],
  necessity: ["necessities"],
  necrology: ["necrologies"],
  necropolis: ["necropolises"],
  nectary: ["nectaries"],
  nectocalyx: ["nectocalyces"],
  neddy: ["neddies"],
  needleful: ["needlefuls"],
  needlewoman: ["needlewomen"],
  negress: ["negresses"],
  negro: ["negroes"],
  nemathecium: ["nemathecia"],
  nematocalyx: ["nematocalyces", "calyxes"],
  nephridium: ["nephridia"],
  nereid: ["nereids", "nereides"],
  nereis: ["nereides"],
  nestful: ["nestfuls"],
  neurapophysis: ["neurapophyses"],
  neuron: ["neura"],
  newsman: ["newsmen"],
  nicety: ["niceties"],
  nidus: ["nidi"],
  nightman: ["nightmen"],
  nimbus: ["nimbi", "nimbuses"],
  ninepence: ["ninepences"],
  ninety: ["nineties"],
  ninny: ["ninnies"],
  nisey: ["nyseys"],
  no: ["noes"],
  nobleman: ["noblemen"],
  noblewoman: ["noblewomen"],
  nobody: ["nobodies"],
  noctiluca: ["noctilucae"],
  noddy: ["noddies"],
  nomarchy: ["nomarchies"],
  nonentity: ["nonentities"],
  nopalry: ["nopalries"],
  norimon: ["norimons"],
  norseman: ["norsemen"],
  northman: ["northmen"],
  nostrum: ["nostrums"],
  notability: ["notabilities"],
  notandum: ["notanda"],
  notary: ["notaries"],
  notopodium: ["notopodia", "e", "notopodiums"],
  notum: ["nota"],
  novelty: ["novelties"],
  nubecula: ["nubeculae"],
  nucellus: ["nucelli"],
  nucha: ["nuchae"],
  nucleolus: ["nucleoli"],
  nucleus: ["nucleuses", "nuclei"],
  nudity: ["nudities"],
  nullity: ["nullities"],
  nuncio: ["nuncios"],
  nuncius: ["nuncii"],
  nunnery: ["nunneries"],
  nuptial: ["nuptials"],
  nursery: ["nurseries"],
  nurseryman: ["nurserymen"],
  nympha: ["nymphae"],
  o: ["o’s", "oes"],
  oath: ["oaths"],
  obelus: ["obeli"],
  obituary: ["obituaries"],
  oblatum: ["oblata"],
  obliquity: ["obliquities"],
  oblongum: ["oblonga"],
  obolus: ["oboli"],
  obscenity: ["obscenities"],
  obsequy: ["obsequies"],
  observandum: ["observanda"],
  observatory: ["observatories"],
  occiput: ["occipita", "occiputs"],
  ocellus: ["ocelli"],
  ochrea: ["ochreaee"],
  octavo: ["octavos"],
  octodecimo: ["octodecimos"],
  oculus: ["oculi"],
  oddity: ["oddities"],
  offertory: ["offertories"],
  oilman: ["oilmen"],
  olfactory: ["olfactories"],
  oligarchy: ["oligarchies"],
  omentum: ["omenta"],
  ommateum: ["ommatea"],
  ommatidium: ["ommatidia"],
  onager: ["onagri", "onagers"],
  ooecium: ["ooecia"],
  oögonium: ["oögonia", "oögoniums"],
  oöphoridium: ["oöphorida", "oöphoridiums"],
  oösporangium: ["oösporangia", "oösporangiums"],
  oötheca: ["oöthecae"],
  operculum: ["opercula", "operculums"],
  ophidion: ["ophidia"],
  opportunity: ["opportunities"],
  opus: ["opera"],
  opusculum: ["opuscula"],
  orangeman: ["orangemen"],
  oratory: ["oratories"],
  orchis: ["orchises"],
  orderly: ["orderlies"],
  ordinary: ["ordinaries"],
  orgy: ["orgies"],
  orrery: ["orreries"],
  ort: ["orts"],
  orthostichy: ["orthostichies"],
  os: ["ossa", "ora", "osar"],
  osculatrix: ["osculatrixes"],
  osculum: ["oscula"],
  osmanli: ["osmanlis"],
  osmaterium: ["osmateria"],
  osphradium: ["osphradia"],
  ossiculum: ["ossicula"],
  ossuary: ["ossuaries"],
  ostensorium: ["ostensoria", "ostensories"],
  osteocomma: ["osteocommata", "osteocommas"],
  osteoma: ["osteomata"],
  osteosarcoma: ["osteosarcomata"],
  ostiary: ["ostiaries"],
  ostium: ["ostia"],
  otary: ["otaries"],
  ottoman: ["ottomans"],
  outgo: ["outgoes"],
  outlawry: ["outlawries"],
  ovarium: ["ovaria", "ovariums"],
  ovary: ["ovaries"],
  oversman: ["oversmen"],
  ovococcus: ["ovococci"],
  ovulum: ["ovula"],
  ovum: ["ova", "ovums"],
  owlery: ["owleries"],
  ox: ["oxen"],
  packman: ["packmen"],
  paddy: ["paddies"],
  padrone: ["padroni", "padrones"],
  pagina: ["paginae"],
  pailful: ["pailfuls"],
  palama: ["palamme"],
  palea: ["paleae"],
  paleola: ["paleolae"],
  palestra: ["palestrae", "palestras"],
  palisado: ["palisadoes"],
  pallium: ["pallia", "palliums"],
  palmarium: ["palmaria"],
  palpebra: ["palpebrae"],
  palpus: ["palpi"],
  palsy: ["palsies"],
  paludamentum: ["paladumenta"],
  paludina: ["paludinae", "paludinas"],
  palulus: ["paluli"],
  palus: ["pali"],
  panful: ["panfuls"],
  pansy: ["pansies"],
  pantry: ["pantries"],
  papilla: ["papillae"],
  papilloma: ["papillomata"],
  papula: ["papulae"],
  papule: ["papules"],
  papyrus: ["papyri"],
  parabola: ["parabolas"],
  parabronchium: ["parabronchia"],
  paradactylum: ["paradactyla"],
  parados: ["paradoses"],
  paradox: ["paradoxes"],
  paraglossa: ["paraglossae"],
  paragnathus: ["paragnathi"],
  paramo: ["paramos"],
  paraphagma: ["paraphragmata"],
  paraphysis: ["paraphyses"],
  parapleura: ["parapleurae"],
  parapodium: ["parapodia"],
  parapophysis: ["parapophyses"],
  parapterum: ["paraptera"],
  parascenium: ["parascenia"],
  paraselene: ["paraselenae"],
  parathesis: ["paratheses"],
  parenthesis: ["parentheses"],
  parhelion: ["parhelia"],
  paries: ["parietes"],
  parley: ["parleys"],
  parody: ["parodies"],
  parry: ["parries"],
  particularity: ["particularities"],
  party: ["parties"],
  passman: ["passmen"],
  passus: ["passus", "passuses"],
  pastry: ["pastries"],
  pasty: ["pasties"],
  patagium: ["patagia"],
  patella: ["patellae"],
  patellula: ["patellulae"],
  patera: ["paterae"],
  paterfamilias: ["pateresfamilias"],
  path: ["paths"],
  pathology: ["pathologies"],
  pathopoela: ["pathiopoeilas"],
  patrimony: ["patrimonies"],
  patrolman: ["patrolmen"],
  patty: ["patties"],
  paugie: ["paugies"],
  paxillus: ["paxilli"],
  pea: ["peas", "pease"],
  pease: ["peases", "peasen"],
  peccadillo: ["peccadillos"],
  peccary: ["peccaries"],
  pectus: ["pectora"],
  peculiarity: ["peculiarities"],
  pedary: ["pedaries"],
  pedicellaria: ["pedicellariae"],
  pediculus: ["pediculi"],
  pedipalpus: ["pedipalpi"],
  pelma: ["pelmata"],
  pelta: ["peltae"],
  penalty: ["penalties"],
  pendulum: ["pendulums"],
  penitentiary: ["penitentiaries"],
  penknife: ["penknives"],
  penman: ["penmen"],
  penna: ["pennae"],
  pennatula: ["pennatulae", "pennatulas"],
  penny: ["pennies", "pence"],
  pensionary: ["pensionaries"],
  pentacron: ["pentacra", "pentacrons"],
  pentecosty: ["pentecosties"],
  penwoman: ["penwomen"],
  peony: ["peonies"],
  perfidy: ["perfidies"],
  peri: ["peris"],
  perichaetium: ["perichaetia"],
  periclinium: ["periclinia"],
  periculum: ["pericula"],
  peridium: ["peridia"],
  perigonium: ["perigonia"],
  perigynium: ["perigynia"],
  perihelion: ["perihelia"],
  periodicity: ["periodicities"],
  periostracum: ["periostraca"],
  periphery: ["peripheries"],
  periphrasis: ["periphrases"],
  perisoma: ["perisomata"],
  perispomenon: ["perispomena"],
  peristoma: ["peristomata"],
  perjury: ["perjuries"],
  perplexity: ["perplexities"],
  persona: ["personae"],
  personality: ["personalities"],
  perula: ["perulae"],
  pes: ["pedes"],
  pessary: ["pessaries"],
  pessulus: ["pessuli"],
  petalum: ["petala"],
  peterman: ["petermen"],
  pettifoggery: ["ies"],
  pfennig: ["pfennigs", "pfennige"],
  phacellus: ["phacelli"],
  phalangid: ["phalangides"],
  phalanstery: ["phalansteries"],
  phalanx: ["phalanxes", "phalanges"],
  phallus: ["phalli"],
  pharynx: ["pharynges"],
  phase: ["phases"],
  phasis: ["phases"],
  phenix: ["phenixes"],
  phenomenon: ["phenomena"],
  philosophy: ["philosophies"],
  phiz: ["phizes"],
  pholas: ["pholades"],
  phosphorus: ["phosphori"],
  photo: ["photos"],
  phratry: ["phratries"],
  phylactery: ["phylacteries"],
  phyle: ["phylae"],
  phyllobranchia: ["phyllobranciae"],
  phyllocladium: ["phyllocladia"],
  phyllodium: ["phyllodia"],
  phylon: ["phyla"],
  phylum: ["phyla"],
  phyma: ["phymata"],
  physiognomy: ["physiognomies"],
  physiology: ["physiologies"],
  phyton: ["phytons"],
  phytozoön: ["phytozoa"],
  piazza: ["piazzas"],
  pickaninny: ["pickaninnies"],
  pickpenny: ["pickpennies"],
  picus: ["pici"],
  pieman: ["piemen"],
  piggery: ["piggeries"],
  pignus: ["pignora"],
  pigsty: ["pigsties"],
  pikeman: ["pikeman"],
  pilentum: ["pilenta"],
  pileorhiza: ["pilorhizae"],
  pileus: ["pilei"],
  pilidium: ["pildia"],
  pillery: ["pilleries"],
  pillory: ["pillories"],
  pinax: ["pinaces"],
  pinery: ["pineries"],
  pinna: ["pinnae", "pinnas"],
  pinnula: ["pinnulae"],
  pintado: ["pintados"],
  pipa: ["pipas"],
  pipra: ["pipras"],
  piracy: ["piracies"],
  pistillidium: ["pistillida"],
  pitcherful: ["pitcherfuls"],
  pitman: ["pitmen"],
  pity: ["pities"],
  pixy: ["pixies"],
  placeman: ["placemen"],
  placenta: ["placentae", "placentas"],
  placitum: ["placita"],
  plaga: ["plagae"],
  plagiary: ["plagiaries"],
  plainsman: ["plainsmen"],
  planaria: ["planariae", "planarias"],
  planula: ["planulae"],
  plasmodium: ["plasmodia"],
  plateau: ["plateaux", "plateaus"],
  plateful: ["platefuls"],
  pleasantry: ["pleasantries"],
  plectrum: ["plectra"],
  plenipotentiary: ["plenipotentiaries"],
  plenty: ["plenties"],
  pleopod: ["pleopods", "pleopoda"],
  plesiosaurus: ["plesiosauri"],
  plethron: ["plethra"],
  pleura: ["pleurae", "pleuras"],
  pleurapophysis: ["pleurapophyses"],
  pleurobranchia: ["pleuroeranchiae"],
  pleuron: ["pleura"],
  pleurosteon: ["pleurostea", "pleurosteons"],
  pleurotoma: ["pleurotomae", "pleurotomas"],
  plexus: ["plexus", "plexuses"],
  plowman: ["plowmen"],
  pluma: ["plumae"],
  plumula: ["plumule", "plumulas"],
  plumularia: ["plumularlae", "plumularias"],
  plurality: ["pluralities"],
  pluteus: ["plutei", "pluteuses"],
  pocketful: ["pocketfuls"],
  pocketknife: ["pocketknives"],
  podarthrum: ["podarthra"],
  podetium: ["podetia", "podetiums"],
  podium: ["podia"],
  podobranchia: ["podobranchle"],
  podotheca: ["podothecae"],
  podura: ["podurae", "poduras"],
  pointsman: ["pointsmen"],
  policeman: ["policemen"],
  policy: ["policies"],
  polity: ["polities"],
  pollex: ["pollices"],
  pollinium: ["pollinia"],
  polyacron: ["polyacra", "polyacrons"],
  polyanthus: ["polyanthuses"],
  polyhedron: ["polyhedrons", "polyhedra"],
  polypary: ["polyparies"],
  polyporus: ["polypori"],
  polypus: ["polypuses", "polypi"],
  polyzoarium: ["polyzoaria"],
  polyzoön: ["polyzoa"],
  pomey: ["pomeys"],
  pompelmous: ["pompelmouses"],
  pomposity: ["pomposities"],
  poncho: ["ponchos"],
  ponderosity: ["ponderosities"],
  pons: ["pontes"],
  pontifex: ["pontifices"],
  pony: ["ponies"],
  poppy: ["poppies"],
  popularity: ["popularities"],
  porgy: ["porgies"],
  porphyry: ["porphyries"],
  porta: ["portae"],
  portico: ["porticoes", "porticos"],
  portman: ["portmen"],
  portmanteau: ["portmanteaus"],
  possibility: ["possibilities"],
  postcava: ["postcavae"],
  postcornu: ["postcornua"],
  postfix: ["postfixes"],
  postfurca: ["postfurcae"],
  postman: ["postmen"],
  "postmaster-general": ["postmasters-general"],
  postulatum: ["postulata"],
  postzygapophysis: ["postzygapophyses"],
  posy: ["posies"],
  potato: ["potatoes"],
  potman: ["potmen"],
  pottery: ["potteries"],
  pound: ["pounds", "pound", "pounds"],
  pourparty: ["pourparties"],
  praecornu: ["praecornua"],
  praenomen: ["praenomina"],
  praetexta: ["praetextae", "praetextas"],
  preachman: ["preachmen"],
  preceptory: ["preceptories"],
  prelacy: ["prelacies"],
  preliminary: ["preliminaries"],
  premaxilla: ["premaxillae"],
  premise: ["premises"],
  premium: ["premiums"],
  prepollent: ["prepollices"],
  presbytery: ["presbyteries"],
  prescutum: ["prescuta"],
  preservatory: ["preservatories"],
  presidency: ["presidencies"],
  pressman: ["pressmen"],
  prezygapophysis: ["prezygapophyses"],
  "prima donna": ["prima donnas", "prime", "donne"],
  primary: ["primaries"],
  primitia: ["primitiae"],
  principality: ["principalities"],
  priory: ["priories"],
  privacy: ["privacies"],
  privateersman: ["privateersmen"],
  privity: ["privities"],
  privy: ["privies"],
  prizeman: ["prizemen"],
  probability: ["probabilities"],
  proboscis: ["proboscides"],
  procoelia: ["procoeliae"],
  procuracy: ["procuracies"],
  prodigy: ["prodigies"],
  profundity: ["profundities"],
  proglottis: ["proglottides"],
  programma: ["programmata"],
  prolatum: ["prolata"],
  prolegomenon: ["prolegomena"],
  proletary: ["proletaries"],
  promontory: ["promontories"],
  pronotum: ["pronota"],
  pronucleus: ["pronuclei"],
  proöstracum: ["proöstraca"],
  propagulum: ["propagula"],
  propensity: ["propensities"],
  properispomenon: ["properispomena"],
  property: ["properties"],
  prophecy: ["prophecies"],
  prophragma: ["prophragmata"],
  propodiale: ["propodialia"],
  propodium: ["propodia"],
  proprietary: ["proprietaries"],
  propriety: ["proprieties"],
  propterygium: ["propterygia"],
  propylaeum: ["propylaea"],
  propylon: ["propyla"],
  proscenium: ["proscenia"],
  proscolex: ["proscolices"],
  prosocoelia: ["prosocoelle"],
  prosoma: ["prosomata"],
  prostomium: ["prostomia"],
  prothallium: ["prothallia"],
  prothallus: ["prothalli"],
  prothonotary: ["prothonotaries"],
  "pro thyalosoma": ["prothyalosomata"],
  protiston: ["protista"],
  protonema: ["protonemata"],
  protovertebra: ["protovertebrae"],
  protozoön: ["protozoa"],
  proviso: ["provisos"],
  proxy: ["proxies"],
  prudery: ["pruderies"],
  prytanis: ["prytanes"],
  psalterium: ["psalteria"],
  psaltery: ["psalteries"],
  pseudobranchia: ["pseudobranchiae"],
  pseudofilaria: ["pseudofilariae"],
  pseudohalter: ["pseudohalteres"],
  pseudonavicella: ["pseudonavicullae"],
  pseudonavicula: ["pseudonaviculae"],
  pseudopodium: ["pseudopodia"],
  pseudopupa: ["pseudopupae", "pseudopupas"],
  pseudostella: ["pseudostellae"],
  pseudostoma: ["pseudostomata"],
  pseudotinea: ["pseudotineae"],
  pseudovary: ["pseudovaries"],
  pseudovum: ["pseudova"],
  psychology: ["psychologies"],
  psylla: ["psyllae"],
  pterostigma: ["pterostigmata"],
  pterygium: ["pterygiums", "pterygia"],
  pterygopodium: ["pterygopodia"],
  pteryla: ["pterylae"],
  puerility: ["puerilities"],
  pulley: ["pulleys"],
  pullus: ["pulli"],
  pulvillus: ["pulvilli"],
  pulvinulus: ["pulvinuli"],
  punctilio: ["punctilios"],
  pupa: ["pupae", "pupas"],
  puppy: ["puppies"],
  purple: ["purples"],
  purseful: ["pursefuls"],
  pycnidium: ["pycnidia"],
  pygidium: ["pygidia"],
  pygmy: ["pygmies"],
  pyla: ["pylae", "pylas"],
  pylangium: ["pylangia"],
  pylorus: ["pylori"],
  pyramidion: ["pyramidia"],
  pyramis: ["pyramides"],
  pyrena: ["pyrenae"],
  pyrexia: ["pyrexiae"],
  pyrite: ["pyrites"],
  pyxidium: ["pyxidia"],
  quackery: ["quackeries"],
  quadra: ["quadrae"],
  quadrans: ["quadrantes"],
  quadratrix: ["quadratrixes", "quadratrices"],
  quadriga: ["quadrigae"],
  quality: ["qualities"],
  quandary: ["quandaries"],
  quantity: ["quantities"],
  quantum: ["quanta"],
  quarry: ["quarries"],
  "quarry-man": ["quarrymen"],
  quarterly: ["quarterlies"],
  quarterstaff: ["quarterstaves"],
  quarto: ["quartos"],
  query: ["queries"],
  questman: ["questmen"],
  quiddity: ["quiddities"],
  quindecemvir: ["quindecemvirs", "quindecemviri"],
  quinquevir: ["quinquevirs", "quinqueviri"],
  quipu: ["quipus"],
  rachis: ["rachises", "rachides"],
  radiale: ["radialia"],
  radius: ["radii", "radiuses"],
  radix: ["radices", "radixes"],
  radula: ["radulae"],
  raftsman: ["raftsmen"],
  ragman: ["ragmen"],
  rally: ["rallies"],
  ramulus: ["ramuli"],
  ramus: ["rami"],
  ranchero: ["rancheros"],
  ranchman: ["ranchmen"],
  rancho: ["ranchos"],
  ranunculus: ["ranunculuses", "ranunculi"],
  rarity: ["rarities"],
  rascality: ["rascalities"],
  rationality: ["rationalities"],
  reactionary: ["reactionaries"],
  reality: ["realities"],
  rebus: ["rebuses"],
  receptaculum: ["receptacula"],
  recipe: ["recipes"],
  rectory: ["rectories"],
  rectrix: ["rectrices"],
  rectus: ["recti"],
  redia: ["rediae", "redias"],
  refectory: ["refectories"],
  refinery: ["refineries"],
  reformatory: ["reformatories"],
  refrigeratory: ["refrigeratories"],
  regatta: ["regattas"],
  regency: ["regencies"],
  regulus: ["reguluses", "reguli"],
  rei: ["reis"],
  reliquary: ["reliquaries"],
  "remainder-man": ["remainder-men"],
  remedy: ["remedies"],
  rendezvous: ["rendezvouses"],
  reply: ["replies"],
  res: ["res"],
  responsibility: ["responsibilities"],
  responsory: ["responsories"],
  reticulum: ["reticula"],
  retinaculum: ["retinacula"],
  retineum: ["retinea"],
  retinophora: ["retiniphorae"],
  retinula: ["retinulae"],
  reverie: ["reveries"],
  rex: ["reges"],
  rhachis: ["rhachises", "rhachides"],
  rhamphotheca: ["rhamphothecae"],
  rhapsody: ["rhapsodies"],
  rhinencephalon: ["rhinencephala"],
  rhinotheca: ["rhinothecae"],
  rhizoma: ["rhizomata"],
  rhonchus: ["rhonchi"],
  rhopalium: ["rhopalia"],
  ribbonman: ["ribbonmen"],
  rifleman: ["rifleman"],
  rima: ["rimae"],
  ringman: ["ringmen"],
  rivalry: ["rivalries"],
  robbery: ["robberies"],
  roberdsman: ["robertdsmen"],
  rodsman: ["rodsmen"],
  rookery: ["rookeries"],
  roomful: ["roomfuls"],
  rosary: ["rosaries"],
  rostellum: ["rostella"],
  rostrulum: ["rostrula"],
  rostrum: ["rostra", "rostrums"],
  rouleau: ["rouleaux", "rouleaus"],
  roundsman: ["roundsmen"],
  rowdy: ["rowdies"],
  royalty: ["royalties"],
  ruby: ["rubies"],
  ruga: ["rugae"],
  rummy: ["rummies"],
  rurality: ["ties"],
  sacculus: ["sacculi"],
  saccus: ["sacci"],
  sacellum: ["sacella"],
  sackful: ["sackfuls"],
  sacramentary: ["sacramentaries"],
  sacrarium: ["sacraria"],
  sacristy: ["sacristies"],
  saga: ["sagas"],
  sagum: ["saga"],
  salary: ["salaries"],
  salesman: ["salesmen"],
  saleswoman: ["saleswomen"],
  salix: ["salices"],
  sally: ["sallies"],
  salmon: ["salmons", "salmon"],
  salpa: ["salpae", "salpas"],
  salvo: ["salvos"],
  sanctity: ["sanctities"],
  sanctuary: ["sanctuaries"],
  sarcobasis: ["sarcobases"],
  sarcoma: ["sarcomata", "sarcomas"],
  sarcophagus: ["sarcophagi", "sarcophaguses"],
  sarcoseptum: ["sarcosepta"],
  satrapy: ["satrapies"],
  saury: ["sauries"],
  savant: ["savants"],
  saxicava: ["saxicavas", "saxicavae"],
  scala: ["scalae"],
  scamillus: ["scamilli"],
  scapula: ["scapulae", "scapuolas"],
  scarf: ["scarfs", "scarves"],
  sceneman: ["scenemen"],
  schema: ["schemata", "schemas"],
  scholium: ["scholia", "scholiums"],
  schoolman: ["schoolmen"],
  scirrhus: ["scirrhi", "scirrhuses"],
  sclerotium: ["sclerotia"],
  scolex: ["scoleces"],
  scopula: ["scopulas", "scopulae"],
  scoria: ["scorlae"],
  scorpio: ["scorpiones"],
  scotchman: ["scotchmen"],
  scriptorium: ["scriptoria"],
  scrobicula: ["scrobiculae"],
  scudo: ["scudi"],
  scullery: ["sculleries"],
  scutella: ["scutelle"],
  scutellum: ["scutella"],
  scutum: ["scuta"],
  scypha: ["scyphae"],
  scyphistoma: ["scyphistomata", "scyphistomae"],
  scyphus: ["scyphi"],
  scytheman: ["scythemen"],
  seaman: ["seamen"],
  seawife: ["seawives"],
  secondary: ["secondaries"],
  secrecy: ["secrecies"],
  secretary: ["secretaries"],
  sectary: ["sectaries"],
  security: ["securities"],
  seed: ["seed", "seeds"],
  seedsman: ["seedsmen"],
  seigniory: ["seigniories"],
  selectman: ["selectmen"],
  self: ["selves"],
  semen: ["semina"],
  seminary: ["seminaries"],
  semita: ["semitae"],
  sensibility: ["sensibilities"],
  sensorium: ["sensoriums", "sensoria"],
  sensery: ["sensories"],
  sentry: ["sentires"],
  separatrix: ["separatrices", "separatrixes"],
  sepia: ["sepias", "sepiae"],
  septarium: ["septaria"],
  septemvir: ["septemvirs", "septemviri"],
  septulum: ["septula"],
  septum: ["septa"],
  sequela: ["sequelae"],
  sequestrum: ["sequestra"],
  seraph: ["seraphs", "seraphim"],
  sergeancy: ["sergeancies"],
  serpula: ["serpulae", "serpulas"],
  seta: ["setae"],
  setula: ["setulae"],
  seventy: ["seventies"],
  severality: ["severalities"],
  severity: ["severities"],
  sextary: ["sextaries"],
  sexto: ["sextos"],
  sextodecimo: ["sextodecimos"],
  shanny: ["shannies"],
  shanty: ["shanties"],
  sheaf: ["sheaves"],
  shearman: ["shearmen"],
  sheetful: ["sheetfuls"],
  shelf: ["shelves"],
  shepherdia: ["shepherdias"],
  shindy: ["shindies"],
  shipful: ["shipfuls"],
  shipman: ["shipmen"],
  shoe: ["shoes", "shoon"],
  shopman: ["shopmen"],
  shopwoman: ["shopwomen"],
  shot: ["shot", "shots"],
  shovelful: ["shovelfuls"],
  showman: ["showmen"],
  shrubbery: ["shrubberies"],
  sickleman: ["sicklemen"],
  sidesman: ["sidesmen"],
  sightsman: ["sightsmen"],
  sigillum: ["sigilla"],
  sigma: ["sigmas"],
  signalman: ["signalmen"],
  signatory: ["signatories"],
  siliqua: ["siliquae"],
  silkman: ["silkmen"],
  silva: ["silvas", "silvae"],
  similarity: ["similarities"],
  simile: ["similes"],
  simulacrum: ["simulacra"],
  singularity: ["singularities"],
  sinuosity: ["sinuosities"],
  sinus: ["sinus", "sinuses"],
  siphonium: ["siphonia"],
  sirocco: ["siroccos"],
  "sister-in-law": ["sisters-in-law"],
  sixpence: ["sixpences"],
  sixty: ["sixties"],
  skerry: ["skerries"],
  skinful: ["skinfuls"],
  sky: ["skies"],
  slaughterman: ["slaughtermen"],
  slav: ["slavs"],
  slavery: ["slaveries"],
  smithery: ["smitheries"],
  snuggery: ["snuggeries"],
  society: ["societies"],
  socman: ["socmen"],
  sodality: ["sodalities"],
  sofa: ["sofas"],
  sofi: ["sofis"],
  solarium: ["solaria"],
  soldo: ["soldi"],
  solemnity: ["solemnities"],
  soliloquy: ["soliloquies"],
  solo: ["solos", "soli"],
  "son-in-law": ["sons-in-law"],
  sophi: ["sophis"],
  soprano: ["sopranos", "soprani"],
  sorcery: ["sorceries"],
  soredium: ["soredia"],
  sors: ["sortes"],
  sorus: ["sori"],
  sou: ["sous"],
  sovereignty: ["sovereignties"],
  spadeful: ["spadefuls"],
  spadix: ["spadices", "spadixes"],
  spado: ["spadones"],
  spatha: ["spathae"],
  spearman: ["spearmen"],
  speciality: ["specialities"],
  specialty: ["specialties"],
  speciosity: ["speciocities"],
  spectrum: ["spectra"],
  speculum: ["specula", "speculum"],
  spermatheca: ["spermathecae"],
  spermatium: ["spermatia"],
  spermatoön: ["spermatoa"],
  spermatozoön: ["spermatozoa"],
  spermidium: ["spermidia"],
  sphaeridium: ["sphaeridia"],
  spica: ["spicae"],
  spicula: ["spiculae"],
  spiculum: ["spicula"],
  spinney: ["spinneys"],
  spinny: ["spinnies"],
  spirituality: ["spiritualities"],
  spitful: ["spitfuls"],
  splanchnapophysis: ["splanchnapophyses"],
  splayfoot: ["splayfeet"],
  splaymouth: ["splaymouths"],
  splenculus: ["splenculi"],
  spoilsman: ["spoilsmen"],
  spokesman: ["spokesmen"],
  spontaneity: ["spontaneities"],
  spooney: ["spooneye"],
  spoonful: ["spoonfuls"],
  sporangium: ["sporangia"],
  sporidium: ["sporidia"],
  sportsman: ["sportsmen"],
  sportula: ["sportulae"],
  sputum: ["sputa"],
  spy: ["spies"],
  squacco: ["squaccos"],
  squama: ["squamae"],
  squamella: ["squamellae"],
  squamipen: ["squamipennes"],
  squamula: ["squamulae"],
  squilla: ["squillas", "squillae"],
  stadium: ["stadia"],
  staff: ["staves", "staffs", "staffs"],
  staffman: ["staffmen"],
  stagecoachman: ["stagecoachmen"],
  stalactite: ["stalactites"],
  stallman: ["stallmen"],
  stamen: ["stamens", "stamina"],
  staminodium: ["staminodia"],
  stannary: ["stannaries"],
  stanza: ["stanzas"],
  stasimon: ["stasmia"],
  statesman: ["statesmen"],
  stateswoman: ["stateswomen"],
  stationary: ["stationaries"],
  statuary: ["statuaries"],
  steersman: ["steersmen"],
  stela: ["stelae"],
  stemma: ["stemmata"],
  sternebra: ["sternebrae"],
  sternum: ["sterna", "sternums"],
  stichidium: ["stichida"],
  stickful: ["stickfuls"],
  stigma: ["stigmas", "stigmata"],
  stiletto: ["stilettos"],
  stillatory: ["stillaries"],
  stimulus: ["stimuli"],
  stipendiary: ["stipendiaries"],
  stipes: ["stipites"],
  stipula: ["stipulas", "stipulae"],
  stirps: ["stirpes"],
  stockman: ["stockmen"],
  stola: ["stolae"],
  stoma: ["stomata"],
  story: ["stories"],
  stragulum: ["stragula"],
  strait: ["straits"],
  strappado: ["strappadoes"],
  strategus: ["strategi"],
  stratum: ["stratums", "strata"],
  streptococcus: ["streptococci"],
  stria: ["striae"],
  strobila: ["strobilae"],
  strokesman: ["strokesman"],
  stroma: ["stromata"],
  strophe: ["strophes"],
  struthio: ["struthiones"],
  stucco: ["stuccoes", "stuccos"],
  studio: ["studios"],
  study: ["studies"],
  sty: ["sties"],
  stylopodium: ["stylopodia"],
  subcontrary: ["subcontraries"],
  subgenus: ["subgenera"],
  subindex: ["subindices"],
  sublimity: ["sublimities"],
  sublingua: ["sublinguae"],
  submentum: ["submenta"],
  subsellium: ["subsellia"],
  subsidiary: ["subsidiaries"],
  subsidy: ["subsidies"],
  substratum: ["substrata"],
  subtlety: ["subtleties"],
  subtreasury: ["subtreasuries"],
  subvariety: ["subvarieties"],
  succedaneum: ["succedanea"],
  succuba: ["succubae"],
  succubus: ["succubi"],
  succus: ["succi"],
  sudatorium: ["sudatoria"],
  sudatory: ["sudatories"],
  sulcus: ["sulci"],
  sulky: ["sulkies"],
  sully: ["sullies"],
  summary: ["summaries"],
  summerhouse: ["summerhouses"],
  summons: ["summonses"],
  sundryman: ["sundrymen"],
  sunglass: ["sunglasses"],
  superfluity: ["superfluities"],
  superintendency: ["superintendencies"],
  supernumerary: ["supernumeraries"],
  superstatum: ["superstrata"],
  suppletory: ["suppletories"],
  supply: ["supplies"],
  suppository: ["suppositories"],
  supramaxilla: ["supramaxillae"],
  surety: ["sureties"],
  surfman: ["surmen"],
  surveillant: ["surveillants"],
  susceptibility: ["susceptibilities"],
  suspensorium: ["suspensoria"],
  suster: ["susters", "sustres", "sustren"],
  sutra: ["sutras"],
  switchman: ["switchmen"],
  swordman: ["swordmen"],
  swordsman: ["swordsmen"],
  syllabarium: ["syllabaria"],
  syllabus: ["syllabuses", "syllabi"],
  sylva: ["sylvae"],
  sympathy: ["sympathies"],
  symphony: ["symphonies"],
  symphysis: ["symphyses"],
  sympodium: ["sympodia"],
  symposium: ["symposia"],
  synangium: ["synangia"],
  synapticula: ["synapticulae"],
  synarthrosis: ["synarthroses"],
  syncarpium: ["syncarpia"],
  synchondrosis: ["synchondroses"],
  synclinorium: ["synclinoria"],
  syncytium: ["syncitia"],
  syndesmosis: ["syndesmoses"],
  synneorosis: ["synneuroses"],
  synonym: ["synonyms"],
  synopsis: ["synopses"],
  synosteosis: ["synosteoses"],
  synthesis: ["syntheses"],
  syrinx: ["syringes"],
  syrtis: ["syrtes"],
  syzygy: ["syzygies"],
  tableau: ["tableaux"],
  "tableau vivant": ["tableaux vivants"],
  "table d’hôte": ["tables d’hôte"],
  tableman: ["tablemen"],
  tablespoonful: ["tablespoonfuls"],
  tabula: ["tabulae"],
  tachina: ["tachinae"],
  tacksman: ["tacksmen"],
  taenia: ["taeniae"],
  taenidium: ["taenidia"],
  taeniola: ["taeniolae"],
  talesman: ["talesmen"],
  talisman: ["talismans"],
  tally: ["tallies"],
  tallyman: ["tallymen"],
  talma: ["talmas"],
  talus: ["tali"],
  tammy: ["tammies"],
  tannery: ["tanneries"],
  tapestry: ["tapestries"],
  tapeti: ["tapetis"],
  tarantula: ["tarantulas", "tarantulae"],
  targum: ["targums", "targumim"],
  tarsale: ["tarsalia"],
  tarsometatarsus: ["tarsometatarsi"],
  tarsus: ["tarsi"],
  tattoo: ["tattoos"],
  tavernman: ["tavernmen"],
  tawdry: ["tawdries"],
  teacupful: ["teacupfuls"],
  teaspoonful: ["teaspoonfuls"],
  technicality: ["technicalities"],
  tegmen: ["tegmina"],
  tegmentum: ["tegmenta"],
  tegula: ["tegulae"],
  telotrocha: ["telotrochae"],
  telson: ["telsons"],
  temporality: ["temporalities"],
  tenaculum: ["tenacula", "tenaculums"],
  tenancy: ["tenacies"],
  tendency: ["tendencies"],
  tentaculum: ["tentacula"],
  tentful: ["tentfuls"],
  tenuiroster: ["tenuirosters"],
  tenuis: ["tenues"],
  teocalli: ["teocallis"],
  teraph: ["teraphs"],
  terebra: ["terebras", "terebrae"],
  terebratula: ["terebratulae"],
  teredo: ["teredos", "teredines"],
  tergum: ["terga"],
  termes: ["termities"],
  terminus: ["termini"],
  termite: ["termites"],
  ternary: ["ternaries"],
  territory: ["territories"],
  tertiary: ["tertiaries"],
  tessera: ["tesserae"],
  test: ["tests", "testae"],
  testimony: ["testimonies"],
  testis: ["testes"],
  testudo: ["testudines"],
  "tête-de-pont": ["têtes-de-pont"],
  tetrarchy: ["tetrarchies"],
  teuton: ["teutons", "teutones"],
  textman: ["textmen"],
  thalamus: ["thalami"],
  thallus: ["thalli"],
  thank: ["thanks"],
  theca: ["thecae"],
  theology: ["theologies"],
  theophany: ["theophanies"],
  theory: ["theories"],
  thesaurus: ["thesauri"],
  thesis: ["theses"],
  thief: ["thieves"],
  thimbleful: ["thimblefuls"],
  thirty: ["thirties"],
  this: ["these"],
  thrombus: ["thrombi"],
  thyrsus: ["thyrsi"],
  tibia: ["tibiae"],
  tibiale: ["tibialia"],
  tibiotarsus: ["tibiotarsi"],
  tidesman: ["tidesmen"],
  tidy: ["tidies"],
  tie: ["ties"],
  tilbury: ["tilburies"],
  tilery: ["tileries"],
  tillman: ["tillmen"],
  timberman: ["timbermen"],
  time: ["times"],
  timpano: ["timpani"],
  tineman: ["tinemen"],
  tinman: ["tinmen"],
  tipstaff: ["tipstaff"],
  tiptoe: ["tiptoes"],
  tipula: ["tipulae", "tipulas"],
  "tire-woman": ["tire-women"],
  tithingman: ["tithingmen"],
  titmouse: ["titmice"],
  titulary: ["titularies"],
  toady: ["toadies"],
  toftman: ["toftmen"],
  toga: ["togas", "togae"],
  tollhouse: ["tollhouses"],
  tollman: ["tollmen"],
  tomato: ["tomatoes"],
  tomentum: ["tomenta"],
  tomium: ["tomia"],
  tony: ["tonies"],
  tooth: ["teeth"],
  toparchy: ["toparchies"],
  tophus: ["tophi"],
  topman: ["topmem"],
  topsman: ["topsmen"],
  tornado: ["tornadoes"],
  tornaria: ["tornariae"],
  torpedo: ["torpedoes"],
  torso: ["torsos", "torsi"],
  torteau: ["torteaus"],
  torula: ["torulae"],
  torus: ["tori"],
  tory: ["tories"],
  townsman: ["townsmen"],
  trabea: ["trabeae"],
  trabecula: ["trabeculae"],
  "tracer/y": ["traceries"],
  trachea: ["tracheae"],
  tracheobranchia: ["tracheobranchlae"],
  tradesman: ["tradesmen"],
  tradeswoman: ["tradeswomen"],
  traditionary: ["traditionaries"],
  tragedy: ["tragedies"],
  trainband: ["trainbands"],
  trajectory: ["trajectories"],
  transparency: ["transparencies"],
  trapezium: ["trapeziums", "trapezia"],
  travesty: ["travesties"],
  trawlerman: ["trawlermen"],
  tray: ["trays"],
  trayful: ["trayfuls"],
  treasury: ["treasuries"],
  treaty: ["treaties"],
  treeful: ["treefuls"],
  "trencher-man": ["trencher-men"],
  triarchy: ["triarchies"],
  tributary: ["tributaries"],
  trica: ["tricae"],
  trichina: ["trichinae"],
  triclinium: ["triclinia"],
  trierarchy: ["trierarchises"],
  trigonometry: ["trigonomettries"],
  trilithon: ["trilithons"],
  tripeman: ["tripemen"],
  tripos: ["triposes"],
  triquetrum: ["triquetra"],
  tritovum: ["tritova"],
  triumvir: ["triumviri", "triumvirs"],
  triungulus: ["triunguli"],
  triviality: ["trivialities"],
  trochilus: ["trochili"],
  trochiscus: ["trochisci"],
  trochus: ["trochi"],
  troilus: ["troili", "troiluses"],
  trophy: ["trophies"],
  "trou-de-loup": ["trous-de-loup"],
  trowelful: ["trowelfuls"],
  truckman: ["truckmen"],
  trunkful: ["trunkfuls"],
  truth: ["truths"],
  tuberculum: ["tubercula"],
  tuberosity: ["tuberosities"],
  tubful: ["tubfuls"],
  tubman: ["tubmen"],
  tumblerful: ["tumblerfuls"],
  tumulus: ["tumuli"],
  tunicary: ["tunicaries"],
  tunny: ["tunnies"],
  tupman: ["tupmen"],
  turbary: ["turbaries"],
  turcoman: ["turcomans"],
  tureenful: ["tureenfuls"],
  turf: ["turfs", "turves"],
  turfman: ["turfmen"],
  turio: ["turiones"],
  tyrkey: ["turkeys"],
  turko: ["turkos"],
  turkoman: ["turkomans"],
  turnkey: ["turnkeys"],
  "turn-out": ["turn-outs"],
  "tut-workman": ["tut-workmen"],
  twenty: ["twenties"],
  tylarus: ["tylari"],
  tylosis: ["tyloses"],
  tympano: ["tympani"],
  tympanum: ["tympanums", "tympana"],
  tyro: ["tyros"],
  ubiquitary: ["ubiquitaries"],
  ulnare: ["ulnaria"],
  ultimatum: ["ultimatums", "ultimata"],
  umbo: ["umbones", "umbos"],
  umbra: ["umbrae"],
  uncertainty: ["uncertainties"],
  uncia: ["unciae"],
  uncinus: ["uncini"],
  uncus: ["unci"],
  understratum: ["understrata", "understratums"],
  unguis: ["ungues"],
  ungula: ["ungulae"],
  unity: ["unities"],
  universality: ["universalties"],
  university: ["universities"],
  unpleasantry: ["unpleasantries"],
  urceolus: ["urceoli"],
  urnful: ["urnfuls"],
  urosteon: ["urostea", "urosteons"],
  vacancy: ["vacancies"],
  vacuum: ["vacuums", "vacua"],
  vagary: ["vagaries"],
  vagina: ["vaginae"],
  valedictory: ["valedictories"],
  valency: ["valencies"],
  vallecula: ["valleculae"],
  valley: ["valleys"],
  vallum: ["valla", "vallums"],
  valvula: ["valvulae"],
  vanity: ["vanities"],
  variety: ["varieties"],
  varix: ["varices"],
  vas: ["vasa"],
  vascularity: ["vascularities"],
  vasculum: ["vascula"],
  vatful: ["vatfuls"],
  velarium: ["velaria"],
  velocity: ["velocities"],
  velum: ["vela"],
  vena: ["venae"],
  ventriculus: ["ventriculi"],
  verbosity: ["verbosities"],
  verity: ["verities"],
  vertebra: ["vertebrae"],
  vertex: ["vertexes", "vertices"],
  vertigo: ["vertigoes", "vertigines"],
  vesicula: ["vesiculae"],
  vespillo: ["vespilloes"],
  vesselful: ["vesselfuls"],
  vestibulum: ["vestibula"],
  vestry: ["vestries"],
  vestryman: ["vestrymen"],
  veto: ["vetoes"],
  vettura: ["vetture"],
  vetturino: ["vetturini"],
  vexillum: ["vexilla"],
  vibraculum: ["vibracula"],
  vibrio: ["vibrios", "vibriones"],
  vibrissa: ["vibrissae"],
  viceman: ["vicemen"],
  victory: ["victories"],
  "vigesimo-quarto": ["vigesimoquartos"],
  villa: ["villas"],
  villainy: ["villainies"],
  villanella: ["villanelle"],
  villus: ["villi"],
  vinculum: ["vincula", "vinculums"],
  virago: ["viragoes"],
  virtuoso: ["virtuosos", "virtuosi"],
  viscus: ["viscera"],
  visionary: ["visionaries"],
  vista: ["vistas"],
  vitta: ["vittae"],
  vivarium: ["vivariums", "vivaria"],
  vivary: ["vivaries"],
  vocabulary: ["vocabularies"],
  volcano: ["volcanoes"],
  volkslied: ["volkslieder"],
  volley: ["volleys"],
  volta: ["volte"],
  voluntary: ["voluntaries"],
  voluptuary: ["voluptuaries"],
  voluta: ["volutas", "volutae"],
  vomitory: ["vomitories"],
  vortex: ["vortexes", "vortices"],
  vorticella: ["vorticellas", "vorticellae"],
  votary: ["votaries"],
  wady: ["wadies"],
  waggery: ["waggeries"],
  wagonful: ["wagonfuls"],
  wallaby: ["wallabies"],
  wardsman: ["wardsmen"],
  warehouse: ["warehouses"],
  warehouseman: ["warehousemen"],
  warranty: ["warranties"],
  washerman: ["washermen"],
  washerwoman: ["washerwomen"],
  watchhouse: ["watchhouses"],
  watchman: ["watchmen"],
  waterman: ["watermen"],
  wealsman: ["wealsmen"],
  webfoot: ["webfeet"],
  weekly: ["weeklies"],
  weighhouse: ["weigh-houses"],
  welshman: ["welshmen"],
  werewolf: ["werewolves"],
  whaleman: ["whalemen"],
  wharf: ["wharfs", "wharves"],
  wheelman: ["wheelmen"],
  wherry: ["wherries"],
  whimsey: ["whimseys", "whimsies"],
  whinny: ["whinnies"],
  whiskey: ["whiskeys", "whiskies"],
  wife: ["wives"],
  wineglassful: ["wineglassfuls"],
  witchery: ["witcheries"],
  withy: ["withies"],
  wolf: ["wolves"],
  woman: ["women"],
  woodman: ["woodmen"],
  woodsman: ["woodsmen"],
  woolman: ["woolmen"],
  workhouse: ["workhouses"],
  workingman: ["workingmen"],
  workman: ["workmen"],
  workwoman: ["workwomen"],
  worry: ["worries"],
  worthy: ["worthies"],
  wreath: ["wreaths"],
  wringstaff: ["wringstaves"],
  wye: ["wyes"],
  xenium: ["xenia"],
  xiphiplastron: ["xiphiplastra"],
  xiphisternum: ["xiphisterna"],
  y: ["y’s", "ys"],
  yachtsman: ["yachtsmen"],
  yardful: ["yardfuls"],
  yë: ["yën"],
  yeoman: ["yeomen"],
  yourself: ["yourselves"],
  youth: ["youths", "youth"],
  zambo: ["zambos"],
  zany: ["zanies"],
  zero: ["zeros", "zeroes"],
  zingaro: ["zingari"],
  zona: ["zonae"],
  zoöcytium: ["zoöcytia"],
  zoödendrium: ["zoödendria"],
  zooecium: ["zooecia"],
  zoölogy: ["zoölogies"],
  zoön: ["zoa"],
  zoösporangium: ["zoosporangia"],
  zygantrum: ["zygantra"],
  zygapophysis: ["zygapophyses"],
};

var names = {
  cosmos: new Names(""),
};
