from abc import abstractmethod, ABC
from py4j.java_gateway import JavaGateway
import subprocess
import time

class CTestGeneratorStrategy(ABC):
    '''
    This abstract method describes the architecture of a CTestGenerator Strategy.
    It tries to be generic enough to avoid constrains on the generation of a ctest with any desired strategy.

    '''

    def __init__(self,texts,evaluator,target,java):
        '''
        The initialization does assign the class variables but also sets up the connection to the java
        difficultyPrediction system. The jar of the system has to be placed in the same folder as this class.
        :param texts: array of texts to generate a test from
        :param evaluator: evaluator for the test
        :param target: target difficulty level
        '''
        self.texts = texts
        self.evaluator = evaluator
        self.targetLevel = target

        #proc = subprocess.Popen(['java', '-jar', 'GenerationSystem/DifficultyPrediction.jar'], stdout=subprocess.PIPE)
        #line = proc.stdout.readline()
        #while not line.rstrip().__contains__(b"Initialization took"):
        #    line = proc.stdout.readline()
        time.sleep(2)
        #self.gateway = JavaGateway(start_callback_server=True)
        #self.java = self.gateway.entry_point
        self.java=java



    def predict(self,ctest):
        '''
        This method predicts the difficulty of a c-test utilizing the difficulty prediction java package.
        The java package is started in the init method and the communication between java and python is
        done with the py4j framework.

        :param ctest: ctest in the tc format of DKPro
        :return: float array of prediction values
        '''

        return self.result_string_to_tuple(self.java.predict(ctest))


    def result_string_to_tuple(self,resultString):
        """
        This method converts the big string submitted by the java difficulty prediction system into the
        its constitutes. The string contains the prediction values, feature values and feature names
        :param resultString: String containg the prediction result and the feature annotation
        :return: Seperated tuple containing the prediction scores, feature names and feature values
        """
        features, predictions = resultString.split('Predictions:')
        featureNames, featureValues = features.split('@data')

        featureNames=[s_inner.split(',') for s_inner in featureNames.splitlines()[2:-1]]
        featureValues = [s_inner.split(',') for s_inner in featureValues.splitlines()[1:]]
        for n,line in enumerate(featureValues):
            featureValues[n] = [float(v) for v in line[:-1]]#Cut off empty feature outcome
        predictions=[s_inner.split(',') for s_inner in predictions.splitlines()[1:]]
        predictions=[float(v) for v in predictions[0]]
        return (predictions,featureNames,featureValues)



    @abstractmethod
    def generate_ctest(self, txt,):
        '''
        This method generates a ctest in the DKPro TC Format according to the desired strategy.

        :param txt: Text to produce ctest from
        :return:None if text is not appropriate for the test generation or ctest in TC Format if successful
        '''
        pass


    @abstractmethod
    def retrieve_ctest(self):
        '''
        This method produces a c-test with the desired difficulty level. If the exact difficulty level can not be hit,
        the closes c-test gets retrieved.
        :return:ctest as tuple of tc-Format, Human readable and the predicted difficulty.
        '''
        pass

    def shutdown(self):
        '''
        Turns off the connection to the java system
        Importend because the py4j framework uses a localhost port
        :return: void
        '''
        self.gateway.shutdown()



