import functools
import inspect
from argparse import ArgumentParser


def with_argparse(func):
    @functools.wraps(func)
    def inner(*inner_args, **inner_kwargs):
        info = inspect.getfullargspec(func)
        args = ArgumentParser()
        mappings = {}

        def add_argument(x, typ, default, required):
            nonlocal mappings
            if typ == bool:
                default = default if default is not None else False
                if default:
                    mappings["no_" + x] = x
                    args.add_argument(f"--no_" + x, action="store_false", default=default)
                else:
                    args.add_argument(f"--" + x, action="store_true", default=default)
            elif typ == list[str]:
                args.add_argument("--" + x, type=str, default=default, required=required, nargs="+")
            else:
                args.add_argument("--" + x, type=typ, default=default, required=required)

        defaults = (tuple(None for _ in range(len(info.args) - len(info.defaults or []))) +
                    (info.defaults or (None for _ in range(len(info.args)))))
        for i, arg in enumerate(info.args or []):
            if i < len(inner_args):
                continue
            assert arg in info.annotations
            arg_type = info.annotations[arg]
            arg_default = defaults[i]
            arg_required = info.defaults is None or len(info.defaults) <= (len(info.args) - (i + 1))
            add_argument(arg, arg_type, arg_default, arg_required)
        for i, arg in enumerate(info.kwonlyargs or []):
            if arg in inner_kwargs:
                continue
            assert arg in info.annotations
            arg_type = info.annotations[arg]
            arg_default = info.kwonlydefaults.get(arg, None) if info.kwonlydefaults is not None else None
            arg_required = info.kwonlydefaults is None or arg not in info.kwonlydefaults
            add_argument(arg, arg_type, arg_default, arg_required)
        args = args.parse_args()
        args_dict = dict()
        for k, v in args.__dict__.items():
            if k in mappings:
                args_dict[mappings[k]] = v
            else:
                args_dict[k] = v

        return func(*inner_args, **args_dict, **inner_kwargs)
    return inner
