RULES_FILE = 'data/compiled-rules.txt'
STATES_FILE = 'data/states.txt'
LITERALS_FILE = 'data/literals.txt'


def make_states(state_strings):
    states = []
    for string in state_strings:
        index = int(string)
        states.append(((index >> 4), (index & 0xf)))
    return tuple(states)


def make_equation(line, literals):
    num_string, *var_strings = line.strip().split(' ')
    var_count = int(num_string)

    assert(var_count == len(var_strings))

    equation = []
    for var_string in var_strings:
        var = int(var_string)
        if var >= 0:
            equation.append(((var >> 4), (var & 0xf)))
        elif var == -16:
            equation.append((-1, 0))
        elif var <= -32:
            equation.append(literals[-var - 32])
        else:
            equation.append(var)

    return tuple(equation)


def load_tokens(filename):
    tokens = []
    token2index = {}
    for line in open(filename).readlines():
        token = line.strip()
        token2index[token] = len(tokens)
        tokens.append(token)

    return tokens, token2index


def load_rules(rules_file=RULES_FILE):
    states = []
    literals = []
    state2index = {}
    literal2index = {}
    rules = {}

    states, state2index = load_tokens(STATES_FILE)
    literals, literal2index = load_tokens(LITERALS_FILE)

    rule_index = 0
    for rule_string in open(rules_file).read().split('\n\n'):
        rule_string = rule_string.strip()
        if rule_string == '':
            continue
        rule_lines = rule_string.split('\n')
        if rule_lines[0][0].isdigit():  # skip total number
            rule_lines = rule_lines[1:]

        node_label, *state_strings = rule_lines[0].strip().split(' ')
        in_states_count = int(state_strings[0])
        out_states_count = int(state_strings[in_states_count + 1])

        assert(in_states_count + out_states_count + 2 == len(state_strings))

        rule = node_label, \
            make_states(state_strings[1:in_states_count + 1]), \
            make_states(state_strings[in_states_count + 2:])

        eqs_count = int(rule_lines[1])

        assert(eqs_count == len(rule_lines) - 2)

        eqs = tuple(make_equation(line, literals)
                    for line in rule_lines[2:])

        rules.setdefault(rule, {})[eqs] = rule_index
        rule_index += 1

    return rules, states, state2index, literals, literal2index
