#!/usr/bin/env python
##
##  Copyright (c) 2004 onward  Yusuke Shinyama <yusuke at cs dot nyu dot edu>
##
##  Permission is hereby granted, free of charge, to any person
##  obtaining a copy of this software and associated documentation
##  files (the "Software"), to deal in the Software without
##  restriction, including without limitation the rights to use,
##  copy, modify, merge, publish, distribute, sublicense, and/or
##  sell copies of the Software, and to permit persons to whom the
##  Software is furnished to do so, subject to the following
##  conditions:
##
##  The above copyright notice and this permission notice shall be
##  included in all copies or substantial portions of the Software.
##
##  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
##  KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
##  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
##  PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
##  COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
##  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
##  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
##  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
##

# pinyin.py - Chinese Pinyin to ASCII converter.
# 
# It converts Chinese characters (hanzi) into pronunciation
# strings (pinyin).  The <code>-C</code> (Cantonese) option
# instructs to use Cantonese pronunciation instead of
# Mandarin. When the <code>-T</code> (tone) option is given, each
# pronunciation string is added with the tone numbers.  Other
# non-ASCII characters that are not Chinese remain untouched.
# When the <code>-H</code> (HTML) option is given, pronunciation
# strings are attached as superscript letters for each Chinese
# character using <code>&lt;sup&gt;</code> tag.
#
# Usage:
#   $ ./pinyin.py [-i input_encoding] [-C] [-T] [-H] file ...
#

import sys, fileinput, codecs, re
stdout = sys.stdout
stderr = sys.stderr

PINYIN_TABLE_MANDARIN_SRC = {
 '4Bie': '38bc 4272 42e2 43df 5f46 5fb6',
 '2Pei': '37dd 48d9 4ab9 4ae0 4cb9 57f9 6bf0 88f4 88f5 8ce0 8d54 90b3 962b 966a 966b',
 '1Fu': '4f15 544b 592b 598b 59c7 5b75 5c03 6024 61ef 6577 65c9 67b9 67ce 6ced 739e 7806 7a03 7b5f 7cd0 7d92 80a4 819a 8374 886d 8c67 8dba 8dd7 909e 911c 9207 9cfa 9ea9 9eac 9eb1 9eb8',
 '4Jiang': '47aa 531e 5320 5905 5d79 5f1c 5f36 647e 6d1a 729f 7ce1 7ce8 7d73 7edb 8503 88b6 8b3d 9171 91a4 91ac 964d',
 '2Pen': '6e53 74eb 76c6 8450',
 '3Nan': '3af1 406a 4212 4533 6201 63c7 6e73 7f71 8169 877b 8d67',
 '1Hang': '592f',
 '4Lei': '3b69 3c95 3d03 426a 4365 44b9 48ae 48e6 4b91 57d2 6502 6cea 6d99 6dda 790c 7927 79b7 7c7b 7d2f 7e87 808b 8631 9179 9287 9311 982a 985e 98a3',
 '3Gou': '3e83 5ca3 72d7 73bd 7b31 8007 8008 8009 82df 8329 86bc 8c7f',
 '3Dou': '37b3 3ab7 4571 551e 6296 6568 6597 6793 67a1 86aa 9627 9661',
 '3Kan': '3e1d 4dab 4f83 5058 519a 574e 57f3 586a 5d01 60c2 6b3f 6b41 6b5e 780d 83b0 8f21 8f57 9851',
 '2Gan': '4ee0',
 '1Fa': '50a0 53d1 5f42 6a43 6cb7 767a 767c 7782',
 '2Xuan': '3623 3cec 3e61 4062 45e0 4b84 4c82 4cbb 4d09 4d0b 4f2d 59b6 5ad9 60ac 61f8 65cb 66b6 6a88 6f29 7384 73b9 7401 7487 74bf 8701 8ab8 9e6e',
 '4Yan': '36ea 389b 3994 3b2b 3dd4 3df3 3dfc 40a9 4167 444d 4729 48ad 4a04 4b58 4cdb 4cfa 4d0f 5050 50bf 538c 53ad 5501 55ad 56a5 5830 5895 599f 59f2 5a69 5b0a 5b3f 5bb4 5f65 5f66 6079 6565 664f 66a5 66d5 66e3 693b 6edf 704e 7054 7067 7069 7114 7130 7131 71c4 71d5 7213 726a 781a 786f 8273 8276 8277 898e 89c3 89fe 8afa 8b8c 8b9e 8c1a 8c33 8c53 8c54 8d0b 8d17 8d5d 917d 91bc 91c5 95eb 9681 96c1 990d 995c 9a10 9a34 9a57 9a60 9a8c 9ceb 9d08 9d33 9dc3 9df0',
 '4Yao': '353d 3754 3781 3d78 3fd1 3fe2 42e4 446c 4645 66dc 718e 71ff 735f 77c5 7a7e 7b44 8000 825e 836f 846f 85ac 85e5 888e 8981 899e 8b91 8dad 9470 94a5 977f 9dc2 9e5e',
 '1Ou': '543d 5455 5614 6ad9 6b27 6b50 6bb4 6bc6 71b0 74ef 750c 770d 7798 8b33 8bb4 93c2 9d0e 9dd7 9e25',
 '3Zi': '373d 39d7 3ead 4502 4623 4ed4 5407 5470 5559 59c9 59ca 5b50 674d 6893 699f 6ed3 77f7 79c4 79ed 7b2b 7c7d 7d2b 8014 80cf 8293 8308 8678 8a3e 8a3f 91e8',
 '1Dang': '5105 5679 5d63 5f53 6fa2 73f0 74ab 7576 7b5c 7c39 8261 87f7 88c6 8960 943a 94db',
 '4Xi': '3476 367e 369b 3792 38df 3938 3961 3b61 3e0d 3e6b 422a 42a0 443c 44c7 467d 46b7 46e5 4701 49cd 4a33 4a64 4b8e 4caa 4fc2 534c 546c 54a5 55ba 568a 56b1 588d 5c43 5c53 5c6d 5fe5 602c 6044 620f 622f 6231 6232 6673 691e 6c50 6eca 6f5f 6f99 7182 7294 76fb 77fd 78f6 798a 7a27 7a78 7cfb 7d30 7d8c 7e4b 7e6b 7ec6 7ee4 7fd5 7fd6 80b8 8203 8204 856e 8669 884b 89a4 8b11 8d65 8d69 90c4 90e4 910e 91f3 91f8 938e 95df 960b 9699 969f 993c 9969 9b29 9ed6 9f42',
 '3Fou': '5426 6b95 7f36 7f39 7f3b 96ec 9d00',
 '3Hen': '44f3 4f77 54cf 5f88 72e0 8a6a',
 '1Xie': '4e9b 6954 6b47 7332 880d 8909',
 '4Pei': '3904 39e9 3af2 3cc8 4283 4f02 4f69 59f5 5e14 65be 65c6 6c9b 6d7f 73ee 7b29 82dd 8f61 8f94 914d 9708 99b7',
 '2Reng': '3b41 3eb1 4127 46ae 4ecd 793d 8fb8 967e',
 '4Huo': '35f2 361e 399c 39af 3a2f 3bc9 3e0c 4068 4084 4100 425f 42ed 499a 4c25 5268 549f 55c0 5684 56af 56bf 596f 5f5f 5f60 60d1 6216 6347 639d 6509 65e4 66e4 6947 6ab4 6c8e 6e71 6fe9 7016 7372 74c1 7668 7713 77c6 77d0 77f1 7809 790a 7978 798d 7a6b 802f 81d2 81db 8267 83b7 84c3 85ff 8816 8b0b 8b97 8ca8 8d27 944a 956c 9584 96d8 970d 9729 9743 97c4 9a1e 9c6f 9ce0 9e0c 9e71 fa0d',
 '4Hun': '3975 4159 4171 46a0 49f0 4fd2 5031 5702 6141 6df7 6eb7 711d 89e8 8ae2 8be8',
 '4Hui': '3470 3479 3491 3707 37a7 392c 3963 3a24 3a39 3a68 3b29 3c25 3c71 3dc4 3ec5 4095 414f 41fb 4307 4547 464c 4661 46db 470b 4927 49e5 4a48 4aed 4f1a 50e1 5136 532f 5349 5599 5612 5667 5696 571a 5b07 5bed 5ec6 5f57 5f59 5f5a 605a 6075 60e0 6167 6193 6666 66b3 6703 69e5 6a5e 6ad8 6ba8 6c47 6ccb 6e4f 6ed9 6f53 6fca 70e9 71f4 7369 74af 77ba 79fd 7a62 7bf2 7d75 7e62 7e6a 7ed8 7f0b 7fd9 7ffd 835f 8527 8559 8588 85f1 879d 87ea 8a6f 8aa8 8af1 8b53 8b7f 8bb3 8bf2 8cc4 8d3f 93f8 942c 95e0 9613 9767 97e2 982e 9892 992f 9956',
 '4Sou': '3d7b 55fd 7636 8184',
 '1Tang': '528f 5621 5763 6c64 6e6f 7fb0 876a 8e5a 93dc 9557 9f1e',
 '4Xu': '3428 355b 3585 35f5 3627 369c 3705 373f 378a 3922 39bd 3c32 3d70 3de6 3eb7 3fa5 4086 42f6 460f 4652 46d9 4721 4895 48f1 48f4 4997 49bd 4b14 4c5b 4cf3 4f35 4f90 52d6 52d7 5379 53d9 57bf 58fb 5a7f 5e8f 6034 6064 6149 654d 6558 65ed 65ee 65ef 6702 69d2 6b88 6c7f 6c80 6d2b 6e86 6f35 6f4a 70c5 73ec 76e2 7781 77b2 7a38 7aa2 7d6e 7dd2 7dd6 7e8c 7eea 7eed 805f 82a7 84c4 85c7 85da 8a39 8cc9 9157 928a 980a 987c 9b46 9b63 9c6e',
 '4Rao': '7e5e 7ed5 9076 96a2 9864',
 '4Hua': '34f0 3566 3577 368c 3822 398e 3a47 46e1 4809 5283 5316 5a73 5aff 6466 6779 6866 69ec 6a3a 6f85 753b 756b 7575 7e63 8219 89df 8a71 8ae3 8b6e 8bdd',
 '3Zu': '4503 4595 4fce 723c 73c7 7956 7d44 7ec4 8a5b 8bc5 963b 977b',
 '3Dui': '966e',
 '3Jiu': '3871 4e45 4e46 4e5d 4e63 597a 7078 7396 7d24 820f 8d73 9152 9579 97ed 97ee',
 '3Lian': '3991 3a98 3bec 3c08 3c38 41dc 431e 5b1a 6459 6582 7fb7 8138 81a6 81c9 83b6 859f 913b',
 '3Liao': '3dab 4126 4460 4a4d 61ad 71ce 77ad 84fc 911d 91d5 957d',
 '4Jue': '9d02',
 '5Lou': '55bd 560d',
 '3Da': '6253',
 '4Jiao': '3810 3b2d 3c3e 4083 4197 48e4 4a92 53eb 544c 5602 5626 564d 566d 5b13 5ce4 5da0 5fbc 630d 654e 6559 656b 65a0 6ed8 6f16 6f50 73d3 76ad 7a8c 7a96 8a06 8b65 8f03 8f4e 8f7f 8f83 9175 91ae 91b6 91c2',
 '4Jian': '34fa 350b 38e4 3997 3a34 3bfa 3c04 3e9d 41df 47c5 4914 495c 49d6 4b3b 4b48 4b55 4d56 4d5b 4ef6 4f9f 4ff4 4fff 5065 50e3 50ed 5251 5263 5271 528d 528e 5292 5294 5efa 5fa4 64f6 65d4 67a7 682b 6898 6957 6997 69db 6abb 6bfd 6d0a 6da7 6e10 6e85 6f38 6f97 6ffa 7033 726e 73d4 78f5 7bad 7ccb 7e5d 8171 81f6 8230 8266 8350 852a 85a6 87b9 8949 898b 89b5 89b8 89c1 8acc 8ad3 8aeb 8b7c 8c0f 8cce 8ce4 8d31 8d9d 8df5 8e10 8e3a 91fc 9274 92fb 9373 9375 93e9 9451 9452 946c 9473 952e 991e 996f 9c0e',
 '1Qia': '6390',
 '4Wo': '381b 3c67 404a 480e 5053 5367 5e44 637e 63e1 64ed 65a1 6943 6c83 6db4 6e25 6fe3 7125 809f 81e5 84a6 9f77 9f8c',
 '4Jun': '347a 349e 34f4 3559 3766 3d2b 3ed2 3f59 41f9 4551 472d 474d 4fca 5101 5441 5bef 5cfb 61cf 6343 6508 651f 6659 6bbe 6d5a 6fec 710c 73fa 756f 7743 7ae3 7b98 7b9f 8470 8528 90e1 96bd 96cb 9915 9982 99ff 9a8f 9d58',
 '4Bei': '3528 36dd 38c1 3c06 3d94 3df6 3e22 3e2c 3e3d 3ed7 3f0e 3fb1 4045 42f3 4512 4819 4876 4a40 4c3d 4ffb 500d 505d 5079 5099 50c3 5907 5b5b 6096 60eb 6102 618a 6601 6896 7119 726c 7295 72c8 72fd 73fc 7432 789a 7cd2 7d34 80cc 84d3 88ab 8919 8c9d 8d1d 8ef0 8f29 8f88 90b6 90e5 9101 92c7 943e 94a1 97b4 97db',
 '2Zhou': '36e9 59af 8ef8 8f74',
 '1Zhou': '4f9c 5468 5541 558c 5a64 5dde 5f9f 6d00 6d32 6dcd 70d0 73d8 76e9 77ea 7ca5 813d 821f 8b78 8bea 8cd9 8d52 8f08 8f16 8f80 9031 9282 9a06 9d43 9e3c',
 '1Luo': '5570 56c9 7f57',
 '3Jia': '3545 445d 4eee 5047 53da 5a7d 5cac 5fa6 659a 659d 6935 698e 69da 6a9f 73be 7532 7615 80db 8cc8 8d3e 9240 94be',
 '4Wu': '3433 3854 3f7e 40d6 43b8 4441 46e9 498d 4cf1 4fc9 5140 52a1 52d9 52ff 537c 575e 57ad 5862 5966 5a7a 5be4 5c7c 5c89 5d4d 5fe2 609e 609f 60ae 620a 6264 6544 6664 674c 6c95 6ea9 7110 7183 7269 75e6 77f9 7ab9 7c85 82b4 8601 8aa4 8bef 8fd5 901c 907b 92c8 94a8 9622 9696 96fe 971a 9727 9770 9a16 9a9b 9da9 9f40 fa0c',
 '3Juan': '3df7 57e2 6372 81c7 83e4',
 '3Jin': '377b 3b10 3bf8 3e4f 4436 44ba 4910 4946 4b59 4ec5 4fad 50c5 5118 537a 5807 5c3d 5df9 5ed1 616c 69ff 6f0c 747e 7d27 7dca 83eb 89b2 89d0 8b39 8c28 9326 9526 9949 9991',
 '5Ba': '5427',
 '4Chou': '450f 6ba0 7c09 81ed 81f0',
 '2Kuang': '392e 6282 72c2 8a91 8bf3 9d5f',
 '1Kuan': '5bbd 5bdb 5bec 81d7 9acb 9ad6',
 '1Cha': '505b 53c9 55cf 5ac5 6260 633f 63d2 63f7 6748 7580 809e 81ff 8256 929f 976b 9937 9987',
 '3Ang': '4369 8eee 96f5',
 '2Jiao': '56bc 5af6',
 '1Chuan': '5ddb 5ddd 6c1a 744f 7a7f',
 '4Fen': '35b9 3ba5 3fce 4efd 507e 50e8 574b 594b 596e 5fff 6124 61a4 7035 7caa 7cde 81b9 9c5d 9cbc',
 '4Fei': '3517 3a4c 3b6d 3d52 418f 426c 4454 4560 46a8 46cd 480a 4935 4c01 4ff7 5255 539e 5420 5c5d 5e9f 5ec3 5ee2 6632 66ca 676e 6ae0 6cb8 6ff7 72d2 75bf 75f1 7648 7829 80ba 80c7 82be 87e6 8cbb 8d39 9428 9544 9745 9f23',
 '1Dong': '4e1c 5032 51ac 549a 57ec 5a3b 5cbd 5d20 5d2c 6638 6771 6c21 6c2d 6db7 7b17 82f3 83c4 8740 9bdf 9d87 9e2b 9f15',
 '2Hou': '35cb 3927 3b0b 3ba2 3e85 4089 45d4 4648 4adb 4ce7 4faf 5589 5e3f 7334 760a 777a 77e6 7bcc 7cc7 7fed 8454 9297 936d 9931 9bf8',
 '3Guai': '62d0 67b4 67fa 7b89',
 '2Mian': '3499 3770 3b06 3b8c 3c03 40c7 43c3 4af5 4c13 5a42 5a94 5b35 5b80 68c9 6ab0 6acb 7720 77c8 77ca 77cf 7dbf 7ddc 7ef5 81f1 8287 8752 918e',
 '2Miao': '3464 383a 4067 45a2 63cf 7784 7de2 82d7 9c59 9d93 9e4b',
 '3Huan': '38ea 3b0a 4220 550d 5d48 6fa3 7746 7de9 7f13 85e7 8f10',
 '4Dong': '3448 34ca 35e2 3731 38a5 3f2f 414d 4376 4792 50cd 51bb 51cd 52a8 52d5 578c 6219 630f 680b 68df 6d1e 772e 80e8 80f4 8156 9718 99e7',
 '3Lun': '57e8 7a10 8023',
 '3Luo': '34a9 39ac 3a61 3c01 3c7b 502e 651e 66ea 7630 7673 7822 81dd 84cf 8803 88f8 8eb6 93af',
 '2Men': '3848 3a3a 429f 4767 4ad2 626a 636b 6a20 748a 7a48 83db 864b 9346 9494 9580 9585 95e8',
 '2Xun': '3704 3741 3a1a 3c0a 3c2c 3f66 42f8 45b2 4649 5071 565a 5bfb 5c0b 5ccb 5de1 5ef5 5faa 6042 6326 648f 65ec 674a 6794 6812 686a 69c6 6a41 6be5 6d35 6d54 6f43 6f6f 7065 71d6 73e3 7495 7543 7d03 8340 8368 8541 87f3 8a62 8be2 90c7 99b4 99e8 9a6f 9c4f 9c58 9c9f',
 '4Kong': '3e1c 63a7 979a',
 '4Chuang': '43ab 521b 5231 524f 5259 5275 6006 6134 734a',
 '1Juan': '5276 59e2 5a1f 6350 6718 6d93 8832 88d0 93b8 942b 954c 9d51 9e43',
 '2Chun': '3744 3747 3d6e 3e2a 43dd 4407 44d0 451a 48e8 48e9 494e 4ac3 5507 618c 6d71 6df3 6e7b 6ee3 6f18 7d14 7eaf 8123 83bc 8493 84f4 9187 9195 9659 9bd9 9d89 9e51',
 '2Xue': '35b8 39d2 3d85 3ff1 4afb 4c51 4e74 5b66 5b78 5cc3 5da8 6588 6cf6 6fa9 71e2 7a74 8313 89f7 96e4 9dfd 9e34',
 '3Zhui': '6c9d',
 '1Fang': '531a 574a 65b9 678b 6c78 6dd3 7265 82b3 8684 90a1 9201 94ab 9d0b',
 '4Kuai': '351e 3655 3657 37f4 3b2e 3c6e 421b 4b5d 4be4 4d90 4fa9 5108 51f7 54d9 5672 5757 584a 58a4 5ee5 5feb 6867 6b33 6d4d 6fae 736a 79ac 7b77 7ce9 810d 81be 90d0 9136 99c3 9b20 9c60 9c99',
 '4Xun': '38b2 46dc 478a 4b40 4f28 4f9a 5640 57c8 595e 5dfa 5dfd 5f87 6b89 6c5b 720b 72e5 8548 8a0a 8a13 8a19 8bad 8baf 8cd0 8fc5 8fff 900a 905c 9442 9656 97d7 9868 9d54 9d55',
 '4Shui': '3f77 4b68 5a37 5e28 6d97 6d9a 7761 7793 7971 7a05 7a0e 8715 88de',
 '2Peng': '36d4 394a 4118 486b 4c03 4d36 5017 50b0 57c4 580b 585c 5873 5f38 5f6d 6189 6337 670b 68da 6916 6a25 6ddc 6f8e 75ed 787c 7a1d 7afc 7bf7 7e84 81a8 8283 84ec 87da 87db 8f23 930b 945d 97f8 97fc 9a61 9afc 9b05 9b14 9d6c 9e4f',
 '4Shuo': '4025 423e 4303 5981 63f1 6420 6714 69ca 6b36 6d2c 70c1 720d 7361 77df 7855 78a9 7bbe 84b4 9399 9460 94c4',
 '4Shun': '3967 4022 4035 445e 6a53 779a 77ac 821c 8563 9806 987a 9b0a',
 '1Zhuan': '4e13 53c0 587c 5ae5 5c02 5c08 747c 750e 7816 78da 7bff 8011 819e 87e4 911f 9853 989b 9c44 9dd2',
 '3Lu': '352a 389a 3bed 4561 4c90 5364 5877 63b3 64b8 64c4 64fc 6a10 6a79 6ad3 6ef7 7002 7875 78e0 823b 8263 826a 84fe 864f 865c 93c0 942a 9465 9565 9b6f 9c81 9e75',
 '4Xue': '37bd 4380 4915 4afc 4b02 4b65 5437 5779 5ca4 6856 6cec 70d5 8b1e 8d90',
 '4Ling': '4ee4 53e6 6395 70a9',
 '2Chui': '34c3 377d 3968 3a7e 4132 434b 4ce0 5015 5782 57c0 6376 6425 68f0 69cc 6e77 7ba0 7bc5 8144 83d9 9318 939a 9524 9672 9840',
 '1Pan': '3434 6500 6f58 7568 7705 7819',
 '1Pao': '629b 62cb 812c',
 '3N': '343b',
 '1Chui': '5439 708a',
 '1Po': '5761 6cfc 6e8c 6f51 7679 9255 93fa 94b7 9817',
 '1Suan': '72fb 75e0 9178',
 '3Gao': '368f 3696 3fb8 45e3 5930 641e 66a0 6772 69c1 6aba 7a3e 7a3f 7b76 7e1e 7f1f 83d2 85c1 85f3',
 '3E': '3f22 5641 6799',
 '2Zei': '621d 8cca 8d3c 9c61',
 '3Bing': '3a00 42d1 44d1 4d35 4e19 504b 6032 62a6 661e 663a 67c4 70b3 772a 7980 79c9 7a1f 7a89 82ea 86c3 90b4 927c 9643 979e 9905 9920 997c',
 '2Ang': '3b7f 401a 44a2 4a55 4b79 4b7a 536c 5c87 6602 663b',
 '2Xin': '36af 4c3c 6533 6a33 8951 9129',
 '3Er': '36b7 38bd 42d9 433a 5c12 5c13 5c14 5ccf 682e 6d31 723e 73e5 8033 85be 8fe9 9087 927a 990c 9975 99ec',
 '1Bu': '5cec 5eaf 6661 900b 9914 9d4f',
 '1Biao': '5126 5882 5e56 5f6a 6807 6a19 6eee 700c 706c 719b 7202 730b 762d 78e6 7a6e 813f 81d5 85e8 8b24 8d06 93e2 9463 9556 9573 98a9 98ae 98c6 98c7 98cd 98d1 98d9 98da 9a6b 9a89 9adf 9e83 9e85',
 '2Xie': '35bf 365d 395f 3a19 3a49 3a66 3a6a 3b68 4511 4575 464e 4861 4b4e 5055 52a6 52f0 534f 5354 53f6 55cb 57a5 594a 604a 6136 6140 62f9 631f 633e 641a 64b7 64f7 651c 659c 65ea 7181 71f2 7c7a 7d5c 7d8a 7df3 7e88 7f2c 7fd3 80c1 8105 8107 818e 8762 887a 896d 8ae7 8c10 90aa 92e3 978b 97b5 9fa4',
 '3Ning': '6a63 77c3',
 '4Tui': '398c 3ddf 5a27 717a 86fb 9000 99fe',
 '2Xia': '35c7 3621 3c3a 3f60 458e 4596 4625 46c5 4996 4a97 4ad7 4fa0 4fe0 51be 5323 5ce1 5cfd 6433 656e 6687 67d9 70a0 70da 72ce 72ed 72f9 73e8 7455 7856 7864 78ac 78cd 796b 7b1a 7b6a 7e00 7e16 7fc8 821d 823a 8578 8d6e 8f44 8f96 9050 938b 965c 971e 9a22 9b7b 9c15 9ee0',
 '4Wai': '483f 5916 61c0',
 '3Tuan': '471d 4d6f 757d 7583 9ed7',
 '4Chang': '4a68 5021 5531 6005 60b5 66a2 713b 7545 757c 8aaf 97d4 9b2f',
 '1Chou': '601e 62bd 640a 72a8 72ab 7633 7bd8 970c',
 '4Zao': '3ca7 3ff7 470a 5515 5523 566a 6165 688d 7076 71e5 7681 7682 7ac3 7ac8 8241 8b5f 8dae 8e81 9020 9ade',
 '2Tang': '347d 3676 370d 3b7b 3ca5 3f3a 416f 424e 4305 48d8 49dc 508f 5510 557a 5802 5858 642a 645a 68e0 69b6 6a18 6a56 6e8f 6f1f 717b 746d 78c4 799f 7bd6 7cc3 7cd6 7cdb 8185 819b 84ce 859a 8797 87b3 8d6f 8e3c 910c 91a3 9395 95db 969a 9939 9944 9db6',
 '4Zai': '4d67 4fa2 50a4 510e 518d 5728 6257 8f09 8f7d 9168',
 '3Cha': '4c08 8e45 9454 9572',
 '1Mie': '4e5c 5400 54a9 54f6 5b6d 8288',
 '3En': '4170',
 '4An': '37c1 3c58 3e29 4141 438f 43a8 4b13 4b97 4be5 5111 530e 5813 5cb8 6309 6697 6848 6d1d 72b4 834c 8c7b 930c 95c7 9b9f 9eec 9eef',
 '4Ao': '356d 362c 362d 371c 3729 3817 397f 4071 4712 4ae8 4baf 50b2 5773 5787 58ba 5961 5965 5967 5aef 5c99 5db4 6160 61ca 6277 64d9 6fb3 8a4f 9a41',
 '1Zuo': '562c',
 '3Bai': '35d7 3f23 4653 4f70 636d 6446 64fa 67cf 767e 77f2 7ca8 7d54 896c',
 '2Wu': '3dfb 3e73 3ecd 4362 499c 4ad3 4b8f 5433 5434 543e 5449 5514 5a2a 65e0 68a7 6bcb 6d16 6d6f 7121 727e 73f8 7491 7966 7991 829c 8323 8381 856a 8708 8a88 8bec 90da 92d8 92d9 94fb 9bc3 9d50 9de1 9e40 9e8c 9f2f',
 '4Zhi': '3544 35cc 35e7 3609 36ff 3742 38e5 3a16 3d1b 413a 4188 41fd 4245 425c 43c4 43ef 442d 4447 44cc 454c 46a6 4777 4783 4879 494d 49af 4ad5 4b39 4b41 4c68 506b 5082 5236 5394 5781 5a21 5bd8 5cd9 5d3b 5e19 5e1c 5e5f 5ea2 5ea4 5ecc 5f58 5f8f 5f9d 5fd7 5fee 6184 61e5 6303 631a 6431 646f 64f2 65d8 667a 684e 68bd 6956 6acd 6cbb 6d37 6ddb 6ede 6eef 6f4c 7099 71ab 72fe 7318 748f 7564 7590 75d4 75e3 7929 7951 79e9 79f2 7a1a 7a3a 7a49 7a92 7b6b 7d29 7dfb 7f6e 7fd0 80f5 815f 81a3 81f3 81f4 81f8 8296 86ed 87b2 889f 88a0 88fd 899f 89d7 89ef 89f6 8a8c 8c51 8c52 8c78 8cea 8d04 8d3d 8df1 8e2c 8e93 8f0a 8f7e 8fe3 90c5 928d 92d5 9455 94da 9527 965f 96c9 99e4 9a2d 9a3a 9a47 9a98 9bef 9d19 9dd9 9e37',
 '1Miao': '55b5',
 '1Hei': '563f 6f76 9ed1 9ed2',
 '2Yi': '36a6 3756 3794 3974 39be 3c18 3ebf 412c 41f5 4405 4416 458a 4785 4a5f 4b01 4b2e 4b8a 4c4c 4c91 4d0a 4e41 4eea 4f87 5100 519d 51d2 531c 54a6 572f 5937 59e8 5b90 5b9c 5ba7 5bf2 5cd3 5dac 5db7 5df8 5f2c 5f5b 5f5c 5f5d 5f5e 6021 605e 6245 6686 675d 67b1 67c2 684b 6938 6b4b 6c82 6cb6 6d1f 71aa 73c6 74f5 7591 75cd 7724 7731 79fb 7c03 7fa0 80f0 82d0 8413 86e6 8794 8898 88b2 89fa 8a11 8a52 8b3b 8b89 8bd2 8c0a 8ca4 8cbd 8d3b 8de0 8ffb 9057 907a 914f 9236 93d4 9809 9824 984a 9890 98f4 9974 9ba7 9d3a 9e03',
 '2Rong': '3607 3750 38d1 3b5c 3c93 3ca8 3e8e 3f38 41c0 41ef 4236 462c 481c 4846 4865 490a 4a78 5ab6 5bb9 5d58 5db8 620e 66e7 6804 6995 69ae 69b5 6be7 6eb6 70ff 7194 7203 72e8 7462 7a41 7d68 7e19 7ed2 7fa2 809c 8319 8338 8363 84c9 877e 878d 878e 8811 8923 9394 9555 99e5 9af6 9c2b 9ddb',
 '2Wan': '3774 4bc8 4e38 5213 5b8c 628f 6c4d 70f7 73a9 7413 7d08 7ea8 8284 9811 987d',
 '3Kong': '391f 5b54 6050',
 '2Ya': '39ce 4130 42a6 4f22 5393 5810 5c88 5d15 5d16 6792 6860 6daf 6f04 7259 72bd 731a 7b0c 82bd 869c 8859 9f56',
 '4You': '34dc 3557 3911 3e68 3ea0 4001 419c 46fb 47a5 4eb4 4f51 4f91 53c8 53f3 56ff 59f7 5ba5 5cdf 5e7c 67da 7270 72d6 7950 7cff 8a98 8bf1 8c81 8ff6 916d 91c9 9d22 9f2c',
 '1Chuo': '6233 9388 9f79',
 '2Xiang': '37c4 4517 4736 4f6d 5ea0 6819 7965 7d74 7fd4 8a73 8be6 8ded fa1a',
 '4Lun': '7896 8ad6 8bba',
 '3Qing': '3deb 451b 4be7 5ebc 5ece 6abe 8acb 8bf7 9803 9877',
 '2Shao': '3c88 3e1b 52fa 6753 724a 73bf 7af0 828d 97f6',
 '4Heng': '5548 6494 6f8b 7d4e 7ed7',
 '4Luo': '34e2 3d16 3fda 4029 41d4 4237 4331 4334 55e0 5cc8 645e 6d1b 6ffc 70d9 7296 73de 784c 7b3f 7d61 7e99 7edc 8366 843d 88bc 96d2 99f1 9a86 9ba5 9d3c 9d45',
 '3Rang': '56b7 58cc 58e4 7219',
 '2Yu': '349c 36a5 3924 3954 395a 3965 399b 3a80 3b02 3b30 3cdb 3d9b 3dd2 3e9e 3eae 3f36 4069 409b 40cb 410f 4128 4342 43f8 4433 4521 45e8 473d 4893 4a52 4c3b 4c77 4ca3 4e88 4e8e 4e8f 4e90 4f03 4f59 4fde 516a 5823 582c 59a4 5a1b 5a2f 5a31 5a80 5b29 5d33 5d4e 5d5b 6109 611a 6275 63c4 65bc 65df 6745 6859 6961 6970 6986 6b24 6b48 6b5f 6b76 6e14 6e1d 6e61 6f01 6f9e 72f3 7397 7399 745c 74b5 756c 756d 7572 76c2 776e 7862 79ba 7aac 7afd 7c45 7df0 7fad 8167 8174 81fe 8201 8206 8245 8330 8438 854d 861b 865e 8753 87b8 8867 8915 89a6 89ce 8adb 8b23 8c00 8e30 8f3f 8f5d 903e 9098 9103 91ea 935d 9683 9685 96d3 96e9 9918 9980 9aac 9ac3 9b5a 9bbd 9c05 9c7c 9d4c 9de0 9e06 9e12',
 '2Tong': '3809 383d 38da 38e0 390f 3b94 3e17 3f27 3f3f 4088 419a 4bb5 4ccb 4d00 4db1 4edd 4f5f 50ee 52ed 540c 54c3 5cc2 5cd2 5cdd 5e9d 5f64 664d 66c8 6723 6850 6c03 6d75 6f7c 70d4 71d1 721e 729d 735e 77b3 783c 7850 79f1 7a5c 7ae5 7ca1 7d67 7f7f 81a7 825f 833c 856b 8692 8a77 8ff5 916e 9256 9275 9285 94dc 9907 9ba6 9c96 9f28',
 '2Ren': '433e 4d26 4eba 4ebb 4ec1 58ec 5fc8 5fce 6732 79c2 7d4d 7eb4 82a2 9213 928b 9b5c 9d39',
 '4Zhu': '344f 3749 3916 39e3 3ac2 3e65 3e9b 3fbb 3ffe 41e1 41e7 4346 43b7 4422 4752 476c 4b21 4b56 4f2b 4f47 4f4f 52a9 577e 5d40 677c 67f1 67f7 6a26 6bb6 6ce8 70b7 75b0 771d 795d 7ada 7b6f 7bb8 7d35 7d38 7ebb 7f9c 7fe5 82e7 8387 8457 86c0 8a3b 8ad4 8dd3 8ef4 9252 92f3 9444 94f8 9714 98f3 99b5 99d0 9a7b 9e86',
 '1Yue': '66f0 66f1 7bb9 7d04 7ea6',
 '2Yun': '3683 370f 4259 48b5 4e91 52fb 5300 5998 612a 6600 6985 69b2 6a52 6c84 6da2 6eb3 6f90 7189 7547 7703 79d0 7b60 7b7c 7bd4 7d1c 7e1c 7ead 8018 803a 82b8 837a 84b7 8553 90e7 9116 92c6 96f2 9942',
 '1Su': '56cc 6aef 7526 7a23 7a4c 82cf 8607 8613 9165',
 '1Yuan': '51a4 5248 566e 56e6 5b3d 5bc3 6081 60cc 68e9 6df5 6e01 6e06 6e0a 6e15 7041 7722 7ba2 8099 847e 84ac 870e 8735 88f7 92fa 99cc 9cf6 9d1b 9d77 9e22 9e33 9e53 9f18 9f1d',
 '1Ha': '54c8 927f',
 '2Quan': '34b0 37e8 37eb 402c 444f 47d2 4830 4f7a 5168 5573 5a58 5b49 5cd1 5dcf 606e 60d3 62f3 643c 6743 6a29 6b0a 6cc9 6d24 6e76 7277 7288 7454 75ca 7842 7b4c 8343 8472 8737 8838 89e0 8a6e 8be0 8de7 8e21 8f07 8f81 919b 9293 94e8 9874 98a7 99e9 9a21 9b08 9c01 9cc8 9de4 9f64',
 '1Wen': '586d 6637 6b9f 6e29 6eab 7465 761f 7783 8c71 8f3c 8f40 8f92 99a7 9c1b 9c2e 9cc1',
 '1Chen': '68fd 741b 778b 7e1d 8ac3 8b13 8cdd 90f4',
 '4Xiang': '37df 441f 48bd 50cf 5411 56ae 5842 5df7 6066 66cf 6a61 73e6 842b 8683 87d3 8801 8950 8c61 940c 95c0 95c2 9805 9879 9c4c',
 '3Chen': '4ad6 588b 6375 7876 789c 78e3 7973 8d02 8dbb 8e38 9356',
 '3Liang': '34b3 351d 44e3 4803 4a6b 4e21 4e24 5169 5521 5562 639a 7dc9 813c 873d 88f2 9b49 9b4e',
 '3Yue': '54d5 5666',
 '2Teng': '4485 4568 482e 4c8d 4ca2 512f 5e50 6ed5 6f1b 75bc 7c50 7c58 7e22 817e 855b 85e4 87a3 8a8a 8b04 9086 99e6 9a30 9a63 9c27',
 '4Zhun': '7a15',
 '4Gan': '4bce 4cba 501d 51ce 5e79 65f0 69a6 6a8a 6c75 6dbb 6de6 7c33 7d3a 7ec0 8a4c 8d1b 8d63 9aad',
 '4Gao': '53dd 543f 544a 7170 796e 7970 799e 8aa5 8bf0 90dc 92ef 9506 9bcc',
 '4Gai': '3562 39c9 3ba3 43d7 4e10 4e62 5303 5304 6224 6461 6982 69e9 69ea 6e89 6f11 74c2 76d6 8462 84cb 8ccc 9223 9499',
 '4Kun': '3afb 56f0 6d83 774f',
 '3Za': '9bba 9c9d',
 '4Dou': '36d2 3884 3dc6 4108 4546 46e0 4b26 68aa 6bed 6d62 75d8 7aa6 7ac7 8130 8373 8c46 9017 92c0 9597 95d8 9916 997e 9b25 9b26 9b2a 9b2c 9b2d',
 '2Pang': '3936 396c 3ac4 416d 4a26 4bbe 5390 5396 5ace 5e9e 5eac 5f77 5fac 65c1 7be3 80ee 823d 8783 9004 96f1 9736 9a2f 9ac8 9f8e 9f90',
 '1Shan': '5220 522a 527c 59cd 59d7 5c71 5e53 5f61 633b 6427 6749 6a86 6f78 6f98 70fb 717d 72e6 73ca 7a47 7b18 7e3f 7fb4 7fb6 8120 8222 829f 82eb 886b 8dda 8ed5 9096',
 '3Mang': '3641 37d0 3b12 4073 448e 459f 58fe 6f2d 7865 833b 83bd 83be 87d2 880e',
 '4Qian': '3438 371e 37fb 3aa0 3be0 4234 42b4 4945 49b2 4ae1 5029 5094 5119 520b 5811 5879 58cd 5d4c 6093 614a 68c8 6920 69e7 6b20 6b49 7698 7bcf 7bdf 7daa 82a1 831c 84a8 8533 8f24',
 '4Qiao': '3897 3aa3 3d25 40dd 41bb 41cc 4fcf 50fa 5ced 5e29 64ac 64bd 7a8d 7ac5 8a9a 8bee 8e88 9657 9798 97d2 9ada',
 '4Nong': '5f04 630a',
 '4Yang': '3a3e 3e8a 3fee 4b3a 4b50 4d6e 600f 6059 69d8 6a23 6f3e 7001 7f95 8a47',
 '3Xiu': '3c59 673d 7cd4 7d87',
 '1Jun': '519b 541b 5747 59f0 687e 6c6e 76b2 76b8 76b9 7885 8399 83cc 8690 8880 89a0 8ecd 921e 9281 929e 94a7 9835 9bb6 9caa 9e87 9e8f 9e95',
 '4Pie': '5af3',
 '4Cun': '434e 540b 5bf8 7c7f',
 '4Nen': '371b 3bce 3da7 5ae9 5af0',
 '4Nei': '356f 358f 3628 3a05 487e 4cd6 5167 5185 6c1d',
 '4Cui': '376e 395e 39d8 3bd4 3bdc 3c56 3cc3 3dea 40b1 40c0 411f 418a 4f1c 5005 5550 555b 5ff0 60b4 6bf3 6dec 7120 7417 75a9 7601 7ac1 7c8b 7cb9 7d23 7db7 7fc6 7fe0 8103 8106 813a 81ac 81b5 81ce 8403 9847',
 '2Cuan': '381d 3b6b 4198 4c16 5297 5dd1 6ad5',
 '3Xia': '959c',
 '3Cuo': '40b3 7473 7e12 811e',
 '2Pin': '3c0b 56ac 5a26 5ad4 5b2a 7371 7415 85b2 860b 8ca7 8d2b 983b 9870 9891 98a6',
 '3Cui': '3d4f 3dc3 42ab 49fd 6f3c 7480 76a0 8da1',
 '5Mo': '5e85 9ebd',
 '3Rui': '60e2 6875 6a64 7e60 854a 854b 8602 8603',
 '3Xie': '3550 374d 4971 497e 5199 51a9 5beb 85db 8840 8ea0',
 '3Gui': '3533 39ea 3a33 3a7b 3cb9 3e35 405b 436f 4788 47a8 48c0 4925 4f79 5326 532d 53ac 579d 59fd 5b84 5e8b 5eaa 6051 6531 6677 6c3f 7678 796a 7c0b 86eb 87e1 89e4 8a6d 8be1 8ecc 8f68 9652 9b3c',
 '2Pa': '4bf2 4d95 6777 6f56 722c 7436 7b62 8dc1',
 '4Pai': '3b5b 45b0 54cc 6c56 6d3e 6e43 848e 9383',
 '4Pao': '3610 3be1 4d8c 5945 6ce1 70ae 75b1 76b0 7832 791f 792e 9764 9af1 9ead',
 '4Pan': '3cea 51b8 5224 53db 62da 6c9c 6cee 6ebf 708d 7249 7554 76fc 88a2 897b 8a4a 92ec 947b 9816',
 '3Guo': '3785 4668 4d39 60c8 679c 6901 69e8 6dc9 7313 7cbf 7db6 83d3 873e 88f9 8f20 9301 9439 991c 9983',
 '3Gun': '3665 3ace 3bfb 40c2 43be 4707 4e28 638d 6eda 6efe 78d9 7dc4 7ef2 84d8 8509 886e 889e 88ac 8f25 8f8a 9b8c 9bc0 9ca7',
 '3Gua': '34b7 4211 518e 5250 526e 53e7 5be1',
 '4Ban': '3698 3ab5 4570 4f34 529e 534a 59c5 6011 626e 62cc 6e74 74e3 79da 7d46 7eca 8fa6 9261 977d',
 '1Bin': '5bbe 5f6c 658c 6915 69df 6ab3 6c43 6ee8 6fd2 6ff1 7015 73a2 7478 74b8 77c9 7e7d 7f24 8819 8c69 8c73 8cd3 8cd4 90a0 944c 9554 9726 986e 99aa 9a5e',
 '4Tu': '514e 5154 580d 83df 9d75',
 '2Ming': '3760 4119 41a8 41a9 4285 4ae4 4f72 51a5 540d 5ac7 660e 669d 6719 69a0 6d3a 6e9f 733d 7700 7791 8317 84c2 879f 89ad 910d 9169 9298 94ed 9cf4 9e23',
 '4Ti': '35e3 3b31 3be9 43ae 4657 49c5 4bdc 4d91 4fe4 501c 5243 568f 5694 5c49 5c5c 608c 6090 60d5 60d6 63a6 63e5 66ff 6b52 6ba2 6d95 777c 7c4a 8599 8905 8daf 8fcf 9016 9037 941f 9511 9ae2 9af0 9b00 9b04',
 '2Tun': '3798 3a54 3e60 3f0a 5749 5c6f 5ff3 7096 81c0 81cb 829a 8a30 8c58 8c5a 8ed8 9715 98e9 9968 9b68 9c80',
 '2Cun': '5b58 62f5 88b8',
 '3Ma': '3437 48d5 48d6 6ea4 739b 746a 7801 78bc 8682 879e 93b7 99ac 9a6c 9c22 9dcc',
 '3Tu': '5410 550b 571f 5721 828f 91f7 948d',
 '1Bai': '63b0',
 '4Ta': '3493 36e5 38db 38f5 39fa 3b7c 3bd3 3ce0 3ceb 3e7a 3ff9 40bf 420b 4233 4308 4347 435d 4393 445c 44e0 471a 4d6c 4d80 4d81 4ea3 55d2 5683 56ba 5d09 631e 6428 64bb 69bb 6a7d 6bfe 6c93 6dbe 6ebb 6f2f 6fbe 6fcc 72e7 736d 7f8d 8abb 8b76 8df6 8e0f 8e4b 8e79 8e82 8ea2 905d 9062 9314 9389 9449 95d2 95e5 95fc 9618 979c 97b3 9b99 9c28 9cce 9f96 9f98',
 '1Nen': '9ec1',
 '4Te': '3942 39f9 5fd1 5fd2 615d 68cf 7279 8126 86ae 87d8 8ca3 92f1 94fd 9d0f',
 '2Niu': '4014 449c 725b',
 '1Dai': '5446 5454 61db 7343',
 '2Min': '37e9 37ed 38af 4055 40a5 40c9 42cb 47e8 4851 487b 4ab8 4c84 59c4 5cb7 5d0f 5fde 5fdf 600b 636a 656f 65fb 65fc 6c11 739f 73c9 7418 7449 75fb 76ff 7807 7888 7dcd 7de1 7f17 7f60 82e0 9231 9309 9372',
 '1Tuo': '4edb 4f82 4fbb 5483 6258 6265 62d5 62d6 6329 635d 6754 6c51 6cb0 6db6 77fa 812b 8131 8a17 8bac 98e5 9966 99b2 9a5d',
 '3Nian': '361d 378b 4693 637b 649a 64b5 6506 6d8a 6df0 78be 7c10 8dc8 8e68 8e8e 8f26 8f87',
 '3Niao': '349f 3735 3821 3b64 40f5 4401 465a 498a 4b8d 5acb 5b1d 5b32 6a22 8311 8526 8885 88ca 892d 9ce5 9e1f',
 '3Wo': '39b1 39f4 4c00 5a51 6211 6370',
 '4Si': '340c 357d 36b6 38c8 3b52 3e3b 3e51 3f85 41c3 43a3 43e4 4999 4e96 4f3a 4f3c 4f40 4fdf 5129 5155 55e3 56db 59d2 5a30 5b60 5bfa 5df3 676b 67f6 6c5c 6cd7 6ce4 6d0d 6d98 7003 726d 7940 79a9 7ae2 7b25 801c 8082 8086 857c 8c84 8d50 91f2 923b 98e4 98fc 9972 99df 9a77 fa2b',
 '3Wa': '39da 3f18 4f64 5493 74e6 90b7',
 '4Se': '348a 3976 3ba6 3c47 3d14 3eed 4262 453c 4a1b 556c 55c7 61ce 6b6e 6b70 6da9 6f80 6f81 6fc7 7012 745f 74b1 7a51 7a61 7e6c 7fdc 8272 8b45 8d87 8f56 92ab 94ef 96ed 98cb',
 '3Meng': '369e 43f5 52d0 61f5 731b 74fe 824b 8722 8813 9333 9530 9bed',
 '4Sa': '3aaa 3cd0 3f42 462e 4663 4b03 5345 644b 6503 6ad2 810e 8428 85a9 8dbf 939d 98af 98d2 99ba',
 '3Song': '39d0 3a26 3a73 4265 472c 50b1 5d77 6002 609a 612f 616b 6352 6964 7ae6 8038 8073 99f7',
 '2Chuang': '3856 40e5 4692 4874 4b5a 5647 5e62 5e8a 6a66 7240 7592',
 '2Tai': '3497 3606 3675 38cd 3b03 3dd8 3e00 421a 4453 48b0 5113 51ad 53f0 576e 5b2f 62ac 64e1 65f2 6aaf 70b1 70b2 7c49 81fa 82d4 83ed 85b9 8dc6 90b0 98b1 99d8 9a80 9b90 9c90',
 '2Zhan': '859d',
 '4Su': '3449 345b 34d8 3504 3556 371a 375b 3a1e 3a4b 3aa9 3b18 3bc8 3d0b 3d11 3d3c 40e4 4147 431a 4398 43cb 447f 450e 463b 46fe 4954 5083 50f3 55c9 5850 5851 5919 5aca 5bbf 612b 612c 619f 681c 69a1 6a15 6a5a 6b90 6cdd 6d91 6eaf 6eb8 6eb9 6f65 738a 73df 749b 78bf 7aa3 7c0c 7c9b 7c9f 7d20 7e24 8083 8085 8186 84ff 850c 85d7 8736 89eb 8a34 8a8e 8b16 8bc9 8c21 8d9a 901f 9061 906c 92c9 9917 9a4c 9a95 9c50 9deb 9e54',
 '3Wu': '3405 3444 3b33 3d72 4489 47fc 4867 4cc7 4e94 4ef5 4f0d 4fae 5035 511b 5348 554e 58b2 59a9 5a12 5a2c 5af5 5d68 5e91 5ee1 5fe4 6003 61ae 6342 6440 65ff 6a46 6b66 6f55 7193 739d 73f7 7512 7894 821e 8e8c 9d61 9e49',
 '4Guang': '3adb 6844 6ace 81e6 81e9 901b',
 '2Tao': '3e57 4b1e 530b 54b7 5555 6843 68bc 6aae 6d2e 6dd8 7979 7daf 7ef9 8404 88ea 8fef 9003 9184 92fe 932d 9676 9780 9789 9940 99e3 9a0a 9f17',
 '2Tan': '3c9c 3dcb 3f51 40ea 4261 42a4 454a 5013 563e 575b 58b0 58b5 58c7 58dc 5a52 60d4 619b 6619 66c7 6983 6a80 6f6d 75f0 7f48 7f4e 8262 85eb 8983 8ac7 8b5a 8c08 8c2d 8c9a 90ef 9188 91b0 931f 9414 952c 9561 9924 9a54',
 '2Wa': '3c2a 5a03',
 '1Sa': '4ee8 6492',
 '3Chang': '3ae4 454b 4800 50d8 5382 53b0 5ee0 60dd 655e 6636 6c05 92f9',
 '1Nan': '56e1',
 '4Fan': '3568 3743 3906 3d00 3d97 3f1d 3f79 424a 442a 44a6 48f2 597f 5b0e 68b5 6c3e 6c4e 6cdb 6efc 72af 7548 76d5 7b35 7bc4 8303 8a09 8ca9 8d29 8ed3 8eec 98ef 98f0 996d fa2a',
 '4Ji': '3467 34ab 3668 3816 3831 386d 3876 394d 3b70 3c1f 3c85 3cf5 3e04 3e44 3ed1 3f92 3fb5 42df 4400 4481 44eb 44fd 45c1 471e 4778 480f 488b 4987 4a16 4bba 4c0f 4d93 4da9 4f0e 517e 5180 5242 5264 5291 54dc 568c 5756 578d 5848 5993 5b63 5bc2 5bc4 5f50 5f51 5fcc 60b8 60ce 61fb 6280 65e1 65e2 65e3 66a8 66c1 689e 6a95 6ab5 6d0e 6d4e 6e08 6f08 6fdf 7031 7324 74be 75f5 7608 7660 77a1 796d 7a29 7a37 7a44 7a4a 7a67 7d00 7d99 7e7c 7eaa 7ee7 7f7d 81ee 82b0 830d 8324 846a 84df 8507 858a 860e 862e 863b 88da 8900 89ac 89ca 8a08 8a18 8a8b 8ba1 8bb0 8dfd 9645 969b 9701 973d 9a65 9aa5 9afb 9b3e 9b5d 9b86 9bda 9bfd 9c36 9c40 9c6d 9c9a 9cab 9d4b 9f4c',
 '1Gao': '69d4 69f9 6a70 6adc 6edc 768b 7690 777e 7bd9 7cd5 7f94 7f99 818f 81ef 993b 9ad8 9ad9 9dce 9f1b',
 '1Si': '4e1d 5072 51d8 53ae 53b6 53f8 549d 5636 565d 5aa4 5edd 601d 6495 65af 6952 69b9 6cc0 6f8c 71cd 78c3 7997 79a0 79c1 7c1b 7c6d 7cf9 7d72 7de6 7e9f 7f0c 7f73 856c 8652 86f3 8724 8784 87d6 87f4 8997 9270 92d6 9376 9401 98b8 98d4 9a26 9de5 9e36',
 '4Xiong': '5910 657b 8a57 8bc7',
 '3Chu': '3e7c 42b0 458f 4658 50a8 5132 51e6 61b7 6775 6918 695a 696e 6a9a 6fcb 7840 790e 891a 9f6d 9f7c',
 '3Qun': '877d',
 '1Dui': '5796 5806 5860 5d5f 75fd 78d3 9827',
 '4Nai': '3b88 3b8f 3ca1 5037 5948 67f0 6992 6e3f 8010 8418 879a 8926 933c 9f10',
 '4Ju': '360c 3718 37ab 382a 3a3f 3a40 3b2c 3ce5 41bd 46ef 48f0 4c5f 4d55 4d99 4ff1 5028 5036 5177 51a3 5267 5287 52ee 53e5 57e7 57fe 58c9 59d6 5be0 5c66 5c68 5ca0 5de8 6007 601a 60e7 6133 61fc 62d2 62e0 636e 64da 661b 6b6b 6cc3 6d30 6fbd 70ac 7123 728b 77bf 79ec 7aad 7ab6 7c34 7c94 7d47 801f 805a 82e3 8661 86b7 8a4e 8bb5 8c97 8ddd 8e1e 8e86 907d 90ad 91b5 9245 92f8 943b 949c 952f 98b6 98d3 99cf 9b94',
 '2Po': '3a07 3c34 5619 5a46 6ac7 76a4 8522 9131',
 '3Xian': '372a 39cb 39e5 3aeb 3b0e 3b60 3bc0 3d8d 3fc5 4273 45fe 4606 469a 4722 48be 496a 49cb 49ee 51bc 5c1f 5c20 5d04 5dae 5e70 641f 6507 663e 6af6 6be8 7066 70cd 71f9 72dd 7303 736b 736e 7381 7663 7906 7992 7b45 7bb2 85d3 861a 8d7b 8de3 9291 934c 94e3 9669 967a 96aa 97c5 9855 986f 9f38 9f74',
 '4Ye': '3416 35a1 35b6 35fc 366a 3763 3977 3a4e 3a91 3c49 3e23 420e 4933 4936 495f 4961 49e8 4b5f 4c9c 4e1a 4eb1 505e 50f7 5558 58b7 591c 5daa 5dab 5fe6 62fd 6396 64db 64ea 64eb 66c4 66c5 66d7 66f3 66f5 67bc 67bd 696a 696d 6b97 6b9c 6db2 6fb2 70e8 7160 71c1 7217 76a3 77b1 77b8 8449 8b01 8b7a 8c12 90ba 9134 9371 9391 9437 9765 9768 9801 9875 9923 9941 998c 9a5c 9d7a 9e08',
 '3Tan': '3cad 41b1 43d9 47a1 4994 55ff 5766 5fd0 61b3 61bb 6bef 74ae 79ab 83fc 8892 8962 8d09 9193 9eee',
 '3Tao': '46af 46b5 4d5a 8a0e 8ba8',
 '4Long': '345d 36de 37d6 3885 54e2 5fbf 6335 6887 7866 8856 8d1a',
 '1Chao': '5f28 600a 6284 6b29 7f7a 8a2c 8d85 9214 949e',
 '2Ping': '3eb8 3ec2 4348 4d84 51ed 51f4 546f 576a 5c4f 5c5b 5e21 5e32 5e48 5e73 617f 6191 67b0 6cd9 6d34 73b6 74f6 7501 7539 7aee 7bb3 7c08 7f3e 8060 80d3 8275 82f9 8353 840d 84f1 86b2 86e2 8a55 8bc4 8eff 90f1 9b83 9c86 9d67',
 '2Ze': '35bd 38f1 3cfb 40ce 41e5 4549 456a 4c39 4da6 5219 5247 5567 5616 5d31 5e3b 5e58 629e 62e9 6351 64c7 6ca2 6cfd 6ead 6fa4 769f 7794 7b2e 7ba6 7c00 802b 8234 835d 8434 8808 880c 8ace 8b2e 8cac 8cfe 8d23 8d5c 8fee 9c02 9c97',
 '2Sui': '3d66 3edf 4714 48ab 9040 968b 968f 96a8',
 '2Za': '4579 4799 4a9e 507a 56d0 5dbb 6742 7838 78fc 894d 96d1 96dc 96e5 97f4 9b73',
 '1Hui': '54b4 5645 5655 5a4e 5e51 5fbd 6062 62fb 6325 63ee 649d 6656 6689 6d03 7008 7070 70e3 7147 7988 7fda 7fec 8886 8918 8a7c 8bd9 8c57 8f1d 8f89 9693 96b3 986a 9c34 9ebe',
 '4Yong': '37b2 3db2 4f63 7528 82da 919f',
 '1Ting': '5385 539b 542c 5722 5e81 5ef0 5ef3 686f 6c40 7d8e 8013 8035 8074 807c 807d 827c 9793',
 '1Tun': '541e 5451 554d 564b 65fd 66be 671c 6d92 711e',
 '1Xi': '35e9 4080 4fd9 5092 50c1 50d6 516e 51de 5365 5380 5438 550f 553d 563b 564f 5915 595a 5a2d 5a90 5b06 5b09 5c56 5d60 5db2 5dc7 5e0c 5f86 5faf 5fda 6037 6038 6053 606f 6089 6095 60c1 60dc 6271 6278 637f 643a 64d5 6614 665e 6670 66e6 6790 6878 69bd 6a28 6a40 6b37 6b56 6b59 6c25 6d60 6dc5 6eaa 6f5d 70ef 7108 7155 7188 7199 71b9 71ba 71bb 71e8 7214 727a 7280 72a7 740b 74d7 7699 774e 7773 77a6 7852 7a00 7ab8 7c9e 7ce6 7e18 7e65 7fb2 80b9 819d 823e 8383 83e5 84b5 8725 8785 8787 87cb 8835 897f 8980 89f9 89fd 89ff 8b46 8c3f 8c40 8c68 8c6f 8c95 8e4a 9145 91af 93ed 9474 96b5 96df 9a31 9a68 9d57 9eca 9f37 fa15',
 '4Shang': '4e04 4e0a 59e0 5c19 5c1a 8830 9284 945c',
 '2Nian': '412d 4139 4a5e 4b2f 5e74 62c8 79ca 79e5 7c98 9b8e 9bf0 9c87 9cb6 9ecf',
 '3Yin': '3406 3865 396f 397c 39a9 39c8 3c43 3f99 41d9 4325 44a1 493a 4a38 4e5a 5c39 5dbe 5ef4 5f15 6a83 6afd 6dfe 6fe6 763e 766e 78e4 7d16 7e2f 7ebc 861f 8693 87be 8b94 8d9b 920f 9690 96a0 96b1 9777 98ee 98f2 996e 99bb',
 '2Zu': '37ba 3c35 3c9e 4178 469d 4bff 4c63 5346 5352 54eb 5d12 5d2a 637d 65cf 7a21 8db3 8e24 8e3f 8e75 9390 93c3 955e',
 '3Che': '3a0b 3d54 42f2 47a3 4c29 5056 626f 64a6',
 '2Qian': '39ae 3a1c 3a6e 3e2b 406e 4224 4349 456d 458d 4a3f 4ef1 524d 5a8a 5c92 5ff4 6272 62d1 63ae 63f5 69a9 6b6c 6f27 6f5b 6f5c 6ff3 704a 71c2 7b9d 8465 8654 9210 9257 92ad 9322 94a4 94b1 94b3 9a1a 9b35 9c2c 9ed4 9eda',
 '2Qiao': '3681 376f 4009 4397 4a4c 4c41 4e54 4fa8 50d1 55ac 563a 5da3 6194 6865 6a35 6a4b 71c6 7644 77a7 785a 7904 7fd8 7ff9 835e 854e 85ee 8b59 8c2f 8dab 8dac 9408 97bd 9866',
 '2Neng': '3d30 43fb 5b23 80fd 85b4',
 '2Mang': '3751 37cc 37ff 385b 3eca 402e 4152 420d 47e5 4d68 5402 54e4 5a0f 5c28 5fd9 607e 6757 6767 6c13 6c52 6d5d 7264 727b 72f5 75dd 76f2 76f3 786d 7b00 8292 832b 8609 86d6 9099 91ef 92e9 94d3 99f9 9f06',
 '1Que': '7f3a 849b',
 '3Geng': '3f98 42c1 4304 54fd 57c2 5cfa 632d 6897 7d86 7ee0 803f 8384 90e0 9abe 9bc1 9ca0',
 '4Man': '35c8 3862 3b05 3d58 4555 4761 4762 486c 50c8 5881 5ada 5e54 6162 66fc 6f2b 6fab 6fb7 71b3 734c 7e35 7f26 8504 8513 9124 93dd 9558 9b17',
 '4Mao': '34b5 34bb 384c 39cc 3a9e 3aef 3b98 3d18 3eba 3fde 4024 42c3 44ee 485a 4ac9 5190 5192 5aa2 5e3d 61cb 6693 67d5 6959 6bf7 7441 7683 770a 7780 7790 8004 8252 82bc 8302 843a 84e9 88a4 8c8c 8cbf 8d38 912e',
 '2Bie': '4825 4b71 5225 522b 8382 87de 8952 8e69',
 '3Chi': '361c 37c2 3881 388b 3c40 3ccf 3db4 42bc 445b 4735 473b 4f88 53fa 544e 5791 59fc 5c3a 6040 6065 62f8 640b 6b3c 6b6f 7c8e 803b 8687 88b3 88ed 892b 8a83 8c49 9279 9f52 9f7f',
 '4Wei': '395c 39a3 3dc9 45fd 4619 467f 471c 487a 4a8b 4b11 4b73 4bb9 4c81 4d73 4e3a 4f4d 536b 53de 5473 5582 5aa6 5b12 5c09 5fbb 6170 672a 6e2d 70ba 715f 71ad 7232 729a 72a9 732c 754f 7859 78d1 7ded 7f7b 80c3 82ff 83cb 851a 8589 85ef 8636 873c 875f 87b1 885b 885e 893d 8b02 8b86 8b8f 8c13 8d00 8e97 8e9b 8ece 8f4a 9317 93cf 9728 9927 9935 9b4f 9b87 9cda',
 '4Mai': '423f 4611 4715 4a2b 4bae 4f45 52a2 52f1 5356 58f2 773f 8108 8109 8750 8ce3 8fc8 9081 9722 9ea5 9ea6',
 '1Hun': '5a5a 60db 60fd 660f 662c 68d4 6b99 6dbd 7767 776f 8364 8477 95bd 960d',
 '4Wen': '4f06 554f 598f 6286 63fe 6435 6c76 6e02 74ba 7d0a 83ac 95ee',
 '4Juan': '38a7 38be 3abb 3bde 4105 4338 45ad 4688 4853 4cea 5026 52b5 52cc 52ec 5377 5dfb 5e23 617b 684a 6dc3 72f7 7367 74f9 7737 774a 7760 7d79 7ee2 7f65 7f82 9104 9308 9529 97cf 98ec 9b33',
 '5Guo': '556f',
 '4Zhao': '387d 3a04 3dd6 40cd 4207 435c 436e 4b93 5146 53ec 5797 65d0 66cc 679b 6ac2 70a4 7167 71f3 72e3 77be 7b0a 7b8c 7f69 7f84 8081 8087 8088 8a54 8bcf 8d75 8d99 96ff 9d6b',
 '1Lang': '5577',
 '3Bei': '3933 5317 9273',
 '4Tie': '4d34 4d7f 86c8 98fb 992e',
 '3Ba': '378e 628a 9200 94af 9776',
 '2Meng': '3679 3813 3a5a 4004 41c7 425a 4443 4445 4490 44dd 45c8 4666 4669 4913 4c12 4c9b 4d0c 4d3f 4d46 511a 51a1 5e6a 61de 66da 6726 6a57 6aac 6c0b 6fdb 7374 750d 753f 76df 77a2 77c7 77d2 791e 8268 82ce 8394 840c 8420 8499 8544 867b 8771 9133 9138 96fa 9740 995b 9bcd 9e0f 9e72',
 '4Niao': '3799 3cee 5c3f 8132',
 '3Ben': '3877 3bba 5932 5959 672c 694d 755a 82ef',
 '1Hua': '54d7 5629 82b1 82b2 9335 9de8',
 '3Zen': '600e',
 '3Dia': '55f2',
 '4Tiao': '6713 773a 7ab1 7c9c 7cf6 7d69 8101 899c 8df3',
 '2Cang': '3d74 3d93 85cf 9476',
 '3Guan': '43d3 45c6 463e 498e 4a6a 4a80 4c98 742f 75ef 7b66 7ba1 8118 8218 8f28 9327 9928 9986 9ce4 fa2c',
 '1Piao': '560c 5f6f 647d 65da 6f02 7ff2 85b8 87b5 98c3 98c4 98d8 9b52',
 '2Ceng': '3b1d 406c 4255 5c42 5c64 5d92 66fe 7880 7af2 912b',
 '4Gou': '35d5 370c 3745 3764 3a0c 3cf6 40d3 476d 4780 508b 5193 5778 57a2 591f 5920 59e4 5abe 5f40 6406 6480 69cb 7179 89af 89cf 8a3d 8a6c 8bdf 8cfc 8d2d 9058 96ca 9aba',
 '4Huang': '3a2a 66c2 76a9 8ee6',
 '2Er': '3587 3728 39eb 3b95 42e9 439f 43a0 4b98 4f95 513f 5150 5152 5532 682d 6abd 6d0f 800c 800f 804f 80f9 834b 88bb 8f00 8f5c 9651 96ad 9af5 9b9e 9c95 9d2f 9e38',
 '3Nen': '9f73',
 '3Nei': '3f0f 4c8e 5a1e 812e 8147 9912 9981 9bbe 9bd8',
 '3Pian': '8991 8ade 8cb5',
 '3Piao': '3a60 3d71 3e7e 6b8d 72a5 76ab 779f 7e39 7f25 8198 8508 91a5',
 '1Jiang': '50f5 58c3 59dc 5c06 5c07 646a 6a7f 6bad 6c5f 6d46 6f3f 74e8 7555 757a 7585 7586 7913 7e6e 7f30 7fde 8333 8441 8591 8780 87bf 8c47 97c1 9c42 9cc9',
 '3Ming': '37f0 3ae5 51d5 59f3 614f 7733',
 '3Ken': '5543 57a6 58be 6073 61c7 808e 80af 80bb 8c64 9339 9f57 9f66 9f82 9f88',
 '1Shuang': '53cc 5b40 5b47 6b06 7935 826d 96d9 971c 9a66 9aa6 9dde 9e18 9e74',
 '1Xun': '52cb 52db 52f2 52f3 5743 57d9 5864 58ce 58e6 66db 7104 718f 71fb 736f 77c4 7e81 81d0 8512 85ab 85b0 860d 91ba',
 '2Xing': '3429 34dd 38dc 3f1b 48c6 492f 4f80 5211 5753 578b 5a19 5f62 6d10 6ece 6ff4 70c6 784e 7b55 80fb 884c 90a2 90c9 9203 9276 9292 92de 9498 94cf 9649 9658 96fd 9933 9967',
 '2Wen': '43b9 43bd 4607 4c1a 5f63 6587 7086 73f3 741d 7612 7d0b 7eb9 805e 82a0 8689 868a 87a1 87c1 95ba 95bf 95c5 95e6 95fb 960c 96ef 99bc 9b70 9cfc 9d0d 9d16 9f24',
 '3Quan': '428e 545f 5946 6c71 72ac 7404 753d 754e 7da3 7e13 7efb 8647 8a43',
 '1Ao': '51f9 67ea 688e 8eea',
 '3Wang': '34c1 3cbf 3cf9 3d0f 42c4 42de 4c23 5f80 5f83 5ff9 60d8 6680 6789 68e2 7007 7db2 7f51 7f52 7f54 83f5 86e7 8744 8ab7 8f1e 8f8b 8fec 9b4d',
 '2Nuo': '345a 352e 3c19 50a9 513a 5a1c 632a 637c 689b 90cd',
 '1Xue': '524a 5419 5c75 8486 859b 8fa5 9774 97be',
 '2Wei': '3552 38f2 39d1 4260 444a 453a 4705 4750 4969 49e6 552f 56d7 56f2 56f4 570d 5729 58dd 5cd7 5cde 5d6c 5e0f 5e37 5e43 60df 6845 6b08 6ca9 6d08 6da0 6e4b 6ea6 6f4d 6f7f 6ff0 7022 741f 7653 77c0 7dad 7ef4 848d 853f 8587 89b9 8fdd 9055 912c 9180 934f 95c8 95f1 973a 97cb 97e6 9ba0',
 '1Kou': '527e 5f44 62a0 6473 82a4 88a7',
 '4Lin': '3581 426e 45f2 4af0 541d 6061 608b 6a49 711b 7ca6 853a 85fa 8cc3 8e78 8e8f 8e99 8eaa 8f65 95b5',
 '3Ya': '3fff 4ab5 538a 54d1 5516 555e 5e8c 75d6 7602 8565 96c3 96c5',
 '4Nang': '3484 513e 9f49',
 '2Li': '34bf 34ef 381f 3992 3c00 3c1a 3d1d 3df0 3e48 3fdb 411c 417b 426b 428d 42e5 4360 4366 4506 4523 4527 45a5 45bd 45bf 4670 48d3 48eb 4c58 4d3b 4d53 4d69 5215 5253 527a 5299 5398 55b1 569f 56c4 5ae0 5b4b 5b77 5ef2 60a1 6521 68a8 68a9 68b8 68c3 6a06 6f13 6f26 7055 7281 7282 729b 72f8 7483 74c8 775d 79bb 7a72 7bf1 7c6c 7c8d 7c9a 7cce 7e2d 7f21 7f79 8243 83de 84e0 853e 85dc 863a 870a 87cd 8935 8b27 8c8d 908c 91a8 91d0 92eb 9305 93eb 9457 96e2 9a39 9a6a 9a8a 9bec 9c7a 9ca1 9d79 9e1d 9e42 9ece 9ee7',
 '3Ye': '3652 4e5f 51b6 5414 5622 57dc 58c4 6f1c 91ce',
 '3Jie': '59d0 5a8e 6a9e 89e3 89e7 98f7',
 '4Wang': '46c3 4911 5984 5fd8 65fa 671b 6722 8fcb',
 '3Yi': '3565 382f 3a58 3aca 3c1d 3c3b 475d 49e7 4c19 4e59 4ee5 501a 506f 5d3a 5df2 5ea1 6246 6261 639c 653a 65d1 65d6 6672 6818 6905 6aa5 77e3 7912 7b16 8094 80e3 8223 8264 82e1 82e2 8681 8798 87fb 886a 8f22 8f59 8fc6 8fe4 9018 91d4 926f 9487 94f1 9857 9ce6 9f6e',
 '2La': '3547 63e6 908b',
 '1Lun': '6384',
 '1Zhuo': '3441 502c 5353 6349 684c 68f9 6dbf 7a5b 7a71',
 '1Zhun': '5b92 68c6 7a80 80ab 8860 8ac4 8c06 8fcd',
 '4Rui': '34f9 36f1 3aab 3c8a 433c 44f2 53e1 58e1 6798 6c6d 745e 777f 82ae 868b 8739 92b3 92ed 9510',
 '1Wei': '502d 504e 5371 55b4 5a01 5a81 5a99 5d34 5dcd 5fae 6104 63cb 6933 6972 6e28 6ebe 70d3 7168 71f0 78a8 840e 8468 8473 875b 89a3 8a74 9036 9687 9688 9c03 9c04 9cc2',
 '3Chuai': '3a9c 63e3',
 '3Chuan': '3c5b 50e2 5598 583e 8168 821b 8348',
 '3Yu': '3468 3481 3531 3651 3762 3818 3870 38c3 3cbe 3e84 3f0c 48c1 494f 4a1e 4e0e 4f1b 4fc1 4fe3 504a 50b4 5673 5704 5709 5b87 5bd9 5c7f 5cff 5dbc 5ebe 61d9 6554 6594 659e 6940 7440 7610 7964 79b9 7ab3 7bfd 7fbd 8065 8207 842d 84f9 860c 87a4 8a9e 8bed 8c90 9105 9151 96e8 9f6c 9f89 fa1e',
 '4Run': '3808 499e 6a4d 6da6 6f64 81b6 958f 95a0 95f0',
 '2Lu': '3820 38b3 3aad 3b54 3c7a 3fd6 484e 4b89 4c15 5362 56a7 5786 58da 5e90 5eec 650e 66e5 680c 6ae8 6cf8 7018 7089 7210 7379 7388 74b7 74d0 76e7 77d1 7c5a 7e91 7f4f 80ea 81da 826b 82a6 8606 8826 8f64 8f73 946a 9871 9885 9ad7 9b72 9c78 9c88 9e15 9e2c 9ef8',
 '4Liu': '3640 3a28 3daf 3f4c 4102 516d 5774 586f 5ec7 6f91 7542 78c2 7fcf 9402 96e1 9724 993e 998f 9b38',
 '4Ning': '38f7 3fe6 452d 4f5e 4fab 6cde 6fd8',
 '3Pin': '54c1 6980',
 '1Ju': '3790 51e5 521f 5a35 5c45 5ca8 5d0c 6285 62d8 6910 6a8b 6cae 6dba 72d9 741a 75bd 7717 7f5d 8152 824d 82f4 83f9 8445 871b 88fe 8d84 8dd4 8e19 92e6 9671 96ce 97a0 99d2 9a79 9b88 9d21 9d8b',
 '1An': '4f92 5a95 5b89 5cce 5cd6 5eb5 6849 6c28 75f7 76e6 76eb 8164 83f4 843b 844a 84ed 8a9d 8af3 8c19 94f5 978c 978d 99a3 9d6a 9d95 9e4c',
 '3Zheng': '4855 6138 628d 62ef 639f 649c 6574 6c36 7cfd',
 '4Kai': '3c89 4877 52d3 5ffe 6112 613e 6b2c 708c 708f 70d7 791a',
 '2Nie': '386a 82f6',
 '4Miao': '5999 5e99 5ebf 5edf 7385 7ad7',
 '2Ke': '54b3 58f3 6bbc',
 '4Kan': '400d 4613 5888 770b 77b0 77d9 78e1 7af7 884e 95de 961a',
 '4Kao': '4427 7292 92ac 94d0 9760',
 '4Po': '36d8 4138 438a 479f 48ea 48ee 4a30 4a96 4bd9 5cb6 5ef9 6734 7087 70de 72db 73c0 7834 7836 7c95 84aa 8feb 9166 9197 91b1 91d9 948b 9b44',
 '2Nin': '391b 42fb 46be 46d8 56dc 60a8',
 '4Ben': '3913 3a27 3c75 4b31 5034 574c 6379 64aa 6873 7b28 8f3d 9029',
 '2Chou': '341c 36f6 393d 399e 3a36 3d5e 3fe7 41fa 42ad 4327 4337 44d3 46ec 4952 4c96 4ec7 4fe6 5114 568b 5b26 5e31 5e6c 60c6 6101 61e4 6826 6906 7118 71fd 7574 7587 7697 7a20 7b79 7c4c 7d2c 7d52 7da2 7ef8 83d7 85b5 88ef 8a76 8b8e 8b90 8e0c 8e8a 9167 916c 91bb 96d4 96e0',
 '4Tang': '47d6 6465 70eb 71d9 8d9f 940b 94f4',
 '3Sheng': '35c2 3f33 3faa 405e 4687 4abf 5057 7701 771a 7bb5',
 '4Sheng': '43b4 4789 5269 5270 52dd 5723 58ad 5d4a 665f 69ba 741e 76db 8056 80dc 8542 8cb9 8cf8',
 '3Zan': '3930 5127 5139 5646 5bc1 62f6 648d 6505 6512 6522 661d 685a 6caf 7938 79b6 8db1 8db2',
 '3Zao': '4c83 65e9 67a3 68d7 6fa1 74aa 7e70 7f32 85bb 85fb 86a4',
 '4Chuo': '369f 3c8b 4090 40d7 412a 41af 41cd 42d8 4373 44ce 4b95 555c 56bd 5a16 5a65 60d9 64c9 6b60 6db0 73ff 7577 78ed 7dbd 7e5b 7ef0 814f 8ac1 8da0 8f1f 8f8d 8fb5 8fb6 9034 916b 918a 9461 9f6a 9f8a',
 '3Zai': '37a8 3c70 3d13 43c1 48ec 4ba8 5bb0 5d3d 7e21',
 '3Ru': '39ba 4e73 4f9e 64e9 6c5d 8097 8fb1 94f7',
 '1Ceng': '66fd',
 '4Deng': '482c 4bb4 50dc 51f3 58b1 5d9d 6ac8 77aa 78f4 8e6c 9093 9127 9419 96a5 972f',
 '4Song': '356c 3bb8 46e6 48a0 5b8b 8a1f 8aa6 8bbc 8bf5 9001 980c 9882 9938',
 '1Sun': '5b59 5b6b 640e 69c2 72f2 733b 836a 84c0 8575 859e 98e7 98f1',
 '1Ai': '54c0 54ce 5509 57c3 6328 8bf6 92b0 9384 953f',
 '2Ning': '3755 3cb0 45ff 4b62 511c 51dd 549b 5680 5b81 5bcd 5bd5 5bd7 5bdc 5be7 62e7 64f0 67e0 6ab8 72de 7370 752f 804d 8079 944f 9b21 9e0b',
 '1Yang': '4f52 5489 592e 59ce 67cd 6b83 6cf1 770f 79e7 7d3b 80e6 9260 9d26 9e2f',
 '2Rui': '3b83 4151 7524 7dcc 8564',
 '5De': '7684',
 '1Zhu': '4f8f 52af 6731 682a 69e0 6a65 6ae7 6aeb 6d19 6f74 7026 732a 73e0 7843 7969 79fc 7d51 8331 86db 876b 8829 88be 8a85 8af8 8bdb 8bf8 8c6c 8de6 90be 9296 94e2 9ba2 9bfa 9d38 9f04 fa16 fa22',
 '3Rong': '3c9d 4887 5087 5197 5b82 6411 6c04 8ef5',
 '3Gong': '3928 39ec 3ad2 3b5f 3eac 3f26 40ac 4857 5de9 5efe 62f1 62f2 6831 6c5e 73d9 7a6c 86ec 92be 92db 978f',
 '1Zhi': '4e4b 536e 5431 5767 588c 5b02 5df5 6418 652f 679d 67b3 6800 6894 6925 69b0 6c41 6c65 75a7 77e5 7957 796c 79d3 79d6 79ea 7d95 7e54 7ec7 80a2 80d1 80dd 8102 829d 8718 887c 96bb 99b6 9cf7 9d32 9d44 9e31 9f05',
 '1Xing': '5174 57b6 60fa 661f 66d0 714b 7329 7446 76a8 7bc2 8165 8208 86f5 89ea 89f2 935f 99ab 9a02 9b8f 9bf9',
 '1Zhe': '55fb 5aec 6663 906e',
 '1Huo': '5290 8c41 9343 952a',
 '1Zha': '5067 5273 54f3 55b3 624e 62af 6313 63f8 6463 67e4 6942 6a1d 6e23 7339 76b6 76bb 89f0 9f44 9f47',
 '1Du': '53be 561f 7763 90fd 918f 95cd 9607 fa26',
 '3Ku': '41e2 82e6',
 '1Jia': '4e6b 4f73 50a2 52a0 5609 5939 593e 5bb6 5e4f 62c1 67b7 6be0 6cc7 728c 7333 73c8 75c2 7b33 801e 8175 846d 8888 8c6d 8c91 8dcf 8fe6 926b 93b5 9553 9d10 9e9a',
 '1Zan': '5142 7c2a 7c2e 9415',
 '1Zao': '50ae 7cdf 8e67 906d',
 '1Jie': '5588 55df 5826 5dc0 63a5 63b2 63ed 64d1 6904 6e5d 744e 7686 79f8 7a2d 8149 83e8 85a2 8754 8857 8b2f 9636 968e 9d9b',
 '4Hang': '47d8 48c8 49b3 4d02 9150',
 '2Cheng': '37bc 3a03 3c82 3f29 404e 4107 4191 41b5 41b8 41f8 4532 45ca 49d5 4ac6 4baa 4e1e 4e57 4e58 5448 57ce 57d5 5818 584d 5856 5a0d 5bac 5cf8 60e9 6195 61f2 6210 627f 6330 6381 63e8 67a8 68d6 6909 6a59 6d06 6d48 6e5e 6f82 6f84 7013 73f5 73f9 7a0b 7a9a 7b6c 7d7e 8100 812d 837f 8670 88ce 8aa0 8bda 90d5 9172 92ee 94d6 9a2c',
 '1Jin': '4eca 57d0 5d9c 5dfe 60cd 65a4 6d25 73d2 7467 77dc 7972 7b4b 7d1f 8355 83f3 887f 895f 89d4 91d1 9485 9e76 9ec5',
 '4Ou': '36cf 4302 4314 6004 616a 6ca4 6f1a',
 '3Ke': '37b9 3a83 3a99 3aa1 3abc 3c24 3d63 53ef 5777 5ca2 5db1 6564 6e07 6e34 70a3 790d',
 '2Chong': '34fd 3e50 432c 459d 4cef 5d07 5d08 6f34 75cb 8327 866b 8769 87f2 8908 9680',
 '3Shang': '4b15 6244 664c 8cde 8d4f',
 '2A': '55c4',
 '1Jiu': '4e29 52fc 557e 63c2 63ea 63eb 673b 6a1b 725e 7a76 7cfa 7cfe 7ea0 841b 89d3 8f47 9194 9604 9b0f 9b2e 9ce9 9e20',
 '3Zeng': '3f6a',
 '1Shuai': '5b48 6454 7f1e 8870',
 '1Wo': '5529 6da1 6db9 6e26 7327 7a9d 7aa9 83b4 8435 8e12',
 '1A': '5416 9312 9515 963f',
 '3Qiu': '3f12 47ad 641d 7cd7',
 '4Suo': '9024',
 '4Sui': '34b8 37b8 3d1a 3eea 3efd 4157 424c 4341 4539 4814 4875 4959 4e97 57e3 5b18 5c81 5d57 65de 6a85 6a96 6b72 6b73 6fbb 716b 71e7 74b2 74cd 775f 7815 788e 795f 79ad 7a42 7a57 7a5f 7e40 7e50 7e78 895a 8ab6 8b62 8c07 8ce5 9042 9083 9406 9429 96a7',
 '3Zhua': '722a',
 '2Chang': '364a 3982 45c5 4806 4bf4 4ee7 507f 511f 514f 5617 5690 573a 5834 5872 5ae6 5c1d 5e38 5f9c 747a 74fa 751e 80a0 8178 8193 82cc 8407 88f3 92ff 93db 9577 9578 957f 9c68 9cbf',
 '3Zhuo': '4e35',
 '3Zhun': '51c6 57fb 6e96 7da7 96bc 9dbd',
 '2Heng': '3530 3d87 4113 449b 4b16 4b1d 4bd2 59ee 6046 6052 63d8 6841 697b 6a2a 6a6b 73e9 8605 8861 8a99 9404 9445 97f9 9d34 9e3b 9ec9 9ecc',
 '3Sun': '353c 3844 398f 405a 635f 640d 69ab 7b0b 7b4d 7bb0 7c28 93a8',
 '4Fa': '36f2 44a5 73d0 743a 855f 9aea 9aee',
 '1Wa': '52b8 54c7 55d7 5a32 5aa7 5f8d 6316 6432 6528 6d3c 6e9b 6f25 7556 7a75 7a8a 7aaa 8049 86d9 9f03',
 '4Jing': '354b 38a3 38cf 3b0c 3d7e 3e75 4514 4856 4d5e 4fd3 501e 50b9 51c0 51c8 52b2 5883 598c 5a59 5a67 5f2a 5f33 5f84 5f91 64cf 656c 66d4 6871 68b7 6d44 6de8 6fea 701e 734d 75c9 75d9 7ade 7adf 7ae7 7aeb 7af6 7af8 80eb 811b 845d 8aa9 8e01 8ff3 9015 93e1 955c 9753 9756 9759 975a 975c fa1c',
 '3Qie': '4e14',
 '1Ling': '62ce 6624',
 '2Fang': '3903 57c5 59a8 623f 80aa 9632 9b74 9c1f 9c82 9cd1',
 '4Ying': '345e 4050 466c 491d 565f 5ab5 5e94 646c 6620 668e 6ee2 7005 786c 81a1 84e5 8b4d 8ccf 93a3 941b 9795',
 '4Fu': '364f 3686 3931 3b3c 3cc7 3d57 3f6c 3f88 40a4 4385 4487 4600 4604 477e 479c 47af 47b8 47d4 4835 49a3 49de 4a31 4b78 4b9b 4bf1 4ed8 5069 5085 51a8 526f 5490 5711 577f 590d 5987 5a4f 5a66 5a8d 5b14 5bcc 5cca 5fa9 6991 7236 7954 7a2a 7d28 7dee 80d5 8179 842f 8567 86b9 86d7 875c 876e 889d 8907 8914 8986 8a03 8a42 8ba3 8ca0 8ce6 8cfb 8d1f 8d4b 8d59 8d74 8f39 9351 9362 961c 961d 9644 965a 99a5 99d9 9a78 9b92 9c12 9c8b 9cc6',
 '4Tou': '35a3 47ac 47dd 900f',
 '1Wu': '4e4c 526d 545c 55da 572c 5a89 5c4b 5deb 5f19 6747 6b4d 6c59 6c5a 6c61 6d3f 70cf 7466 7a8f 7bbc 815b 8790 8aa3 90ac 9114 93a2 9c1e 9d2e 9f3f',
 '3Gang': '37e0 37f5 3f58 4d1a 5c97 5d17 6e2f',
 '4Xiao': '3505 35db 390a 3e72 42a5 4567 4ff2 509a 52b9 54b2 5578 5628 562f 5a0e 5b5d 6054 6548 6569 6585 6586 6821 6b57 6d8d 71bd 7b11 8096 85e0 8a9f 97a9',
 '4Xian': '3535 387e 3993 3a48 3a87 3b17 3e8c 400f 4042 40f1 40f8 426f 43f9 4781 493c 49df 4a18 4a37 4c64 4d47 4d9f 4f23 4fd4 50e9 50f4 53bf 549e 54ef 57b7 59ed 5a0a 5baa 5c98 5cf4 61b2 648a 665b 6a4c 6d80 7017 732e 737b 73b0 73fe 770c 774d 7caf 7cee 7d64 7dab 7dda 7e23 7ebf 7f10 7fa1 7fa8 817a 81d4 81fd 82cb 83a7 86ac 8706 88a8 8aa2 8c4f 8f5e 92e7 930e 9650 9665 9677 9730 9921 9985 9eb2',
 '3Weng': '3622 3732 4425 4930 52dc 5855 5963 66a1 6d7b 6ec3 7788 806c 84ca',
 '3Tang': '3489 3fe9 4f16 5018 5052 50a5 513b 5e11 6203 66ed 6dcc 7223 77d8 8025 8eba 93b2 9482 954b',
 '2Fu': '3695 3711 37ca 3805 3914 3a84 3ad9 40fd 42f9 433f 4356 4467 4480 4530 454e 4620 479e 47ee 484d 4a17 4a99 4d57 4d7e 4e40 4f0f 4fd8 51b9 51eb 521c 5310 5488 56a9 5798 5b5a 5caa 5dff 5e17 5e45 5e5e 5f17 5f7f 602b 6255 6276 62c2 670d 678e 67eb 683f 6874 68f4 6c1f 6d11 6d6e 6daa 6f93 70a5 70f0 73b8 7408 7536 7549 7550 7641 7953 798f 7b26 7b30 7b99 7cb0 7d31 7d3c 7d65 7d8d 7e1b 7ec2 7ecb 7f1a 7f58 7f66 7fc7 8240 8274 8299 82a3 82fb 8300 832f 83a9 83d4 844d 8659 86a8 8709 8760 88b1 8ae8 8e3e 8f3b 8f90 90db 9258 925c 97cd 97e8 98ab 9af4 9b84 9ce7 9cec 9d14 9d69 9d9d 9efb fa1b',
 '3Lou': '3ab9 587f 5d5d 5d81 6402 645f 750a 7bd3 7c0d',
 '1Cao': '64a1 64cd 7cd9 9135',
 '3Bie': '3fdc 765f 86c2',
 '4Fang': '653e 8dbd',
 '3Zou': '8d70 8d71',
 '4Zha': '45b3 47a2 4e4d 5412 548b 54a4 5bb1 643e 67f5 6805 69a8 6ea0 7079 70b8 75c4 781f 7c0e 81aa 86b1 8a50 8bc8 91a1',
 '3Shi': '355c 3e6c 3e77 44a8 49a0 4e68 4f7f 5158 53f2 59cb 5ba9 5c4e 77e2 7b36 8c55 9242 99db 9a76',
 '2Gui': '6f59',
 '2Fa': '3579 363a 41c5 48f9 4e4f 4f10 57a1 59c2 6830 75ba 781d 7b4f 7f5a 7f70 7f78 8337 85c5 95a5 9600',
 '3Shu': '3494 37ec 3bee 3cc6 3eff 4455 4796 4831 4a73 5a4c 5c5e 5c6c 6691 6f7b 7659 76e8 7f72 85a5 85af 85f7 8700 8969 9483 9ecd 9f20 9f21',
 '1Bi': '506a 5c44 6bf4 7680 7695 7a2b 8795 8c4d 903c 98b7',
 '2Zuo': '3e32 43b0 476b 47b0 6628 690a 79e8 7a13 7b70 8443 923c',
 '1Wang': '344c 5c22 5c23 5c29 5c2a 5c2b 6c6a',
 '1Mao': '732b 8c93',
 '1Zheng': '4e89 4f42 57e9 59c3 5ce5 5d1d 5d22 5f81 5fb0 5fb4 5fb5 6014 6323 6399 70a1 70dd 722d 72f0 7319 7665 7710 7741 775c 7b5d 7b8f 7bdc 8047 84b8 8acd 8be4 8e2d 9266 931a 93f3 94b2 94ee 9b07',
 '2Fo': '4ecf 4f5b 5772',
 '1Liao': '64a9 8e7d',
 '1Fei': '5561 5983 5a53 6249 6e04 7306 7dcb 7eef 83f2 871a 88f6 970f 975e 975f 98db 98dd 98de 99a1 9a11 9a1b 9be1 9cb1',
 '1Fen': '515d 5206 5429 5c8e 5e09 6610 6706 6c1b 71d3 780f 7d1b 7eb7 7fc2 82ac 886f 8a1c 915a 9216 96ab 96f0 9934 9959 9cfb',
 '3Shuang': '3f3d 45ee 4aea 587d 6161 6a09 723d 7e14 9a3b',
 '2Mei': '3dac 3eb3 4288 4359 44bd 4902 5445 5833 587a 5a92 5d4b 5fbe 6517 679a 6802 6885 6963 6973 69d1 6c92 6ca1 6e44 6e48 7164 7338 73ab 73fb 7442 7709 7742 7996 7bc3 8104 8122 815c 82fa 8393 847f 90ff 9176 92c2 9387 9545 9709 9da5 9e5b 9ef4',
 '3Kai': '3841 4057 4429 4493 51ef 51f1 5240 5274 5605 57b2 584f 607a 6137 6168 669f 6977 8488 8f06 9347 93a7 94e0 9534 95d3 95ff 98bd',
 '2Ao': '37fc 3802 3ff0 443f 46ab 495d 498b 4d45 53ab 55f7 55f8 5d85 5ed2 646e 6556 6ef6 71ac 720a 7352 7353 7488 78dd 7ff1 7ffa 8071 851c 87af 8b37 8b38 9068 93ca 93d6 969e 9a9c 9c32 9ccc 9dd4 9f07',
 '3Peng': '6367 6dce 768f',
 '3Ting': '42fc 4990 4d7a 4fb9 633a 6883 6d8f 70f6 73fd 753a 753c 8121 8247 8a94 92cc 94e4 9832',
 '2Ai': '3c6f 4839 4da3 5540 560a 5d66 6371 6571 6573 764c 7691 769a 9a03',
 '1Pa': '556a 593f 5991 7685 8225 8469 8686 8db4',
 '4Ka': '9ac2',
 '1Beng': '4f3b 5623 5d29 5d6d 794a 7db3 7e43 7ef7 958d',
 '1Reng': '6254',
 '3Yang': '3526 41e6 4446 448b 4b2c 4ef0 509f 517b 52e8 5771 5c9f 6143 61e9 62b0 6501 6c27 6c31 70b4 75d2 7662 8746 9785 990a 990b 99da',
 '1Qiong': '5314 712a 828e 928e',
 '1Peng': '5309 562d 6026 6072 62a8 6888 6f30 70f9 7830 7851 78de 8eef 959b 99cd',
 '4Xuan': '39e6 3cd9 4357 4629 476e 49ce 4a59 4a70 57cd 6030 6621 6965 6966 6ceb 6e32 70ab 7729 7734 78b9 7d62 7e3c 7e4f 7eda 8519 8852 8b82 8d19 9249 93c7 94c9 955f 9799 98b4 99fd',
 '3Nai': '4e43 5976 59b3 5b2d 5efc 6c16 7593 827f 8ffa',
 '3Zhe': '556b 7987 8005 8d6d 8e37 937a',
 '2Tuan': '3a5b 429c 5278 56e2 56e3 5718 6171 629f 6476 69eb 6f19 7cf0 93c4 9dfb',
 '3Zha': '3480 3878 3de2 42fe 4562 46fd 4d59 538f 7728 82f2 9b93 9c8a',
 '3Nao': '344e 36f4 3e81 4700 4727 5318 57b4 5816 5ad0 607c 60a9 60f1 7459 78af 8111 8166',
 '3Tian': '3401 35ad 3649 394f 413c 413d 440c 4804 48ef 4a44 553a 5fdd 60bf 666a 6b84 6ddf 7753 8146 8214 89a5 89cd 8cdf 932a 9369 9518 9766 9902',
 '3Tiao': '3e20 4837 5b25 5ba8 6640 7a95 8a82',
 '3Pang': '448d 55d9 802a 89ab',
 '3Zhi': '39fb 3bb9 3c9b 4169 41db 4324 43ba 46d7 4cc5 52a7 538e 53ea 54ab 5740 5741 5902 5e0b 6049 627a 62a7 6307 65e8 664a 683a 6b62 6c9a 6d14 6dfd 6ecd 75bb 780b 7949 7d19 7eb8 82b7 85e2 8967 8a28 8dbe 8ef9 8f75 916f 962f 9ef9',
 '5Me': '9ebc',
 '4Kuan': '492d',
 '3Zhu': '3d6d 4622 4c1e 4e36 4e3b 5631 56d1 58f4 5b4e 5b94 62c4 65b8 6b18 6e1a 6fd0 7151 716e 77a9 77da 782b 7f5c 8a5d 8caf 8d2e 967c 9e88',
 '3Lie': '47e9 54a7',
 '5Ma': '5417 55ce 561b',
 '4Kuang': '42af 4d43 5123 51b5 535d 5739 58d9 5cb2 61ec 61ed 65f7 663f 66e0 6846 6cc1 720c 7716 7736 77cc 77ff 7926 7d56 7e8a 7ea9 8cba 8d36 8e80 909d 913a 9271 945b',
 '2Zhao': '7740',
 '2Niang': '5a18 5b22 5b43',
 '2Luo': '3469 3f08 3f4b 42a8 4bc1 5138 6924 6b0f 6c07 6c0c 7321 7380 7ba9 7c6e 7f85 841d 8502 863f 87ba 8999 89b6 89bc 903b 908f 93cd 947c 9523 9559 9960 9a3e 9a58 9aa1 9e01',
 '2Lun': '35ae 3dcd 4201 4473 4ed1 4f26 4f96 502b 56f5 5707 5a68 5d18 5d19 60c0 62a1 6ca6 6dea 6ea3 7db8 7eb6 8140 83d5 8726 8e1a 8f2a 8f6e 9300 966f 9be9',
 '1Qi': '4e03 501b 50db 51c4 552d 5601 5884 59bb 5a38 60bd 617c 617d 621a 6532 671f 67d2 6816 6864 687c 68f2 69bf 6b3a 6c8f 6dd2 6f06 78ce 78e9 7dc0 840b 8ac6 90ea 970b 9863 9b4c 9d88 9e02',
 '5Ne': '5436',
 '1Die': '7239 8dcc',
 '2Hui': '4f6a 56d8 56de 56ec 5efb 5efd 605b 6d04 75d0 8334 8698 86d4 86d5 8716 8ff4 9025 9bb0',
 '1Kao': '5c3b 9adb 9df1',
 '1Za': '531d 5482 5e00 62b8 6c9e 7d2e 8fca 9254',
 '2Zhai': '386f 5b85 790b',
 '4Kang': '3c20 4ea2 4f09 531f 56e5 6297 7095 72ba 780a 909f 9227 94aa 958c 95f6',
 '4Yi': '3439 344a 345c 3465 34f7 3534 3582 3601 360a 3660 366f 36a4 36d5 370b 3712 386b 387c 389e 38c2 38fb 3989 39a4 3c45 3c72 3cbc 3cd1 3d01 3d12 3d5d 3d69 3da0 3e53 3e6d 3f48 4101 413f 41bf 41e9 4268 42da 42f5 433b 4388 44c3 44c8 44f9 452c 454d 4581 45d1 45df 45f7 461d 4638 4758 476f 4883 48e7 49b4 4b25 4b42 4b47 4b5e 4b7f 4bc6 4c52 4d2c 4e42 4e49 4e84 4ea6 4ebf 4ee1 4f07 4f3f 4f5a 4f7e 4fcb 5104 517f 5208 5293 52ae 52da 52e9 533b 5453 546d 5479 5508 55cc 56c8 571b 57bc 57f6 57f8 58bf 5955 5ad5 5b11 5b1f 5bf1 5c79 5cc4 5da7 5e1f 5e20 5e46 5ed9 5f02 5f08 5f0b 5f79 5fc6 6008 603f 6092 60a5 610f 61b6 61cc 61ff 6291 62b4 6339 6359 648e 6561 6581 6613 6679 66c0 66ce 6759 678d 67bb 6827 68ed 698f 69f7 6a8d 6b2d 6b5d 6b94 6baa 6bb9 6bc5 6c49 6d02 6d42 6d65 6d73 6e59 6ea2 6f69 6fba 7037 7088 7132 71a0 71a4 71bc 71e1 71f1 7348 73b4 7570 75ab 75ec 7617 761e 7631 7654 76ca 776a 7796 79c7 7a53 7ae9 7dc6 7e0a 7e76 7e79 7ece 7f22 7f9b 7fa9 7fbf 7fca 7fcc 7ff3 7ffc 8084 808a 814b 8189 81c6 8257 827a 8285 82c5 84fa 858f 85d9 85dd 8619 8649 86e1 8734 87a0 88a3 88d4 88db 8939 897c 8a32 8a33 8a4d 8a63 8abc 8b6f 8b70 8b9b 8bae 8bd1 8be3 8c59 8c5b 8c77 8dc7 8efc 8f76 9038 9091 91b3 91b7 91f4 9220 93b0 943f 9552 9571 9623 96bf 972c 977e 97a5 9861 9950 99c5 9a5b 9a7f 9aae 9be3 9d82 9d83 9de7 9dfe 9ed3 9f78 fa17 fa25',
 '2Huo': '4106 4111 48f6 4bcf 4f78 6d3b 79ee 79f3 843f 94ac',
 '2Hun': '346e 3a21 46f0 4adf 4c1f 4d37 581a 5ff6 68a1 68de 694e 6d51 6e3e 73f2 743f 7e49 8f4b 9850 991b 992b 9984 9b42 9f32',
 '2Hua': '3572 37c6 380f 398a 3b49 3baf 417f 43e6 4522 4c7b 4da4 5212 534e 5b05 5d0b 6ed1 72ef 733e 78c6 83ef 8550 8796 8b41 91eb 93f5 94e7 9a4a 9a85',
 '3Sa': '348e 429b 6d12 7051 9778',
 '3Lia': '4fe9 5006',
 '2Hu': '35c5 3ab6 3bdb 3e71 3fb0 3fe5 406b 4238 4249 427f 4280 4381 452f 469b 47b1 4812 49fc 4a74 4b45 4b4c 4b4d 4bb8 5596 561d 56eb 58f6 58f7 58fa 5aa9 5f27 6287 6430 659b 695c 69f2 6e56 702b 7100 7173 72d0 7322 745a 74f3 7bb6 7cca 7d57 7e20 80e1 846b 851b 8774 879c 885a 89f3 9190 9378 9836 992c 9b0d 9b71 9c17 9d60 9d98 9da6 9e44 9e55',
 '4Ya': '375e 3c33 4149 475f 489d 49aa 4bc9 4c32 4d5d 4e9a 4e9c 4e9e 4ff9 5688 5720 57e1 5a05 5a6d 631c 6397 63e0 6c29 6c2c 7330 73a1 7811 7a0f 7aab 8050 897e 8a1d 8bb6 8ecb 8f67 8fd3 930f 941a 94d4 9f7e',
 '4Nao': '6dd6 9599 95f9 9b27',
 '4Nan': '5a7b',
 '3Chai': '831d',
 '2Ha': '86e4',
 '4Yu': '3828 3cda 3f63 404c 408a 41b7 4205 425b 42d6 435e 4587 4618 4631 46d5 48a9 4a12 4b04 4b87 4b99 4d01 4d6b 4ffc 54ca 5585 5590 55a9 55bb 564a 572b 57df 5809 59aa 5ad7 5bd3 5cea 5d8e 5ebd 5f67 5fa1 5fec 6086 6087 60d0 6108 617e 622b 6631 68db 68dc 68eb 6af2 6b0e 6b1d 6b25 6b32 6bd3 6d74 6de2 6def 6eea 6fa6 706a 7134 715c 71cf 71e0 7229 72f1 7344 735d 7389 7419 7609 7652 77de 7821 7872 7907 7916 791c 79a6 79d7 7a22 7a36 7a65 7c5e 7c72 7dce 7e58 7f6d 807f 8080 80b2 828b 828c 831f 842e 84ae 84e3 8577 8581 871f 872e 88d5 8a89 8aed 8b7d 8c15 8c6b 8ec9 8f0d 9033 9047 9079 90c1 91a7 923a 9289 92ca 9325 942d 94b0 95be 9608 96a9 9731 9810 9884 98eb 9947 996b 99ad 9a1f 9a48 9a6d 9b30 9b31 9b3b 9b4a 9cff 9d25 9d2a 9d52 9df8 9e46 9e6c 9ee6 9fa5',
 '4Pi': '3fd9 4440 4444 4818 485f 4928 4d19 50fb 5ab2 5ad3 5c41 63ca 64d7 6de0 6e12 6f4e 6fbc 6fde 7513 7914 8406 8b6c 8f9f 95e2 9dff 9e0a',
 '4Miu': '8b2c 8c2c',
 '2He': '34ed 3520 3561 397a 3a89 3b58 3b71 3b9d 3bab 3e47 3fe3 40d2 4142 44a9 4563 47a6 4897 4ad8 4cda 4d85 4f55 52be 5392 5408 548a 548c 54ec 555d 59c0 5ec5 60d2 6546 66f7 67c7 6838 6941 6bfc 6cb3 6db8 6ec6 6f95 7186 72e2 76c7 76c9 76cd 76d2 7909 79be 79f4 7bd5 7d07 7ea5 7fee 8020 8377 83cf 8402 849a 86b5 874e 879b 8910 8988 8a38 8a65 8c88 8c89 90c3 91db 924c 95a1 95d4 9602 97a8 981c 9932 9b7a 9c84 9da1 9e56 9f55 9f81 9fa2',
 '1Ni': '59ae',
 '3Tong': '36da 3a8c 41b9 59db 6345 6876 7b52 7b69 7d71 7d82 7edf',
 '4Shuai': '48a6 535b 54b0 5e05 5e25 7e42 87c0',
 '4Yun': '36ba 378c 37e6 468b 4a75 4cb0 4d74 56e9 593d 5b55 607d 60f2 6120 614d 679f 7185 71a8 7df7 816a 8570 8574 8580 85f4 860a 8fd0 904b 90d3 9106 915d 9196 919e 97de 97eb 97f5 97fb 9da4',
 '4Lv': '3527 3825 3cb6 451e 4896 4968 52f4 5bfd 5d42 5f8b 616e 6c2f 6ee4 6ffe 7387 7bbb 7da0 7dd1 7eff 819f 844e 8651 9462',
 '4Lu': '34d0 35a8 36ec 3719 37e4 3987 3a90 3a96 3afd 3bdd 3bdf 3f3e 40d9 4312 4391 43bc 4402 4635 4684 47ff 485c 4a6e 4c5a 4d2a 4f93 50c7 5279 52ce 52e0 5725 578f 5a3d 5ccd 5ed8 5f54 5f55 622e 645d 6902 6a1a 6dd5 6de5 6e0c 6f09 6f5e 719d 742d 7490 752a 76dd 7769 7849 788c 78df 797f 7984 7a11 7a4b 7b93 7c0f 7c2c 7c36 7c59 7cb6 8194 83c9 850d 8557 8642 87b0 89ee 89fb 8cc2 8d42 8da2 8def 8e1b 8e57 8f05 8f46 8f82 8f98 902f 9181 9304 9332 9334 93d5 93f4 9646 9678 9732 9a04 9a3c 9be5 9d66 9d71 9dfa 9e6d 9e7f 9e93',
 '4Tong': '6078 615f 6185 75db 8623 8855',
 '4Pin': '403b 725d 8058',
 '4Yue': '3727 3730 3b26 3c1b 3e4a 42d0 4583 47e0 482f 4847 4881 48b2 4926 4943 4db3 5216 599c 5c84 5cb3 5dbd 6071 6085 60a6 6209 6288 6373 6708 6782 6a3e 6ce7 7039 721a 72d8 73a5 793f 79b4 7bd7 7c46 7c65 7c70 7ca4 7cb5 8625 868e 868f 8d8a 8dc0 8dc3 8e8d 8ecf 9205 925e 94ba 95b1 95b2 9605 9e11 9e19 9fa0',
 '4Mei': '3b51 401b 424b 428a 4c28 4c2a 4d62 59b9 5a9a 5bd0 62ba 6627 6cac 715d 75d7 771b 7778 7959 875e 8882 8dca 97ce 9b3d 9b45',
 '3Bang': '3519 3b84 47fa 699c 7253 7d81 7ed1 8180',
 '4La': '3e0a 3ecb 3edd 40b0 40f3 43c0 44e5 45f6 4753 4a89 4c6b 4d9b 524c 63e7 650b 694b 7209 74ce 760c 814a 81c8 81d8 8721 874b 8772 881f 8fa2 8fa3 945e 9574 9b0e 9bfb f90e',
 '1Quan': '5708 570f 59fe 5f2e 609b 68ec 7b9e 7d5f 9409',
 '3Zhou': '43bb 459e 5e1a 666d 759b 776d 7b92 8098 83f7 9bde',
 '3Nv': '5973 7c79 91f9 9495',
 '4Le': '3539 3580 39a1 4e50 4ec2 52d2 53fb 54f7 5fc7 6250 634b 697d 6a02 6c3b 6cd0 738f 7833 7afb 7c15 827b 961e 97f7 9831 9c33 9cd3',
 '2Zhuo': '38ff 3aac 3b6c 3e9f 4175 4574 4d82 5262 53d5 5544 5545 5734 59b0 5f74 62d9 64af 64c6 64e2 6580 65ab 65ae 65b1 65b2 65b5 666b 68b2 68c1 68f3 6913 69d5 6ae1 6c4b 6cce 6d4a 6d5e 6fc1 6fef 7042 707c 70aa 70f5 712f 7422 7438 787a 799a 7be7 7c57 7c71 7ecc 7f6c 8301 8817 8ad1 8b36 8bfc 8e14 914c 92dc 9323 942f 9432 956f 9ddf',
 '4Li': '3466 34a7 350f 3578 35da 3611 37f3 3842 3921 3926 39f0 3b0f 3b9a 3be4 3c79 3ea1 3ece 3efa 3f16 3f41 3f5d 3f90 3fe8 407b 40ef 4144 41d0 42aa 437d 44de 4501 4509 4608 4695 47cf 47d0 4843 4919 4976 4b05 4b06 4b8b 4ba5 4c1b 4c1c 4c9e 4d04 4d21 4d98 4e3d 4f8b 4fd0 4fea 5088 512e 5137 51d3 5229 529b 52b1 52f5 5386 5389 53a4 53af 53b2 540f 5456 550e 5533 56a6 56c7 575c 585b 58e2 5a6f 5c74 5ca6 5dc1 60a7 60b7 6144 623b 623e 642e 650a 6526 652d 6584 66a6 66c6 66de 6738 67a5 680e 6817 681b 6835 68d9 6ad4 6adf 6aea 6b74 6b77 6ca5 6cb4 6d96 6ea7 6fff 701d 7204 720f 72a1 7301 73d5 740d 746e 74c5 74d1 74e5 75a0 75ac 75e2 7658 7667 76aa 76ed 77cb 7805 782c 783a 783e 78ff 792a 792b 7930 79b2 79dd 7acb 7b20 7b63 7be5 7c92 7c9d 7cf2 7d9f 7e85 8137 82c8 82d9 8318 8354 8385 8389 849e 85f6 86b8 86ce 86e0 8727 8777 8807 8823 882b 8a48 8b88 8d72 8dde 8e92 8f62 8f63 8f79 90e6 9148 925d 96b6 96b7 96b8 96f3 96f4 9742 974b 9b01 9b32 9c71 9c73 9ce8 9d17 9dc5 9dd1 9e97 9e9c',
 '3Hai': '6d77 70f8 80f2 917c 91a2',
 '2Nang': '4087 4e6a 56a2 56ca 6b1c',
 '1O': '5594 5662',
 '1Chuang': '41ab 5205 6227 6450 724e 7255 75ae 7621 7a93 7a97 7abb',
 '1Pang': '4e53 6ec2 8196',
 '1Gen': '522f 5246 6839 8ddf',
 '4Wa': '43b3 46b4 481a 55e2 5a60 6df4 817d 8183 889c 896a 97c8 97e4',
 '4Ke': '3549 3929 3fa7 4650 4d97 514b 523b 524b 52c0 52ca 55d1 578e 5801 5a14 5ba2 5c05 606a 6118 6119 63e2 6415 69bc 6c2a 6e98 78a6 7dd9 7f02 8849 8ab2 8bfe 9198 9a0d 9a92',
 '1Gu': '4f30 5471 5495 5903 59d1 5af4 5b64 67e7 6a6d 6cbd 6cd2 7b1f 7b8d 7b9b 7f5b 82fd 83c7 83f0 86c4 89da 8ef1 8f9c 9164 9232 94b4 9b95 9d23 9e2a',
 '3Kua': '57ae 6057 9299 951e',
 '2Yin': '3542 3597 366c 3759 37a4 3e12 3e5c 3e5e 44c4 459c 4aa9 5198 541f 552b 5656 569a 5701 57a0 5924 5a6c 5bc5 5d1f 5d2f 65a6 6aad 6ba5 6cff 6deb 6edb 70ce 72be 72fa 748c 7892 82c2 8376 8529 87eb 8a14 8a1a 8a21 8abe 911e 91ff 921d 9280 94f6 972a 9ccf 9de3',
 '1Shai': '3a44 7b5b 7be9',
 '3Lin': '342d 3a06 4572 50ef 51db 51dc 5ee9 5eea 61cd 61d4 649b 6a81 6aa9 6f9f 765b 765d 83fb',
 '2Hai': '373e 3a1f 483d 4bd0 4c7a 5b69 9826 988f 9ab8',
 '2Ti': '3592 3857 38e2 4160 4536 46a3 46f1 4770 48e1 4a11 4b3e 504d 5397 557c 55c1 5a82 5a9e 5d39 60ff 63d0 6f3d 7445 78ae 7994 79b5 7a0a 7d88 7df9 7ee8 7f07 7f64 8351 876d 8906 8b15 8da7 8e44 8e4f 9046 918d 9357 984c 9898 9a20 9bb7 9d5c 9d97 9d99 9e48 9f36',
 '1Liu': '6e9c 7198 8e53',
 '4Niang': '4586 917f 91b8 91c0',
 '2Tu': '3b78 3ecc 3ee0 3eec 3eef 4177 4598 4808 48c4 48dd 4905 4cdc 51c3 56f3 56fe 5716 5717 5857 5b8a 5c60 5cf9 5d5e 5ea9 5f92 6022 6348 63ec 688c 6d82 6e65 6f73 7479 760f 7a0c 7b61 816f 837c 8456 84a4 8dff 9014 9174 922f 934e 999f 99fc 9d5a 9d9f 9dcb 9df5 9f35',
 '4Chui': '9fa1',
 '4Ku': '3482 3838 4fc8 55be 56b3 5e93 5eab 7105 7614 77fb 79d9 7d5d 7ed4 88b4 88e4 8932 8db6 9177',
 '3Ou': '3496 3f34 4686 4bda 5076 5418 5878 8026 8162 8545 85d5',
 '3Ding': '3ac0 3d3f 3f17 5d7f 6a99 6fce 85a1 914a 9424 9802 9876 9f0e 9f11',
 '1Gang': '5188 51ae 521a 525b 5808 583d 5ca1 6386 6760 68e1 7268 7285 7598 7899 7b10 7db1 7eb2 7f38 7f41 7f53 7f61 809b 8221 91ed 92fc 93a0 94a2',
 '2Pai': '3d7a 4ff3 5f98 6392 68d1 724c 7305 7bfa 7c30 7c32 8f2b',
 '1Bian': '7178 7251 7335 7502 782d 7a28 7b3e 7baf 7c69 7cc4 7de8 7f16 8759 8fb9 8fba 9089 908a 937d 97ad 9bfe 9bff 9cca',
 '1Zhen': '4fa6 5075 583b 5a9c 5ac3 5bca 5e2a 63c1 6438 659f 6815 6862 686d 6968 699b 6a3c 6eb1 6f67 6fb5 7349 73cd 73ce 744a 7504 771e 771f 7827 78aa 78cc 796f 798e 799b 7bb4 7c48 7f1c 80d7 81fb 8474 8496 84a7 84c1 85bd 8c9e 8d1e 8f43 9049 9159 91dd 9241 9331 937c 9488 9755 99d7 9c75',
 '3Ruan': '3f31 3f6d 4782 4aad 5a86 670a 744c 74c0 76f6 789d 791d 7ddb 800e 815d 8761 8edf 8f2d 8f6f 962e',
 '3Lv': '3b5a 3ef2 3f94 4fa3 4fb6 507b 50c2 5122 5415 5442 5c61 5c62 5c65 6314 635b 65c5 68a0 6e87 6f0a 7963 7a06 7a5e 7a6d 7d7d 7e37 7f15 8182 8190 891b 8938 90d8 92c1 94dd',
 '1Diu': '4e1f 4e22 92a9 94e5',
 '1Xiong': '342b 5144 5147 51f6 5308 54c5 5ff7 605f 6c79 6d36 80f7 80f8 8a29 8a7e 8bbb 8cef',
 '1Zhua': '6293 631d 64be 6a9b 81bc 9afd',
 '4Huai': '54b6 574f 58ca 58de 5b6c 8639 863e 8ad9',
 '4Huan': '34c9 3555 3ab1 3b07 3e56 3f2b 4013 4028 41a0 4bd8 5524 559a 559b 57b8 5942 5950 5ba6 5e7b 60a3 610c 6362 63db 64d0 6899 69f5 6da3 6e19 6f36 70c9 7115 7165 744d 75ea 7613 8c62 8f58 902d 9c00',
 '1Ji': '4920 4e0c 4e69 50df 51e0 5209 520f 525e 52e3 53fd 54ad 5527 559e 55d8 5630 57fa 58bc 59eb 59ec 5c10 5c50 5cdc 5d46 5d47 64ca 671e 673a 6785 6a0d 6a5f 6ac5 6bc4 6fc0 7284 7391 74a3 7578 757f 77f6 78ef 79a8 79ef 7a18 7a3d 7a4d 7b04 7b53 7b95 7c0a 7ddd 7e3e 7ee9 7f81 7f87 7f88 802d 807b 808c 8640 866e 8940 8989 898a 89ed 8ac5 8b4f 8b64 8ba5 8ceb 8cf7 8d4d 8de1 8dfb 8e5f 8e8b 8eb8 8ff9 9288 9416 9447 9459 96ae 96de 97bf 97f2 98e2 9951 9965 9c3f 9cee 9d8f 9dc4 9e21 9ea1 9f4e 9f4f 9f51',
 '1Ping': '4e52 4fdc 5a09 7129 782f 9829',
 '4Tuan': '5f56 8916',
 '4Zhen': '34c4 38c0 3bb3 3be2 3d28 42b6 43d6 4769 47f4 4a2f 4cb4 4cf2 4fb2 5733 5866 630b 632f 63d5 6552 6576 6715 681a 7739 8704 8aab 8cd1 8d48 93ad 93ae 9547 9635 9663 9707 9d06 9e29',
 '3Tie': '496b 9244 9295 92e8 9421 9435 94c1 9507 9a56 9d29',
 '3Kuo': '64c3',
 '2Cong': '3f3b 4258 457a 4cf7 4e1b 4ece 53e2 5a43 5b6e 5f93 5f96 5f9e 60b0 6152 6dd9 6f0e 6f40 7047 721c 742e 8ab4 8ce8 8ce9 931d',
 '5Men': '4eec 5011',
 '1Tian': '500e 5172 5929 5a56 6c97 6dfb 915f 9754 975d 9ec7',
 '1Tiao': '4f7b 5ea3 604c 6311 65eb 7967 804e 84e8 9ba1',
 '2Cen': '37a5 3ef8 40e1 417e 4901 4a19 4bd4 4c8b 57c1 5c91 68a3 6a6c 6d94 7b12',
 '4Dui': '3642 37cb 381a 3b23 3cd4 3d7d 41cf 41e4 452a 4a34 4a3a 4b08 4b3d 4bdf 514a 514c 5151 5bf9 5bfe 5c0d 603c 619d 619e 61df 6fe7 7029 7893 794b 7d90 85b1 8b48 8f5b 931e 9413 941c 9566 961f 968a',
 '4Duo': '39f7 3ee7 415c 4374 4468 4643 465f 493b 4a54 5234 5241 5815 58ae 58af 5c2e 5d9e 60f0 67c1 67ee 6857 70a7 70a8 8235 8de5 8dfa 8e31 964a 964f 98ff 9973',
 '4Dun': '3b3f 491c 4f05 56e4 5d38 5e89 627d 6c8c 6f61 71c9 76fe 7818 8e32 9007 9041 906f 920d 949d 9813 987f 9da8',
 '3Ci': '383f 4f4c 6b64 6cda 73bc 7689 7d2a 8dd0',
 '3Gen': '4ac0',
 '3Du': '403e 421e 4417 5835 5e3e 668f 743d 7779 7b03 7be4 88fb 89a9 8ced 8d4c 9316',
 '1Cui': '50ac 55fa 5894 5d14 6467 69b1 69ef 7355 78ea 7af4 7e17 93d9',
 '4Zhuo': '5285',
 '1Cuo': '6413 64ae 7749 78cb 84ab 8e49 9073',
 '1Cun': '6751 6f8a 76b4 90a8',
 '4Liang': '4eae 54f4 55a8 60a2 667e 6e78 8ad2 8c05 8f0c 8f1b 8f86 91cf 9344',
 '4Men': '3943 3996 3c6a 3d4d 60b6 61d1 61e3 7116 71dc 95f7',
 '3Na': '4e78 54ea 96eb',
 '4Bu': '3635 36b4 3ccd 3ec9 3f9f 4287 434c 43fd 4470 4775 4b0f 4cdd 4d1d 4d3a 4e0d 4f48 5425 5498 57d7 57e0 5e03 6016 6091 6357 6b65 6b68 6b69 7bf0 7c3f 8379 8500 90e8 90f6 923d 949a 94b8 9922 9bac',
 '1Xiang': '342e 4e61 53a2 554c 5ec2 5fc0 6b00 6e58 74d6 76f8 7bb1 7dd7 7e95 7f03 81b7 8297 8459 858c 8944 90f7 9109 910a 9115 9472 9576 9999 9a64 9aa7 9e98',
 '4Feng': '385d 4ff8 51e4 5949 6e57 7128 7148 752e 8cf5 8d57 9cef 9cf3 9d0c',
 '4Bi': '34d6 35c9 3620 3629 3644 3840 3859 38b0 38b6 38b8 39d9 3aa4 3bbf 3bc7 3c78 3cfc 3d65 3d68 3e43 3eeb 3feb 4023 4079 4136 42a7 42d4 431f 43b5 43f6 4557 45a9 47c6 47e4 4998 49d7 4a5b 4a90 4ac1 4afe 4b1b 4b6e 4ba1 4bd7 4f56 5487 54d4 55f6 5752 581b 58c1 5970 59bc 5a62 5b16 5e01 5e63 5e87 5ee6 5f0a 5f3b 5f3c 5f43 5fc5 602d 610a 610e 655d 6583 6788 67f2 6890 6945 6a98 6bd5 6bd6 6bd9 6e62 6eed 714f 719a 72f4 7358 7359 73cc 74a7 7540 7562 75aa 75f9 75fa 7764 7765 78a7 7b5a 7b84 7b85 7b86 7be6 7bf3 7c8a 7e2a 7f7c 805b 8177 81c2 82fe 835c 84d6 84fd 853d 859c 870c 8890 88e8 895e 8963 89f1 8a56 8bd0 8cb1 8cc1 8d14 8d32 8d51 8df8 8e55 8e83 8e84 907f 90b2 9128 912a 924d 939e 93ce 9434 94cb 9587 9589 959f 95ed 965b 97b8 97e0 98f6 9946 999d 99dc 9a46 9ab3 9ac0 9b53 9b85 9c0f 9cbe 9d56 9ddd 9de9 9f0a',
 '1Zhui': '5a3a 690e 8ffd 9310 9525 96b9 9a05 9a93 9d2d 9d7b',
 '1Guang': '4f8a 50d9 5149 54a3 5799 59ef 6d38 706e 7097 709b 70e1 73d6 80f1 832a 8f04 92a7 9ec6',
 '1Ku': '5233 54ed 5710 5800 625d 630e 67af 684d 7a9f 80d0 8dcd 8ef2 8f71 90c0 985d 9ab7',
 '4Bo': '3595 5b79 6300 64d8 6a97 7588 7c38 7e74 8617 8b52',
 '4Zong': '435f 474b 500a 662e 7314 75ad 7632 7882 7cbd 7cc9 7ced 7e26 7e31 7eb5 7efc',
 '4Ba': '35a0 3d9a 40fb 4189 41d1 43ac 43b1 496f 4a57 4a7b 4c3e 4c5d 575d 57bb 58e9 5f1d 6b1b 705e 7238 7308 7f62 7f77 8019 8987 9738',
 '1Huang': '569d 5843 5ddf 614c 671a 8093 8352 8841',
 '2Hang': '391a 402a 4615 4cb3 542d 65bb 676d 6c86 80ae 822a 82c0 86a2 8ca5 8fd2 980f 9883 9b67',
 '2Die': '3459 3948 39b6 3a78 3a79 3afc 3b2a 3b6f 3cb2 3cb3 3df8 3ee1 43f2 462d 4787 4795 481f 4aa5 4ba2 4c80 4cc0 4d11 53e0 558b 57a4 581e 5ccc 5d7d 604e 621c 6315 6633 66e1 6c0e 7243 7252 74de 7573 7589 758a 7723 7730 789f 7d70 7ed6 8051 80c5 81f7 8253 82f5 8728 8776 890b 893a 8a44 8adc 8c0d 8d83 8dd5 8e40 8fed 957b 9c08 9cbd',
 '3Di': '3873 3a86 3b7d 40a1 4355 4891 48cc 4c43 5467 5758 57de 5e95 5f24 62b5 62de 638b 67e2 6c10 7274 7825 805c 82d6 830b 83e7 89dd 8a46 8bcb 8ee7 90b8 963a 9ab6',
 '3Li': '3e1a 3f96 45cd 491a 49c9 4fda 5163 5a0c 5cdb 5ce2 5cf2 674e 6b10 6b1a 6d6c 6fa7 7406 76e0 793c 79ae 7cb4 87f8 8821 88cf 88e1 8c4a 9026 9090 91b4 91cc 92f0 9502 9bc9 9c67 9ca4 9ce2 fa18',
 '4Tao': '3690 5957',
 '4Tan': '443a 4716 509d 50cb 53f9 5606 57ee 63a2 6b4e 6e60 70ad 78b3 8ce7 8d55',
 '4Die': '60f5 800a 800b',
 '1Song': '502f 51c7 5a00 5d27 5d69 5eba 61bd 677e 6780 67d7 68a5 6aa7 6dde 6fcd 7879 83d8 8719 9536 9b06',
 '1Xu': '5066 5401 5474 5618 5653 589f 5aad 5b03 5b2c 5e41 620c 63df 65f4 6647 6948 6b28 6b30 6b3b 6b54 6b58 759e 76f1 7a30 7e03 7e7b 80e5 8566 8657 865a 865b 8751 8a0f 8b43 9450 9700 9808 987b 9a49 9b1a 9b56',
 '2Long': '3685 376b 3863 3995 3c0d 40e7 418d 43ca 466a 4962 4a8a 5499 56a8 5c78 5d90 5dc3 5dc4 663d 66e8 6727 680a 6af3 6cf7 6e70 6edd 6f0b 7027 7216 73d1 74cf 7643 772c 77d3 783b 7931 7932 7abf 7adc 7b3c 7bed 7c3c 7c60 804b 807e 80e7 830f 856f 8622 882a 882c 8971 8c45 8e98 93e7 9468 9686 9733 9747 9e17 9f8d 9f92 9f93 9f99',
 '4Tai': '3477 396d 48ed 592a 5933 5ff2 6001 614b 66c3 6c70 6cf0 6e99 71e4 80bd 8226 915e 9226 949b',
 '3Dan': '350a 356a 3f4e 40eb 425e 4bb0 4c4b 4eb6 4f14 5210 63b8 64a2 64a3 6fb8 73ac 74ed 75b8 7d1e 80c6 81bd 8874 9ef5',
 '3Dao': '3800 3fd2 4183 4326 4cbd 5012 58d4 5bfc 5c0e 5c9b 5cf6 5d8b 5d8c 5db9 6363 636f 6417 64e3 7977 7982 79b1 8e48 969d 96af',
 '2Bu': '91ad',
 '2Chen': '3574 3af3 3c80 3d34 3f78 41a3 449e 4698 471f 47e2 4885 4888 48bb 48c5 491f 5875 5bb8 5c18 5c52 5ff1 6116 6550 6668 66df 6a04 6c88 6c89 7141 81e3 831e 8380 8390 852f 85bc 87b4 8a26 8af6 8c0c 8ed9 8fb0 8fe7 9202 9648 9673 9703 9dd0 9e8e',
 '3Fa': '40b2 4f71 6cd5 704b',
 '2Bo': '35d8 377f 37d1 39f3 3a67 3a6d 3a8d 3b0d 3b27 3c5f 3d3e 3dbf 3e40 408d 42bf 4368 4378 4448 4484 45da 464f 47b3 47db 488c 48aa 496c 4a87 4aac 4aca 4b2a 4b66 4b6f 4b80 4b82 4bcb 4c0a 4d88 4eb3 4f2f 4fbc 50f0 52c3 535a 5697 58c6 5d8f 5e1b 613d 61ea 632c 640f 6540 6822 6872 6b02 6cca 6cfa 6d61 6e24 717f 7254 72bb 733c 74dd 7921 7934 7b94 7c19 7cea 80c9 8116 818a 8236 824a 82e9 8467 8584 88af 894f 896e 8a96 8b08 8c70 8e04 8e23 90e3 9238 9251 92cd 939b 946e 94c2 9548 993a 997d 998e 999e 99c1 99ee 9a73 9ac6 9b8a 9c8c 9d53 9e41',
 '3Yuan': '46c4 46c7 4a69 59b4 8fdc 903a 9060',
 '2Xiao': '347e 36a3 3b35 449d 6d28 7b05 80f6 8a24 8ab5 90e9',
 '2Xian': '3605 360b 36fe 3849 38ba 3b79 3bad 3cc4 3ced 3d6a 44b8 4554 4768 49a5 4c97 4da2 54b8 550c 5563 55db 5a34 5a39 5a71 5acc 5afa 5afb 5f26 61aa 6d8e 71c5 7509 75c3 75eb 7647 764e 77af 77b7 7925 7d43 7fac 80d8 8237 85d6 86bf 86dd 8854 8858 8af4 8ce2 8d12 8d24 8f31 929c 9591 9592 95f2 9df3 9df4 9dfc 9e47 9e79',
 '2Qie': '3697 49a7 4f3d 767f 8304',
 '2Bi': '3bb0 4a06 4d44 8378 9b6e 9f3b',
 '4Zhuang': '58ee 58ef 58f5 649e 710b 72b6 72c0',
 '1Jiao': '4d14 4ea4 50ec 5604 59e3 5a07 5b0c 5ce7 5d95 618d 61bf 6912 6d47 6f86 7126 71cb 7901 7a5a 7c25 81a0 81b2 8281 832d 832e 8549 86df 87c2 87ed 8a68 8de4 90ca 940e 9a55 9a84 9bab 9d35 9d41 9dcd 9de6 9dee 9e6a',
 '3Ying': '3bcb 3c9f 4034 4a0d 4b4a 4b57 5dca 5f71 68ac 6d67 6f41 7034 7484 763f 766d 77e8 7a4e 90e2 9834 988d 9895 9896',
 '2Ba': '351c 39ca 47e6 4cc1 4cca 53d0 577a 58a2 59ad 629c 62d4 70a6 72ae 79e1 80c8 8307 83dd 8a59 8dcb 8ef7 94b9 98b0 999b 9b43 9f25',
 '3Rou': '7163 7c88 7cc5',
 '1Qu': '380a 4f39 4f49 51f5 5324 533a 5340 547f 5765 5c48 5c96 5cb4 5d87 6188 62be 657a 66f2 7820 795b 80e0 86c6 86d0 88aa 8a58 8ab3 8d8b 8da8 8eaf 8ec0 957c 9639 99c6 99c8 9a45 9a71 9af7 9b7c 9c38 9c4b 9d8c 9ee2',
 '2Duo': '38de 443e 51d9 526b 593a 596a 60b3 6387 6553 655a 6560 656a 692f 6bf2 75e5 838c 88f0 8957 922c 9438 94ce 9bb5',
 '5La': '5566 97a1',
 '2Mu': '4c6f 6048 6bea 6c01 734f 8b55',
 '2Huai': '3733 40f6 422d 4d1c 5f8a 6000 61d0 61f7 69d0 6af0 6dee 7024 8032 8922 8931 8e1d',
 '4Gen': '3ad4 3b93 4e98 4e99 826e 831b',
 '2Huan': '3872 39a5 3d79 3d8e 3fea 437a 4760 4967 49a1 4b74 4b9d 4d1f 5b1b 5bcf 5bf0 61c1 6356 6853 6d39 6fb4 72df 73af 74b0 74db 7ceb 7e6f 7f33 7fa6 8092 8341 8408 8411 8c72 8c86 8fd8 9084 9370 9436 953e 956e 95e4 961b 96c8 9b1f',
 '3Bian': '399a 4075 533e 60fc 6241 78a5 7a86 85ca 890a 8cb6 8d2c',
 '3Biao': '37fd 3812 3bf1 3bf9 4538 5a4a 6aa6 8868 88f1 893e 8ad8 9336 98c8',
 '2Ma': '3984 45eb 4cf8 7298 75f2 8534 87c6 87c7 9ebb',
 '3Xu': '3454 346f 37b0 3960 4161 4513 5194 55a3 59c1 662b 6829 6e51 7166 73dd 7cc8 8a31 8a61 8add 8bb8 8be9 8c1e 9126 9191',
 '3Bo': '7ba5 86be 8ddb 99ca',
 '2Mi': '3737 37dc 3827 38c6 3a62 3e0f 42b3 42db 4315 4318 4358 4573 4577 46e7 490d 4978 4abe 4d22 519e 5f25 5f4c 6202 64df 6520 6ab7 7030 7222 7315 737c 74d5 7962 7c4b 7cdc 7e3b 7f59 84be 862a 863c 8a78 8b0e 8c1c 8ff7 919a 91be 91bf 91c4 957e 9e0d 9e8a 9e8b 9e9b',
 '1Ban': '6273 642c 653d 6591 6592 670c 73ed 7622 764d 822c 8668 878c 8929 8fac 9812 9881',
 '2Mo': '40fa 4251 4be2 5298 56a4 56b0 5aeb 6469 6479 64f5 6a21 6a45 78e8 7ce2 8611 8b28 8c1f 9943 995d 998d 9acd 9b54',
 '4Bang': '3b4b 3bc1 3fa6 409c 43a7 45ab 49db 4c37 508d 585d 68d2 73a4 78c5 7a16 8255 84a1 868c 872f 8b17 8c24 938a 9551',
 '1Kai': '5952 5f00 63e9 75ce 9426 950e 958b',
 '3Pai': '4fd6',
 '4Zui': '3761 3811 3c0e 4639 666c 6700 682c 69dc 6a87 6a8c 797d 7d4a 7f6a 855e 894a 8fa0 9154 917b 9189 92f7 930a',
 '4Xin': '3430 36db 3b44 4496 46b1 46e8 4717 4f29 4fe1 5342 56df 5b5e 712e 7161 812a 820b 8845 8a2b 8ed0 91c1 9620 9856 99b8',
 '3Shuai': '7529',
 '3Men': '66aa',
 '1Qin': '4eb2 4fb5 5a87 5d5a 5d94 6b3d 7019 7d85 887e 89aa 8a9b 94a6 9849 99f8 9a8e 9bbc',
 '4Xia': '35ff 3664 4e05 4e0b 5413 5687 590f 5913 61d7 6b31 759c 7771 7f45 8290 93ec 9db7',
 '3Pie': '4955 4e3f 9405',
 '4Zheng': '3860 3867 3c4f 3f40 40bb 4223 46eb 494c 496d 499b 49b6 5e27 5e40 653f 6b63 75c7 8a3c 8b49 8bc1 90d1 912d 92e5 9503 9d0a',
 '4Xie': '34a0 34d4 350e 3591 3670 3795 38ef 38f0 39aa 3a1d 3c14 3c21 3ce6 3cff 3d2c 3d2e 3d3d 3e09 3f4a 424f 4263 429d 4548 464a 465d 46b3 46b8 48a1 498f 4991 4a67 4c92 4d66 4eb5 4f33 5070 5368 5378 586e 598e 5a9f 5c51 5c5f 5c67 5db0 5ee8 5fa2 61c8 63f3 65ba 66ac 68b0 698d 69ad 6cc4 6cfb 6d29 6e2b 6fa5 7009 7023 707a 710e 71ee 7215 736c 75b6 7944 79bc 7ccf 7d32 7d4f 7d6c 7de4 7ec1 7f37 85a4 87f9 880f 8878 893b 8b1d 8c22 8e9e 9082 97a2 97f0 99f4 9ab1 9f58 9f65',
 '1Yin': '4f8c 51d0 5591 567e 56d9 56e0 5794 5819 59fb 5a63 5cfe 6114 6147 647f 6b45 6bb7 6c24 6d07 6d15 6e6e 6eb5 7616 798b 79f5 7b43 7d6a 7df8 8335 8491 88c0 8af2 92a6 94df 95c9 9625 9634 9670 967b 9682 9712 9720 9787 97f3 97fd 97fe 99f0 9a83 9eeb',
 '4Zhuan': '4275 50ce 556d 56c0 581f 64b0 7077 7451 7bc6 7bf9 7c51 7e33 815e 8483 8948 8b54 8cfa 8d03 8d5a 994c 9994',
 '3Kuan': '3bd8 4540 4957 4c8c 6b35 6b3e 6b40 7abe',
 '3Kuai': '39df 44d2 64d3 84af',
 '3Min': '37b6 3978 3a09 4845 50f6 51ba 5221 52c4 60af 610d 615c 61ab 62bf 6543 654f 668b 6cef 6e11 6e63 6f63 76bf 7b22 7c22 9594 95a9 95f5 95fd 9c35 9cd8 9efd 9efe',
 '4Beng': '3c76 3def 4a3b 5874 6412 69f0 6cf5 882f 8df0 8e66 8ff8 902c 93f0 955a',
 '4Hai': '3925 399f 39e1 3e94 41cb 4ea5 55d0 5bb3 6c26 9900 995a 99ed 9a87',
 '4Xiu': '35dc 3f8b 55c5 563c 5cab 5cc0 6eb4 73db 7407 79c0 7d89 7e4d 7e61 7ee3 8896 890e 890f 92b9 93e5 93fd 9508 9f45',
 '4Cuan': '3d40 3e11 6ba9 71b6 7228 7a9c 7ac4 7be1 7c12 9479 9569',
 '3Shua': '800d',
 '3Diao': '3e7f 4246 5c4c 948c',
 '1Hai': '548d 55e8',
 '2Qiu': '3424 3564 3797 37c8 3939 3962 39e8 3b5d 3dd5 3eab 42b5 43bf 472a 47f5 48c7 491b 4fc5 53b9 53f4 5512 56da 5d37 5def 5df0 624f 6739 6882 6bec 6c42 6c53 6cc5 6d57 6e1e 716a 72b0 738c 7403 7486 76b3 76da 7d0c 7d7f 808d 8119 827d 82ec 838d 866c 866f 86f7 8764 88d8 89e9 8a04 8a05 8cd5 8d47 900e 9011 9052 914b 91da 92b6 9486 9b82 9bc4 9c0c 9c3d 9f3d',
 '4Zhang': '3569 3663 3f74 4e08 4ed7 5d82 5e10 5e33 5e5b 6259 6756 6db1 75ee 762c 7634 7795 7dd4 80c0 8139 8cec 8d26 969c',
 '2Yang': '37c5 3b15 4051 45b9 4b17 4f6f 52b7 579f 5d35 5f89 626c 63da 656d 65f8 661c 6698 6768 6837 694a 6d0b 7080 70ca 716c 739a 73dc 7452 75a1 760d 773b 7993 7f8a 7f8f 86d8 8af9 8f30 935a 940a 9626 9633 967d 9737 98ba 98cf 9c11 9e09',
 '4Fou': '6931 7ace',
 '1Ben': '5954 6ccd 6e00 7287 931b 951b',
 '1Qie': '5207',
 '3Qu': '4d9a 53d6 5a36 6d40 7aec 7b41 7d36 8a53 9f72 9f8b',
 '4Han': '3475 3488 3726 38a8 3a14 3a8b 3ca6 3d44 3f73 4054 430d 43af 43f7 4404 457f 4594 4636 49f2 4af2 50bc 54fb 57be 5a28 5c7d 608d 61be 625e 634d 64bc 650c 65f1 6658 6665 66b5 6c57 6d5b 6d86 6dca 6f22 6f8f 701a 710a 71af 7302 7694 7745 7ff0 839f 83e1 86ff 872d 8792 8b40 8c8b 91ec 92b2 92ce 9588 95ec 96d7 9837 9844 988c 9894 99af 99fb 9dbe',
 '3Qi': '3485 3793 3953 3a69 3ad3 3fa8 410e 412b 425d 42ef 43a2 43ff 44bb 4507 46f4 4854 4b6b 4b6c 4e5e 4f01 542f 5447 5518 5553 5554 555f 5a4d 5c7a 5c82 6567 6675 675e 68e8 69e3 76c0 7dae 7dba 7eee 8291 829e 88ff 8aec 8c48 8d77 9094 95d9',
 '2Nong': '3db6 4078 4889 4fac 5102 519c 54dd 5665 6a82 6b01 6d53 6fc3 7651 79af 79fe 7a60 8113 81bf 857d 895b 8fb2 8fb3 91b2 9b1e 9f48',
 '1Suo': '509e 5506 55cd 55e6 5a11 6331 6332 644d 686b 68ad 6a0e 7c11 7c14 7e2e 7f29 7fa7 838f 84d1 8d96 8e5c',
 '2Ruan': '44f4 5827 58d6 648b',
 '3She': '4b37 6368 9a07',
 '5Bo': '5575 8514',
 '1Qiu': '3400 418b 4e18 4e20 5775 5a9d 6058 6978 79cb 79cc 7a50 7bcd 7de7 8429 84f2 86af 8775 87d7 8824 8da5 90b1 97a6 97a7 9c0d 9c43 9cc5 9d96 9e59 9f9d',
 '4Wan': '3e18 4951 4e07 4ef4 534d 5350 5fe8 6365 7d84 7feb 8155 842c 87c3 8ca6 8d0e 93ab',
 '4Lie': '3920 39dc 3b2f 3b5e 3bff 3cb1 3e39 3f32 3f5f 407d 4140 426d 44df 4732 47f9 4d15 5120 51bd 5217 52a3 52bd 57d3 59f4 6312 6369 64f8 6d0c 6d56 70c8 716d 72a3 7375 7759 8057 811f 8322 86da 86f6 88c2 8d94 8e90 8ffe 98b2 9b1b 9b23 9ba4 9c72 9d37',
 '3Leng': '51b7',
 '1Pou': '5256',
 '3Luan': '5375',
 '4Tuo': '553e 62d3 67dd 69d6 6be4 6bfb 7ba8 7c5c 841a 8600 8dc5',
 '4Hu': '3546 39bf 3a2d 3e26 3e89 41d8 4353 4576 46ce 4a25 4a3c 4a9d 4cb5 4e92 51b1 51b4 569b 5a5f 5aed 5aee 5cb5 5e0d 5f16 6019 6236 6237 6238 623d 6248 62a4 6462 6608 6791 695b 69f4 6c8d 6caa 6eec 71a9 74e0 795c 7b0f 7c04 7d94 8530 8b3c 8b77 9120 9800 9cf8',
 '4Guai': '39d4 3dc7 3f47 40af 42bd 53cf 592c 602a 6060 65dd 7650',
 '4Guan': '3ba1 3d26 432f 439a 45f0 465b 466e 477a 4e31 60b9 60ba 60ef 6163 63bc 645c 6a0c 6bcc 6cf4 6dab 6f45 704c 721f 74d8 76e5 77d4 7936 797c 7f46 7f50 8b34 8cab 8d2f 9066 93c6 9475 96da 9c79 9e1b 9e73',
 '3Ju': '40ca 4114 4153 422e 48b9 4da5 4e3e 5480 5f06 6319 64e7 67b8 6907 6989 6998 6af8 6b05 77e9 7ad8 7b65 8209 8392 849f 877a 8893 8dd9 8e3d 9f5f 9f83',
 '2Nao': '37aa 3e80 40e9 4129 444b 46dd 4af8 4d03 5476 5912 5cf1 5da9 5dce 5dd9 6013 61b9 6320 6493 6861 6a48 7331 7376 737f 7847 7e77 8a49 8b4a 9403 94d9 9ad0',
 '2Nan': '34d3 3f56 451c 457c 46c1 4db2 4fbd 5357 5583 597b 5a1a 6694 678f 67ac 67df 6960 7537 7558 83ae 8433 8af5 96be 96e3',
 '3Ji': '36a1 3786 379b 37a6 39b8 3a08 3d09 4364 48b3 4912 4e2e 5980 5df1 5e7e 621f 6324 638e 64a0 64e0 6cf2 72b1 7398 7a56 87e3 8e26 9218 9b55 9b62 9c7e 9e82',
 '2Nai': '3f8d 4372 4605 4bee 6468 718b 91e2',
 '2Chuo': '72b3',
 '4He': '3570 39a6 3b1e 3d51 3dce 3de4 438b 44fc 4682 4a9a 4cfd 4d33 4d71 4f6b 4ff0 55c3 58d1 668d 7103 7142 7187 7200 764b 76ac 788b 7fef 881a 8894 8cc0 8d3a 8d6b 96ba 974e 974f 9db4 9e16 9e64 fa2d',
 '2Luan': '3748 3869 3c4d 4582 470c 571d 571e 5a08 5b4c 5b6a 5b7f 5ce6 5dd2 631b 6523 66eb 683e 6b12 6ee6 7053 7064 7674 7675 7f89 8114 81e0 864a 92ae 947e 9d49 9e1e 9e3e',
 '3Gai': '4ab1 5fcb 6539 7d60',
 '2Piao': '3f3c 456f 4d29 6df2 74e2 7ac2 86ab 95dd',
 '1Chang': '4f25 5000 5a3c 660c 667f 6dd0 7316 7429 83d6 88ee 9329 9520 95b6 960a 9be7 9cb3 9f1a',
 '1Qiao': '5281 589d 58bd 5d6a 5e67 6572 6a47 7857 78fd 7e51 8df7 8e0d 8e7a 8e7b 90fb 9121 9125 936b 936c 93d2 9430 9539 9792 981d 9ab9 9adc',
 '4Cheng': '343c 4015 725a 79e4 9757',
 '3Lei': '344d 34a6 367c 385e 3d9f 3f0d 3fd4 4242 46f6 48c2 4d0e 50ab 5121 513d 53bd 5792 58d8 6a0f 6ad0 6ad1 6d21 7045 7623 7657 78ca 78e5 7928 7d6b 8012 854c 857e 85df 863d 8a84 8b84 8bd4 8f60 9478 981b 9e13',
 '2Pu': '3492 3bf7 3cab 3eaa 422c 423b 4451 4515 45f1 49e4 4d06 50d5 530d 6a38 6bde 6fee 735b 749e 77a8 7a59 7b81 7e80 83d0 83e9 8461 84b1 84b2 8946 8965 8e7c 8f50 917a 93f7 9564',
 '4Pa': '5e0a 5e15 6015 8899',
 '1Nang': '56d4',
 '4Gu': '39fd 3f7d 435b 44e2 4d9c 50f1 51c5 56fa 580c 5d13 5d2e 6018 6545 688f 68dd 69be 727f 75fc 797b 7a12 932e 9522 96c7 980b 9867 987e 9bdd 9cb4 9dae',
 '3Han': '360e 3615 361a 3673 3d4e 3e01 3e96 3f49 4350 4351 44cd 44ff 46de 5388 558a 6d6b 7f55 850a 8c43 9b2b',
 '3Re': '558f 60f9',
 '2Rao': '3e5b 4ade 835b 8558 86f2 87ef 8953 9952 9976',
 '2Ran': '3723 3caf 3e10 3f86 4459 4584 4ac7 562b 7136 71c3 7e4e 80b0 86a6 86ba 86c5 887b 8887 88a1 9ae5 9aef',
 '2Hen': '3bca 62eb 75d5 978e',
 '3Hao': '597d 604f 90dd',
 '2Dian': '47cd',
 '4Ang': '39b9 3f1c 678a 76ce 91a0',
 '2Tuo': '3e30 3e31 3f20 3f83 436b 4850 4a91 4b7e 4c3f 4d31 4f57 5768 5836 5cae 6a50 6cb1 6cb2 7260 72cf 7823 7824 78a2 7d3d 8889 8a51 8dce 8ff1 9161 9624 9640 9641 9781 99b1 99c4 99dd 99de 9a28 9a52 9a6e 9a7c 9b60 9b80 9d15 9e35 9f09 9f0d 9f27',
 '4Zan': '373a 37db 38c5 56cb 6682 66ab 6b11 6ffd 7052 74c9 74d2 74da 7a73 8978 8b83 8b9a 8cdb 8d0a 8d5e 913c 9142 9147 933e 93e8',
 '2Tui': '3888 3891 3faf 3fbd 3fc9 3fd7 4003 416a 437e 4acb 5c35 5f1a 6a54 7a68 8608 8e6a 96a4 9839 983a 983d 9893 9b4b',
 '2Xiong': '49fa 718a 8d68 96c4',
 '4Ge': '49c4 4e2a 500b 5404 55f0 7b87 867c 927b 94ec',
 '2Pi': '3525 3bc5 5564 57e4 58c0 6707 6787 6bd7 6bd8 7137 72a4 73ad 7435 75b2 76ae 7b13 7eb0 7f74 7f86 80b6 813e 8157 818d 8298 86bd 8731 87b7 8c7c 8c94 90eb 921a 930d 9630 9674 9b7e 9f19',
 '3Chui': '47bc',
 '2Wang': '4ea1 4ebe 4efc 5166 5f7a 738b 83a3 869f',
 '1Duo': '525f 5484 54c6 5689 591a 591b 5d1c',
 '1Shen': '4f38 4f81 515f 547b 59bd 59fa 5a20 5c7e 5cf7 625f 67db 6c20 6df1 71ca 73c5 7521 7533 7837 7c76 7c78 7d33 7ec5 80c2 8398 8460 8593 88d1 8a37 8a75 8bdc 8eab 926e 99ea 9bf5 9cb9 9d62',
 '1Gan': '4e79 4e7e 4e81 51f2 5769 5c32 5c34 5c36 5c37 5d45 5e72 5fd3 653c 6746 67d1 6cd4 7395 7518 75b3 77f8 7aff 7b78 7c93 809d 8289 82f7 8677 872c 8fc0 9cf1',
 '3Chun': '35ba 3fe4 411d 43db 440f 4790 49ae 5046 60f7 7776 7bba 8436 8822 8e33',
 '3Kao': '4bea 4e02 62f7 6537 6832 69c0 6d18 70e4 71fa 7a01 8003 85a7 9bb3 9c93',
 '4Zhai': '3a5f 503a 50b5 5be8 7635 7826',
 '2Lang': '3757 37cd 3883 3c62 3fbf 41a1 4859 4bd6 4c76 4fcd 52c6 5acf 5eca 6879 6994 6a03 6b34 72fc 7405 746f 7a02 7b64 8246 83a8 84c8 84e2 870b 8782 8e09 8eb4 90ce 90de 92c3 9512',
 '4Zhan': '37de 3e98 3ef5 42ce 45c3 463a 4a8c 4c60 4f54 5061 5d98 6218 6226 6230 6808 685f 68e7 693e 6e5b 7ad9 7dbb 7efd 83da 8638 8665 8666 89b1 8e54 8f1a 8f4f 986b 98a4',
 '4Bing': '34c8 4202 45d2 4e26 4f75 5002 50a1 5bce 5e76 5e77 6452 6824 68c5 75c5 7add 9235 9750 9ba9',
 '4Nuan': '5044',
 '2Cui': '51d7 615b',
 '1Ta': '4ed6 584c 5979 5b83 7942 79a2',
 '5Sai': '567b',
 '3Ca': '7924',
 '1Ma': '5988 5abd 5b24 5b37',
 '2Ting': '3e76 45f4 4c53 4ead 505c 5a77 5d49 5ead 5ef7 673e 695f 69b3 6e1f 7b73 8064 839b 8476 8713 874f 8aea 9092 95ae 9706 9bc5 9f2e',
 '4Liao': '387b 3e92 427c 4361 4386 48a7 5c1e 5c25 5c26 5ed6 6482 6599 7093 7212',
 '4Lian': '3703 373b 3a9d 3c68 3d91 3f11 581c 5aa1 604b 6200 655b 695d 6b93 6bae 6d70 6e45 6f4b 6fb0 7032 70bc 7149 7453 7df4 7e9e 7ec3 8430 8539 861d 861e 88e3 895d 932c 934a 93c8 94fe 9c0a',
 '2Liao': '3669 374b 3d73 470d 472e 4740 4a05 50da 5639 5afd 5be5 5bee 5c6a 5d7a 5d9a 5d9b 5eeb 6180 644e 6579 66b8 6f3b 71ae 7360 7499 7597 7642 7ab7 7c1d 7e5a 7f2d 804a 818b 81ab 87df 8c42 8cff 8e58 8fbd 907c 9410 9563 985f 98c2 98c9 9ace 9dda 9def 9e68 9e69',
 '2Lian': '34ce 3715 377a 37c0 3858 3898 3955 3981 3d8c 3ea6 3f13 3fbe 4060 40db 4182 43c8 467a 4965 4a2c 4b51 4eb7 50c6 5286 5332 5333 55f9 5652 5941 5969 5971 5afe 5e18 5ec9 601c 6169 6190 68bf 69e4 6ae3 6d9f 6e93 6f23 6fc2 6fd3 7191 71eb 740f 7489 78cf 7a34 7c3e 7c62 7c68 7e3a 7ff4 8054 806b 806e 806f 81c1 83b2 84ee 8595 878a 880a 88e2 8933 899d 8b30 8b67 8e65 8fde 9023 938c 942e 9570 9b11 9c31 9ca2',
 '1Tui': '63a8 84f7 85ec',
 '1Keng': '52a5 5748 5751 595f 5994 6333 647c 727c 7841 785c 787b 92b5 935e 93d7 94ff 962c',
 '3Tui': '3782 3c63 3fbc 4fc0 50d3 817f 8e46 9abd',
 '3Tun': '3594 6c3d',
 '3Tuo': '37ce 44d5 4c8a 59a5 5aa0 5af7 5eb9 5f75 64b1 692d 6955 6a62 8ec3 9c16 9d4e',
 '4Luan': '4e71 4e82 858d 91e0',
 '3Bao': '3645 3c8f 3ec4 4382 4b4b 4cc8 4cf0 4d10 4fdd 5821 5822 5b9d 5bb2 5bda 5bf3 5bf6 6009 73e4 7de5 8446 8913 8934 8cf2 974c 98fd 9971 99c2 9cf5 9d07 9e28',
 '1Cu': '7c97 89d5 9e81 9e84 9ea4',
 '4Mian': '3d10 7cc6 9762 9763 9eb5 9eba',
 '4Zhong': '343a 3cb4 4c70 4e51 4ef2 4f17 5045 5839 5995 5a91 6e69 72c6 773e 794c 7b57 7ddf 833d 869b 8846 8876 8ae5 91cd',
 '4Ruo': '441e 504c 53d2 5a7c 5f31 6949 712b 7207 7bac 7bdb 82e5 84bb 9100 9c19 9c2f 9db8',
 '3Hang': '6c7b',
 '1Zui': '539c 5d89 6a36 7e97 8127 87d5',
 '4Pu': '3b25 66dd 7011 8216 8217',
 '4Mie': '349d 407e 423c 460a 4a4f 5e6d 61f1 6423 6ad7 6ec5 706d 7923 7bfe 8511 858e 881b 884a 8995 9456 9c74 9d13',
 '4Chuai': '499f 49a4 49b7 8e39',
 '2Ling': '35ab 3875 3944 39ad 3aae 3b21 3bea 3c65 3c86 3e33 3ecf 3f89 4125 420a 4241 4256 4279 4322 4345 4516 4558 4585 4665 4696 4832 487c 487f 49d9 4a29 4bcd 4c31 4d07 4d12 4d2b 4f36 51cc 5222 5464 56f9 577d 590c 59c8 5b41 5cad 5cba 670e 67c3 68c2 6afa 6b1e 6ce0 6de9 6faa 7075 71ef 7227 72d1 73b2 740c 74f4 768a 7756 7831 7890 797e 79e2 7adb 7b2d 7d37 7dbe 7eeb 7f9a 7fce 8046 8232 82d3 83f1 8506 8576 8626 86c9 8851 88ec 8a45 8dc9 8ee8 8f18 9143 91bd 9234 9302 94c3 959d 9675 96f6 970a 971b 971d 9748 99d6 9b7f 9bea 9cae 9d12 9e30 9e77 9ea2 9f61 9f62 9f84 9f97',
 '4Chuan': '4e32 6c4c 7394 8cd7 91e7 948f',
 '3Hua': '8633',
 '3Hun': '7754 9bf6',
 '3Huo': '4f19 5925 706b 90a9 9225',
 '2Bai': '3f1f 3fdf 4cc6 767d',
 '1Ti': '4d18 5254 68af 8e22 92bb 9dc8 9dc9',
 '2Bao': '3d61 3ffa 420f 4964 4a0c 4a14 4aa8 5ad1 74df 7a87 96f9',
 '1Zuan': '8e9c 8ea6 947d',
 '3Duan': '77ed',
 '3Hui': '3a53 3dd0 40e3 43e8 46fc 6094 6a93 6bc0 6bc1 6bc7 70e0 71ec 8294 867a 8770 8b6d',
 '2Chai': '346a 3fb9 44f1 4faa 5115 558d 67f4 7961 8c7a',
 '1Lu': '565c 5695',
 '3Lai': '40be 553b',
 '2Chao': '413b 4b24 4c2b 5632 5de2 5de3 6641 6a14 6f6e 7ab2 8b3f 8f48 911b 9f02 9f0c',
 '2Chan': '3506 367b 3886 3da3 3e25 3ea5 4081 471b 486a 4872 48d1 492b 49ef 50dd 5103 5296 56b5 5a75 5b0b 5b71 5dc9 5edb 68ce 6b03 6bda 6e79 6f79 6f7a 6fb6 700d 703a 7158 7351 78db 79aa 7dfe 7e75 7e8f 7e92 7f20 826c 8749 87ec 87fe 8a97 8b92 8c17 8e94 913d 9141 92cb 9471 9575 995e 998b',
 '4Lou': '3537 5c5a 6f0f 763a 93e4 9542 964b',
 '1Sheng': '5347 544f 58f0 6598 6607 6b85 6ce9 6e66 713a 7272 72cc 73c4 751f 7525 7b19 8072 82fc 924e 9629 965e 9d7f 9f2a',
 '2Kui': '3653 369d 3a12 456b 47f8 4906 49f6 4bd3 4ceb 55b9 5914 594e 6223 63c6 6646 668c 694f 6951 6ac6 6e40 72aa 777d 8067 8475 8637 8641 8ea8 9035 9108 9368 9804 982f 9997 9a24 9a99 9b41',
 '4Qiang': '4d41 5534 709d 7197 7fbb',
 '2Feng': '3980 3d6f 43ce 465c 4a7c 51af 5838 5906 6453 6ca8 6d72 6e22 6f28 7d98 7e2b 7f1d 8242 9022 99ae',
 '4Gang': '6205 6206 6207 69d3 7135 7b7b',
 '3Nong': '4d5c',
 '1Zhan': '38f6 5360 5383 5da6 65c3 65dc 6834 6be1 6c08 6c0a 6cbe 77bb 8998 89c7 8a40 8a79 8b6b 8b9d 8c35 8d88 9085 9711 98e6 9958 9a59 9c63 9ce3 9e07 9e6f',
 '4Zun': '6358 71c7 928c 9c52 9cdf',
 '3Shou': '374a 4b6d 57a8 5b88 624b 63b1 824f 9996',
 '1Zhai': '635a 6458 658b 658e 69b8 9f4b',
 '4Chong': '39e4 63f0 9283 94f3',
 '4Chua': '4ac4',
 '5Le': '4e86 990e 9979',
 '1Guo': '5613 57da 581d 588e 5d1e 6fc4 8748 87c8 90ed 921b 934b 9505',
 '1Shua': '5237 5530',
 '3Nang': '3d9e 652e 66e9 7062 9962 9995',
 '3Kang': '4849',
 '3Mei': '4180 44fa 4738 51c2 5a84 5aba 5b0d 5d44 6334 6bce 6bcf 6d7c 6e3c 71d8 7f8e 9382 9541 9ee3',
 '4Peng': '3f1e 63bd 692a 750f 78b0 8e2b',
 '5A': '554a',
 '3Zhai': '7a84 9259 98f5',
 '2Zao': '4963 51ff 91a9 947f',
 '3Zhao': '355a 3e90 4203 4756 627e 6cbc 7475 83ec',
 '3Zhan': '37a1 382d 4074 4392 47cb 4840 4a45 4c7c 5af8 5c55 5d2d 5d83 5d84 62c3 640c 65a9 65ac 692b 6990 6a3f 6a4f 7416 76bd 76cf 76de 8e4d 8f3e 8f97 9186 98ad 98d0 9b59',
 '3Nin': '62f0',
 '3Qiong': '82d8',
 '1Mi': '54aa 7787',
 '1Cuan': '64ba 651b 6c46 8e7f 8ea5 92d1',
 '1Kong': '5025 57ea 5d06 60be 6db3 787f 7a7a 7b9c 9313 9d7c',
 '4Ze': '3cc1 4ec4 5928 5e82 6603 6617 6c44 7a04',
 '3Niu': '3eb2 43d4 5ff8 626d 7084 72c3 7d10 7ebd 83a5 9215 94ae 9775',
 '1Mo': '6478',
 '4Zi': '3c23 3c37 3c74 4146 4409 5033 525a 5b57 6063 627b 6e0d 6f2c 7278 7725 7726 80d4 80fe 81ea 8321',
 '5Lei': '561e',
 '1Qiang': '545b 55c6 5d88 6215 6217 65a8 67aa 690c 69cd 6eac 7244 7310 73b1 7472 77fc 7bec 7f8c 7f97 7fab 8154 8723 8b12 8dc4 8e4c 8e61 9306 9397 93d8 93f9 9516 9535 956a',
 '3Shui': '6c34',
 '1Bei': '4f13 5351 5ca5 5eb3 60b2 63f9 676f 686e 6911 76c3 7891 7986 7dbc 85e3 9303 9642 9d6f 9e4e',
 '1Dun': '5428 5678 58a9 58aa 5f34 60c7 6489 64b4 6566 729c 7905 8733 8e72 8e7e 9a50',
 '2Que': '7638',
 '1Zhao': '59b1 5df6 62db 662d 671d 67d6 76c4 7abc 91d7 924a 9363 948a 99cb',
 '3Xue': '385c 4a2e 96ea 9c48 9cd5',
 '1Dian': '508e 53a7 5d6e 5dc5 5dd3 5dd4 6382 6527 6541 69c7 69d9 6ec7 7628 766b 7672 8e4e 985a 985b 98a0 9f7b',
 '1Diao': '51cb 5201 53fc 595d 5f6b 625a 6ba6 6c48 7431 7797 7889 866d 86c1 8c82 96d5 9b89 9bdb 9cb7 9ced 9d70 9f26',
 '2Jiu': '3eb5',
 '2Su': '4fd7',
 '2Qun': '3a8a 3fcf 4b7d 5bad 5e2c 7fa3 7fa4 88d9 88e0',
 '3Wan': '3736 3f5c 3ff8 40ba 414b 45a4 45d5 463c 46f7 4779 4a4a 4cc3 5007 57e6 5a49 5b9b 60cb 633d 665a 6669 667c 689a 6900 6d63 742c 7579 7696 76cc 774c 7755 7897 7da9 7db0 7efe 8115 83c0 8416 8e20 8f13 92c4 92d4 933d 9794 9bc7 9ca9',
 '4Qi': '34de 379a 38ec 3b91 3f24 4008 4019 4048 4049 4164 430c 43c5 43cc 43e0 4497 4644 4689 468d 47c4 4880 4ad4 4b23 4c34 546e 54a0 5650 5668 5921 5951 5f03 5f9b 5fd4 6187 61a9 6456 66a3 6814 68c4 6c14 6c17 6c23 6c54 6c7d 6ce3 6e46 6e47 7081 7508 76f5 77f5 780c 789b 78b6 78dc 78e7 7918 7da5 7f09 7f4a 847a 87ff 8a16 8aff 8bab 8fc4 93da 9f1c',
 '1Shang': '4f24 50b7 5546 5892 616f 6b87 6ba4 6ef3 6f21 71b5 850f 87aa 89de 89f4 8b2a 9b3a',
 '1Geng': '5e9a 63ef 6404 6d6d 754a 79d4 7a09 7cb3 7d59 7d5a 7dea 7e06 7fae 7fb9 8015 83ee 8ce1 8d53 9d8a 9e52',
 '3Pen': '7fc9 7ff8',
 '4Qu': '39c1 3ae2 3c26 4066 44a7 4810 521e 53ba 53bb 6e68 89b0 89b7 89bb 89d1 8bce 8da3 95b4 95c3 9612 9eae 9f01',
 '2Jia': '3ab4 3b96 3f2a 3fd3 402b 4039 455b 46df 4a61 550a 573f 57c9 605d 621b 621e 6274 689c 6d43 6d79 7848 835a 83a2 86f1 86fa 88b7 88cc 8df2 90cf 90df 92cf 94d7 94ea 9782 9788 982c 9830 988a 9904 9978 9d4a',
 '1Sen': '66d1 68ee 692e 69ee 7527 7a7c 7bf8 8518 8942',
 '1Yong': '342f 50ad 55c8 5670 5889 58c5 5eb8 5ef1 6175 64c1 6fad 701c 7049 75c8 7655 7670 8579 9095 90fa 9118 93de 955b 96cd 96dd 9954',
 '2Jie': '34d7 34e4 353e 3636 36c3 374c 37af 39a2 3a17 3a29 3b9e 3bae 3e05 4037 4092 409d 40b6 4165 4316 4559 45fb 48e0 495b 4c99 5022 5048 507c 5091 5226 5227 523c 52ab 52bc 5369 536a 55bc 5a55 5b51 5c8a 5d28 5d65 5e6f 622a 62ee 6377 6429 64f3 6605 6770 6809 6828 6840 6854 696c 6976 69a4 6adb 6aed 6ed0 6f54 7004 7596 7664 776b 78a3 7aed 7bc0 7d50 7ed3 7faf 813b 8282 83ad 845c 84f5 86e3 8710 8818 881e 883d 8871 88ba 8a10 8a70 8ab1 8ba6 8bd8 8d8c 8e15 8ffc 9263 937b 9411 9821 9889 9a14 9b9a 9c92',
 '3Pei': '6622 7423',
 '1Jiong': '5182 518b 5770 57a7 57db 6243 860f 8614 99c9 99eb',
 '3Liu': '5b3c 67f3 6801 687a 6a6e 73cb 7db9 7efa 7f76 7f80 925a 92f6 94c6 950d 98f9',
 '4Zhe': '409e 43f3 45ea 4826 4a7e 4d6d 67d8 6aa1 6d59 70e2 8517 87c5 8fd9 9019 9dd3 9e67',
 '1Ying': '5040 5624 56b6 5a74 5a96 5ac8 5b30 5b46 5b7e 5fdc 61c9 6484 6516 6720 685c 6967 6a31 6afb 6e36 7150 745b 748e 74d4 7507 7516 78a4 792f 7dd3 7e93 7eec 7f28 7f42 7f43 7f4c 81ba 82f1 83ba 8621 8767 8833 892e 8b7b 9348 944d 9533 9719 97fa 9d2c 9da7 9daf 9dea 9df9 9e0e 9e1a 9e66 9e70',
 '4Nou': '4176 462b 4c2d 5542 69c8 7373 8028 8b73 9392 941e',
 '1Tou': '5077 5078 5a7e 5aae 936e',
 '2Qia': '480d',
 '4Hen': '6068',
 '2Sheng': '4c46 61b4 6660 6e97 757b 7e69 7ef3 8b5d 9c66',
 '3Ling': '5dba 5f7e 888a 963e 9818 9886',
 '1Chu': '51fa 521d 5c80 6a17 8c99 9f63',
 '1Wai': '359e 54bc 558e 6b6a',
 '4Niu': '42f4 8842',
 '1Wan': '525c 5e35 5f2f 5f4e 6e7e 6f6b 7063 873f 8c4c',
 '2Tiao': '347f 37d8 4384 4492 45ba 47ed 4a66 4bfe 4c54 4cc2 5ca7 5cb9 6761 689d 6a24 7952 7b24 8280 82d5 8414 8729 8d92 8fe2 92da 93a5 9797 9aeb 9bc8 9c37 9ca6 9f60 9f86',
 '2Tian': '39c2 445a 47e7 4852 4858 4956 55d4 5861 586b 5c47 606c 6437 6cba 6e49 74b3 751b 751c 7530 754b 7551 76f7 7ab4 80cb 83fe 95d0 9617 9dc6 9dcf',
 '1Chong': '5145 51b2 5603 5fb8 5fe1 6183 61a7 644f 6c96 6d7a 73eb 7fc0 8202 833a 885d 8e56',
 '4Zhou': '3447 3473 350c 396e 3f19 3fad 41e0 4219 42d3 440d 46c6 4a5c 4d87 4f37 50fd 5191 546a 5492 54ae 5663 5b99 663c 665d 7503 76b1 76ba 7c40 7c52 7c55 7c99 7d02 7e10 7ea3 7ec9 80c4 836e 8464 8a4b 914e 99ce',
 '1Gua': '522e 5280 61d6 681d 6870 7171 74dc 7611 7b48 7dfa 8052 80cd 8136 8161 8440 8717 8778 8d8f 8e3b 92bd 9822 98b3 9a27 9d30 9e39',
 '3Jing': '362b 4718 4e3c 4e95 5106 522d 5244 5b91 61ac 61bc 666f 66bb 6c6c 70c3 70f4 71db 749f 74a5 7a7d 80bc 87fc 8b66 9631 981a 9838 9888',
 '4Tian': '3b87 3dba 63ad 7154 7471',
 '1La': '56b9 5783 62c9 67c6 78d6 7fcb 83c8',
 '4Nie': '3596 363f 365e 3694 3738 3a76 3b86 3d2a 3e0e 40bc 4112 431c 4713 4bc0 4bc5 4bf5 556e 55a6 55eb 565b 5699 56c1 56d3 573c 5b7c 5b7d 5d72 5e07 6470 655c 67bf 68ff 69f8 6af1 6d85 6e7c 758c 7bde 7cf1 7cf5 8042 8076 81ec 81f2 83cd 8616 8825 8e02 8e51 8ea1 93b3 9477 9480 954a 954d 95d1 9667 9689 9873 989e 9f67',
 '3Lang': '34ea 365f 3af0 3bbe 3f97 4036 4041 5871 6716 6717 6724 70fa 7860 8a8f 95ac 9606',
 '4Shen': '3972 3c2e 3d55 41a6 4fba 613c 614e 62bb 661a 6939 6d81 6e17 6ef2 7606 760e 762e 7712 7718 80be 8124 814e 845a 8703 92e0',
 '1Pai': '62cd',
 '4Gui': '367a 3a88 41c8 4306 4377 4434 45af 4646 477f 4829 4be3 4c0e 4ccf 523d 523f 528a 528c 532e 5331 5da1 5ddc 6530 660b 66a9 67dc 6842 6a9c 6ac3 6e8e 7094 7b40 84d5 8958 8cb4 8d35 8dea 97bc 9c56 9c65',
 '2Jue': '34f8 3503 3522 37f2 391c 3a71 3b48 3b7e 3c10 3d50 3dfe 3e15 3e5f 3ed5 4017 4077 4195 41a2 41f6 42c9 434a 43d0 43e3 4418 45bc 463f 4660 474c 47b7 4807 4848 4986 49bc 4e85 5014 5095 50ea 51b3 5214 5282 52ea 53a5 5671 5b52 5b53 5c6b 5d1b 5d2b 5d51 5da5 5f21 5f4f 61a0 61b0 6204 6289 6317 6354 6398 64a7 652b 658d 67fd 6877 6a5b 6a5c 6b14 6b2e 6b8c 6c12 6c7a 6f4f 7106 7133 71a6 7211 721d 7234 7235 7357 7383 73a6 73a8 73cf 7474 75a6 761a 77cd 77e1 7804 7d55 7d76 7edd 81c4 82b5 855d 8568 8697 87e8 87e9 883c 8990 899a 89ba 89c9 89d6 89fc 8a23 8b4e 8bc0 8c32 8c3b 8c9c 8d7d 8db9 8e76 8e77 8ea9 902b 920c 940d 941d 9481 9562 9c4a 9cdc 9d03 9de2 9fa3',
 '4Zeng': '4662 4c1d 7511 8d08 8d60',
 '2Liu': '342c 3f5e 4267 42f7 45dc 46a7 4b1f 4b77 4c18 4c56 4c5e 4d89 5218 5289 56a0 5ab9 5d67 61f0 65c8 65d2 69b4 6a4a 6ca0 6d41 6d4f 700f 7409 7460 746c 74a2 7544 7559 7571 7581 7624 7645 786b 84a5 84c5 85f0 87c9 88d7 905b 938f 93a6 93d0 954f 9560 98c0 98c5 98d7 99e0 99f5 9a2e 9a51 9a9d 9c21 9db9 9e60 9e8d',
 '4Gun': '4d6a 68cd 74ad 7774',
 '4Ren': '3834 3db5 3e3e 403c 41ee 42d5 4ede 4eed 4efb 5203 5204 598a 59d9 5c7b 6041 6268 6752 6895 7263 7d09 7d1d 7eab 8095 887d 88b5 8a12 8a8d 8ba4 8bb1 8d41 8ed4 8ee0 8f6b 976d 9771 97cc 97e7 98ea 9901 996a 9d40',
 '4Gua': '5366 5569 576c 6302 639b 7d53 7f63 7f6b 8902 8a7f 8bd6 9afa',
 '2Beng': '752d',
 '2Yan': '35f4 3616 3619 3adf 3cc2 3d84 3fd5 3ffc 400b 403d 40b4 41fe 4299 432a 44c2 457e 4597 45e1 48a5 4937 4c72 4dae 4e25 53b3 56b4 57cf 5869 58db 58e7 598d 59f8 5a2b 5a2e 5b4d 5ca9 5d52 5d53 5dcc 5dd6 5dd7 5ef6 63c5 6616 694c 6a90 6ae9 6cbf 6e7a 708e 72ff 7402 76d0 7814 784f 789e 7939 7b75 7c37 7d96 839a 8455 8505 8664 8712 8a00 8a7d 8ba0 90d4 9206 9586 95bb 960e 9843 984f 9854 989c 9e7d 9e99 9ea3',
 '2Yao': '3a31 3b81 409a 4199 4270 42c2 430a 431b 4343 4504 45b4 46ba 46bb 48a3 4b19 5004 509c 55c2 579a 582f 59da 5ab1 5c27 5c2d 5ce3 5d24 5da2 5da4 5fad 612e 63fa 6416 6447 669a 69a3 6bbd 6dc6 70d1 723b 7307 733a 73e7 7464 7476 7a91 7aaf 7ab0 7e47 80b4 8628 8b20 8b21 8c23 8efa 8f7a 9059 9065 908e 929a 94eb 98bb 98d6 9906 991a 9c29 9cd0',
 '3Gei': '7d66 7ed9',
 '2Rang': '3682 4274 5134 52f7 703c 737d 74e4 79b3 7a63 7a70 8618 8e9f 9b24',
 '1Niu': '599e',
 '1Pen': '5460 55af 55b7 5674 6b55',
 '2Lin': '3502 375d 3de0 46ac 48af 4ad0 4bbc 4e34 4e83 53b8 58e3 5d99 60cf 65b4 667d 66bd 6797 6dcb 6f7e 7036 71d0 735c 7433 7498 7510 7584 75f3 77b5 77dd 7884 78f7 7b96 7cbc 7d9d 7e57 7f67 7ff7 81e8 8f54 8f9a 9074 90bb 9130 93fb 96a3 9716 9a4e 9c57 9cde 9e90 9e9f',
 '4Nuo': '3421 46a5 611e 61e6 61e7 633c 63bf 6426 643b 7a2c 7a64 7ce5 7cef 8afe 8bfa 8e43 903d',
 '3Sang': '4866 4ad9 55d3 6421 78c9 892c 939f 9859 98a1',
 '2Ou': '9f75',
 '4Meng': '3734 3771 4822 4942 4970 5922 5923 5b5f 61dc 68a6 6e95 9725 973f',
 '1Gong': '4f9b 516c 529f 5311 53b7 5868 5bab 5bae 5de5 5e4a 5f13 606d 6129 653b 675b 739c 78bd 7be2 7cfc 80b1 86a3 89e5 89f5 8eac 8eb3 9b5f 9f8f 9f94 9f9a',
 '1Chun': '5a8b 65fe 6625 6699 6776 693f 6ac4 81a5 8405 8f34 9d9e',
 '4Cu': '35e4 3c17 40da 438c 466f 46e4 47df 4813 481e 4944 4958 4b28 4fc3 5648 5876 61b1 6880 69ed 6ba7 731d 7604 762f 7c07 7e2c 8128 851f 8d97 8e27 8e59 8e74 918b 9f00',
 '3Gu': '34b4 3689 3bcf 3f0b 3fb6 4007 401c 4026 4030 417d 42ba 434d 4428 4869 4d7b 53e4 5502 5503 560f 5c33 6132 6262 675a 6996 6bc2 6c69 6dc8 6ff2 7014 7138 726f 72dc 76b7 76bc 76ec 77bd 7a40 7cd3 7e0e 7f5f 7f96 80a1 8135 81cc 84c7 85a3 86ca 86cc 8831 8a41 8bc2 8c37 8f42 9237 9936 9989 9aa8 9f13 9f14',
 '1Zu': '79df 84a9',
 '1Kuai': '5459',
 '1Kang': '5add 5eb7 5ffc 6177 69fa 6f2e 7a45 7c87 7ce0 8ebf 93ee 9c47',
 '3Lan': '36e6 3a2b 3a5c 3c16 4296 432b 58c8 5b3e 5b44 5b4f 61d2 61f6 63fd 64e5 652c 6984 6b16 6d68 6f24 7060 89a7 89bd 89c8 9182 9872',
 '3Lao': '37d9 39ef 41ed 4569 4764 4cd3 4d4f 4f6c 54be 6045 6833 6a51 6f66 72eb 8001 8356 8f51 92a0 94d1',
 '1Kua': '4f89 54b5 5938 59f1 8342 8a87',
 '1Li': '54e9',
 '1Hong': '53ff 544d 54c4 63c8 6e39 70d8 7122 85a8 8a07 8c3e 8ee3 8f37 8f5f 8f70 9367 986d',
 '4Shua': '8a9c',
 '4Ci': '3796 3879 3880 3a5e 3e42 44e7 45f9 49bb 4bf8 4c0d 4cd0 4f7d 523a 523e 673f 6b21 7d58 83bf 869d 86d3 8786 8cdc',
 '3Ga': '5c15 738d',
 '2Guo': '3575 35aa 368d 3d81 40b8 4190 4438 490b 4b0e 56ef 56f6 56fb 56fd 5700 570b 5e3c 5e57 6156 63b4 6451 6f0d 7c02 805d 8158 8195 8662 9998',
 '4Ce': '397d 3a32 3a4d 41f2 421f 4282 4534 473a 4fa7 5074 518a 518c 5395 53a0 5ae7 5ec1 607b 60fb 61a1 62fa 6547 6d4b 6e2c 755f 77e0 7b27 7b56 7b5e 7b74 7ba3 7ca3 8326 8417 84db 906a 9819',
 '3Ge': '54ff 64d6 7b34 8238',
 '4Ca': '47c3 4d7d 56c3',
 '1Xian': '4ed9 4eda 50ca 5148 5615 597e 5b10 5b45 5c73 5eef 5ffa 61b8 6380 63ba 66b9 6774 67ae 6c19 6f96 73d7 7946 79c8 7c7c 7e8e 7e96 7ea4 82ee 84d2 8610 893c 8973 8a2e 8df9 8e6e 8e9a 9170 929b 9341 94e6 9528 97ef 97f1 99a6 9bae 9c7b 9c9c',
 '2Zan': '5003 54b1 5592 7ccc',
 '4Ming': '448c 547d 8a7a',
 '3Huang': '393a 3b3b 405c 4319 4420 5bba 5e4c 6033 604d 6130 6643 6644 69a5 6ec9 70be 7180 769d 8a64 8b0a 8c0e 93a4',
 '2Gu': '473c 4ba9 9dbb 9e58',
 '1Guai': '4e56',
 '4Tun': '892a',
 '3Qiang': '36e8 415a 50b8 52e5 588f 62a2 6436 78e2 7e48 7e66 7fa5',
 '2Ga': '5c1c 91d3 9337',
 '1Ca': '5693 64e6',
 '2Ge': '35c6 3753 3837 39b4 3a30 3abe 3d67 3df4 419f 4213 4419 457b 45d8 4601 46cb 46ff 4894 4a23 4a50 4a82 4aba 4ae6 4f6e 530c 5444 55dd 5676 5865 6105 630c 643f 654b 683c 69c5 7366 8188 81f5 8316 845b 86d2 88d3 89e1 8afd 8f35 8f55 9398 9549 95a3 95a4 9601 9616 9694 9769 97b7 97d0 97da 9abc 9baf 9c2a 9f43',
 '1Ci': '5068 5e9b 75b5 858b 8800 8d7c 8d80 8d91 9aca',
 '3Dai': '469e 469f 50a3 6b79 6b7a',
 '4Mou': '6117',
 '4Jie': '3458 374f 3839 3f8f 3fcd 4053 41d2 453f 46fa 4bf0 4c3a 4c44 4cb8 4e2f 4ecb 501f 5424 5536 583a 5c46 5c4a 5c95 5e8e 5fa3 6088 6212 6950 7297 73a0 743e 754c 754d 75a5 780e 7d12 7e72 8265 82a5 85c9 86a7 892f 8aa1 8beb 8ea4 9385 9b40 9b6a',
 '2Ni': '37be 3a92 3e78 4626 463d 46cf 475a 502a 5462 576d 57ff 5a57 5c3c 5c54 6029 6ce5 6de3 72cb 730a 79dc 7c7e 81e1 86ad 873a 89ec 8c8e 8ddc 8f17 90f3 922e 9713 9bd3 9be2 9cb5 9e91 9f6f',
 '3Kuang': '4fc7 593c 9ecb',
 '4Jia': '404d 4ef7 4fa1 50f9 5ac1 67b6 69a2 7a3c 99d5 9a7e',
 '3Zhuai': '8de9',
 '1Da': '366e 54d2 5660 58b6 642d 6498 8037 8345 8921',
 '2Na': '43e7 46d4 4af1 55f1 62cf 62ff 8abd 93bf 954e',
 '4Jin': '3a37 3b1c 3bf2 3c39 3c48 3d06 3da6 3db3 4006 41ae 42ee 431d 4464 4590 45ef 4772 4af4 4d96 4f12 50f8 51da 52a4 52c1 5664 568d 5890 5997 5ae4 5b27 5bd6 6422 6649 664b 669c 6783 6b4f 6ba3 6d55 6d78 6e8d 6fc5 6fdc 70ec 71fc 740e 7468 74a1 74b6 763d 76e1 7981 7e09 7f19 80b5 8369 84f3 85ce 8cee 8d10 8d46 8fd1 8fdb 9032 9773 9f7d',
 '2Lou': '37fa 396a 3c8e 3e8f 411b 4179 474f 48da 4aeb 4bab 4c7e 5245 5a04 5a41 5ed4 617a 697c 6a13 71a1 8027 802c 825b 848c 851e 877c 87bb 8b31 8ec1 9071 97bb 9ac5 9acf 9ddc',
 '1Di': '4efe 4f4e 5824 5891 6ef4 7721 78fe 7f9d 83c2 889b 8d86 9349 9684 97ae',
 '4Jiu': '3807 39d5 3a46 3c83 3dad 3ea9 4162 4192 4286 4298 44d8 46ee 4842 4cce 4cd4 50e6 5313 531b 5336 53a9 548e 5aa8 5c31 5ec4 5ecf 5ed0 6166 6344 6551 65e7 67e9 67fe 6855 759a 81fc 8205 820a 9be6 9df2 9e6b 9e94 9f68',
 '4Bin': '4414 50a7 5110 6448 64ef 6ba1 6baf 8191 81cf 9acc 9ad5 9ae9 9b02 9b13 9b22',
 '3Yun': '41ac 41d6 47ab 485d 491e 49be 4a36 4ab3 508a 5141 5597 628e 6b92 6b9e 72c1 73a7 78d2 891e 8cf1 8f11 9217 962d 9668 9695 9723',
 '3Long': '3659 3d33 4841 5131 5784 5785 58df 58e0 62e2 650f 7ac9 9647 96b4',
 '2Ding': '5a48',
 '4Shuan': '49e0 6dae',
 '2Nu': '36a2 4f2e 5974 5b65 7b2f 8498 99d1 9a7d 9d11',
 '4Qin': '37ac 3908 421c 5422 5423 551a 5bf4 628b 63ff 6407 64b3 6c81 83e3',
 '3Zhuan': '4871 5b68 7af1 8ee2 8f49 8f6c',
 '3Fen': '3979 7c89 7fb5 9efa',
 '1Yun': '596b 6655 6688 6c32 6c33 7174 7dfc 7e15 7f0a 8480 8495 8779 8d07 8d5f',
 '2Ying': '388d 3a15 3d04 3d6c 3d88 3e59 3e5a 3fd8 405d 40f7 4449 4566 4aaf 50cc 55b6 584b 5b34 5dc6 5eee 650d 6979 6aff 6e81 6f46 6fd9 6fda 701b 7020 702f 7192 71df 7469 76c1 76c8 7c5d 7c6f 7e08 8314 8365 8367 83b9 8424 8425 8426 843e 85c0 86cd 8747 877f 87a2 8805 89ae 8b0d 8d0f 8d62 8fce',
 '3Fei': '3971 4541 4a3d 4a3e 532a 595c 60b1 6590 670f 68d0 69a7 7bda 7fe1 855c 8ab9 8bfd 9925',
 '2Pou': '39f5 6294 6299 634a 638a 88d2 9307 952b',
 '4Zang': '3638 585f 5958 5f09 81d3 81df 846c 92ba',
 '4Qie': '34f6 35eb 36cd 36d9 3932 3966 3af8 3c30 3c3c 3e64 3f80 3f9c 47d9 493f 4b0a 5327 59be 602f 608f 60ec 611c 6308 6705 6d2f 6dc1 7a55 7a83 7aca 7b21 7ba7 7bcb 7dc1 85d2 86ea 8e25 9365 9532 9bdc',
 '2Shui': '8ab0 8c01',
 '4Qia': '34e3 384a 3909 4711 4a10 4bca 4d9d 533c 5736 5e22 6070 6b8e 6d3d',
 '3Gan': '3e82 40ed 41de 4508 4917 4d5f 611f 64c0 6562 687f 6a44 6f89 7068 76af 76f0 79c6 7a08 8866 8d11 8d76 8d95 9c64 9ce1',
 '4Dan': '35d6 387a 3adc 3c7d 3cb7 3d45 3e97 4137 4277 4a22 4a35 4a65 4b5b 4cc9 4f46 50e4 5556 5557 557f 5649 56aa 5e0e 5f39 5f3e 5f48 60ee 619a 61ba 62c5 65e6 66ba 67e6 6c2e 6c8a 6de1 6f6c 6fb9 72da 758d 7605 7649 765a 7a9e 8145 81bb 8215 840f 86cb 8711 89db 8a95 8bde 926d 94bd 972e 994f 99be 99f3 9ae7 9d20',
 '1Kan': '520a 52d8 582a 5d41 6221 681e 9f95 9f9b',
 '1Zai': '54c9 683d 6e3d 707d 707e 70d6 8cf3',
 '4Dai': '3432 37ad 3af9 3bc2 3da1 3ed6 3fc3 4206 44ab 4ee3 53c7 57ed 5cb1 5e12 5e26 5e2f 5e36 5ed7 5f85 6020 6234 67cb 6b86 6c4f 703b 73b3 7447 7519 7c24 7d3f 7dff 7ed0 825c 888b 8976 8cb8 8d37 8de2 8e5b 8eda 8ee9 8fe8 902e 9734 9746 9edb 9ef1',
 '2Fan': '3836 3e0b 3e95 4000 401f 4252 42a9 42e3 42e6 4313 484a 4a9b 4aa4 4af6 4b75 4bb3 51e1 51e2 51e3 52eb 58a6 5fdb 61a3 67c9 68e5 6a0a 6a4e 702a 703f 70e6 7169 71d4 74a0 77fe 792c 7b32 7c75 7dd0 7e41 7fb3 81b0 8227 8543 85a0 85e9 8629 881c 894e 8e6f 91e9 9407 9492 98bf 9ded',
 '2Yong': '3758 45e4 5581 5ac6 5ade 69e6 6efd 7245 9852 9899 9c45 9cd9',
 '4Qiu': '47ec 4817 6b8f 8791',
 '4Gong': '34cb 3536 3bef 41e8 489a 5171 7fbe 8ca2 8d21',
 '4Nian': '3cbd 5344 5538 57dd 59e9 5eff 5ff5 824c 9f30 9f5e',
 '4Shao': '3df9 43f4 467c 4b30 4f4b 52ad 5372 54e8 5a0b 6f72 7744 7d39 7da4 7ecd 8891 90b5 98b5',
 '4Shan': '38cc 3aa8 4120 4184 46b2 4947 4982 4985 4c47 4c49 4d2e 5093 50d0 5584 58a0 58a1 5b17 6247 639e 6472 64c5 6c55 7057 718c 759d 78f0 7985 7e55 7f2e 81b3 87ee 87fa 8a15 8b06 8b71 8baa 8d0d 8d61 8d78 912f 91e4 928f 9425 9490 994d 9a38 9a9f 9c53 9c54 9cdd',
 '3Hou': '3583 3e38 543c 72bc',
 '4Shai': '3b20 6652 66ec 95b7',
 '1Pian': '504f 56e8 5aa5 728f 7bc7 7fe9 8439 9828 9da3',
 '1Ding': '4e01 4ec3 53ee 5975 5e04 5eb0 738e 7594 76ef 91d8 9489 976a',
 '4Reng': '82bf',
 '3Pu': '3e52 4c95 5703 5724 57d4 666e 6a8f 6c06 6d66 6ea5 6f7d 70f3 8aa7 8ae9 8b5c 8c31 9420 9568',
 '1Ang': '9aaf',
 '3Pi': '3a3d 43d8 46b0 46b9 490f 4acc 4c26 4ef3 5339 567d 56ad 572e 5d25 5e80 758b 75de 7656 8134 82c9 8ac0 92a2 96a6',
 '1Han': '5505 5ae8 61a8 6b5b 70b6 751d 86b6 8c3d 9163 9807 9878 99a0 9b7d 9f3e',
 '1Qun': '56f7 590b 5cee 8e06 9021 9c06',
 '2Qin': '3626 3899 3a52 3a81 3b97 4537 458c 49a6 52e4 53aa 55ea 5659 5ac0 5e88 61c3 61c4 6366 64d2 65b3 6a8e 6fbf 73e1 7434 7439 79bd 79e6 8039 82a9 82b9 83e6 8699 8793 8804 8d7a 9219 96c2 9772 9cf9 9d6d',
 '1Shu': '4e66 500f 5010 5135 53d4 59dd 6292 6445 6474 6504 66f8 6778 67a2 68b3 6a1e 6a7e 6b8a 6bb3 6bf9 758e 758f 7d13 7d80 7ebe 8212 852c 8e08 8ed7 8f38 8f93 964e 9d68',
 '1Shao': '5f30 634e 65d3 68a2 70e7 713c 71d2 7a0d 7b72 8244 83a6 86f8 8f0e 9afe',
 '3Zhen': '3431 3a9b 40a6 40a7 4450 4ab4 4aec 5f2b 628c 62ae 6623 6678 6795 755b 75b9 7715 7a39 7b09 7d3e 7d7c 7e25 8044 8897 88d6 8a3a 8bca 8eeb 8f78 8fb4 9b12 9ef0',
 '3Mai': '4e70 562a 836c 8552 8cb7 9df6',
 '3Nuo': '36c2 3845 6a60 7808 7839',
 '3Mao': '36b9 39c7 5187 536f 5918 5cc1 623c 6634 6cd6 7b37 8306',
 '3Man': '36e7 4731 5c58 6e80 6ee1 6eff 77d5 87a8 87ce 8954 93cb',
 '1Gun': '88e9',
 '3Wei': '3590 3787 3791 37ea 3815 38bb 3a0a 3b19 3b4f 3c6c 40ec 4227 4794 4a98 4b10 4b3f 4d4b 4eb9 4f1f 4f2a 5049 507d 50de 5130 55a1 59d4 5a13 5bea 5c3e 5d23 5d54 5fab 6107 6596 6690 68b6 6932 6d27 6d58 6ffb 709c 7152 7325 73ae 744b 75cf 75ff 784a 78c8 7def 7eac 8172 8249 829b 82c7 8371 8466 84f6 85b3 8624 8732 8ac9 8bff 8e13 9361 9697 97d1 97d9 97e1 97ea 9820 98b9 9aa9 9aaa 9aab 9baa 9c94',
 '4Pang': '370a 7090 80a8 80d6',
 '1Huai': '7af5',
 '4Mu': '3487 3708 38ce 39c5 3f87 4032 42be 4475 49d4 4eeb 52df 5776 5893 5e55 5e59 6155 66ae 6728 6958 6be3 6c90 7091 7267 72c7 76ee 7766 7a46 82dc 83af 869e 926c 94bc 96ee 9702 97aa 9e5c',
 '3Dang': '48ca 48e3 515a 6321 64cb 6529 6863 6b13 7059 8b61 8b9c 8c20 9ee8',
 '3Wen': '349a 35a7 35c3 3767 3cf7 520e 543b 5461 687d 7a33 7a4f 7a69 80b3 8117',
 '1Gou': '52fe 6784 6c9f 6e9d 7bdd 7df1 7f11 82b6 8920 920e 9264 94a9 97b2 97dd',
 '1Huan': '56be 61fd 6b22 6b53 6b61 72bf 737e 8b99 8c9b 9144 9a69 9d05 9d4d',
 '2Tou': '34f1 384f 388f 4d49 4ea0 532c 5744 5934 6295 724f 9158 982d 9ab0',
 '1Shuo': '54fe 8aaa 8aac 8bf4',
 '1Chan': '5e68 60c9 6400 6519 68b4 88a9 88e7 895c 8fbf 9246 92d3',
 '4Ma': '347b 372b 3a38 3fba 4be6 508c 551b 561c 5e13 6769 69aa 72b8 7341 7770 7943 79a1 7f75 9581 99e1 9a82',
 '4Dao': '49c2 5230 5675 60bc 6aa4 71fe 74d9 76d7 76dc 7a32 7a3b 7e9b 7fff 83ff 885c 885f 8ec7 9053',
 '1Zhuang': '5986 599d 5a24 5e84 6869 6889 6a01 7ca7 7cda 8358 838a 88c5 88dd',
 '1Yao': '4e48 5406 5593 592d 5996 5e7a 6796 6946 7945 8170 847d 8a1e 9080 9d01',
 '1Yan': '5063 5266 54bd 5571 5ae3 5b2e 5d26 61d5 61e8 6df9 6f39 70df 7109 7159 7312 73da 7bf6 80ed 814c 81d9 83f8 852b 9122 9140 9183 95b9 9609 9eed',
 '2Rou': '35bb 3f65 4413 49f7 4c06 5a83 63c9 67d4 697a 6c91 6e18 7448 74c7 79b8 811c 816c 8447 875a 8e42 8f2e 9352 9355 97a3 97d6 9a25 9c07 9d94',
 '4Mi': '3706 3a20 3ad8 3cf4 3d35 3d4b 3e13 4047 427e 430f 4310 4329 4591 46c9 46d1 48fe 4909 4b69 4bad 5196 51aa 5627 5853 5b93 5bbb 5bc6 5cda 5e42 5e4e 5e66 6993 6a12 6ac1 6c68 6ccc 6de7 6dff 6ef5 6f1e 6fd7 7190 7955 79d8 7c1a 7cf8 7f83 8524 85cc 871c 8820 8993 8994 899b 89c5 8b10 8c27 9f0f',
 '2Hao': '357a 3819 3a5d 3b14 4765 49eb 512b 52c2 55e5 5637 5651 568e 58d5 6903 6beb 6fe0 7346 734b 7354 7c47 8814 8ad5 8b79 8c6a',
 '2Han': '35a4 3648 3654 37cf 37d4 3b80 3db0 3f28 4204 45d9 4934 4941 4a21 4ba7 4d83 4f44 51fd 51fe 542b 5705 5a22 5bd2 5d21 6657 6892 6db5 7113 7400 7b68 80a3 9097 90af 92e1 97d3 97e9',
 '2Qiong': '348c 39ed 3baa 3dc0 3f07 4143 41b3 4284 44d6 46ea 483b 511d 536d 5b86 60f8 684f 68fe 6a69 712d 7162 743c 749a 74ca 7758 778f 7a77 7a79 7aae 7ac6 7b3b 7b47 8315 85d1 85ed 86e9 8d79 8deb 8f01 909b',
 '1Bao': '52f9 5305 5b62 7172 7b23 80de 82de 8554 8912 8943 95c1 9ab2',
 '4Zuo': '3445 3600 3634 36d7 3b6e 42cf 4518 4f5c 4fb3 505a 5511 5750 590e 5c9d 5c9e 5ea7 600d 67de 795a 7cf3 80d9 8444 888f 9162 963c',
 '4Mo': '3c44 3c73 3dec 3df5 3e6e 407c 407f 43de 44ac 4603 4706 4b34 4bac 4c45 4cee 4d32 55fc 569c 573d 587b 58a8 59ba 5afc 5bde 5e1e 6154 6629 66af 672b 67ba 6b7e 6b7f 6b81 6cab 6d26 6e50 6f20 700e 7205 763c 768c 771c 773d 7799 781e 7933 79e3 7c96 7d48 7e38 7e86 8031 819c 8309 8388 83ab 84e6 85e6 86e8 87d4 8847 88b9 8b29 8c83 8c8a 8c98 911a 9286 93cc 9546 964c 9721 977a 9a40 9b15 9b69 9ed8 9ed9',
 '4Xing': '34d1 3f2c 4044 4094 44f7 46ed 4c22 5016 59d3 5a5e 5b39 5e78 6027 60bb 674f 6dac 7dc8 81d6 8347 8395',
 '4Di': '38a9 3985 3f35 4038 403f 43d1 446d 446f 47b6 47e1 49dd 4a58 4a5a 4c71 4d8d 5059 50c0 54cb 5572 5730 5754 57ca 5886 58ac 5943 5a23 5db3 5e1d 5f1f 601f 6178 6455 65f3 6755 688a 68e3 6974 6a00 6e27 710d 7393 73f6 750b 7747 78b2 7976 7998 7b2c 7de0 7f14 8091 8163 8482 8515 8673 8743 87ae 8ae6 8c1b 8e36 8ed1 8f6a 9012 9013 905e 9070 91f1 926a 99b0',
 '3Kun': '3a72 4805 58f8 58fc 6083 6346 68b1 7871 7975 7a07 7a1b 7d91 88cd 95ab 95b8 9603 9f6b',
 '3Kui': '36fb 4811 49b1 4ae5 7143 8dec 8e5e 980d',
 '2E': '3f02 3f30 4109 454f 45b8 4a79 4c6e 4cd7 4cd8 4fc4 542a 54e6 56ee 5a25 5ce8 5ce9 6d90 73f4 7692 774b 786a 78c0 83aa 86fe 8a1b 8a90 8b4c 8bb9 8fd7 920b 96b2 981f 984d 989d 9a00 9b64 9d5d 9d5e 9e45',
 '3Tou': '3a97 4c4f 59b5 65a2 7d0f 9204 94ad 9ec8',
 '4Da': '5927 7714',
 '2Fou': '4cd5 527b 54f9 7d11 88e6',
 '3Ni': '38c7 3d6b 4565 49b5 49c7 4b72 4c2f 4f31 4f32 4f60 5117 511e 5b74 62b3 62df 64ec 65ce 67c5 72d4 79b0 82e8 85bf 9268 94cc 96ac 999c',
 '3Dong': '35a6 3a02 40a2 4d54 58a5 5b1e 61c2 7bbd 8463',
 '3Nu': '52aa 5f29 782e 80ec',
 '3Beng': '345f 42fd 4640 4a6c 4b70 4cde 57f2 742b 83f6 8a81 979b',
 '4Kua': '3404 42c0 80ef 8de8 9abb',
 '3Xi': '4884 5338 559c 56cd 58d0 5c63 5f99 6198 6199 657c 66bf 67b2 6d17 6f07 72f6 73ba 74bd 77d6 79a7 7c01 7e30 7e9a 8448 8478 84f0 87e2 8af0 8e5d 8ea7 917e 91c3 973c 9c5a',
 '3Ka': '4f67 5494 80e9 9272',
 '3Dian': '36b2 3e03 4344 44e6 5178 56b8 594c 5a70 655f 70b9 7420 75f6 7898 8547 8e2e 9ede',
 '4Kuo': '35e5 47ef 49a2 4bfa 59e1 5ed3 5f49 5f4d 6269 62e1 62ec 6304 64f4 6f37 6ff6 86de 95ca 9614 979f 97b9',
 '4Hao': '362a 3671 36aa 3740 37bb 3b36 3d46 44b5 46bd 475e 49da 4abd 4b09 4beb 5090 53f7 54e0 5cfc 608e 660a 6626 6667 66a4 66ad 66cd 6d69 6dcf 6ec8 6f94 7025 704f 705d 7693 769c 769e 76a1 76a5 79cf 8017 8055 8583 865f 9117 93ac 9550 9865 98a2 9c1d',
 '4Kui': '355f 4210 436a 455a 559f 5633 5abf 5c2f 6126 6127 6192 648c 69f6 6a3b 6e83 6f70 7786 77b6 7bd1 7c23 7c44 8069 806d 8075 8143 8489 852e 8562 9400 944e 993d 994b 9988 9a29',
 '4Du': '410d 414a 47fb 4ca7 55a5 5992 59ac 5ea6 675c 6bac 6e21 79fa 7c35 809a 8370 8799 8827 8839 934d 9540 976f',
 '4Rou': '413e 5b8d 8089',
 '2Xi': '3512 3804 3934 39bb 3a57 3ce7 3d7f 3f6f 3fc7 4018 43ee 4ae3 4e60 5ab3 5d8d 5e2d 68e4 693a 69e2 6a84 6b2f 6f1d 711f 712c 7184 730e 7365 761c 7fd2 84a0 84c6 8582 88ad 88fc 8972 89a1 89cb 8b35 8d98 90cb 932b 93b4 9496 9521 96b0 972b 98c1 9a3d 9c3c 9cdb',
 '4Biao': '39fc 4784 4ff5 8985 9c3e 9cd4',
 '4Bian': '3778 38d0 3b53 3ca2 3cce 3cd2 3d1c 3d77 3eb9 3ede 4278 44aa 46d2 4862 4abb 4fbf 5325 535e 53d8 5909 5f01 5fa7 5fed 6283 63d9 662a 6c73 6c74 73a3 7df6 7f0f 8251 82c4 898d 8b8a 8fa1 8fa7 8fa8 8fa9 8fab 8fae 8faf 904d 91c6 959e 9d18',
 '4Cou': '50b6 51d1 6971 6e4a 8160 8f33 8f8f',
 '1Tan': '5574 563d 574d 6039 62a9 644a 64f9 6524 6a5d 6ee9 7058 762b 7671 7dc2 8211 821a 8b60 8caa 8d2a',
 '1Sao': '6145 63bb 6414 6e9e 7619 7e45 7f2b 81ca 8258 98be 9a12 9a37 9a9a 9c20 9c3a 9c62 9ccb',
 '1San': '4e09 53c1 5607 5f0e 6515 6bf5 6bff 7299 7cc1 9b16',
 '3Yong': '3672 39b7 3dcf 3f6b 47bb 4fd1 509b 52c7 52c8 548f 57c7 584e 5d71 5f6e 607f 6080 60e5 6111 6139 6142 62e5 6408 67e1 6810 6c38 6cf3 6d8c 6e67 752c 7867 799c 81c3 86f9 8a60 8e0a 8e34 9bd2 9cac',
 '4Sun': '613b 6f60',
 '1Sai': '585e 6122 63cc 6be2 816e 984b 9c13 9cc3',
 '4Leng': '468f 4b9a 580e 6123 8e1c',
 '4Zuan': '4938 63dd 6525 945a 94bb 9961',
 '3Hu': '45c2 4e55 552c 6d52 6ef8 7425 8400 864e 865d',
 '4Weng': '74ee 7515 7f4b 9f46',
 '2Xu': '4371 4fc6 5f90 84a3',
 '3Cai': '3952 433d 4406 48cb 5038 5a47 5f69 63a1 68cc 776c 7db5 8df4 8e29 91c7',
 '1Fan': '50e0 5643 5b0f 5e06 5e61 65d9 65db 674b 756a 7c53 7e59 7ffb 8f53 98dc 9c55',
 '3Shan': '3692 38e3 3a1b 3a8e 3d92 483e 647b 6671 7752 89a2 91a6 9583 95ea 9655 965d 967f',
 '3Shao': '3aa2 449a 4520 5c11',
 '3Shai': '7e7a',
 '3Cao': '4491 613a 61c6 8278 8349 9a32',
 '3Can': '3614 3717 39a7 3fca 415f 4b2b 60e8 6158 61af 6701 9eea 9ef2',
 '1Jue': '5658 5c69 6485',
 '1Lie': '5de4',
 '4Guo': '3cc0 8142 8fc7 904e',
 '3Fang': '3442 356b 39cd 3bd0 488d 4988 4cb1 4eff 5023 65ca 6609 6618 74ec 7706 7d21 7eba 822b 8a2a 8bbf 9ae3 9dad',
 '1E': '59b8 59bf 5a3f 5a40 5a50 5c59 5cc9 75fe 94b6',
 '1Sha': '4e77 5526 5e34 6740 686c 699d 6a27 6bba 6c99 715e 7300 75e7 7802 7870 7c86 7d17 7eb1 838e 8531 88df 93a9 94e9 9aff 9b66 9bbb 9bca 9bcb 9ca8',
 '1Zhang': '50bd 5887 5adc 5f20 5f35 5f70 615e 66b2 6a1f 6f33 7350 748b 7ae0 7cbb 8501 87d1 9067 9123 9926 9a3f 9c46 9e9e',
 '4Geng': '4c4d 4c4e 4c6d 4c74 5829 66f4',
 '1Cai': '731c',
 '3Chuang': '3d42 41ec 6464 6f3a 95d6 95ef',
 '2Yai': '775a',
 '1Shuan': '62f4 6813 9582 95e9',
 '4Ken': '3e27 63af 73e2 784d 88c9 8903',
 '5Tian': '988b',
 '4Rong': '9d27',
 '3Xiao': '4495 4975 5c0f 6653 6681 66c9 76a2 7b71 7b7f 7be0 8b0f',
 '2Cu': '44da 4890 5f82 6b82 8c60',
 '3Deng': '44ad 6225 6729 7b49',
 '1Tuan': '6e4d 7153 732f 8c92',
 '1You': '4f18 512a 5466 5698 5e7d 5fe7 602e 60a0 6182 6538 6acc 7000 7e8b 8030 913e 9e80',
 '1Ya': '4e2b 529c 538b 5440 5727 58d3 5b72 5e98 62bc 690f 9d09 9d28 9d76 9e26 9e2d',
 '4Shou': '359f 3945 46f5 517d 53d7 552e 58fd 5900 5bff 6388 6dad 72e9 7378 74b9 75e9 7626 7dac 7ef6 93c9',
 '2Ci': '34e8 3602 3639 3935 40a3 4185 4218 45aa 46d0 49f3 4a0f 4b63 4cbf 4cc4 5790 5b28 6148 67cc 6fe8 73c1 74f7 7506 78c1 7920 7960 7ccd 8328 85ba 8a5e 8bcd 8f9d 8f9e 8fa4 8fad 96cc 98fa 9908',
 '3Ban': '3a6f 3e1e 3e9c 427d 4b33 5742 5c85 6604 677f 7248 7c84 8228 8742 9211 94a3 95c6 962a',
 '4Yin': '38e7 3aa6 3d08 3f09 4543 46bf 485b 4903 4c9f 5370 57bd 5837 5ed5 616d 6196 6197 61da 6704 6abc 6e5a 6fe5 730c 764a 7aa8 80e4 831a 836b 852d 9173 9ba3',
 '1Can': '50aa 53c2 53c3 53c4 53c5 55b0 6e4c 84e1 8b32 98e1 9910 9a42 9a96',
 '1Ge': '5272 54af 54e5 572a 6208 6213 6228 6401 64f1 6b4c 6e2e 6ed2 7241 726b 7271 72b5 7599 8090 80f3 8b0c 93b6 9d1a 9d3f 9e3d 9ea7',
 '2Ru': '3a0e 3e58 42c8 4c30 5112 5685 5982 5b7a 5e24 6310 66d8 6847 6e2a 6fe1 71f8 7b4e 81d1 8339 8560 85b7 8815 88bd 8966 909a 91b9 92a3 986c 98a5 9c6c 9d3d',
 '1Ga': '560e 5620',
 '1Chi': '5403 54e7 55ab 55e4 5644 5ab8 5f68 5f72 645b 6b2b 74fb 75f4 7661 7735 779d 7b1e 7d7a 86a9 87ad 8a35 8aba 90d7 9ab4 9b51 9d1f 9ed0 9f5d',
 '1Qing': '503e 50be 537f 5568 570a 5bc8 6c22 6c2b 6c30 6df8 6e05 72c5 873b 8efd 8f15 8f7b 90ec 9751 9752 9bd6 9cad',
 '2Ju': '3632 394c 3a74 3b82 3f64 42f0 43f1 456e 45c7 472f 485e 490e 4a95 4c2c 4c61 4d17 4fb7 530a 5a45 5c40 5dc8 6336 63ac 68ae 6908 6a58 6be9 6bf1 6ce6 6dd7 7117 7291 72ca 7ba4 7cb7 83ca 861c 8aca 8d9c 8dfc 8e18 8e6b 8eb9 8f02 90f9 9113 9514 95b0 97ab 99f6 9a67 9d59 9d74 9daa 9f33',
 '1Cong': '5306 56ea 56f1 5fe9 6031 60a4 6181 66b0 679e 68c7 6a05 6a2c 6a37 6f17 7127 71ea 747d 7481 779b 7bf5 7deb 7e71 8061 8066 806a 8070 82c1 8471 84ef 8525 87cc 9350 936f 93d3 93e6 9a18 9a44 9aa2',
 '1Tong': '4f97 55f5 606b 6a0b 70b5 71a5 72ea 75cc 84ea 901a',
 '1Hu': '3ada 4e4e 5322 532b 547c 553f 5552 5611 5780 5be3 5cd8 5e60 5ffd 60da 6612 66f6 6b51 6cd8 6ef9 70c0 70fc 81b4 82f8 864d 8656 8ee4 8f77 96d0',
 '1Che': '4f21 4fe5 5513 7817 7868 8397 86fc 8eca 8f66',
 '3Ran': '36a9 3bd7 3ff5 4383 44a3 48f8 4921 5184 5189 5465 59cc 5aa3 67d3 6a6a 73c3 82d2',
 '3Rao': '3471 5a06 5b08 6270 64fe',
 '3Pou': '357b 52cf 54e3 5a44 5ecd 68d3 7283 74ff',
 '1Shi': '4d13 545e 5931 5c38 5c4d 5e08 5e2b 65bd 6d49 6e64 6e7f 6eae 6ebc 6fd5 72ee 7345 7bb7 7d41 8479 8492 84cd 8671 8768 8937 8979 8a69 8bd7 90bf 91f6 921f 9247 9248 9bf4 9c24 9cba 9cf2 9cfe 9db3 9e24',
 '3Feng': '47ea 552a 8982 8af7',
 '4Cuo': '4423 47f6 4c5c 5249 5252 539d 632b 63aa 6b75 839d 83a1 84cc 902a 92bc 932f 9509 9519 9f5a 9f70',
 '1Pei': '5478 57ba 599a 5a1d 5caf 600c 67f8 80a7 80da 8843 9185',
 '1He': '5475 559d 55ec 5cc6 62b2 8a36 8bc3',
 '3Xin': '4f08',
 '1Nie': '60d7 634f 63d1 8e17 9222 9269 931c 9448',
 '1Sou': '55d6 5ec0 5ecb 635c 641c 6449 6eb2 7340 8490 878b 910b 9199 93aa 953c 98bc 98d5 993f 998a 9a2a',
 '3Suo': '3abd 40b9 4174 4217 441d 459b 45e2 4786 47bd 48d4 4d40 5522 55e9 6240 6475 64cc 669b 6d13 6e91 7410 7463 7485 7637 7d22 8928 938d 9396 93bb 93c1 93fc 9501',
 '2Ji': '3515 35a2 35ca 35f1 360d 366b 3783 380d 380e 3847 386e 3902 395b 39c0 3b72 3b9f 3ba8 3c5e 3cba 3d15 3ef7 3f7a 3f8a 4052 4415 441a 4690 4798 47cc 48e2 4a6f 4bc2 4caf 4ced 4e9f 4ebc 4f0b 4f76 506e 51fb 5359 5373 537d 53ca 5409 573e 5832 5849 59de 5ac9 5c8c 5d74 5daf 5eb4 5f76 5fe3 6025 6131 6222 63e4 6483 64ee 6781 68d8 696b 6975 69c9 6a76 6a9d 6b9b 6c72 6d01 6e52 6f03 6f57 6fc8 710f 72e4 75be 7620 768d 790f 799d 7b08 7bbf 7c4d 7d1a 7ea7 8024 810a 818c 82a8 838b 84ba 84fb 8540 857a 874d 878f 894b 89d9 8e16 8e50 8f2f 8f5a 8f91 90c6 9212 92a1 9353 93f6 9491 96c6 96e6 96e7 9735 978a 9d36 9dba 9e04 9e61',
 '1Tai': '56fc 5b61 80ce',
 '4Ri': '44a4 56f8 65e5 6c1c 8875 91f0 9224 99b9 9a72',
 '1Pie': '6486 6487 66bc 6c15 762a 77a5',
 '1Bing': '4ecc 5175 51ab 51b0 63a4 681f 68b9 6c37 7d63',
 '2Shu': '40de 476a 4d30 57f1 587e 5b70 5c17 6393 6dd1 7102 719f 79eb 83fd 866a 8961 8d16 8d4e 8dfe 9e00',
 '3Ta': '35f3 3e9a 5854 5896 6999 737a',
 '1Kui': '5232 5cbf 5dcb 609d 76d4 7aa5 7aba 85c8 862c 8667 9377 95da',
 '3Chou': '4aae 4e11 4e12 4fb4 5062 541c 677b 677d 7785 77c1 919c 9b57',
 '4Hou': '3548 3ad7 47a7 4ab7 5019 539a 540e 5795 5820 5f8c 6d09 7f3f 8c5e 9005 90c8 9107 9b9c 9c5f 9c8e 9c98',
 '3Sui': '4b49 4bdd 5dc2 7021 81b8 9ac4 9ad3',
 '1Pin': '59d8 62fc 6d84 7917',
 '2Liang': '3e41 4776 48fc 4b6a 51c9 589a 60ca 6881 690b 6a11 6dbc 7cae 7cb1 7ce7 7da1 826f 8f2c 8f8c 99fa',
 '4Re': '6e03 70ed 71b1',
 '3Ti': '4321 48fd 4a86 4d8f 4f53 632e 8eb0 8ec6 9ab5 9ad4',
 '4San': '393e 3a94 3a9a 4ac5 4fd5 6563 6f75 9590',
 '4Sao': '3fcb 4439 45a3 55bf 57fd 6c09 77c2',
 '2Yuan': '37f6 3973 3e49 45a0 4b27 4cae 4cd2 4ce3 5143 539f 53a1 53b5 5458 54e1 56ed 5706 570e 5712 5713 571c 57a3 586c 59a7 5ab4 5ac4 5c8f 63f4 676c 699e 69ac 6a7c 6ade 6c85 6e72 6e90 6e92 7230 7328 732d 733f 7342 7b0e 7de3 7e01 7f18 7fb1 82ab 8432 849d 8597 8696 875d 876f 8788 8881 8c9f 8d20 8f45 8f95 908d 90a7 93b1 9a35 9b6d 9da2 9db0 9eff 9f0b',
 '1Guan': '500c 5173 51a0 5b98 68fa 761d 764f 839e 8484 898c 89b3 89c0 89c2 95a2 95d7 95dc 9c25 9c5e',
 '2Fei': '4208 6ddd 80a5 8153 8409 8730',
 '2She': '3b59 3c12 404b 40a0 478c 4f58 63f2 820c 8675 86c7 86e5 9366 94ca',
 '2Fen': '3dca 3e2e 4a7f 4be8 4d05 5746 575f 58b3 59a2 5e69 5f05 678c 68a4 68fb 68fc 6a68 6c7e 6fc6 7083 711a 71cc 7356 79ce 7f92 80a6 84b6 8561 86a0 86a1 8c6e 8c76 8f52 943c 999a 99a9 9b75 9ec2 9f16 9f22',
 '3Si': '6b7b',
 '4Que': '3541 3a41 3c0c 3c4b 3c7f 3d36 3fa1 41ce 49ac 49ff 5374 537b 5491 57c6 5859 58a7 5bc9 5d05 60ab 6128 6164 6409 69b7 6bbb 6bc3 704d 71e9 7437 76b5 785e 786e 788f 78ba 78bb 7910 792d 822d 8d9e 95cb 95d5 9615 9619 96c0 9d72 9e4a',
 '2Shi': '35b7 3ad1 3d53 4096 45a8 49b9 4da1 4e6d 4ec0 5341 57d8 5852 5b9e 5b9f 5bd4 5be6 5cd5 62fe 65f6 65f9 6642 69af 6e41 6e5c 6ea1 70bb 77f3 794f 83b3 8494 8680 8755 9048 9250 98df 98e0 9963 9c23 9ca5 9f2b 9f2d',
 '3Pan': '5762 5964',
 '1Cheng': '4793 5041 564c 57e5 6186 6490 6491 68e6 6a55 6a89 6cdf 6d7e 722f 7424 77a0 79f0 7a31 7a6a 7ac0 7dfd 86cf 87f6 8d6a 8d6c 93ff 9637 9833 9953',
 '4Ru': '5089 5165 55d5 5ab7 6d33 6ebd 7e1f 7f1b 84d0 8925 910f',
 '1Kun': '5764 5803 5a6b 5d10 5d11 60c3 6606 665c 711c 7311 7428 747b 83ce 872b 88c8 890c 8c87 918c 9315 951f 9a09 9ae0 9ae1 9ae8 9be4 9cb2 9d7e 9e4d',
 '3Fan': '44a0 46c0 53cd 74ea 8ee1 8fd4 9b6c',
 '3Pao': '8dd1',
 '2Cha': '3889 3892 3aaf 3ac5 405f 419b 4458 4553 4929 4ca6 4daa 579e 5bdf 5d56 643d 67e5 67fb 69ce 6aab 78b4 79c5 8336 8a67 9364 9538',
 '1Zang': '5328 7242 7f98 810f 81dc 81e2 81e7 8535 8ccd 8cd8 8d13 8d1c 8d43 9ad2',
 '3Cun': '520c 5fd6',
 '1Tu': '51f8 5817 5d80 5edc 6378 6d8b 75dc 79bf 79c3 7a81 92f5',
 '1Lou': '4056 779c',
 '2Chi': '34fe 365c 37b4 38ae 3b9b 4215 4424 451f 4659 46c2 4704 47be 4aa7 4b88 4d94 4db5 5001 5319 577b 5880 5cbb 5f1b 5f7d 5fa5 5fb2 6301 6b6d 6c60 6c66 6cdc 7afe 7b42 7b88 7b8e 7bea 801b 830c 832c 834e 86b3 8b18 8cbe 8d7f 8d8d 8e1f 8fdf 9045 905f 9072 99b3 9a70',
 '1Duan': '5073 526c 5a8f 7aef 890d 9374',
 '4Cang': '416e 48a2 8cf6',
 '4Chu': '3518 35f0 3647 3915 3918 4655 4734 47e3 498c 49c1 4b9e 4e8d 4ff6 5097 510a 5904 6035 62c0 6410 654a 65a3 65b6 6b2a 6b5c 6ccf 6ec0 7421 755c 77d7 7acc 7ad0 7d40 81c5 8372 8655 89f8 8c56 8e00 905a 9110 95a6 9edc',
 '1Qian': '3868 4edf 4f65 50c9 515b 5343 5731 5732 5977 5a5c 5b6f 5c8d 5e75 60ad 6106 6173 6266 62ea 6394 6434 6481 6510 6511 6513 6744 6ab6 6acf 6b26 6c58 6c67 7275 727d 7b7e 7c3d 7c64 7c81 7e34 7f9f 80b7 8181 81e4 828a 8688 8930 8ad0 8b19 8b63 8c26 8c38 8fc1 9077 91fa 925b 92df 9431 948e 94c5 9513 9621 97c6 9869 9a2b 9a9e 9b1c 9b1d 9d6e 9e50',
 '4Ceng': '38d2 8e6d',
 '4Ai': '354c 35d2 3637 3776 3905 3fc4 4033 416c 453d 477d 4b35 4f0c 50fe 53c6 55f3 5867 58d2 5ad2 5b21 611b 61d3 61dd 66a7 66d6 6fed 7231 7477 74a6 76a7 77b9 7828 784b 788d 7919 827e 8586 8b6a 8cf9 9440 9698 9749 9d31',
 '4Chi': '3486 34fc 3511 37bf 387f 3f5a 41fc 45d6 47f7 4820 4932 4bbb 4c21 4cf5 4e7f 4f99 50ba 52c5 53f1 557b 5f73 605c 6157 618f 61d8 62b6 6555 65a5 6758 6dd4 707b 70bd 70fe 71be 75d3 75f8 761b 7719 7fc4 7fc5 7fe4 89e2 8d64 8da9 8dee 8e05 906b 9253 9290 98ed 994e 996c 9d92 9dd8',
 '1Gai': '4f85 5793 59df 5cd0 6650 7561 7974 7d6f 8344 8a72 8be5 8c65 8cc5 8d45 90c2 9654 9691',
 '2Chu': '354f 3551 3861 3d86 3f25 4173 439d 43a4 47de 4802 4827 520d 53a8 5e6e 5eda 66ef 6a71 6ac9 6ae5 6ec1 7293 7be8 801d 8021 82bb 84a2 84ad 854f 85f8 870d 8d8e 8e70 8e87 8e95 924f 92e4 9504 9664 96cf 96db 9db5',
 '1Shou': '53ce 6536 834d',
 '1Deng': '5654 5b01 706f 71c8 7492 767b 7af3 7c26 89b4 8c4b 956b',
 '2You': '34a1 3571 3625 36ad 36dc 3acd 3cfa 3ec0 3f55 447b 45bb 4683 488a 489f 5c24 5cf3 6023 65bf 6962 6afe 6c8b 6cb9 6d5f 6e38 6efa 72b9 7336 7337 7531 75a3 79de 80ac 83a4 83b8 8555 86b0 8763 8a27 8f0f 8f36 900c 9030 904a 90ae 90f5 923e 94c0 99c0 9b77 9b8b 9c7f 9c89',
 '3Jiong': '34cf 35a5 38a0 392f 3dd7 3de1 4339 4403 489b 4fb0 50d2 518f 56e7 5e5c 6cc2 7085 70af 70f1 715a 715b 71b2 769b 7a98 7d45 7d97 8927 8fe5 9008 9848 988e',
 '1Tao': '53e8 5acd 5f22 6146 638f 642f 69c4 6d9b 6ed4 6fe4 746b 7d5b 7e1a 7e27 7ee6 7fe2 872a 8a5c 8b1f 8f41 97b1 97dc 97ec 98f8 9955',
 '4Cha': '36f3 388e 38fe 391e 42ac 44b2 44ed 47d5 4868 4f98 5239 524e 597c 59f9 5c94 5dee 6c4a 7d01 8869 8a0d 8a6b 8be7',
 '2Huang': '37b7 3fa0 3fae 4163 4297 42a3 437f 4435 445f 47b9 4a84 4bb2 4ce8 505f 5164 51f0 55a4 582d 58b4 5a93 5d32 5fa8 60f6 6497 6e5f 6f62 714c 71bf 735a 745d 749c 7640 7687 78fa 7a54 7bc1 7c27 824e 845f 8757 87e5 8daa 9051 9360 953d 968d 992d 9a1c 9c09 9c51 9cc7 9dec 9ec3 9ec4',
 '1Xuan': '4f61 5107 5405 55a7 5847 5a97 5ba3 5f32 6103 610b 63ce 660d 6645 6684 688b 714a 7444 777b 77ce 79a4 7fe7 7ffe 8431 857f 85fc 8756 8809 8ae0 8afc 8b5e 8c16 8ed2 8f69 92d7 9379 9db1',
 '3Chong': '57eb 5ba0 5bf5',
 '4Che': '352d 3949 3b1a 3bd9 3c4c 3d43 3f9d 3fed 4064 4472 4486 46a2 46f8 4720 49ea 4a01 52f6 577c 5c6e 5f7b 5fb9 63a3 64a4 6f88 70f2 7221 77ae 7869 8045 8f4d 8f99 8fe0',
 '2Qiang': '3a56 4e2c 5899 58bb 5af1 5b19 5ee7 5f37 5f3a 5f4a 6a2f 6aa3 6f12 723f 7246 8537 8594 8620 8ed6',
 '4Chen': '39f1 478b 512c 512d 56ab 5926 6987 6aec 75a2 85fd 886c 896f 8b96 8c36 8d81 8d82 9f53 9f54 9f80',
 '3Hong': '3b34 4027 551d 55ca 664e',
 '3Mu': '4948 4ea9 59c6 59e5 5cd4 62c7 6bcd 7261 7273 7546 7552 755d 755e 756e 782a 80df 9267',
 '2Mou': '3b4c 43ec 45cb 4950 4c55 4f94 52ba 6d20 725f 7738 77b4 7e46 7f2a 86d1 87f1 8b00 8c0b 927e 936a 9d3e 9eb0',
 '3Dun': '76f9 8db8 8e89',
 '3Duo': '350d 35bc 3650 36c6 36ca 3969 3ed4 44b3 4664 4824 492a 4a63 4ac2 4bec 4eb8 54da 56b2 579b 579c 57f5 5972 619c 6305 6306 6735 6736 75d1 7d9e 7f0d 8d93 8eb1 8eb2 9517 9b0c',
 '1Dou': '3a2e 515c 5160 543a 5517 6a77 7bfc 8538 90d6',
 '3Mo': '4a4b 61e1 62b9',
 '3Cou': '3af6',
 '3Zuan': '408e 4323 7c6b 7e64 7e82 7e89 7e98 7f35',
 '3Mi': '3765 395d 3c3d 3cfd 4b67 4c4a 4f8e 5b4a 5f2d 6549 6cb5 6d23 6e33 6fd4 7056 772b 772f 7c73 7f8b 8112 845e 851d 92a4 9761',
 '3Qin': '3772 3f9b 4834 5745 5bd1 5bdd 5be2 6611 66cb 87bc 8d7e',
 '4Cai': '4328 4c02 57f0 5bc0 7e29 83dc 8521',
 '4Can': '36d1 38d3 3eee 3f69 46f9 6faf 707f 71e6 74a8 7cb2 8592',
 '4Cao': '4483 808f 8959',
 '4Nve': '4588 458b 4a0b 759f 7627 7878 8650 8b14 8c11',
 '2Qu': '3586 3739 38c4 3beb 3c98 4020 4082 42e7 4763 47a4 47ca 4d76 4f62 52ac 5337 5fc2 61c5 6235 65aa 6710 6b0b 6c0d 6ded 6e20 7048 7496 74a9 75c0 766f 78f2 7c67 7fd1 7ff5 80ca 81de 83c3 844b 8556 8627 87b6 87dd 8837 8862 8c66 8ea3 8ee5 947a 9d1d 9e1c 9e32 9eaf 9eb4 9eb9 9f29',
 '5Shi': '8b58',
 '1She': '5953 5962 6aa8 8b47 8cd2 8cd6 8d4a 8f0b',
 '3Bu': '365b 3a10 402f 42e0 4488 4a81 4a94 535c 535f 54fa 6355 8865 88dc 9cea',
 '2Kang': '625b 6443',
 '1Xiu': '4f11 4fe2 4fee 54bb 5ea5 6a07 70cc 7f9e 8129 81f9 8320 84da 8c85 929d 9380 93c5 9948 9990 9ae4 9af9 9d42 9e3a',
 '3Keng': '4870',
 '2Lei': '348d 3523 3d62 3e4e 4253 4363 442f 4a13 58e8 5ad8 64c2 6a91 6b19 74c3 757e 7e32 7e8d 7e9d 7f27 7f4d 7fb8 8632 8646 881d 9433 9458 956d 96f7 9741 9f3a',
 '2Leng': '4244 4b0b 5030 5844 5d1a 68f1 695e 7a1c 8590',
 '2Can': '3947 3a3b 3c5a 3e91 437c 43fc 4476 45dd 45de 4609 4641 4773 48df 4aee 4cfb 5b20 5b31 60ed 6159 615a 6b8b 6b98 8695 8745 8836 883a',
 '2Cao': '3716 3be5 411a 43c6 442c 5608 5d86 66f9 66fa 69fd 6f15 825a 84f8 87ac 893f 93ea',
 '1Xin': '4723 4ffd 5153 5677 59a1 5ede 5fc3 5ffb 60de 65b0 6615 677a 6b23 6b46 7098 76fa 82af 85aa 8a22 8f9b 920a 92c5 946b 950c 99a8 9a8d',
 '4Lve': '347c 3500 3a3c 40ae 430e 46da 4923 5719 63a0 64fd 7565 7567 7a24 92dd 92e2 950a',
 '4Lang': '435a 455e 57cc 5d00 6d6a 8497',
 '2Cai': '34b2 4d2d 624d 6750 6ea8 72b2 7e94 88c1 8ca1 8d22',
 '4Piao': '3b13 3e93 43c7 50c4 527d 52e1 5ad6 5fb1 6153 7968 7bfb 9860 9a43 9aa0',
 '4Pian': '34f2 3e24 43d2 7247 9a17 9a19 9a97',
 '3Bi': '36b0 3832 3a8f 3ef6 40fe 41f7 43e2 4621 480b 48e5 4f4a 4ffe 5315 5421 555a 5936 59a3 5f7c 673c 67c0 6bd4 6c98 6ed7 6f77 7595 79d5 7b14 7b46 7c83 868d 8c8f 9119',
 '1Xia': '5084 5477 7146 7175 75a8 778e 867e 8766 8c3a 935c 9595 98ac',
 '1Jian': '5042 517c 51bf 56cf 575a 5805 5978 59e6 59e7 5c16 60e4 620b 6214 63c3 641b 6937 6a2b 6afc 6b7c 6bb2 6e54 7010 7038 714e 719e 71b8 724b 728d 730f 73aa 76d1 76e3 7777 788a 791b 7b3a 7b8b 7bef 7c5b 7dd8 7e11 7f04 7f23 80a9 8270 8271 83c5 83fa 844c 84b9 8551 8573 8643 8c5c 8c63 946f 9593 95f4 976c 97ac 97af 97c0 97c9 9845 9930 99a2 9a1d 9b0b 9b50 9c1c 9c39 9ca3 9cd2 9cfd 9d73 9dbc 9e63 9e89',
 '2Qi': '370e 37d3 37da 37e2 3a7d 3be6 4122 4172 427b 4421 4474 44c5 479a 47da 484b 49d8 49f5 4a53 4b76 4b7c 4c07 4c48 4cac 4ce2 4d92 4d9e 4e93 4e9d 5176 5258 573b 57fc 5898 5947 5c90 5c93 5d0e 5e3a 5fef 612d 61e0 6391 6589 658a 65c2 65d7 68ca 68cb 6ab1 6ac0 6b67 6dc7 6fdd 7309 7382 7426 742a 7482 7541 7566 75b7 7881 7895 7941 7947 7948 797a 79a5 7ad2 7cb8 7da6 7da8 7e83 7fd7 8006 8110 81cd 8269 82aa 8360 8401 8415 8572 85c4 8604 8691 8694 869a 86f4 871d 871e 8810 8879 8dc2 8e11 8edd 8fc9 913f 91ee 9321 9324 951c 980e 9880 9a0e 9a0f 9a90 9a91 9b10 9b3f 9ba8 9bd5 9c2d 9caf 9ccd 9d78 9d80 9e92 9f4a 9f50',
 '2Du': '3c69 3e3f 3f84 44ef 4671 48b1 4a85 4af3 4bb7 5125 51df 5335 5b3b 691f 6add 6bb0 6bd2 6d9c 6e0e 7006 724d 7258 728a 72a2 72ec 7368 74c4 76be 78a1 8773 8aad 8b80 8b9f 8bfb 8c44 8d15 945f 97c7 97e3 97e5 9a33 9ad1 9ee9 9ef7',
 '2Run': '7289 77a4',
 '1Lao': '635e 6488',
 '1Sang': '4e27 55aa 6851 6852',
 '4Zou': '353f 3d75 482b 594f 63cd 9a5f 9aa4',
 '2Qing': '3bf3 477c 478d 4c94 5260 52cd 591d 60c5 64ce 6674 6692 6a08 6aa0 6b91 7520 9ee5 fa12',
 '4Cong': '6b09 85c2 8b25',
 '2De': '3775 392b 3940 3941 3bd6 4677 4678 561a 5f97 5fb3 5fb7 6074 60ea 6dc2 9340 951d',
 '1Ba': '4ec8 516b 53ed 54f5 5c9c 5df4 6252 634c 6733 7390 75a4 7b06 7c91 7f93 82ad 8c5d 91df 9c83',
 '3Chan': '389f 3983 3bc6 3e4c 3e7d 42b2 442e 444e 4918 4940 4a76 4d3c 4d50 4e33 4ea7 5181 522c 5257 5277 56c5 5d7c 5e5d 644c 65f5 6d50 6efb 705b 71c0 7522 7523 7c05 7e5f 8487 8546 8ac2 8b42 8b87 8c04 93df 94f2 95b3 95e1 9610 9a4f 9aa3',
 '3Chao': '3da4 3dc5 4390 43da 5435 5dd0 7092 717c 7727 9ea8',
 '2Da': '3713 3bda 3f91 3fef 40ee 440a 447d 4a62 4cf4 4d63 5312 547e 59b2 601b 6e9a 709f 7557 7563 7b2a 7b54 7e68 8359 8598 87fd 891f 8a5a 8fbe 8fd6 9039 9054 943d 977c 9791 97c3',
 '1Mou': '54de',
 '2Ye': '44c9 497a 5ceb 6353 63f6 64e8 6930 7237 723a 740a 7458 8036 91fe 9381 94d8',
 '1Chai': '62c6 8286 91f5 9497',
 '4Lao': '5aea 61a6 6a6f 6d9d 6f87 8022 802e 8ec2 916a 9add',
 '4Lan': '372e 3c6b 40f9 5682 58cf 6ee5 6feb 70c2 71d7 7201 721b 7224 74d3 7e9c 7f06 9567',
 '2Di': '38d9 3c05 3e4d 42ae 4a00 4a24 4ba4 4bfc 4d1e 4d60 4ee2 5519 5547 5600 5681 5ae1 5ef8 654c 6575 6891 6da4 6ecc 6f6a 72c4 7b1b 7bf4 7c74 7cf4 7fdf 837b 850b 8510 85e1 89bf 89cc 8c74 8e62 8fea 93d1 955d 976e 9814 9e10',
 '4Lai': '3823 3fa2 4124 44f6 4c9a 52d1 5a15 6af4 6fd1 7028 702c 765e 7669 7750 775e 7c41 7c5f 85fe 8970 8cda 8cf4 8d49 8d56 983c 9842 9d63',
 '1Bo': '525d 5265 54f1 5d93 62e8 64a5 64ad 67ed 6ce2 73bb 7676 76cb 7835 7886 7f3d 83e0 889a 8e73 9262 94b5 9911 9a4b 9ac9 9b81 9c4d 9c85',
 '1Seng': '50e7 9b19',
 '1En': '5940 6069 717e 84bd',
 '2Shen': '4c20 751a 795e fa19',
 '4Yuan': '392a 3950 3b47 4148 43cd 4b07 4b3c 5086 5917 5a9b 6028 613f 63be 7457 7990 82d1 884f 88eb 8911 8924 8b1c 9662 9858',
 '3Qia': '5361 8dd2 9160',
 '1Zi': '5179 5472 54a8 55de 59d5 59ff 5b56 5b5c 5b73 5b76 5d6b 6825 6914 6dc4 6e7d 6ecb 6fac 7386 753e 798c 79f6 7a35 7ca2 7d0e 7dc7 7f01 830a 8332 83d1 8458 89dc 8aee 8c18 8cb2 8cc7 8d40 8d44 8da6 8f1c 8f3a 8f8e 9111 922d 9319 937f 93a1 9531 9543 983e 983f 9aed 9bd4 9cbb 9d1c 9d85 9dbf 9dc0 9e5a 9f12 9f4d 9f5c 9f87',
 '1Ka': '5496 5580',
 '3Sha': '50bb 510d 8a2f',
 '1Ke': '5cc7 67ef 68f5 6a16 7290 73c2 75b4 778c 78d5 79d1 7a1e 7aa0 7c3b 80e2 82db 842a 8596 874c 8db7 8efb 8f72 9233 9846 9897 9ac1',
 '3La': '47d1 5587 85de',
 '2Pao': '36bf 46cc 4a5d 530f 5486 5789 5e96 70b0 722e 72cd 888d 8ef3 9784 9f59 9f85',
 '2Pan': '40d1 40f2 4232 4c09 4c14 5abb 5e4b 642b 67c8 69c3 700a 76d8 76e4 78d0 78fb 7e0f 84b0 87e0 8dd8 8e63 939c 97b6',
 '1Zong': '3bf6 5027 582b 5b97 5d4f 5d55 5d78 60fe 6721 68d5 6936 71a7 7323 78eb 7a2f 7d9c 7dc3 7df5 7fea 8159 8250 847c 876c 8c75 8e28 8e2a 8e64 9441 9a0c 9a23 9a94 9b03 9b09 9b37 9bee 9bfc',
 '3Jiang': '38a1 3bcd 4070 4243 42cc 4482 4679 5956 5968 596c 6868 69f3 6ef0 734e 8029 8199 848b 8523 8941 8b1b 8bb2 985c',
 '4Sai': '50ff 55ee 7c3a 8cfd 8d5b',
 '3Nuan': '3b09 43a1 4647 6696 6e1c 7156 7157 992a',
 '4Qing': '42dc 4320 51ca 5e86 6176 6385 6bb8 6c6b 6f00 7883 78d8 78ec 7f44 8b26 944b 9758',
 '3Kou': '351a 52b6 53e3',
 '1Hao': '4fbe 5686 84bf 8585',
 '4Zhui': '410c 5760 589c 60f4 6858 7500 787e 7908 7aa1 7b0d 7db4 7e0b 7f00 7f12 8187 8ac8 8b75 8d05 8d58 9446 991f 9d7d',
 '3Shen': '3524 3724 3c02 3d4a 3f95 5432 54c2 5a76 5b38 5ba1 5bb7 5be9 5f1e 6e16 700b 77ab 77e4 77e7 89be 8a20 8ad7 8b85 8c02 8c09 90a5 9823 9825 9b6b',
 '2Cuo': '37c7 3f68 4821 48dc 4d3e 5d6f 5d73 75e4 77ec 8516 8658 919d 9e7a 9e7e',
 '3Cheng': '4fb1 5863 5eb1 5f8e 609c 7748 901e 9a01 9a8b',
 '1Hou': '9f41',
 '3Jue': '47b5 47fe',
 '3Mou': '4352 67d0 8e07',
 '4Suan': '7958 7b07 7b6d 7b97 849c',
 '3Suan': '352f 5334',
 '4Ga': '5c2c',
 '3Jun': '8720',
 '3Sai': '35f7 4222',
 '3Xing': '376d 3a18 3b90 4cd9 64e4 6e3b 7772 9192',
 '1Cen': '5d7e 68ab',
 '3Sao': '36ee 4545 5ac2 626b 6383',
 '3San': '39f2 4010 4248 4289 4ae9 4ed0 4f1e 5098 7cc2 7cdd 7ce3 7ce4 7e56 93fe 994a 9993',
 '3Jian': '3513 3a35 3ce8 3d95 412f 4150 424d 46f3 47f0 4a46 4b60 4bbf 4bdb 4be1 4d61 4d64 4da0 4fed 5039 5109 51cf 526a 56dd 583f 5bcb 5f3f 6229 622c 62e3 6338 6361 63c0 64bf 6695 67ec 68c0 691c 6aa2 6e1b 6e55 703d 7450 7751 77bc 7877 78b1 7b15 7b67 7b80 7c21 7d78 7e6d 7fe6 85c6 8812 88e5 8947 897a 8b07 8b2d 8b7e 8c2b 8dbc 8e47 9417 9427 950f 9c14 9e78 9e7b 9e7c',
 '3Jiao': '3a70 3b42 3cc5 3f71 3f72 400a 4076 4628 46a9 481b 495e 4d1b 4f7c 4fa5 50e5 510c 527f 528b 52e6 5b42 5fba 6322 6341 6405 6477 649f 64b9 652a 657d 657f 6648 669e 66d2 6e6b 6e6c 6f05 705a 70c4 714d 72e1 74ac 768e 76a6 77eb 77ef 7b4a 7d5e 7e73 7ede 7f34 811a 8173 81eb 8660 87dc 89d2 8b51 8ccb 8e0b 9278 94f0 9903 997a 9c4e 9c9b',
 '4Bao': '3678 3ae7 3c92 4916 5124 5228 52fd 5831 5fc1 62a5 62b1 66b4 66d3 7206 72a6 83e2 8663 888c 8c79 8db5 924b 9464 94c7 9b91 9c8d 9e14',
 '2Pian': '36f9 3f10 4b81 6944 6969 80fc 8141 8ada 8c1d 8cc6 8e41 8f27 99e2 9a08 9a88 9abf',
 '4Bai': '35d1 3814 4494 4899 4d3d 5457 5504 5e8d 62dc 62dd 6557 7a17 7cba 86fd 8d01 8d25',
 '1Cang': '4ed3 4efa 4f27 5009 5096 51d4 5d62 6ca7 6ec4 8231 8259 82cd 84bc 87a5 9dac 9e27',
 '1Zun': '58ab 58ff 5c0a 5d9f 6a3d 7e5c 7f47 9075 940f 9df7',
 '4Pou': '5485',
 '4Shi': '34be 353a 3bb6 3c41 3e37 3e5d 407a 4293 43e1 46c8 47d7 4931 4a43 4b44 4c04 4e16 4e17 4e8a 4e8b 4ed5 4f8d 519f 52bf 52e2 534b 5376 53d3 5469 55dc 566c 58eb 596d 5b15 5ba4 5d3c 5e02 5f0f 5f11 5f12 5fd5 6043 6220 623a 62ed 63d3 662f 6630 67be 67f9 67ff 683b 6c0f 6fa8 70d2 7702 770e 7757 793a 793b 7b6e 7c2d 7fe8 8210 8213 87ab 896b 8996 89c6 8a66 8a93 8adf 8ae1 8b1a 8bc6 8bd5 8c25 8cb0 8d33 8efe 8f7c 9002 901d 9069 907e 91c8 91ca 91cb 9230 9243 927d 92b4 94c8 98fe 9919 991d 9970 9bf7 9cc0 9f5b',
 '4Rang': '61f9 6518 8b72 8b93 8ba9',
 '3Po': '53f5 5c00 7b38 9887',
 '3Ren': '38fc 43d5 43f0 4b43 5fcd 6820 6823 68ef 79f9 7a14 814d 834f 8375',
 '4Neng': '3c8c',
 '4Sha': '36ab 36fc 3c31 4209 474a 4b9c 4d58 4d8e 503d 53a6 553c 5551 5565 55a2 5e39 5ec8 6b43 7b91 7fe3 8410 95af 970e',
 '4Hong': '3db9 6f92 857b 9359 95a7 9b28',
 '4Diao': '349b 3a95 40aa 40bd 4519 483c 4d72 4f04 540a 5f14 6389 7639 7a8e 7ab5 7ae8 839c 84e7 85cb 8a0b 8abf 8c03 91e3 92b1 92fd 9443 9493 94de 9b61',
 '4Dian': '34e0 376a 379f 3946 3d64 3d98 3f2d 49c3 4f43 576b 57ab 588a 58c2 5960 5a5d 5e97 60e6 6242 6a42 6bbf 6dc0 6fb1 73b7 7414 7535 7538 75c1 765c 78f9 7c1f 8714 923f 94bf 963d 96fb 975b',
 '4She': '34a4 38b5 3d07 4005 406f 4115 4330 4836 492e 4d65 538d 5399 5c04 5f3d 6151 6174 61fe 6442 6444 651d 6b07 6d89 6e09 6ee0 7044 731e 793e 820d 850e 8802 8a2d 8bbe 8d66 97d8 9e9d',
 '3Miao': '399d 417a 52b0 676a 6dfc 6e3a 7707 79d2 7bce 7df2 7f08 85d0 9088',
 '3Mian': '36ef 3901 3ef0 400e 4904 4e0f 506d 514d 5195 52c9 52d4 5595 5a29 6110 6c45 6c94 6e4e 6fa0 7704 7d7b 7dec 7f05 817c 8442 9bb8 9eaa 9eab',
 '3Guang': '5e83 5ee3 72b7 7377',
 '5Da': '37f7 7629',
 '1Gui': '4e80 5080 572d 59ab 5aaf 5ae2 5b00 5e30 5f52 646b 691d 69fb 69fc 6b78 6e88 73ea 7470 749d 74cc 7688 7845 7a90 81ad 8325 88bf 898f 89c4 90bd 90cc 95a8 95fa 9b36 9b39 9bad 9c91 9f9c 9f9f',
 '4Duan': '3ac1 3c6d 482a 5845 65ad 65b7 6934 6bb5 6bc8 7145 7456 78ab 7c16 7c6a 7dde 7f0e 8176 846e 8e96 935b 953b',
 '3Xiang': '35fd 4291 45ae 4eab 4eaf 54cd 595b 5d91 60f3 6651 97ff 98e8 9909 9957 995f 9977 9b9d 9bd7 9c76 9c9e',
 '2Man': '34bc 35c4 3662 417c 42a1 443d 4471 46f2 47c2 4bf6 4c0b 59cf 6097 6172 6471 69fe 7792 779e 86ee 883b 8b3e 8c29 8e52 9862 989f 9945 9992 9b18 9c3b 9cd7',
 '2Mao': '375f 3ca0 4166 515e 5825 5a8c 5af9 65c4 6786 6bdb 6c02 6e35 7266 77db 7f5e 8305 8765 87ca 8992 8ede 9155 9328 951a 9ae6 9af3 9d9c',
 '2Mai': '3725 3f2e 4072 4691 4a2a 57cb 85b6 973e',
 '4Shu': '36f8 3710 38fd 3d96 3dc2 3f70 3f81 419d 4240 4389 4624 4739 4742 48a4 4a71 4fb8 51c1 5885 5c0c 5eb6 5ebb 6055 620d 6570 6578 66d9 672e 675f 6811 6a1c 6a39 6cad 6f31 6f44 6f8d 6fd6 7ad6 7aea 7d49 8357 8481 8853 8895 88cb 8c4e 8ff0 9265 9330 93e3 9b9b 9d90',
 '4Zen': '4ac8 8b56 8b5b 8c2e',
 '4Nu': '489e 6012 6290 6419',
 '1Feng': '4e30 4ef9 5051 50fc 51e8 51ec 51ee 583c 59a6 5bf7 5c01 5cef 5cf0 5d36 6340 67ab 687b 6953 6a92 6ca3 7043 70fd 71a2 728e 7326 75af 760b 76fd 781c 78b8 7bc8 8391 8451 8634 8702 882d 8bbd 8c50 9137 9146 92d2 93e0 950b 974a 98a8 98cc 98ce 9eb7',
 '4Nv': '3d56 45a1 4610 46bc 4d8a 6067 6712 8844',
 '2Nou': '3779 4a32 7fba 8b68',
 '4Ding': '374e 5576 5a17 5b9a 5fca 6917 77f4 7887 78a0 78f8 815a 8a02 8ba2 9320 952d 9841 98e3 9964',
 '1Xiao': '547a 54d3 54ee 554b 560b 5610 5635 56a3 56bb 56c2 5a4b 5baf 5bb5 5ea8 5f47 61a2 64a8 67ad 67b5 689f 6af9 6b4a 6bca 6d88 6f47 6f5a 701f 7071 7072 70cb 7107 7362 75da 75df 785d 7863 7a58 7a99 7bab 7c18 7c2b 7d83 7ee1 7fdb 81ae 8427 8437 856d 85c3 8648 8653 87cf 87f0 8828 8e03 900d 92b7 9500 9704 9a4d 9a81 9ac7 9b48 9bb9 9d1e 9e2e',
 '3Yao': '37f1 3893 3acf 3ad0 3d2d 404f 4058 419e 4d20 4da7 4ef8 5060 54ac 5a79 5b8e 5c86 5d3e 62ad 6773 67fc 699a 6b80 6e94 7711 7a85 7a88 7a94 8200 82ed 84d4 95c4 9a15 9dd5 9f3c 9f69',
 '3Yan': '34e7 3563 36a7 3882 3ac3 3b7a 4059 410b 42bb 43a6 45fa 48cd 4c93 4e75 4fe8 5043 513c 5156 5157 5261 533d 53a3 53b4 565e 5935 5944 59f6 5d43 5d96 5dd8 5dda 5e7f 5f07 611d 622d 624a 6281 63a9 63dc 66ee 68ea 693c 6abf 6c87 6e30 6e37 6f14 7430 7517 773c 787d 7f68 8412 8758 884d 88fa 8917 8ebd 9043 90fe 9153 9692 9a13 9b47 9b58 9c0b 9da0 9ee1 9ee4 9ef6 9f34 9f39 9f91',
 '2An': '4719 557d 73b5 96f8',
 '4Jiong': '344b 6f83',
 '4Kou': '34c2 3c2f 3c84 3f5b 4cdf 4cf9 4f5d 51a6 53e9 5bbc 5bc7 6010 6263 6542 6ef1 7789 7a9b 7b58 7c06 8532 853b 91e6 9dc7',
 '4Ne': '43aa 4b46 5450 8a25 8bb7',
 '1Teng': '81af 9f1f',
 '4Ng': '3576',
 '4Na': '35d9 3a25 3d38 4011 415e 41e3 41f1 422b 438e 4593 45a7 47dc 4a8f 4c79 59a0 637a 7b1d 7d0d 7eb3 80ad 84b3 8872 88a6 8c7d 8c80 8edc 90a3 9209 94a0 9779 9b76',
 '3Shun': '542e 63d7 696f 8cf0',
 '1Zhong': '4e2d 4f00 5223 5990 5e52 5f78 5fe0 5fea 67ca 6c77 6cc8 6f68 7082 7144 76c5 7c66 7d42 7ec8 822f 8520 87bd 8873 8877 8e71 9221 92bf 937e 9418 949f 953a 9d24',
 '3Zhang': '4ec9 638c 6da8 6f32 7903 7ef1 979d',
 '4Ni': '381c 397e 3990 3cbb 4065 460c 4d51 4d52 533f 5804 5adf 5b3a 5c70 60c4 6135 6635 66b1 6c3c 6eba 75c6 7768 7cd1 7e0c 80d2 817b 81a9 8fe1 9006 9dc1 9dca 9e5d 9e62',
 '4Dang': '4457 4992 51fc 5735 57b1 58cb 5a78 5b95 6113 6a94 6c39 6f52 7497 74fd 76ea 778a 7800 78ad 7911 7c1c 8361 83ea 8569 862f 8da4 903f 95e3 96fc',
 '3Zang': '99d4 9a75',
 '3Ai': '3478 35e8 388a 4442 4a20 4f41 566f 5a3e 6639 6b38 6bd0 77ee 853c 85f9 8eb7 972d 9744 99a4',
 '1Pi': '4e15 4f3e 5288 567c 576f 6036 6082 61b5 6279 62ab 62b7 65c7 708b 72c9 72d3 7812 78c7 7915 79db 79e0 7d15 7fcd 801a 82e4 8c7e 91fd 9208 9239 925f 9294 94cd 9739 99d3 9aec 9af2 9b8d 9c8f 9d04',
 '3Ao': '3443 3907 479d 4be0 4d08 5aaa 5abc 629d 62d7 72d5 82ba 8884 8956 957a',
 '3An': '371d 3f62 4046 4156 4ffa 5535 57b5 57ef 63de 667b 7f6f 92a8 968c',
 '4En': '4290 4b36 4b53 4b61 6441',
 '4Shuang': '39bc 486f 7040',
 '1Dao': '5200 5202 5e4d 5fc9 6737 6c18 8220 91d6 9b5b 9c7d',
 '1Sui': '5020 54f8 590a 6535 6bf8 6d7d 6ed6 6fc9 71a3 772d 7762 7d8f 7ee5 8295 837d 837e 867d 96d6 9796',
 '2Lv': '6988 6ad6 6ada 6c00 7208 7618 763b 81a2 85d8 95ad 95fe 99bf 9a62 9a74',
 '1Bie': '618b 864c 9c49 9cd6 9f08 9f9e',
 '1Kuang': '52bb 5321 5329 54d0 6047 6d2d 7844 7b50 8a86 8bd3 8eed 90bc',
 '1Zeng': '5897 589e 618e 6a67 71b7 7494 77f0 78f3 7e52 7f2f 7f7e 8b44 9a53',
 '3Zong': '3894 3dd3 3e45 4c0c 506c 50af 603b 60e3 6121 6374 63d4 6403 6460 719c 7dcf 7e02 7e3d 84d7',
 '4Er': '3483 36c5 43b6 43ea 48f5 4e8c 4f74 5235 54a1 5f0d 5f10 6a32 6be6 7732 8848 8a80 8cae 8cb3 8d30 94d2',
 '3Qiao': '36bd 3851 390d 4cbe 5de7 6084 6100 91e5',
 '3Qian': '39c4 4a88 4b64 5d70 6496 69cf 6d45 6dfa 7e7e 7f31 8125 8738 8b74 8c34 9063',
 '1Pu': '4ec6 5657 58a3 6251 62aa 64b2 64c8 6534 75e1 92ea 94fa 9660',
 '2Zha': '3c5c 3d19 4977 4d75 5284 672d 7250 7534 7b9a 7d25 86bb 883f 8b57 9358 94e1 9598 95f8 9705',
 '2Zhe': '378f 3abf 3bf0 429e 43b2 4411 4432 44c6 4743 4755 4e47 5387 54f2 5560 5586 569e 57d1 608a 6298 647a 6662 6b7d 74cb 7813 78d4 7c77 8034 8674 8707 87c4 8936 8975 8a5f 8b2b 8b3a 8b81 8b8b 8b98 8c2a 8f12 8f19 8f84 92b8 9bbf 9e05',
 '1Tie': '546b 5e16 6017 841c 8cbc 8d34',
 '2Zhi': '3677 373c 3a01 4408 47c8 4c65 4d42 4f84 5024 503c 5128 5295 57f4 57f7 59b7 59ea 6179 61eb 6267 63b7 646d 64ff 67e3 690d 6a34 6b96 6f10 7286 74c6 74e1 76f4 7983 79f7 7a19 7d77 7e36 8040 804c 8077 81b1 8635 86f0 87d9 8901 8cad 8d28 8dd6 8e2f 8e60 8e91 8ec4 91de 99bd',
 '1Dan': '4e39 510b 52ef 5330 5355 5358 55ae 5661 5989 5a85 64d4 6b9a 6bab 7514 7708 7803 7baa 7c1e 803c 803d 8043 8078 894c 8ead 90f8 9132 9156 9815 9ed5',
 '4Chao': '4ee6 4eef 8016 89d8',
 '4Chan': '3674 3b04 3d4c 4021 4828 4a9c 4c7f 5133 5fcf 61f4 61fa 785f 7fbc 97c2',
 '2Zhu': '3509 4335 457d 461a 47c9 496e 4bb1 529a 672f 705f 70a2 70db 71ed 7225 7603 7a8b 7af9 7afa 7b01 7b1c 7b51 7bc9 7beb 8233 833f 84eb 880b 883e 8d89 8e85 9010 9c41',
 '4Chai': '3cd7 460d 56c6 7625 867f 8806 8883',
 '1Jing': '4d16 4eac 4eb0 5162 5755 5759 5a5b 5de0 65cc 65cd 6676 6cfe 6d87 7304 775b 7b90 7cbe 7d4c 7d93 7ecf 8059 8148 830e 8346 834a 8396 83c1 844f 9a5a 9be8 9cb8 9d5b 9d81 9d84 9e96 9ea0 9f31 fa1d',
 '3Zui': '4b94 5634 567f 5d8a 5db5 6d05 6fe2 74bb',
 '4E': '34f5 3529 354e 35be 35c1 37e7 380b 384b 398d 39d6 3a75 3b99 3c66 3dc8 4193 4465 446a 44ca 453e 4673 46d6 4748 47a9 48de 4af7 4cec 5054 50eb 537e 5384 5443 545d 54a2 54b9 5669 5714 57a9 580a 5828 582e 5c8b 5d3f 5dad 6076 60aa 60e1 6115 6239 627c 6424 6439 6aee 6e42 7427 7810 7846 80fa 816d 82ca 843c 855a 8685 8741 89a8 8a7b 8a92 8ae4 8b8d 8c14 8c5f 8c96 8edb 8ef6 8f6d 904c 904f 9102 9354 9469 9537 95bc 960f 9628 9638 981e 984e 989a 9913 9929 997f 9b65 9c10 9c77 9cc4 9d9a 9e57 9f76',
 '1Zou': '5ab0 63ab 68f7 68f8 7b83 7dc5 83c6 8acf 8b05 8bcc 8bf9 90b9 90f0 9112 9139 966c 9a36 9a7a 9beb 9cb0 9ec0 9f71 9f7a',
 '3Zun': '50d4 5642 6499 8b50',
 '3Zuo': '377e 4f50 5497 5528 5de6 6bd1 7e53',
 '2Lai': '3693 394e 4158 42f1 4685 482d 49d2 4f86 4feb 5008 5a61 5b7b 5d03 5d0d 5eb2 5f95 5fa0 6765 68be 68f6 6d9e 6df6 730d 741c 7b59 7b82 83b1 840a 9028 90f2 9338 94fc 9a0b 9be0 9d86 9eb3',
 '1Weng': '55e1 5d61 7fc1 8789 9393 9710 9db2 9e5f',
 '3Sou': '35a9 36d0 4239 4264 43c2 4b9f 5081 53dc 53df 55fe 64de 64fb 6ae2 6eeb 778d 7c54 85ae 85ea',
 '2Lao': '35e6 37a0 37c9 37f9 3a13 40d5 470e 4741 4c8f 50d7 52b3 52b4 52de 54f0 5520 562e 5d02 5d97 61a5 6d76 7262 75e8 7646 78f1 7a82 7c29 87e7 91aa 9412 94f9',
 '2Lan': '3463 3613 37a9 39a8 3cd5 41be 4340 444c 49a8 4a8d 4c10 4cff 5116 5170 53b1 5549 56d2 5a6a 5c9a 5d50 5e71 61e2 62e6 6514 6593 6595 680f 6b04 6b17 6f9c 703e 7046 7061 71e3 71f7 74bc 7937 7bee 7c43 7c63 7cf7 7e7f 847b 84dd 85cd 862b 862d 8955 8964 8974 8b4b 8b95 8c30 8e9d 946d 9484 95cc 9611 97ca',
 '1Yu': '5539 625c 6bfa 6de4 7600 76d3 7a7b 7b8a 7d06 7ea1 8676 8fc2 8fc3 9653',
 '3You': '3b8b 3c36 3f9e 414e 44b4 4b00 4c42 4cd1 4e23 5064 5363 53cb 5500 5cb0 5eae 61ee 6709 682f 6884 69f1 6cd1 6e75 7256 7989 7f90 7f91 8048 82c3 83a0 86b4 870f 9149 92aa 94d5 9edd',
 '1Bang': '57b9 5e2e 5e47 5e5a 5e6b 6360 6886 6d5c 7e0d 90a6 90ab 97a4',
 '3Fu': '34e1 356e 42e8 4317 44db 45c4 4a49 4acd 4add 4bfd 4e76 4fcc 4fdb 4fef 5452 5638 5e9c 5f23 629a 62ca 636c 64ab 65a7 668a 6ecf 7124 752b 76d9 79ff 7c20 812f 8150 8151 8386 86a5 8705 8f14 8f85 90d9 91dc 91e1 982b 9b34 9bc6 9efc',
 '3Xuan': '3f8c 437b 4823 54ba 6685 70dc 766c 9009 9078',
 '4Quan': '4110 5238 529d 52e7 52f8 70c7 7276 7d6d',
 '1Heng': '4ea8 54fc 6099 811d 8afb',
 '2Hong': '3593 38ac 40ab 40d4 4196 427a 4354 472b 4791 484c 484f 49c6 4a0e 4a51 4aa6 4af9 4afa 4ca8 4edc 5430 57ac 5985 5a02 5b8f 5b96 5cf5 5d64 5f18 5f4b 6c6f 6cd3 6d2a 6d64 6e31 6f42 7074 7392 7854 7861 7ad1 7ae4 7bca 7ca0 7d05 7d18 7d2d 7d8b 7ea2 7eae 7fc3 7fdd 803e 823c 82f0 836d 8452 8453 8679 89e6 8a0c 8ba7 8c39 8c3c 921c 9277 92d0 958e 95f3 971f 9783 9d3b 9e3f',
 '1Yo': '5537',
 '1Yi': '3608 4e00 4f0a 4f9d 541a 54bf 566b 58f1 58f9 5901 5adb 5f0c 6098 63d6 6ab9 6b39 6bc9 6cc6 6d22 6eb0 6f2a 71da 7317 747f 794e 7995 7a26 7e44 86dc 8863 8b69 90fc 91ab 92a5 966d 990f 997b 9dd6 9e65 9edf 9ef3',
 '1Ye': '503b 564e 6654 882e',
 '2Chuan': '3bcc 3f37 4063 4f1d 4f20 50b3 570c 66b7 693d 6b42 8229 8239 8aef 8f32 9044',
 '2Chuai': '3a93 8197',
 '5Yo': '54df 55b2',
 '3Zhong': '38eb 51a2 55a0 585a 5c30 6b71 7607 79cd 7a2e 80bf 816b 8e35 fa10',
 }

PINYIN_TABLE_CANTONESE_SRC = {
 '2Gau': '36a3 3e83 3eb5 3fdf 400a 488b 4c18 4e45 4e5d 5902 5ca3 67b8 72d7 7396 7b31 7cfa 7ea0 8007 82df 830d 8a6c 8bdf 97ed 97ee',
 '5Koeng': '5f36 7e48 7f9f 8503 8941 93f9 956a',
 '1Dong': '3f15 3f46 5679 5b95 5f53 6b13 73f0 74ab 7576 7b5c 7c1c 7c39 8261 87f7 88c6 8960 9f1e',
 '2Gaa': '35ce 445d 5047 560f 659d 67b6 698e 69da 6a9f 7615 8cc8 8d3e',
 '2Gai': '5048',
 '2Gan': '377b 4150 430d 44ba 4d61 4d64 4ec5 50c5 5807 5890 5df9 5ed1 69ff 6ba3 747e 763d 7b15 7d27 7dca 83eb 84f3 89b2 89d0 8b39 8c28 9949 9991',
 '2Gam': '40ed 519a 5481 5745 611f 6562 6f89 9326 9526',
 '1Ou': '3a60 495d',
 '2Cyu': '51e6 6775 696e 782b 8655 891a 8caf',
 '1Co': '3f68 4173 4406 495a 4c08 509e 520d 521d 5d6f 7293 7473 78cb 82bb 838f 84ad 8e49 919d 96cf 96db',
 '3Hek': '4086 55ab',
 '4Pei': '3bb0 4bf1 4cb9 57e4 6787 6bd7 7435 75b2 76ae 7c83 813e 868d 8731 8c94 90b3 90eb 921a 9630 9674 9f19',
 '3Hei': '3793 38df 3961 3ad3 3e0d 3ea7 4164 42a0 439f 4b23 4caa 546c 5668 56b1 5848 5f03 6187 61a9 620f 6231 68c4 6c14 6c23 6c7d 7081 7508 7f9b 993c 9969',
 '1Hei': '35ed 3680 372f 3743 380c 383b 3a58 3a6c 3c15 3c28 3c3b 3c3f 3c46 3c47 3da7 3de9 3e77 3ea3 4080 45b7 46e5 4908 4a33 4acf 4ad1 4b8e 4ce2 4fd9 50d6 50db 550f 55e8 563b 563f 5a38 5a90 5b09 5c4e 5dc7 5e0c 6095 665e 66e6 69bf 6b37 6b3a 6d60 70ef 7199 71b9 7214 727a 72a7 72f6 774e 79a7 7a00 7fb2 8ac6 8b46 8c68 91af 91d0 9b4c',
 '6Dong': '4992 5a78 6113 6219 6f52 739a 7497 74fd 76ea 7800 78ad 8361 83ea 8569 8e3c 903f',
 '3Baan': '4414 626e',
 '1On': '3ae8 4002 48bf 4bc3 6849 76ce 94f5 978c 978d',
 '3Baak': '3f23 4f70 5ef9 67cf 6822 6a97 767e 8236',
 '1Oi': '3dbc 54c0',
 '3Baai': '3620 38c1 3f61 42a7 6e43',
 '3Jit': '564e 8b01',
 '6Ngo': '5367 81e5 9913 997f',
 '3Jip': '64eb 6b97 9768 9941 998c',
 '6Be': '568a',
 '6But': '351c 36d8 37d1 3a8d 3af2 3d3e 3dbf 3e40 4368 4aac 4b6f 4b80 4b82 52c3 54f1 62e8 64a5 6872 6d61 6e24 7676 7d8d 8116 8378 8467 88af 894f 92cd 9911 997d 999e 9c4d 9d53 9e41',
 '6Buk': '3492 3678 3b25 3bf7 3cab 4451 45f1 49e4 4a81 4d06 4f0f 50d5 5e5e 66dd 6a38 6fee 7011 7e80 8946 8e7c 8f50 91ad 93f7 9564',
 '6Bui': '3788 5b5b 7119 8a96',
 '1De': '55f2 7239',
 '6Bun': '3698 4f34 53db 67c8 7554 7d46 7eca 80d6',
 '1Luk': '3bdf 4684 47ff 645d 7769 788c 78df 7c0f 8f46 8f98 9e93',
 '1Dung': '3448 36b5 3ae1 4376 4c24 4e1c 51ac 51bb 51cd 549a 6771 67ca 6c21 6db7 70b5 8740 9d87 9e2b 9f15',
 '6Ning': '38f7 3c8c 3fe6 452d 4f5e 5bd7 6cde 6fd8',
 '3Jin': '5b3f 5bb4 66e3 70fb 8b8c 9140 91bc 9df0',
 '3Si': '38c8 55dc 5f11 5f12 67f6 6cd7 8a66 8ae1 8b1a 8bd5 8c25 99df 9a77',
 '6Fei': '5255 539e',
 '3Se': '4263 534c 5378 538d 5399 6cfb 731e 8d66 9a07',
 '3Kaai': '34de 379a 3f07',
 '2Aai': '35e8 388a 3dad 6b38 77ee 8a92 8bf6',
 '4Jin': '550c 57cf 598d 5ef6 5f26 63c5 6d8e 7136 71c3 72ff 7814 7b73 7b75 7d43 7d96 8237 839a 86bf 8712 8a00 8ce2 8d24 90d4 92cb',
 '2Aam': '3f62 424f 83e1',
 '2Syu': '4455 6691 7659 9ecd 9f20',
 '1Fe': '5561',
 '1Di': '5572 7684',
 '3Kaau': '34c2 3c2f 425a 4286 438b 4cf9 92ac 94d0 9760',
 '2Aau': '351a 629d 62d7 8a4f',
 '4Jiu': '509c 582f 59da 5a06 5ab1 5b08 5c27 5ce3 5da2 5fad 612e 6416 6447 69a3 6a48 6d2e 733a 73e7 7464 7476 7a91 7aaf 7ab0 835b 8558 86f2 87ef 8953 8b20 8c23 8efa 8f7a 9059 9065 9676 98bb 98d6 9952 9976 9c29 9cd0 9dd5',
 '2Gwo': '3785 4668 4d39 5a50 60c8 7313 7cbf 83d3 873e 88f9 8901 8f20 9301 951e 991c',
 '1Miu': '55b5',
 '1Mit': '6423',
 '4Joeng': '5134 52f7 579f 5f89 626c 63da 6518 65f8 661c 6698 6768 694a 6d0b 703c 7080 70ca 716c 737d 75a1 760d 7993 79b3 7a70 7f8a 8618 86d8 8830 935a 9496 9633 967d 98ba 98cf 9e09',
 '1Jing': '5040 5a74 5ac8 5b30 5e94 61c9 6484 6516 685c 6a31 6afb 7034 745b 748e 74d4 7e93 7f28 81ba 82f1 8621 8ccf 9719 97fa 9df9 9e1a 9e66 9e70',
 '6Gan': '8fd1',
 '6Gam': '3a52 63ff 6407 64b3',
 '5Leng': '5dba 9818',
 '3Baat': '3b6d 3eb4 516b 634c',
 '3Baau': '7206 7832 8c79',
 '6Gat': '4798 7599 8d8c 8db7 8dc0',
 '6Gau': '3ea9 5003 56bf 65e7 67e9 820a',
 '6Gap': '39c0 4039 53ca 5c8c',
 '3Pun': '5224 6c9c 6cee 80d3 9816',
 '3Pui': '3904 39e9 3fa6 4f69 65c6 6622 6c9b 73ee 7432 914d 9708',
 '4Maai': '3706 3725 3a62 4072 4450 4814 57cb 85b6',
 '4Maan': '35c3 43b9 487b 59cf 71b3 8513 86ee 883b 8b3e 8c29 9945 9992 9b18 9c3b 9cd7',
 '4Maau': '43ec 45a5 490a 5a8c 77db 8305 8765 87ca 9328 951a 9af3',
 '1Hoeng': '35fd 3fdd 4b15 4e61 81b7 8297 858c 9109 9115 9999',
 '3Kyut': '3727 391c 3b48 3d50 4017 4195 41a2 42c9 4373 43d0 4660 49bc 4bb9 4e85 51b3 53a5 5658 5b53 6289 6485 6c7a 7357 73a6 7f3a 82b5 849b 8568 87e8 89d6 89fc 8a23 8b4e 8bc0 8c32 8db9 920c 940d 95cb 95d5 9615 9619 99c3 9d03',
 '3Put': '3507 35f6 39ca 497d 6cfc 6f51 9166 91b1 93fa',
 '2Piu': '3e7e 76ab',
 '4Nong': '56ca 56d4 74e4',
 '5Ji': '4ee5 5235 5c12 5c14 5d3a 5df2 62df 6d31 723e 77e3 8033 82e1 85be 85bf 8bae 8f59 8fe9 9087 926f 927a 94d2 99ec',
 '5Je': '51b6 558f 5622 57dc 60f9 82e5 91ce',
 '2Pin': '4075',
 '1Aai': '54ce 5509 57c3 5a2d 6328',
 '1Deot': '361e 67ee',
 '1Aak': '39d6 4748 480e 49c4 4bb8 5384 5443 63e1 6439 8edb 8ef6 8f6d 9f77 9f8c',
 '1Aam': '36ea 36fa 3784 4156 4937 4a04 4cfa',
 '4To': '3e30 3e31 3f20 3f83 4468 4850 4a91 4b7e 4c3f 5768 5836 67c1 68bc 6aae 6cb1 6cb2 7823 7d3d 8235 8a51 8dce 9161 9640 99b1 99dd 9a52 9a6e 9a7c 9b80 9d15 9e35 9f09 9f0d',
 '1Aat': '621e 63e0 6854 95bc 960f',
 '6Hang': '3f2c 44f7 46ed 5016 5a5e 5e78 674f 6dac 8347 8395',
 '1Dai': '34f3 3add 40c5 4355 4f4e 5f7d 6c10 7825 78fe 7f9d 80dd 889b 97ae',
 '1Dak': '3776 3941 4678 5f97 5fb7 60b3 6dc2',
 '1Dam': '3c9c 3d02 3d37 43d9 4571 4901 4cc9 63fc',
 '1Dan': '4872 58a9 58aa 7905 8e7e',
 '1Daa': '6253',
 '1Zeot': '3518 3915 3c35 461a 469d 5d12 6035 8736',
 '1Zeoi': '3879 3a7e 3b85 420c 4ab6 4ce1 5ca8 6cae 72d9 75bd 7820 82f4 83f9 8445 8d84 8dd9 8ffd 9310 9525 96b9 96ce 9a05 9a93 9d7b',
 '1Zeon': '3852 3c09 3ef1 5c0a 6971 699b 6a3c 6a3d 6d25 6eb1 7349 7a80 7f47 80ab 8127 81fb 84c1 8c06 8f43 8fcd 9075',
 '1Dap': '35f3 55d2',
 '1Dat': '63ec 816f',
 '1Dau': '3a2e 3fe1 515c 5517 7bfc 90d6',
 '3Tau': '35a3 47dd 7093 900f',
 '4Je': '3b68 6353 63f6 6930 6b4b 7237 723a 73a1 740a 8036',
 '3Tap': '4a9a',
 '2Dai': '3a86 3b7d 40a1 48cc 5e95 5f24 62b5 63d0 67e2 7274 89dd 8a46 8bcb 8ee7 90b8 963a',
 '3Hyun': '4110 428e 529d 52f8 6965 6966 7734 79a4 7d62 7da3 7eda 7efb 8b82',
 '1So': '36d6 36d7 384f 39e7 3f70 44fe 47fd 492c 5506 55e6 5a11 6331 6332 686b 68ad 68b3 758e 758f 7c11 7d80 7fa7 84d1 852c 8d96',
 '4Wong': '37b7 4113 4163 4297 42a3 445f 4a84 4bb2 4ce8 505f 51f0 55a4 582d 5fa8 60f6 6e5f 6f62 714c 735a 738b 749c 7640 7687 78fa 7bc1 7c27 824e 8757 87e5 8daa 9051 9360 953d 968d 992d 9a1c 9c09 9cc7 9ec3 9ec4',
 '2Dan': '8db8 8e89',
 '3Hyut': '37bd 44b8 47d2 4a67 4afc 4cf3 5437 6cec 72d8 77b2 8840',
 '4Taai': '35b7 3e6b 4160 49d1',
 '2Dau': '37b3 3ab7 6296 6597 6793 7cfe 86aa 8c46 8d73 9204 94ad 9661',
 '2Zung': '3750 3c9d 3dcf 3dd3 3e45 3efe 41ef 45f8 4865 4887 4c0c 5045 506c 50af 603b 6460 6b71 719c 7607 7a2e 7cbd 7cc9 7e3d 80bf 816b',
 '4Taam': '3f4e 3f51 42a4 5013 58b0 60d4 619b 6619 66c7 6f6d 75f0 7f48 85eb 87eb 8983 8ac7 8b5a 8c08 8c2d 90ef 91b0 931f 9414 952c 9561 9a54',
 '3Tan': '892a 8e46',
 '4Long': '37cd 3883 3c62 3fbf 41a1 4859 4bd6 4c76 5d00 5eca 6879 6994 6d6a 72fc 7405 746f 7860 7a02 7b64 8782 90ce 90de 92c3 93af 9512 f928',
 '6Zoek': '3482 3730 3d78 3fd1 42e4 441e 4588 458b 46a5 47d1 482f 4a0b 4db3 721d',
 '3Tai': '3b31 48ed 4d8d 4d91 5243 568f 5c49 5c5c 66ff 6d95 8599 9b00 9b04',
 '1Ceot': '48ba 51fa 710c 7d40 9edc 9f63',
 '2Ze': '3a0b 59ca 59d0 8005 8d6d 9019 937a 9517',
 '2Zo': '377e 4fce 5497 5528 5de6 963b',
 '2Zin': '3563 37a1 382d 38ba 3a35 3ac3 3b17 3cbd 403d 4059 4074 437e 48a5 4a45 4c7c 4d50 526a 5297 5c55 6229 622c 640c 705b 76bd 7e5f 7fe6 8ad3 8b7e 8c2b 8f97',
 '2Zi': '3873 3876 3953 39d7 3aca 3b52 3bb9 3c9b 4169 41db 425d 4920 49bb 4b72 4cc5 538e 5470 54ab 5740 5741 59c9 5b50 5fb5 6049 627a 6307 65e8 6893 6b62 6c9a 6ed3 75bb 7949 79c4 79ed 7c7d 7d19 7d2b 7eb8 8014 80cf 8293 82b7 8879 8a3e 8a3f 8dbe 8ef9 8f75 916f 962f 9ef9',
 '2Jau': '673d 67da 6cd1 7163 79b8 7cc5 8447 86b4 923e 9edd',
 '2Ziu': '3a70 3e5b 4203 481b 6cbc',
 '2Mang': '5fdf',
 '1Ceon': '3ae9 411d 4ca0 6625 6776 693f 6a41 81a5 8f34 9c06',
 '1Ceoi': '3720 38a1 3d4f 42ab 4651 486a 4db4 50ac 539c 5439 55fa 5894 5d14 63a8 6467 69b1 708a 7480 78ea 7e17 7f1e 86c6 8870 8c99 8d8b 8d8d 8da8 93d9 9fa1',
 '5Saan': '3c2e',
 '2Dyun': '77ed',
 '2Cing': '343c 4855 62ef 8bf7 901e 90e2 9a01 9a8b',
 '2Kong': '389c',
 '6Hong': '48bd 48c8 5df7 8856 9805 9879',
 '3Coek': '347f 34f8 352c 38ff 3aac 3c8b 3e7f 41ce 4519 4750 4756 49ff 4b93 4d60 502c 52fa 5353 5671 5a65 5a7c 6233 684c 6f5f 707c 712f 76b5 788f 7a5b 7dbd 7ef0 828d 8da0 8e14 8fb5 9034 9d72 9e4a',
 '6Maak': '4047 423f 4603 4a2b 4bac 4cee 9ea5',
 '2Ceng': '8acb',
 '6Ngang': '37c1 491d',
 '6Maan': '35c8 3862 3b05 3d58 4555 4761 4762 486c 4e07 50c8 534d 5881 5ada 5e54 6162 66fc 69fe 6f2b 7e35 7f26 842c 9124 93dd 9558',
 '2Ho': '37b9 3abc 4614 55ec 5ca2',
 '6Gwat': '3b7e 3ed5 5014 5d1b 6398 957c 9d8c',
 '1Kat': '6b2c',
 '1Kap': '3486 3b1b 3cb8 4f0b 5438 573e 6271 6b59 6c72 7b08 7d1a 7d66 7ea7 7ed9 9212',
 '6Gwan': '73fa 90e1',
 '1Kam': '887e 895f',
 '5Hon': '46de 482d 608d 65f1 850a 99fb',
 '6Ngong': '6205 6206 6207',
 '1Kai': '5d47 5d60 6eaa 78ce 7a3d 8c3f 9e02',
 '6Gwai': '455a 532e 5331 648c 6a3b 6ac3 7085 7bd1 7c23 7c44 8489 8562 8dea 9400 993d 9988',
 '6Maat': '401b 407e 4285 481a 896a',
 '6Maau': '3b98 4ac9 8c8c',
 '2Lo': '34a9 39ac 3a61 3c01 3c7b 7630 7822 81dd 84cf 8803 88f8 8eb6',
 '4Nang': '5749 80fd',
 '4Wang': '38ac 4196 437f 47b9 484c 4a0e 4a51 4afa 5430 5b8f 5f18 5f4b 6c6f 6cd3 6d64 7ad1 7d18 7eae 7fc3 803e 8c39 921c 92d0 958e 95f3 9783',
 '4Bou': '57d7',
 '5Kau': '6855',
 '1Wik': '7aa2',
 '6Ce': '707a',
 '1Paang': '3923 39f8 5309 6072 62a8 6cd9 6f30 6f8e 70f9 8eef',
 '1Caap': '380d 3d15 4052 4245 4281 430c 4415 4d93',
 '3Jim': '4ffa 538c 53a3 990d 995c',
 '1Coi': '554b',
 '2Kwaa': '4f89 9abb',
 '5Seoi': '4bdd 5885 60b4 7021 7d6e 7dd2 7e78 7eea 9ad3',
 '2Kwai': '47a8 8dec 8e5e',
 '4Haam': '35a4 3605 36fe 3937 3b80 438f 4554 4c97 51fd 54b8 5563 55db 6db5 8854 8af4 929c 9e79 9e99',
 '1Zak': '3cc1 4ec4 4fa7 5074 5219 5247 5d31 5e82 6603 8434 9c02',
 '1Zai': '3a08 4091 4774 4801 4aa1 4aa2 4aa3 4b29 4bba 5242 5291 6324 64e0 6ac5 8640 8d4d 8dfb 8e8b 96ae 9f4c 9f4e 9f4f 9f51',
 '1Zan': '40a6 40cc 73cd 73ce 771e 771f 799b 7c48 80d7 85bd',
 '1Zam': '3630 38c5 40e1 5d9c 659f 744a 7827 7972 7bb4 8474 91dd 9356 937c 9488 9c75',
 '1Long': '5577 84e2',
 '1Zaa': '35ec 3aa5 3fb4 4412 4a28 4d59 507a 5412 548b 54a4 54b1 5592 6313 63f8 67e4 6a1d 6a9b 6e23 76bb 781f 7ccc 89f0 9afd 9f47',
 '1Zap': '57f7 6179 6267 6c41 7d77 7e36 84fb 99bd',
 '1Zat': '3731 4447 6303 63e4 67b3 6acd 7929 8cea 8d28 9455 9527 96b2 9a2d 9a98',
 '1Zau': '44df 49d3 4f9c 5468 5541 557e 558c 5a35 5a64 5dde 63ab 63ea 68f8 6d32 6e6b 76e9 7740 7dc5 821f 83c6 8acf 8b05 8b78 8bcc 8bea 8bf9 8cd9 8d52 8f08 8f16 8f80 9031 90b9 90f0 9112 9139 966c 9a36 9a7a 9beb 9cb0 9d43 9e3c 9ec0 9f7a',
 '1Kaam': '4633',
 '4Cou': '3716 3be5 411a 43c6 442c 4890 50ae 5608 5f82 66f9 69fd 6b82 6f15 825a 87ac 8dff 9c3d',
 '3Man': '5461',
 '1Sung': '3721 38dd 4265 442b 48a8 4bf3 4bf7 5a00 5d27 5d69 5fea 644f 6dde 83d8 9b06',
 '1Ling': '3a55 62ce 9234 94c3',
 '4Jam': '4efb 5198 541f 58ec 598a 5a6c 5d1f 6deb 7c37 7d1d 88b5 928b 972a 9d40',
 '3Bin': '38d0 53d8 5e77 8b8a 904d',
 '4Jan': '4eba 4ec1 513f 5924 5bc5 6ba5',
 '3Gwaa': '5366 5459 54bc 5569 6302 639b 7d53 7f63 8902 8a7f 8bd6',
 '6Haap': '3621 3b58 3b71 3c2d 4426 4563 4596 4c51 4fa0 4fe0 5323 5ce1 5cfd 631f 633e 67d9 72ce 72ed 76d2 7856 796b 7ba7 7bcb',
 '4Kaau': '3424 39c4',
 '3Faat': '36f2 40b2 53d1 6cd5 73d0 743a 767c 781d 9aee',
 '2Bing': '3a00 403b 42d1 4d35 4e19 5bce 5c4f 5c5b 6032 62a6 6452 663a 70b3 79c9 90b4 997c',
 '3Faai': '3655 3a4c 401f 418f 421b 5080 54d9 5672 5757 584a 5feb 78c8 7b77',
 '3Faan': '4d0a 6c3e 6c4e 6cdb 7548 8982 8ca9 8d29',
 '3Bit': '3521 38bc 40e9 42e2 43df 4ce4 5f46 618b 864c 9c49 9cd6 9de9 9f08',
 '5Leoi': '344d 348d 367c 36ce 3b5a 3d7d 3d9f 3ef2 3f0d 3f94 3fd4 4242 442f 4d0e 4fa3 4fb6 5121 5122 5415 5442 5792 58d8 5c61 5c62 65c5 68a0 6f2f 7045 7657 78ca 78e5 7927 7928 7a06 7a6d 7f15 8182 85df 863d 881d 891b 90d8 92c1 94dd 9e13',
 '5Nin': '3b9f 6d8a 78be 8e4d 8f3e',
 '6Mai': '8882',
 '6Ngau': '9a33',
 '6Ngat': '5140 537c 5c79 5c7c 674c 6c54 77f9 77fb 9622 9623',
 '4Mang': '34bc 420d 750d 76df 840c',
 '6Ngap': '7810',
 '6Ngan': '3df3 4b97',
 '6Ngai': '3637 38fb 4883 4b73 4f2a 507d 50de 5453 56c8 57f6 5804 5bf1 69f8 6bc5 7768 7fbf 827a 84fa 85d9 85dd 897c 8a63 8be3 9b4f',
 '1O': '36b3 42cd 42ea 5a40 5c59 67ef 73c2 75b4 75fe 8efb 8f72 94b6',
 '6Ngak': '4a79 4c6e',
 '6Ngaa': '7811 8a1d 8bb6 8fd3',
 '5Niu': '349f 3735 3b64 40f5 465a 4b8d 5acb 5b1d 8311 8526 8885 88ca 892d 9ce5 9e1f',
 '5Zoeng': '3526 4369 4446 444b',
 '1Jeoi': '6875',
 '2Tim': '3401 35ad 3901 413c 5fdd 8214',
 '1Lung': '7abf 7ac9',
 '2Hoeng': '4291 4eab 54cd 56ae 664c 8801 97ff 98e8 9909 9957 995f 9977',
 '1Kaap': '3874 3aa7 3c5e 3d14 4375 43e9 4d8b 95df',
 '3Ngau': '8162',
 '6Gwaat': '3a7f 4097 4bc7',
 '5Wui': '3c71 86d5',
 '5Wun': '6665 7696',
 '6No': '3421 43a0 611e 61e6 61e7 7a2c 7cef',
 '2Sei': '6b7b',
 '1Bun': '4232 4313 4512 642b 642c 7c53',
 '1No': '632a',
 '1Buk': '3a10 3a63 535c 535f',
 '3Ngan': '47f4 8d0b 8d5d',
 '1Bui': '3b8e 676f 686e 76c3',
 '5Keoi': '62d2',
 '3Hing': '3deb 42dc 42ef 5910 5e86 6176 657b 77ce 78ec 7dae 7f44 8b26',
 '6Haai': '34a0 3ce6 4089 4aa5 4ad4 53e1 55d0 68b0 7023 85a4 89df 97f0 9f58',
 '1Zong': '3b9c 3f75 428b 4367 468e 5958 5986 599d 5e84 6869 6a01 7242 7ca7 81e7 838a 88c5 88dd 8ccd 8d13 8d1c 8d43 9ad2',
 '1Pang': '382e',
 '4Ke': '7638 9a91',
 '2Hai': '5338 55ba',
 '6Ngaang': '786c',
 '3Ding': '6917 7887 8ba2 9320 952d 9841 98e3 9964',
 '1Pong': '43fa 6ec2',
 '4Mo': '40fa 5298 587a 78e8 85e6 8611 9943 998d',
 '2Caang': '6a59',
 '2Bung': '3dbb 4a6c 4b70 6367 73a4 742b 83f6',
 '6Syu': '36f8 3710 4742 58f4 5c0c 6811 6a39 6f8d 7ad6 7aea 8167 88cb 8c4e',
 '4Mung': '3679 3813 3a5a 4004 4443 4445 4669 4711 4913 4c12 4c9b 4d0c 4d3f 4d46 5e6a 66da 6726 6c0b 6fdb 77a2 77c7 791e 8268 8499 9138 9725 995b 9e0f',
 '4Paan': '3c0b 4928',
 '5San': '80be 814e',
 '4Paai': '4c5d 4ff3 6392 68d1 724c 7bfa 7c30 9784',
 '2Bong': '3b84 4a54 4c03 699c 7253 78c5 7d81 7ed1 8180 84a1 938a 9551 9ac8',
 '4Mei': '38f2 3cfd 3d5f 3e0f 4260 42b3 4315 4577 4902 4969 4d22 519e 5833 5a9a 5d4b 5f25 5f4c 5fae 6520 6963 6e44 6ea6 7030 7222 7709 7cdc 7e3b 8587 863c 90ff 91be 9e8b 9ef4',
 '4Paau': '4819 5228 530f 5486 5e96 70b0 924b 9464 94c7',
 '4Tam': '4261 443a 4a35 4da8',
 '4Tai': '3592 3857 4536 4a11 4b2b 4c71 504d 557c 55c1 5824 5a9e 7994 7a0a 7d88 7df9 7ee8 7f07 855b 8b15 8e44 8e4f 918d 9684 984c 9898 9a20 9bf7 9cc0 9d5c 9d97 9d99 9dc8 9e48',
 '2Ceon': '35ba 3fe4 43db 440f 4b9e 5046 60f7 8822 8cf0 8e33',
 '6Aap': '3529',
 '4Taa': '4f57',
 '2Ceoi': '3a9c 4c9d 53d6 5a36 63e3 6f3c 8da1',
 '6Ging': '4514 501e 52b2 6b91 75c9 75d9',
 '4Tau': '388f 4d49 4ea0 5934 6295 982d',
 '6Geng': '64cf',
 '5Faan': '3ba5',
 '2Me': '6b6a',
 '1Cong': '4ed3 5009 51d4 5231 5275 6ca7 6ec4 75ae 7621 8231 8259 82cd 84bc 9dac 9e27',
 '2Mo': '6478',
 '6Mok': '3c73 3dec 3e6e 44ac 55fc 5bde 5e55 5e59 6f20 763c 7799 819c 83ab 911a 93cc 9546',
 '3Bek': '58c1',
 '3Pung': '63bd 692a 78b0 8e2b',
 '6Mou': '34bb 375f 384c 3854 39cc 3a9e 3aef 3d18 3eba 3fde 42c3 4a01 4cf1 52a1 52d9 52df 5893 5a7a 5aa2 5e3d 6155 620a 66ae 6958 6bf7 7441 770a 8004 8750 96fe 9727 9a16 9a9b 9da9 9e5c',
 '2Hiu': '4975 6653 66c9',
 '1Ang': '4aaf 4d0d 5624 56b6 66f4 7f43 83ba 97a5 9daf',
 '1Gung': '34b6 34db 38c9 39ec 3ad2 40b5 4221 48cf 4c38 4cb2 4ccd 4f9b 516c 5171 529f 5868 5bab 5bae 5de5 5f13 606d 653b 6db3 7d05 828e 86a3 8eac 8eb3 9b5f 9f94 9f9a',
 '2Him': '5d04 5dae 7303 736b 7381 9669 96aa',
 '6Loeng': '4036 4041 4eae 55a8 60a2 8ad2 8c05',
 '2Hin': '3aeb 3b0e 3c3d 3e42 4299 45fe 4606 496a 49cb 4a88 4b64 4bc8 5e70 663e 7e7e 7f31 86ac 8706 884d 8b74 8c34 9063 97c5 9855 986f',
 '6Zuk': '36e9 37ba 38c3 3954 39ba 3d40 3e9f 413e 42ed 457d 463b 479d 47c9 4831 4bb1 4fd7 65cf 6d4a 6fc1 7e8c 7eed 8233 84eb 85da 8e85 8ef8 8f74 9010 93c3 9432 955e 956f 9c41',
 '6Gaap': '386e 4caf',
 '6Zui': '5760',
 '5Piu': '3c8f 527d 6153 6b8d 779f 7f25 91a5 9860 9c3e 9cd4',
 '1Zek': '5527',
 '2Jung': '4fd1 5197 5b82 607f 62e5 64c1 6e67 6ec3 752c 81c3 84ca 86f9 8e0a 8e34',
 '2Nau': '599e 5ff8 626d 72c3 7d10 7ebd 9215 94ae',
 '1Niu': '3b42',
 '2Soeng': '60f3 8cde 8d4f 9bd7 9c9e',
 '1Nip': '93b3 954d',
 '1Nin': '59e9 810c',
 '2Nan': '393b 649a',
 '2Nam': '60d7',
 '1Nim': '4b2f 4d7f 62c8 68ef 7c98 9ecf',
 '1Nik': '397e 4065 460c 4d51 4d52 533f 5adf 615d 6426 6635 66b1 6eba 783e',
 '1Haang': '4aab 542d 5751 592f 8a99 962c',
 '4Pun': '3a6f 4c14 5e4b 69c3 6d00 6e53 76c6 76d8 76e4 78d0 78fb 7e0f 97b6',
 '4Pui': '37dd 4aca 57f9 5f98 6bf0 88f4 8ce0 8d54 966a',
 '3Co': '3d3d 47f6 4958 4d2d 4fb3 5249 55bf 61c6 632b 839d 84cc 92bc 932f 9509 9519',
 '1Aang': '44e8 47b7 7507 7516 7f42 7f4c',
 '6Fuk': '3686 3a07 3b3c 47ee 4a31 4d57 590d 5b93 670d 6d11 7b99 832f 8567 8659 88b1 9d14 9d69',
 '6Koek': '35b8 3dfe 4418 81c4',
 '3Ce': '3f9d 659c',
 '6Zin': '3993 8ce4 8d55',
 '6Zim': '389b 3dd4 4dab 807b 87b9 9b59',
 '6Zik': '344a 3534 3601 3677 36a4 381c 38c2 3d12 3da0 3f8a 4129 42da 433b 44c8 461d 48e7 49b4 4b5e 4bc6 4d2c 5024 503c 57f4 5915 5bc2 5e2d 690d 6b96 6c50 6f03 76f4 77fd 7a19 7a78 7c4d 8024 892f 8ea4',
 '6Ziu': '3781 387d 3a04 3fe2 49c2 564d 56bc 5797 8d75 8d99',
 '6Zit': '3a17 3a29 3b86 4316 4559 4601 4bf5 622a 776b 8e15',
 '1Cang': '34cc 47d6 47eb 4800 564c 7ac0 7f49 943a 94db',
 '6Zip': '3416 3595 35fc 3738 3a91 3c49 3d2a 3e23 420e 49e8 4b4e 4b5f 4c9c 55eb',
 '3Cip': '37a5 3af8 3c31 546b 553c 59be 83e8 8a40 8dd5 8e25 9bdc',
 '1Waat': '3ba7 44db',
 '3Ciu': '3d25 41cc 4fcf 5ced 5e29 8096 8a9a 8bee 9798',
 '3Cit': '3610 3b1a 3b6f 3bd9 3c4c 4486 4c68 5c6e 5f7b 5fb9 61d8 63a3 64a4 6f88 7869 850e 8a2d 8bbe 8e05 8f4d 8f99',
 '4Waan': '3872 3975 3a21 3d8e 437a 468b 4c1f 5bf0 6fb4 73a9 73af 74b0 7d84 7f33 8fd8 9370 9436 953e 956e 95e4 961b 9811 9b1f',
 '6Dat': '51f8 6022 7a81 8456 87d8 9f35',
 '2Kiu': '563a',
 '6Dap': '36e5 38db 39fa 3b7c 3ceb 3e7a 3fef 420b 435d 447d 44e0 4cf4 4d80 4d81 6e9a',
 '4Waai': '3522 3a0a 40f6 6000 61f7 69d0 6af0 6dee 7372 8639 8922 8931',
 '6Dam': '41b1 4716 4b5b',
 '1Ceng': '4672 6e05 9752',
 '6Dan': '40b4 7096 71c9',
 '6Dai': '57ed 5a23 5f1f 608c 6755 68e3 777c 7b2b 7b2c 8e36 9012 902e 905e 9070 96b6 96b7 96b8 9de4',
 '6Dak': '7279 7286 87a3',
 '1Waan': '3624 3f14 5a60 5f2f 5f4e 6e7e 6f6b 7063 8c69',
 '3Cim': '37fb 3c07 3c48 4b55 50ed 5811 5879 58cd 6920 69e7',
 '1Waai': '37ea 3e7b',
 '1Kung': '7b3b',
 '2Ling': '64f0',
 '5Wo': '798d',
 '1Ti': '4e05',
 '2Kui': '3c6e 3dd0 3e5f 4210 4661 46e9 483f 4aed 4b5d 4be4 4fa9 5108 523d 5ee5 6126 6192 65dd 6703 6867 6a9c 6d4d 6e83 6f70 6fae 72ef 736a 7e62 7f0b 8069 8075 810d 81be 835f 8588 90d0 9136 9c60 9c99',
 '1Jaa': '5414',
 '5Haam': '58cf 6496 69db 6abb 8c4f 9921 9985',
 '3Gwong': '7926 945b',
 '4Ngou': '37fc 3802 498b 4d45 5ed2 646e 6556 6ef6 71ac 720a 7488 78dd 7ff1 7ffa 8071 87af 8b37 9068 93ca 93d6 969e 9a41 9a9c 9c32 9ccc 9f07',
 '4Jyun': '4f2d 5143 5213 539f 5458 54e1 56ed 5706 5712 5713 571c 57b8 5827 58d6 5a9b 5ac4 5b8c 5c8f 60ac 61f8 628f 63f4 676c 69ac 6a7c 6ade 6c4d 6c85 6cbf 6d39 6e72 6e90 70ab 7230 7328 733f 7384 76f7 7d08 7de3 7ea8 7f18 7fb1 8284 82ab 849d 8696 875d 876f 8788 8881 8c72 8f45 8f95 908d 90a7 9206 925b 94c5 9695 9a35 9da2 9eff 9f0b',
 '1Mo': '4a4b 4b69 4be2 56a4 6469 64f5 9b54 9ebd',
 '4Ngoi': '9a03',
 '1Me': '54a9 54f6 5b6d 7f8b 8288',
 '6Diu': '390d 3a95 40ad 40bd 7a95 7f40 84e7 8c03 929a 94eb',
 '6Nan': '3b2e',
 '3Wan': '6120 614d 63fe 7174 7f0a 8574 860a 915d 919e 97de 97eb',
 '3Wai': '5612 5c09 6167 6170 729a 7369 754f 78a8 79fd 7a62 7f7b 7ffd 851a 8589 893d 9728 9935',
 '3Pek': '431f 5288',
 '3Pei': '3d68 3fd9 581b 5b16 5c41 5e14 6bd8 6de0 6fde 8b6c',
 '6Nap': '3555 35a0 3a25 41f1 422b 47dc',
 '6Nau': '6abd 8028 81a9 9392',
 '6Nat': '80ad 8a25 8bb7',
 '3Do': '6736',
 '2Cin': '389f 4914 5181 563d 56c5 5e5d 6d45 6dfa 71c0 8487 8546 9322 94b1 95e1 9610',
 '3Zoek': '3a71 3e71 4175 4cb5 4d82 5981 65ab 65ae 6ae1 7235 799a 8457 914c 96c0',
 '2Ciu': '6084 6100',
 '6Zau': '3807 413b 4219 4a5c 4d87 4f37 50e6 5191 5b99 5c31 5cab 7c40 7d02 7ea3 80c4 8896 914e 9df2 9e6b',
 '2Bou': '3645 365b 3ec4 42e0 4382 4a94 4cc8 4cf0 4d10 4fdd 5703 5821 5aac 5b6c 5b9d 5bf6 8446 8865 88dc 8913 99c2 9d07 9d4f 9e28',
 '4Mong': '37cc 385b 3d07 4152 459f 45c8 4970 4d68 4ea1 5166 5396 54e4 5c28 5eac 5fd8 5fd9 6757 6767 6c52 727b 72f5 786d 8292 832b 8609 867b 8771 9099 92e9 94d3 99f9',
 '4Pong': '3751 3936 396c 3ac4 416d 43fe 4a26 4bbe 5e9e 5fac 65c1 7be3 8783 9004 96f1 9736 9f8e 9f90',
 '2Seng': '9192',
 '3Joek': '7d04 7ea6 85ac 8dc3 8e8d',
 '1Mok': '3ffa',
 '3San': '393e 3a9a 4ac5',
 '2Gun': '43d3 463e 498e 4a6a 4c98 65a1 670a 742f 75ef 7b66 7ba1 8118 8218 9327 9928 9986',
 '4Ping': '3eb8 3ec2 4d84 576a 5e21 5e32 67b0 6d34 7aee 7f3e 82f9 8353 840d 8a55 8bc4 8eff 90f1 9a08',
 '5Ming': '37f0 76bf',
 '2Gui': '464c',
 '2Daai': '373d 3c80 469f 50a3 5e26 5e36 6b7a',
 '3Paai': '3b5b 848e',
 '6Zaai': '345c 3582 36f3 36ff 3a5f 3d69 3e6d 413f 41e9 4388 46d6 5be8 5ecc 7725 7826 8c78',
 '3Paak': '62cd 73c0',
 '2Daam': '40eb 4ab4 4c4b 80c6 81bd 9ed5 9ef5',
 '6Zaan': '36ba 3834 38e7 3d08 3e3e 3ef5 463a 4a2f 4c60 4d74 63dd 6525 6808 7dbb 7efd 8b54 8f1a 8f4f 994c 9994',
 '6Zaam': '391b 3bb3 3db5 4015 41ee 42d5 49f5 6682 66ab 7ad9 8cfa 8d5a 8e54 933e 93e8',
 '2Daan': '758d 86cb',
 '1Nuk': '6067',
 '1Ce': '3a02 5513 5962 6aa8 7817 7868 8eca 8f66',
 '6Baang': '7e43 8e66',
 '1Ci': '35a2 35e9 36d3 3796 3c1e 3c87 3df0 3e88 40a3 415a 41ea 428d 49dd 49f3 4b1c 4cad 4cc4 5068 54e7 55e4 5601 5ab8 645b 6521 74fb 75b5 75f4 7661 7721 7735 7b1e 7d7a 80d4 80f5 86a9 87ad 90d7 96cc 98fa 9ab4 9b51 9d1f 9e31 9ed0 9f5d',
 '6Zaap': '3804 380e 39bb 3fc7 4481 4e60 4ebc 5d8d 6742 69e2 7fd2 88ad 8972 9491 9598 95f8 96b0 96c6 96dc 96e5 972b 98c1 9c3c 9cdb',
 '2Daau': '36d2 39f7 41fa 4487 46e0',
 '3Paau': '4d8c 5945 75b1 76b0 791f 792e 8db5',
 '6Zaau': '3911 3a46 3e68 3ea0 4001 4162 419c 4298 68f9 6ac2 9a5f 9aa4 9d6b',
 '6Zaat': '35cc 35e7 378f 3cd1 3d01 43c4 44a4 4ad5 4b39 4b7f',
 '3Koek': '3541 3fa1 5374 537b',
 '1Ngaau': '3f5b 4b32 630d',
 '1Ngoi': '6c28',
 '6Bit': '3fdc 4825 4b71 5225 522b 762a 765f 8952 8e69 999d',
 '6Bin': '3b53 3cce 3d77 3e95 3eb9 4278 44aa 46d2 4abb 4fbf 535e 5f01 5fed 6283 662a 6c73 6c74 72bf 8251 8fa8 8fa9 8faf 91c6 959e',
 '6Bik': '714f',
 '6Zaa': '5d56 62c3 67de 81aa',
 '1Ngaam': '5571',
 '2Fau': '5256 5426 7f36',
 '3Zou': '7076 7ac8 8444',
 '6Aa': '90a3',
 '1Gyun': '3cd9 414c 48fa 4a70 4ccc 5a1f 6081 61c1 6350 6d93 72f7 7404 74f9 774a 8832 8eab 9799 99fd 9d51 9e43',
 '2Pou': '4c95 666e 6c06 6d66 6ea5 8b5c 8c31 9420 9568',
 '2Wun': '3f1d 4471 463c 4bdb 6365 6900 6d63 6fa3 76cc 7746 7897 8155',
 '6Jyun': '613f 63be 6621 7457 7729 7e6f 8852 8911 9858',
 '2Wui': '36f1 4f1a',
 '6Deoi': '4a34 4bdf 603c 619d 61df 85b1 8b48 931e 9413 961f 968a 99fe',
 '6Jyut': '5216 54d5 5666 6085 60a6 6209 6288 6708 6a3e 6ce7 73a5 7a74 7ca4 7cb5 807f 8aaa 8d8a 8ecf 925e 9487 94ba 95b1 95b2 9605',
 '6Deon': '491c 4f05 56e4 6c8c 76f9 9041 906f 920d 949d 9813 987f',
 '2Fan': '3f9c 46c0 7c89',
 '3Zok': '3445 3600 3930 3e32 43b0 476b 4f5c 6628 98f5 9ddf',
 '2Fai': '75bf 75f1',
 '3Zoi': '4d67 518d',
 '4Peng': '546f 5e73 74f6',
 '4Lou': '35e6 37c9 3820 38b3 3a13 3aad 3b54 3c7a 3fd6 470e 472e 4741 484e 4c15 4c8f 50d7 52b3 52de 5362 5520 562e 5786 58da 5e90 5eec 680c 6ae8 6cf8 6d76 7018 7089 7210 7262 7388 75e8 7646 76e7 7c29 7c5a 7e91 7f4f 80ea 81da 823b 826b 82a6 8606 87e7 8826 8f51 8f64 8f73 91aa 9412 946a 94f9 9871 9885 9c78 9c88 9e15 9e2c',
 '3Heoi': '3966 39c1 3ae2 3c26',
 '1Cou': '64cd 7c97 89d5 9ea4',
 '4Syun': '3bc0 3cec 4062 4063 45e0 4967 4c82 4d09 4d0b 65cb 6f29 7401 7487 74bf 8239 9084 93c7 955f',
 '2Jam': '98f2 996e',
 '4M': '5452 5514',
 '4Pang': '394a 3c76 3d57 3e24 51ed 670b 78de',
 '4Laau': '3737 37fa 385e 3920 3cc5 3e8f 411b 42f7 44dd 45dc 4bab 4c56 635e 6488',
 '1Gou': '37f8 3912 3c0f 4181 69d4 69f9 6adc 768b 777e 7bd9 7cd5 7f94 818f 81ef 993b 9ad8 9f1b',
 '2Bang': '752d',
 '6Doek': '8bfc',
 '1Gon': '3da5 3ffb 49f2 4e79 4e7e 5e72 6746 687f 6f27 7395 77f8 7aff 809d 8677 9150 9cf1',
 '1Goi': '3a1f 3c6f 3c7e 402d 44a7 4da3 4f85 5793 7974 7d6f 8344 8a72 8be5 8cc5 8ccc 8d45 9654 9691',
 '4Laai': '394e 3992 3c1a 4527 4d53',
 '4Laan': '3613 3cd5 49a8 4a8d 5170 62e6 6514 6593 6595 680f 6b04 6f9c 703e 7061 7c63 7cf7 862d 8974 8b95 8c30 946d 9567 95cc 9611',
 '4Laam': '37a9 41be 4340 4ad0 4c10 5682 5a6a 5c9a 5d50 60cf 7201 791b 7b96 7bee 7c43 84dd 85cd 8934 8964',
 '6Zyut': '3503 42d0 4380 43b3 47e0 4881 4926 4b02 7edd',
 '3Caai': '35eb 3d1e 460d 47d9 562c 56c6 7625 867f 8806',
 '3Tou': '3690 514e 5154 580d 5957 7ee6 83df',
 '3Caak': '397d 3a32 3a4d 3fed 4282 518a 518c 5720 577c 62c6 67f5 6805 7b56 7b74 7c0e 8326',
 '3Caam': '471f 47c3 4c7f 5133 5fcf 61fa 6749 8215 8b96',
 '3Caan': '36d1 38d3 3eee 47e2 5b71 6faf 71e6 74a8 7cb2 7fbc',
 '3Caap': '3b01 3b91 63d2 81ff 9364 9538',
 '3Caau': '360a 43eb 4d4e 6477 808f',
 '3Caat': '355e 40f0 4553 4daa 5237 5239 524e 5530 5693 5bdf 64e6 736d 737a 7924 8a67',
 '2Sing': '376d 3b90 4cd9 7bb5',
 '3Tok': '3d75 4e47 4f82 5eb9 6258 67dd 69d6 6a50 6cb0 7ba8 7c5c 8600 8a17 8bac 8dc5 98e5 9966 99b2 9b60',
 '6Zyun': '400f',
 '2Dou': '3800 3a36 3d54 403e 5012 5835 58d4 5c9b 5cf6 5e3e 6363 6417 64e3 7779 7982 79b1 89a9 8ced 8d4c',
 '6Leoi': '3466 3527 3825 3d03 41d4 426a 44de 451e 4968 52f4 5533 616e 623e 6cb4 6cea 6dda 6ee4 6ffe 76ed 790c 79b7 7c7b 7e87 8012 8631 8651 8a84 985e',
 '6Leon': '3581 3f08 426e 4af0 541d 7d0a 853a 85fa 8bba 8e78 8e8f 8eaa 95b5',
 '6Leot': '39d2 3b9a 3bdd 3cda 42d6 4501 4b06 4d36 51d3 564a 5d42 5f8b 6144 642e 6817 6ea7 7387 746e 7849 7be5 7e42 7e58 819f 844e 9079 9731 9dc5 9e9c',
 '5Miu': '3acf 3ad0 3cd0 3d2d 404f 419e 43da 46ba 676a 6dfc 6e3a 7707 79d2 7df2 7f08 85d0 9088',
 '5Min': '36ef 3a3a 3d10 400e 445a 4904 4e0f 506d 5195 52c9 52d4 6110 6c94 6e4e 7704 7dec 7f05 817c 9766 9bb8',
 '6Kek': '360c 4e6a 5267 5287 5c50',
 '1Cau': '403a 418b 42fa 4410 4708 4a02 4ca1 5062 62bd 63eb 640a 6978 714d 7633 79cb 79cc 7bd8 7d2c 7de7 8429 97a6 97a7 9c0c 9c0d 9cc5 9d96 9e59 9f9d',
 '1Cat': '3b4d 3bc3 4e03 67d2 687c 6f06',
 '6Wing': '3bcb 548f 6cf3 6f41 799c 7a4e 8a60 988d 988e 9896',
 '1Cap': '38ec 54a0 6222 6fc8 7dc1 7ddd 7f09 847a 857a 8f2f 8f91',
 '3Deng': '639f 77f4',
 '6Joeng': '58cc 6059 6837 6a23 6f3e 7001 7f95 8b93 8ba9 917f 91c0',
 '1Caa': '398b 4461 4c96 53c9 55b3 55cf 5c94 5dee 6260 6748 6b3b 6c4a 8256 8e45 91f5 9497',
 '1Can': '3947 3972 3c5a 4254 441c 4641 4eb2 55d4 778b 89aa',
 '1Cam': '4081 4737 4fb5 5d7e 68fd 7d85 8ac3 99f8 9a8e',
 '1Cak': '5ec1 607b 6d4b 755f',
 '1Cai': '377d 4424 51c4 59bb 60bd 6816 68f2 6c8f 6dd2 840b 90ea',
 '2Zyu': '4c1e 4e36 4e3b 62c4 6e1a 7151 716e 967c 9e88',
 '2Ji': '501a 506f 6246 65d6 7dba 7eee 9321 951c 966d 9f6e',
 '6Geoi': '34da 3718 382a 48f0 4d99 5177 5be0 5ca0 5de8 60e7 61fc 70ac 728b 79ec 7aad 7ab6 7c94 82e3 8661 86b7 8a4e 8bb5 8c66 8ddd 907d 91b5 9245 943b 949c 98b6 98d3 99cf',
 '3Fat': '399f 6cb7',
 '6Laam': '372e 3e8c 432b 6ee5 6feb 7e9c 7f06 8230 8266 8f5e',
 '6Laan': '3c6b 40f9 70c2 721b',
 '6Laai': '4124 42e5 44f6 4608 4695 4732 48e6 6fd1 7028 7c41 7c5f 843d 85fe 8cf4 8d56 9179 983c',
 '6Laak': '3580 52d2 7833 808b',
 '6Laau': '3669',
 '6Laat': '3ecb 3edd 4c6b 4d9b 5217 524c 8fa2 8fa3 8ffe 9b0e',
 '6Laap': '40b0 40f3 45f6 4d98 5783 7acb 7fcb 814a 81c8 81d8 881f 908b 945e 9c72',
 '6Juk': '5125 5809 5ab7 5b8d 5cea 617e 6b32 6d74 6def 6ebd 72f1 7344 7389 7e1f 7f1b 8089 80b2 84d0 8925 8fb1 910f 923a 9325 94b0 9b3b 9cff 9d52 9e46',
 '3Fai': '351e 3657 368d 3755 37f4 3d52 3fb1 4454 4560 4626 46a8 46cd 4c01 4d90 4ff7 5e9f 5ec3 5ee2 602b 6ae0 7648 80ba 82be 889a 8d39 9428 9544',
 '3Fan': '34e9 3c75 3f79 4000 4b1a 7035 7793 7caa 7cde 8a13 8bad',
 '6Ping': '959b',
 '4Saan': '4888 6f7a',
 '2Naan': '6201',
 '4Saam': '37a4 4960',
 '5Jau': '4e23 5363 53cb 6709 69f1 7256 7f91 83a0 8a98 8bf1 9149 92aa 94d5',
 '1Bei': '3817 3859 3bc5 45b0 469c 4bd7 4d3d 5351 5421 5c44 5ca5 5eb3 60b2 6911 72a4 7891 7be6 7f74 7f86 8406 84d6 85e3 88e8 8a56 8bd0 9642 9d6f 9e4e',
 '5Jan': '542c 5f15 622d 6704 763e 7e2f 8693 87be 9777',
 '5Jam': '834f 887d 98ea',
 '5Jai': '66f3',
 '1Lei': '550e 55b1 9b01',
 '2Naau': '36bc 3eb2 4087 43d4',
 '5Jaa': '4e5f',
 '4Cau': '393d 399e 4327 436d 43bf 4ec7 4fe6 5114 56da 5d37 5e31 5e6c 60c6 61e4 6cc5 72a8 7574 7587 7a20 7b79 7c4c 7da2 7ef8 8764 88ef 8a76 8b8e 8b90 8e0c 8e8a 9052 914b 9158 9167 916c 96d4 96e0 9b82',
 '3Hong': '7095 70d8',
 '4Caa': '3810 3889 3892 405f 4c79 579e 643d 67e5 67fb 6942 69ce 78aa 78b4 8336',
 '1Saang': '3f52 4cbc 4d24 7272 7424 751f 7525 7b19 80dc',
 '4Cam': '36af 371e 3726 3da3 3e25 3e91 437c 449e 471b 4aee 4c3c 565a 5bfb 5c0b 6326 648f 6c88 6c89 6d54 6f6f 704a 71c5 71d6 8368 8541 9129 9703 9b35 9b6b 9c58 9c9f',
 '4Can': '5875 5c18 852f 87b4 9648 9673',
 '3Gun': '342b 3ba1 3d26 3dc4 45f0 466e 477a 60ba 6bcc 6dab 704c 721f 74d8 77d4 7936 797c 7f50 8cab 8d2f 9475 9e1b 9e73',
 '1Fuk': '364f 3805 42f9 433f 454e 4add 5e45 798f 8179 844d 8760 876e 8907 8f39 8f3b 8f90 9351 99a5 9c12 9cc6',
 '1Fui': '369d 3780 3a12 3b22 3dc7 3faf 45af 4747 47e9 594e 6062 609d 6ac6 7070 867a 8a7c 8bd9 8c57 9b41',
 '1Fun': '3a72 3e55 3f2b 4754 56be 5bbd 5bec 61fd 6b22 6b61 737e 81d7 8b99 8c9b 9a69 9acb 9ad6',
 '6Se': '4005 4836 9607 9e9d',
 '6Si': '34be 3c41 3e37 3e5d 4293 4734 473b 4b44 4e8b 4ed5 4f8d 5033 525a 58eb 623a 662f 6c0f 793a 8906 8996 89c6 8adf 8c49 8df1',
 '3Bo': '5643 64ad 8b52 9131 9643',
 '6Sik': '8680 8755',
 '6Sin': '4120 4476 48ad 5584 58a0 58a1 5b17 64c5 6a3f 7e55 7f2e 7fa8 81b3 912f 994d',
 '6Sim': '8d0d 8d61',
 '1Be': '3f30 5564',
 '6Siu': '436e 45ba 5146 52ad 5372 53ec 65d0 7bbe 7d39 7ecd 8087 8088 8891 90b5 9ba1',
 '2Gwaa': '4211 5be1',
 '5Nou': '344e 36f4 3e81 3f85 4700 4727 52aa 5f29 607c 60a9 60f1 7459 782e 78af 8111 8166',
 '1Dyun': '526c 7aef 8011',
 '2Gwai': '3e35 40fd 436f 4f79 5326 532d 53ac 579d 59fd 5b84 6677 6739 7c0b 8a6d 8be1 8ecc 8f68 9b3c',
 '2Gwan': '3bfb 40c2 4707 638d 6eda 6efe 7dc4 7ef2 8509 886e 889e 88ac 88f7 8f25 8f8a 9bc0 9ca7',
 '5Lou': '352a 389a 39ef 3bed 4764 4977 4c90 4cd3 4d4f 5364 59e5 6045 63b3 64c4 6833 6a79 6ad3 6c07 6c0c 6ef7 72eb 78e0 8001 8263 826a 84fe 864f 865c 92a0 942a 94d1 9b6f 9c81 9e75 f936',
 '2Cong': '3ae4 45ee 53b0 5ee0 60dd 6203 655e 6636 6c05 92f9 95d6 95ef 99d4',
 '1Aau': '4071 4269 4271 4654 4723 4895 4972 4cfc 51f9 5773 6293',
 '3Neoi': '3d56',
 '3To': '553e 6bfb',
 '5Kaam': '3c39',
 '1Hon': '520a 54b9 770b 9807 9878',
 '5Mun': '36e7 6ee1 6eff',
 '5Mui': '4180 6334 6bcf 6d7c',
 '1Zuk': '3922 3c32 425b 4335 496e 4cd1 54eb 5631 56d1 6349 65b8 67f7 6b18 70db 71ed 7603 77a9 77da 795d 7af9 7afa 7b51 7bc9 7ca5 81c5 880b',
 '3Faa': '5316 6779',
 '3Taat': '3493 38f5 3ce0 3ff9 631e 7f8d 8df6 95e5 95fc 9791 97c3',
 '3Taau': '47ac',
 '3Ngou': '5db4 82ba',
 '3Taap': '366e 3bd3 40bf 4308 584c 5854 62d3 6428 69bb 6bfe 6dbe 891f 8abb 9062 9389 95d2 9618 9c28 9cce',
 '1Tau': '5077 5078 5aae',
 '3Ngon': '6848',
 '1Lo': '56c9 56d6 651e 65ef',
 '3Taan': '36f6 53f9 5606 6b4e 70ad 78b3 8d03 9d20',
 '4Ciu': '43ad 4a4c 5af6 5d95 6194 6641 671d 6a35 6f6e 71cb 7744 77a7 8b59 8c2f 936b 9866',
 '1Tam': '3d45',
 '1Tan': '3ce9 4d4d 541e 66be 6d92 98e9 9968',
 '3Ngoi': '611b 7231 74a6',
 '1Taa': '4ed6 5b83 6039 7260 7942 9248 94ca',
 '1Le': '54a7 54e9',
 '1Nguk': '526d 5c4b',
 '3Zaang': '499b 49b6 8be4',
 '1Po': '6a16 9817 9887',
 '5Pang': '3bc1 4c37',
 '4Se': '3d43 47d7 4f58',
 '3Zing': '3860 4223 466c 496d 5e27 5e40 653f 8a3c 8b49 8bc1',
 '2Seon': '353c 398f 69ab 7b0b 7b4d 7c28',
 '3Geoi': '422e 46ef 4d55 5028 5c66 5c68 64da 6fbd 77bf 8e1e 8e86 92f8 952f',
 '2Seoi': '3454 4161 48a4 6c34 7cc8 9191',
 '2Cung': '3509 51a2 585a 5ba0 5bf5',
 '6Dung': '34ca 357c 38a5 3f2f 419a 4792 4f97 52a8 52d5 578c 5cd2 606b 6078 615f 630f 6d1e 7850 80f4 8855 8ff5',
 '3Laat': '760c',
 '6Zyu': '3cbe 3fbb 408a 4618 4631 4d01 4f4f 7b6f',
 '6Ngaan': '8d17 96c1 9d08',
 '6Ngaai': '366f 3712 3f48 4268 42d9 4758 4e42 5208 827e',
 '4Fong': '59a8 9632 9b74 9c82',
 '6Ngaak': '5cc9 8a7b 981f 984d 989d',
 '3Deoi': '34a6 3642 37cb 381a 3b23 3cd4 3d5d 41cf 41d5 4b08 4b3d 514c 5151 5689 5bf9 5c0d 7893 794b 8187 8f5b',
 '4Lei': '34bf 34ef 3ca0 3e48 40be 411c 417b 4360 4366 4506 45bd 45bf 4670 4c58 4d3b 5398 5ae0 5b4b 68a8 68a9 68c3 6a06 6c02 6f13 6f26 7055 729b 72f8 7315 737c 740d 7483 74c8 79bb 7b63 7bf1 7c6c 7e2d 7f21 7f79 84e0 863a 870a 8935 8b27 8c8d 91a8 96e2 973e 9a6a 9a8a 9c7a 9ca1 9d79 9e1d 9e42',
 '3Cyu': '458f 5904',
 '6Ngaat': '3433 3795 3f7e 43cc',
 '6Caak': '4c39 621d 8808 8cca 8d3c 9c97',
 '4Ceng': '6674',
 '6Deng': '57de 5b9a 8a02',
 '6Hip': '52f0 534f 53f6 62f9',
 '1Wong': '3d0f 5c22 5c2a 6c6a',
 '4Baang': '562d',
 '3Gaau': '35d5 370c 3745 3871 3939 3c83 3cf6 3daf 40d3 44d8 46ee 4ab7 4b28 654e 65a0 6821 6ed8 6f16 73d3 7a8c 7a96 8990 899a 89ba 89c9 8f03 8f83 9175 9ab9',
 '3Gaat': '8ecb 8f67 91d3 9486 9782',
 '6Mik': '3a20 3d4b 4310 4bad 5196 51aa 5853 5e42 5e4e 5e6d 7cf8 7f83 8993 89c5 9f0f',
 '3Gaap': '3545 3753 3fd3 402b 419f 455b 4a61 550a 5939 593e 5cac 643f 689c 6d79 7532 80db 823a 835a 83a2 86e4 86f1 86fa 88b7 88cc 8df2 90cf 90df 9240 92cf 94be 94d7 9788 97d0 9830 988a 9d3f 9d4a',
 '6Min': '9762 9eaa 9eb5',
 '4Wo': '44a9 548a 548c 76c9 79be 924c 9fa2',
 '2Goeng': '36e8',
 '4Wu': '35c5 3bdb 3f47 3fb0 406b 427f 4280 4381 4812 4a74 4b45 4b4c 58f6 58fa 5f27 695c 6e56 72d0 7322 745a 7cca 80e1 846b 8774 885a 9190 992c 9b0d 9b71 9d98 9da6 9e55',
 '3Gaam': '3a37 3a5c 41bf 4af4 4d96 5c32 6a44 6de6 9274 9451 9452',
 '5Pong': '43a7 45ab 802a 868c',
 '3Gaan': '6a7a 6da7 6f97 77b7 8aeb 8c0f',
 '6Mit': '349d 423c 4329 460a 4a4f 6ec5 706d 70d5 7bfe 8511 881b 884a',
 '6Miu': '5999 5e99 5edf 7385',
 '3Gaak': '35c6 39b4 4419 46ff 4a23 4a82 4ee1 54af 55dd 572a 5865 6105 630c 683c 6ec6 7fee 80f3 8188 8316 89e1 9398 9549 9694 9769 9abc 9b32',
 '1Se': '3c54 3f9a 4e9b 8cd2 8d4a',
 '2Hou': '597d',
 '4Ji': '3794 4e84 4eea 4f41 4f87 5100 5152 531c 53f0 54a6 5532 572f 5937 59e8 5b9c 5ba7 5db7 5f5d 6021 64ec 6818 682d 684b 6938 6c82 6cb6 6d0f 6d1f 72cb 7591 75cd 7691 769a 7731 79fb 7bb7 7c03 7fa0 800c 800f 804f 80f0 80f9 81d1 8351 86c7 8794 886a 8898 88b2 89fa 8a11 8a52 8abc 8bd2 8c0a 8ca4 8cbd 8d3b 8f00 8fc6 8fe4 8ffb 914f 9651 9824 9890 98f4 9974 9b9e 9c95 9d2f 9d3a 9e03 9e38',
 '2Hoi': '4057 51ef 51f1 5240 5274 57b2 584f 5d66 607a 6137 6d77 78d1 91a2 93a7 94e0 95d3 95ff 98bd',
 '2Hon': '3e01 4350 4351 43af 4cda 4f83 520b 634d 7f55 884e',
 '1Si': '348b 3567 36b8 37c3 383c 3d32 3e87 3ea8 3eb0 3f44 41c1 4324 4333 452e 45d0 467e 4873 4ae2 4c89 4ca9 4d13 4e1d 4fec 5072 50ff 51d8 53ae 53f8 5c38 5c4d 5e08 5e2b 5edd 601d 6495 65af 6eae 6f8c 72ee 7345 7997 79a0 79c1 7cf9 7d41 7d72 7de6 7e9f 7f0c 7f73 8479 84cd 8652 86f3 8724 8784 87f4 8937 8979 8a69 8bd7 90bf 917e 91c3 9270 9376 9536 98b8 98d4 9c24 9cf2 9de5 9e24 9e36 9f36',
 '3Syu': '38fd 3d42 4240 4b04 55fb 5eb6 6055 620d',
 '2Dam': '356a 3c7d 3dcb 628c 6c8a',
 '2Taai': '3e1f',
 '2Taam': '55ff 64a2 6a5d 6bef 83fc',
 '2Taan': '3adc 4994 4eb6 5766 5fd0 75b8 8892 8962',
 '6Lei': '35da 3926 4144 41d0 4fd0 5229 540f 6d70 6d96 7301 75e2 8137 8385 8389 849e 8a48',
 '6Lek': '58e2 7667',
 '4Waa': '37c6 398a 3b49 417f 4522 4da4 5212 534e 5d0b 6a3a 93f5 94e7 9a4a 9a85',
 '4Wan': '370f 392a 3baf 4259 4290 4b9d 4d37 4e91 52fb 5300 544d 5998 6600 6655 6688 6c84 6d51 6e3e 6eb3 6f90 73f2 743f 7547 79d0 7b7c 7bd4 7d1c 7ead 8018 82b8 8553 90e7 9116 92c6 96f2 987d 991b 992b 9984 9b42 9f32',
 '4Wai': '3822 465f 49e6 4e3a 56d7 56f4 570d 5e0f 5e37 5e43 60df 6da0 6e4b 6f4d 6f7f 6ff0 70ba 7232 73ae 744b 7dad 7ef4 8fdd 9055 9057 907a 912c 95c8 95f1 97cb 97e1 97e6',
 '6Lang': '9b05',
 '4Taan': '40ea 575b 58c7 679f 6a80',
 '6Pun': '8450',
 '5Kwaai': '39df 3a24 405b',
 '6Baan': '4a7f 4b31 529e 6e74 74e3 8fa6',
 '5Ling': '9886',
 '4Cyun': '34b0 37e8 37eb 381d 3bcc 3f37 402c 4c16 4f20 4f7a 50b3 5168 5b58 5dd1 606e 6505 6512 6522 693d 6b11 6cc9 7277 75ca 7b4c 8343 8a6e 8be0 8de7 8e72 8f07 8f81 9044 919b 9293 94e8',
 '3Mut': '4b34 6cab 79e3',
 '1Gwong': '3ad5 5149 6844 6d38 80f1 92a7',
 '1Hing': '3b62 417d 5144 5174 537f 6c22 6c2b 8208 944b 99a8',
 '2Jyu': '4f1b 50b4 59aa 5ad7',
 '1Him': '5ffa 8b19 8c26',
 '1Hin': '35d4 39db 3dff 3e1d 41c2 466d 46c8 47bf 4853 5615 5c8d 6106 6380 6434 6510 6513 6774 6c67 7275 727d 7945 7e34 8930 8ed2 8f69 9a2b 9a9e 9db1',
 '6Baau': '9ab2 9c8d',
 '3Soek': '407b 524a 70c1 9460 94c4',
 '1Juk': '5590 5662 5f67 65ed 6631 6bd3 6c83 6fb3 715c 71e0 8581 9033 90c1 92c8 96a9',
 '4Cai': '3a93 4d92 61e0 8269 85ba 86f4 8810 9f4a 9f50',
 '1Hiu': '357a 36a0 3809 3897 48aa 547a 54d3 5635 56a3 56c2 67ad 67b5 689f 6b4a 735f 7362 7e51 81ae 85c3 8648 8dac 8df7 8e7a 9121 9a4d 9a81 9d1e 9e2e',
 '2Lyun': '3703 373b 604b 6200',
 '3Daam': '77f3',
 '4Pai': '4ab9',
 '1Hung': '34d9 3573 35e2 3c81 3e1c 415d 41aa 5025 5147 51f6 5308 543d 54c5 5d06 5ff7 605f 60be 6c79 6d36 7a79 7a7a 7b9c 80f7 80f8',
 '3Daai': '35e3 3985 3bc2 3be9 45d6 47e1 4b3e 6234 8515 8976 8e5b',
 '3Him': '3543 3898 3aa0 3c38 4234 42b4 4ae1 5094 614a 6b20 6b49 82a1',
 '4Paa': '43ac 43b1 63b1 6733 6777 722c 7436 7b62 8019',
 '3Daat': '4d63 547e 59b2 601b 64bb 7b2a 8e82 977c',
 '3Hip': '365d 38b5 3e64 3f80 43ee 47d0 52a6 5354 55cb 602f 60ec 611c 80c1 8105',
 '3Daap': '3713 3bda 3f91 3ff4 430b 509d 5312 5683 642d 7563 7629 7b54 8037 8345 8921 939d',
 '4Pau': '36bf 4a5d 6294 638a',
 '3Hiu': '40dd 493f 7a8d 7ac5 9f3c',
 '6Gwaang': '4fc7 901b',
 '6Faa': '64ed',
 '4Coeng': '364a 37c4 4108 4517 45c5 4806 4bf4 511f 573a 5834 5872 5899 58bb 5af1 5b19 5ea0 5ee7 6215 6a2f 6aa3 7246 7965 7fd4 80a0 8178 82cc 8407 8537 8594 8620 8a73 8be6 9577 957f',
 '3Wo': '6db4',
 '2Mik': '5e66',
 '6Fai': '5420',
 '6Fan': '3d00',
 '2Zim': '3b7a 410b 4320 50e3 98ad 98d0',
 '6Fau': '5cca 5fa9 86d7 8986 961c',
 '6Fat': '3579 363a 41c5 48f9 4e4f 4f10 5772 5caa 75ba 7782 7b4f 7f5a 7f70 8337 95a5 9600',
 '6Fu': '3914 3cc7 3f88 40a4 479c 47b8 47d4 4835 4b78 4b9b 4ec6 4ed8 5069 5085 7954 86b9 875c 8a03 8ba3 8ca0 8cfb 8d1f 8d59 8d74 8f14 8f85 9644 99d9 9a78 9b92 9c8b',
 '6Long': '435a 455e 8497',
 '1Gwang': '53b7 6e39 80b1 85a8 89e5 89f5 8a07 8f37 8f5f 8f70 9367 9835',
 '1Zoeng': '3495 3e67 4103 42a2 46eb 4b2c 50bd 5adc 5c06 5c07 5f20 5f35 5f70 615e 6a1f 6d46 6f33 6f3f 7350 748b 7ae0 7cbb 8780 87bf 87d1 9123 9c46 9e9e',
 '2Wo': '40ba 4c00 5a51',
 '6Jyu': '516a 55bb 5bd3 5fa1 6108 6eea 6fa6 7609 7652 79a6 84e3 8577 85c7 88d5 8a89 8aed 8b7d 8c15 8c6b 9047 9810 9884 99ad 9a6d 9fa5',
 '5Lyun': '5a08 5b4c 8114 81e0',
 '2Wu': '40d6 575e 5862 6342 6440 6d52 6ef8 795c 969d',
 '6Ling': '4b9a 53e6 6123 7756',
 '6Duk': '3aa9 3e3f 3f84 44ef 48b1 4a85 4af3 4bb7 5335 691f 6add 6bb0 6bd2 6e0e 7006 724d 7258 728a 72a2 72ec 7368 76be 78a1 8961 8b9f 8bfb 8d15 97c7 97e3 9ad1 9ee9 9ef7',
 '2Kei': '4f01',
 '2Ping': '504b',
 '2Biu': '37fd 3812 39fc 4348 4784 4ff5 5a4a 8868 88f1 893e',
 '2Bin': '399a 533e 60fc 6241 78a5 7a28 7a86 890a 8cb6 8d2c 9bff',
 '3Geng': '93e1 955c',
 '2King': '3a69 451b 5ece 9803 9877',
 '4Naang': '3d30 43fb 4605 4bee',
 '3Haak': '5580 5ba2 8ad5 8d69',
 '3Haam': '558a',
 '3Haap': '5477 55d1 6390',
 '5Jung': '52c7 6142 6c04',
 '2Pung': '42fd',
 '3Haau': '3e38 5b5d',
 '2Cang': '3c82 4191',
 '1Au': '362c 362d 4adc 52fe 533a 5340 5614 616a 6ad9 6b27 6b50 6ca4 6f1a 71b0 74ef 750c 8b33 8bb4 9264 93c2 9dd7 9e25',
 '1At': '6264',
 '1Ap': '564f 6d3d',
 '1Aa': '375e 3c33 40c1 4105 4e2b 5382 5416 5440 54d1 554a 555e 627c 690f 9312 9515 96c5 9d09 9e26',
 '1Am': '557d 5677 57b5 5a95 5eb5 76e6 83f4 8af3 8c19 97fd 9d6a 9e4c',
 '2Zaau': '355a 37ab 3c36 42f4 43bb 4c42 5e1a 627e 722a 722b 8098',
 '1Ak': '3b2c 5053 5594 5e44 6239 6424 6e25 815b 9628 9929',
 '6Paai': '4079',
 '5Saai': '445b 54b6 8213',
 '3Pit': '4e3f 5af3 5fb6 6486 6487 6c15 6f4e 77a5',
 '3Piu': '3d71 43c7 50c4 7bfb 9a43 9aa0',
 '2Gei': '3533 379b 37a6 3ead 434d 48b3 5c7a 5df1 5e8b 5eaa 638e 675e 7d00 7eaa 8291 866e 87e3 8e26 9094 9e82',
 '4Ning': '3cb0 45ff 4b62 511c 549b 5680 5bcd 5be7 62e7 67e0 72de 7370 752f 804d 8079 85b4 9e0b',
 '3Pin': '34f2 3d1c 43d2 4862 5fa7 7247 9a19 9a97',
 '1Geoi': '3752 3790 4155 4505 4ff1 5036 5579 5c45 5d0c 636e 6910 741a 8152 88fe 9d8b',
 '3Ging': '3e75 44f3 4fd3 52c1 5f33 5f84 5f91 656c 6871 734d 7ade 7af6 811b 8396 8ff3',
 '3Zat': '4d75',
 '3Zau': '353f 3fad 42d3 4472 50fd 5492 54ae 594f 63cd 663c 665d 7503 76b1 76ba 7e10 7ec9',
 '3Zam': '3d06 5bd6 63d5 6b41 6d78 8c2e',
 '3Zan': '38c0 4769 4b24 4cf2 4fb2 5733 632f 8704 88d6 8cd1 8d48 93ad 93ae 9547 9707',
 '3Zai': '43ba 4a16 4b41 5236 6cf2 6f08 72fe 7318 7660 796d 7a44 7a67 88fd 9645 969b 9701 973d',
 '3Zak': '38f1 4549 4da6 8d5c',
 '3Zaa': '3878 3d19 40ce 419b 47a2 4e4d 538f 643e 69a8 6ea0 70b8 7160 75c4 82f2 86b1 8a50 8bc8 91a1',
 '4Leon': '3502 35ae 3dcd 3de0 4201 42c4 4473 46ac 48af 4bbc 4ed1 4f26 4f96 502b 56f5 5707 5d18 5d19 5d99 62a1 6384 6a49 6ca6 6dea 6f7e 71d0 7498 7510 7584 77b5 78f7 7ca6 7cbc 81a6 83d5 8ad6 8f2a 8f54 8f6e 8f9a 9074 90bb 9130 93fb 96a3 9a4e 9c57 9cde 9e9f',
 '4Dung': '9ba6',
 '2Ngam': '63de',
 '2Ngan': '9280 94f6',
 '3Gong': '69d3 6d1a 7d73 7edb 964d',
 '2Fung': '3b34 4027 47ea 4ff8 552a',
 '5Bang': '3b4b',
 '1Jik': '4ebf 5104 5fc6 61b6 6291 6a8d 6fba 76ca 7e76 808a 8189 81b1 81c6 960b 9b29 fa17',
 '5Haai': '4701 5db0 6fa5 736c 87f9 880f 9082 99ed 99f4 9a87',
 '1Jin': '50bf 54bd 56a5 5ae3 70df 7109 7159 71d5 80ed 81d9 83f8 852b 9122 9a60 9d1b',
 '1Giu': '392d 4ba6 4fa5 50e5 510c 5a07 5b0c 618d 6d47 6f86 9a55 9a84 9dee',
 '1Jim': '393f 5266 53ad 565e 5944 5b2e 5d26 6079 61e8 6dca 6df9 6e30 814c 9183 95b9 9609',
 '1Gik': '3998 39b8 3ef7 4076 44e7 4e9f 51fb 58bc 621f 64a0 64bd 64ca 68d8 6b9b 6bc4 6fc0 894b 8f5a',
 '1Gin': '34fa 3b74 3e2b 3e47 4295 42cc 42d7 4311 4533 4b48 4dac 575a 5805 5e75 728d 80a9 83fa 8c5c 9203 976c 97ac 9c39 9ca3 9d73',
 '1Gim': '3438 517c 641b 7e11 7f23 84b9 9c1c 9cd2 9dbc 9e63',
 '1Jiu': '4e48 5406 5593 579a 592d 5996 5e7a 7946 8170 847d 8981 9080',
 '3Zeng': '6b63',
 '4Tim': '48f6 606c 6e49 751c 83fe',
 '1Paan': '3434 6273 6500 7705',
 '1Paak': '556a',
 '1Paai': '6d3e',
 '4Fu': '3ad9 4b4d 4d7e 51eb 6276 678e 6991 7b26 818d 8299 82fb 86a8 98ab 9ce7',
 '2Waai': '3787 38bb 3c6c 4227 4323 44d5 4b10 4be3 4c0e',
 '1Wu': '38aa 459a 4867 4a15 4e4c 545c 55da 572c 6747 6b4d 6c5a 6c61 6d3f 70cf 90ac 9114 91eb 93a2 94a8 9770',
 '1Paau': '4b0c 629b 62cb 6ce1 812c 98d1',
 '3Kwaan': '35a5 3afb 4403',
 '2Waan': '349a 7db0 7efe',
 '2Fu': '34e1 356e 35b9 41e2 42e8 4317 45c2 45c4 477e 4a49 4acd 4fdb 4fef 552c 5638 5e9c 5f23 629a 62ca 64ab 65a7 695b 6ecf 7236 7425 752b 76d9 7c20 812f 8150 8151 82e6 8386 864e 8705 8ef5 90d9 91dc 982b 9efc',
 '4Fang': '63c8',
 '5To': '37ce 4c81 4c8a 5af7 692d 9cda',
 '6Maai': '52a2 52f1 5356 8ce3 8fc8 9081',
 '2Gwaan': '3665 3a30 43be 465b',
 '2Fo': '368c 3a99 3da1 41a7 4f19 5419 5801 5925 706b 9225 94ac 9846 9897',
 '2Gwaai': '3564 39ea 3a33 4306 44d2 592c 62d0 67b4 67fa 7f6b 84af',
 '2Fui': '4d4b 6d27 75cf 79ac 8958 8cc4 8d3f 95e0 9613 9baa 9c94',
 '4Kwaai': '395f 480f 4bd3',
 '2Fun': '3bd8 4540 48b5 4957 4c8c 68a1 6b35 6b3e 76e5 7abd 96da',
 '1Kei': '3a7b 5d0e 6532 6864 8e11',
 '5Cau': '4817',
 '5Cai': '8360 9b86 9c6d 9c9a',
 '5Can': '7d16 7ebc',
 '5Cam': '8548',
 '5Caa': '3431',
 '1Bo': '475b 5575 5761 5d93 6ce2 73bb 7886 822c 83e0',
 '1Kau': '5f44 62a0 6473 7f11 82a4',
 '1Seng': '58f0 661f 8072 8165',
 '1Ko': '9233',
 '5Hou': '6d69',
 '6Cai': '54dc 568c',
 '6Cat': '4d7d',
 '1Kak': '63e2',
 '5Fu': '3dc6 5987 5a66',
 '6Go': '35fb',
 '1Caau': '361c 3a28 3a45 3ecc 4337 4c2b 4c78 4d78 527f 52e6 6284 7f7a 9214 949e',
 '4Haan': '3bd7 5a34 5afa 5afb 61aa 75eb 7647 9591 95f2 9df3 9df4 9e47',
 '6Wat': '4b87 4d33 6287 6430 71cf 735d 77de 9a48 9c4a 9d25 9e6c',
 '4Kaan': '3b97',
 '4Haai': '3476 370e 373e 3a19 4018 448a 4839 4ae3 4c7a 56a1 5b69 8ae7 8c10 978b 9ab8 9fa4',
 '4Kaam': '421c',
 '6Wan': '37e6 3dc9 3e18 46a0 46f0 4a75 4adf 4cb0 5031 5702 607d 60f2 6141 6df7 6eb7 78d2 8ae2 8be8 8fd0 904b 90d3 9106 97d7 97f5 9850',
 '4Haau': '4ce7 6d28',
 '6Wak': '41c8',
 '6Wai': '392c 3963 3a39 3a68 3ef0 439a 4619 487a 49e5 4a8b 536b 5aa6 5f59 605a 60e0 6193 6a5e 6e2d 715f 732c 80c3 8559 875f 87ea 885b 885e 8b02 8b53 8b86 8c13 8e97 93cf 93f8',
 '1Caak': '60fb 6e2c',
 '1Caai': '3bb2 3f2e 4797 4da9 640b 6413 731c',
 '1Caan': '3e97 403c 46b1 4b38 4c57 6b8b 6b98 98e1 9910',
 '1Caam': '3474 34ce 3558 37e5 3801 3c2b 4098 41ae 53c2 53c3 53c4 63ba 6400 647b 6519 68ab 7e3f 8fbf 9a42 9a96',
 '6Waa': '3566 399c 46e1 6466 6866 69ec',
 '1Gwaa': '39d3 3dbd 4ada 74dc 7dfa 9a27',
 '2Tin': '394f 440c 48ef 4a44 6ddf 8146 89a5',
 '2Gyun': '38a7 3df7 45ad 5377 6372 83e4',
 '1Gwan': '36ac 519b 541b 5747 5d11 6606 665c 711c 7428 76b2 76b8 7b60 7e1c 7eb6 8399 872b 8880 88c8 890c 8ecd 921e 9281 9315 94a7 951f 9bb6 9be4 9cb2 9da4 9e87',
 '1Gwai': '3c55 3ec1 4145 44d9 489c 5232 572d 59ab 5aaf 5ec6 5f52 6b78 6c3f 6ca9 6e88 73ea 7470 7688 7845 862c 90bd 95a8 95fa 9a29 9bad 9c91',
 '1Gwat': '34b4 3fb6 55d7 6132 61b0 6262 69be 6a58 6c69 6dc8 6ed1 7e0e 84c7 9aa8 9dbb 9e58',
 '1Coeng': '3867 39ff 415b 41ab 43ab 45c9 490c 494a 4983 4b96 4c7d 4d41 4f25 5021 545b 55c6 56ea 56f1 5a3c 6436 65a8 660c 67aa 69cd 7244 7255 7316 73b1 7472 7a93 7a97 7abb 7fa5 83d6 8b12 8dc4 8e4c 8e61 9306 9329 9397 93d8 9520 9535 95b6 960a 9be7 9cb3 9f1a',
 '1Kaa': '4f67 5361 5494 64d6',
 '2To': '59a5 6a62',
 '4Paang': '5017 5f6d 6189 68da 6ddc 787c 81a8 87da 87db 8f23 9d6c 9e4f',
 '6Gaak': '66f1',
 '6Gaat': '7534',
 '5Mei': '372b 3791 3827 414f 41fb 44fa 4794 4b3f 4b67 4eb9 5a13 5aba 5c3e 6e3c 6fd4 7f8e 7f99 8249 845e 9382 9541 9761',
 '4Seon': '3744 3747 3d6e 4407 44d0 451a 48e8 48e9 494e 49e0 4ac3 5507 6df3 6e7b 6f18 711e 7289 7d14 7eaf 8123 819e 83bc 84f4 9187 9d89 9e51',
 '5Noi': '9927 9981',
 '1Cuk': '3449 345b 35e4 3647 3705 3af9 3bc8 3f19 3fa5 40d7 40da 41cd 438c 450e 466f 46e4 4739 481e 4a73 4e8d 4fc3 4ff6 5a16 6149 61b1 6410 654a 65b6 675f 6a15 6b5c 6d91 6ec0 755c 762f 77d7 7c07 7c0c 84c4 850c 851f 89e6 89eb 89f8 8ad4 8e27 8e59 8e74 8e75 901f 906b 9110 9863 9917 9f6a 9f71 9f8a',
 '1Cun': '58ab 58ff',
 '4Tyun': '3798 3a54 3f0a 429c 5278 56e2 5718 5c6f 5e89 5ff3 6171 629f 6476 6f19 7bff 7cf0 81c0 8c58 8c5a 8ed8 9b68 9c80 9dd2 9dfb',
 '4Zeoi': '4151 4c91',
 '4Zeon': '3da6',
 '2Hang': '5543 80af',
 '3Dan': '627d 6489 64b4',
 '3Ping': '8058',
 '3Dai': '4c65 5886 5d7d 5e1d 63e5 6e27 78b2 7998 7de0 7f14 8482 8743 87ae 8ae6 8c1b',
 '4Ngo': '34df 4109 454f 45b8 4cd7 4cd8 4fc4 542a 54e6 5a25 5ce8 5ce9 6d90 774b 786a 83aa 8a1b 8a90 8b4c 8bb9 92e8 9507 9d5d 9d5e 9e45',
 '3Dat': '4a62',
 '3Dau': '4b26 7ac7 9b25 9b26 9b2a 9b2d',
 '3Dap': '37f7 441b 6498',
 '1Zyu': '36a5 377c 39b5 39e3 3a80 3b4c 3b70 3d86 3eae 40f4 41b0 4342 43ed 4417 4893 48f7 4a52 4a7d 4f8f 6731 682a 69e0 6a65 6ae7 6aeb 6f74 7026 732a 73e0 7843 7969 8331 85f8 86db 88be 8a85 8af8 8bdb 8bf8 8c6c 8de6 90be 94e2',
 '2Wong': '3cbf 3cf9 4435 6789 7007',
 '3Kwong': '4d43 5739 58d9 6269 65f7 66e0 7e8a 7ea9 909d 913a',
 '5Lam': '342d 34d3 36e6 3a06 4572 51db 51dc 58c8 5ee9 5eea 61cd 61d4 6a81 6aa9',
 '5Lai': '45cd 527a 5299 652d 6a0f 6fa7 793c 79ae 8821 8c4a 91b4 9c67 9ce2',
 '3Pong': '55d9 80a8 8255 8b17 8c24',
 '5Lau': '39d5 587f 5d5d 5d81 67f3 7bd3 7c0d 7db9 7e37 7efa 7f76 8938',
 '6Wok': '3bc9 425f 4a9d 6ab4 6fe9 74c1 74e0 7a6b 83b7 8816 944a 956c',
 '2Paai': '4c26',
 '1Gwaan': '46aa 474d 493d 5173 761d 7db8 95a2 95d7 95dc 9c25 9ccf',
 '1Gwaai': '3733 3fa9 40b7 40ec 4daf 4e56',
 '1Gwaat': '346e 3828 4ba9 4c2c 4c7b',
 '2Paau': '4283 8dd1',
 '5Maang': '4045 731b 824b 8722 9333 9530',
 '1Suk': '3450 34d8 3504 371a 375b 3a4b 3b18 3d0b 3d3c 3d96 40e4 4147 4398 43cb 447f 4954 4a39 4c59 500f 5010 50f3 5135 53d4 5919 6a5a 6dd1 7102 7c9f 7e2e 7f29 8083 8085 84ff 8b16 8c21 8e5c 9a4c 9a95 9b9b 9c50 9deb 9e54',
 '6Pet': '577a',
 '3Gau': '374c 3764 3a0c 4192 476d 4848 508b 5193 53a9 548e 57a2 591f 5920 59e4 5abe 5ec4 5f40 6344 6406 6551 6784 69cb 7078 759a 7a76 89af 89cf 8cfc 8d2d 9058 96ca',
 '3Gat': '3ab4 3b96',
 '1Pou': '92ea',
 '3Laap': '39dc 59a0 64f8',
 '3Gap': '3c44 3f2a 41f2 46df 49fb 5408 5cc6 84cb 90c3 95a4 981c 988c 9e3d',
 '3Gwo': '3cc0 8fc7',
 '3Kit': '34f6 3a6a 3f24 4165 52bc 5b51 6308 63ed 6705 696c 78a3 7df3 7e88 7f2c 7faf 8a10 8ba6 9365 9532',
 '3Gaa': '426f 50f9 5ac1 5e4f 7a3c 99d5 9a7e',
 '3Hung': '63a7 6f92 7074 8a0c 979a',
 '3Gam': '3bf2 3f49 4cba 50f8 51ce 5664 7068 7d3a 7ec0 8d1b 8d63 9c14',
 '3Gan': '3ad4 495c 4f12 5cce 826e 831b 9773',
 '3Gai': '386d 3c85 3fb5 4053 4987 4bfb 5f50 7031 7094 7b40 7d12 7e7c 7ee7 7f7d 84df 858a 87ff 8a08 8ba1 9afb',
 '3Laai': '765e 7669',
 '3Gak': '350f 37f3 3f97 40d2',
 '3Laak': '561e',
 '3Goeng': '3a40 4c5f',
 '3Dang': '482c 51f3 58b1 5d02 5d97 5d9d 6ac8 78f4 956b',
 '1Tung': '71a5 75cc 84ea 901a',
 '3Coeng': '4a68 5531 6005 60b5 6217 6227 66a2 709d 7197 7452 7545 97d4 9b2f',
 '5Ceoi': '3dc3',
 '1Nung': '71f6',
 '4Min': '3499 3b8c 3c03 40c7 43c3 4af5 4c13 5a94 5b80 68c9 6acb 7720 77ca 7dbf 7ddc 7ef5',
 '6Haang': '34d1 39bf 4044 4094',
 '1Tap': '4f6e',
 '4Miu': '3464 4067 45a2 63cf 7784 82d6 82d7',
 '6Luk': '34d0 35a8 36ec 3719 37e4 3987 3a90 3a96 3afd 3cb6 3f3e 40d9 4312 4361 4391 43bc 4402 4635 485c 4a6e 4c5a 4d2a 50c7 516d 5774 5f54 5f55 622e 6c2f 6de5 6e0c 6f09 719d 742d 752a 76dd 797f 7984 7a4b 7b93 7c59 7da0 7eff 83c9 89fb 8e1b 902f 9181 9304 9334 9646 9678 9a04 9be5 9e7f',
 '5Taai': '8226 825c',
 '5Zin': '3be0',
 '5Zim': '36a9 3882 3ff5 4383 44a3 4921 4c93 4dae',
 '4Sim': '4184 4a9c 5a75 5b0b 8749 87ec 87fe 88e7',
 '4Sin': '4a37 6fb6 7fa1',
 '1Tai': '68af 92bb 9511',
 '2Saam': '34c4 3524 3724 38e3 7cc1 7cdd',
 '2Saan': '4248 6563 7ca3 7ce4 994a',
 '3Kiu': '41bb 8e88',
 '2Saai': '4884 5c63 5f99 67b2 73ba 74bd 7e30 7e9a 8478 84f0 8af0 8e5d',
 '2Saau': '37ec 3aa2 4239 4264 447c 4b6d 7a0d',
 '5Daam': '8d31',
 '4Siu': '3c88 3e1b 8fe2 97f6',
 '3Taai': '3477 396d 43ae 46a3 48e0 592a 5fd5 6001 614b 6ba2 6c70 6cf0 8cb8 8d37 8eda 9226 949b',
 '2Jiu': '5b8e 6270 64fe 6773 6b80 7711 7a7e 7a85 7a88 7a94 7e5e',
 '2Keoi': '87dd',
 '1Geng': '9a5a 9ea0',
 '2Jin': '5043 533d 5830 6c87 6f14 8758 8ebd 90fe 9c0b 9da0 9f34 9f39',
 '2Jim': '53b4 5e7f 5f07 63dc 6abf 7f68 8181 88fa 9153 9692 9b47 9b58 9ee1 9ee4 9ef6 9f91',
 '1Maang': '63b9 9bed',
 '3Zeon': '349e 3559 3766 3b10 3b1c 3f31 3f59 45ef 472d 4fca 5bef 5cfb 6422 6649 664b 6d5a 6fec 74a1 756f 7ae3 7e09 7f19 81c7 8fdb 9032 92d1 9915 9982 99ff 9a8f 9d54',
 '4Niu': '3cee',
 '3Zeoi': '3761 3c0e 410c 5073 666c 6700 6a87 7db4 7f00 855e 8d05 8d58 8db3 9189 991f',
 '4Kin': '3a1c 3a6e 458d 63ae 63f5 8654 9c2c',
 '6Mung': '3734 3771 4822 4942 5922 68a6 973f',
 '2Kwan': '58f8 58fc 6083 6343 6346 6517 68b1 7a07 7a1b 7d91 83cc 95ab 9603 9e95',
 '4Nim': '9bf0 9cb6',
 '4Kiu': '3681 376f 4397 4c41 4e54 4fa8 50d1 55ac 6865 6a4b 7904 7fd8 7ff9 834d 835e 854e 8dab',
 '4Nin': '412d 4139 5e74',
 '3Gaang': '632d 6d6d 9015',
 '5Maai': '3765 395d 39b9 4e70 8cb7',
 '2Liu': '6599 6f66',
 '1Lau': '55bd 560d 6402 645f 6e9c 8356 8b31',
 '5Maan': '3770 3a09 3e2a 3ff8 4c84 5a29 665a 77d5 8115',
 '6Tim': '3dba',
 '3Doek': '39fb 4401 5544 5545 5663 65b2 6913 6dbf 7422 8ad1',
 '1Laa': '5566 5587 62c9 782c 97a1',
 '1Zyun': '383e 3d0a 3d9c 3f93 43dd 45a4 45d5 4779 4949 4b3c 4b84 4cc3 4ce3 4e13 587c 5ae5 5c08 750e 7816 78da 8ea6 911f 947d 94bb 954c 9853 989b 9c44',
 '1Lai': '378c 3a49 6b10',
 '5Seon': '696f 76fe',
 '5Maau': '42de 4352 536f 6634 6cd6 7261 8306 925a 94c6',
 '1Lam': '51a7 5549 83fb',
 '2Lin': '40db',
 '6Dei': '54cb 5730 58ac',
 '3Lat': '4033',
 '3Gu': '39fd 3f7d 435b 44e2 50f1 56fa 580c 5d2e 6545 75fc 7a12 932e 9522 96c7 980b 9867 987e',
 '3Lap': '3547',
 '6Deu': '6389 8abf',
 '3Go': '4e2a 500b 5404 7b87',
 '3Lai': '77cb',
 '6Mong': '671b 6722',
 '3Ge': '5605',
 '3Laa': '3664 4583 5611 56b9 7f45',
 '5Se': '4115 793e',
 '6Jit': '556e 5699 56d3 5b7c 5d72 6470 69f7 6af1 70ed 71b1 7cf1 81ec 81f2 8616 8617 882e 9480 9f67',
 '2Gu': '3689 4007 4026 4f30 53e4 679c 726f 72dc 76b7 76ec 77bd 7f5f 7f96 80a1 81cc 86ca 86cc 8831 8a41 8bc2 94b4 9f13 9f14',
 '6Jip': '4e1a 505e 5629 5b7d 5daa 6654 66c4 696d 70e8 71c1 8449 8825 90ba 9134 95d1 9765 9801 9875',
 '5Ngaan': '773c',
 '3Sim': '639e 7985 79aa',
 '5Ngaai': '487e 4ae5',
 '6Huk': '3570 3ab6 3d36 3de4 3e89 3fe5 4238 49fc 561d 659b 69f2 7e20 89f3 9177',
 '3Sin': '3a87 3fc5 493c 4947 4982 4985 4a18 5029 6247 6427 717d 785f 7daa 7dab 7dda 7ebf 817a 84a8 8b06 8f24 93fe 9730 9a38 9a9f',
 '5Ngaau': '3496 3f34 45e4 54ac 9f69',
 '3Siu': '3505 35db 42a5 5578 562f 7b11',
 '3Sit': '350e 38ef 3a1d 3c14 3cff 3e09 3ee1 429d 465d 478c 4eb5 4f33 5070 5a9f 5c51 5c67 5dad 6954 698d 6e2b 75b6 7a83 7aca 7d32 7d4f 7de4 7ec1 820c 859b 893b 8ea0 977e',
 '6Jin': '5501 5c98 5cf4 5f65 5f66 6d80 73b0 73fe 781a 786f 82cb 83a7 8afa 8b9e 8c1a 8c33 92e7 9f5e 9f74',
 '6Jim': '7069 7131 71c4 7213 8273 8276 8277 8c53 8c54 917d 91c5 9a10 9a57 9a8c',
 '6Jik': '4ea6 571b 57bc 57f8 5955 5cc4 5da7 5e1f 5f08 5f0b 5f79 603f 61cc 6759 7037 71da 71e1 75ab 776a 7e79 7ece 7fca 7fcc 7ffc 8285 8734 876a 8b6f 8b7a 8bd1 9006 91b3 9a5b 9a7f 9d82 9d83 9dc1 9dca 9e5d 9e62 9ed3',
 '2Go': '54ff 55f0 7b34 8238',
 '3Saap': '36ab 3d64 4209 4b0a 55a2 6b43 6b70 6fc7 7b91 7fdc 7fe3 8410 8dbf 9254 970b 970e 9778 98af 98d2 99ba',
 '3Saau': '3d7b 54e8 6f72',
 '6E': '35a1',
 '3Saai': '353a 49d8 55ee 6652 66ec 6bba',
 '3Saak': '6880 7d22',
 '3Saam': '38cc 3908 6472',
 '3Saan': '3a94 46b2 4d2e 4f1e 5098 6c55 6dae 759d 7be1 7e56 8a15 8baa',
 '3Lyut': '3b5e 3b69 3c95 3e39 3f5f 4923 52a3 54f7 57d2 6318 634b 811f 92dd 950a',
 '5Soeng': '4e0a 979d',
 '4Kim': '4224 5c92 62d1 7b9d 9210 9246 9257 94a4 94b3 9ed4 9eda',
 '4Gong': '80ae',
 '3Hot': '34ed 3aa1 3b1e 3d63 3dce 3fe3 4ad8 4d8e 559d 5d51 668d 66f7 6bfc 6e34 8910 97a8 9da1 9e56',
 '3Hou': '362a 36aa 3740 4427 44b5 46bd 7292 79cf 8017',
 '3Ngaap': '64ea',
 '3Ngaak': '922a',
 '3Hoi': '5117',
 '3Hok': '55c0 58f3 6bbc',
 '3Hon': '3d44 66b5 6c49 6f22 71af',
 '5Pei': '5a62',
 '6Lim': '3d91 655b 6582 6b93 6bae 6f4b 6fb0 7032 88e3 895d',
 '3Gung': '3bef 41e8 4508 551d 55ca 8ca2 8d21',
 '6Lik': '3823 39f0 3b0f 3be4 3c79 3df4 3ea1 3efa 3f41 3fe8 4025 4365 437d 4843 4976 4b05 4ba5 4c1b 4c1c 529b 5386 53a4 5456 56a6 575c 66c6 6b74 6b77 6ca5 76aa 78ff 8280 82c8 85f6 8f63 90e6 9148',
 '6Liu': '387b 427c 48a7 5c25 5ed6',
 '6Lit': '3b2f 3ed3 3f32 4140 4b8b 4d15 51bd 6d0c 70c8 7d9f 8322 86da 88c2 8d94 98b2 9ba4 9d37',
 '6Lip': '3bff 3cb1 407d 4753 4a89 5120 5de4 72a3 730e 7375 8e90 9b23',
 '6Do': '3590 3c10 3ee7 415c 493b 5815 58ae 58af 5d9e 60f0 964a',
 '3Lok': '34e2 39a1 3abe 4237 6d1b 70d9 7296 73de 7d61 7edc 8366 916a 96d2 99f1 9a86',
 '2Kaai': '9347 9534',
 '5Nyun': '43a1 6696 6e1c 992a',
 '4Pung': '7bf7 8283 9afc',
 '3Kut': '3ab1 3ea2 44b7 47ef 499a 4bfa 4f78 62ec 681d 6fca 7b48 8052 8c41 9002 9afa 9b20 9d30 9e39',
 '1Lap': '7b20 7c92 82d9 9dd1',
 '6Nong': '9f49',
 '1Lat': '7529',
 '1Lyun': '7675',
 '1Tong': '34e5 528f 6c64 7fb0 93dc 9557 95db',
 '1Jyu': '4e8e 625c 6275 65bc 6745 6de4 7397 7600 7afd 7d06 7ea1 8867 9098 9653',
 '1Zyut': '525f',
 '6Lou': '5aea 6d9d 6f5e 6f87 7490 7c2c 8557 8cc2 8d42 8def 8f05 8f82 9dfa 9e6d',
 '1Lyut': '6d56',
 '2Teoi': '3782 3bd0 3c63 817f',
 '1Zaang': '3b39 4e89 5d22 6399 68e6 72f0 7319 7741 775c 7b5d 7b8f 8acd 8e2d 931a 94ee',
 '4Ngau': '3b81 401a 409a 4270 42c2 725b',
 '1Gaan': '39c6 4535 4c19 5978 59e6 8271 83c5 844c 8551 9592 9593 95f4',
 '1Gu': '3f0b 4249 44da 49f8 5471 5495 59d1 5b64 67e7 6cbd 6cd2 7b9b 7f5b 80cd 82fd 83c7 83f0 86c4 89da 8ef1 8f9c 9164 9237 9d23 9e2a',
 '1Gaai': '3fac 3ffd 40c8 4578 4cf6 4f73 5055 5588 5826 6977 6e5d 75ce 7686 79f8 85a2 8754 8857 9636 968e',
 '1Go': '3439 38a6 390e 54e5 65ee 6b4c 6e2e 7241 72b5 83cf 8b0c',
 '1Gaau': '3b35 3cf0 3e28 434a 4430 4628 4892 49f1 4cab 4d14 4ea4 6559 80f6 81a0 832d 86df 8de4 8f47 90ca 9bab 9c9b 9d41',
 '1Gaat': '3515 4497 4880',
 '4Ngai': '502a 5371 5d6c 5dcd 6845 72a9 730a 873a 8f17 90f3 9713 9ba0 9be2 9cb5 9e91 9f6f',
 '4Ngan': '3616 569a 5701 57a0 72fa 8abe 911e 9f57 9f82',
 '4Ngam': '3484 36a7 6272',
 '3Gyut': '5282 6a5c 6f4f 761a',
 '2Gaang': '4c74',
 '3Pou': '8216 94fa',
 '3Pok': '4138 41da 4a99 5657 58a3 6251 64b2 749e 7c95 84aa 91d9 948b',
 '2Fei': '3971 4a3e 532a 5c5d 60b1 6590 670f 68d0 69a7 7bda 7fe1 80d0 8ab9 8bfd 966b',
 '3Hoeng': '5411 66cf',
 '1Kuk': '3586 3ec3 4897 66f2 86d0',
 '1Kui': '7e6a 7ed8',
 '2San': '3598 4ae9',
 '2Sam': '39f2 3c02 4289 428f 4f08 5a76 5b38 5ba1 5be9 6e16 700b 77ab 8b85 8c02 90a5',
 '2Sai': '4222 4f7f 6d17 99db 9a76',
 '2Jing': '6620 763f 766d',
 '4Ngaai': '3a92 3e78 4416 463d 48c0 5393 5540 5810 5d16 6371 6daf 775a',
 '4Ngaan': '366c 3e5c 3e5e 41f5 4d26 8664 984f 989c',
 '4Ngaam': '55a6 5ca9 5d52 5dcc 5dd6 764c 789e 9eec',
 '2Saa': '6d12 7051 800d',
 '2Bun': '3877 3a27 3bba 40f2 4a30 672c 755a 82ef',
 '4Ngaau': '449c 5d24 6bbd 6dc6 723b 80b4 991a',
 '2Sau': '374a 4230 43c2 43f4 5081 53df 55fd 55fe 5b88 624b 64de 64fb 68f7 72e9 778d 7c54 824f 85ae 85ea 8b0f 9996',
 '3Fong': '4084 51b5 653e 6cc1 7d56 8cba 8d36',
 '1Dou': '4462 4a72 5200 53e8 561f 5fc9 6c18 8220 90fd 95cd 9b5b 9c7d 9ced',
 '6Wong': '65fa 8fcb',
 '6Git': '3636 5091 6770 6840 69a4 6ed0',
 '6Giu': '5ce4 5da0 64ac 8f7f',
 '6Gip': '72f9 965c',
 '1Baai': '49ec 62dc 63b0',
 '6Gin': '4ef6 5065 6957 9375 952e',
 '6Gim': '4fed 5109',
 '6Dek': '38d9 3c05 42ae 4a00 7b1b 7bf4 7c74 7cf4',
 '6Beng': '75c5',
 '2Hung': '3491 54c4 5b54 5ffc 6050 6177',
 '3Cek': '357d 4735 4a0f 544e 8d64',
 '2Do': '36c6 36ca 3969 3ed4 3f87 492a 4bec 4d31 4eb8 5234 5241 54da 56b2 579b 579c 57f5 6735 8dfa 8eb1 8eb2 9b0c',
 '4Tung': '383d 390f 3e17 3f27 3f3f 4088 4bb5 4ccb 4d00 4db1 4edd 4f5f 50ee 540c 5f64 66c8 6723 6850 6c03 6f7c 70d4 729d 735e 77b3 7a5c 7ae5 81a7 825f 833c 8d68 916e 9285 94dc',
 '2Pau': '74ff',
 '3Kwok': '35e5 5f49 979f 97b9',
 '3Tyut': '4fbb 812b 8131 838c',
 '5Kam': '5997',
 '2Sou': '36d0 36ee 4545 5ac2 626b 6383 6570',
 '5Fan': '507e 50e8 5674 594b 596e 5fff 6124 61a4 6b55',
 '2Jeng': '5f71',
 '3Fung': '8af7 8bbd 8cf5 8d57',
 '4Lyun': '3748 3869 3c4d 3d8c 3ea6 43c8 4582 470c 571e 5b6a 5b7f 5ce6 5dd2 631b 6523 683e 6b12 6ee6 7064 7f89 8054 806f 92ae 947e 9e1e 9e3e',
 '3Ceoi': '3bd4 3cc3 3d76 4066 40c0 411f 43e3 4580 4639 477b 499f 49b7 5005 5550 6a47 6bf3 6dec 7120 7ac1 7db7 7fe0 8103 8106 813a 81ac 89b0 89b7 89d1 8da3',
 '5Zaam': '4b43',
 '5Zaan': '39c8 3aa6 41d9 42b6 43d6 44a1',
 '6Bong': '39f5 49db 508d 57f2',
 '1Gun': '45c6 500c 51a0 5b98 68fa 839e 898c 89c0 89c2',
 '6Zeon': '3808 3db3 4006 499e 5c3d 6fdc 70ec 76e1',
 '6Zeoi': '3428 347a 34f9 3627 373f 3811 3818 3aab 3c8a 3d70 3fbd 40b1 40cd 41e4 433c 44f2 49bd 53d9 589c 5e8f 601a 654d 6558 6e86 6f35 7500 787e 7e0b 7f12 7f6a 805a 8fa0 9c6e',
 '3But': '7835 7f3d 8e73 9262 94b5',
 '6Jiu': '6130 66dc 71ff 7b44 8000 9dc2 9e5e',
 '4Jau': '56ee 5c24 5eae 601e 60a0 63c9 6538 65bf 67d4 6962 697a 6c8b 6cb9 6e38 72b9 7336 7337 7531 75a3 7e47 83b8 8555 8628 86b0 8763 8a27 8e42 8f2e 8f36 900c 904a 90ae 90f5 94c0 97a3 9b77 9bc8 9c7f 9f2c',
 '1Bik': '3c78 3df6 3ff0 4a48 4afe 4ba0 506a 610a 610e 61b5 63ca 6945 6e62 74a7 7695 78a7 7dbc 8177 895e 8e84 8f9f 903c',
 '2Zai': '4ed4 6d4e 6fdf',
 '1Bin': '3ca2 47cd 7178 7335 7502 782d 7b3e 7baf 7c69 80fc 8439 8759 8fab 8fae 8fb9 8fba 908a 97ad',
 '2Zan': '37db 37de 42ce 45c3 62ae 755b 7a39 7e1d 7f1c 8eeb 8f78 9b12 9ef0',
 '2Zam': '4aec 5e0e 600e 6795 7d7c',
 '2Zaa': '3de2 9b93 9c8a',
 '2Saang': '35c2 3a18 3faa 405e 40cf 6e3b 7701 771a',
 '3Zong': '3638 585f 58ee 58ef 846c',
 '4Hung': '3593 40d4 4791 49c6 49fa 4aa6 4af9 4ca8 6d2a 718a 7bca 7ea2 836d 8452 857b 8679 8ba7 8c3c 96c4 9d3b 9e3f 9ec9 9ecc',
 '2Zau': '3af6 3e90 459e 45a3 546a 776d 7b92 8d70 9152 9bde',
 '1Biu': '3bf9 3dbe 4043 406d 417a 4538 4673 4bbd 5126 5f6a 6753 6807 6a19 6eee 700c 719b 730b 762d 7968 7a6e 8198 81d5 8508 85e8 9336 93e2 9463 9556 9573 98a9 98ae 98c6 98c7 98d9 9a6b 9a89 9adf 9e83',
 '1Wun': '525c 8c4c',
 '1Wui': '42ff 504e 6933 6e28 7168 7859 9688',
 '2Zyun': '3b47 3dea 3f5c 408e 4148 46f7 4871 4a4a 4aad 50d4 556d 5642 56c0 6499 7e82 7e98 7f35 8e9c 8f49 8f6c 9142',
 '5Si': '5e02 9230 94c8',
 '1Taai': '4d18 4d8f 5454',
 '1Taan': '3594 3631 3a0f 5574 574d 644a 6524 6ee9 7058 762b 7671',
 '1Taam': '43e5 63a2 7dc2 8caa 8d2a 9193',
 '6Bang': '6191',
 '4Hong': '402a 44b0 45b1 4615 4736 4bd1 4cb3 676d 6c86 7b10 822a 86a2 8fd2 980f 9883 9aaf',
 '6Wu': '3546 3a2d 3e26 41d8 42ba 4576 4a12 4a3c 4e92 51b1 5a5f 5aed 5aee 5cb5 6019 6236 6237 6248 62a4 6791 6c8d 6caa 6eec 81d2 828b 8b77 9120 9800',
 '3Fu': '3838 3931 3f6c 4484 526f 5490 5bcc 5e93 5eab 623d 7d5d 88b4 88e4 8914 8932 8ce6 8d4b',
 '3Fo': '3fa7 8ab2 8bfe 8ca8 8d27 9a0d 9a92',
 '6Wo': '4100 555d 7978',
 '3Fe': '3e43',
 '1Fu': '343b 34ec 37ca 3986 3a64 3b6a 3c60 3c97 40ff 412e 4250 42bf 438d 4394 4467 4530 4a17 4bfd 4c50 4d23 4d38 4e4e 4f15 4fd8 5233 547c 57ba 592b 5b5a 5b75 5c03 5e60 6232 630e 6577 67af 67b9 67ce 6874 6b51 6ced 6ef9 75e1 7806 7b8d 80a4 80d5 819a 81b4 8374 83a9 864d 8656 886d 8b3c 8dba 8dd7 90db 911c 9207 9ab7 9cfa 9ea9 9eb1 9eb8',
 '1Siu': '3c96 3cb5 3d05 3dae 4303 431a 4525 4a2d 4af8 4c11 4d1b 5bb5 63f1 6af9 6d88 6f47 6f5a 701f 70e7 71d2 75df 785d 7bab 7c2b 7d83 7ee1 7fdb 8427 8437 856d 87cf 8828 900d 92b7 9500 9704 9b48',
 '3Bou': '3635 36b4 3ccd 3ffe 4f48 5124 57d4 5831 5e03 6016 62a5 62aa 923d 94b8',
 '1Sin': '34a8 413d 4273 4674 4aea 4ed9 50ca 5148 79c8 7c7c 893c 8df9 8e6e 8e9a 9bae 9c9c',
 '1Sim': '36b2 7b18 82eb',
 '1Fo': '3e2f 41bc 4216 4300 436b 68f5 79d1 7a1e 7aa0 7abe 874c 9ac1',
 '1Sik': '36ed 375c 38f0 3afa 3b4a 3ba9 3f4a 407a 454a 492d 4a1b 4b52 4d71 50c1 556c 55c7 596d 5ab3 5f0f 606f 6089 60dc 62ed 6614 6662 6670 6dc5 7184 761c 7699 7a51 7a61 7ab8 7dc6 8204 8272 83e5 84a0 856e 8725 8785 87cb 884b 896b 8b58 8bc6 8efe 8f56 8f7c 910e 91ca 91cb 92ab 93b4 94ef 98fe 9919 9970 9ab0',
 '3Bok': '35d8 3ae7 3b0d 3b27 4378 45da 4b2a 4b66 4c0a 4d88 4eb3 535a 5697 58c6 5da8 61ea 640f 6b02 733c 74dd 7921 7934 7c19 7e1b 7f1a 80c9 818a 896e 939b 946e 9548 993a 998e 99c1 99ee 9a73 9ac6 9b44',
 '3Dou': '34c3 414a 5230 5992 59ac 636f 6581 8839 9053',
 '6Lung': '345d 36de 37d6 3885 3ce5 54e2 5f04',
 '4Cyu': '354f 3861 47de 6a71 8e70 9e86',
 '6Ding': '374e 5576',
 '3Zi': '3544 366a 386b 39a4 39d8 3a16 3c23 3c37 3c45 3c74 3d1b 4146 42f5 43ef 4409 44cc 44f9 4581 45f7 4720 4777 4783 4787 494d 4961 49af 4b25 4c80 4d5d 5bd8 5fd7 61eb 631a 646f 667a 6e0d 6f2c 7278 7590 75e3 7dfb 7f6e 80fe 81f3 81f4 89ef 89f6 8a8c 8d04 8d3d 8e2c 8e93 8f0a 8f7e 92d5 99e4 9a3a 9dd9 9e37',
 '3Zo': '4f50 8a5b 8b2f 8bc5',
 '3Ze': '409e 43d1 45ea 4a7e 501f 556b 67d8 8517 85c9 87c5 9dd3 9e67',
 '1Deng': '7594 91d8 9489',
 '3Bong': '750f',
 '3Fok': '3e0c 3e15 4870 4986 5f4f 6204 6509 652b 7383 77cd 81db 85ff 8837 883c 8c9c 8ea9 9481 9562 970d',
 '2Ting': '4fb9 633a 753a',
 '3Saat': '429b 644b 6492 6740 6a27 715e 8428 8531 85a9 93a9 94e9 95b7',
 '1Kaak': '7dd9 7f02',
 '6Gwaai': '367a',
 '6Gwaan': '3d2b',
 '1Ziu': '3658 3cac 3e08 410f 432d 4645 4ce9 4db0 50ec 62db 6912 7126 7901 81b2 8549 87ed 8dad 940e 9de6 9e6a',
 '4Fei': '4208 6ddd 80a5 8153 8730',
 '1Laang': '5525',
 '1Zik': '360d 3938 3ba8 3c72 3cba 3e44 3e93 422a 441a 45fb 49a7 4ced 52e3 5373 537d 5832 5e3b 5e58 6956 6a34 766a 79ef 7a37 7a4d 7e54 7ec7 7ee9 804c 8077 8635 874d 87d9 8e16 8e5f 8ec4 8ff9 965f 9bfd 9c3f 9cab',
 '4Coi': '34b2 624d 6750 7e94 88c1 8ca1 8d22',
 '1Zin': '35b6 3850 38f6 3acb 3b8d 3bb5 3c04 3dd9 3e36 4085 41f3 4379 444d 45e1 4c73 5e34 620b 6214 63c3 65c3 6834 6be1 6c08 6c0a 6e54 6e85 6ffa 714e 724b 7b3a 7b8b 7c5b 7fb6 9085 97af 97c9 9958 9b0b 9c63 9ce3 9e07 9e6f',
 '1Zim': '384b 3994 3c58 4021 41dd 42bb 42ec 46f3 4828 48cd 4993 5360 5c16 60c9 6cbe 6e10 6f38 71b8 77bb 79e5 81dc 81e2 859d 8998 89c7 8a79 8b6b 8c35 9711',
 '5Paang': '68d2',
 '1Pat': '5339 9d04',
 '1Pau': '3b13 4afd',
 '4Syu': '3e21 3f21 6778 6b8a 6bb3 6d19 85af 85f7 8d8e 9296',
 '1Pai': '6279 78c7 930d',
 '1Pak': '3577',
 '5Hing': '80eb',
 '1Paa': '6252 8db4 8dc1',
 '2Ang': '54fd',
 '3Kang': '63af 88c9',
 '5Pui': '500d 789a',
 '2Geng': '9838 9888',
 '6Dang': '4bb4 9093 9127',
 '1Fung': '343d 34a5 36d4 3702 3826 3ded 3e3c 4031 44a0 45e6 45ec 4d44 4e30 5906 59a6 5c01 5cef 5cf0 5d36 67ab 6953 6ca3 6e22 7043 70fd 728e 75af 760b 8451 8634 8702 882d 8c50 9146 92d2 950b 98a8 98cc 98ce 9eb7',
 '3Baa': '3d9a 40fb 4a57 4a7b 4f2f 575d 57bb 57e7 58e9 5f1d 705e 8987 9738',
 '1To': '62d5 62d6 63d3',
 '3Ban': '50a7 574b 574c 5b2a 6379 6448 64ef 6ba1 6baf 8191 81cf 9acc 9ad5 9b13 9b22',
 '3Bai': '3814 4411 4a06 7b85 853d 8d14 8d51 9589 95ed',
 '3Bak': '34e6',
 '2Ging': '3b0c 4718 5106 522d 5244 5883 5887 5e5c 61ac 61bc 666f 749f 7adf 87fc 8b66',
 '3Sing': '5723 6027 714b 8056',
 '1Gwaang': '39e6 4354 4395',
 '3Kong': '3c20 4f09 531f 593c 61ed 6297 720c 72ba 77ff 909f 9227 94aa 958c 95f6',
 '4Ngong': '3b7f 4a55 4b79 4b7a 6602',
 '5Naa': '42fe 438e 4b01',
 '1Dang': '3caa 3f45 4532 465e 4cfe 5654 706f 71c8 767b 7c26 8c4b 8e6c 9419',
 '5Nai': '36b7 82e8',
 '5Nan': '370a 3af1',
 '1Hong': '344c 3cdd 3f39 3fa4 41b2 45e7 52bb 5321 5eb7 6047 6846 690c 6d2d 6f2e 7736 7a45 7b50 7ce0 8154 8a86 8bd3 93ee',
 '5Nam': '4212 7a14',
 '5Jyu': '4e73 4fc1 4fe3 5673 5704 5709 5973 5b87 5ebe 6554 659e 6c5d 6d33 7440 7610 791c 79b9 7ab3 7c79 7fbd 860c 8a9e 8bed 8c90 9105 92d9 94fb 96e8 9e8c 9f6c 9f89',
 '4Mou': '3dfb 4251 4362 4b77 4c6f 5aeb 5af5 5deb 5e91 5ee1 6479 65c4 65e0 6a21 6a46 6bcb 6bdb 7121 7ce2 829c 82bc 856a 8aa3 8b28 8bec 8c1f 9155 9ae6',
 '1Joek': '836f 846f',
 '1Fong': '3829 3835 3843 3846 3903 3c42 3d41 402e 4131 531a 574a 5ddf 614c 65b9 678b 6c78 8093 80aa 82b3 8352 8841 8b0a 8c0e 90a1 9201 94ab',
 '4Moi': '7266',
 '4Kei': '37bf 37d3 37da 37e2 3ac5 3be6 4172 427b 4474 47da 4953 4a53 4b76 4b7c 4c07 4c48 4d6d 4d9e 4e93 4fdf 573b 57fc 5c90 65c2 65d7 679d 68ca 68cb 6b67 6dc7 7426 742a 75a7 7881 7895 7941 7947 7948 797a 7ad2 7da6 8006 80b5 82aa 8401 8572 8604 8691 871d 871e 8dc2 8edd 980e 9880 9a0f 9a39 9a90 9b10 9b3f 9ba8 9bd5 9c2d 9ccd 9e92',
 '4Nei': '37be 37dc 38c6 4272 4318 4358 4573 46cf 475a 4978 59ae 5a57 5c3c 6029 7962 79b0 922e 94cc',
 '3Cuk': '8c56',
 '3Gui': '528a',
 '5Hang': '60bb',
 '5Ngau': '37f1 4da7 5076 8026 8545 85d5',
 '5Ce': '4d66',
 '5Ngaa': '39da 3f18 3fff 4ab5 4f64 5e8c 74e6 758b',
 '5Co': '5750',
 '6Zeng': '51c0 51c8 6de8 90d1 912d',
 '5Ngai': '639c 6aa5 7912 8223 8264 8681 86fe 8798 87fb 9697 9820 9857',
 '5Ci': '36b6 3a01 41c3 43a3 4f3c 59d2 5cd9 6043 6c5c 6ecd',
 '4So': '50bb 510d',
 '1Gwing': '405d 4339 4b90 5182 5770 6243 99c9',
 '4Si': '3ad1 4215 57d8 5852 65f6 65f9 6642 69af 83b3 8494 9c23 9ca5',
 '1Daak': '3775 392b 3940 3bd6 4677 5491',
 '1Daai': '38e2 440e 4a58 4a5a 4a98 5446 7343',
 '1Daan': '3806 38cb 4437 449f 4e39 50e4 52ef 5330 5355 55ae 6b9a 6bab 7649 7baa 7c1e 894c 90f8 9132',
 '1Daam': '36ae 4121 510b 62c5 64d4 6e5b 7514 7708 7d1e 803d 8043 8ead 9156 9815',
 '5Zyu': '3468 349c 3531 3565 3651 3762 382f 3870 399b 3e84 3f0c 48c1 494f',
 '6Mei': '424b 428a 467f 4c28 4c2a 5473 5bd0 672a 715d 83cb 9b45 9b87',
 '1Daau': '3c8e',
 '1Hang': '3561 39f6 3c22 40ab 40d8 4ea8 54fc 5548 647c 727c 7841 785c 811d 8e01 93d7 94ff',
 '3Ngaa': '9f7e',
 '4Dam': '6983',
 '3Sung': '3bb8 48a0 5b8b 9001 9938',
 '1Heng': '8f15 8f7b',
 '6Jung': '4f63 7528',
 '6Tok': '9a5d',
 '6Toi': '3e00',
 '4Noeng': '5a18 5b43',
 '4Tiu': '37d8 4083 4a66 4bfe 4c54 5ca7 6761 689d 7b24 82d5 839c 8729 9797 9aeb 9c37 9ca6 9f60 9f86',
 '6Zou': '4483 505a 7681 7682 795a 7c09 80d9 8443 963c',
 '6Zing': '345e 362b 38cf 3c4f 3f40 4014 51ca 5a67 7a7d 7aeb 9631 9756 9759 975c',
 '1Ne': '5462',
 '3Nim': '8e68',
 '6Zoi': '5728',
 '4Tin': '39c2 47e7 4852 4858 4956 49c3 5861 586b 6437 6cba 7530 754b 78cc 7ab4 95d0 9617',
 '6Zok': '42cf 4963 51ff 600d 6d5e 6fef 7b70 838b 9369 947f',
 '4Naam': '5357 5583 56dd 56e1 678f 67df 6960 7537 8169 877b 8af5',
 '3Cong': '416e 521b 6006 6134',
 '4Naan': '5da9 96be 96e3',
 '4Caang': '404e 4f27 5096 6381 649c 673e',
 '3Kei': '5180 61fb 66a8 81ee 9a65 9aa5',
 '4Naau': '5476 5912 5cf1 6013 6320 6493 6861 7331 7376 737f 8b4a 9403 94d9',
 '4Lin': '3715 37c0 3955 3981 50c6 55f9 601c 6190 68bf 69e4 6d9f 6f23 7489 83b2 84ee 88e2 8933 8b30 8fde 9023 93c8 9c31 9ca2',
 '4Lim': '377a 3858 3f13 4060 467a 4965 4a2c 4b51 5941 5969 5e18 5ec9 6e93 6fc2 78cf 7c3e 81c1 8539 861e 880a 938c 942e 9570 9b11',
 '1Soeng': '342e 3560 4276 432e 4744 4a07 4a25 4d6e 4d70 4d7c 4f24 50b7 53a2 53cc 5546 5891 5b40 5ec2 5fc0 6b00 6b87 6ba4 6e58 6e6f 71b5 74d6 76f8 7935 7bb1 7dd7 7e95 7f03 826d 8459 850f 8944 89de 89f4 8b2a 9472 9576 96d9 971c 9a64 9a66 9aa6 9aa7 9b3a 9dde',
 '5Tiu': '3e20 5b25 7ab1',
 '5Tim': '3d98 7c1f 9902',
 '4Loeng': '3e41 48fc 4b6a 4fcd 51c9 6881 690b 6a11 6dbc 7cae 7cb1 7ce7 826f 83a8 870b 8e09 8f2c 8f8c 91cf',
 '4Liu': '37a0 37f9 3d73 3dab 41ed 470d 4a05 4a4d 50da 5639 5afd 5be5 5bee 5c6a 5d7a 5d9a 6180 6579 6a51 71ce 7360 7597 7642 77ad 7c1d 7e5a 7f2d 804a 818b 81ab 87df 8c42 8fbd 907c 9410 9563 98c9 9def 9e69',
 '4Sing': '3f29 49d5 4baa 4c46 4e1e 4e58 5818 584d 5bac 627f 665f 76db 7e69 7ef3 8aa0 8bda 90d5 92ee 94d6 9a2c',
 '1Naa': '54ea',
 '2Nin': '378b 8dc8',
 '2Nim': '361d 3a59 637b',
 '1Nai': '3f8d',
 '1Nat': '5ad0',
 '1Nau': '5b32 7373',
 '1Nap': '57dd',
 '3Jam': '5ed5 836b 852d',
 '3Jan': '5370 764a 820b 8845 91c1',
 '3Gaai': '3458 374f 3839 3fcd 41d2 46fa 4bf0 4c3a 4c44 4cb8 4ecb 4ef7 598e 5c2c 5c46 5c4a 5ee8 6088 61c8 6212 7297 73a0 754c 75a5 82a5 86a7 8aa1 8beb 9385 9ab1',
 '5Neoi': '8147 91f9 9495 9912',
 '6Noi': '3a05 3b88 3b8f 3ca1 415e 4c5e 5037 5185 5948 67f0 6c1d 8010 933c',
 '6Nok': '8afe 8bfa',
 '6Nou': '42d2 6012',
 '1Jung': '55c8 55e1 5670 5855 58c5 5ef1 6fad 7049 75c8 7670 7fc1 8579 8789 9095 96cd 96dd 9954 9db2 9e5f',
 '3Jau': '5e7c',
 '6Lok': '3d16 4029 40d5 4118 4334 4e50 6a02',
 '6Loi': '3fa2 46f6 48c2 7750 775e 8bd4 8cda 8d49',
 '2Kaa': '9272',
 '4Hang': '3530 4bd2 59ee 6046 6052 63ef 73e9 80fb 830e 8605 8861',
 '2Kai': '3485 3841 43ff 4b6b 4b6c 542f 5553 555f 68e8',
 '6Ji': '4e49 4e8c 4f74 5293 52da 54a1 5ed9 5f02 5f0d 65bd 6613 6a32 6f69 73e5 7570 7fa9 8084 8848 8cb3 8d30 98df',
 '2Kau': '53b9',
 '2M': '5463',
 '2O': '5684',
 '6Je': '504c 591c 5c04 6e03',
 '2Gwing': '34cf 38a0 392f 3dd7 3de1 489b 518b 518f 56e7 6abe 6cc2 70af 70f1 715a 7d45 8927 8a57 8bc7 8fe5 9848',
 '2So': '3abd 4217 441d 4692 4786 4d40 5522 55e9 60e2 6240 7410 7463 7485 9396 93c1 9501',
 '2Si': '3e6c 53f2',
 '2Se': '3550 374d 4971 497e 4b37 5199 51a9 5beb 6368 63f3 7009 820d',
 '3Zung': '343a 3cb4 3d29 435f 4f17 75ad 7632 773e 7eb5 7efc 8846',
 '1Syun': '3a4a 4556 475c 5b59 5b6b 5ba3 609b 63ce 640e 6718 69c2 72f2 72fb 733b 7444 75e0 836a 84c0 9178 98e7',
 '2Gwong': '392e 3adb 5ee3 72b7 7377 7a6c 8a91 8bf3',
 '3Gyun': '38be 3abb 3bde 4338 5238 5a58 5e23 7367 7737 7760 7d6d 7d79 7ee2 7f65 9104 97cf 9b33',
 '1Mau': '54de 8e0e',
 '1Mat': '4e5c',
 '2Kwaai': '36fb',
 '2Kwaan': '3ed2 41f9',
 '6Lin': '3a9d 3f11 695d 6e45 70bc 7149 7df4 7ec3 934a 94fe',
 '1Maa': '55ce 569c 5988 5abd 5b24 5b37 5b56 8682 879e 9ebc',
 '1Man': '4607 4f29 5445 7086 868a',
 '1Mai': '54aa',
 '1Mak': '551b 561c',
 '5Wai': '3b19 3b4f 4377 4f1f 5049 6690 709c 7152 7def 7eac 82c7 8466 8af1 8bb3 97d9 97ea',
 '2Laai': '8210',
 '1Bau': '4b6e',
 '2Laam': '652c',
 '6Cing': '3d7e',
 '2Laau': '3ab9',
 '4Fung': '3980 3d6f 43ce 465c 4a7c 51af 6453 7e2b 7f1d 84ec 9022 99ae',
 '6Bok': '399d 3a67 3d61 3f0e 402f 408d 420f 4331 464f 4964 4a0c 4a14 4a87 4aa8 4bcb 6cca 6cfa 6ffc 72a6 74df 7b94 8584 8b08 9251 94c2 96f9',
 '6Syun': '7451 7bc6 815e 942b',
 '3Jaak': '7330 7aab',
 '2Dung': '35a6 38eb 40a2 4d54 61c2 757d 8463 856b 8e35',
 '6Bou': '3c92 3ec9 3f9f 4287 434c 43fd 4470 4480 4916 4c27 4d3a 52fd 57e0 62b1 6355 66b4 6b65 7bf0 7c3f 83e2 8500 8663 90e8',
 '2Gin': '3513 40a9 412f 4bbf 5bcb 8b07 8dbc 8e47',
 '2Gim': '6361 64bf 68c0 6a90 6aa2 7751 77bc',
 '2Giu': '4009 495e 5fbc 649f 657f 76a6 77eb 77ef 7e73 7f34 87dc 8b51 8b65 8e7b 8f4e 9c4e',
 '4Lo': '3469 3f4b 42a8 5138 6924 6b0f 7321 7380 7ba9 7c6e 7f57 7f85 8136 8161 841d 863f 87ba 89b6 903b 908f 947c 9523 9a3e 9a58 9aa1 9e01',
 '1Gei': '3668 36a6 36bb 3786 3a3f 3af7 3c66 3f04 3f22 4008 41eb 47c7 484b 488e 4a6d 4e0c 4e69 501b 5176 51e0 5209 525e 53fd 552d 5630 57fa 5947 59ec 5e7e 6567 671e 671f 673a 6a5f 7284 7391 74a3 7578 757f 77f6 78ef 79a8 7a18 7b95 7f81 7f87 7f88 808c 898a 89ed 8b4f 8ba5 9324 9416 97bf 98e2 9951 9965 9aa9',
 '1Dik': '38a9 3c89 5600 5ae1 63a6 65f3 6ef4 7393 750b 83c2 8c74 8e62 9069 93d1 955d 976e',
 '1Dim': '6382 6541',
 '1Din': '34b9 508e 5dc5 5dd3 5dd4 6ec7 7538 7628 766b 7672 8e4e 985a 985b 98a0 9f7b',
 '1Heoi': '35fe 3f73 42b8 43cf 4589 5401 5474 5618 5653 5729 589f 59c1 65f4 6b54 76f1 865a 865b 8a0f 8fc2 9a49 9b56 9b7c',
 '1Diu': '34ee 368b 408f 4384 496b 4cc2 4e1f 4e22 51cb 5201 53fc 5f6b 7431 7889 86c1 8c82 92a9 932d 94e5 96d5 9bdb 9cb7 9d70',
 '3Tin': '7471',
 '3Tim': '3b87 63ad',
 '3Tit': '9244 9295 9435 94c1 992e 9a56',
 '3Tiu': '6713 773a 7c9c 7cf6 8101 899c 8d92 8df3',
 '3Tip': '3910 494c 4d34 5e16 6017 8cbc 8d34',
 '6Daam': '35d6 3cb7 4a22 5556 5557 5649 56aa 61ba 6c2e 6de1 6fb9 79ab 7a9e 840f 8ce7 972e 9924 9ae7',
 '6Daan': '3b3f 4f46 5f39 5f48 60ee 619a 63b8 64a3 8711',
 '6Daai': '3432 37ad 3f35 4038 403f 4206 446f 47be 4ca6 5927 6c4f 8ed1 8f6a 91f1',
 '2Bei': '3a8f 40fe 43e2 4621 480b 48e5 4ffe 5315 59a3 5ab2 5f7c 67c0 6bd4 6c98 7540 7595 79d5 80b6 979e 9ac0',
 '6Ban': '3913 4f53 7b28',
 '2Geoi': '3aaf 4114 4153 4d9a 4e3e 504a 67dc 6907 6940 6989 6af8 77e9 7b65 8209 8392 842d 849f 8e3d 9f72 9f8b',
 '6Bak': '39ee 3d80 484d 49f6',
 '6Bai': '35d1 3644 3840 3bc7 3d94 4899 5752 5e01 5e63 5f0a 655d 6583 6890 6bd9 72f4 7358 859c 870c 9128 965b',
 '6Bat': '38b6 47e6 4cc1 4f5b 5f3c 62d4 6ccc 72ae 80c8 82fe 83dd 8dcb 8ef7 9238 94b9 98f6 9f25',
 '2Dim': '3e03 4344 70b9 84a7 9ede',
 '2Din': '44e6 5178 75f6 7898 8547',
 '6Daau': '4c4f',
 '6Daat': '37ee 40ee 54d2 5660 7e68 8359 8598 8fbe 9054 943d',
 '1Gep': '55bc',
 '6Daap': '4233 4347 4393 471a 4d6c 6c93 758a 8e0f 8e4b 905d 9314 979c 97b3 9f98',
 '5Po': '888c',
 '5Pe': '6036',
 '6Dyut': '38de 593a 596a 6553',
 '6Gyun': '4275 5026 5708 57e2',
 '2Bo': '377f 7c38',
 '6Gyut': '6a5b',
 '6Dyun': '3ac1 3c6d 482a 65ad 6934 6bb5 6bc8 7146 7c16 7c6a 7dde 7f0e',
 '1Haau': '3e92 40aa 543c 54ee 55c3 5610 589d 5c3b 62f7 6572 6bc3 70cb 70e4 7307 75da 7857 78fd 8653 8b1e 9117 9ad0',
 '1Haap': '384a 3f60 4092 4a10 4d9d',
 '1Fat': '35f5 398c 39fe 3ada 44e4 4746 4a1a 4b0d 4d2f 521c 5488 553f 56eb 5800 5be3 5dff 5e17 5f17 5f7f 5ffd 60da 6612 66f6 6c1f 6df4 72d2 7953 7a9f 7b0f 7b30 7d31 7d3c 7ec2 7ecb 8274 82b4 8300 97cd 97e8 9af4 9b46 9efb',
 '1Fan': '36f0 37a3 390b 3be3 3c6a 4332 439c 4667 4b7b 4d6b 5206 52cb 52db 52f3 5429 5a5a 60db 660f 66db 678c 68d4 68fb 6b99 6c1b 6dbd 7104 718f 71fb 736f 75fb 7767 780f 7aa8 7c78 7d1b 7de1 7e81 7eb7 7f17 7f92 7fc2 81d0 82ac 8364 8477 85b0 915a 91ba 95bd 960d 96f0 9959',
 '1Fai': '39d1 3ace 3e46 5fbd 6325 63ee 649d 6632 6656 6689 694e 7147 7fec 8667 8886 8918 8f1d 8f89 96b3 9c34 9ebe',
 '1Haan': '6173 6394',
 '1Haam': '3583 361b 3991 4689 4869 5d4c',
 '1Haak': '514b 523b 524b 5687 8d6b 9ed1',
 '1Faa': '3a5b 3cf8 82b1 83ef 8624',
 '5Heoi': '4f62',
 '1Gwo': '3ce1 3fc6 48ec 6208 631d 64be 6da1 6e26 7c3b 8596 904e',
 '4Caau': '341c 3f72 3fe7 421e 44d3 48d4 4ca7 5de2 6a14 6f05 8f48 911b',
 '4Caai': '346a 3fb9 410d 44f1 472c 4faa 5115 558d 67f4 8308 8c7a',
 '4Caan': '3c69 43fc 50dd',
 '4Caam': '3506 3574 3704 3742 3c0a 3ea5 3ef8 3f78 45dd 45de 4609 492b 49ef 4cfb 5296 56b5 5d83 5dc9 60ed 6159 615a 67a8 68d6 6b03 6bda 703a 7351 78db 8695 8836 8b92 8c17 9471 9575 995e 998b',
 '5Jeoi': '64e9 6a64 7e60 854a',
 '4Maang': '41c7 6c13 76f2 76f3 8394 9133',
 '3Zek': '53ea 646d 7099 7e3e 810a 818c 8dd6 8de1 96bb',
 '3Leng': '9753 975a',
 '3Ji': '610f 61ff 6baa 7617 761e 7631 858f 8b70 91b7 943f 9571 9950 9dfe',
 '5Maa': '48d5 48d6 69aa 739b 746a 7801 78bc 93b7 99ac 9a6c',
 '1Bou': '354a 3ddb 54fa 5cec 6661 7172 7f39 8912 8943 900b 9914 9bc6',
 '5Mai': '42db 4c4a 5f2d 6549 6e33 772f 7c73 92a4',
 '5Man': '3736 3978 3b06 4b74 50f6 520e 52c4 543b 600b 60af 610d 61ab 6286 62bf 6543 654f 668b 6cef 6e11 6e63 6fa0 7d7b 80b3 8117 9594 95f5 9c35 9c36 9efd 9efe',
 '5Mau': '36b9 4ea9 53b6 67d0 755d',
 '1Bok': '525d 5265 6534 7cea',
 '2Po': '53f5 5c00 7b38',
 '5Hai': '4c92 8b11',
 '5Ham': '3e9d 872d 9837 9844 9894',
 '5Haa': '4e0b',
 '4Fui': '3470',
 '5Hau': '3ad7 539a 5795',
 '4Bui': '68d3',
 '6Baa': '4189 41d1 496f 7f62 7f77',
 '2Gaai': '6a9e 89e3 89e7',
 '1Cyu': '5132 5e6e 6ae5 6ec1 8e87 8e95',
 '2Gaam': '3536 3e82 51cf 6e1b',
 '2Gaan': '3899 3d4e 3d95 3e4f 424d 4b59 583f 62e3 63c0 67a7 67ec 6898 7877 78b1 7b67 7b80 7c21 7e6d 8327 8949 897a 9e7b 9e7c',
 '2Gaau': '3a6d 3f71 4f7c 59e3 6405 641e 64b9 652a 72e1 768e 7b4a 7d5e 7ede 9278 94f0 9903 997a',
 '6Neot': '356f 43aa 4b46 4b53 9df8',
 '4Keoi': '3739 38c4 3beb 3c98 4020 4082 41bd 42e7 4466 4763 4d76 52ac 61c5 65aa 6710 6c0d 6e20 7048 74a9 766f 78f2 7ad8 7c67 7d47 80ca 81de 8556 8627 8862 8ea3 8ee5 9d1d 9e32 9f29',
 '2Diu': '412a 4246 5c4c',
 '2Dong': '48ca 48e3 515a 6321 64cb 6529 6863 6a94 8b9c 8c20 9482 9ee8',
 '1Lip': '4882',
 '2Waa': '5250 526e 6432 753b 756b 8a71 8bdd',
 '2Heoi': '37b0 4168 4513 48f4 5194 53bb 54bb 55a3 662b 6829 6b28 7166 8a31 8a61 8bb8 8be9 9126',
 '1Cyun': '41a4 41a5 5ddd 6751 6c1a 6c46 7a7f 8e06 8e7f 8ea5 90a8 91e7',
 '5Jin': '4f23 4fd4 5156 5157 55ad 5dd8 665b 7517 774d',
 '5Jim': '4fe8 513c 5189 5261 59cc 624a 67d3 6a6a 7430 82d2 9869',
 '3Fiu': '8985',
 '5Loeng': '34b3 351d 44e3 4803 4a6b 4e21 4e24 4fe9 5006 5169 7dc9 88f2 9b49 9b4e',
 '5Jiu': '5060 62ad 6e94 7ed5 8200 9076 9a15',
 '5Lei': '3e1a 3f96 49c9 4fda 59b3 5a0c 5c65 5cdb 674e 6d6c 7406 88cf 88e1 9026 9090 91cc 92f0 9502 9bc9 9ca4',
 '3Ung': '74ee 9f46',
 '5Wong': '4c23 5f80 5f83',
 '4Sung': '34fd 5d07',
 '1Syu': '34f1 3eec 42e1 49fe 4e66 59dd 6292 6445 6474 6504 66f8 67a2 6a17 6a1e 7d13 7ebe 8212 8f38 8f93 9103',
 '5Gaa': '35c7',
 '1Tou': '37b5 38ad 3ade 3ae6 3e57 401e 4231 4b22 4b2d 5f22 6146 69c4 6ed4 7d5b 7e1a 8b1f 97dc 97ec 9955',
 '1Waa': '359e 3c2a 4a1f 4bc4 4d77 54c7 54d7 558e 5a03 6316 6c59 6d3c 6f25 7a8a 7a90 7aaa 86d9 8717 8778 8b41 9f03',
 '3Ong': '3f1c 4c40 7515 7f4b 8eee 91a0',
 '1Wai': '364e 38e6 3b83 4d27 502d 55b4 59d4 5a01 5d34 840e 8473 875b 9036',
 '1Zou': '385f 3bfe 3dee 470a 6152 79df 7cdf 84a9 8e67 906d',
 '1Zoi': '39b2 54c9 683d 6e3d 707d 707e 70d6',
 '4Cim': '3480 406e 456d 4ac8 648d 6f5b 6f5c 6ff3 71c2',
 '4Cin': '367b 3886 39ae 48d1 5103 524d 5a8a 5edb 700d 7e8f 7f20 8e94',
 '1Wat': '4587 4d65 55e2 5c48 6b1d 7140 7229 817d 8183 8a58 8bce 9b31 9ee6',
 '2Fong': '3442 356b 393a 39cd 3a2a 405c 4130 4319 488d 4cb1 4eff 5023 5e4c 5f77 6033 604d 623f 6609 6643 69a5 6ec9 7180 74ec 769d 7d21 7eba 822b 8a2a 8a64 8bbf 9ae3',
 '2Haau': '36bd 4bea 4cbe 5de7 6537 6832 7187 8003 85a7',
 '6Mun': '3943 3996 3d4d 60b6 61d1 61e3 7116 71dc 95f7',
 '6Mui': '3b51 59b9 6627 6cac 771b 97ce',
 '6Muk': '3641 3708 3821 39c5 4032 42be 4475 5776 6728 6be3 6c90 7267 76ee 7766 7a46 82dc 926c 94bc 9702',
 '1Goeng': '3cfe 3e54 456c 45f5 4c94 50f5 59dc 5f4a 6a7f 6bad 7586 7913 7e6e 7f30 7f8c 7f97 8591 8723 97c1',
 '3Gei': '34ab 3c1f 4400 44fd 5bc4 5f9b 65e2 65e3 6d0e 7a4a 8507 89ac 89ca 8a18 8bb0',
 '3Teoi': '3946 3ddf 4d6f 9000',
 '2Haam': '43a6 49a6 4af2 4b60 83b0 95de 961a 9b2b',
 '2Haan': '3e27 77af',
 '6Sap': '5341 62fe',
 '6Sat': '5b9e 5bd4 5be6 6e5c',
 '6Sau': '359f 3945 46f5 53d7 552e 58fd 5900 5bff 6388 7dac 7ef6',
 '1Zaau': '3584 35a9 35c0 35d9 3633 36b1 3951 3962 3c4a 3c57 3ccb 3cdc 3d17 3e6f 3f8b 41b6 4387 440d 4757 4973 4aae 4b00 4b8c 4c0d 5632',
 '1Zi': '36c4 37a2 37a8 383f 394b 3a7c 3bc4 3c8d 3cd6 3d2f 3f67 3f7b 3fa8 3ff2 40dc 4154 41ec 42b7 4336 43a9 44be 44e1 46b7 475d 48fd 49c7 49f4 4ab0 4c35 4d79 4e4b 5179 536e 5431 5472 54a8 55de 59ff 5b5c 5b73 5d6b 5df5 6418 652f 6800 6894 6925 69b0 6dc4 6ecb 753e 77e5 7957 79d6 7ca2 7dc7 7f01 80a2 80d1 8102 829d 83d1 8718 8786 8aee 8c18 8cb2 8cc7 8d40 8d44 8d91 8f1c 8f8e 9111 9319 93a1 9531 9543 9aed 9bd4 9cbb 9cf7 9f12 9f4d 9f5c 9f87',
 '4Ci': '352d 3602 3639 365c 37b4 38ae 3935 3b9b 3e51 4185 4218 4421 4659 46d0 4704 4aa7 4aef 4b63 4b88 4cbf 4d94 4db5 5319 577b 5880 6148 6301 6b6d 6c60 6cbb 6cdc 7386 74f7 78c1 7960 7b88 7b8e 7bea 7ccd 8110 81cd 830c 8328 832c 8332 834e 858b 86b3 8a5e 8bcd 8e1f 8f9e 8fa4 8fad 8fc9 8fdf 9072 9908 99b3 9a70 9dbf 9dc0 9e5a',
 '6Sai': '4931 566c 6fa8 7b6e 7c2d 8a93 901d 907e',
 '6San': '613c 614e 80c2',
 '1Ze': '3e19 4988 546e 55df 7f5d 906e',
 '6Sam': '4ec0 6939 751a 845a 9eee',
 '5Sin': '4c47 4c49 6f6c 87fa 9c53 9c54 9cdd',
 '1Ging': '3964 4eac 5162 5a5b 5de0 60ca 6cfe 6d87 77dc 7d93 7ecf 8346 834a 9d5b 9d81',
 '5Ngo': '39b1 39f4 6211',
 '4King': '348c 3500 3741 38a3 3bf3 3dc0 405a 4d5e 52cd 5898 60f8 64ce 6aa0 712d 7162 743c 74ca 778f 8315 85d1 9be8 9cb8 9ee5',
 '6Sing': '4789 5269 5d4a 8cf8',
 '2Dang': '44ad 6225 7b49',
 '4Jim': '400b 4e25 56b4 58db 5acc 708e 7130 76d0 868e 8692 86ba 86c5 9586 95bb 95eb 960e 9aef 9e7d',
 '1Do': '364d 4ac2 591a',
 '1Seon': '3844 3a1a 3c2c 4f9a 5ccb 5f87 6042 6b89 6d35 6f60 72e5 73e3 76b4 8340 8a62 8be2 9021 90c7',
 '1Seoi': '346f 3960 3ca4 41d3 421d 42f6 4492 4498 4500 4551 49cc 4a8e 4b3a 4b50 4bef 4c6c 5020 590a 5b03 6454 6948 6e51 6fc9 7762 7a30 7d8f 7df0 7e7b 7ee5 80e5 837d 837e 8470 867d 8751 8add 8c1e 9450 96d6 9700 9808 987b',
 '6Nyun': '4782 5a86 5ae9',
 '3Bing': '4202 4e26 4f75 8ff8',
 '1Seot': '3585 378a 3c9e 3eb7 460f 48a6 5379 6064 620c 7aa3 87c0 88c7',
 '4Tong': '347d 3676 370d 3b7b 3ca5 3f3a 416f 424e 4305 454b 48d8 5510 5802 5858 642a 68e0 6a18 6e8f 717b 746d 819b 8797 87b3 8d6f 91a3',
 '1Gaang': '3a85 43b4 754a 8015',
 '6Pak': '3e52',
 '4Wui': '4f6a 56d8 56de 56ec 5efb 5f8a 605b 6d04 75d0 8334 8698 86d4 8ff4',
 '4Wun': '3d79 3fea 4d1f 57a3 5cd8 6853 72df 7342 74db 7de9 7fa6 8341 8411 8c86 96c8',
 '6Dau': '54e3 68aa 6d62 75d8 7aa6 8130 8373 8b80 9017 92c0 9916 997e',
 '4Tang': '4568 4c8d 4ca2 6ed5 75bc 7c50 7e22 817e 85e4 8a8a 8b04 9086 99e6 9a30',
 '3Maa': '3fba 45eb 4be6 4d62 5417 561b',
 '5Aa': '5493',
 '1Gaam': '390c 3997 42ee 4da0 4da2 5c34 5c37 6937 76d1 76e3 7dd8 7f04',
 '2Hau': '482e 53e3',
 '3Bui': '3528 36dd 3e2c 3e3d 42f3 47fa 4c3d 505d 6096 63f9 6896 6d7f 72c8 72fd 80cc 84d3 8919 8c9d 8d1d 8ef0 8f29 8f88 90b6 92c7 94a1',
 '6Haau': '390a 4cce 509a 52b9 6054 6548 6586',
 '6Haat': '385a 3bab 3d7f',
 '3Bun': '3ab5 534a 59c5 62cc 7249 977d',
 '5Pai': '7764 7765',
 '1Mung': '61dc 6aac',
 '5Pan': '725d',
 '2Haa': '5413',
 '2Ham': '3673 57f3 627b 6b3f 780d 8f57 9851',
 '2Han': '4f77 57a6 58be 5f88 6073 61c7 72e0 8c64 9f66 9f88',
 '6Haam': '3a14 3f28 4404 49df 4c64 4d9f 9677',
 '6Haan': '387e 4042 4998 648a 9650',
 '6Meng': '547d',
 '4Hau': '35cb 3927 3b0b 3ba2 3e85 45d4 4648 4adb 4faf 5589 7334 760a 7bcc 7cc7 7fed 9107 9297 936d 9931 9bf8',
 '4Ngaa': '39ce 414e 4353 4f22 6792 6860 7259 72bd 82bd 869c 8859',
 '2Syun': '3535 437b 4423 4823 50ce 5334 635f 640d 64b0 7bf9 9009 9078',
 '4Haa': '3c3a 4a97 4ad7 7455 8578 8d6e 9050 971e 9a22',
 '4Soeng': '3982 5018 507f 5617 5690 5ae6 5c1d 5e38 5f9c 751e 88f3 9c68 9cbf',
 '4Hai': '4bd0 5092 516e 595a 5faf 8787 8c95 8e4a 90cb 9a31 9f37',
 '4Han': '3bca 49a5 62eb 75d5',
 '4Ham': '37cf 37d4 3bad 3db0 4204 4934 4941 4a21 4d83 542b 7113 90af 9163',
 '4Zing': '3429 34dd 3732 3a15 3b41 3d04 3d88 3e59 3e5a 3eb1 3f1b 3fd8 3fee 40f7 4127 4449 4566 46ae 492f 4a0d 9649',
 '6Zi': '3465 3483 387c 389e 38e5 3cbc 3e3b 3e53 4101 413a 4188 425c 452c 454c 458a 458c 476f 4999 4f3a 5082 5155 55e3 5b57 5bfa 5df3 5ea4 5f58 6d98 7564 75d4 7726 7940 7a1a 7a3a 7ae2 7b25 801c 81ea 8997 98e4 98fc 9972',
 '6Zo': '3634 52a9 5511 5ea7',
 '6Ze': '34d4 3d2c 464a 4933 69ad 8b1d 8c22',
 '1Laau': '4056 4c7e',
 '1Laai': '5b7b',
 '1Laan': '56d2 8e9d',
 '3Jeoi': '8739',
 '3Cai': '4019 43c5 4d1d 50ba 5207 6455 780c',
 '6Dit': '39b6 3cb3 43f2 4c43 4cc0 54a5 57a4 5e19 6633 67e3 74de 7723 79e9 7d29 7d70 7ed6 7fd0 800b 80c5 81f7 82f5 87b2 889f 8a44 8efc 8fed',
 '3Cak': '3a5e 4349 47a3',
 '3Cam': '3a3b 5423 551a 8b56 8c36',
 '6Dip': '3459 373c 3948 3a78 3a79 3b2a 3cb2 3d2e 4548 4ac6 4ba2 4d11 53e0 5551 558b 581e 60f5 63f2 696a 7252 789f 8253 8728 8776 890b 8936 8adc 8c0d 8e40 9371 9c08 9cbd',
 '3Can': '3f69 512d 56ab 6987 6aec 7019 707f 7a31 886c 896f 8d81 9f54 9f80',
 '3Caa': '38c7 38fe 391e 3cd7 42ac 45b3 47d5 4f98 597c 59f9 79c5 8721 8869 8a6b 8be7',
 '5Naai': '4e43 5b2d 6c16 8ffa 91e2 9f10',
 '2Gang': '3b93 3f98 4304 4c4d 4c4e 4c6d 4e99 57c2 5829 6897 71b2 7d86 7ee0 803f 90e0 9abe 9bc1 9ca0',
 '6Zat': '3475 4f84 5394 59ea 5ac9 5ea2 684e 69c9 75be 79f7 7a92 81a3 84ba 86ed 86f0 87c4 90c5 928d 94da',
 '5Naam': '4112 6df0',
 '5Naan': '8d67',
 '6Zap': '35f1 366b 3ce7',
 '6Zan': '9635 9663',
 '6Zam': '3be2 3d28 3f09 4cb4 6715 681a 7739 9d06 9e29',
 '4Zung': '3758 37fe 38d1 3b5c 3c93 3ca8 3e8e 3f38 41c0 4236 462c 481c 49e1 4a78 9c45',
 '6Zak': '409d 4431 4d42',
 '6Zai': '3e04 3fc3 6ede 6eef',
 '3Cap': '350d 36fc 388e 3934 3d6c 4579 46fd 49ba 4b9c',
 '6Din': '3649 379f 3e60 4f43 57ab 5960 6dc0 6fb1 7535 765c 8714 923f 94bf 96fb 975b',
 '3Cau': '35dc 50b6 51d1 55c5 6ba0 6e4a 6eb4 7cd7 8160 81ed 8f33 8f8f 9f45',
 '3Cat': '3e9a 4328 44ed',
 '6Dik': '3e4d 4183 4a24 4bfc 4d1e 654c 6575 6d5f 6da4 6ecc 72c4 7c4a 7fdf 837b 85cb 85e1 89bf 89cc 8fea 9349 9e10',
 '1Soi': '46a1 4c04 6122 6be2 6bf8 816e 984b 9c13 9cc3',
 '2Hyun': '72ac 72ad 753d 754e 8647',
 '5Joeng': '4ef0 517b 536c 56b7 57cc 58e4 6c27 75d2 7662 990a 9b24',
 '2Co': '40b3 4802 510a 61b7 695a 6a9a 6fcb 7840 790e 811e',
 '2Ci': '3881 388b 3989 3c40 3db4 40a0 42bc 4490 44a8 46c2 48ce 49a0 4c29 4f4c 4f88 5953 59cb 59fc 5f1b 6040 6065 6261 62f8 675d 67c2 67ff 6b3c 6b64 6cda 73bc 77e2 803b 80e3 892b 8a83 8b3b 8c55 8dd0 9aca 9f52 9f7f',
 '2Ce': '43e1 4e14 54c6 5972 5c3a 626f 62af 64a6',
 '3Sei': '350c 355c 43e4 56db 6b94 8082 8086 857c',
 '3Sek': '34aa 36eb 932b 9521',
 '1Haai': '54b3 63e9 7e72',
 '2Cou': '4491 6145 8278 8349 9a32',
 '1Bing': '34c8 4e52 5175 51ab 51b0 5e76 63a4 681f 6c37',
 '1Hou': '5686 8320 84bf 8585 9ac7',
 '4Pou': '422c 423b 4515 4a63 530d 83e9 8461 84b2 888d 917a',
 '1Hoi': '368a 3769 3c29 3c7c 3f82 548d 5f00 958b',
 '1Teng': '3554 5385 5ef3 807d',
 '1Hoe': '9774',
 '2Coi': '3952 433d 48cb 5bc0 5f69 63a1 776c 7db5 831e 8df4 91c7',
 '1Zang': '42eb 4396 448f 4c62 589e 5ce5 5d1d 618e 6a67 77f0 78f3 7e52 7f2f 7f7e 9b19',
 '1Lou': '565c 5695',
 '6Nik': '3cbb 60c4 7592',
 '6Nim': '3dd1 49d4 5538 5ff5',
 '4Zyu': '340c 3924 395a 3965 3a0e 3b02 3b30 3c1d 3cdb 3d9b 3dd2 3e58 3e9e 3ec0 3f92 4069 409b 40cb 42c8 43f8 4433 451f 4521 45e8 462b 473d 4795 48a9 4a71 4af1 4b14 4c2d 4c3b 4c77 4ca3',
 '6Nip': '358f 3628 363f 365e 3694 37bc 3a76 3e0e 40bc 41e3 431c 46be 4706 4bc0 4bc5 4cd6 634f 63d1 651d 655c 6d85 6e7c 8042 8076 82f6 8e02 8e51 8ea1 9477 954a 9667 9689 9873 989e',
 '6Niu': '3799 5c3f',
 '4Lung': '3685 376b 3863 3995 3c0d 3eaa 40e7 418d 43ca 4a8a 5499 56a8 5dc3 663d 66e8 6727 680a 6af3 6cf7 7027 73d1 74cf 7643 772c 77d3 783b 7931 7adc 7b3c 7c60 804b 807e 80e7 830f 8622 882a 8971 8e98 9468 9686 9f8d 9f99',
 '6Jeon': '648b 6da6 6f64 81b6 958f 95f0',
 '6Jeoi': '6798 6c6d 777f 82ae 868b 88d4 92b3 92ed 9510',
 '2Koi': '6168 6982',
 '6Baak': '3f1f 47b3 4cc6 50f0 5310 5e1b 767d 83d4 8514 8e23',
 '6Baai': '3525 3c59 43f6 4b86 5457 5504 60eb 618a 6557 7a17 7cba 8d25',
 '2Mou': '4cc7',
 '3Cok': '4c5c 64c9',
 '3Daan': '387a 4725 65e6 72da 7605 7e75 8a95 8bde 926d 94bd',
 '3Coi': '35f7 3614 4c02 57f0 585e 7e29 83dc 8521 8cfd 8d5b',
 '3Leoi': '6ada 9462',
 '6Ho': '3d51 8cc0 8d3a',
 '2Gung': '3928 3b5f 3eac 3f26 40ac 4857 5de9 5efe 62f1 62f2 6831 73d9 86ec 8f01 978f',
 '6Baat': '3832 38b8 3a3d 4136 496c 4cca 8307 9b43',
 '3Hin': '49ee 5baa 61b2 732e 737b',
 '4Pan': '56ac 5ad4 73ad 77c9 860b 8ca7 8d2b 983b 9870 9891 98a6',
 '3Zaan': '4c9f 56cb 7052 74d2 74da 8b9a 8cdb 8d0a 8d5e 9147 9961',
 '2Zaan': '3406 373a 3865 38fc 396f 397c 39a9 4170 4325 4392 43f0 45ce 4a38 5139 661d 68e7 76cf 76de 8db1 8db2',
 '2Zaam': '3c43 406a 5bc1 5d2d 5d84 65a9 65ac 7728',
 '3Zaam': '3d34 5061 8638',
 '3Zaak': '35bd 3cfb 41e5 4bd4 5536 5567 5616 5c9d 77fa 789b 78e7 7a84 7b2e 7ba6 7c00 8234 8cac 8cfe 8d23 8fee 9f70',
 '2Zaai': '3d09 3d13',
 '3Zaai': '3afc 41e7 41fd 4c0f 503a 50b5 7635',
 '2Cau': '4390 4e11 677b 7785 919c',
 '1Ting': '42fc 4a60 4bd5 686f 70c3 70f4 7d8e',
 '1Ho': '3a83 3c52 412c 43dc 5475 5777 82db 8a36 8bc3',
 '2Can': '3983 42b2 442e 4d3c 5432 54c2 62bb 6375 6b84 75a2 75b9 7715 77e7 7d3e 8897 8a20 8a3a 8bca 8fb4',
 '3Seon': '3430 3b44 4022 4035 445e 46dc 46e8 476e 4abf 4b40 4d04 4fe1 5640 56df 5dfd 613b 6c5b 779a 77ac 821c 8563 8a0a 8baf 8fc5 8fff 900a 905c',
 '2Cam': '3538 3717 3772 3a65 3f9b 3fca 4ad6 588b 5bd1 5bdd 5be2 789c 78e3 8e38',
 '3Seoi': '37b8 3f77 4b68 5c81 5d57 5e05 5e25 5e28 6329 6b72 6d97 788e 7a05 7a0e 7e40 86fb 8715 88de',
 '3Zaat': '3789 39ab 3c5c 3ce8 54f3 624e 62f6 672d 7d25 7d2e 86bb 9358 94e1',
 '3Zaau': '3447 3473 34dc 3918 396e 3df9 4207 435c 46c6 7b0a 7f69',
 '3Zaap': '5273 5284 531d 647a 7838',
 '1Sang': '50e7 72cc 9f2a',
 '6Seoi': '34b8 395e 3bdc 3c56 3d1a 3dc2 3eea 3efd 4157 418a 41b3 424c 4341 4539 4875 4959 5f57 69e5 6a96 71e7 745e 74b2 7601 775f 7761 795f 7a57 7a5f 7bf2 7cb9 7e50 81b5 8403 8527 895a 8ab6 8c07 8ce5 9042 9083 9406 9429 96a7',
 '6Seon': '3967 9806 987a',
 '6Seot': '3f81 419d 672e 672f 6cad 79eb 8853 8a39 8ff0 9265',
 '3Sik': '88fc',
 '1Hat': '43a2 4e5e 5fd4 8fc4',
 '1Hau': '3851 3aa3 3af4 3e72 4bcc 4be8 777a 84f2 9f41',
 '6Goeng': '4952 5534 5f37 7ce8',
 '1Hap': '404d 406f 533c 5e22 6070 6415',
 '1Caang': '5000 6490 6491 6a55 722d 77a0',
 '1Haa': '54c8 5c88 867e 8766 959c 98ac 9c15',
 '1Ham': '3f4d 51f5 52d8 5505 574e 5769 582a 5d41 61a8 6221 86b6 8c3d 9f95 9f9b',
 '1Han': '60ad',
 '1Hai': '80b9 95aa',
 '1Ku': '8ef2',
 '1Hak': '39a6 3d7a 44c7 4682 53ef 5c05 6c2a',
 '3Hap': '4d97',
 '5Teng': '8247',
 '1Zung': '3661 3672 3687 3709 3895 38ed 3a11 3aea 3bac 3bf6 3e63 3e8b 3f6b 3fc8 41d7 4226 4235 423a 4478 45e5 4634 4642 486e 4a7a 4c75 4e2d 4f00 5027 5990 5b97 5c30 5d55 5f78 5fe0 60fe 68d5 6936 72c6 76c5 7a2f 7d42 7d9c 7e31 7e4c 7ec8 7fea 8202 822f 8520 87bd 8c75 8e2a 8e64 9350 937e 9418 949f 953a 9a23 9b03 9b37',
 '6Bung': '5873',
 '2Baai': '636d 6446 64fa 896c',
 '3Ham': '350b 360e 3615 3e96 400d 457f 5888 5d01 7400 77b0 77d9 78e1',
 '3Wok': '77f1 96d8',
 '2Mung': '61f5',
 '2Gou': '368f 3696 3fb8 66a0 6772 69c0 69c1 7693 7a3e 7a3f 7e1e 7f1f 85c1 93ac',
 '1Piu': '3f3c 4408 560c 5f6f 647d 6f02 7e39 85b8 87b5 98c4 98d8',
 '2Zeoi': '40ca 4864 4b94 4da5 5480 5634 89dc 9f5f 9f83',
 '2Zeon': '3bf8 4772 5118 51c6 57fb 6e96 71fc 7da7 8369 85ce 8cee 8d10 8d46 96bc',
 '3Hit': '6b47 7332 874e 880d',
 '2Goi': '4ab1 6539 80f2',
 '2Gon': '4d5f 76af 79c6 7a08 7b78 8d76 8d95',
 '5Syun': '542e 96bd 96cb',
 '6Joek': '5f31 66f0 6949 6c4b 7039 721a 759f 7627 793f 79b4 7bac 7bdb 7c65 7c72 84bb 85e5 8625 8650 8b14 8c11 9100 9205 9470 94a5 9fa0',
 '4Ng': '3e73 3ecd 44ca 499c 4ad3 4b8f 5433 5434 543e 5cff 68a7 6d6f 73f8 75e6 8708 90da 92d8 9f2f',
 '4No': '50a9 513a 633c 637c',
 '1Ding': '34c5 38d4 47d3 4e01 4ec3 53ee 5722 6c40 738e 76ef 8035 914a 976a',
 '1Muk': '3572',
 '1Mun': '9862',
 '6Cuk': '4652 46d9',
 '5Zyun': '34f4 371b 3b95 3f6d 4193',
 '1Mut': '6b7e',
 '2Hing': '34b7',
 '6Maang': '5b5f',
 '4Zoeng': '3682 3b15 4051 4274 45b9 4b17',
 '3Sip': '34a4 404b 4330 6151 61fe 6444 6d89 6ee0 7044 71ee 8e9e 97d8',
 '2Hong': '37df 391a 427a',
 '6Gei': '3831 394d 42df 4f0e 5993 5fcc 5fee 60ce 6280 82b0 8a8b 8dfd 9a0e 9b3e 9d4b',
 '6Bing': '7add',
 '2Tung': '36da 38da 3cc6 6876 7b52 7b69 7d71 7d82 7edf',
 '2Pei': '46b0 46b9 490f 4acc 4ef3 56ad 572e 5e80 75de 82e4 9119',
 '3Zim': '4f54',
 '3Zin': '3b2b 3dfc 4729 4840 4d0f 6218 6230 682b 6d0a 726e 7bad 8350 8465 85a6 986b 98a4 991e 996f',
 '2Joeng': '4f52 5771 600f 62b0',
 '3Zik': '7620 8e50 8e60 9dba 9e61',
 '3Zit': '353e 3609 37af 39a2 3abf 3b9e 3e05 42a6 43f3 44c6 4b47 4c99 5369 54f2 5586 5a55 5c8a 608a 6298 6377 6976 6d59 6ddb 758c 7596 7664 7bc0 8282 8707 8c12',
 '3Ziu': '3754 3b2d 3dd6 66cc 7042 70a4 7167 76ad 8a54 8bcf 91ae 91c2',
 '3Zip': '3847 3a4e 5022 56c1 6174 63a5 6904 696b 6d43 8034 8975 8b8b 8b98 8f12 8f84 9705 9bbf',
 '5Ng': '3405 3487 3b33 4e94 4ef5 4f0d 5348 554e 5fe4 727e',
 '5No': '3845 3c19',
 '1Baang': '3b3b 409c 494b 4a1c',
 '1Cin': '3868 4012 4133 4266 48ac 48b4 4edf 5343 6266 68b4 74e9 828a 8fc1 9077 9621 97c6',
 '1Cim': '3510 3674 3884 39b0 3bf0 3ea4 3f50 3f86 3fbe 3fcc 40f1 40f8 418e 42b1 42b9 4bed 4bf9 4d4c 4f65 50c9 5b10 5b45 5e68 61b8 6515 66b9 6afc 6b7c 6bb2 7038 7b7e 7c3d 7c64 7e8e 7e96 7ea4 83b6 859f 8973 915f 929b 92df 94e6 9513',
 '1Cik': '387f 3a7d 3c17 3f5a 45e9 4cf5 52c5 52d1 53f1 5f73 617c 617d 621a 62b6 6475 6555 65a5 6790 683b 69ed 78e9 87ab 93da 98ed 996c 9d92 9dd8 9f1c',
 '2Deoi': '349b 3650 44b3',
 '2Faan': '3906 484a',
 '1Kwaa': '4bde 57ae 5938 59f1 8342 8a87 8de8 9299',
 '1Kwai': '4093 4c85 4e8f 5ae2 5cbf 5dcb 69fb 69fc 76d4 777d 77a1 7aa5 7aba 898f 89c4 95da 980d',
 '1Ciu': '38cd 3e66 4aff 4d72 5281 5e67 5f28 600a 662d 7e70 7f32 8d85 91d7 936c 948a 9539',
 '1Kwan': '37d2 45b5 56f7 5764 5803 5cee 7311 7885 7b98 83ce 9a09 9ae1',
 '4Zaam': '3542 3597 3e12 433e 457e 46d8 4a19 4aa9',
 '4Zaan': '3759',
 '5Nong': '3d9e 64c3 652e 66e9 7062 8e9f',
 '4Zaau': '34a1 3571 35bb 3625 36ad 36dc 3acd 3cca 3cfa 3dd5 3f55 3f65 3f9e 4128 4301 431b 4413 447b 45bb 488a 489f 4c06',
 '2Cyun': '3c5b 453f 47bc 520c 5598 570c 5fd6 6b42 7ac4 821b 8348',
 '3Tek': '8e22',
 '6Lyun': '4e71 4e82 858d',
 '1Daang': '3501',
 '2Gong': '4070 441f 6e2f 8029 8b1b 8bb2 985c',
 '4Ling': '35ab 3875 3944 39ad 3aae 3b21 3bea 3c65 3c86 3e33 3ecf 3f89 4125 420a 4241 4244 4256 4279 4322 4345 4516 4558 4585 4665 4696 4832 487c 487f 49d9 4a29 4b0b 4bcd 4c31 4d07 4d12 4d2b 4f36 51cc 5464 56f9 580e 590c 5cad 5d1a 67c3 68c2 68f1 695e 6ab8 6afa 6b1e 6ce0 6de9 7075 72d1 73b2 74f4 7a11 7a1c 7b2d 7dbe 7eeb 7f9a 7fce 8046 8232 82d3 83f1 8506 8590 8626 86c9 8a45 8ee8 8f18 9143 91bd 9675 96f6 971d 9cae 9cf9 9d12 9e30 9f61 9f84',
 '2Zong': '9a75',
 '3Lo': '645e',
 '2Naa': '4e78',
 '5Cyu': '344f 3749 3916 41e0 41e1 42b0 4346 48dd 4f2b 4f47 50a8 5b81 66d9 677c 67f1 7ada 7d35 7ebb 7f72 7f9c 82a7 82ce 82e7 8d2e',
 '2Zing': '388d 3c9f 4034 4b4a 4b57 4b58 6574',
 '5Mou': '3444 37c2 39c7 3d72 4489 47fc 4948 4fae 5187 59a9 59c6 6003 61ae 62c7 6b66 6bcd 6f55 73f7 7512 782a 7894 821e 9267 9d61 9e49',
 '5Lung': '3659 3d33 4841 5131 5784 58df 5fbf 62e2 650f 7be2 9647 96b4',
 '3Seng': '59d3 935f',
 '1Zaai': '3ab0 4122 4364 471e 658b 9f4b',
 '3Kwaa': '42c0 80ef',
 '1Zaak': '421f',
 '1Zaam': '3a9b 3fa3 40e2 4167 4543 46f8 473e 4903 4a27 5142 7c2a 9415',
 '1Zaan': '363b 3919 39e2 3b92 3d8f 3f99 4104 422f 4370 44f0 49e3 4b3b 7416 9186',
 '3Kwai': '5abf 6127',
 '2Wai': '355f 3a53 40e3 4374 46fc 49b1 4f4d 5349 552f 5582 559f 58dd 5bea 5d23 6bc0 6bc1 6bc7 71ec 7325 75ff 8172 848d 873c 8ac9 8b6d 8bff 9aab',
 '2Wan': '3767 6435 7a33 7a69',
 '3Kwan': '56f0 774f 7a98',
 '1Zaap': '36d5 3b24 3df8 4046 4257 43e0 44c3 4531 4879 4b42 4be5 4c52',
 '3Ci': '34e8 35d3 3880 41fc 45f9 47f1 4826 4bf8 4c21 4cd0 4f7d 5129 523a 5395 53a0 557b 5e1c 5e5f 5e9b 6063 61e5 673f 676e 6b21 70bd 71be 7719 7d58 7fc4 7fc5 7fe8 83bf 869d 86d3 8cdc 8d50 8dee 994e',
 '3Sang': '64e4 8e6d',
 '2Tong': '3489 3a8c 3f12 3fe9 50a5 513b 6345 66ed 6dcc 7223 77d8 7cd6 8e5a 8eba 93b2 940b 94f4',
 '5Long': '34ea 365f 3af0 3bbe 5871 6717 70fa 95ac 9606',
 '6Wik': '369c 3f63 4205 57df 68eb 7dce 7f6d 872e 95be 9608 9b4a',
 '6Faau': '4600',
 '2Zeng': '4e3c 4e95 6c6b 80bc',
 '6Jing': '5ab5 8ba4',
 '6Faan': '3568 72af 7b35 7bc4 8303 92c4 98ef 996d',
 '4Jyu': '4e0e 4e88 4f03 4f59 4fde 5112 5685 5982 59a4 5a1b 5a31 5b2c 5b7a 5c7f 5d4e 5dbc 5e24 6109 611a 6310 63c4 65df 6970 6986 6b24 6b48 6b5f 6bf9 6e14 6e1d 6e61 6f01 6fe1 724f 72f3 745c 74b5 756c 76c2 776e 79ba 7aac 7fad 8174 81fe 8201 8206 8207 8245 8339 8438 8498 854d 8560 85b7 865e 8753 8761 8815 88bd 8915 8966 89a6 89ce 8adb 8b23 8c00 8e30 8f3f 8f5d 903e 91b9 91ea 92a3 94f7 9683 9685 96e9 9918 9980 9b5a 9c7c 9d3d 9e06 9e12 9f75',
 '2Zang': '3f6a',
 '1Jyun': '51a4 596b 5b9b 5bc3 6df5 6e0a 7722 8099 847e 870e 873f 9cf6 9d77 9e22 9e33 9e53 9f18',
 '5Taam': '8d09',
 '2Tau': '3a97 3f25 551e 59b5 6568 9ec8',
 '1Tiu': '3b38 446c 4837 4f7b 5ea3 604c 6311 7967 825e',
 '2Faau': '357b 3c34 4b0f 4be0 4cdd',
 '1Teoi': '379c 84f7',
 '1Teon': '4974 4cea 4d4a 554d 6e4d 829a 8c92',
 '2Tai': '4321 4a86 7747 9ad4',
 '2Faai': '3ed7',
 '1Tik': '3942 39f9 47b6 48b0 4bdc 501c 5254 5fd1 5fd2 6090 60d5 60d6 64ff 8ca3 8daf 9016 92f1 94fd',
 '2Tam': '3cad',
 '1Tim': '3b32 3d53 6dfb',
 '1Tin': '4016 44b6 46b6 5929',
 '2Tan': '350a 47a1 6c3d',
 '4Fan': '3d97 3e2e 424a 4252 447a 4af6 4d05 5746 575f 58b3 59a2 5e69 68fc 6c7e 6fc6 711a 7fb5 84b6 8561 86a1 8c6e 8c76 8f52 9b75 9ec2 9f16 9f22',
 '6Zong': '72b6 72c0 810f 81df',
 '4Zyun': '3623 3774 37f2 37f6 3973 3adf 3cc2 3e49 3e61 4357 45a0 4760 4a59 4b27 4c72 4cae 4cbb 4cd2',
 '4Ceoi': '3551 3968 3d66 3d6d 3edf 4132 434b 4371 4714 474b 48ab 4c70 4d5a 4d86 53a8 5eda 5f90 6376 6425 68f0 690e 69cc 7ba0 7be8 8144 8197 83d9 870d 8ac8 9318 939a 9524 964f 9664 968b 968f 96a8',
 '4Fau': '6d6e 6daa 70f0 7408 7a03 7d11 7f58 7f66 82a3 8709 88d2 9d00',
 '4Ceon': '3d4c 3f66 42f8 45b2 4629 4649 4845 5de1 5ef5 5faa 63d7 65ec 6812 79e6 7d03 8793 99b4 9a6f',
 '3Cung': '39e4 63f0 9283 94f3',
 '3Zyun': '3950 43cd 4938 6358 7e33 940f',
 '4Mau': '35bc 45cb 4950 4c55 4f94 725f 7738 7e46 7f2a 8b00 8c0b 936a 96fa 97aa 9eb0',
 '1Song': '462e 4ba3 4e27 55aa 55d3 6851',
 '1Tuk': '46e2 79bf 79c3 9d5a',
 '4Maa': '3984 4cf8 5af2 75f2 8534 87c6 9ebb',
 '1Keoi': '380a 38d8 3b55 3e16 49e2 4b83 4f49 547f 5c96 5d87 62d8 75c0 795b 80e0 88aa 8dd4 8eaf 8ec0 9639 99d2 9a45 9a71 9a79 9b88',
 '4Mai': '46e7 4a2a 7787 86d1 8b0e 8c1c 8ff7 919a 9e9b',
 '4Man': '35c4 37b6 37e9 37ed 4055 40a5 40c9 417c 42cb 43bd 46f2 47e8 4851 4ab8 4bf6 4c0b 4c1a 5cb7 5fde 6587 65fb 65fc 6c11 6c76 739f 73c9 753f 7b22 7d0b 7eb9 7f60 805e 82a0 82e0 95a9 95bf 95fb 95fd 960c 96ef 9b70 9f06 9f24',
 '3Zyut': '3441 4090 41af 4abc 4ace 4b95 555c 60d9 62d9 6387 656a 68b2 68c1 68f3 6b60 7421 7577 7a8b 7d55 7ecc 7f6c 814f 8301 855d 88f0 8ac1 8f1f 8f8d 918a 9323',
 '5Nei': '42fb 43d5 4f60 60a8 65ce 67c5 6ab7',
 '3Bei': '3a38 3cea 3f8c 3feb 4023 427e 43b5 45d2 4a40 4a90 4ba1 5e87 67f2 6bd6 75f9 75fa 7955 79d8 81c2 8298 8cbb 8f61 8f94 90b2 912a 924d 959f 99dc',
 '3Cing': '4064 6385 9758',
 '6Zang': '4c1d 7511 8d08 8d60',
 '5Liu': '374b 4126 4460 4e86 61ad 66b8 84fc 911d 91d5 948c',
 '4Jung': '342f 509b 50ad 5581 5889 5bb9 5d71 5eb8 6175 620e 6995 6be7 6eb6 6efd 6fc3 701c 7194 72e8 7462 7d68 7ed2 7fa2 809c 8319 8338 84c9 878d 878e 9118 9394 93de 9555 955b 9852 9899 9c05 9c2b 9ddb',
 '3Fut': '49a2 6ff6 86de 95ca 9614',
 '6Hung': '3db9 6c5e 95a7 9b28',
 '3Fun': '4028 559a',
 '3Fui': '4095 5599 6094 6666 75d7 7623 8aa8 8bf2 9767 986a',
 '5Lim': '3a98 3bec 3c08 3c68 41dc 431e 6b5b 7fb7 8138 81c9 895c',
 '2Caan': '3bc6 3e4c 3e7d 40a7 4abe 4e33 4ea7 522c 5257 5277 6d50 6efb 7522 93df 94f2',
 '4Koeng': '3a56 5d79 5f3a',
 '2Caai': '8e29 8e39',
 '1Ji': '4f0a 4f9d 533b 541a 54bf 566b 5979 5adb 6905 6b39 6bc9 6f2a 7317 747f 794e 7995 7e44 86dc 8863 90fc 91ab 92a5 94f1 9dd6 9e65 9edf 9ef3',
 '1Jo': '54df 5537 55b2',
 '3Syun': '352f 4509 7b6d 7b97 849c',
 '6Gik': '3783 395b 3b72 4a6f 6781 6975',
 '3Syut': '385c 395c 4a2e 8aac 8bf4 96ea 9c48 9cd5',
 '4Kam': '354b 3a81 4537 4945 4946 5659 6366 64d2 6a8e 7434 79bd 82a9 8804 9772',
 '4Kan': '3626 52e4 616c 61c3 61c4 65b3 82b9',
 '4Kau': '36cf 37c8 39e8 3b5d 42b5 47f5 48c7 491b 4fc5 53f4 5def 5df0 6882 6bec 6c42 72b0 7403 7486 7c06 7d7f 827d 866c 866f 86f7 88d8 89d3 89e9 8a04 8cd5 8d47 9011 91da 92b6 9f3d',
 '5Jyun': '6ceb 74c0 791d 800e 85b3 8edf 8f2d 8f32 8f6f 8fdc 9060 9249 94c9',
 '6Maa': '3437 347b 3b04 49de 508c 7943 79a1 7f75 99e1 9a82',
 '3Giu': '3c3e 4197 53eb 544c 5602 566d 5b13',
 '3Git': '34d7 36c3 3bae 495b 4a64 62ee 64b7 64f7 6d01 6f54 7aed 7d50 7d5c 7ed3 88ba 896d 9821 9889 9b9a 9c92',
 '6Mak': '3df5 407c 407f 4611 4bae 5192 58a8 7666 773d 7e86 8108 8109 84e6 8847 8899 88b9 899b 8c8a 8c98 964c 9721 9722 9a40 9ea6 9ed8 9ed9',
 '6Hau': '4780 47a7 5019 540e 5820 5f8c 7f3f 8329 9005 90c8 9c5f 9c8e',
 '6Hat': '3520 3b9d 52be 6433 6838 6a84 778e 7909 7c7a 821d 8988 89a1 89cb 8f44 8f96 95a1 9602 9ea7 9ee0 9f55 9f81',
 '6Man': '3662 3848 38ce 47c2 514d 554f 74ba 95ee',
 '6Hai': '3591 3670 3c21 3c27 3f8f 4fc2 6a95 76fb 798a 7cfb 7e6b',
 '6Zek': '84c6',
 '3Gin': '3a34 41df 4d56 4d5b 5efa 6bfd 8171 898b 89c1',
 '6Mat': '3ad8 3cf4 3d35 3d65 3e13 430f 436a 46d1 4909 52ff 5627 5bc6 6c95 6ef5 7269 8524 871c 87d4 889c 8b10 8c27',
 '6Mau': '34b5 37aa 4024 44ee 61cb 6959 776f 7780 8302 88a4 8b2c 8c2c 8cbf 8d38 912e',
 '3Gik': '8871',
 '6Han': '41ad 6068 7d4e 7ed7',
 '6Haa': '53a6 590f 5ec8 6687 8290',
 '1Zing': '3855 3de5 3fc2 4050 41f0 4292 4294 42ca 42dd 44b1 4544 48d0 49ab 49f9 4a09 4bda 4d16 4fa6 5075 5f81 6014 65cd 6676 6862 6968 6d48 6e5e 70dd 75c7 7665 7710 775b 796f 798e 7b90 84b8 8c9e 8d1e 9049 9266 94b2 9d84',
 '4Waang': '3d87 472b 484f 4b16 4b1d 6a2a 6a6b',
 '4Ou': '93d5',
 '6Suk': '3494 3bee 3ce4 3eff 40de 4671 4d30 587e 59af 5b70 5c5e 5c6c 719f 83fd 8700 8d16 8d4e 9483 9e00',
 '1Pei': '353b 3949 3a22 3a7a 3c5f 3da8 4876 4ae0 4b92 4e15 4f3e 5478 62ab 72c9 7812 79e0 7d15 7eb0 7fcd 9239 94cd 99d3 9aec 9b7e',
 '1Pek': '567c',
 '4Jing': '4ecd 4f80 51dd 5211 578b 584b 5a19 5b34 5db8 5f62 6254 6979 6ece 6ee2 6f46 6ff4 7005 701b 7020 702f 7192 71df 7469 76c8 784e 793d 7c6f 7e08 827f 8314 8365 8367 83b9 8424 8425 8426 84e5 8747 87a2 8805 8b0d 8fce 90a2 9276 93a3 94cf 9658 967e',
 '1Joeng': '592e 59ce 67cd 6b83 6cf1 79e7 9260 9785 9d26 9e2f',
 '4Ziu': '3478 347e 3a31 4058 4199 430a 4343 4504 45b4 46bb 48a3 4b19',
 '4Wing': '3607 5d58 69ae 8363 877e 8811 919f',
 '6Gung': '34cb',
 '2Caau': '3da4 3dc5 5435 7092 8016 8a2c',
 '3Saang': '38d2',
 '3Dyun': '65b7 7145 78ab 8176 935b 935c 953b',
 '4Zim': '3caf 3fd5 432a 49b2 4ac7 4db2',
 '4Zin': '35f4 360b 3723 3849 3b79 3cc4 3ced 3d6a 3e10 3ffc 41fe 44c2 4584 45fa 4768 4cff',
 '3Kaa': '3909 3c24 4bca 9ac2',
 '6Dou': '42ad 4326 4b30 55a5 5bfc 5c0e 60bc 675c 6e21 7118 71fe 76d7 76dc 79fa 7a0c 7a3b 7e9b 7fff 8254 828f 8e48 934d 9540',
 '2Aa': '538a 55c4 75d6 7602 963f',
 '2An': '371d',
 '2Am': '5535 57ef 63a9 667b 95c7 9eed 9eef',
 '2Ai': '8eb7',
 '2Au': '3443 5455 657a 6bb4 6bc6',
 '3Kau': '4f5d 53e9 5bc7 6010 6263 6ef1 7b58 81fc 8205 853b 91e6 9dc7',
 '6Dok': '386f 526b 5ea6 8957 8e31 9438 94ce',
 '6Doi': '3ed6 44ab 4ee3 506b 5cb1 5e12 5f85 6020 6b86 73b3 7447 7d3f 7ed0 8773 888b 8fe8 9168 9746 9edb',
 '4Seng': '57ce',
 '1Deoi': '34e7 354d 3fbc 4003 429a 43e8 4703 4b54 5806 6566',
 '6Hou': '3671 37bb 3b36 3d46 475e 49da 4abd 4beb 53f7 660a 6626 6dcf 6ec8 704f 705d 769c 769e 8583 9550 9865 98a2 9c1d',
 '1Deon': '38af 396b 3a9f 40e6 453b 4a83 5428 5678 60c7 8733 8ac4 9566',
 '6Jan': '4ede 4eed 5203 57bd 5b55 5c7b 6196 7263 730c 7d09 7eab 8095 80e4 8a12 8a8d 8bb1 8ed4 8f6b 9173 97cc 97e7',
 '3Maak': '64d8',
 '4Kong': '42af 72c2 9d5f',
 '3Maat': '62b9',
 '6Hon': '3488 37e7 38a8 3a8b 3ca6 4054 43f7 625e 6d86 701a 710a 7745 7ff0 8792 8c7b 91ec 92b2 9588 95ec',
 '6Hok': '367e 3c12 3d85 3ff1 4ba4 4cfd 5b66 5b78 6588 6fa9 77d0 7fef 89f7 8c89 9db4 9dfd 9e34 9e64',
 '6Hoi': '3925 39e1 41cb 4ea5 5bb3 6c26',
 '4Naa': '345a 39f1 3d4a 43e7 46d4 46d5 55f1 5a1c 62cf 62ff 8a49',
 '6Jai': '52e9 62b4 62fd 678d 67bb 6cc4 6d29 8a4d 8dc7 8de9',
 '2Deng': '9802 9876',
 '4Nai': '4372 576d 62b3 6ce5 81e1',
 '4Nam': '3f56 46c1 713e 7141',
 '4Kwong': '3fa0',
 '4Nan': '457c',
 '4Nau': '3471 34f7 3e80 4166 4d03',
 '2Jan': '5ef4 5fcd 6a83 6fe6 766e 8375 8b94 9690 96b1',
 '1Zit': '64f3 6809 6adb 7004',
 '5Cyun': '8b50 9c52 9cdf',
 '3Tong': '4171 645a 70eb 71a8 71d9 8d9f',
 '3Gwat': '3479',
 '6Saai': '3ccf',
 '4Pin': '36f9 3f10 4b81 6944 6969 7df6 7f0f 8ade 8c1d 8df0 8e41 99e2 9a88 9abf',
 '6Saap': '49b9 70a0',
 '6Saat': '4da1',
 '4Piu': '3bf1 456f 4d29 5ad6 6734 74e2',
 '2Caam': '39a7 3ab8 5646 60e8 6158 61af 6701 7bf8 9eea 9ef2',
 '3Gwan': '4d6a 68cd 7754',
 '5Lin': '6459 64b5 6506 740f 8f26 8f87',
 '3Gwai': '3467 39d4 3a88 3cf5 477f 4788 523f 528c 5b63 5da1 60b8 6842 6e8e 75f5 7678 7786 77b6 8cb4 8d35 8e76 994b 9bda 9c56 9cdc',
 '6Ng': '3576 43b8 4fc9 5be4 5fe2 609e 609f 6664 7110 8601 8aa4 8bef 8fd5 907b',
 '3Oi': '3905 5ad2 5b21 66a7 7477 8586 9440',
 '3Ok': '57a9 580a',
 '1Liu': '6482 64a9 8e58',
 '3On': '4141 4b36 6309 6441',
 '1Lik': '40ef 47cf 64fd 67a5 680e 6adf 6aea 701d 720d 74c5 792b 8dde 8e92 8f62 8f79 96f3 9742',
 '3Ou': '3729 3fae 443f 4712 58ba 5965 5967 5c99 61ca',
 '1Lim': '4ee4',
 '1Gang': '3e74 3e79 489a 5e9a 79d4 7a09 7cb3 7dea 7fae 7fb9 8ce1 8d53 9d8a 9e52',
 '4Faau': '4356 4cd5',
 '2Jyun': '4e38 5007 53bf 5917 5a49 60cb 667c 70f7 742c 7579 7755 7e23 82d1 83c0 8e20 962e 9662',
 '1Ping': '3519 36c1 436c 4995 4f3b 4fdc 59d8 5a09 6026 62da 62fc 7539 7830 9829',
 '4Kwai': '3512 3a57 3a66 3f6f 456b 47f8 4906 4ceb 5914 5db2 5dc2 6223 63c6 643a 651c 668c 72aa 7566 772d 8067 8475 8835 89ff 9035 9145 9474 96df 9804 982f 9997 9a24 9a68 9a99',
 '3Aau': '3eab 4302 4314 977f',
 '3Aat': '538b 5688 5828 58d3 605d 621b 6b79 7312 904f 981e 9f43',
 '4Kwan': '3a8a 3fcf 5bad 7fa3 7fa4 88d9',
 '3Aap': '62bc 9d28 9e2d',
 '3Aan': '664f 9d33 9dc3',
 '3Aam': '3fc4 43a8 4b13',
 '3Aai': '3660 3792 3977 3d67 55cc 9698 9932',
 '3Tung': '75db 9c96',
 '6Soeng': '5c1a',
 '2Mui': '71d8',
 '3Gwaat': '34c9 43e6 4bcf 4d1c 522e 5280 6356 98b3',
 '1Ngou': '55f7 64d9 7352',
 '1Mei': '5c58 7778 875e',
 '3Gwaan': '4e31 60ef 6163 63bc 645c 64d0 8e80',
 '3Gwaak': '63b4 6451',
 '3Gwaai': '380f 3ed1 40af 42bd 4896 4ccf 602a 6060',
 '6Guk': '3a74 42f0 43a4 4fb7 5c40 6336 68ae 7105 7117 8dfc',
 '6Gui': '4829 6530 7650',
 '4Teoi': '3888 3891 3d84 3fc9 3fd7 416a 42d8 4acb 50d3 58a4 7a68 85ec 96a4 9839 9893 9b4b',
 '1Leon': '5652',
 '1Maau': '732b 8c93',
 '3Pak': '3711 4385 438a 4a96 4bd9',
 '3Pan': '5460 55b7',
 '3Sau': '517d 5bbf 6f31 6f43 7378 7407 7626 79c0 7d89 7e61 7ee3 924e 92b9 93fd 9508',
 '3Paa': '5e0a 5e15 6015',
 '1Cing': '36f5 3d59 455d 4793 4a1d 5041 570a 67fd 6a89 6c30 79e4 79f0 8100 83c1 86cf 873b 87f6 8d6a 8d6c 9bd6 9cad 9f31',
 '6Nei': '36c5 38bd 3990 3f0f 43b6 43ea 48f5 6be6 7732 817b 990c 9975',
 '4Ging': '749a',
 '1Maan': '6471',
 '3Pau': '3be1 5a44',
 '5Laan': '39a8 5b3e 61d2 61f6',
 '1Cung': '37b1 3894 391d 3956 3b94 3cd8 3cde 408c 4194 41b9 440b 44d7 45d3 462a 4751 486f 48bc 5145 51b2 5306 57c7 5fe1 6031 60a4 6183 61a7 6450 679e 6a05 6c96 6d7a 6d8c 73eb 747d 7481 7f7f 7fc0 8066 806a 8070 82c1 833a 8471 84ef 8525 885d 8877 93e6 9a18 9a44 9aa2',
 '4Nung': '3db6 3e9c 4078 4889 4fac 5102 519c 54dd 5665 6d53 79fe 7a60 8113 81bf 895b 8b68 8fb2 91b2',
 '4Ho': '4f55 6cb3 8377 86b5',
 '2Loeng': '5521 5562 8f1b 8f86',
 '2Leon': '5375',
 '1Gap': '3902 3f7a 5fe3 6025 82a8',
 '1Gau': '3619 3797 49b0 4c17 4ca5 52fc 53e5 644e 673b 6a1b 6c9f 6cc3 6e9d 7bdd 7c3c 7df1 82b6 88a7 920e 94a9 9604 97dd 9b2e 9ce9 9e20 9f9c 9f9f',
 '1Gat': '34e4 4c34 4f76 5403 5409 54ad 59de 7d07 7ea5 8090 867c 86e3 8a16 8a70 8bab 8bd8',
 '4Po': '5a46 5abb 76a4 84b1',
 '1Gaa': '3699 39dd 3e62 409f 4214 4552 4d25 4f3d 50a2 52a0 5496 5609 560e 5676 5bb6 5c15 67b7 6935 6cc7 728c 73c8 75c2 767f 7b33 801e 8304 846d 8888 8c6d 8dcf 8fe6 93b5 9553 9d10',
 '1Ngan': '5940',
 '1Gai': '36f7 3bb7 4861 6785 7b04 862e 96de 9dc4 9e21',
 '1Gan': '4225 423d 44f8 461c 493a 4cac 54cf 5dfe 65a4 6839 784d 7b4b 8270 89d4 8ddf 91ff',
 '1Gam': '361a 41de 431d 46d3 4eca 67d1 6cd4 7518 75b3 7981 7d1f 82f7 887f 91d1',
 '1Duk': '53fe 7763 7b03 7be4 88fb',
 '3Cyut': '3b6e 47df 4bff 4c63 5352 5484 637d 64ae 690a 731d 7e53 8e24',
 '1Huk': '3ba3 3c84 52d6 52d7 54ed 702b',
 '4Haang': '38dc 6841 884c',
 '6Zung': '356c 37b2 3db2 46e6 4ef2 8a1f 8aa6 8bbc 8bf5 980c 9882',
 '1Jap': '4fcb 5508 6092 6339 63d6 6ce3 6d65 6e46 6e47 6f5d 71a0 71a4 7fd5 88db 9091',
 '1Jat': '4e00 4e28 58f9 5f0c',
 '1Jau': '3400 4e18 4f11 4f18 512a 5466 5698 5775 5e7d 5ea5 5fe7 602e 6182 61ee 6acc 7e8b 8030 86af 8c85 90b1 913e 9af9 9d42 9e3a 9e80',
 '1Jan': '56e0 5794 5819 59fb 5ffb 6069 6147 6615 6b23 6b45 6bb7 6c24 6d07 6e6e 7098 712e 7504 76fa 798b 7d6a 8335 88c0 8a22 8af2 92a6 94df 9528 95c9 967b 99f0 9a83 9eeb',
 '1Jam': '5465 5591 5d5a 5d94 5ede 6114 6b3d 6b46 7616 946b 94a6 9634 9670 9682 9712 9720 97f3',
 '3Cyun': '3e11 434e 4e32 540b 5bf8 60f4 64ba 651b 7228 7394 7a9c 7e13 948f',
 '4Leoi': '3523 35ca 3d62 3e4e 4363 4a13 4b89 4bc1 513d 5ad8 60c0 64c2 6502 6988 6a91 6ad0 6ad1 6b19 6c00 757e 7d2f 7d6b 7e32 7e8d 7f27 7f4d 7fb8 8502 857e 85d8 8632 8646 8f60 9433 956d 95ad 95fe 96f7 9a62 9a74',
 '6Bei': '3517 3629 38b0 3c06 3e22 485f 4ac1 4ffb 5099 5907 5970 7b13 7b84 7cd2 88ab 8963 907f 9781 9ab3 9af2 9f3b',
 '3Jyu': '68dc 9157 91a7 98eb 9947 996b',
 '2Baa': '378e 628a 9776',
 '1Waang': '8eed',
 '2Bai': '35d7 4653',
 '2Ban': '3e1e 427d 4b33 54c1 7980 7a1f',
 '5Ciu': '9657',
 '4Seoi': '4ce0 5015 5782 813d 8ab0 8c01 9672',
 '2Sang': '3f33 4687',
 '1Toi': '80ce 82d4',
 '5Cin': '38e4 49d6 4ff4 8df5 8e10 9a4f 9aa3',
 '5Le': '8926',
 '4Cung': '35f0 3ba4 3e50 3f3b 4258 432c 457a 459d 4cef 4cf7 4e1b 4ece 53e2 5f9e 60b0 677e 6dd9 6f0e 6f40 721e 742e 79cd 85c2 866b 87f2 8ce8 91cd',
 '1Naai': '5976',
 '5Zaau': '353d 3557 355b 3b8b 44b4 454d 46fb',
 '4Ze': '44c9 497a',
 '4Zi': '3587 3608 370b 3756 3816 3974 39be 39eb 3c18 3ebf 42e9 4405 44f4 4785 49c5 4a5f 4b2e 4b8a 4b98 4c4c',
 '1Guk': '3632 394c 39bd 3b82 3bcf 3e7c 3f64 4030 43f1 4428 455f 456e 45c7 472f 4a95 4c61 4cd4 4d48 4d9c 530a 5502 5503 55be 56b3 63ac 688f 6908 6996 6bc2 6ff2 7014 727f 7a40 7cd3 83ca 861c 8aca 8c37 8d9c 8e18 8ed7 8f02 8f42 92ca 92e6 9514 97a0 97ab 980a 987c 9d60 9e44 9eaf 9eb4',
 '2Siu': '4495 449a 5c0f 5c11 7b71 7be0',
 '1King': '503e 50be',
 '2Sin': '39e5 3b60 3d8d 51bc 59fa 5c1f 5eef 6be8 71f9 72dd 736e 7663 766c 7b45 85d3 861a 8de3 9291 94e3',
 '2Sim': '3692 3a1b 3a8e 3d38 3d92 483e 6671 7154 7752 9583 95ea 9655 965d',
 '3Zyu': '3ac2 3e65 3e9b 439d 43b7 48f1 4a1e 4a92 4b21 4b56 6bb6 6ce8 70b7 75b0 7bb8 7d38 7f5c 7fe5 86c0 8a3b 9252 9444 94f8 99b5 99d0 9a7b',
 '2Sung': '39d0 3a26 3a73 50b1 5d77 6002 609a 616b 7ae6 8038 8073 99f7',
 '3Jyut': '4b91 4e59 7207 91d4 9ce6',
 '3Jyun': '6028 60cc',
 '6Hap': '36cd 3932 397a 3a89 3c30 3c3c 4894 69bc 6b31 6e98 70da 76cd 778c 78d5 927f 94ea 95d4 9616',
 '3Lang': '79dd',
 '2Song': '3f3d 452a 4866 4ad9 587d 6421 723d 78c9 9859 98a1',
 '2Zoi': '3c70 43c1 4502 4ba8 5bb0 5d3d 7e21 8f09 8f7d',
 '2Beng': '3778 9905',
 '3Gip': '3cb9 4037 40b6 5226 5227 52ab 62be 6f80 7864 9d56',
 '2Zou': '4503 45e2 4c83 65e9 67a3 68d7 7475 74aa 7956 7d44 7ec4 85fb 86a4 977b',
 '4Kung': '344b 39ed 3baa 4143 4284 44d6 46ea 483b 5311 7a77 7aae 7ac6 7b47 85ed 86e9 8deb 909b 928e',
 '3Gim': '4be1 5251 528d',
 '4Jeoi': '4f6f 7dcc 8564',
 '5Mang': '42f2',
 '5Zi': '3d6b 433a 4c2f 96c9',
 '4Meng': '540d',
 '5Ze': '3652 8fd9',
 '1Baak': '3933 5317 8feb',
 '1Baan': '3866 3979 3fc0 40d1 4399 4694 653d 6591 6592 73ed 7622 878c 9812 9881',
 '6Loek': '347c 35c9 3a3c 40ae 430e 63a0 7565 7567',
 '1Baat': '3aa4 41f7',
 '1Baau': '52f9 5305 5b62 70ae 80de 82de 9b91 9f59 9f85',
 '1Gwik': '3d81 3de6 435e 4997 49cd 49d0 4b18 4b99 4c5b 4d17 4f90 6b88 6d2b 6de2 6e68 72ca 7d8c 7ee4 805d 8662 8669 89a4 90e4 90f9 95c3 95c5 9612 9699 9998 9d59 9daa',
 '6Ham': '61be 64bc 9849',
 '3Ngok': '5641 6076 60e1',
 '4Ming': '3760 3ae5 4119 41a8 41a9 4cdf 51a5 5ac7 660e 669d 69a0 6d3a 6e9f 7733 7791 8317 84c2 879f 910d 9169 9298 94ed 9cf4 9e23',
 '1Hyun': '34bd 39a5 3b09 3ef9 432f 4ce6 5107 54ba 55a7 57d9 5864 58ce 5b1b 5f2e 6103 6645 6684 68ec 6e32 70dc 7156 79a2 7ffe 8431 8432 8809 8ae0 8afc 8c16 92d7',
 '5Mong': '34c1 37d0 37ff 3b12 4073 43f5 448e 44bd 5984 60d8 6f2d 7db2 7f51 7f54 833b 83bd 83f5 87d2 8f1e 8f8b 9b4d',
 '5Kiu': '4ade 769b',
 '5Wing': '6c38',
 '5Laang': '51b7',
 '3Paan': '466a 76fc 88a2 897b',
 '2Ng': '55ef',
 '4Kyun': '444f 4830 5dcf 60d3 62f3 6743 6b0a 8737 8838 8e21 9874 98a7 9b08 9f64',
 '3Long': '667e',
 '2Ding': '3ac0 3d3f 4d7a 6fce 827c 85a1 9f0e',
 '2Cim': '8ac2 8b87 8c04',
 '3Dit': '8dcc',
 '3Diu': '540a 5f14 76c4 7a8e 7ab5 8a82 91e3 9493',
 '1Wan': '3683 3b08 460e 6b9f 6c32 6c33 6e29 6eab 7185 761f 7e15 8570 8580 891e 8c71 8d07 8d5f 8f40 8f92',
 '3Din': '6bbf',
 '2Coeng': '62a2 78e2',
 '3Dim': '34e0 376a 3f2d 576b 588a 5e97 60e6 6242 73b7 75c1 78f9 8e2e 963d',
 '5Mung': '511a 61de 8813',
 '6Naau': '3f36 4176 4c30 6dd6 9599 95f9 9b27',
 '6Naat': '637a',
 '6Naap': '5167 5436 5450 7d0d 7eb3 8872 8edc 9209 94a0 9b76',
 '3Haang': '4c22',
 '2Nung': '4d5c',
 '1Mong': '68e2 9794',
 '3Po': '7834',
 '5Wan': '414b 41ac 41d6 491e 4a36 4ab3 5141 5c39 628e 6b92 6b9e 72c1 837a 9668 9723 97fb',
 '1Bat': '3cfc 3ef6 47c6 4e0d 51b9 5425 5487 54d4 55f6 5f43 6bd5 6ed7 6eed 6f77 7562 7b14 7b46 7b5a 7bf3 7e2a 835c 84fd 8df8 8e55 97e0',
 '2Zoeng': '41e6 4243 4482 4ec9 5956 596c 638c 6868 69f3 6da8 6f32 734e 7ef1 848b 8523',
 '5Haan': '50e9',
 '2Tou': '3fd2 46af 4bbb 5410 571f 7977 8a0e 8ba8 91f7 948d',
 '3Naai': '7593',
 '5Waa': '8e1d',
 '3Naam': '63c7',
 '1Baa': '3cd2 3e2d 3fec 4bf2 4c3e 4d95 53ed 5427 5cc7 5df4 7238 75a4 7b06 7c91 7f93 82ad 8469 8c5d 9200 94af 9c83',
 '3Naat': '712b',
 '1Bak': '479f',
 '1Bai': '39f3 8ddb 939e',
 '1Ban': '37d7 3bfd 3e8d 3ede 3f43 4a08 4c09 4efd 5110 5954 5bbe 5f6c 658c 68b9 69df 6ab3 6c43 6ee8 6fd2 6ff1 6ff5 7015 7287 73a2 74b8 7e7d 7f24 8819 8c73 8cc1 8cd3 8cd4 8d32 90a0 931b 944c 951b 9554',
 '1Bam': '4e53 6cf5',
 '3Am': '6697 8164',
 '3Beng': '67c4 7a89',
 '3Ngong': '4425',
 '1Mang': '345f 4ae4 64dd',
 '1Saa': '383a 3aaa 3c9a 3e3a 4a56 5345 5526 5565 6c99 7300 75e7 7802 7d17 7eb1 838e 88df 9024 9b66 9bca 9ca8',
 '6Neoi': '4b61',
 '1San': '36d9 3c11 4bc2 4f38 4f81 515f 547b 5a20 5c7e 65b0 6c20 71ca 73c5 7521 7533 7837 795e 7d33 7ec5 8398 85aa 8a75 8bdc 8f9b 92c5 950c 99ea',
 '1Sam': '38fa 3d31 415f 41a6 4d47 5fc3 68ee 69ee 6df1 741b 7d9d 82af 8460 8942 8cdd 90f4',
 '1Sak': '3976 63cc',
 '6Sek': '3c64 4096 4137 45a8 4cbd 7855 78a9 794f 9f2b',
 '1Sai': '34fe 38b4 3bd5 3fb7 5636 6053 6a28 7280 7852 7b5b 7be9 7c01 7c9e 823e 831c 897f',
 '4Leng': '9748 9bbb 9bea',
 '1Sau': '39c3 3bb4 423e 4479 4520 462f 4907 4b9f 4fee 53ce 55d6 5ecb 641c 6536 6eb2 6eeb 7340 7cd4 7f9e 8129 8258 8490 84e8 910b 93aa 953c 98bc 98d5 993f 9948 998a 9990 9a2a',
 '1Sat': '3ece 3eed 48db 5931 5ba4 745f 74b1 819d 8671 8768',
 '4Ting': '3e76 3f17 4c53 4ead 505c 5a77 5ead 5ef7 695f 6e1f 839b 8476 8713 9706 9f2e',
 '1Sap': '5482 6da9 6e7f 6ebc 6f81 6fd5 8b45',
 '6Ngou': '371c 397f 4ae8 4baf 50b2 5961 9dd4',
 '3Waan': '404c',
 '3Waai': '3c25 4d73',
 '3Waat': '3c67 4011 404a 7a75',
 '6Ngon': '3e29 5cb8',
 '3Dung': '5d20 680b 68df 6e69',
 '6Ngok': '34f5 35be 35c1 380b 398d 3a75 3b99 3c1b 3e4a 48de 4af7 54a2 5669 582e 5cb3 5d3f 5dbd 6115 6aee 816d 843c 8ae4 8c14 9102 9354 9537 984e 989a 9c10 9c77 9cc4 9d9a 9e11 9e57 9f76',
 '6Ngoi': '354c 5916 788d 7919',
 '6Lap': '426d 491a 4c9e 641a 9574 9d17',
 '1Wo': '3f7f 57da 581d 5a32 5aa7 6fc4 7171 7327 7a9d 7aa9 83b4 8435 8e12 934b 9505',
 '6Wut': '4106 4111 6d3b',
 '6Wui': '37a7 3ec5 532f 6c47 6ed9 70e9 71f4',
 '6Wun': '38ea 3b07 3b0a 4013 4220 4809 4bd8 5524 5942 5950 5fe8 6362 63db 6da3 6e19 7115 7165 75ea 7613 7f13 7feb 8f10 902d',
 '3Cou': '3ca7 3ff7 450f 539d 566a 6165 63aa 6c09 6fa1 71e5 77c2 7cd9 8b5f 8dae 8e81 9020 9135 9162 918b',
 '1Pui': '36b0 3cc8 574f 576f 67f8 80a7 80da 8843 9185',
 '1Pun': '3896 41ba 6f58',
 '6Waat': '41b7 4389 4847 4afb 4b07 59e1 733e 8796',
 '3Teon': '5f56 8916',
 '1Bit': '34d6 39d9 3bbf 3eeb 426c 42d4 45a9 47e4 4a5b 4b1b 4f56 5fc5 602d 73cc 8995 89f1 94cb 9b85',
 '6Waan': '3e56 3f02 4159 41a0 449b 5ba6 5e7b 60a3 650c 6f36 8c62 8f58',
 '1Sou': '389d 3970 3e98 4c05 4c86 56cc 6414 7526 7a23 7a4c 7e45 7f2b 81ca 82cf 8607 9165 98be 9a37 9a9a 9b1a',
 '6Waai': '3707 3b29 4927 4b11 4d7b 58de 863e',
 '6Waak': '34f0 35f2 398e 39af 3a47 4b09 4c25 5283 5a73 5aff 60d1 6216 6e71 7809 7e63 8b0b 97c4 9a1e',
 '4Hou': '3819 3a5d 3b14 4765 49eb 55e5 568e 58d5 6beb 6f94 6fe0 865f 8814 8c6a',
 '2Baan': '5742 6604 677f 723f 7248 8228 8742 9211 94a3 95c6 962a 9b6c',
 '1Pin': '3fab 504f 5aa5 7bc7 7de8 7f16 7fe9',
 '4Sam': '417e 4c8b 5c91 5ff1 6116 68a3 6d94 8a26 8af6 8c0c',
 '4San': '3af3 41a3 4885 48bb 48c5 5bb8 6668 686d 7973 8124 81e3 8703 8fb0 9dd0',
 '1Pik': '4440 4d19 50fb 64d7 6fbc 7513 7656 91fd 95e2 9739 9dff',
 '6Mut': '3dac 43de 4d32 59ba 672b 6b7f 6b81 6c92 6ca1 700e 8309 977a',
 '2Baau': '4b4b 98fd 9971',
 '4Hon': '3654 45d9 4ba7 5bd2 6c57 72b4 751d 9097 97d3 97e9 99af 9dbe 9f3e',
 '4Hoi': '9826 988f',
 '4Sau': '3d5e 6101',
 '1Zeng': '7cbe',
 '5Zung': '39b7 47bb',
 '1Ngon': '5b89 80fa 92a8',
 '3Gou': '544a 8aa5 8bf0 90dc 92ef 9506',
 '3Got': '34e3 4213 5272 5db1 845b 8f35 8f55',
 '3Gok': '3837 4077 41f6 4aba 4ae6 5095 57c6 6354 6401 64f1 6877 73a8 784c 88bc 89d2 90dd 927b 94ec 95a3 9601',
 '3Goi': '76d6 8462',
 '3Gon': '4bce 5e79 65f0 69a6 7c33 9aad',
 '4Mun': '3eca 429f 42a1 443d 4ad2 4eec 5011 6097 626a 636b 6a20 748a 7792 779e 7a48 864b 8e52 8e63 9346 9494 9580 95e8 989f',
 '4Mui': '3eb3 4288 4359 5a92 679a 6885 7164 73ab 7996 8122 815c 82fa 8393 9176 92c2 9709',
 '3Goek': '45bc 5c69 811a 8173 8c3b',
 '2Hei': '4048 4049 410e 412b 43b2 4507 559c 56cd 5c82 87e2 8c48 8d77',
 '3Kwaai': '5633',
 '1Saai': '3a44 4675 4aa0 5625 6f07',
 '1Gong': '34fb 3757 37e0 37f5 39cf 3b43 3b4e 3f1a 3f58 4309 4697 4d1a 4ea2 5188 521a 525b 5808 583d 5c97 5ca1 5d17 625b 6443 6760 68e1 6c5f 7285 74e8 7598 77fc 7899 7db1 7eb2 7f38 7f41 7f61 809b 8221 8333 8c47 91ed 92fc 94a2',
 '1Saan': '3457 3596 36db 36fd 372a 38b2 3c94 3e6a 3f95 4178 4200 4c20 5220 522a 59cd 59d7 5c71 62f4 633b 6813 6c19 6c25 6f78 73ca 81bb 8222 8dda 9582 95e9',
 '1Saam': '384e 3aa8 4010 43f9 4e09 4ee8 53c1 5e53 5f61 6bf5 6bff 7a47 829f 852a 886b 91e4 93d2 9490 9b16',
 '1Saap': '348a 4262 43c9',
 '1Saau': '3c99 3ca3 3e4b 422d 45db 486d 4939 4b12 4b49 5f30 634e 65d3 68a2 7b72 8244 83a6 86f8 9afe 9bb9',
 '5Ting': '414d 6883 73fd 8121 92cc 94e4 9832 988b',
 '4Lau': '342c 396a 3f5e 4267 46a7 4740 474f 48da 4aeb 4b1f 4d89 507b 50c2 5218 5289 56a0 5a04 5a41 5ed4 617a 61f0 65c8 65d2 697c 69b4 6a13 6a4a 6d41 6d4f 6f0a 6f3b 700f 7409 7460 74a2 7559 7571 7624 7645 779c 786b 8027 802c 825b 848c 851e 877c 87bb 87c9 8e53 905b 938f 93a6 93d0 954f 9560 97bb 98c0 98c2 98d7 993e 9a2e 9a9d 9ac5 9acf 9c21 9db9 9e60',
 '4Toi': '3497 3606 3675 3b03 3dd8 421a 4453 5113 62ac 64e1 67b1 6aaf 70b1 7c49 81fa 85b9 8dc6 90b0 9236 98b1 99d8 9a80 9b90 9c90',
 '4Cong': '3856 3d74 3d93 40e5 4874 4b5a 5e62 5e8a 649e 6a66 7240 85cf',
 '4Tou': '38e0 3b78 3ee0 3eef 4177 46cc 46ec 47ed 48c4 493e 4b1e 4cdc 530b 54b7 5555 56fe 5716 5857 5c60 5edc 5f92 6087 638f 642f 6843 6d82 6d9b 6dd8 6fe4 760f 7daf 7ef9 837c 8404 9003 9014 9174 9184 9780 9789 99e3 99fc 9a0a 9d4c 9dcb 9f17',
 '4Laa': '35ff 3648',
 '4Lam': '375d 3a2b 4182 4277 4e34 6797 6dcb 7433 75f3 81e8 9716',
 '1Sing': '365a 3bb1 3c61 3de3 3f6e 45cc 52dd 5347 544f 60fa 65cc 6607 7329 7446 965e 9a02 9a8d',
 '4Lai': '381f 3c00 3d1d 426b 4523 4d69 4f86 56c4 7281 7282 853e 85dc 9ece 9ee7',
 '2Oi': '35d2 4a20 50fe 55f3 566f 58d2 66d6 6bd0 77b9 853c 85f9 972d 9744 9749',
 '6Jam': '6041 7eb4 814d 8cc3 8d41 996a',
 '3Kok': '3929 3a41 3c4b 3c7f 4068 4142 4253 4d85 5859 58d1 606a 60ab 6118 6128 6164 6409 69b7 6db8 786e 78ba 78bb 7910',
 '3Koi': '39c9 43d7 4e10 5ffe 613e 6224 69ea 6e89 6f11 9223 9499',
 '4Co': '37c7 3b6b 4612 4d3e 5252 5ab0 75e4 77ec 8021 924f 92e4 9504 9624 9db5 9e7a 9e7e',
 '4Faai': '480a',
 '6Jaa': '5344 5eff',
 '4Faan': '3836 3dca 3e0b 3fce 42a9 42e3 42e6 46a9 4aa4 4b75 4bb3 51e1 51e2 58a6 5e06 68b5 6a0a 703f 70e6 7169 71d4 74a0 77fe 792c 7b32 7e41 7fb3 81b0 8543 85a0 85e9 8629 87e0 881c 8e6f 8ed3 8eec 91e9 9407 9492 98bf',
 '4Cing': '4107 41f8 477c 478d 5448 57d5 60c5 60e9 61f2 6210 6f82 6f84 73f5 77aa 7a0b 88ce 9172 9933 9967',
 '6Jau': '4f51 4f91 53c8 53f3 56ff 5ba5 72d6 7950 890e 91c9',
 '6Jat': '4f5a 4f7e 52ae 6396 65e5 6c68 6cc6 6db2 6ea2 80b8 814b 8257 8875 8f76 9038 9224 93b0 9552 99b9 9a72',
 '2Ou': '37c5 3907 4d08 4d20 5aaa 5abc 8884 8956 957a',
 '6Jap': '5165',
 '2Lai': '569f',
 '3Sap': '3f42 453c 4b03',
 '5Laau': '4179',
 '1Bong': '3643 3a0d 3ff6 48b6 4a77 5e2e 5e47 5e6b 6337 6412 6886 7e0d 90a6 97a4',
 '3Sat': '348e',
 '3Sai': '3b20 3b61 4d58 4e16 52bf 52e2 58fb 5a7f 7d30 7ec6 8cb0 8d33',
 '3Sak': '3bb6 4307 459b 462d',
 '5Laam': '3c16 4386 63fd 64e5 6984 6b16 6f24 7060 7226 89bd 89c8',
 '3Sam': '3d55 6c81 6e17 6ef2',
 '5Laai': '3f16',
 '5Tou': '3b6c 809a',
 '4Ce': '35bf 3697 4511 8f0b 90aa',
 '4Cang': '3a03 3b1d 406c 41b5 41b8 4255 425e 49dc 5c42 5c64 5d92 66fe 912b',
 '4Loi': '3693 4158 42f1 4685 49d2 5008 5d03 5d0d 5f95 5fa0 6765 68f6 6d9e 6df6 83b1 840a 90f2 9338 94fc 9a0b 9be0 9d86',
 '4Nou': '36a2 4f2e 5974 5b65 5e11 7b2f 99d1 9a7d',
 '1Faat': '3695 3a82 3a84 3cf7 3d60 401c 4296 4620 463f 479e 4a50 4a80',
 '6Zaak': '37d9 3b59 429e 456a 4bb0 5b85 62e9 63b7 6458 64c7 64e2 64f2 6aa1 6cfd 6fa4 70e2 78d4 880c 8b2b 8b81 8c2a 8e2f 8e91',
 '2Lou': '4f6c',
 '3Gwok': '3575 35aa 3a2f 3c0c 40b8 4190 4438 490b 4b0e 4b7d 5613 56ef 56fd 570b 588e 5d1e 5e3c 5e57 5ed3 64f4 6901 69e8 6f37 7c02 8195 8748 87c8 90ed 9729',
 '1Lak': '7afb',
 '1Faan': '35a7 45fc 4a9b 53cd 5e61 65db 7568 756a 7e59 7ffb 8f53 8fd4 9ded',
 '6Zoeng': '3a3e 3e8a 3f74 4586 47aa 4b65 4d02 4e08 50cf 5320 6756 6a61 87d3 8950 8c61',
 '1Faai': '3552 3e94 4617',
 '1Faak': '62c2 6cb8',
 '6Zaang': '40bb 4662 6323',
 '3Ai': '416c 5ad5 5c6d 66c0 7e0a 7f22 7ff3 8c77',
 '3An': '4781 4cdb',
 '6Lo': '3fda',
 '6Lam': '3bfa 7f67',
 '3Aa': '4149 489d 4cec 4e9a 4e9e 57e1 5a05 5a6d 631c 6397 6c29 6c2c 897e 930f 94d4',
 '6Lai': '34a7 3578 3611 3842 3921 3e0a 3f5d 3f90 3fdb 42aa 457b 4c9a 4d21 4e3d 4f8b 4fea 5137 52b1 52f5 5389 53b2 60b7 6369 6526 6835 68d9 6fff 75a0 7658 77d6 7805 783a 792a 7c9d 7cf2 8354 86ce 8823 8fe3 9e97',
 '6Lak': '3539 4ec2 53fb 5c74 6250 6738 6cd0 7c15 961e 9c33 9cd3',
 '6Lau': '3537 3640 3f4c 4102 5ec7 6f0f 7618 763a 763b 7fcf 816c 93e4 9542 964b 9724 9732 998f 9dda 9e68',
 '6Lat': '43c0',
 '5Waai': '3815',
 '5Waan': '49be 633d 8f13 9bc7 9ca9',
 '1Kwaan': '34ad 3853 3a48 3c4e 440a 4ab2',
 '6Nuk': '4610 46bc 4d8a 6712 8844',
 '3Au': '356d 42c1 6004',
 '3At': '3548 39a3 3ccc 3dc8 3f8e 4198 475f 4b35 4bc9 4be7 4c32',
 '6Fung': '385d 51e4 5949 752e 9cf3',
 '6Koeng': '8199',
 '1Fei': '3af5 4935 4a41 4b20 5983 5a53 6249 7dcb 7eef 83f2 871a 88f6 970f 975e 98db 98de 99a1 9a11 9be1',
 '3Zoeng': '3569 3663 3893 3bcd 4ed7 5d82 5e10 5e33 5e5b 7634 80c0 8139 8cec 8d26 9171 91ac 969c',
 '3Sou': '3481 3556 3a1e 3d11 3fcb 46fe 5083 55c9 57fd 5850 612b 612c 6cdd 6eaf 7619 7d20 8186 8a34 8bc9 9061',
 '2Lam': '3463 8ad7',
 '2Laa': '63e6 650b',
 '5Tam': '51fc 6c39',
 '1Bang': '3def 3f1e 4640 4656 4cde 580b 5874 5d29 5f38 75ed 794a 7d63 7db3 7ef7 927c',
 '3Kai': '5951 6112 6814 7608 761b',
 '4Jeng': '8d0f 8d62',
 '2Kung': '391f',
 '3Sok': '3ba6 40b9 4174 47bd 55cd 5851 6420 6578 6714 69ca 6b36 6eb9 84b4 938d 9399',
}

PAT_ALPHA = re.compile(r'\w', re.UNICODE)
def is_hanzi(n):
  return (0x3400 <= n and n <= 0x9fff) or (0xf900 <= n and n <= 0xfaff)

def pinyin(lines, decoder, encoder, table, html=False, tone=False):
  for rawline in lines:
    try:
      line = decoder(rawline)[0]
    except UnicodeError:
      print '%s:%d: cannot decode: %s' % (lines.filename(), lines.filelineno(), rawline)
      return 1
    prev = 0  # 1:alpha, 2:hanzi, 0:other
    s = ''
    for (i,c) in enumerate(line):
      if not PAT_ALPHA.match(c):
        s += c
        prev = 0
      else:
        n = ord(c)
        if is_hanzi(n):
          # hanzi
          if prev:
            s += ' '
          prev = 2
          if n not in table:
            s += c
          else:
            a = table[n]
            if html:
              s += u'%s<sup>%s%s</sup>' % (c, a[1:], a[0])
            elif tone:
              s += u'%s%s' % (a[1:], a[0])
            else:
              s += a[1:]
        else:
          # other alphabets
          if prev == 2:
            s += ' '
          prev = 1
          s += c
    if html:
      stdout.write(s.encode('ascii', 'xmlcharrefreplace'))
    else:
      stdout.write(encoder(s)[0])
  return 0

def main(argv):
  import getopt
  def usage():
    print 'usage: %s [-d] [-i input_encoding] [-C)antonese] [-T)one] [-H)tml] [file ...]' % argv[0]
    return 100
  try:
    (opts, args) = getopt.getopt(argv[1:], 'di:THC')
  except getopt.GetoptError:
    return usage()
  (debug, enc_in, html, tone, srcmap) = (0, 'utf-8', False, False, PINYIN_TABLE_MANDARIN_SRC)
  for (k, v) in opts:
    if k == '-d': debug += 1
    elif k == '-i': enc_in = v
    elif k == '-T': tone = True
    elif k == '-H': html = True
    elif k == '-C': srcmap = PINYIN_TABLE_CANTONESE_SRC
  table = {}
  for (a,u) in srcmap.iteritems():
    for n in u.split(' '):
      table[int(n,16)] = a
  decoder = codecs.getdecoder(enc_in)
  encoder = codecs.getencoder(enc_in)
  lines = fileinput.input(args)
  return pinyin(lines, decoder, encoder, table, html, tone)

#
if __name__ == '__main__': sys.exit(main(sys.argv))
