"""
 Copyright (c) 2022, salesforce.com, inc.
 All rights reserved.
 SPDX-License-Identifier: BSD-3-Clause
 For full license text, see the LICENSE file in the repo root or https://opensource.org/licenses/BSD-3-Clause
"""

import argparse
import json
import os
import random
import os.path as op

import numpy as np
import torch
import torch.backends.cudnn as cudnn
from torch.nn.functional import normalize

from peft import LoraConfig, get_peft_model
import torch.distributed as dist
from tqdm import tqdm

import lavis.tasks as tasks
from lavis.common.config import Config
from lavis.common.dist_utils import get_rank, init_distributed_mode, is_dist_avail_and_initialized, is_main_process, \
    get_world_size
from lavis.common.logger import setup_logger
from lavis.common.optims import (
    LinearWarmupCosineLRScheduler,
    LinearWarmupStepLRScheduler,
)
from lavis.common.registry import registry
from lavis.common.utils import now
from lavis.datasets.data_utils import prepare_sample
from lavis.models import load_model_and_preprocess

# imports modules for registration
from lavis.datasets.builders import *
from lavis.models import *
from lavis.models.blip2_models.blip2_rank_extraction import Blip2RankExtraction
from lavis.models.blip_models.blip_prompt_model import BlipPromptTuning
from lavis.processors import *
from lavis.runners import *
from lavis.tasks import *

from lavis.models.blip2_models.blip2_prompt_model import Blip2PromptTuning
from lavis.models.blip_models.blip_outputs import BlipOutput, BlipOutputFeatures


def save_predict(prediction, filename):
    with open(filename, 'w') as fp:
        json.dump(prediction, fp)

def concat_cache_files(cache_files, predict_file):
    results = []
    for f in cache_files:
        temp = json.load(open(f))
        results += temp
        os.remove(f)

    save_predict(results,predict_file)
    return

def save_predict_ddp(results, predict_file):
    world_size = get_world_size()

    if world_size == 1:
        save_predict(results, predict_file)
        print("Inference file saved")
        return

    else:
        cache_file = op.splitext(predict_file)[0] \
                     + f'_{get_rank()}_{world_size}' \
                     + op.splitext(predict_file)[1]

        save_predict(results, cache_file)
        if is_dist_avail_and_initialized():
            dist.barrier()

        if is_main_process():
            cache_files = [op.splitext(predict_file)[0] + '_{}_{}'.format(i, world_size) + \
                           op.splitext(predict_file)[1] for i in range(world_size)]
            concat_cache_files(cache_files, predict_file)
            print("Inference file saved")

    return

def SVD_text_img_feat(args, test_dataloader, model):
    model.eval()
    img_all_singular_values = None
    txt_all_singular_values = None
    with torch.no_grad():
        for step, samples in enumerate(tqdm(test_dataloader)):

            samples = prepare_sample(samples, cuda_enabled=True)

            outputs = model(samples, output_features=True)
            if is_dist_avail_and_initialized():
                dist.barrier()

            # ---------- image singular values ---------- #
            img_singular_values = [
                normalize(torch.svd(h).S, p=1.0, dim=-1, eps=1e-12, out=None).tolist()
                for h in [outputs["image_embeds_proj"]]
            ]

            if img_all_singular_values == None:
                img_all_singular_values = img_singular_values
            else:
                for l_i, l in enumerate(img_singular_values):
                    img_all_singular_values[l_i] += l

            # ------------------------------------------ #

            if is_dist_avail_and_initialized():
                dist.barrier()

            # ---------- Text singular values ---------- #
            txt_singular_values = [
                normalize(torch.svd(h).S, p=1.0, dim=-1, eps=1e-12, out=None).tolist()
                for h in [outputs["text_embeds_proj"]]
            ]

            if txt_all_singular_values == None:
                txt_all_singular_values = txt_singular_values
            else:
                for l_i, l in enumerate(txt_singular_values):
                    txt_all_singular_values[l_i] += l

            # ------------------------------------------ #

    path = f'lavis/output/BLIP/rank_extract/{args.folder_name}'
    # # # img_l0_file = f'{path}/svd_image_l0.json'
    # # # txt_l0_file = f'{path}/svd_text_l0.json'
    # # #
    # # # save_predict_ddp(img_all_singular_values[0], img_l0_file)
    # # # if is_dist_avail_and_initialized():
    # # #     dist.barrier()
    # #
    # # save_predict_ddp(txt_all_singular_values[0], txt_l0_file)
    # if is_dist_avail_and_initialized():
    #     dist.barrier()
    img_l12_file = f'{path}/svd_image_out.json'
    txt_l12_file = f'{path}/svd_text_out.json'

    save_predict_ddp(img_all_singular_values[-1], img_l12_file)
    if is_dist_avail_and_initialized():
        dist.barrier()

    save_predict_ddp(txt_all_singular_values[-1], txt_l12_file)
    if is_dist_avail_and_initialized():
        dist.barrier()

    print("SVD Done:")

def parse_args():
    parser = argparse.ArgumentParser(description="Training")

    parser.add_argument("--cfg-path", required=True, help="path to configuration file.")
    parser.add_argument(
        "--options",
        nargs="+",
        help="override some settings in the used config, the key-value pair "
        "in xxx=yyy format will be merged into config file (deprecate), "
        "change to --cfg-options instead.",
    )

    args = parser.parse_args()
    # if 'LOCAL_RANK' not in os.environ:
    #     os.environ['LOCAL_RANK'] = str(args.local_rank)

    return args


def setup_seeds(config):
    seed = config.run_cfg.seed + get_rank()

    random.seed(seed)
    np.random.seed(seed)
    torch.manual_seed(seed)

    cudnn.benchmark = False
    cudnn.deterministic = True


def get_runner_class(cfg):
    """
    Get runner class from config. Default to epoch-based runner.
    """
    runner_cls = registry.get_runner_class(cfg.run_cfg.get("runner", "runner_base"))

    return runner_cls


def main():
    # allow auto-dl completes on main process without timeout when using NCCL backend.
    # os.environ["NCCL_BLOCKING_WAIT"] = "1"

    # set before init_distributed_mode() to ensure the same job_id shared across all ranks.
    job_id = now()

    cfg = Config(parse_args())

    init_distributed_mode(cfg.run_cfg)

    setup_seeds(cfg)

    # set after init_distributed_mode() to only log on master.
    setup_logger()

    cfg.pretty_print()

    task = tasks.setup_task(cfg)
    datasets = task.build_datasets(cfg)

    # ---------------------------------- Model ---------------------------------- #
    model = task.build_model(cfg)

    if cfg.run_cfg["lora"]["enable"]:
        _ = cfg.run_cfg["lora"].pop("enable")
        ckpt = cfg.run_cfg["lora"].pop("from_checkpoint")
        config = LoraConfig(**cfg.run_cfg["lora"])
        model = get_peft_model(model, config)
        if ckpt is not None:
            model.load_state_dict(torch.load(ckpt), strict=False)

    if cfg.run_cfg["prompt_tuning"]["enable"]:
        _ = cfg.run_cfg["prompt_tuning"].pop("enable")
        ckpt = cfg.run_cfg["prompt_tuning"].pop("from_checkpoint")

        if "blip2_" in cfg.model_cfg["arch"]:
            model = Blip2PromptTuning(cfg.run_cfg["prompt_tuning"], model)
        elif "blip_" in cfg.model_cfg["arch"]:
            model = BlipPromptTuning(cfg.run_cfg["prompt_tuning"], model)
        else:
            ValueError("Prompt learning is not supported for this architecture.")

        if ckpt is not None:
            model.load_checkpoint(ckpt)
    else:
        if "blip2_" in cfg.model_cfg["arch"]:
            model = Blip2RankExtraction(model)
        else:
            ValueError("Rank extraction is not supported for this architecture.")


    model.print_trainable_parameters()
    runner = get_runner_class(cfg)(
        cfg=cfg, job_id=job_id, task=task, model=model, datasets=datasets
    )
    # --------------------------------------------------------------------------- #
    model = runner.model
    dataloader = runner.dataloaders['val']
    SVD_text_img_feat(cfg.run_cfg, dataloader, model)


if __name__ == "__main__":
    main()
