import json
import os
import re
import time

from dotenv import load_dotenv
from openai import OpenAI

from .final_feedback import generate_feedback
from .turnwise_feedback import turnwise_error_detection

load_dotenv(override=True)

client = OpenAI(
    api_key=os.environ["OPENAI_API_KEY"],
    organization=os.environ["OPENAI_ORGANIZATION_KEY"],
)


def process_input_history(history):
    convo = []
    for i, line in enumerate(history):
        role = "Buyer" if line["origin"] == "user" else "Seller"
        content = line["text"]
        convo.append({"id": line["id"], "role": role, "content": content})
    return convo


def process_turnwise_feedback(turnwise_feedback):
    answ = []
    for line in turnwise_feedback:
        answ.append(
            dict(
                message_id=line["id"],
                origin="bot" if line["role"] == "Seller" else "user",
                text=line["content"],
                feedback=line["feedback"],
            )
        )
    return answ


def read_json(file_path):
    with open(file_path, "r") as f:
        return json.load(f)


class FeedbackModule:
    def __init__(self):
        self.id = None

    def get_turnwise_feedback(self, history, params):
        target_price = float(params["targetPrice"])
        reservation_price = float(params["reservationPrice"])
        conversation = process_input_history(history)

        processed_turns, flagged_errors = turnwise_error_detection(
            conversation, reservation_price, target_price
        )
        return dict(data=process_turnwise_feedback(processed_turns))

    def get_overall_feedback(self, history):
        conversation = process_input_history(history)
        general_feedback = generate_feedback(conversation)
        return dict(data=general_feedback)
