"""Utility functions for message records and outputs."""



from openpyxl import load_workbook
from openpyxl.styles import PatternFill

from processing.number_utils import parse_number
from processing.position_utils import LinkedPosition


def convert_to_data(ws, data_args):
    """Ensure the worksheet cells in the self.data_args, all have 'n' type. """
    if data_args is None: return 

    for pos, _ in data_args:
        ws_cell = ws[pos.ws_str]
        if (ws_cell.data_type == 'n') or (ws_cell.value is None): continue
        org_value = ''.join([part for part in ws_cell.value.split(',')])
        num_value = parse_number(org_value)
        if num_value is None: continue
        ws_cell._bind_value(num_value)


# %% message keys

# Error Types and Filling Patterns
_LOSS = 'LOSS'
_WRONG = 'WRONG'
_FORMULA = 'FORMULA'
_ERROR = 'ERROR'

FILL_PATTERN = {
    _LOSS: PatternFill(
        fill_type='solid', 
        start_color='00CCFFFF', 
        end_color='00CCFFFF'
    ), 
    _WRONG: PatternFill(
        fill_type='solid', 
        start_color='00FF99CC', 
        end_color='00FF99CC'
    ), 
    _FORMULA: PatternFill(
        fill_type='solid', 
        start_color='00CCFFFF', 
        end_color='00CCFFFF'
    ), 
    _ERROR: PatternFill(
        fill_type='solid', 
        start_color='00FFCC99', 
        end_color='00FFCC99'
    )
}

# Spurious Objects and SubBlock Offsets
_ANNO_SENTENCE = 'SENTENCE'
_ANNO_KEYPART = 'KEYPART'
_ANNO_ANSWER = 'ANSWER'

OFFSETS = {
    _ANNO_SENTENCE: 0, 
    _ANNO_KEYPART: 2, 
    _ANNO_ANSWER: 6, 
}



def write_results(filename, results, col_index=7, pattern=FILL_PATTERN[_FORMULA]):
    """Write the list of results to the 'check' sheet."""

    # copy and create a new annotated worksheet
    wb = load_workbook(filename)
    anno_ws = wb['labeling']

    # remove previous tempts
    old_check_names = [name for name in wb.sheetnames if name.startswith('check')]
    old_check_sheets = [wb[name] for name in old_check_names]
    for ocs in old_check_sheets:
        wb.remove(ocs)

    ws = wb.copy_worksheet(anno_ws)
    ws.title = 'check'

    for formula in results:
        ws.cell(formula.row_index, col_index).value = formula.excel_reads
        ws.cell(formula.row_index, col_index).fill = pattern
        
    wb.save(filename)

    
def write_results_and_errors(filename, revisename, results, errors, res_col=7, err_col=8):
    # copy and create a new annotated worksheet
    wb = load_workbook(filename)
    anno_ws = wb['labeling']
    wb_data = load_workbook(filename, data_only=True)
    anno_ws_data = wb_data['labeling']

    # remove previous tempts
    old_check_names = [name for name in wb.sheetnames if name.startswith('check')]
    old_check_sheets = [wb[name] for name in old_check_names]
    for ocs in old_check_sheets:
        wb.remove(ocs)

    ws = wb.copy_worksheet(anno_ws)
    ws.title = 'check'

    for msg in errors:
        if (msg is None): continue
        if (msg.msg_object == _ANNO_SENTENCE) and (msg.msg_type == _LOSS): continue
        row_index = msg.global_row_index()
        ws.cell(row_index, err_col).fill = FILL_PATTERN[_WRONG]
        ws.cell(row_index, err_col).value = str(msg)

    nerr = 0
    for formula in results:
        if formula.excel_reads == '<ERROR>': 
            nerr += 1
            ws.cell(formula.row_index, res_col).fill = FILL_PATTERN[_ERROR]
            ws.cell(formula.row_index, res_col).value = formula.excel_reads

            if ('ARG' in formula.name):
                # add argument-formula template on the right
                r = formula.row_index
                # ws.cell(r, res_col + 1).value = 'header cells'
                # linked_header_pos = LinkedPosition.from_ws_coords((r, res_col+1))
                # ws.cell(r, res_col + 2).value = 'data cells'
                # linked_data_pos = LinkedPosition.from_ws_coords((r, res_col+2))
                # ws.cell(r, res_col + 3).value = 'index'
                # linked_index_pos = LinkedPosition.from_ws_coords((r, res_col+3))

                if formula.name.endswith('MAX'):
                    func = 'LARGE'
                else:
                    func = 'SMALL'
                header = '__header__'
                data = '__data__'
                index = '__index__'
                ws.cell(r + 1, res_col).value = f"=XLOOKUP({func}({data}, {index}), {data}, {header})"
        else:    # if ('ARG' in formula.name)
            if hasattr(formula, 'data_args'):
                convert_to_data(ws, formula.data_args)
            ws.cell(formula.row_index, res_col).value = formula.excel_reads

            # label out if the calc result unmatch with the anno result
            anno_res = anno_ws_data.cell(formula.row_index, 2).value
            calc_res = formula.get_answer()
            if isinstance(anno_res, str) and (anno_res != calc_res):
                ws.cell(formula.row_index, res_col).fill = FILL_PATTERN[_FORMULA]
            elif isinstance(anno_res, int) or isinstance(anno_res, float):
                try: 
                    calc_float = float(calc_res)
                    if round(float(anno_res), 2) != round(calc_float, 2):
                        ws.cell(formula.row_index, res_col).fill = FILL_PATTERN[_FORMULA]
                except:
                    ws.cell(formula.row_index, res_col).fill = FILL_PATTERN[_FORMULA]

    # set 'check' as the default sheet
    check_index = wb.sheetnames.index('check')
    anno_ws.sheet_view.tabSelected = False
    wb.active = check_index
    ws.sheet_view.tabSelected = True
    
    wb.save(revisename)
    wb.close()

    return nerr




# %% Message Class
class AnnoMessage(object):
    """Error message type during annotation reading and parsing."""

    def __init__(self, msg_type, msg_object, sub_block_start=None):
        self.msg_type = msg_type
        self.msg_object = msg_object
        self.fill_pattern = FILL_PATTERN[self.msg_type]
        self.offset = OFFSETS[self.msg_object]
        self.sub_start = sub_block_start

    def __str__(self):
        return f'{self.msg_type}-{self.msg_object}'
    
    def global_row_index(self):
        """Get the worksheet-global index of the starting row."""
        if self.sub_start is None:
            print(f'Message has NONE sub-block index. Please fill in first.')
            return None
        return self.sub_start + self.offset



def highlight_annotation_messages(filename, messages, col_index=5):
    """Highlight a Cell to indicate that an Annotation Error has Occurred."""

    # copy and create a new annotated worksheet
    wb = load_workbook(filename)
    anno_ws = wb['labeling']

    # remove previous tempts
    old_check_names = [name for name in wb.sheetnames if name.startswith('check')]
    old_check_sheets = [wb[name] for name in old_check_names]
    for ocs in old_check_sheets:
        wb.remove(ocs)

    ws = wb.copy_worksheet(anno_ws)
    ws.title = 'check'

    for msg in messages:
        if msg is None: continue
        row_index = msg.global_row_index()
        ws.cell(row_index, col_index).fill = FILL_PATTERN[_WRONG]
        ws.cell(row_index, col_index).value = str(msg)
        # print(f'error type [{msg.msg_type}] on object [{msg.msg_object}]')

    # print(f'highlighted {len(messages)} pieces of messages.')
    # print(f'\n\n')
    wb.save(filename)
