# -*- coding: utf-8 -*-

import functools
import os
import pickle

from .logger import LOGGER
from .utils import Singleton, identity

CACHE_TAGS = set()
IN_MEMORY_CACHE = {}
FILE_CACHE_DIR = '.cache'
MISSING = Singleton.create('MISSING')


def _cache_memory_get(tag):
    ret = IN_MEMORY_CACHE.get(tag, MISSING)
    if ret is not MISSING:
        LOGGER.info(f'Load "{tag}" from memory cache')
    return ret


def _cache_memory_set(tag, value):
    LOGGER.info(f'Add "{tag}" into memory cache')
    IN_MEMORY_CACHE[tag] = value


def _cache_file_get(tag):
    ret = MISSING
    file_name = os.path.join(FILE_CACHE_DIR, tag)
    if os.path.exists(file_name):
        with open(file_name, 'rb') as fp:
            ret = pickle.load(fp)

    if ret is not MISSING:
        LOGGER.info(f'Load cached file "{tag}"')
    return ret


def _cache_file_set(tag, value):
    try:
        os.makedirs(FILE_CACHE_DIR, exist_ok=True)

        file_name = os.path.join(FILE_CACHE_DIR, tag)
        with open(file_name, 'wb') as fp:
            pickle.dump(value, fp)
        LOGGER.info(f'Cached file generated "{tag}"')
    except Exception:
        LOGGER.exception(f'Failed to generte cache file "{tag}"')


def cache_get(tag):
    ret = _cache_memory_get(tag)
    if ret is MISSING:
        return _cache_file_get(tag)


def cache_set(tag, value):
    _cache_memory_set(tag, value)
    _cache_file_set(tag, value)


def make_cache(tag, mode: str='memory-file'):
    if mode == 'disable' or mode is None:
        return identity

    assert tag not in CACHE_TAGS, f'{tag} is already used'
    assert mode in {'memory', 'file', 'memory-file'}

    CACHE_TAGS.add(tag)

    post_hook = None
    if mode == 'memory':
        getter = _cache_memory_get
        setter = _cache_memory_set
    elif mode == 'file':
        getter = _cache_file_get
        setter = _cache_file_set
    else:
        getter = cache_get
        setter = _cache_file_set
        post_hook = _cache_memory_set

    def wrapper(fn):

        @functools.wraps(fn)
        def wrapped(*args, **kwargs):
            if callable(tag):
                name = tag(*args, **kwargs)
            else:
                name = tag

            ret = getter(name)
            if ret is MISSING:
                ret = fn(*args, **kwargs)
                setter(name, ret)

            if post_hook is not None:
                post_hook(name, ret)

            return ret

        return wrapped

    return wrapper
