///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// This file is part of ModelBlocks. Copyright 2009, ModelBlocks developers. //
//                                                                           //
//    ModelBlocks is free software: you can redistribute it and/or modify    //
//    it under the terms of the GNU General Public License as published by   //
//    the Free Software Foundation, either version 3 of the License, or      //
//    (at your option) any later version.                                    //
//                                                                           //
//    ModelBlocks is distributed in the hope that it will be useful,         //
//    but WITHOUT ANY WARRANTY; without even the implied warranty of         //
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          //
//    GNU General Public License for more details.                           //
//                                                                           //
//    You should have received a copy of the GNU General Public License      //
//    along with ModelBlocks.  If not, see <http://www.gnu.org/licenses/>.   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "nl-cpt.h"
#include "nl-dtree.h"

char psX[]="";
char psSlash[]="/";
char psComma[]=",";
char psSemi[]=";";
char psSemiSemi[]=";;";
char psDashDiamondDash[]="-<>-";
char psTilde[]="~";
char psBar[]="|";
//char psOpenBrace[]="{";
//char psCloseBrace[]="}";
char psLangle[]="<";
char psRangle[]=">";
char psLbrack[]="[";
char psRbrack[]="]";
char psLparen[]="(";
char psRparen[]=")";

const char* BEG_STATE = "S/end;-/-;-/-;-/-;-;;0~(3,3)[-|-|-|-]";
const char* END_STATE = "S/end;-/-;-/-;-/-;-;;0~(0,0)[-|-|-|-]";
//const char* BEG_STATE = "S/end;-/-;-/-;-/-;-;;0~0";
//const char* END_STATE = "S/end;-/-;-/-;-/-;-;;0~0";


////////////////////////////////////////////////////////////////////////////////
//
//  Random Variables
//
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////// Simple Variables

//// D: depth (input only, to HHMM models)...
DiscreteDomain<char> domD;
//typedef DiscreteDomainRV<char,domD> D;
class D : public DiscreteDomainRV<char,domD> {
 public:
  D ( )                : DiscreteDomainRV<char,domD> ( )    { }
  D ( int i )          : DiscreteDomainRV<char,domD> ( i )  { }
  D ( const char* ps ) : DiscreteDomainRV<char,domD> ( ps ) { }
};
const D D_0("0");
const D D_1("1");
const D D_2("2");
const D D_3("3");
const D D_4("4");
const D D_5("5");


//// B: boolean
DiscreteDomain<char> domB;
//typedef DiscreteDomainRV<char,domB> B;
class B : public DiscreteDomainRV<char,domB> {
 public:
  B ( )                : DiscreteDomainRV<char,domB> ( )    { }
  B ( const char* ps ) : DiscreteDomainRV<char,domB> ( ps ) { }
};
const B B_0 ("0");
const B B_1 ("1");

//// P: part of speech category...

DiscreteDomain<short> domainP;
typedef DiscreteDomainRV<short,domainP> P;

//// L: letter...
DiscreteDomain<char> domainLt;
typedef DiscreteDomainRV<char,domainLt> Lt;

//// W: word (array of letters, arranged last to first)...
//typedef StaticSafeArray<5,Lt> W;
DiscreteDomain<int> domainWd;
class Wd : public DiscreteDomainRV<int,domainWd>, public StaticSafeArray<5,Lt> {
 public:
  Wd ( ) { }
  Wd ( const char* ps ) : DiscreteDomainRV<int,domainWd>(ps) {
    char psTemp[2]="-"; int n=strlen(ps);
    for(int i=0;i<5;i++) {
      psTemp[0]=(i<n)?ps[n-i-1]:'_';
      //cout<<"!!!!!!!!!!!!!!!!!!"<<psTemp<<endl;
      StaticSafeArray<5,Lt>::set(i)=Lt(psTemp);
    }
  }
  friend pair<StringInput,Wd*> operator>> ( const StringInput ps, Wd& rv ) { return pair<StringInput,Wd*>(ps,&rv); }
  friend StringInput operator>> ( pair<StringInput,Wd*> si_x, const char* psDlm ) {
    if(si_x.first==NULL)return si_x.first; String s; StringInput si=si_x.first>>s>>psDlm; *si_x.second=s.c_array(); return si; }
  bool   operator== ( const Wd& w ) const { return DiscreteDomainRV<int,domainWd>::operator==(w); }
  size_t getHashKey ( )            const { return DiscreteDomainRV<int,domainWd>::getHashKey();  }
};

const static Wd W_EMPTY("-");

//// G: constituent category...
DiscreteDomain<int> domG;
//typedef DiscreteDomainRV<int,domG> G;
class G : public DiscreteDomainRV<int,domG> {
 private:
  static SimpleHash<G,B> hToTerm;
  void calcDetModels ( string s ) {
    if (!hToTerm.contains(*this)) {
      hToTerm.set(*this) = ('A'<=s[0] && s[0]<='Z') ? B_0 : B_1;
    }
  }
 public:
  G ( )                : DiscreteDomainRV<int,domG> ( )    { }
  G ( const DiscreteDomainRV<int,domG>& rv ) : DiscreteDomainRV<int,domG>(rv) { }
  G ( const char* ps ) : DiscreteDomainRV<int,domG> ( ps ) { calcDetModels(ps); }
  //G ( string s ) : DiscreteDomainRV<int,domG> ( s )  { calcDetModels(s); }
  B getTerm ( ) const { return hToTerm.get(*this); }
  friend pair<StringInput,G*> operator>> ( StringInput si, G& m ) { return pair<StringInput,G*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,G*> si_m, const char* psD ) {
    if ( si_m.first == NULL ) return NULL;
    StringInput si=si_m.first>>(DiscreteDomainRV<int,domG>&)*si_m.second>>psD;
    si_m.second->calcDetModels(si_m.second->getString()); return si; }
};
SimpleHash<G,B> G::hToTerm;
const G G_BOT("-");
const G G_TOP("ROOT");
const G G_RST("REST");
const G G_et("et");
const G G_et1("et1");
const G G_et2("et2");

//// C: same as G, since no role labels...
typedef G C;


//// F: final-state...
DiscreteDomain<int> domF;
//typedef DiscreteDomainRV<char,domF> F;
class F : public DiscreteDomainRV<int,domF> {
 private:
  static SimpleHash<F,G> hToG;
  static SimpleHash<G,F> hFromG;
  void calcDetModels ( string s ) {
    if (!hToG.contains(*this)) {
      hToG.set(*this) = (this->toInt()>1) ? G(this->getString().c_str()) : G_BOT;
      hFromG.set(G(this->getString().c_str())) = *this;
    }
  }
 public:
  F ( )                : DiscreteDomainRV<int,domF> ( )    { }
  F ( const DiscreteDomainRV<int,domF>& rv ) : DiscreteDomainRV<int,domF>(rv) { }
  F ( const char* ps ) : DiscreteDomainRV<int,domF> ( ps ) { calcDetModels(ps); }
  F ( const G& g )                                         { *this = hFromG.get(g); }
  G        getG ( )     const { return hToG.get(*this); }
  static F getF ( G g )       { return hFromG.get(g); }
  friend pair<StringInput,F*> operator>> ( StringInput si, F& m ) { return pair<StringInput,F*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,F*> si_m, const char* psD ) {
    if ( si_m.first == NULL ) return NULL;
    StringInput si=si_m.first>>(DiscreteDomainRV<int,domF>&)*si_m.second>>psD;
    si_m.second->calcDetModels(si_m.second->getString()); return si; }
};
SimpleHash<F,G> F::hToG;
SimpleHash<G,F> F::hFromG;
const F F_0("0");
const F F_1("1");
const F F_BOT("-");

//// Q: syntactic state...
typedef DelimitedJoint2DRV<psX,G,psSlash,G,psX> Q;
//class Q : public DelimitedJoint2DRV<psX,G,psSlash,G,psX> {
// public:
//  Q ( )                                                   : DelimitedJoint2DRV<psX,G,psSlash,G,psX> ( )    { }
//  Q ( char* ps )                                          : DelimitedJoint2DRV<psX,G,psSlash,G,psX> ( ps ) { }
//  Q ( const G& g1, const G& g2 )                          : DelimitedJoint2DRV<psX,G,psSlash,G,psX> ( g1, g2 ) { }
//};
#define Q_BOT Q(G_BOT,G_BOT)
#define Q_TOP Q(G_TOP,G_RST)



//////////////////////////////////////// Joint Variables

//// R: collection of syntactic variables at all depths in each `reduce' phase...
typedef DelimitedJointArrayRV<4,psSemi,F> R;

//// S: collection of syntactic variables at all depths in each `shift' phase...
class S : public DelimitedJoint2DRV<psX,DelimitedJointArrayRV<4,psSemi,Q>,psSemi,G,psX> {
 public:
  operator G()  const { return ( ( (second      !=G_BOT) ? second       :
                                   (first.get(3)!=Q_BOT) ? first.get(3).second :
                                   (first.get(2)!=Q_BOT) ? first.get(2).second :
                                   (first.get(1)!=Q_BOT) ? first.get(1).second : first.get(0).second ) ); }
  bool compareFinal ( const S& s ) const { return(*this==s); }
};

//// IP: Represents state of repair (none, ET, alteration)
DiscreteDomain<int> domI;
typedef DiscreteDomainRV<int,domI> I;

const I I_F("0");
const I I_ET("ET");
const I I_ET1i("ET1i"); const I I_ET1you("ET1you");
const I I_ET2know("ET2know"); const I I_ET2mean("ET2mean"); const I I_ET2guess("ET2guess"); const I I_ET2think("ET2think");
const I I_ALT("1");
const I I_REPEAT("REPEAT");

// Buffer size and position
const static int bufferSize = 4;
DiscreteDomain<int> domainInd;
typedef DiscreteDomainRV<int,domainInd> Ind;

//class Ind : public DiscreteDomainRV<int,domainInd>{
//  public:
//    int toAbs(){
//      int i = atoi(getString().c_str());

//};

// A Buffer is a combination of an int pair (Indices) and an array of words (NakedBuffer)
// The ints Ind into the array, first int representing the `cur', and the
// 2nd representing the end position (if we've gone backwards for a repair)
//class Buff : public DelimitedJoint2DRV<psX,DelimitedJoint2DRV<psLparen,Ind,psComma,Ind,psRparen>,psLbrack,DelimitedJointArrayRV<bufferSize,psBar,Wd>,psRbrack> {
//};
typedef DelimitedJoint2DRV<psLparen,Ind,psComma,Ind,psRparen> Indices;
typedef DelimitedJointArrayRV<bufferSize,psBar,Wd> NakedBuffer;
typedef DelimitedJoint2DRV<psX,Indices,psLbrack,NakedBuffer,psRbrack> Buff;

class IB : public DelimitedJoint2DRV<psX,I,psTilde,Buff,psX> {
  public:
    operator I() const { return first; }
    operator Buff() const { return second; }
};

/*
class SI : public DelimitedJoint2DRV<psX,S,psSemiSemi,I,psX> {
   public:
   operator S() const { return first;}
   bool compareFinal ( const SI& s ) const {
     //return(*this==s);
     return first.compareFinal(s.first);
   }
};
*/

class SIB : public DelimitedJoint2DRV<psX,S,psSemiSemi,IB,psX> {
  public:
   operator S() const { return first;}
   operator IB() const { return second; }
   bool compareFinal ( const SIB& s ) const {
     //return(*this==s);
     return first.compareFinal(s.first);
   }
};

//// H: the set of all (marginalized) reduce and (modeled) shift variables in the HHMM...
class H : public DelimitedJoint2DRV<psX,R,psDashDiamondDash,SIB,psX>
{ public:
  operator R() const {return first;}
  operator SIB() const {return second;}
  operator D() const {return ( (first.get(3)==F_0) ? D_4 :
                               (first.get(2)==F_0) ? D_3 :
                               (first.get(1)==F_0) ? D_2 :
                               (first.get(0)==F_0) ? D_1 : D_0 );}
};


////////////////////////////////////////////////////////////////////////////////
//
//  Models
//
////////////////////////////////////////////////////////////////////////////////

typedef HidVarCPT2DModel<P,G,LogProb> PgivGModel;

//// Generative model of word given tag...
class WModel {
 private:
  TrainableDTree2DModel<P,Wd,LogProb> modPgivWdt;
  //Unk2DModel modWunkgivP;

  RandAccCPT2DModel<P,Wd,LogProb> modPgivWs;
  RandAccCPT1DModel<P,LogProb> modP;
  RandAccCPT1DModel<Wd,LogProb> modW;

 public:

  //LogProb getProb ( const W& w, const HidVarCPT1DModel<P,LogProb>::IterVal& p ) const {
  LogProb getProb ( const Wd& w, const P::ArrayIterator<LogProb>& p, int loc ) const;
  //LogProb getBufProb ( const Word& w, const P::ArrayIterator<LogProb>& p, int loc) const;
  void writeFields ( FILE* pf, string sPref ) { //modPgivWdt.writeFields(pf,sPref);
  }
  friend pair<StringInput,WModel*> operator>> ( StringInput si, WModel& m ) { return pair<StringInput,WModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,WModel*> delimbuff, const char* psD ) {
    StringInput si;
    return ( (si=delimbuff.first>>"W "   >>delimbuff.second->modW      >>psD)!=NULL ||
             (si=delimbuff.first>>"Pw "  >>delimbuff.second->modPgivWs >>psD)!=NULL ||
//             (si=delimbuff.first>>"UNK ">>delimbuff.second->modWunkgivP>>psD)!=NULL ||
             (si=delimbuff.first>>"PwDT ">>delimbuff.second->modPgivWdt>>psD)!=NULL ||
             (si=delimbuff.first>>"P "   >>delimbuff.second->modP      >>psD)!=NULL ) ? si : StringInput(NULL);
  }
};

//// Wrapper class for model
class OModel {

 private:

  PgivGModel modPgivG;
  WModel     modWgivP;
  static RandAccCPT2DModel<Wd,I,LogProb> modWgivI;
  static int word_num;

 public:
//  typedef MapKey2D<I,G> IPGKey;
//  static Buff wBuf;
  static Wd curWord;
  static SimpleHash<G,Prob> hcpCache;

  class DistribModeledWgivG {
   private:

    // word buffer
   public:
    DistribModeledWgivG& set ( const Wd& w, const OModel& m ){
//      cerr << "Setting word..." << w << endl;
      word_num++;
      curWord = w;
      m.calcProb(*this,w);
//      wBuf.set() = Word(w);
      //wBuf.dump();
      return *this;
    }

    void    clear   ( )                     { hcpCache.clear(); }
    Prob&   setProb ( const G& g )       { return hcpCache.set(g); }
    LogProb getProb ( const G& g ) const { return LogProb(hcpCache.get(g)); }
    LogProb getProb ( const SIB& s ) const {
//        cerr << "  OModel::getProb consulted with word " << curWord << " and s: " << s << endl;
        if(s.second.first == I_REPEAT){
            int cur = atoi(Ind(s.second.second.first.first).getString().c_str());
            if(s.second.second.second.get(cur) == curWord){
              return LogProb(1.0);
            }else return LogProb(0.0);
        }else if(s.second.first == I_ET || s.second.first == I_ET1i || s.second.first == I_ET1you ||
                 s.second.first == I_ET2know || s.second.first == I_ET2mean || s.second.first == I_ET2think || s.second.first == I_ET2guess){
            return modWgivI.getProb(curWord, s.second.first);
        }else if((s.second.first == I_F && s.second.second.first.first == s.second.second.first.second)){
           // if condition: no repair (IP_F and curindex = endindex) or editing term
          return LogProb(hcpCache.get(s.first));
        }else if(s.second.first == I_ALT){
            if(hcpCache.get(s.first) > Prob(0.0)){
              return LogProb(1.0);
            }else{
              return LogProb(0.0);
            }
        }else{
          // condition: continuation of multi-word repair (iP = IP_F but cur
          // index not caught up to end index)
          return LogProb(hcpCache.get(s.first));
        }
    }
    const Wd& getW ( ) { return curWord; }
  };

  typedef DistribModeledWgivG RandVarType;

  void calcProb ( OModel::RandVarType& o, const Wd& w) const;
  LogProb getProb ( const OModel::RandVarType& o, const SIB& sib ) const { return o.getProb(sib); }

  friend pair<StringInput,OModel*> operator>> ( StringInput si, OModel& m ) { return pair<StringInput,OModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,OModel*> delimbuff, const char* psD ) {
    StringInput si;
    return ( (si=delimbuff.first>>"Pc ">>delimbuff.second->modPgivG>>psD)!=NULL ||
             (si=delimbuff.first>>"Wi ">>delimbuff.second->modWgivI>>psD)!=NULL ||
             (si=delimbuff.first>>       delimbuff.second->modWgivP>>psD)!=NULL ) ? si : StringInput(NULL);
  }

  void writeFields ( FILE* pf, string sPref ) { modWgivP.writeFields(pf,sPref); }
};

SimpleHash<G,Prob> OModel::hcpCache;
RandAccCPT2DModel<Wd,I,LogProb> OModel::modWgivI;

//////////////////////////////////////// "Wrapper" models for individual RVs...

//// Model of F given D and F and Q (from above) and Q (from previous)
class FModel {
 private:
  HidVarCPT4DModel<F,D,G,G,LogProb> mFr;            // Reduction model: F given D, G (active cat from prev), G (awaited cat from above) (assume prev awaited = reduction)
  static const HidVarCPT1DModel<F,LogProb> mF_0;    // Fixed F_ZERO model.
  static const HidVarCPT1DModel<F,LogProb> mF_BOT;  // Fixed F_BOT model.
 public:
  //static BphModel mBph;
  static bool F_ROOT_OBS;
  LogProb setIterProb ( F::ArrayIterator<LogProb>& f, const D& d, const F& fD, const I& iC, const Q& qP, const Q& qU, int& a ) const {
    LogProb pr;
    if(iC==I_REPEAT || (iC != I_F && iC != I_ALT)){
       // not fluent, not alteration, means editing terms...
       pr = mF_0.setIterProb(f,a);
    }else if ( fD==F_BOT && (qP.second==G_BOT) ) {
      // >1 (bottom) case...
      pr = mF_BOT.setIterProb(f,a);
    }
    //else if ( fD>F_1 && (qP.second==F(fD).getG() || (qP.second.getTerm()==B_1 && fD==F_BOT)) ) {
    else if ( fD==F_BOT && qP.second.getTerm()==B_1 ) {
      // >1 (middle) case...
      pr = mFr.setIterProb(f,d,qP.first,qU.second,a);
      /*
      if ( F(f) > F_1 )
        pr /= mBph.getProb(B_1,d,qU.second);
      */
    }
    //if ( fD==F_0 || (qP.second!=F(fD).getG() && !(qP.second.getTerm()==B_1 && F(fD).getG()==G_NONE)) ) ) {
    else {
      // 0 (top) case...
      pr = mF_0.setIterProb(f,a);
    }
    // Report error...
    if ( a<-2 && pr==LogProb() ) cerr<<"\nERROR: no condition Fr "<<d<<" "<<qP.first<<" "<<qU.second<<"\n\n";
    // Iterate again if result doesn't match root observation...
    if ( a>=-2 && d==D_1 && F_ROOT_OBS!=(F(f)>F_1) ) pr=LogProb();
    ////cerr<<"  F "<<d<<" "<<fD<<" "<<qP<<" "<<qU<<" : "<<f<<" = "<<pr<<" ("<<a<<")\n";
    return pr;
  }
  friend pair<StringInput,FModel*> operator>> ( StringInput si, FModel& m ) { return pair<StringInput,FModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,FModel*> si_m, const char* psD ) {
    StringInput si;
    return ( //(si=si_m.first>>"Bph ">>si_m.second->mBph>>psD)!=NULL ||
             (si=si_m.first>>"Fr " >>si_m.second->mFr >>psD)!=NULL ) ? si : StringInput(NULL);
  }
};
//BphModel FModel::mBph;
const HidVarCPT1DModel<F,LogProb> FModel::mF_0(F_0);
const HidVarCPT1DModel<F,LogProb> FModel::mF_BOT(F_BOT);
bool FModel::F_ROOT_OBS = false;


//// Model of Q given D and F and F and Q(from prev) and Q(from above)
class QModel {
 public:
  HidVarCPT3DModel<G,D,G,LogProb> mGe;     // Expansion model of G given D, G (from above)
 private:
  HidVarCPT4DModel<G,D,G,G,LogProb> mGtm;  // Awaited transition model of G (active) given D, G (awaited cat from previous), G (from reduction)
  HidVarCPT4DModel<G,D,G,G,LogProb> mGtp;  // Active Transition model of G (active) given D, G (active cat from previous), G (awaited cat from above)
  HidVarCPT4DModel<G,D,G,G,LogProb> mGtq;  // Active Transition completion of G (awaited) given D, G (active cat from current), G (active cat from previous)
  static const HidVarCPT1DModel<G,LogProb>   mG_BOT;   // Fixed G_BOT model.
  static       HidVarCPT2DModel<G,G,LogProb> mG_COPY;  // Cached F_COPY model  --  WARNING: STATIC NON-CONST is not thread safe!
 public:
  LogProb setIterProb ( Q::ArrayIterator<LogProb>& q, const D& d, const F& fD, const F& f, const I& iC, const Q& qP, const Q& qU, int& a ) const {
    LogProb pr,p;
    if(iC==I_REPEAT || (iC!=I_F && iC!=I_ALT)){
        // iC is some ET state...
        if(!mG_COPY.contains(qP.first)) mG_COPY.setProb(qP.first,qP.first) = 1.0;
        pr = p = mG_COPY.setIterProb(q.first, qP.first, a);
        if(!mG_COPY.contains(qP.second)) mG_COPY.setProb(qP.second,qP.second) = 1.0;
        pr *= p = mG_COPY.setIterProb(q.second,qP.second, a);
    }else if (fD>F_1) {
      if (f>=F_1) {
        if (f>F_1) {
          if (qU.second.getTerm()==B_1 || qU==Q_BOT) {
            ////cerr<<"a\n";
            // >1 >1 (expansion to null) case:
            pr  = mG_BOT.setIterProb(q.first, a);
            pr *= mG_BOT.setIterProb(q.second,a);
          }
          else {
            ////cerr<<"b\n";
            // >1 >1 (expansion) case:
            pr  = p = mGe.setIterProb(q.first,d,qU.second,a);
            if ( a==-2 && p==LogProb() ) cerr<<"\nERROR: no condition Ge "<<d<<" "<<qU.second<<"\n\n";
            if ( !mG_COPY.contains(G(q.first)) ) mG_COPY.setProb(G(q.first),G(q.first))=1.0;
            pr *= p = mG_COPY.setIterProb(q.second,G(q.first),a);
            /*
            pr *= FModel::mBph.getProb(B_1,d,qU.second);
            */
          }
        }
        else {
          ////cerr<<"c\n";
          // >1 1 ('plus' transition following reduction) case:
          pr  = p = mGtp.setIterProb(q.first,d,qP.first,qU.second,a);
          if ( a==-2 && p==LogProb() ) cerr<<"\nERROR: no condition Gtp "<<d<<" "<<qP.first<<" "<<qU.second<<"\n\n";
          pr *= p = mGtq.setIterProb(q.second,d,G(q.first),qP.first,a);
          if ( a==-2 && p==LogProb() ) cerr<<"\nERROR: no condition Gtq "<<d<<" "<<G(q.first)<<" "<<qP.first<<"\n\n";
        }
      }
      else {
        ////cerr<<"d\n";
        // >1 0 ('minus' transition without reduction) case:
        if ( !mG_COPY.contains(qP.first) ) mG_COPY.setProb(qP.first,qP.first)=1.0;
        pr  = p = mG_COPY.setIterProb(q.first,qP.first,a);
        pr *= p = mGtm.setIterProb(q.second,d,fD.getG(),qP.second,a);
        if ( a==-2 && p==LogProb() ) cerr<<"\nERROR: no condition Gtm "<<d<<" "<<fD.getG()<<" "<<qP.second<<"\n\n";
      }
    }
    else {
      ////cerr<<"e\n";
      // <=1 0 (copy) case:
      if ( !mG_COPY.contains(qP.first) )  mG_COPY.setProb(qP.first, qP.first )=1.0;
      pr  = p = mG_COPY.setIterProb(q.first, qP.first, a);
      if ( !mG_COPY.contains(qP.second) ) mG_COPY.setProb(qP.second,qP.second)=1.0;
      pr *= p = mG_COPY.setIterProb(q.second,qP.second,a);
    }
    ////cerr<<"  Q "<<d<<" "<<fD<<" "<<f<<" "<<qP<<" "<<qU<<" : "<<q<<" = "<<pr<<" ("<<a<<")\n";
    return pr;
  }
  friend pair<StringInput,QModel*> operator>> ( StringInput si, QModel& m ) { return pair<StringInput,QModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,QModel*> si_m, const char* psD ) {
    StringInput si;
    return ( (si=si_m.first>>"Gtm ">>si_m.second->mGtm>>psD)!=NULL ||
             (si=si_m.first>>"Gtp ">>si_m.second->mGtp>>psD)!=NULL ||
             (si=si_m.first>>"Gtq ">>si_m.second->mGtq>>psD)!=NULL ||
             (si=si_m.first>>"Ge ">>si_m.second->mGe>>psD)!=NULL ) ? si : StringInput(NULL);
  }
};
const HidVarCPT1DModel<G,LogProb> QModel::mG_BOT(G_BOT);
HidVarCPT2DModel<G,G,LogProb> QModel::mG_COPY;


//////////////////////////////////////// Joint models...

//////////////////// Reduce phase...

//// Model of R given S
class RModel : public SingleFactoredModel<FModel> {
 public:
  LogProb setIterProb ( R::ArrayIterator<LogProb>& r, const I& iC, const SIB& sibP, int& a ) const {
    const S& sP = sibP.first;
    const FModel& mF = getM1();
    LogProb pr;
    pr  = mF.setIterProb ( r.set(4-1), 4, F(sP.second) , iC, sP.first.get(4-1), sP.first.get(3-1), a );
    pr *= mF.setIterProb ( r.set(3-1), 3, F(r.get(4-1)), iC, sP.first.get(3-1), sP.first.get(2-1), a );
    pr *= mF.setIterProb ( r.set(2-1), 2, F(r.get(3-1)), iC, sP.first.get(2-1), sP.first.get(1-1), a );
    pr *= mF.setIterProb ( r.set(1-1), 1, F(r.get(2-1)), iC, sP.first.get(1-1), Q_TOP            , a );
    return pr;
  }
};


//////////////////// Shift phase...

//// Model of S given R and S
class SModel : public SingleFactoredModel<QModel> {
 private:
  static const HidVarCPT1DModel<G,LogProb>   mG_BOT;
 public:
  LogProb setIterProb ( S::ArrayIterator<LogProb>& s, const R::ArrayIterator<LogProb>& r, const I& iC, const S& sP, int& a ) const {
    const QModel& mQ = getM1();
    LogProb pr;
    pr  = mQ.setIterProb ( s.first.set(1-1), 1, F(r.get(2-1)), F(r.get(1-1)), iC, sP.first.get(1-1), Q_TOP                        ,a );
    pr *= mQ.setIterProb ( s.first.set(2-1), 2, F(r.get(3-1)), F(r.get(2-1)), iC, sP.first.get(2-1), Q().setVal(s.first.set(1-1)) ,a );
    pr *= mQ.setIterProb ( s.first.set(3-1), 3, F(r.get(4-1)), F(r.get(3-1)), iC, sP.first.get(3-1), Q().setVal(s.first.set(2-1)) ,a );
    pr *= mQ.setIterProb ( s.first.set(4-1), 4, sP.second,     F(r.get(4-1)), iC, sP.first.get(4-1), Q().setVal(s.first.set(3-1)) ,a );
    pr *= ( G(s.first.set(4-1).second)!=G_BOT &&
            G(s.first.set(4-1).second).getTerm()!=B_1 )
      ? mQ.mGe.setIterProb ( s.second, 5, G(s.first.set(4-1).second), a )
      : mG_BOT.setIterProb ( s.second, a );
    ////cerr<<"  G "<<5<<" "<<G(q4.second)<<" : "<<g<<" = "<<pr<<" ("<<a<<")\n";
    return pr;
  }
};
const HidVarCPT1DModel<G,LogProb> SModel::mG_BOT(G_BOT);

class IModel{
 private:
  HidVarCPT2DModel<I,I,LogProb> mIP;
  static HidVarCPT1DModel<I,LogProb> mI_ZERO;
public:
  LogProb setIterProb ( I::ArrayIterator<LogProb>& i, const I& iP, int& a ) const{
    LogProb pr;
//    cerr << "    Inside i model: iP = " << iP << ", and new i is: ";
    pr = mIP.setIterProb(i, iP, a);
    //pr = mI_ZERO.setIterProb(i,a);
//    cerr << i << endl;
    return pr;
  }
  friend pair<StringInput,IModel*> operator>> ( StringInput si, IModel& m ) { return pair<StringInput,IModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,IModel*> si_m, const char* psD ) {
    StringInput si;
    return ( (si=si_m.first>>"IP ">>si_m.second->mIP>>psD)!=NULL ) ? si : StringInput(NULL);
    //return StringInput(NULL);
  }
};

HidVarCPT1DModel<I,LogProb> IModel::mI_ZERO("0");

class BuffModel{
  private:
    HidVarCPT2DModel<Ind,Ind,LogProb> mBB; // BB = buffer backward
    HidVarCPT2DModel<Ind,Ind,LogProb> mBM; // BM = buffer movement (f or b)
    static HidVarCPT2DModel<Wd,Wd,LogProb> mW_COPY;
    static HidVarCPT2DModel<Wd,Wd,LogProb> mW_CLEAR;
    static HidVarCPT4DModel<Wd,G,B,Wd,LogProb> mW_SUBCOPY;
    static HidVarCPT4DModel<Wd,G,B,Wd,LogProb> mW_ALIGNCOPY;
    static HidVarCPT2DModel<Ind,Ind,LogProb> mIND_COPY;
    static float subParam; // = 0.5;
 public:
  LogProb setIterProb ( Buff::ArrayIterator<LogProb>& b, const I& iC, const S::ArrayIterator<LogProb>& sC, const R::ArrayIterator<LogProb>& r, const SIB& sibP, int& a ) const{
//    const I& iP = sibP.second.first;
    // Look in the buffer (second.second), first item (index pair).
    const Ind& curP = sibP.second.second.first.first;
    const Ind& endP = sibP.second.second.first.second;
    S s_temp; s_temp.setVal(sC);
    G g = G(s_temp);

    LogProb pr;
    if(iC==I_REPEAT || (iC!=I_F && iC!=I_ALT)){
      // Editing term: Leave pointers where they are, copy words across
      if(!mIND_COPY.contains(curP)){
        mIND_COPY.setProb(curP,curP) = 1.0;
      }
      if(!mIND_COPY.contains(endP)){
        mIND_COPY.setProb(endP,endP) = 1.0;
      }
      // we're in an interruption point...
      pr = mIND_COPY.setIterProb(b.first.first,curP,a);
      pr *= mIND_COPY.setIterProb(b.first.second,endP,a);
      for(int i = 0; i < bufferSize; i++){
        if(!mW_COPY.contains(sibP.second.second.second.get(i))){
            mW_COPY.setProb(sibP.second.second.second.get(i),
                            sibP.second.second.second.get(i)) = 1.0;
        }
        pr *= mW_COPY.setIterProb(b.second.set(i), sibP.second.second.second.get(i), a);
      }
    }else if(iC == I_F && curP == endP){
      // no interruption point, no alteration, so keep both pointers at the end.
      // move old words backwards, put the current word into the end index
      // : Probability is faked here, measured in obs model for easier caching
      if(!mIND_COPY.contains(curP)){
        mIND_COPY.setProb(curP,curP) = 1.0;
      }
      pr = mIND_COPY.setIterProb(b.first.first, curP,a);
      pr *= mIND_COPY.setIterProb(b.first.second, endP,a);
      int icur = bufferSize-1; //atoi(Ind(b.first.first).getString().c_str());
      for(int i = 0; i < bufferSize; i++){
        if(i == icur){
          // copy word from observation into buffer (cheating hack but should
          // improve speed)
          if(!mW_COPY.contains(OModel::curWord)){
            mW_COPY.setProb(OModel::curWord,OModel::curWord) = 1.0;
          }
          pr *= mW_COPY.setIterProb(b.second.set(i), OModel::curWord,a);
        }else{
          if(!mW_COPY.contains(sibP.second.second.second.get(i+1))){
            mW_COPY.setProb(sibP.second.second.second.get(i+1),
                            sibP.second.second.second.get(i+1)) = 1.0;
          }
          pr *= mW_COPY.setIterProb(b.second.set(i), sibP.second.second.second.get(i+1), a);
        }
      }
    }else if(iC == I_ALT){
      // ALTERATION CASE:
      // first word of alteration: move back 'cur' pointer
      // first copy over value of 'end' pointer (stays)
      if(!mIND_COPY.contains(endP)){
        mIND_COPY.setProb(endP,endP) = 1.0;
      }
      pr = mIND_COPY.setIterProb(b.first.second,endP,a);
      // select a distance back to go (0..3 words where 0 is overwriting current word)
      pr *= mBB.setIterProb(b.first.first,curP,a);
      int icur = atoi(Ind(b.first.first).getString().c_str());
//      int iend = bufferSize-1; //atoi(Ind(b.first.second).getString().c_str());
      for(int i = 0; i < bufferSize; i++){
//        int abs_i = (bufferSize-1-iend + i) % bufferSize; //i + (iend+bufferSize-1) % bufferSize;
//        int abs_c = (bufferSize-1-iend + icur) % bufferSize; //icur + (iend+bufferSize-1) % bufferSize;
        if(i == icur){
          // decide if we should substitute or copy
          //mSub.setIterProb(
          if(OModel::curWord == sibP.second.second.second.get(i)){
            // 1-subs probability, prob. of just copying word over
            if(!mW_SUBCOPY.contains(g, B_1, sibP.second.second.second.get(i))){
              mW_SUBCOPY.setProb(sibP.second.second.second.get(i), g, B_1,
                                 sibP.second.second.second.get(i)) = 1-subParam;
            }
            pr *= mW_SUBCOPY.setIterProb(b.second.set(i), g, B_1, sibP.second.second.second.get(i), a);
          }else{
            // subParam, probability of substitution operation
            if(!mW_SUBCOPY.contains(g, B_0, OModel::curWord)){
              if(OModel::hcpCache.get(g) > Prob(0.0)){
                mW_SUBCOPY.setProb(OModel::curWord, g, B_0, OModel::curWord) = subParam * OModel::hcpCache.get(g);
              }else{
                mW_SUBCOPY.setProb(OModel::curWord, g, B_0, OModel::curWord) = subParam;
              }
            }
            pr *= mW_SUBCOPY.setIterProb(b.second.set(i), g, B_0, OModel::curWord, a);
          }
        }/*else if(abs_i < abs_c){
          // in absolute terms (as opposed to the circular implementation),
          // this index is less than the current index
          // we should clear out buffer before retrace point,
          // to prevent weird future repairs.
          if(!mW_CLEAR.contains(sibP.second.second.second.get(i))){
            mW_CLEAR.setProb(Wd("-"), sibP.second.second.second.get(i)) = 1.0;
          }
          pr *= mW_CLEAR.setIterProb(b.second.set(i), sibP.second.second.second.get(i), a);
        }*/
        else{
          if(!mW_COPY.contains(sibP.second.second.second.get(i))){
            mW_COPY.setProb(sibP.second.second.second.get(i),
                            sibP.second.second.second.get(i)) = 1.0;
          }
          pr *= mW_COPY.setIterProb(b.second.set(i), sibP.second.second.second.get(i), a);
        }
      }
    }else if(iC == I_F){
      // Not at the first word of alteration, but recently had repair and
      // the cur pointer is not at the end of the buffer...
      // either stay the same (insert), move forward one position (copy or sub)
      // or move forward > 1 position (deletion)
      pr = mBM.setIterProb(b.first.first, curP, a);
      pr *= mIND_COPY.setIterProb(b.first.second, endP, a);
      int icurP = atoi(curP.getString().c_str());
//      int iendP = atoi(endP.getString().c_str());
      int icur = atoi(Ind(b.first.first).getString().c_str());

      if(icur == icurP){
        // Insert: force an overwrite:
        for(int i = 0; i < bufferSize; i++){
          if(i == icur){
            // copy word from observation into buffer (cheating hack but should
            // improve speed)
            if(!mW_COPY.contains(OModel::curWord)){
              mW_COPY.setProb(OModel::curWord,OModel::curWord) = 1.0;
            }
            pr *= mW_COPY.setIterProb(b.second.set(i), OModel::curWord,a);
          }else{
            if(!mW_COPY.contains(sibP.second.second.second.get(i))){
              mW_COPY.setProb(sibP.second.second.second.get(i),
                              sibP.second.second.second.get(i)) = 1.0;
            }
            pr *= mW_COPY.setIterProb(b.second.set(i), sibP.second.second.second.get(i), a);
          }
        }
      }else{         // if(icur == (icurP+1))
        // Now choose between copy and substitute probabilistically...
        for(int i = 0; i < bufferSize; i++){
          if(i == icur){
            // if word is the same, copy it over with high probability (to
            // make up for the drubbing it will take in the observed probs.)
            if(OModel::curWord == sibP.second.second.second.get(i)){
              if(!mW_ALIGNCOPY.contains(g, B_1, sibP.second.second.second.get(i))){
                if(OModel::hcpCache.get(g) > Prob(0.0)){
                  mW_ALIGNCOPY.setProb(sibP.second.second.second.get(i), g, B_1,
                                     sibP.second.second.second.get(i)) = (1.0-subParam) / OModel::hcpCache.get(g);
                }else{
                  // will be zeroed out eventually, but can't put a zero here...
                  mW_ALIGNCOPY.setProb(sibP.second.second.second.get(i), g, B_1,
                                     sibP.second.second.second.get(i)) = 1.0;
                }
              }
              pr *= mW_ALIGNCOPY.setIterProb(b.second.set(i), g, B_1, sibP.second.second.second.get(i), a);
            }else{
              if(!mW_ALIGNCOPY.contains(g, B_0, OModel::curWord)){
                 mW_ALIGNCOPY.setProb(OModel::curWord, g, B_0, OModel::curWord) = subParam;
              }
              pr *= mW_ALIGNCOPY.setIterProb(b.second.set(i), g, B_0, OModel::curWord, a);
            }
          }else{
            if(!mW_COPY.contains(sibP.second.second.second.get(i))){
              mW_COPY.setProb(sibP.second.second.second.get(i),
                              sibP.second.second.second.get(i)) = 1.0;
            }
            pr *= mW_COPY.setIterProb(b.second.set(i), sibP.second.second.second.get(i), a);
          }
        }
      }
    }
    return pr;
  }
  friend pair<StringInput,BuffModel*> operator>> ( StringInput si, BuffModel& m ) { return pair<StringInput,BuffModel*>(si,&m); }
  friend StringInput operator>> ( pair<StringInput,BuffModel*> si_m, const char* psD ) {
    StringInput si;
    return ( 0 ||
             //(si=si_m.first>>"BF ">>si_m.second->mBF>>psD)!=NULL ||
             (si=si_m.first>>"BB ">>si_m.second->mBB>>psD)!=NULL ||
             (si=si_m.first>>"BM ">>si_m.second->mBM>>psD)!=NULL ||
//             (si=si_m.first>>"Bg ">>si_m.second->mBG>>psD)!=NULL ||
//             (si=si_m.first>>"WD ">>si_m.second->mW_DUMMY>>psD)!=NULL ||
             0) ? si : StringInput(NULL);
//    return StringInput(NULL);
  }
};
//HidVarCPT2DModel<Ind,Ind,LogProb> BuffModel::mBG;
//HidVarCPT2DModel<Wd,Wd,LogProb> BuffModel::mW_DUMMY;
//HidVarCPT2DModel<Ind,Ind,LogProb> BuffModel::mBF;
HidVarCPT2DModel<Wd,Wd,LogProb> BuffModel::mW_COPY;
HidVarCPT2DModel<Wd,Wd,LogProb> BuffModel::mW_CLEAR;
HidVarCPT4DModel<Wd,G,B,Wd,LogProb> BuffModel::mW_SUBCOPY;
HidVarCPT2DModel<Ind,Ind,LogProb> BuffModel::mIND_COPY;
HidVarCPT4DModel<Wd,G,B,Wd,LogProb> BuffModel::mW_ALIGNCOPY;
float BuffModel::subParam = 0.32;

class SIModel : public DoubleFactoredModel<SModel,IModel>{
};

class IBModel : public DoubleFactoredModel<IModel,BuffModel>{
};

class SIBModel : public DoubleFactoredModel<SModel,IBModel>{
};

//////////////////// Overall...

//// Model of H=R,SIB given SIB
class HModel : public DoubleFactoredModel<RModel,SIBModel> {
 public:
  static Wd WORD;
  static bool& F_ROOT_OBS;
  typedef H::ArrayIterator<LogProb> IterVal;
  SIB& setTrellDat ( SIB& s, const H::ArrayIterator<LogProb>& h ) const {
    s.setVal(h.second);
    return s;
  }
  R setBackDat ( const H::ArrayIterator<LogProb>& h ) const {
    R r;
    for(int i=0;i<4;i++)
      r.set(i)=F(h.first.get(i));
    return r;
  }
  LogProb setIterProb ( H::ArrayIterator<LogProb>& h, const SIB& sibP, int& a ) const {
    const RModel& mR = getM1();
    const SModel& mS = getM2().getM1();
    const IModel& mI = getM2().getM2().getM1();
    const BuffModel& mB = getM2().getM2().getM2();

    LogProb pr;
//cerr << "Startig iteration... sibP = " << sibP << ", a = " << a << endl;
    pr = mI.setIterProb( h.second.second.first, sibP.second.first, a);
//cerr << "  Done with imodel: " << I(h.second.second.first) << endl;
    pr *= mR.setIterProb ( h.first, I(h.second.second.first), sibP, a );
//cerr << "  done with rmodel: " << endl;
    pr *= mS.setIterProb ( h.second.first, h.first, I(h.second.second.first), sibP, a );
//cerr << "  done with smodel: " << endl;
    pr *= mB.setIterProb( h.second.second.second, I(h.second.second.first), h.second.first, h.first, sibP, a);
//cerr << "  done with buffmodel:" << endl; 
//cerr << "  h = " << h << endl;
    return pr;
  }
  void update ( ) const { }
};
Wd    HModel::WORD;
bool& HModel::F_ROOT_OBS = FModel::F_ROOT_OBS;

