from dotenv import load_dotenv
from genai import Credentials, Client
from utils.chroma_utils import query_chroma
from genai.schema import TextGenerationParameters, TextGenerationReturnOptions
import json
import spotipy

def construct_gofa_prompt_tmdb_examples(apis, utterance):
    preamble = """I will ask you to perform a task, your job is to come up with a sequence of APIs that will perform the task.
You should use the APIs below to answer the question posed of you.

Available APIs: A list of randomly arranged available APIs with definitions to describe the purpose of each."""
    example_apis = """- /search/person: Search for people.
- /person/{person_id}/movie_credits: Get the movie credits for a person, the results contains various information such as popularity and release date.
- /search/movie: Search for movies.
- /movie/{movie_id}/credits: Get the cast and crew for a movie.
- /movie/top_rated: Get the top rated movies on TMDb.
{{tools}}
"""
    examples = """
Use the following format:
Task: give me the number of movies directed by Sofia Coppola
Sequence: /search/person, /person/{person_id}/movie_credits

Task: Who was the lead actor in the movie The Dark Knight?
Sequence: /search/movie, /movie/{movie_id}/credits

Task: Who directed the top-1 rated movie?
Sequence: /movie/top_rated, /movie/{movie_id}/credits

Task: {{utterance}}
Sequence:"""

    tools = ""
    for path in apis:
        tools += "- " + path + ": " + apis[path].replace("\n\n", " ") + "\n"
    prompt = "\n".join([preamble, example_apis, examples])
    prompt = prompt.replace("{{tools}}", tools)
    prompt = prompt.replace("{{utterance}}", utterance)
    return prompt

def construct_gofa_prompt_spotify_examples(apis, utterance):
    preamble = """I will ask you to perform a task, your job is to come up with a sequence of APIs that will perform the task.
You should use the APIs below to answer the question posed of you.

Available APIs: A list of randomly arranged available APIs with definitions to describe the purpose of each."""
    example_apis = """- /search: Get Spotify catalog information about albums, artists, playlists, tracks, shows, episodes or audiobooks\nthat match a keyword string.<br />\n**Note: Audiobooks are only available for the US, UK, Ireland, New Zealand and Australia markets.**\n
- /me/player/currently-playing: Get the track currently being played on the user's Spotify account.\n
- /tracks/{id}: Get Spotify catalog information for a single track identified by its\nunique Spotify ID.\n
- /artists/{id}/albums: Get Spotify catalog information about an artist's albums.\n
{{tools}}
"""
    examples = """
Use the following format:
Task: show me the album information of the song currently playing
Sequence: /me/player/currently-playing, /tracks/{id}

Task: show me the name of Maroon 5's newest album
Sequence: /search, /artists/{id}/albums

Task: {{utterance}}
Sequence:"""

    tools = ""
    for path in apis:
        tools += "- " + path + ": " + apis[path].replace("\n\n", " ") + "\n"
    prompt = "\n".join([preamble, example_apis, examples])
    prompt = prompt.replace("{{tools}}", tools)
    prompt = prompt.replace("{{utterance}}", utterance)
    return prompt

def construct_gofa_prompt_gofa_examples(apis, utterance):
    preamble = """I will ask you to perform a task, your job is to come up with a sequence of APIs that will perform the task.
You should use the APIs below to answer the question posed of you.

Available APIs: A list of randomly arranged available APIs with definitions to describe the purpose of each."""
    example_apis = """- trello_card_retrievewithwhere_Card: Retrieve all cards from Trello
- trello_member_retrievewithwhere_Member: Retrieve members from Trello
- jira_issue_create_Issue: Create operation for jira Issue
- jira_issue_retrievewithwhere_Issue: Retrieve all Jira issues
- slack_message_create_message: A new message is posted in Slack
- box_file_create_File: Create a file in a specified folder in your Box account
{{tools}}
"""
    examples = """
Use the following format:
Task: Create a file in Box for a new Jira issue
Sequence: jira_issue_create_Issue, jira_issue_retrievewithwhere_Issue, box_file_create_File

Task: Get all trello card members and send a slack message with the text "Hello Trello card member!" to all
Sequence: trello_card_retrievewithwhere_Card, trello_member_retrievewithwhere_Member, slack_message_create_message

Task: {{utterance}}
Sequence:"""
    tools = ""
    for path in apis:
        tools += "- " + path + ": " + apis[path].replace("\n\n", " ") + "\n"
    prompt = "\n".join([preamble, example_apis, examples])
    prompt = prompt.replace("{{tools}}", tools)
    prompt = prompt.replace("{{utterance}}", utterance)
    return prompt

def get_bam_response(prompt, model_id):
    load_dotenv()
    credentials = Credentials.from_env()
    parameters = TextGenerationParameters(decoding_method="greedy", max_new_tokens=100, temperature=0.05, stop_sequences=["\n\n"], include_stop_sequence=False)
    client = Client(credentials=credentials)
    responses = list(
        client.text.generation.create(
            model_id=model_id,
            inputs=[prompt],
            parameters=parameters,
        )
    )
    return responses[0].results[0].generated_text

def get_apis(collection_name, utterance, k=10, model_id="meta-llama/llama-2-70b"):
    query_text = [utterance]
    cr_querying_results = query_chroma(collection_name, query_text, recall_limit=k)
    apis = cr_querying_results[0][0]
    if collection_name=="tmdb":
        prompt = construct_gofa_prompt_tmdb_examples(apis, utterance)
    elif collection_name=="spotify":
        prompt = construct_gofa_prompt_spotify_examples(apis, utterance)
    response = get_bam_response(prompt, model_id)
    output = response.strip().split(", ")
    return output

def get_spotify_access_token(spec_path):

    with open(spec_path) as f:
        raw_api_spec = json.load(f)

    scopes = list(raw_api_spec['components']['securitySchemes']['oauth_2_0']['flows']['authorizationCode']['scopes'].keys())
    access_token = spotipy.util.prompt_for_user_token(scope=','.join(scopes))
    headers = {
        'Authorization': f'Bearer {access_token}'
    }
    return access_token


