"""Class implementing a beginner user. Basically maps the error-rates 
for different gaps and returns for a given gap its error rate.
"""
import json
import random

random.seed(42)

class Simulated_Learner():
    #######################################
    #     Initialize, save, and load
    #######################################
    def __init__(self):
        """Initialize the learner"""
        self.gap_error_rate = dict()
        self.overall_error_rate = 0.0

    def create_learner_from_data(self, training_data):
        """Create a learner model based on the given input data and the specified error rate"""
        # Read data and extract all answers of users which have the according error rate
        data = []
        with open(training_data, 'r') as lines:
            next(lines)
            for line in lines:
                splits=line.strip().split('\t')
                try:
                    splits[3] 
                except IndexError:
                    print(line)
                # Questions + Answers - col 4-12
                data.append(splits[4:])
        # Start extracting the individual gaps and add them to the error rate:
        for participant in data:
            for question, answer in self.chunks(participant,2):
                if 'TheThe' in question:
                    continue
                answer_keys = answer.split()
                answer_index = 0
                gaps = [token for token in question.split() if '#GAP#' in token] 
                try:
                    assert(len(gaps) == len(answer.split()))
                except AssertionError:
                    print("Number of gaps are not matching the number of answers!")
                    print(question, answer)         

                # Check if the gap is already in the dictionary, add its appropriate entry
                # Check if the gap was answered correctly or incorrectly
                for token in question.split():
                    if "#GAP#" in token:
                        last_half = answer_keys[answer_index].split(':')[-1].split('#')[0]
                        gap_token = self.clean_errors(token.replace('#GAP#',last_half))
                        learner_answer = token.replace('#GAP#',answer_keys[answer_index].split('#')[1])
                        try:
                            self.gap_error_rate[gap_token]['total_number'] += 1
                        except KeyError:
                            self.gap_error_rate[gap_token] = {'total_number':1, 'errors':0, 'correct':0}
                        if gap_token == learner_answer:
                            self.gap_error_rate[gap_token]['correct'] += 1
                        else:
                            self.gap_error_rate[gap_token]['errors'] += 1
                        answer_index += 1 
        self.only_error_rates = dict()
        for gap,values in self.gap_error_rate.items():
            self.only_error_rates[gap] = float(values['errors'])/float(values['total_number'])
        print(self.only_error_rates.items())
        # Compute the overall error rate as the average over all word error rates
        self.overall_error_rate = sum([error for word, error in self.only_error_rates.items()])
        self.overall_error_rate = self.overall_error_rate / len(self.only_error_rates.items())
        print("Loaded training data. Overall error rate is: {}".format(self.overall_error_rate))
        # Add the overall error rate to the dictionary:
        self.only_error_rates['overall_error_rate'] = self.overall_error_rate

    def store_learner_model(self, target_location):
        """ Save a the current learner model"""
        with open(target_location,'w') as store:
            json.dump(self.only_error_rates, store)

    def load_learner_model(self, source_location):
        """ Load a specific learner model"""
        with open(source_location, 'r') as store:
            self.only_error_rates = json.load(store)
        self.overall_error_rate = self.only_error_rates['overall_error_rate']

    def set_random_seed(number):
        random.seed(number)

    #######################################
    #        Do gap prediction
    #######################################
    
    def predict_gap(self, word):
        """Return true, if the learner answers the gap correctly. Else, false."""
        random_number = random.uniform(0.0,1.0)
        try:
            false_probs = self.only_error_rates[word]
        except KeyError:
            false_probs = self.overall_error_rate 
        return random_number < 1.0 - false_probs

    def is_in_vocabulary(self, word):
        """Return true, if the word is in the vocabulary of the learner """
        try:
            self.only_error_rates[word]
            return True
        except KeyError:
            return False

    def delta_error_rate(self, word):
        """Return the delta of the average and this words error-rate.
        If the word is not in the vocabulary, return 0.0 """
        try:
            return self.overall_error_rate - self.only_error_rates[word]
        except KeyError:
            return 0.0


    #######################################
    #        Helper functions
    #######################################

    def chunks(self, l, n):
        """Yield successive n-sized chunks from l."""
        for i in range(0, len(l), n):
            yield l[i:i + n]

    def clean_errors(self, token):
        """Function cleaning up errors in the source data using a lookup list"""
        if token == 'haas':
            return 'has'
        if token == 'senrse':
            return 'sense'
        if token == 'couse':
            return'course'
        if token == 'strettches':
            return 'stretches'
        if token == 'sleeep':
            return 'sleep'
        if token == 'pereive':
            return 'perceive'
        if token == 'peopple':
            return 'people'
        if token == 'beliieve':
            return 'believe'
        if token == 'schoools':
            return 'schools'
        if token == 'larrge':
            return 'large'
        if token == 'knowedge':
            return 'knowledge'
        if token == 'son':
            return 'soon'
        return token


