from __future__ import division
import torch
import torch.nn as nn
import torch.nn.functional as F
from models.copy_generator import CopyGeneratorLoss


def build_loss_compute(model, vocab, gpu):
    """
    Returns a LossCompute subclass which wraps around an nn.Module subclass
    (such as nn.NLLLoss) which defines the loss criterion. The LossCompute
    object allows this loss to be computed in shards and passes the relevant
    data to a Statistics object which handles training/validation logging.
    Currently, the NMTLossCompute class handles all loss computation except
    for when using a copy mechanism.
    """
    device = torch.device("cuda" if gpu else "cpu")

    copy_attn_force = False
    criterion = CopyGeneratorLoss(
        len(vocab), copy_attn_force,
        unk_index=0, ignore_index=1
    )

    loss_gen = model.generator
    copy_loss_by_seqlength = True
    compute = CopyGeneratorLossCompute(
        criterion, loss_gen, vocab, copy_loss_by_seqlength
    )
    compute.to(device)

    return compute

class LossComputeBase(nn.Module):
    """
    Class for managing efficient loss computation. Handles
    sharding next step predictions and accumulating multiple
    loss computations
    Users can implement their own loss computation strategy by making
    subclass of this one.  Users need to implement the _compute_loss()
    and make_shard_state() methods.
    Args:
        generator (:obj:`nn.Module`) :
             module that maps the output of the decoder to a
             distribution over the target vocabulary.
        tgt_vocab (:obj:`Vocab`) :
             torchtext vocab object representing the target output
        normalzation (str): normalize by "sents" or "tokens"
    """

    def __init__(self, criterion, generator):
        super(LossComputeBase, self).__init__()
        self.criterion = criterion
        self.generator = generator

    @property
    def padding_idx(self):
        return self.criterion.ignore_index

    def _make_shard_state(self, batch, output, range_, attns=None, scales=None):
        """
        Make shard state dictionary for shards() to return iterable
        shards for efficient loss computation. Subclass must define
        this method to match its own _compute_loss() interface.
        Args:
            batch: the current batch.
            output: the predict output from the model.
            range_: the range of examples for computing, the whole
                    batch or a trunc of it?
            attns: the attns dictionary returned from the model.
        """
        return NotImplementedError

    def _compute_loss(self, batch, output, target, **kwargs):
        """
        Compute the loss. Subclass must define this method.
        Args:
            batch: the current batch.
            output: the predict output from the model.
            target: the validate target to compare output with.
            **kwargs(optional): additional info for computing loss.
        """
        return NotImplementedError

    def __call__(self,
                 batch,
                 output,
                 attns,
                 normalization=1.0,
                 shard_size=0,
                 trunc_start=0,
                 trunc_size=None,
                 retain_graph=False,
                 scales=None):
        """Compute the forward loss, possibly in shards in which case this
        method also runs the backward pass and returns ``None`` as the loss
        value.
        Also supports truncated BPTT for long sequences by taking a
        range in the decoder output sequence to back propagate in.
        Range is from `(trunc_start, trunc_start + trunc_size)`.
        Note sharding is an exact efficiency trick to relieve memory
        required for the generation buffers. Truncation is an
        approximate efficiency trick to relieve the memory required
        in the RNN buffers.
        Args:
          batch (batch) : batch of labeled examples
          output (:obj:`FloatTensor`) :
              output of decoder model `[tgt_len x batch x hidden]`
          attns (dict) : dictionary of attention distributions
              `[tgt_len x batch x src_len]`
          normalization: Optional normalization factor.
          shard_size (int) : maximum number of examples in a shard
          trunc_start (int) : starting position of truncation window
          trunc_size (int) : length of truncation window
        Returns:
            A tuple with the loss and a :obj:`onmt.utils.Statistics` instance.
        """
        if trunc_size is None:
            trunc_size = batch["tgt"].size(0) - trunc_start # tgt_max_len
        trunc_range = (trunc_start, trunc_start + trunc_size) # (0, tgt_max_len)
        shard_state = self._make_shard_state(batch, output, trunc_range, attns, scales)
        if shard_size == 0:
            loss = self._compute_loss(batch, **shard_state)
            return loss / float(normalization)
        # batch_stats = onmt.utils.Statistics()
        loss = 0.
        for shard in shards(shard_state, shard_size, retain_graph=retain_graph):
            shard_loss = self._compute_loss(batch, **shard)
            loss += shard_loss.div(float(normalization))
        return loss

    # def _stats(self, loss, scores, target):
    #     """
    #     Args:
    #         loss (:obj:`FloatTensor`): the loss computed by the loss criterion.
    #         scores (:obj:`FloatTensor`): a score for each possible output
    #         target (:obj:`FloatTensor`): true targets
    #     Returns:
    #         :obj:`onmt.utils.Statistics` : statistics for this batch.
    #     """
    #     pred = scores.max(1)[1]
    #     non_padding = target.ne(self.padding_idx)
    #     num_correct = pred.eq(target).masked_select(non_padding).sum().item()
    #     num_non_padding = non_padding.sum().item()
    #     return onmt.utils.Statistics(loss.item(), num_non_padding, num_correct)

    def _bottle(self, _v):
        return _v.view(-1, _v.size(2))

    def _unbottle(self, _v, batch_size):
        return _v.view(-1, batch_size, _v.size(1))

class CopyGeneratorLossCompute(LossComputeBase):
    """Copy Generator Loss Computation."""

    def __init__(self, criterion, generator, tgt_vocab, normalize_by_length):
        super(CopyGeneratorLossCompute, self).__init__(criterion, generator)
        self.tgt_vocab = tgt_vocab
        self.normalize_by_length = normalize_by_length

    def _make_shard_state(self, batch, output, range_, attns, scales=None):
        """See base class for args description."""

        return {
            "output": output,
            "copy": attns.get("copy"),
            "target": batch["tgt"][range_[0] + 1: range_[1], :, 0],
            "tgt_copy_attn": attns.get("tgt_std"),
            "context_copy_attn": attns.get("context_std"),
            "abs_align": batch["abs_align"][range_[0] + 1: range_[1]],
            "context_align": batch["context_align"][range_[0] + 1: range_[1]],
            "scales": scales
        }

    def _compute_loss(self, batch, output, copy, target, tgt_copy_attn, context_copy_attn, abs_align, context_align, scales=None):
        """Compute the loss.
        The args must match :func:`self._make_shard_state()`.
        Args:
            batch: the current batch.
            output: the predict output from the model.
            target: the validate target to compare output with.
            copy_attn: the copy attention value.
            align: the align info.
        """

        # target = target.view(-1)
        target = target.reshape(-1)
        abs_align = abs_align.reshape(-1)
        context_align = context_align.reshape(-1)

        out_scores, abs_scores, context_scores = self.generator(
            self._bottle(output), self._bottle(copy), self._bottle(tgt_copy_attn),
            self._bottle(context_copy_attn), batch["abs_src_map"], batch["context_src_map"]
        )
        out, loss = self.criterion(out_scores, abs_scores, context_scores, abs_align, context_align, target)

        # this block does not depend on the loss value computed above
        # and is used only for stats
        # scores_data = collapse_copy_scores(
        #     self._unbottle(scores.clone(), batch.batch_size),
        #     batch, self.tgt_vocab, batch.dataset.src_vocabs)
        # scores_data = self._bottle(scores_data)

        # this block does not depend on the loss value computed above
        # and is used only for stats
        # Correct target copy token instead of <unk>
        # tgt[i] = align[i] + len(tgt_vocab)
        # for i such that tgt[i] == 0 and align[i] != 0
        # target_data = target.clone()
        # unk = self.criterion.unk_index
        # correct_mask = (target_data == unk) & (align != unk)
        # offset_align = align[correct_mask] + len(self.tgt_vocab)
        # target_data[correct_mask] += offset_align

        # Compute sum of perplexities for stats
        # stats = self._stats(loss.sum().clone(), scores_data, target_data)

        # this part looks like it belongs in CopyGeneratorLoss
        if self.normalize_by_length:
            # Compute Loss as NLL divided by seq length
            tgt_lens = batch["tgt"][:, :, 0].ne(self.padding_idx).sum(0).float()
            # Compute Total Loss per sequence in batch
            loss = loss.view(-1, batch["batch_size"]).sum(0)
            # Divide by length of each sequence and sum
            loss = torch.div(loss, tgt_lens)
            if scales is not None:
                loss = loss.sum(0) * scales
            loss = loss.sum()
        else:
            # 9 * 64, 32, 64
            if scales is not None:
                loss = loss.view(-1, batch["batch_size"]).sum(0) * scales

            loss = loss.sum()
        return loss

def filter_shard_state(state, shard_size=None):
    for k, v in state.items():
        if shard_size is None:
            yield k, v

        if v is not None:
            v_split = []
            if isinstance(v, torch.Tensor):
                for v_chunk in torch.split(v, shard_size):
                    v_chunk = v_chunk.data.clone()
                    v_chunk.requires_grad = v.requires_grad
                    v_split.append(v_chunk)
            yield k, (v, v_split)


def shards(state, shard_size, eval_only=False, retain_graph=False):
    """
    Args:
        state: A dictionary which corresponds to the output of
               *LossCompute._make_shard_state(). The values for
               those keys are Tensor-like or None.
        shard_size: The maximum size of the shards yielded by the model.
        eval_only: If True, only yield the state, nothing else.
              Otherwise, yield shards.
    Yields:
        Each yielded shard is a dict.
    Side effect:
        After the last shard, this function does back-propagation.
    """
    if eval_only:
        yield filter_shard_state(state)
    else:
        # non_none: the subdict of the state dictionary where the values
        # are not None.
        non_none = dict(filter_shard_state(state, shard_size))

        # Now, the iteration:
        # state is a dictionary of sequences of tensor-like but we
        # want a sequence of dictionaries of tensors.
        # First, unzip the dictionary into a sequence of keys and a
        # sequence of tensor-like sequences.
        keys, values = zip(*((k, [v_chunk for v_chunk in v_split])
                             for k, (_, v_split) in non_none.items()))

        # Now, yield a dictionary for each shard. The keys are always
        # the same. values is a sequence of length #keys where each
        # element is a sequence of length #shards. We want to iterate
        # over the shards, not over the keys: therefore, the values need
        # to be re-zipped by shard and then each shard can be paired
        # with the keys.
        for shard_tensors in zip(*values):
            yield dict(zip(keys, shard_tensors))

        # Assumed backprop'd
        variables = []
        for k, (v, v_split) in non_none.items():
            if isinstance(v, torch.Tensor) and state[k].requires_grad:
                variables.extend(zip(torch.split(state[k], shard_size),
                                     [v_chunk.grad for v_chunk in v_split]))
        inputs, grads = zip(*variables)
        torch.autograd.backward(inputs, grads, retain_graph=retain_graph)