import React, { Component } from 'react';
import { withTracker } from 'meteor/react-meteor-data';
import { Random } from 'meteor/random'
import { Link } from "react-router-dom";

import { Assignments, Tutorials, Annotations } from '../api/documents';
import { gup } from '../util/gup.js';

export class Demo extends Component {
  constructor(props) {
    super(props);

    const names = [ // 100 most popular names in the US over the last century
      "Aaron", "Abigail", "Adam", "Alan", "Albert", "Alexander", "Alexis",
      "Alice", "Amanda", "Amber", "Amy", "Andrea", "Andrew", "Angela", "Ann",
      "Anna", "Anthony", "Arthur", "Ashley", "Austin", "Barbara", "Benjamin",
      "Betty", "Beverly", "Billy", "Bobby", "Brandon", "Brenda", "Brian",
      "Brittany", "Bruce", "Bryan", "Carl", "Carol", "Carolyn", "Catherine",
      "Charles", "Cheryl", "Christian", "Christina", "Christine",
      "Christopher", "Cynthia", "Daniel", "Danielle", "David", "Deborah",
      "Debra", "Denise", "Dennis", "Diana", "Diane", "Donald", "Donna",
      "Doris", "Dorothy", "Douglas", "Dylan", "Edward", "Elizabeth", "Emily",
      "Emma", "Eric", "Ethan", "Eugene", "Evelyn", "Frances", "Frank",
      "Gabriel", "Gary", "George", "Gerald", "Gloria", "Grace", "Gregory",
      "Hannah", "Harold", "Harry", "Heather", "Helen", "Henry", "Jack",
      "Jacob", "Jacqueline", "James", "Jane", "Janet", "Janice", "Jason",
      "Jean", "Jeffrey", "Jennifer", "Jeremy", "Jerry", "Jesse", "Jessica",
      "Joan", "Joe", "John", "Johnny", "Jonathan", "Jordan", "Jose", "Joseph",
      "Joshua", "Joyce", "Juan", "Judith", "Judy", "Julia", "Julie", "Justin",
      "Karen", "Katherine", "Kathleen", "Kathryn", "Kayla", "Keith", "Kelly",
      "Kenneth", "Kevin", "Kimberly", "Kyle", "Larry", "Laura", "Lauren",
      "Lawrence", "Linda", "Lisa", "Logan", "Lori", "Louis", "Madison",
      "Margaret", "Maria", "Marie", "Marilyn", "Mark", "Martha", "Mary",
      "Matthew", "Megan", "Melissa", "Michael", "Michelle", "Nancy", "Natalie",
      "Nathan", "Nicholas", "Nicole", "Noah", "Olivia", "Pamela", "Patricia",
      "Patrick", "Paul", "Peter", "Philip", "Rachel", "Ralph", "Randy",
      "Raymond", "Rebecca", "Richard", "Robert", "Roger", "Ronald", "Rose",
      "Roy", "Russell", "Ruth", "Ryan", "Samantha", "Samuel", "Sandra", "Sara",
      "Sarah", "Scott", "Sean", "Sharon", "Shirley", "Stephanie", "Stephen",
      "Steven", "Susan", "Teresa", "Terry", "Theresa", "Thomas", "Tiffany",
      "Timothy", "Tyler", "Victoria", "Vincent", "Virginia", "Walter", "Wayne",
      "William", "Willie", "Zachary",];

    this.state = {
      workerId: gup("workerId", Random.choice(names) + String(Random.fraction() * 1000).split(".")[0]),
      assignmentId: gup("assignmentId", Random.id()),
      hitId: gup("hitId", Random.id()),
      action: gup("action", "annotate"),
      annoId: gup("annoId"),
      textId: gup("textId"),
      tutId: gup("tutId"),
      ui: gup("ui", "actions"),
      instructions: gup("instructions"),
      internalAssignment: gup("internalAssignment"),
      group: "",
    };

    this.handleInputChange = this.handleInputChange.bind(this);
    this.handleSelection = this.handleSelection.bind(this);
    this.handleSubmit = this.handleSubmit.bind(this);
    this.formURL = this.formURL.bind(this);
    this.renderGroups = this.renderGroups.bind(this);
    this.renderUISelection = this.renderUISelection.bind(this);
    this.renderUserInfo = this.renderUserInfo.bind(this);
    this.renderAssignments = this.renderAssignments.bind(this);
    this.renderActions = this.renderActions.bind(this);
    this.renderAnnotations = this.renderAnnotations.bind(this);
    this.renderTutorials = this.renderTutorials.bind(this);
  }

  handleInputChange(event) {
    const target = event.target;
    const value = target.value;
    const name = target.name;

    this.setState({
      [name]: value
    });
  }

  handleSelection(event) {
    const target = event.target;
    const value = target.value;
    const name = target.name;

    this.setState({
      [name]: value,
    });

    if (name == 'action') {
      if (value.includes('tutorial')) {
        const tutorials = this.props.tutorials.filter(
          tutorial => tutorial.ui == this.state.ui
        ).map(
          tutorial => tutorial._id
        );
        const tutorial = tutorials.length > 0 ? tutorials[0] : "";

        this.setState({
          internalAssignment: "",
          textId: "",
          tutId: tutorial,
        });
      } else if (value == 'annotate') {
        this.setState({
          internalAssignment: "",
          textId: "",
          tutId: "",
          annoId: "",
        });
      } else if (value == 'view-annotation') {
        const assignments = this.props.assignments.filter(assignment => {
          return assignment.ui == this.state.ui;
        });
        if (assignments.length > 0) {
          const assignment = assignments[0];
          this.setState({
            textId: assignment.text_id,
            internalAssignment: assignment._id,
            tutId: "",
            annoId: "",
          });
        } else {
          this.setState({
            textId: "",
            internalAssignment: "",
            tutId: "",
            annoId: "",
          });
        }
      } else if (value == 'view-tutorial-annotations') {
        const tutorials = this.props.tutorials.filter(tutorial => {
          return tutorial.ui == this.state.ui;
        });
        if (tutorials.length > 0) {
          const tutorial = tutorials[0];
          this.setState({
            textId: "",
            internalAssignment: "",
            tutId: tutorial,
            annoId: "",
          });
        } else {
          this.setState({
            textId: "",
            internalAssignment: "",
            tutId: "",
            annoId: "",
          });
        }
      }
    } else if (name == 'internalAssignment') {
      if (value == '') {
        this.setState({
          textId: "",
          instructions: "",
        });
      } else {
        const assignment = this.props.assignments.find(assignment => {
          return assignment._id == value;
        });
        this.setState({
          textId: assignment.text_id,
          instructions: assignment.instructions,
          ui: assignment.ui,
        });
      }
    } else if (name == "group") {
      this.setState({
        action: 'annotate',
        internalAssignment: "",
      });
    }
  }

  handleSubmit(event) {
    if (this.state.action == 'annotate') {
      if (this.state.internalAssignment != '') {
        const assignment = this.props.assignments.find(
          a => a._id == this.state.internalAssignment);
        const annotation = this.props.annotations.find(anno => {
          return anno._id == assignment.ann_id;
        });
        Meteor.call('assignments.assign', assignment._id, assignment.text_id, this.state.workerId, annotation.clusters, annotation.labels, annotation.checks, annotation.focus_mention);
      }
    }
  }

  formURL() {
    if (this.state.action == 'annotate') {
      if (this.state.internalAssignment != '') {
        const assignment = this.props.assignments.find(
          a => a._id == this.state.internalAssignment);
        const annotation = this.props.annotations.find(anno => {
          return anno._id == assignment.ann_id;
        });
        if (annotation != undefined) {
          return "/annotate/" + this.state.ui;
        }
      }
      return '/assign/'+ this.state.group;
    } else if (this.state.action.startsWith('view')) {
      return '/view/' + this.state.ui;
    } else {
      return '/tutorial/'+ this.state.tutId;
    }
  }

  renderGroups() {
    const groups = new Set(
      this.props.assignments.map(assignment => assignment.group)
    );
    const groupList = Array.from(groups).map(group => {
      return (
        <option key={group} value={group}>Group: {group}</option>
      );
    });
    return (
      <label>
        Select the group:
        <select value={this.state.group} onChange={this.handleSelection} name="group">
          <option value="">-</option>
          {groupList}
        </select>
      </label>
    );
  }

  renderUISelection() {
    if (this.state.group != "") {
      return "";
    }
    return (
      <label>
        Select the UI:
        <select value={this.state.ui} onChange={this.handleSelection} name="ui">
          <option value="actions">actions</option>
          <option value="check-mentions-exclude">check-mentions-exclude</option>
          <option value="check-mentions-exclude-editable">check-mentions-exclude-editable</option>
          <option value="check-mentions-include">check-mentions-include</option>
          <option value="check-mentions-include-editable">check-mentions-include-editable</option>
          <option value="sentiment">sentiment</option>
          <option value="conditionals">conditionals</option>
          <option value="conditionals-from-actions">conditionals-from-actions</option>
          <option value="one-entity-at-a-time">one-entity-at-a-time</option>
          <option value="full">full</option>
          <option value="link-to-one-after">link-to-one-after</option>
          <option value="link-to-one-before">link-to-one-before</option>
          <option value="mentions">mentions</option>
          <option value="freq-ents">freq-ents</option>
        </select>
      </label>
    );
  }

  renderAssignments() {
    const assignments = this.props.assignments.filter(assignment => {
      if (this.state.group == "") {
        return assignment.ui == this.state.ui;
      } else {
        return assignment.group == this.state.group;
      }
    }).map(assignment => {
      return (
        <option key={assignment._id} value={assignment._id}>{assignment.text_id} - {assignment.ann_id} - {assignment.group} - {assignment.count}</option>
      );
    });
    return (
      <label>
        Select the assignment:
        <select value={this.state.internalAssignment} onChange={this.handleSelection} name="internalAssignment">
          {this.state.action == 'annotate' ? (<option value="">assign-random</option>) : ""}
          {assignments}
        </select>
      </label>
    );
  }

  renderTutorials() {
    const tutorials = this.props.tutorials.filter(tutorial => {
      return tutorial.ui == this.state.ui;
    }).map(tutorial => {
      return (
        <option key={tutorial._id} value={tutorial._id}>Tutorial {tutorial._id}</option>
      );
    });

    return (
      <label>
        Select the tutorial:
        <select value={this.state.tutId} onChange={this.handleSelection} name="tutId">
          {tutorials}
        </select>
      </label>
    );
  }

  renderActions() {
    if (this.state.group != "") {
      return "";
    }
    return (
      <label>
        Select the task:
        <select value={this.state.action} onChange={this.handleSelection} name="action">
          <option value="annotate">annotate</option>
          <option value="view-annotation">view-annotation</option>
          <option value="tutorial">tutorial</option>
          <option value="view-tutorial-annotations">view-tutorial-annotations</option>
        </select>
      </label>
    );
  }

  renderAnnotations() {
    if (! this.state.action.includes('view')) {
      return "";
    } else {
      const textIds = this.state.tutId == "" ? [this.state.textId] : this.props.tutorials.filter(t => t._id == this.state.tutId)[0].tasks.map(t => t.text_id);
      const annotations = this.props.annotations.filter(annotation => {
        return textIds.includes(annotation.text_id);
      }).sort((a, b) => {
        const a_length = a.clusters.filter( cluster => {
          return cluster.mentions.some( mention => {
            return mention.label != "";
          });
        }).length;
        const b_length = b.clusters.filter( cluster => {
          return cluster.mentions.some( mention => {
            return mention.label != "";
          });
        }).length;
        if (a_length > b_length) return -1;
        else if (a_length < b_length) return 1;
        return ('' + a.user).localeCompare(b.user)
      }).map(annotation => {
        const length = annotation.clusters.filter( cluster => {
          return cluster.mentions.some( mention => {
            return mention.label != "";
          });
        }).length;
        return (
          <option key={annotation._id} value={annotation._id}>{annotation.user} - {annotation._id} - {length}</option>
        );
      });

      return (
        <label>
          Select the annotations:
          <select value={this.state.annoId} onChange={this.handleSelection} name="annoId">
            {annotations}
          </select>
        </label>
      );
    }
  }

  renderUserInfo() {
    if (this.state.action == 'view') {
      return "";
    } else {
      return (
        <div>
          <br />
          <label>
            WorkerID (your username):
            <input
              name="workerId"
              type="text"
              value={this.state.workerId}
              onChange={this.handleInputChange} />
          </label>
          <br />
          <label>
            AssignmentID (anything you like):
            <input
              name="assignmentId"
              type="text"
              value={this.state.assignmentId}
              onChange={this.handleInputChange} />
          </label>
          <br />
          <label>
            HitID (anything you like):
            <input
              name="hitId"
              type="text"
              value={this.state.hitId}
              onChange={this.handleInputChange} />
          </label>
        </div>
      );
    }
  }

  render() {
    if (! this.props.areReady) {
        return (
          <div>Loading...</div>
        );
    }
    return (
      <div>
        <form onSubmit={this.handleSubmit} action={this.formURL()} >
          {this.renderGroups()}
          {this.renderUISelection()}
          {this.renderActions()}
          {this.state.action.includes('tutorial') ? this.renderTutorials() : this.renderAssignments()}
          {this.renderAnnotations()}
          {this.state.action.includes('view') ? "" : this.renderUserInfo()}
          <br />
          <input
            name="instructions"
            type="hidden"
            value={this.state.instructions}
            onChange={this.handleInputChange} />
          <input
            name="textId"
            type="hidden"
            value={this.state.textId}
            onChange={this.handleInputChange} />
          <input type="submit" className="button" value="Go!" />
        </form>
        <p>
          Current link: {this.formURL()}
        </p>
      </div>
    );
  }
}

export default DocContainer = withTracker(() => {
  const handles = [
    Meteor.subscribe('tutorials-all'),
    Meteor.subscribe('annotations-all'),
    Meteor.subscribe('assignments-all'),
  ];

  return {
    tutorials: Tutorials.find().fetch(),
    annotations: Annotations.find().fetch(),
    assignments: Assignments.find().fetch(),
    areReady: handles.every(handle => handle.ready()),
  };
})(Demo);
