# -*- coding: utf-8 -*-

import contextlib
import gzip
import logging
import os
import sys
import traceback

ENV_LEVEL_VAR = 'LOGGING_LEVEL'
ENV_VERBOSE_VAR = 'LOGGING_VERBOSE'

BLACK, RED, GREEN, YELLOW, BLUE, MAGENTA, CYAN, WHITE = range(8)

RESET_SEQ = '\033[0m'
COLOR_SEQ = '\033[1;{:d}m'
FMT_LIST = [
    '%(levelname)s %(message)s',
    '%(levelname)s [%(name)s:%(filename)s:%(lineno)s] %(message)s'
]
COLORS = {
    'WARNING': YELLOW,
    'INFO': GREEN,
    'DEBUG': BLUE,
    'CRITICAL': CYAN,
    'ERROR': RED
}


class Logger(logging.Logger):
    def log_to_file(self, path, fmt=FMT_LIST[0], **kwargs):
        file_handler = logging.FileHandler(path, **kwargs)
        file_handler.setFormatter(DefaultFormatter(fmt))
        self.addHandler(file_handler)


class DefaultFormatter(logging.Formatter):
    def formatException(self, exec_info):
        exception_type, exception, tb = exec_info
        imported_tb_info = traceback.extract_tb(tb)[-1]
        filename = os.path.relpath(imported_tb_info[0], os.path.curdir)
        lineno = imported_tb_info[1]

        return f'====> {filename}:{lineno}#{exception_type.__name__} {exception}'


class ColoredFormatter(DefaultFormatter):
    def format(self, record):
        levelname = record.levelname
        if levelname in COLORS:
            record.levelname = ''.join([
                COLOR_SEQ.format(30 + COLORS[levelname]),
                levelname,
                RESET_SEQ
            ])
        if record.filename:
            record.filename = ''.join([
                COLOR_SEQ.format(30 + MAGENTA), record.filename, RESET_SEQ
            ])
        return super().format(record)


def getenv(key, default, convert_fn):
    value = os.getenv(key)
    if value is None:
        value = default
    else:
        try:
            value = convert_fn(value)
        except Exception:
            print(f'env "{value}" is invalid. Fallback to "{default}"', file=sys.stderr)
            value = default
    return value


def get_logger(*, files=None, mode='a', console=True, name=None, level=None, verbose=None):
    if level is None:
        level = getenv(ENV_LEVEL_VAR, logging.INFO, lambda x: getattr(logging, x.upper()))

    if verbose is None:
        verbose = getenv(ENV_VERBOSE_VAR, 1, int)

    this_logger = Logger(name or 'logger', level=level)
    this_logger.propagate = False

    fmt = FMT_LIST[max(0, min(1, verbose))]
    if files is not None:
        if isinstance(files, str):
            files = [files]

        assert isinstance(files, (list, tuple))
        for path in files:
            this_logger.log_to_file(path, fmt=fmt, mode=mode)

    if console:
        console_handler = logging.StreamHandler()
        if verbose != 0:
            fmt_class = ColoredFormatter
        else:
            fmt_class = DefaultFormatter
        console_handler.setFormatter(fmt_class(fmt))
        this_logger.addHandler(console_handler)

    return this_logger


class _SrcFiles:
    def __init__(self, *files):
        self.files = files

    def __eq__(self, file):
        return file in self.files

    def __repr__(self):
        return repr(self.files)


LOGGER = get_logger(name='default')
# Skip frames in curring file and standard logging library
logging._srcfile = _SrcFiles(logging._srcfile,
                             contextlib.__file__,
                             os.path.normcase(get_logger.__code__.co_filename))


def open_file(file, mode, logger=LOGGER, **kwargs):
    if file.endswith('gz'):
        open_fn = gzip.open
    else:
        open_fn = open

    if mode is None:
        mode = 'r'
    if 'w' in mode:
        char = '>'
    elif 'a' in mode:
        char = '>>'
    else:
        char = '<'

    if file == '-':
        if char[0] == '>':
            file = 'stdout'
            fp = sys.stdout
        else:  # char[0] == '<'
            file = 'stdin'
            fp = sys.stdin
    else:
        fp = open_fn(file, mode, **kwargs)

    logger.info('%s %s', char, file)
    return fp


@contextlib.contextmanager
def smart_open_file(file, mode, logger=LOGGER, **kwargs):
    if not isinstance(file, str):
        yield file
    else:
        fp = open_file(file, mode, LOGGER, **kwargs)
        if fp is sys.stdin and fp is sys.stdout:
            yield fp
        else:
            try:
                yield fp
            finally:
                fp.close()


def open_wrapper(fn):
    def _open(name, mode=None, **kwargs):
        name = fn(name)
        if mode is not None:
            return open_file(name, mode, **kwargs)
        return name
    return _open
