# Copyright (c) Facebook, Inc. and its affiliates.
#
# This source code is licensed under the MIT license found in the
# LICENSE file in the root directory of this source tree.

import math
import torch
import os

from fairseq import logging, metrics, utils
from fairseq.criterions import FairseqCriterion, register_criterion
from torch.nn import parallel


def label_smoothed_nll_loss(lprobs, target, epsilon, ignore_index=None, reduce=True):
    if target.dim() == lprobs.dim() - 1:
        target = target.unsqueeze(-1)
    nll_loss = -lprobs.gather(dim=-1, index=target)
    smooth_loss = -lprobs.sum(dim=-1, keepdim=True)
    if ignore_index is not None:
        pad_mask = target.eq(ignore_index)
        nll_loss.masked_fill_(pad_mask, 0.)
        smooth_loss.masked_fill_(pad_mask, 0.)
    else:
        nll_loss = nll_loss.squeeze(-1)
        smooth_loss = smooth_loss.squeeze(-1)
    if reduce:
        nll_loss = nll_loss.sum()
        smooth_loss = smooth_loss.sum()
    eps_i = epsilon / lprobs.size(-1)
    loss = (1. - epsilon) * nll_loss + eps_i * smooth_loss
    return loss, nll_loss


@register_criterion('label_smoothed_cross_entropy_w_parallel')
class LabelSmoothedCrossEntropyWParallelCriterion(FairseqCriterion):
    def __init__(self, task, sentence_avg, label_smoothing, stop_input_gradient=False):
        super().__init__(task)
        self.sentence_avg = sentence_avg
        self.eps = label_smoothing
        self.parallel_loss_fn = torch.nn.MSELoss(reduction="sum")
        self.stop_input_gradient = stop_input_gradient

    @staticmethod
    def add_args(parser):
        """Add criterion-specific arguments to the parser."""
        # fmt: off
        parser.add_argument('--label-smoothing', default=0., type=float, metavar='D',
                            help='epsilon for label smoothing, 0 means no label smoothing')
        parser.add_argument('--stop-input-gradient', action="store_true", help='stop the gradient of input tensor')
        # fmt: on

    def forward(self, model, sample, reduce=True):
        """Compute the loss for the given sample.

        Returns a tuple with three elements:
        1) the loss
        2) the sample size, which is used as the denominator for the gradient
        3) logging outputs to display while training
        """
        net_output = model(**sample['net_input'])
        loss, nll_loss = self.compute_loss(model, net_output, sample, reduce=reduce)
        try:
            parallel_loss = self.compute_parallel_loss(
                sample, model, net_output
            )
        except:
            parallel_loss = None
            pass
        # print("pid: {}, parallel_loss: {}".format(os.getpid(), parallel_loss))
        sample_size = sample['target'].size(0) if self.sentence_avg else sample['ntokens']
        if parallel_loss is not None:
            loss += parallel_loss
        logging_output = {
            'loss': loss.data,
            'nll_loss': nll_loss.data,
            'ntokens': sample['ntokens'],
            'nsentences': sample['target'].size(0),
            'sample_size': sample_size,
        }
        if parallel_loss is not None:
            logging_output['parallel_loss'] = parallel_loss.data
        else:
            logging_output['parallel_loss'] = 0
        # print("pid: {}, 'parallel_loss' in logging_output {}".format(os.getpid(), 'parallel_loss' in logging_output))
        return loss, sample_size, logging_output

    def compute_loss(self, model, net_output, sample, reduce=True):
        lprobs = model.get_normalized_probs(net_output, log_probs=True)
        lprobs = lprobs.view(-1, lprobs.size(-1))
        target = model.get_targets(sample, net_output).view(-1, 1)  # sample["target"]
        loss, nll_loss = label_smoothed_nll_loss(
            lprobs, target, self.eps, ignore_index=self.padding_idx, reduce=reduce,
        )
        return loss, nll_loss

    def compute_parallel_loss(self, sample, model, net_output):
        # print("pid: {}, parallel_input in sample: {}".format(os.getpid(), "parallel_input" in sample))
        if "parallel_input" not in sample:
            return None
        extra = net_output[1]
        assert "encoder_doc_out" in extra, "``encoder_doc_out'' must in extra when using label_smoothed_cross_entropy_w_parallel loss. " \
            "an architecture with this function is --arch bart_summ_abs_large"
        encoder_doc_out = extra['encoder_doc_out']
        if self.stop_input_gradient:
            encoder_doc_out = encoder_doc_out.detach()
        model_encoder_forward_fn = getattr(model, "my_encoder_forward", None)
        assert callable(model_encoder_forward_fn), "``model_encoder_forward_fn'' must be callable when using label_smoothed_cross_entropy_w_parallel loss. " \
            "an architecture with this function is --arch bart_summ_abs_large"
        _, parallel_doc_out = model.my_encoder_forward(
            **sample['parallel_input']
        )
        # loss = torch.nn.MSELoss(encoder_out, parallel_out, reduction="sum") / encoder_out.size(0)
        loss = self.parallel_loss_fn(encoder_doc_out, parallel_doc_out)
        return loss

    @staticmethod
    def reduce_metrics(logging_outputs) -> None:
        """Aggregate logging outputs from data parallel training."""
        loss_sum = sum(log.get('loss', 0) for log in logging_outputs)
        nll_loss_sum = sum(log.get('nll_loss', 0) for log in logging_outputs)

        ntokens = sum(log.get('ntokens', 0) for log in logging_outputs)
        sample_size = sum(log.get('sample_size', 0) for log in logging_outputs)
        nsentences = sum(log.get('nsentences', 0) for log in logging_outputs)

        metrics.log_scalar('loss', loss_sum / sample_size / math.log(2), sample_size, round=3)
        metrics.log_scalar('nll_loss', nll_loss_sum / ntokens / math.log(2), ntokens, round=3)
        metrics.log_derived('ppl', lambda meters: utils.get_perplexity(meters['nll_loss'].avg))

        # if logging_outputs[0].get("parallel_loss", None) is not None:
        parallel_loss_sum = sum(log.get('parallel_loss', 0) for log in logging_outputs)
        metrics.log_scalar('parallel_loss', parallel_loss_sum / nsentences / math.log(2), sample_size, round=3)

    @staticmethod
    def logging_outputs_can_be_summed() -> bool:
        """
        Whether the logging outputs returned by `forward` can be summed
        across workers prior to calling `reduce_metrics`. Setting this
        to True will improves distributed training speed.
        """
        return True


