import numpy as np

from baselines.carla_agent.navigation.basic_agent import BasicAgent
from baselines.carla_agent.navigation.behavior_agent import BehaviorAgent
from leaderboard.autoagents.autonomous_agent import AutonomousAgent, Track
from srunner.scenariomanager.carla_data_provider import CarlaDataProvider
import yaml
from carla_tools.sensors import CameraConfig, get_sensors, SensorConfig
from baselines.lmdrive.autopilot import AutoPilot
from baselines.lmdrive.instruction_planner import InstructionPlanner
from baselines.lmdrive.map_agent import MapAgent
import carla
from carla_tools.planning import get_angle_to
import cv2
from carla_tools.display import DisplayInterface
from baselines.lmdrive.planner import RoutePlanner


def get_entry_point():
    return "LaMPilotAgent"


class WPTAgent(AutoPilot):
    def setup(self, path_to_conf_file):
        super().setup(path_to_conf_file)
        self._hud = DisplayInterface()
        self.current_instruction = 'Drive safely.'
        if not hasattr(self, 'scenario_config_name'):
            self.scenario_config_name = None
        if not hasattr(self, 'town_id'):
            self.town_id = None
        self.sampled_scenarios = None

    def _init(self):
        super()._init()
        self._route_planner = RoutePlanner(5, 50.0)
        self._route_planner.set_route(self._global_plan, True)
        self._instruction_planner = InstructionPlanner(self.scenario_config_name, False)
        print('scenario_config_name:', self.scenario_config_name)
        print('town_id:', self.town_id)

    def run_step(self, input_data, timestamp):
        if not self.initialized:
            self._init()

        tick_data = self.tick(input_data)

        last_instruction = self._instruction_planner.command2instruct(
            self.town_id, tick_data, self._waypoint_planner.route)
        # last_notice = self._instruction_planner.pos2notice(
        #     self.sampled_scenarios, tick_data)
        # last_traffic_light_notice = self._instruction_planner.traffic_notice(
        #     tick_data)
        if self.current_instruction != last_instruction:
            self.current_instruction = last_instruction

        # gps = self._get_position(tick_data)
        # near_node, near_cmd = self._waypoint_planner.run_step(gps)
        gps = tick_data['gps']
        near_node, near_cmd = self._waypoint_planner.run_step(gps)
        far_node, far_cmd = self._command_planner.run_step(gps)
        self.near_cmd = near_cmd
        self.far_cmd = far_cmd
        steer, throttle, brake, target_speed = self._get_control(near_node, far_node, tick_data)

        control = carla.VehicleControl()
        control.steer = steer
        control.throttle = throttle
        control.brake = float(brake)

        display_data = self._prepare_display_data(tick_data, timestamp, control)
        surface = self._hud.render(display_data)
        tick_data['display'] = surface

        return control

    def _prepare_display_data(self, tick_data, timestamp, control):
        display_data = {'rgb_front': cv2.resize(tick_data['rgb_front'], (1200, 900)),
                        'rgb_left': cv2.resize(tick_data['rgb_left'], (280, 210)),
                        'rgb_right': cv2.resize(tick_data['rgb_right'], (280, 210)),
                        'instruction': "Instruction: %s" % self.current_instruction,
                        'timenspeed': f"Time: {timestamp:.3f} s, Speed: {tick_data['speed']:.2f} m/s",
                        'control': f"Tht: {control.throttle:.2f}, Str: {control.steer:.2f}, Brk: {control.brake:.2f}",
                        'agent_message': "Agent: %s" % self.agent_message,
                        }
        return display_data

    def tick(self, input_data):
        result = super().tick(input_data)
        pos = self._get_position(result)

        result['gps'] = pos
        next_wp, next_cmd = self._route_planner.run_step(pos)
        result['next_waypoint'] = next_wp
        result['next_command'] = next_cmd.value
        result['measurements'] = [pos[0], pos[1], result['compass'], result['speed']]

        theta = result['compass'] + np.pi / 2
        R = np.array([[np.cos(theta), -np.sin(theta)], [np.sin(theta), np.cos(theta)]])
        local_command_point = np.array([next_wp[0] - pos[0], next_wp[1] - pos[1]])
        local_command_point = R.T.dot(local_command_point)
        result['target_point'] = local_command_point

        return result

    def _get_control(self, target, far_target, tick_data):
        """
        This function is key for autopilot logic,
        where precise control over the vehicle's steering and acceleration/deceleration
        is necessary to navigate towards a goal while responding dynamically to immediate and future path requirements.
        """
        # FIXME
        loc = self._vehicle.get_location()
        pos = np.array([loc.x, loc.y])
        pos = tick_data["gps"]
        theta = tick_data["compass"]
        speed = tick_data["speed"]

        # Steering
        angle_unnorm = get_angle_to(pos, theta, target)
        # the maximum deviation angle is 90 degrees for full left or right steering.
        angle = angle_unnorm / 90

        steer = self._turn_controller.step(angle)
        steer = np.clip(steer, -1.0, 1.0)  # [-1.0, 1.0] representing full left turn to full right turn
        steer = round(float(steer), 3)  # rounded to three decimal places

        angle_far_unnorm = get_angle_to(pos, theta, far_target)

        # The vehicle should slow if the angle to the far target is greater than 45 degrees
        # or the angle to the near target is greater than 5 degrees (should_slow).
        should_slow = abs(angle_far_unnorm) > 45.0 or abs(angle_unnorm) > 5.0
        self.show_slow = should_slow
        target_speed = 4.0 if should_slow else 6.5

        brake = self._should_brake()
        self.should_brake = brake
        target_speed = 0.0 if brake else target_speed

        # The difference between the target speed and the current speed is limited to a maximum increment (0.25)
        # to ensure smooth acceleration.
        delta = np.clip(target_speed - speed, 0.0, 0.25)

        # A PID controller (_speed_controller) is applied to this speed difference to compute the throttle value,
        # which is then clipped to the range [0.0, 0.75].
        throttle = self._speed_controller.step(delta)
        throttle = np.clip(throttle, 0.0, 0.75)

        if brake:
            # If braking is required, the steering is halved to reduce maneuvering aggressiveness.
            throttle = 0.0
            steer *= 0.5

        return steer, throttle, brake, target_speed
