""" Class for defining specific user strategies """
import random
from user_simulation.simulated_learner import Simulated_Learner

class UserQuery:
    def __init__(self, step_size=8, strategy='random', static_class=2) -> None:
        self.sampled_classes = {0:[], 1:[], 2:[], 3:[], 4:[]}
        self.user_models = {}
        self._init_users()
        self.history = []
        self.step_size = step_size
        self.step_counter = 0
        self.current_proficency = 0
        self.strategy = strategy
        self.static_class = static_class
        self.current_class = 0 # standard class is increasing
        if self.strategy == 'decreasing':
            self.current_class = 4
        self.class_labels = [0, 1, 2, 3, 4]
        
    def _init_users(self) -> None:
        """ Initialize all simulated users """
        prof_0_user = Simulated_Learner()
        prof_0_user.load_learner_model("learner_models/prof_0.json")
        self.user_models[0] = prof_0_user
        prof_1_user = Simulated_Learner()
        prof_1_user.load_learner_model("learner_models/prof_1.json")
        self.user_models[1] = prof_1_user
        prof_2_user = Simulated_Learner()
        prof_2_user.load_learner_model("learner_models/prof_2.json")
        self.user_models[2] = prof_2_user
        prof_3_user = Simulated_Learner()
        prof_3_user.load_learner_model("learner_models/prof_3.json")
        self.user_models[3] = prof_3_user
        prof_4_user = Simulated_Learner()
        prof_4_user.load_learner_model("learner_models/prof_4.json")
        self.user_models[4] = prof_4_user

    def sample_class(self) -> int:
        """ Sample a single class according to the specified strategy. """
        if self.strategy == 'static':
            sample = self.static_class
        elif self.strategy == 'increasing':
            if self.step_counter == self.step_size:
                self.step_counter = 0
                self.current_class = (self.current_class + 1)%len(self.class_labels) # Wrap around 
                sample = self.current_class
            else:
                sample = self.current_class
                self.step_counter += 1
        elif self.strategy == 'decreasing':
            if self.step_counter == self.step_size:
                self.step_counter = 0
                self.current_class = (self.current_class - 1)%len(self.class_labels) # Wrap around 
                sample = self.current_class
            else:
                sample = self.current_class
                self.step_counter += 1
        elif self.strategy == 'interrupted':
            if len(self.history) == 12:
                self.current_class = (self.current_class + 1)%len(self.class_labels) # Wrap around 
                sample = self.current_class
                self.step_counter = 0             
            elif len(self.history) == 16:
                self.current_class = (self.current_class - 1)%len(self.class_labels) # Wrap around 
                sample = self.current_class 
                self.step_counter = 0              
            elif len(self.history) == 20:
                self.current_class = (self.current_class + 1)%len(self.class_labels) # Wrap around 
                sample = self.current_class   
                self.step_counter = 0             
            elif len(self.history) == 24:
                self.step_counter = 0             
                self.current_class = (self.current_class + 1)%len(self.class_labels) # Wrap around 
                sample = self.current_class   
            elif self.step_counter == self.step_size:
                self.step_counter = 0
                self.current_class = (self.current_class + 1)%len(self.class_labels) # Wrap around 
                sample = self.current_class
            else:
                sample = self.current_class
                self.step_counter += 1
        else:   
            # Default strategy is random sampling
            sample = random.sample(self.class_labels,1)[0]
        self.history.append(sample)
        return sample

    def get_ctest_class(self, test : list) -> int:
        """ Scores the gaps of a ctest and returns the estimated class. """
        if self.strategy == 'static':
            sim_user = self.user_models[self.static_class]
        else:
            sim_user = self.user_models[self.history[-1]]
        gaps = [self._extract_token(g) for g in test]
        gap_predictions = [sim_user.predict_gap(g) for g in gaps]
        return self._get_label(sum(gap_predictions))

    def _extract_token(self, gap : str) -> list:
        """ Extract the exact gap tokens """
        return gap.split('_')[-1] 
        
    def _get_label(self, score :int ) -> int: 
        """ Mapping of score to class """
        score = score*5
        if score <= 54:
            return 0
        elif score <= 64:
            return 1
        elif score <= 74:
            return 2
        elif score <= 84:
            return 3
        else:
            return 4

    def set_strategy(self, strategy: str) -> None:
        """ Sets the user query behavior. Possible options are:
            random : random query behavior            
            static : always query a specified class
            increasing : increase the queried class with specified step size    
            decreasing : decrease the queried class with specified step size"""
        self.strategy = strategy


