///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// This file is part of ModelBlocks. Copyright 2009, ModelBlocks developers. //
//                                                                           //
//    ModelBlocks is free software: you can redistribute it and/or modify    //
//    it under the terms of the GNU General Public License as published by   //
//    the Free Software Foundation, either version 3 of the License, or      //
//    (at your option) any later version.                                    //
//                                                                           //
//    ModelBlocks is distributed in the hope that it will be useful,         //
//    but WITHOUT ANY WARRANTY; without even the implied warranty of         //
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          //
//    GNU General Public License for more details.                           //
//                                                                           //
//    You should have received a copy of the GNU General Public License      //
//    along with ModelBlocks.  If not, see <http://www.gnu.org/licenses/>.   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <getopt.h>

#include "nl-iomacros.h"
#include "nl-cpt.h"



char psX[]     = "";
char psSpc[]   = " ";
char psSlash[] = "/";


//// G: grammar rule symbol
DiscreteDomain<int> domG;
typedef DiscreteDomainRV<int,domG> G;
const G G_NONE ("-");

//// P: pos symbol
DiscreteDomain<int> domP;
typedef DiscreteDomainRV<int,domP> P;
const P P_NONE ("-");

//// W: pos symbol
DiscreteDomain<int> domW;
typedef DiscreteDomainRV<int,domW> W;
const W W_NONE ("-");

//// Model of term P given G
typedef CPT2DModel<P,G,Prob> PgModel;
//// Model of term P given W
typedef CPT2DModel<P,W,Prob> PwModel;
//// Model of term P
typedef CPT1DModel<P,Prob> PModel;
//// Model of word W
typedef CPT1DModel<W,Prob> WModel;


//// C: right-corner transformed grammar rule symbol
DiscreteDomain<int> domC;
typedef DiscreteDomainRV<int,domC> C;
const C C_NONE ("-");

//// CC: pair of right-corner transformed grammar rule symbols associated with rule expansion
typedef DelimitedJoint2DRV<psX,C,psSpc,C,psX> CC;
const CC CC_NONE ( C_NONE, C_NONE );

//// Cr: right-corner transformed grammar rule symbol resulting from reduction
DiscreteDomain<int> domCr;
typedef DiscreteDomainRV<int,domCr> Cr;
const Cr Cr_NONE ("-");

//// Model of G (prior)
typedef CPT1DModel<G,Prob> GModel;
//// Model of CC given C (binary branching expansion)
typedef CPT2DModel<CC,C,Prob> CCModel;
//// Model of C given C (unary branching expansion)
typedef CPT2DModel<C,C,Prob> CuModel;
//// Model of Cr given C (unary branching expansion)
typedef CPT2DModel<Cr,C,Prob> CnModel;


//// D: depth (input only, to HHMM models)...
DiscreteDomain<char> domD;
typedef DiscreteDomainRV<char,domD> D;
const D D_0("0");
const D D_1("1");
const D D_2("2");
const D D_3("3");
const D D_4("4");
const D D_5("5");

//// D: depth (input only, to HHMM models)...
DiscreteDomain<char> domK;
typedef DiscreteDomainRV<char,domK> K;
const K K_00("00");
const K K_01("01");
const K K_02("02");
const K K_03("03");
const K K_04("04");
const K K_05("05");
const K K_06("06");
const K K_07("07");
const K K_08("08");
const K K_09("09");
const K K_10("10");
const K K_11("11");
const K K_12("12");
const K K_13("13");
const K K_14("14");
const K K_15("15");
const K K_16("16");
const K K_17("17");
const K K_18("18");
const K K_19("19");
const K K_20("20");


//// Model of C given D (depth-specific prior)
typedef CPT2DModel<C,D,Prob> CdModel;
//// Model of CC given D (depth-specific prior over adjacent pairs)
typedef CPT2DModel<CC,D,Prob> CCdModel;
//// Model of CC given D and C (depth-limited binary-branching expansion)
typedef CPT3DModel<CC,D,C,Prob> CCdcModel;
const C C_TOP ("ROOT/REST");

//// Q: incomplete category state
typedef DelimitedJoint2DRV<psX,G,psSlash,G,psX> Q;
//DiscreteDomain<int> domQ;
//typedef DiscreteDomainRV<int,domQ> Q;
const Q Q_NONE ("-/-");

//// B: boolean
DiscreteDomain<int> domB;
typedef DiscreteDomainRV<int,domB> B;
const B B_0 ("0");
const B B_1 ("1");

//// F: final state
DiscreteDomain<int> domF;
typedef DiscreteDomainRV<int,domF> F;
const F F_0 ("0");
const F F_1 ("1");
const F F_NONE ("-");

//// Expansion model of Q given D and Q (from above)
typedef CPT3DModel<G,D,G,Prob> GeModel;
//// Transition model of G (awaited cat) given D and G (awaited cat from previous) and G (from reduction)
typedef CPT4DModel<G,D,G,G,Prob> GtmModel;
//// Transition model of G (active cat)  given D and G (active cat from previous) and G (awaited cat from above)
typedef CPT4DModel<G,D,G,G,Prob> GtpModel;
//// Transition model of G (awaited cat) given D and G (active cat from current) and G (active cat from previous)
typedef CPT4DModel<G,D,G,G,Prob> GtqModel;
//// Reduction model of F given D and G (active cat from previous) and Q (from above) (only check if previous awaited = below)
typedef CPT4DModel<F,D,G,G,Prob> FrModel;


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
///
///  Main function
///
////////////////////////////////////////////////////////////////////////////////

static struct option long_options[] = {
  {"help", no_argument, 0, 'h'},
  {"port", required_argument, 0, 'p'},
  {"query", required_argument, 0, 'q'},
  {"verbose", no_argument, 0, 'v'},
  {"very-verbose", no_argument, 0, 'V'},
  {0, 0, 0, 0}};


void printUsage(char* progname) {
  //fprintf(stderr,"Usage: cat <RAW_FILE> | %s [OPTIONS]... [MODEL_FILE1] [MODEL_FILE2] ...\n",progname);
  //fprintf(stderr,"Runs the speech recognition system on RAW_FILE using the models in MODEL_FILE1, 2, etc.\n");
  fprintf(stderr," System variable Settings:\n");
  fprintf(stderr,"  -h, --help\t\tPrint this message\n");
  //fprintf(stderr,"  -p, --port=PORT\tListens for updates on port PORT. (default %d.)\n",UPDATE_PORT_NUM);
  //fprintf(stderr,"  -q, --quer=PORT\tListens for queries on port PORT. (default %d.)\n",QUERY_PORT_NUM);
  fprintf(stderr,"  -v, --verbose\t\tTurns on NOISY output. (default off.)\n");
  fprintf(stderr,"  -V, --very-verbose\tTurns on VERY_NOISY and NOISY output. (default off.)\n");
}


int main (int nArgs, char* argv[]) {

  PgModel mPg;
  PwModel mPw;
  PModel  mP;
  WModel  mW;
  GModel  mG;
  GModel  mGr;
  CCModel mCC;
  CuModel mCu;
  CnModel mCn;
  CdModel   mCd;
  CCdModel  mCCd;
  CCdcModel mCCdc;

  const int D_MAX=4;
  const int K_MAX=20;

  ////////// PART 0: COMMAND-LINE OPTIONS

  // Parse command-line options
  char* progname = strdup(argv[0]);
  int option_index = 0;
  char opt;
  while( (opt = getopt_long(nArgs, argv, "hp:q:Vv", long_options, &option_index)) != -1 ) {
    switch(opt) {
    case 'h': printUsage(progname);
              exit(0);
    //case 'p': UPDATE_PORT_NUM  = atoi(optarg); break;
    //case 'q': QUERY_PORT_NUM = atoi(optarg); break;
    //case 'V': WorldModel::OUTPUT_VERY_VERBOSE = true; break; // do break.
    //case 'v': WorldModel::OUTPUT_VERBOSE = true; break;
    default: break;
    }
  }
  // Complain if too few args...
  if (optind > nArgs ) {
    printUsage(progname);
    exit(0);
  }


  ////////// PART I. LOAD MODEL (FROM STDIN)

  FILE* pf = stdin;

  int c=' '; int i=0; int line=1; String sBuff(1000);                          // Lookahead/ctrs/buffers
  CONSUME_ALL ( pf, c, WHITESPACE(c), line);                                   // Get to first record
  while ( c!=-1 && c!='\0' && c!='\5' ) {                                      // For each record
    CONSUME_STR ( pf, c, (c!='\n' && c!='\0' && c!='\5'), sBuff, i, line );    //   Consume line
    StringInput si(sBuff.c_array());
    if ( !( sBuff[0]=='#' ||                                                   //   Accept comments/fields
	    (si>>"Pg ">>mPg>>"\0"!=NULL) ||
	    (si>>"Pw ">>mPw>>"\0"!=NULL) ||
	    (si>>"P " >>mP >>"\0"!=NULL) ||
            (si>>"W " >>mW >>"\0"!=NULL) ||
            (si>>"G " >>mG >>"\0"!=NULL) ||
	    (si>>"Gr ">>mGr>>"\0"!=NULL) ||
	    (si>>"CC ">>mCC>>"\0"!=NULL) ||
	    (si>>"Cu" >>mCu>>"\0"!=NULL) ||
	    (si>>"Cn" >>mCn>>"\0"!=NULL) ) )
      cerr<<"\nERROR: unable to process line "<<line<<": "<<sBuff<<"\n";
    CONSUME_ALL ( pf, c, WHITESPACE(c), line);                                 //   Consume whitespace
  }

  //mCC.dump(cout,"CC");
  //exit(0);

  SimpleHash<C,B> hIsComplete;
  C cA; for ( bool bA=cA.setFirst(); bA; bA=cA.setNext() ) {
    hIsComplete.set(cA) = ( (cA.getString().find('/')==string::npos) ? B_1 : B_0 );
  }
//   SimpleHash<C,B> hIsRight;
//   C cA; for ( bool bA=cA.setFirst(); bA; bA=cA.setNext() ) {
//     CCModel::IterVal ccBC; for ( bool bBC=mCC.setFirst(ccBC,cA); bBC; bBC=mCC.setNext(ccBC,cA) ) {
//       hIsRight.set(ccBC.second)=B_1;
//     }
//   }
//   GModel::IterVal gA; for ( bool bA=mGr.setFirst(gA); bA; bA=mGr.setNext(gA) ) {
//     String sA; sA<<gA; C cA(sA.c_array());
//     hIsRight.set(cA)=B_1;
//   }

  mPg.dump(cout,"Pg");
  mPw.dump(cout,"Pw");
  mP.dump(cout,"P");
  mW.dump(cout,"W");

  mPg.clear();
  mPw.clear();
  mP.clear();
  mW.clear();
  mG.clear();
  mCu.clear();
  mCn.clear();

  ////////// PART II. MODEL TRANSFORMS

  ////////// 3. DEFINE DEPTH-BOUNDED RIGHT-CORNER GRAMMAR PROBS...

  //// 3a. Define depth-bounded subtree probs...

  CPT4DModel<B,K,D,C,Prob> mCbounded;
  /*
  class FdkModel : public CPT4DModel<F,K,D,C,Prob> {
   public:
    bool  setFirst (       CPT4DModel<F,K,D,C,Prob>::IterVal& f, const K& k, const D& d, const C& c ) { return CPT4DModel<F,K,D,C,Prob>::setFirst(f,k.toInt()%2,d,c); }
    bool  setNext  (       CPT4DModel<F,K,D,C,Prob>::IterVal& f, const K& k, const D& d, const C& c ) { return CPT4DModel<F,K,D,C,Prob>::setNext (f,k.toInt()%2,d,c); }
    Prob& setProb  ( const CPT4DModel<F,K,D,C,Prob>::IterVal& f, const K& k, const D& d, const C& c ) { return CPT4DModel<F,K,D,C,Prob>::setProb (f,k.toInt()%2,d,c); }
    Prob  getProb  ( const CPT4DModel<F,K,D,C,Prob>::IterVal& f, const K& k, const D& d, const C& c ) { return CPT4DModel<F,K,D,C,Prob>::getProb (f,k.toInt()%2,d,c); }
  };
  FdkModel mCbounded;
  */
  // For chains of increasing length k...
  for ( int k=0; k<=K_MAX; k++ ) {
    // For decreasing depths d...
    for ( int d=D_MAX+1; d>=1 && d>=D_MAX+1-k; d-- ) {
      cerr<<"3a,"<<k<<","<<d<<"...\n";

      mCbounded.setProb ( B_1, k, d, C_NONE ) = 1.0;
      ////cerr<<"  a "<<C_NONE<<" = "<<mCbounded.getProb ( B_1, k, d, C_NONE )<<"\n";

      // For each A=.../...|...
      for ( CCModel::const_iterator iCC=mCC.begin(); iCC!=mCC.end(); iCC++ ) {
        C cA = iCC->first.getX1();

        //mCbounded.setProb ( B_1, k, d, cA ) = 0.0;

        if ( k==D_MAX+1-d || d==D_MAX+1 ) {   // CODE REVIEW: K CONDITION REDUNDANT?
          if ( mCC.getProb(CC_NONE,cA)!=Prob() ) {
            mCbounded.setProb ( B_1, k, d, cA ) += mCC.getProb(CC_NONE,cA);
            ////cerr<<"  b "<<cA<<" = "<<mCbounded.getProb ( B_1, k, d, cA )<<"\n";
          }
        }
        // Induction step (length + depth)...
        if ( k>0 && d<D_MAX+1 ) {   // CODE REVIEW: K CONDITION REDUNDANT?
          // For each B,C by rule from A...
          CCModel::IterVal ccBC; for ( bool bBC=mCC.setFirst(ccBC,cA); bBC; bBC=mCC.setNext(ccBC,cA) ) {
            ////cerr<<"    "<<cA<<" -> "<<ccBC<<"\n";
            // Obtain equivalent simpler category if any...
            CuModel::IterVal cB; /*if ( !mCu.setFirst(cB,ccBC.first) )*/ cB=ccBC.first;
            //if (k==20&&d==4)
            ////cerr<<"  "<<cA<<" -> "<<ccBC<<"  ("<<cB<<" "<<mCbounded.getProb(B_1,k-1,d,cB)<<" "<<mCbounded.getProb(B_1,k-1,d+1,ccBC.second)<<")\n";
            if ( mCbounded.getProb(B_1,k-1,d,cB)!=Prob() && mCbounded.getProb(B_1,k-1,d+1,ccBC.second)!=Prob() ) {
              // Define prob...
              mCbounded.setProb ( B_1, k, d, cA ) += mCC.getProb(ccBC,cA) * mCbounded.getProb(B_1,k-1,d,cB) * mCbounded.getProb(B_1,k-1,d+1,ccBC.second);
              ////cerr<<"  c "<<cA<<" = "<<mCbounded.getProb ( B_1, k, d, cA )<<"\n";
            }
          }
        }
      }
    }
  }

  ////mCbounded.dump(cout,"Cb");

  //// 3b. Define depth-bounded RC-PCFG probs...

  // For decreasing depths d...
  for ( int d=D_MAX; d>=0; d-- ) {
    cerr<<"3b,"<<d<<"...\n";
    // Base case...
    if ( 0==d ) {
      // For each Ae expanded from level above...
      GModel::IterVal gAe; for ( bool bAe=mGr.setFirst(gAe); bAe; bAe=mGr.setNext(gAe) ) {
	String sAe; sAe<<gAe; C cAe(sAe.c_array());
        CC ccTOP_Ae ( C_TOP, cAe );
        mCCdc.setProb ( ccTOP_Ae, d, C_TOP ) += mGr.getProb(gAe) * mCbounded.getProb(B_1,K_MAX,d+1,cAe);
      }
    }
    // Induction step...
    else {
      // For each A/X|...
      for ( CCModel::const_iterator iCC=mCC.begin(); iCC!=mCC.end(); iCC++ ) {
        C cA = iCC->first.getX1();
        // For each A/V|W,W...
        CCModel::IterVal ccBC; for ( bool bBC=mCC.setFirst(ccBC,cA); bBC; bBC=mCC.setNext(ccBC,cA) ) {
          // Obtain equivalent simpler category if any...
          CuModel::IterVal cB; /*if ( !mCu.setFirst(cB,ccBC.first) )*/ cB=ccBC.first;
          if ( mCbounded.getProb(B_1,K_MAX,d,cB)!=Prob() && mCbounded.getProb(B_1,K_MAX,d+1,ccBC.second)!=Prob() ) {
            // Define prob...
            mCCdc.setProb ( ccBC, d, cA ) = mCC.getProb(ccBC,cA) * mCbounded.getProb(B_1,K_MAX,d,cB) * mCbounded.getProb(B_1,K_MAX,d+1,ccBC.second);
          }
        }
      }
    }
  }
  mCCdc.normalize();

  ////mCCdc.dump(cout,"CCdc");

  //// 3c. Define depth-bounded RC-PCFG left star probs...

  //typedef CPT4DModel<C,K,D,C,Prob> CdkcModel;
  class CdkcModel : public CPT4DModel<C,K,D,C,Prob> {
   public:
    //typedef CPT4DModel<C,K,D,C,Prob>::IterVal IterVal;
    bool  setFirst (       CPT4DModel<C,K,D,C,Prob>::IterVal& c, const K& k, const D& d, const C& cP ) { return CPT4DModel<C,K,D,C,Prob>::setFirst(c,k.toInt()%2,d,cP); }
    bool  setNext  (       CPT4DModel<C,K,D,C,Prob>::IterVal& c, const K& k, const D& d, const C& cP ) { return CPT4DModel<C,K,D,C,Prob>::setNext (c,k.toInt()%2,d,cP); }
    Prob& setProb  ( const CPT4DModel<C,K,D,C,Prob>::IterVal& c, const K& k, const D& d, const C& cP ) { return CPT4DModel<C,K,D,C,Prob>::setProb (c,k.toInt()%2,d,cP); }
    Prob  getProb  ( const CPT4DModel<C,K,D,C,Prob>::IterVal& c, const K& k, const D& d, const C& cP ) { return CPT4DModel<C,K,D,C,Prob>::getProb (c,k.toInt()%2,d,cP); }
  };
  CdkcModel mClstar;
  typedef CPT3DModel<C,D,C,Prob> CdcModel;
  CdcModel mClstarzero;
  CdcModel mClstarsum;
  // For chains of increasing length k...
  for ( int k=0; k<=K_MAX; k++ ) {
    // For increasing depth limits d...
    for ( int d=1; d<=4; d++ ) {
      cerr<<"3c,"<<k<<","<<d<<"...\n";
      // For each A_0...
      C cA0; for ( bool bA0=cA0.setFirst(); bA0; bA0=cA0.setNext() ) if ( B_1==hIsComplete.get(cA0) ) {
        // Base case...
        if ( 0==k ) {
          mClstar.setProb ( cA0, k, d, cA0 ) = 1.0;
        }
        // Induction step...
        else {
          // For each output, clear...
          CdkcModel::IterVal cAk; for ( bool bAk=mClstar.setFirst(cAk,k,d,cA0); bAk; bAk=mClstar.setNext(cAk,k,d,cA0) ) {
	    mClstar.setProb(cAk,k,d,cA0)=0.0;
	  }

          // For each A_k-1 by left-star from A_0...
	  CdkcModel::IterVal cAkdec; for ( bool bAkdec=mClstar.setFirst(cAkdec,k-1,d,cA0); bAkdec; bAkdec=mClstar.setNext(cAkdec,k-1,d,cA0) ) {
            // Obtain equivalent simpler category A_k-1 if any...
            CuModel::IterVal cAkd; /*if ( !mCu.setFirst(cAkd,cAkdec) )*/ cAkd=cAkdec;
            // For each A_k,B_k by rule from simplified A_k-1...
	    CCdcModel::IterVal ccAkBk; for ( bool bAk=mCC.setFirst(ccAkBk,cAkd); bAk; bAk=mCC.setNext(ccAkBk,cAkd) ) {
              if ( mClstar.getProb(cAkdec,k-1,d,cA0)!=Prob() && mCC.getProb(ccAkBk,cAkd)!=Prob() &&
                   mCbounded.getProb(B_1,K_MAX,d+1,ccAkBk.second)!=Prob() &&
                   ccAkBk.first != C_NONE ) {
                // Define prob...
                mClstar.setProb ( ccAkBk.first, k, d, cA0 ) +=
                  mClstar.getProb(cAkdec,k-1,d,cA0) * mCC.getProb(ccAkBk,cAkd) * mCbounded.getProb(B_1,K_MAX,d+1,ccAkBk.second);
              }
            }
          }
        }

        // If k sufficiently large...
        if ( k>=0 ) {
          // For each output, calc running total...
          CdkcModel::IterVal cAk; for ( bool bAk=mClstar.setFirst(cAk,k,d,cA0); bAk; bAk=mClstar.setNext(cAk,k,d,cA0) ) {
            mClstarsum.setProb ( cAk, d, cA0 ) += mClstar.getProb(cAk,k,d,cA0);
          }
        }
	// Store zero-level...
	if ( 0==k ) {
          // For each output, store zero...
          CdkcModel::IterVal cAk; for ( bool bAk=mClstar.setFirst(cAk,0,d,cA0); bAk; bAk=mClstar.setNext(cAk,0,d,cA0) ) {
	    //cerr<<"adding "<<cAk<<" "<<d<<" "<<cA0<<"\n";
            mClstarzero.setProb ( cAk, d, cA0 ) += mClstar.getProb(cAk,0,d,cA0);
          }
	}
      }
    }
  }

  //mClstarzero.dump(cout,"Clstarzero");
  //mClstarsum.dump(cout,"Clstarsum");

  mCbounded.clear();
  mClstar.clear();

  //// 3d. Define prior over incomplete constituents C...

  // For increasing depth d...
  for ( int d=0; d<=4; d++ ) {
    // Base case...
    if ( 0==d ) {
      mCd.setProb ( C_TOP, d ) = 1;
    }
    // Induction step...
    else {
      // For each Ah/Ahm1 at d-1 (if d-1=0, has to be ROOT/REST)...
      CdModel::IterVal cAhAhm1; for ( bool     bAhAhm1=mCd.setFirst(cAhAhm1,d-1);
	  			      bAhAhm1; bAhAhm1=mCd.setNext (cAhAhm1,d-1) ) {
	cerr<<"3d,"<<d<<","<<cAhAhm1<<"...\n";
        // For each Ah/Ahm, Ahm0 expanded from Ah/Ahm1 at d-1 (if d-1=0, has to be ROOT/REST -> ROOT/REST Ahm0)...
        CCdcModel::IterVal ccAhAhm_Ahm0; for ( bool         bAhAhm_Ahm0=mCCdc.setFirst(ccAhAhm_Ahm0,d-1,cAhAhm1);
	  				       bAhAhm_Ahm0; bAhAhm_Ahm0=mCCdc.setNext (ccAhAhm_Ahm0,d-1,cAhAhm1) ) {
	  C cAhm0 = ccAhAhm_Ahm0.second;
	  // If not unary...
	  if ( cAhm0 != C_NONE ) {
	    // For each Ahm0/Ahm0n...
	    CdcModel::IterVal cAhm0nAhm0nu; for ( bool          bAhm0nAhm0nu=mClstarsum.setFirst(cAhm0nAhm0nu,d,cAhm0);
						  bAhm0nAhm0nu; bAhm0nAhm0nu=mClstarsum.setNext (cAhm0nAhm0nu,d,cAhm0) ) {
              C c = cAhm0nAhm0nu;  // Fixes value before tampering with hashes in setProb.
	      mCd.setProb ( c, d ) +=
                mCd.getProb(cAhAhm1,d-1) * mCCdc.getProb(ccAhAhm_Ahm0,d-1,cAhAhm1) * mClstarsum.getProb(cAhm0nAhm0nu,d,cAhm0);
	    }
	  }
	}
      }
    }
  }
  mCd.normalize();
  ////mCd.dump(cout,"Cd");

  //// 3e. Define prior over adjacent constituents at d-1 and d...

  // For increasing depth d...
  for ( int d=1; d<=4; d++ ) {
    // For each Ah/Ahm1 at d-1 (if d-1=0, has to be ROOT/REST)...
    CdModel::IterVal cAhAhm1; for ( bool     bAhAhm1=mCd.setFirst(cAhAhm1,d-1);
				    bAhAhm1; bAhAhm1=mCd.setNext (cAhAhm1,d-1) ) {
      cerr<<"3e,"<<d<<","<<cAhAhm1<<"...\n";
      // For each Ah/Ahm, Ahm0 expanded from Ah/Ahm1 at d-1 (if d-1=0, has to be ROOT/REST -> ROOT/REST Ahm0)...
      CCdcModel::IterVal ccAhAhm_Ahm0; for ( bool         bAhAhm_Ahm0=mCCdc.setFirst(ccAhAhm_Ahm0,d-1,cAhAhm1);
					     bAhAhm_Ahm0; bAhAhm_Ahm0=mCCdc.setNext (ccAhAhm_Ahm0,d-1,cAhAhm1) ) {
	C cAhAhm = ccAhAhm_Ahm0.first;
	C cAhm0  = ccAhAhm_Ahm0.second;
        // If not unary...
	if ( cAhm0 != C_NONE ) {
	  // For each Ahm0n/Ahm0nu reachable from Ahm0...
	  CdcModel::IterVal cAhm0nAhm0nu; for ( bool          bAhm0nAhm0nu=mClstarsum.setFirst(cAhm0nAhm0nu,d,cAhm0);
						bAhm0nAhm0nu; bAhm0nAhm0nu=mClstarsum.setNext (cAhm0nAhm0nu,d,cAhm0) ) {
            mCCd.setProb ( CC(cAhAhm,cAhm0nAhm0nu), d ) +=
              mCd.getProb(cAhAhm1,d-1) * mCCdc.getProb(ccAhAhm_Ahm0,d-1,cAhAhm1) * mClstarsum.getProb(cAhm0nAhm0nu,d,cAhm0);
          }
        }
      }
    }
  }
  mCCd.normalize();
  ////mCCd.dump(cout,"CCd");


  ////////// 4. DEFINE HHMM PROBS...

  //// 4a. Define Q-Expand probs (to preterminal)...

  GeModel mGe;
  // For increasing depth d...
  for ( int d=1; d<=4; d++ ) {
    // For each pair of stack elements at d...
    CCdModel::IterVal ccAhAhm_Ahm0nAhm0n1; for ( bool                bAhAhm_Ahm0nAhm0n1=mCCd.setFirst(ccAhAhm_Ahm0nAhm0n1,d);
                                                 bAhAhm_Ahm0nAhm0n1; bAhAhm_Ahm0nAhm0n1=mCCd.setNext (ccAhAhm_Ahm0nAhm0n1,d) ) {
      C cAhAhm       = ccAhAhm_Ahm0nAhm0n1.first;
      C cAhm0nAhm0n1 = ccAhAhm_Ahm0nAhm0n1.second;
      cerr<<"4a,"<<d<<","<<ccAhAhm_Ahm0nAhm0n1<<"...\n";
      // For each -, - expanded from Ahm0n/Ahm0n1...
      CCdcModel::IterVal ccX_X; for ( bool  bX_X=mCCdc.setFirst(ccX_X,d,cAhm0nAhm0n1);
                                      bX_X; bX_X=mCCdc.setNext (ccX_X,d,cAhm0nAhm0n1) ) {
        if ( C_NONE==ccX_X.first && C_NONE==ccX_X.second && mCC.getProb(CC_NONE,cAhm0nAhm0n1)!=Prob() ) {
          String sAhm0nAhm0n1; sAhm0nAhm0n1<<cAhm0nAhm0n1<<"/"<<cAhm0nAhm0n1;
          mGe.setProb ( Q(sAhm0nAhm0n1.c_array()).first, d, Q(cAhAhm.getString().c_str()).second )
            += ( mCCd.getProb(ccAhAhm_Ahm0nAhm0n1,d)
                 * mCCdc.getProb(ccX_X,d,cAhm0nAhm0n1) );
          mGe.setProb ( Q_NONE.first, d+1, Q(sAhm0nAhm0n1.c_array()).second ) = Prob(1.0);
	}
      }
    }
  }
  mGe.normalize();
  mGe.dump(cout,"Ge");
  mGe.clear();

  //// 4b. Define Q-Transition-minus probs

  GtmModel mGtm;
  // For increasing depth d...
  for ( int d=1; d<=4; d++ ) {
    // For each Ah/Ahm1...
    CdModel::IterVal cAhAhm1; for ( bool     bAhAhm1=mCd.setFirst(cAhAhm1,d);
				    bAhAhm1; bAhAhm1=mCd.setNext (cAhAhm1,d) ) {
      cerr<<"4b,"<<d<<","<<cAhAhm1<<"...\n";
      // If incomplete...
      if ( B_0==hIsComplete.get(cAhAhm1) ) {
        // For each Ah/Ahm, Ahm0 expanded from Ah/Ahm1...
        CCdcModel::IterVal ccAhAhm_Ahm0; for ( bool         bAhAhm_Ahm0=mCCdc.setFirst(ccAhAhm_Ahm0,d,cAhAhm1);
                                               bAhAhm_Ahm0; bAhAhm_Ahm0=mCCdc.setNext (ccAhAhm_Ahm0,d,cAhAhm1) ) {
          C cAhAhm = ccAhAhm_Ahm0.first;
          C cAhm0  = ccAhAhm_Ahm0.second;
          if ( cAhm0 != C_NONE ) {
            mGtm.setProb ( Q(cAhAhm1.getString().c_str()).second, d, Q(cAhAhm.getString().c_str()).second, G(cAhm0.getString().c_str()) )
              += mCd.getProb(cAhAhm1,d) * mCCdc.getProb(ccAhAhm_Ahm0,d,cAhAhm1);
          }
        }
      }
    }
  }
  mGtm.normalize();
  mGtm.dump(cout,"Gtm");
  mGtm.clear();

  //// 4c. Define Q-Transition-plus probs

  GtpModel mGtp;
  GtqModel mGtq;
  // For increasing depth d...
  for ( int d=1; d<=4; d++ ) {
    // For each pair of stack elements at d...
    CCdModel::IterVal ccAhAhm_Ahm0nAhm0nu; for ( bool                bAhAhm_Ahm0nAhm0nu=mCCd.setFirst(ccAhAhm_Ahm0nAhm0nu,d);
                                                 bAhAhm_Ahm0nAhm0nu; bAhAhm_Ahm0nAhm0nu=mCCd.setNext (ccAhAhm_Ahm0nAhm0nu,d) ) {
      C cAhAhm       = ccAhAhm_Ahm0nAhm0nu.first;
      C cAhm0nAhm0nu = ccAhAhm_Ahm0nAhm0nu.second;
      cerr<<"4c,"<<d<<","<<ccAhAhm_Ahm0nAhm0nu<<"...\n";
      // For each Ahm0n0, - expanded from Ahm0n/Ahm0nu...
      CCdcModel::IterVal ccAhm0n0_X; for ( bool       bAhm0n0_X=mCCdc.setFirst(ccAhm0n0_X,d,cAhm0nAhm0nu);
                                           bAhm0n0_X; bAhm0n0_X=mCCdc.setNext (ccAhm0n0_X,d,cAhm0nAhm0nu) ) {
        C cAhm0n0 = ccAhm0n0_X.first;
        if ( cAhm0n0 != C_NONE && ccAhm0n0_X.second == C_NONE ) {
          // Probability of q^d_t given f^d_t q^d-1_t
          mGtp.setProb ( Q(cAhm0nAhm0nu.getString().c_str()).first, d, G(cAhm0n0.getString().c_str()), Q(cAhAhm.getString().c_str()).second )
            += ( mCCd.getProb(ccAhAhm_Ahm0nAhm0nu,d)
                 * mCCdc.getProb(ccAhm0n0_X,d,cAhm0nAhm0nu) );
          mGtq.setProb ( Q(cAhm0nAhm0nu.getString().c_str()).second, d, Q(cAhm0nAhm0nu.getString().c_str()).first, G(cAhm0n0.getString().c_str()) )
            += ( mCCd.getProb(ccAhAhm_Ahm0nAhm0nu,d)
                 * mCCdc.getProb(ccAhm0n0_X,d,cAhm0nAhm0nu) );
	}
      }
    }
  }
  mGtp.normalize();
  mGtq.normalize();
  mGtp.dump(cout,"Gtp");
  mGtq.dump(cout,"Gtq");
  mGtp.clear();
  mGtq.clear();

  //// 4d. Define F-Reduce probs

  FrModel mFr;
  // For increasing depth d...
  for ( int d=1; d<=4; d++ ) {
    // For each Ah/Ahm1 at d-1 (if d-1=0, has to be ROOT/REST)...
    CdModel::IterVal cAhAhm1; for ( bool     bAhAhm1=mCd.setFirst(cAhAhm1,d-1);
				    bAhAhm1; bAhAhm1=mCd.setNext (cAhAhm1,d-1) ) {
      cerr<<"4d,"<<d<<","<<cAhAhm1<<"...\n";
      // For each Ah/Ahm, Ahm0 expanded from Ah/Ahm1 at d-1 (if d-1=0, has to be ROOT/REST -> ROOT/REST Ahm0)...
      CCdcModel::IterVal ccAhAhm_Ahm0; for ( bool         bAhAhm_Ahm0=mCCdc.setFirst(ccAhAhm_Ahm0,d-1,cAhAhm1);
					     bAhAhm_Ahm0; bAhAhm_Ahm0=mCCdc.setNext (ccAhAhm_Ahm0,d-1,cAhAhm1) ) {
	C cAhAhm = ccAhAhm_Ahm0.first;
	C cAhm0  = ccAhAhm_Ahm0.second;
	if ( cAhm0 != C_NONE ) {
 	  // For each Ahm0n reachable from Ahm0...
 	  CdcModel::IterVal cAhm0n; for ( bool    bAhm0n=mClstarsum.setFirst(cAhm0n,d,cAhm0);
 					  bAhm0n; bAhm0n=mClstarsum.setNext (cAhm0n,d,cAhm0) ) {
            // If complete...
            if ( B_1==hIsComplete.get(cAhm0n) ) {

              Prob pr;
              if ( (pr=mClstarzero.getProb(C(cAhm0n),d,cAhm0)) != Prob() ) {
                mFr.setProb ( F(cAhm0n.getString().c_str()), d,
                              G(cAhm0n.getString().c_str()), Q(cAhAhm.getString().c_str()).second )
                  += ( mCd.getProb(cAhAhm1,d-1) * mCCdc.getProb(ccAhAhm_Ahm0,d-1,cAhAhm1) *
                       pr / mClstarsum.getProb(cAhm0n,d,cAhm0) );
              }
              if ( (pr=mClstarsum.getProb(cAhm0n,d,cAhm0)-mClstarzero.getProb(C(cAhm0n),d,cAhm0)) != Prob() ) {
                mFr.setProb ( F_1, d,
                              G(cAhm0n.getString().c_str()), Q(cAhAhm.getString().c_str()).second )
                  += ( mCd.getProb(cAhAhm1,d-1) * mCCdc.getProb(ccAhAhm_Ahm0,d-1,cAhAhm1) *
                       pr / mClstarsum.getProb(cAhm0n,d,cAhm0) );
              }
            }
          }
        }
      }
    }
  }
  mFr.normalize();
  mFr.dump(cout,"Fr");
  mFr.clear();
}



