#!/usr/bin/env python
# coding=utf-8
# Copyright 2021 The HuggingFace Team All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
Fine-tuning the library's seq2seq models for question answering using the 🤗 Seq2SeqTrainer.
"""
# You can also adapt this script on your own question answering task. Pointers for this are left as comments.

import logging
import os
import sys
import json
import random
import copy
import string
from dataclasses import dataclass, field
from typing import List, Optional, Tuple
from tqdm import tqdm
from collections import Counter

import torch
import datasets
from datasets import load_dataset, DatasetDict, concatenate_datasets, load_from_disk
import numpy as np
import pandas as pd
import transformers
from transformers import (
    AutoConfig,
    AutoModelForSeq2SeqLM,
    AutoTokenizer,
    DataCollatorForSeq2Seq,
    HfArgumentParser,
    set_seed
)
from transformers.trainer_utils import get_last_checkpoint
from src.utils import ModelArguments, DataTrainingArguments, Seq2SeqArguments, \
                LoraArguments, TASK_DEMONSTRATION_ONLY, get_dataset_sizes, \
                load_bbh_dataset, load_fs_glue_dataset, load_fs_super_glue_dataset, save_predictions
from src.metrics import compute_metrics as sni_compute_metrics
from src.models import HyperLoRAConfig, HyperLoRAModelForPretrain, HyperLoRAModelForFinetune
# from src.models.bart import HyperLoRAConfig, HyperLoRAModelForPretrain, HyperLoRAModelForFinetune
from src.trainer import Seq2SeqHyperTrainer

import os
os.environ["WANDB_PROJECT"]="HyperLoRA"

logger = logging.getLogger(__name__)

def generate_lora_weights(hyperlora, demo_datasets):
    hyperlora.cuda()
    with torch.no_grad():
        input_ids, attention_mask = demo_datasets['demo_input_ids'], demo_datasets['demo_attention_mask']
        input_ids = torch.tensor(input_ids).cuda()
        attention_mask = torch.tensor(attention_mask).cuda()
        generate_lora_weights = hyperlora.generate_lora_weights(
                                        demo_input_ids=input_ids.unsqueeze(0),
                                        demo_attention_mask=attention_mask.unsqueeze(0))

    return generate_lora_weights

def main():
    # See all possible arguments in src/transformers/training_args.py
    # or by passing the --help flag to this script.
    # We now keep distinct sets of args, for a cleaner separation of concerns.

    parser = HfArgumentParser((ModelArguments, DataTrainingArguments, Seq2SeqArguments, LoraArguments))
    if len(sys.argv) == 2 and sys.argv[1].endswith(".json"):
        # If we pass only one argument to the script and it's the path to a json file,
        # let's parse it to get our arguments.
        model_args, data_args, training_args, lora_args = parser.parse_json_file(json_file=os.path.abspath(sys.argv[1]))
    else:
        model_args, data_args, training_args, lora_args = parser.parse_args_into_dataclasses()

    # update model_args

    # Setup logging
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        handlers=[logging.StreamHandler(sys.stdout)],
    )

    log_level = training_args.get_process_log_level()
    logger.setLevel(log_level)
    datasets.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.enable_default_handler()
    transformers.utils.logging.enable_explicit_format()

    training_args.local_rank = -1
    # Log on each process the small summary:
    logger.warning(
        f"Process rank: {training_args.local_rank}, device: {training_args.device}, n_gpu: {training_args.n_gpu}"
        + f", distributed training: {bool(training_args.local_rank != -1)}, 16-bits training: {training_args.fp16}"
    )
    logger.info(f"Training/evaluation parameters {training_args}")

    # Set seed before initializing model.
    set_seed(training_args.seed)

    # Load data
    train_datasets = None
    sni_datasets = load_dataset("src/dataset/sni_dataset.py",
                                    split_subdir='splits/default',
                                    task_subdir='tasks',
                                    demo_subdir='demo_tasks',
                                    max_num_instances_per_task=64,
                                    max_num_instances_per_eval_task=100,
                                    data_dir=data_args.dataset_name)
    
    all_task_names = set(sni_datasets['test']['Task'])
    demo_datasets, raw_datasets = {}, {}
    for task_name in all_task_names:
        demo_datasets[task_name] = sni_datasets['demo'].filter(lambda example: example['Task'] == task_name, desc=None)
        raw_datasets[task_name] = sni_datasets['test'].filter(lambda example: example['Task'] == task_name, desc=None)

    # Load pretrained model and tokenizer
    #
    # Distributed training:
    # The .from_pretrained methods guarantee that only one local process can concurrently
    # download model & vocab.

    hyperlora_config = HyperLoRAConfig.from_pretrained(
        model_args.hypelora_name_or_path, lora_rank=lora_args.lora_rank
    )
    config = AutoConfig.from_pretrained(
        model_args.config_name if model_args.config_name else model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    tokenizer = AutoTokenizer.from_pretrained(
        model_args.tokenizer_name if model_args.tokenizer_name else model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        use_fast=model_args.use_fast_tokenizer,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    plm_model = AutoModelForSeq2SeqLM.from_pretrained(
        model_args.model_name_or_path,
        from_tf=bool(".ckpt" in model_args.model_name_or_path),
        config=config,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )

    # We resize the embeddings only when necessary to avoid index errors. If you are creating a model from scratch
    # on a small vocab and want a smaller embedding size, remove this test.
    embedding_size = plm_model.get_input_embeddings().weight.shape[0]
    if len(tokenizer) > embedding_size:
        plm_model.resize_token_embeddings(len(tokenizer))

    if plm_model.config.decoder_start_token_id is None:
        raise ValueError("Make sure that `config.decoder_start_token_id` is correctly defined")

    # task2id
    if not model_args.not_use_hyperlora:
        all_task_names = demo_datasets.keys()
        hyperlora_model = HyperLoRAModelForPretrain(config=hyperlora_config,
                                        model_args=model_args,
                                        lora_args=lora_args,
                                        encoder=plm_model.encoder,
                                        pretrain_task_names=all_task_names)
        task2id = hyperlora_model.task2id
        # load pre-trained checkpoint
        if model_args.pretrain_checkpoint is not None:
            logger.info('Loading checkpoint from {}'.format(model_args.pretrain_checkpoint))
            hyperlora_model.load_state_dict(torch.load(os.path.join(model_args.pretrain_checkpoint, "pytorch_model.bin"), map_location='cpu'))

        if not model_args.finetune:
            model = hyperlora_model
    else:
        model = plm_model
        all_task_names = list(raw_datasets.keys())

    # Temporarily set max_answer_length for training.
    max_answer_length = data_args.max_answer_length
    padding = "max_length" if data_args.pad_to_max_length else False

    if training_args.label_smoothing_factor > 0 and not hasattr(model, "prepare_decoder_input_ids_from_labels"):
        logger.warning(
            "label_smoothing is enabled but the `prepare_decoder_input_ids_from_labels` method is not defined for"
            f"`{model.__class__.__name__}`. This will lead to loss being calculated twice and will take up more memory"
        )

    max_seq_length = data_args.max_seq_length
    max_demo_length = data_args.max_demo_length
    input_column = data_args.input_column
    output_column = data_args.output_column

    def preprocess_sni_demonstrations(instance):
        # definition
        if isinstance(instance["Definition"], list):
            definition = (
                "Definition: " + instance["Definition"][0].strip()
            )  # TODO: should we use <Definition>?
        else:
            definition = "Definition: " + instance["Definition"].strip()
        if not definition[-1] in string.punctuation:
            definition += "."
        definition += "\n\n"

        # positive examples
        pos_examples = []
        for idx, pos_example in enumerate(instance["Positive Examples"][:2]):   # 2pos
            pos_example_str = f" Positive Example {idx+1} -\n"
            pos_example_str += f"Input: {pos_example['input'].strip()}"
            if not pos_example_str[-1] in string.punctuation:
                pos_example_str += "."
            pos_example_str += "\n"
            pos_example_str += f" Output: {pos_example['output'].strip()}"
            if not pos_example_str[-1] in string.punctuation:
                pos_example_str += "."
            pos_example_str += "\n"
            pos_example_str += "\n"
            if (
                len(
                    tokenizer(
                        definition + " ".join(pos_examples) + pos_example_str
                    )["input_ids"]
                )
                <= max_demo_length
            ):
                pos_examples.append(pos_example_str)
            else:
                break

        demo_input_str = definition + "".join(pos_examples)
        demo_tokenize_output= tokenizer(demo_input_str.strip(), max_length=max_demo_length, padding=padding, truncation=True)
        demo_output = {'demo_input_ids': demo_tokenize_output['input_ids'],
                       'demo_attention_mask': demo_tokenize_output['attention_mask']
                       }
        return demo_output

    if not model_args.not_use_hyperlora:
        # prepreocess demonstration
        for key in demo_datasets:
            demo_datasets[key] = preprocess_sni_demonstrations(demo_datasets[key][0])

    def preprocess_function(examples):
        definitions =  examples['Definition']
        instances = examples['Instance']
        pos_exps = examples['Positive Examples']
        tasks = examples['Task']
        sources, task_ids, demo_input_ids, demo_attention_mask = [], [], [], []
        for i in range(len(definitions)):
            task_input = ""
            # add the input first.
            task_input += "Now complete the following example -\n"
            task_input += f"Input: {instances[i]['input'].strip()}"
            if not task_input[-1] in string.punctuation:
                task_input += "."
            task_input += "\n"
            task_input += "Output: "

            # definition
            if isinstance(definitions[i], list):
                definition = (
                    "Definition: " + definitions[i][0].strip()
                )  # TODO: should we use <Definition>?
            else:
                definition = "Definition: " + definitions[i].strip()
            if not definition[-1] in string.punctuation:
                definition += "."
            definition += "\n\n"

            # positive examples
            pos_examples = []
            for idx, pos_example in enumerate(pos_exps[i][:2]): # 2pos
                pos_example_str = f" Positive Example {idx+1} -\n"
                pos_example_str += f"Input: {pos_example['input'].strip()}"
                if not pos_example_str[-1] in string.punctuation:
                    pos_example_str += "."
                pos_example_str += "\n"
                pos_example_str += f" Output: {pos_example['output'].strip()}"
                if not pos_example_str[-1] in string.punctuation:
                    pos_example_str += "."
                pos_example_str += "\n"
                pos_example_str += "\n"
                if (
                    len(
                        tokenizer(
                            definition + " ".join(pos_examples) + pos_example_str + task_input
                        )["input_ids"]
                    )
                    <= max_seq_length
                ):
                    pos_examples.append(pos_example_str)
                else:
                    break

            source = definition + "".join(pos_examples) + task_input
            sources.append(source)

            # demo
            task = tasks[i]
            task_ids.append(task2id[task])
            demo_input_ids.append(demo_datasets[task]['demo_input_ids'])
            demo_attention_mask.append(demo_datasets[task]['demo_attention_mask'])

        model_inputs = tokenizer(sources, max_length=max_seq_length, padding=padding, truncation=True)
        if 'output' in instances[0] and instances[0]['output']:
            labels = [random.choice(ex["output"]) for ex in instances]
            labels = tokenizer(text_target=labels, max_length=max_answer_length, padding=padding, truncation=True)
        else:
            labels = None

        # padding in the loss.
        if padding == "max_length" and data_args.ignore_pad_token_for_loss and labels:
            labels["input_ids"] = [
                [(l if l != tokenizer.pad_token_id else -100) for l in label] for label in labels["input_ids"]
            ]

        # if idx is not None: model_inputs['id'] = examples['idx']
        model_inputs["labels"] = labels["input_ids"]
        model_inputs["task_ids"] = task_ids
        model_inputs["demo_input_ids"] = demo_input_ids
        model_inputs["demo_attention_mask"] = demo_attention_mask
        return model_inputs

    # load from checkpoint
    if training_args.resume_from_checkpoint is not None:
        state_dict = torch.load(os.path.join(training_args.resume_from_checkpoint, 'pytorch_model.bin'), map_location='cpu')
        model.load_state_dict(state_dict, strict=True)
        print('Loading model from {} successful.'.format(training_args.resume_from_checkpoint))

    task2em, all_process_preds, all_process_labels, all_process_task_names, all_ems = {}, [], [], [], 0
    for task_name in tqdm(all_task_names, total=len(all_task_names), desc='Evaluating on each task'):
        eval_dataset = raw_datasets[task_name]
        column_names = eval_dataset.column_names

        if data_args.max_eval_samples is not None:
            # We will select sample from whole data
            max_eval_samples = min(len(eval_dataset), data_args.max_eval_samples)
            eval_dataset = eval_dataset.select(range(max_eval_samples))
        # Validation Feature Creation
        with training_args.main_process_first(desc="validation dataset map pre-processing"):
            eval_dataset = eval_dataset.map(
                preprocess_function,
                batched=True,
                num_proc=data_args.preprocessing_num_workers,
                remove_columns=column_names,
                load_from_cache_file=not data_args.overwrite_cache,
                desc="Running tokenizer on validation dataset",
            )
        if data_args.max_eval_samples is not None:
            # During Feature creation dataset samples might increase, we will select required samples again
            max_eval_samples = min(len(eval_dataset), data_args.max_eval_samples)
            eval_dataset = eval_dataset.select(range(max_eval_samples))

        # finetune model
        if not model_args.not_use_hyperlora and model_args.finetune:
            # generate lora modules
            generate_adapters = generate_lora_weights(hyperlora_model, demo_datasets[key])
            # generate_adapters = generate_avg_lora_weights(hyperlora_model, demo_datasets[key])

            # define down_plm model, and inject generate weights
            model = HyperLoRAModelForFinetune(
                config=config,
                model_args=model_args,
                lora_args=lora_args,
                generate_adapters=generate_adapters
            )

        # Data collator
        label_pad_token_id = -100 if data_args.ignore_pad_token_for_loss else tokenizer.pad_token_id
        data_collator = DataCollatorForSeq2Seq(
            tokenizer,
            model=model,
            label_pad_token_id=label_pad_token_id,
            pad_to_multiple_of=8 if training_args.fp16 else None
        )

        def compute_metrics(predictions, eval_dataset):
            references = [instance['Instance']['output'] for instance in eval_dataset]
            result = sni_compute_metrics(predictions=predictions, references=references)
            return result

        # Post-processing:
        def postprocess_text(preds, labels):
            preds = [pred.lower().strip().replace(".", "") for pred in preds]
            labels = [label.lower().strip().replace(".", "") for label in labels]
            return preds, labels

        # no shuffle in inter-task
        # train_dataset = train_dataset.shuffle(seed=training_args.seed)
        # call_backs = [EarlyStoppingCallback(early_stopping_patience=3)]
        # Initialize our Trainer
        trainer = Seq2SeqHyperTrainer(
            use_multitask_sampler=False,
            model=model,
            args=training_args,
            eval_dataset=eval_dataset if training_args.do_eval else None,
            tokenizer=tokenizer,
            data_collator=data_collator,
        )

        # Evaluation
        max_length = (
            training_args.generation_max_length
            if training_args.generation_max_length is not None
            else data_args.val_max_answer_length
        )
        num_beams = data_args.num_beams if data_args.num_beams is not None else training_args.generation_num_beams
    
        def process_prediction_and_labels(preds, labels):
            if isinstance(preds, tuple):
                preds = preds[0]

            if data_args.ignore_pad_token_for_loss:
                # Replace -100 in the labels as we can't decode them.
                labels = np.where(labels != -100, labels, tokenizer.pad_token_id)
                preds = np.where(preds != -100, preds, tokenizer.pad_token_id)

            decoded_preds = tokenizer.batch_decode(preds, skip_special_tokens=True)
            decoded_labels = tokenizer.batch_decode(labels, skip_special_tokens=True)

            # Some simple post-processing
            decoded_preds, decoded_labels = postprocess_text(decoded_preds, decoded_labels)
            return decoded_preds, decoded_labels

        predictions, label_ids, _ = trainer.predict(eval_dataset, max_length=max_length, num_beams=num_beams, metric_key_prefix="predict")
        process_preds, process_labels = process_prediction_and_labels(predictions, label_ids)
        
        metrics = compute_metrics(process_preds, raw_datasets[task_name])
        exact_matches, rougel = metrics['exact_match'], metrics['rougeL']
            
        print('Task Name: ', task_name)
        print('Prediction: ', process_preds[0])
        print('Labels: ', process_labels[0])
        print('Exact Match: ', exact_matches)
        print('RougeL: ', rougel)
        all_process_preds.extend(process_preds)
        all_process_labels.extend(process_labels)
        all_process_task_names.extend([task_name] * len(process_preds))
            
        task2em[task_name] = [exact_matches, rougel, len(eval_dataset)]
        
        # TODO: visualize analysis
        def transfer_to_cpu(data):
            if isinstance(data, torch.Tensor):
                return data.cpu()
            elif isinstance(data, dict):
                return {k: transfer_to_cpu(v) for k, v in data.items()}
        
        # generate lora modules
        generate_adapters = generate_lora_weights(hyperlora_model, demo_datasets[task_name])
        save_name = os.path.join(training_args.output_dir, 'generate_weights', f'sni_{task_name}_rougeL{rougel}.pt')
        torch.save(transfer_to_cpu(generate_adapters), save_name)

    save_predictions(training_args.output_dir, all_process_preds, all_process_labels, all_process_task_names)
    
    print(task2em)
    task2em_df = pd.DataFrame.from_dict(task2em)
    task2em_df.to_excel(os.path.join(training_args.output_dir, 'task2em.xlsx'))
    logger.info("Avg exact match: {}".format(all_ems/len(all_task_names)))


def _mp_fn(index):
    # For xla_spawn (TPUs)
    main()


if __name__ == "__main__":
    main()