function S = statscounters(names)
% Create a structure for statsfunhelper to record counters in manopt
% 
% function S = statscounters(name)
% function S = statscounters(names)
%
% The input can either be one string containing a chosen name for a
% counter, or a cell containing multiple strings designating multiple
% counters. The names must be valid field names for Matlab structures.
%
% The output is a structure S. For each input string, S contains a field
% with that name. That field contains a function handle. Calling that
% function with appropriate inputs (problem, x, stats, store) returns the
% value of the counter saved in store and whose name is the field name.
%
% This manopt tool is meant to be used in conjunction with incrementcounter
% and with statsfunhelper. In the examples folder of the toolbox, the
% example named using_counters demonstrates how to use this feature.
%
% See also: statscounter statsfunhelper using_counters

% This file is part of Manopt: www.manopt.org.
% Original author: Nicolas Boumal, July 27, 2018.
% Contributors: 
% Change log: 

    % If we receive only one string as input, place it in a cell so that
    % the rest of this function's code works the same in both cases.
    if ischar(names)
        names = {names};
    end
    
    assert(iscell(names), ['names must be either one string, or a ' ...
                           'cell of strings. Each string must be a ' ...
                           'valid field name for structures.']);
    
    for k = 1 : numel(names)
        
        name = names{k};
        
        assert(isvarname(name) || iskeyword(name), ...
               'Each input string must be a valid structure field name.');
        
        S.(name) = @(problem, x, stats, store) ...
                           getcountervalue(problem, x, stats, store, name);
        
    end

end

function val = getcountervalue(problem, x, stats, store, name) %#ok<INUSL>
    if isfield(store.shared, name)
        val = store.shared.(name);
    else
        val = 0;
    end
end
